#include <sys/types.h>

#include "dotsrc.h"

/*
  PROGRAM
*/

Public Bool fis_program(object)
     PseudoObject* object;
{
    return object == NULL || object->tag == PROGRAM;
}

Public void decompose_program(program,
			      env_def, exp_def, obj_def, mod_def,
			      link_def, rule_def)
     Program* program;
     EnvDef** env_def;
     ExpDef** exp_def;
     ObjDef** obj_def;
     ModDef** mod_def;
     LinkDef** link_def;
     RuleDef** rule_def;
{
    if (program == NULL) {
	*env_def = NULL;
	*exp_def = NULL;
	*obj_def = NULL;
	*mod_def = NULL;
	*link_def = NULL;
	*rule_def = NULL;
    }
    else if (is_program_non_NULL(program)) {
	*env_def = program->env_def;
	*exp_def = program->exp_def;
	*obj_def = program->obj_def;
	*mod_def = program->mod_def;
	*link_def = program->link_def;
	*rule_def = program->rule_def;
    }
    else
      type_error(program, "PROGRAM", "decompose_program");
}

Public EnvDef* read_env_def(program)
     Program* program;
{
    if (program == NULL)
      return NULL;
    else if (is_program_non_NULL(program))
      return program->env_def;
    else
      type_error(program, "PROGRAM", "read_env_def");
}

Public ExpDef* read_exp_def(program)
     Program* program;
{
    if (program == NULL)
      return NULL;
    else if (is_program_non_NULL(program))
      return program->exp_def;
    else
      type_error(program, "PROGRAM", "read_exp_def");
}

Public ObjDef* read_obj_def(program)
     Program* program;
{
    if (program == NULL)
      return NULL;
    else if (is_program_non_NULL(program))
      return program->obj_def;
    else
      type_error(program, "PROGRAM", "read_obj_def");
}

Public ModDef* read_mod_def(program)
     Program* program;
{
    if (program == NULL)
      return NULL;
    else if (is_program_non_NULL(program))
      return program->mod_def;
    else
      type_error(program, "PROGRAM", "read_mod_def");
}

Public LinkDef* read_link_def(program)
     Program* program;
{
    if (program == NULL)
      return NULL;
    else if (is_program_non_NULL(program))
      return program->link_def;
    else
      type_error(program, "PROGRAM", "read_link_def");
}

Public RuleDef* read_rule_def(program)
     Program* program;
{
    if (program == NULL)
      return NULL;
    else if (is_program_non_NULL(program))
      return program->rule_def;
    else
      type_error(program, "PROGRAM", "read_rule_def");
}

Public Program* new_program()
{
    Program* program = (Program*)dotsrc_malloc(sizeof(Program));

    program->tag = PROGRAM;
    program->env_def = NULL;
    program->exp_def = NULL;
    program->obj_def = NULL;
    program->mod_def = NULL;
    program->link_def = NULL;
    program->rule_def = NULL;
    return program;
}

Public Program* set_program(program, env_def, exp_def, obj_def, mod_def,
			    link_def, rule_def)
     Program* program;
     EnvDef* env_def;
     ExpDef* exp_def;
     ObjDef* obj_def;
     ModDef* mod_def;
     LinkDef* link_def;
     RuleDef* rule_def;
{
    if (program == NULL)
      write_through_NULL("set_program");
    if (! is_program_non_NULL(program))
      type_error(program, "PROGRAM", "set_program");
    if (! is_env_def(env_def))
      type_error(env_def, "ENV_DEF", "set_program");
    if (! is_exp_def(exp_def))
      type_error(exp_def, "EXP_DEF", "set_program");
    if (! is_obj_def(obj_def))
      type_error(obj_def, "OBJ_DEF", "set_program");
    if (! is_mod_def(mod_def))
      type_error(mod_def, "MOD_DEF", "set_program");
    if (! is_link_def(link_def))
      type_error(link_def, "LINK_DEF", "set_program");
    if (! is_rule_def(rule_def))
      type_error(rule_def, "RULE_DEF", "set_program");

    program->env_def = env_def;
    program->exp_def = exp_def;
    program->obj_def = obj_def;
    program->mod_def = mod_def;
    program->link_def = link_def;
    program->rule_def = rule_def;
    return program;
}

Public Program* create_program(env_def, exp_def, obj_def, mod_def,
			       link_def, rule_def)
     EnvDef* env_def;
     ExpDef* exp_def;
     ObjDef* obj_def;
     ModDef* mod_def;
     LinkDef* link_def;
     RuleDef* rule_def;
{
    if (! is_env_def(env_def))
      type_error(env_def, "ENV_DEF", "create_program");
    if (! is_exp_def(exp_def))
      type_error(exp_def, "EXP_DEF", "create_program");
    if (! is_obj_def(obj_def))
      type_error(obj_def, "OBJ_DEF", "create_program");
    if (! is_mod_def(mod_def))
      type_error(mod_def, "MOD_DEF", "create_program");
    if (! is_link_def(link_def))
      type_error(link_def, "LINK_DEF", "create_program");
    if (! is_rule_def(rule_def))
      type_error(rule_def, "RULE_DEF", "create_program");

    {
	Program* program = (Program*)dotsrc_malloc(sizeof(Program));

	program->tag = PROGRAM;
	program->env_def = env_def;
	program->exp_def = exp_def;
	program->obj_def = obj_def;
	program->mod_def = mod_def;
	program->link_def = link_def;
	program->rule_def = rule_def;
	return program;
    }
}

Public Program* write_env_def(program, env_def)
     Program* program;
     EnvDef* env_def;
{
    if (program == NULL)
      write_through_NULL("write_env_def");
    else if (is_program_non_NULL(program))
      if (is_env_def(env_def)) {
	  program->env_def = env_def;
	  return program;
      }
      else
	type_error(env_def, "ENV_DEF", "write_env_def");
    else
      type_error(program, "PROGRAM", "write_env_def");
}

Public Program* write_exp_def(program, exp_def)
     Program* program;
     ExpDef* exp_def;
{
    if (program == NULL)
      write_through_NULL("write_exp_def");
    else if (is_program_non_NULL(program))
      if (is_exp_def(exp_def)) {
	  program->exp_def = exp_def;
	  return program;
      }
      else
	type_error(exp_def, "EXP_DEF", "write_exp_def");
    else
      type_error(program, "PROGRAM", "write_exp_def");
}

Public Program* write_obj_def(program, obj_def)
     Program* program;
     ObjDef* obj_def;
{
    if (program == NULL)
      write_through_NULL("write_obj_def");
    else if (is_program_non_NULL(program))
      if (is_obj_def(obj_def)) {
	  program->obj_def = obj_def;
	  return program;
      }
      else
	type_error(obj_def, "OBJ_DEF", "write_obj_def");
    else
      type_error(program, "PROGRAM", "write_obj_def");
}

Public Program* write_mod_def(program, mod_def)
     Program* program;
     ModDef* mod_def;
{
    if (program == NULL)
      write_through_NULL("write_mod_def");
    else if (is_program_non_NULL(program))
      if (is_mod_def(mod_def)) {
	  program->mod_def = mod_def;
	  return program;
      }
      else
	type_error(mod_def, "MOD_DEF", "write_mod_def");
    else
      type_error(program, "PROGRAM", "write_mod_def");
}

Public Program* write_link_def(program, link_def)
     Program* program;
     LinkDef* link_def;
{
    if (program == NULL)
      write_through_NULL("write_link_def");
    else if (is_program_non_NULL(program))
      if (is_link_def(link_def)) {
	  program->link_def = link_def;
	  return program;
      }
      else
	type_error(link_def, "LINK_DEF", "write_link_def");
    else
      type_error(program, "PROGRAM", "write_rule_def");
}

Public Program* write_rule_def(program, rule_def)
     Program* program;
     RuleDef* rule_def;
{
    if (program == NULL)
      write_through_NULL("write_rule_def");
    else if (is_program_non_NULL(program))
      if (is_rule_def(rule_def)) {
	  program->rule_def = rule_def;
	  return program;
      }
    else
      type_error(rule_def, "RULE_DEF", "write_rule_def");
}

Public void delete_program(program)
     Program* program;
{
    if (program != NULL)
      if (is_program_non_NULL(program)) {
	  delete_env_def(program->env_def);
	  delete_exp_def(program->exp_def);
	  delete_obj_def(program->obj_def);
	  delete_mod_def(program->mod_def);
	  delete_link_def(program->link_def);
	  delete_rule_def(program->rule_def);
	  dotsrc_free((char*)program);
      }
      else
	type_error(program, "PROGRAM", "delete_program");
}

Public Program* copy_program(program)
     Program* program;
{
    if (program == NULL)
      return NULL;
    else if (is_program_non_NULL(program)) {
	Program* p = (Program*)dotsrc_malloc(sizeof(Program));

	p->tag = PROGRAM;
	p->env_def = copy_env_def(program->env_def);
	p->exp_def = copy_exp_def(program->exp_def);
	p->obj_def = copy_obj_def(program->obj_def);
	p->mod_def = copy_mod_def(program->mod_def);
	p->link_def = copy_link_def(program->link_def);
	p->rule_def = copy_rule_def(program->rule_def);
	return p;
    }
    else
      type_error(program, "PROGRAM", "copy_program");
}

/*
  ENV_DEF
*/

Public Bool fis_env_def(object)
     PseudoObject* object;
{
    return object == NULL || object->tag == ENV_DEF;
}

Public void decompose_env_def(env_def, name, author, date, def_libs)
     EnvDef* env_def;
     char** name;
     char** author;
     char** date;
     ObjList** def_libs;
{
    if (env_def == NULL) {
	*name = NULL;
	*author = NULL;
	*date = NULL;
	*def_libs = NULL;
    }
    else if (is_env_def_non_NULL(env_def)) {
	*name = env_def->name;
	*author = env_def->author;
	*date = env_def->date;
	*def_libs = env_def->def_libs;
    }
    else
      type_error(env_def, "ENV_DEF", "decompose_env_def");
}

Public char* read_name(env_def)
     EnvDef* env_def;
{
    if (env_def == NULL)
      return NULL;
    else if (is_env_def_non_NULL(env_def))
      return env_def->name;
    else
      type_error(env_def, "ENV_DEF", "read_name");
}

Public char* read_author(env_def)
     EnvDef* env_def;
{
    if (env_def == NULL)
      return NULL;
    else if (is_env_def_non_NULL(env_def))
      return env_def->author;
    else
      type_error(env_def, "ENV_DEF", "read_author");
}

Public char* read_date(env_def)
     EnvDef* env_def;
{
    if (env_def == NULL)
      return NULL;
    else if (is_env_def_non_NULL(env_def))
      return env_def->date;
    else
      type_error(env_def, "ENV_DEF", "read_date");
}

Public ObjList* read_def_libs(env_def)
     EnvDef* env_def;
{
    if (env_def == NULL)
      return NULL;
    else if (is_env_def_non_NULL(env_def))
      return env_def->def_libs;
    else
      type_error(env_def, "ENV_DEF", "read_def_libs");
}

Public EnvDef* new_env_def()
{
    EnvDef* env_def = (EnvDef*)dotsrc_malloc(sizeof(EnvDef));

    env_def->tag = ENV_DEF;
    env_def->name = NULL;
    env_def->author = NULL;
    env_def->date = NULL;
    env_def->def_libs = NULL;
    return env_def;
}

Public EnvDef* set_env_def(env_def, name, author, date, def_libs)
     EnvDef* env_def;
     char *name, *author, *date;
     ObjList* def_libs;
{
    if (env_def == NULL)
      write_through_NULL("set_env_def");
    if (! is_env_def_non_NULL(env_def))
      type_error(env_def, "ENV_DEF", "set_env_def");
    if (! is_obj_list(def_libs, DEF_LIB))
      type_error(def_libs, "OBJ_LIST of DEF_LIB", "set_env_def");

    env_def->name = name;
    env_def->author = author;
    env_def->date = date;
    env_def->def_libs = def_libs;
    return env_def;
}

Public EnvDef* create_env_def(name, author, date, def_libs)
     char *name, *author, *date;
     ObjList* def_libs;
{
    if (is_obj_list(def_libs, DEF_LIB)) {
	EnvDef* env_def = (EnvDef*)dotsrc_malloc(sizeof(EnvDef));

	env_def->tag = ENV_DEF;
	env_def->name = name;
	env_def->author = author;
	env_def->date = date;
	env_def->def_libs = def_libs;
	return env_def;
    }
    else
      type_error(def_libs, "OBJ_LIST of DEF_LIB", "create_env_def");
}

Public EnvDef* write_name_inEnvDef(env_def, name)
     EnvDef* env_def;
     char* name;
{
    if (env_def == NULL)
      write_through_NULL("write_name_inEnvDef");
    else if (is_env_def_non_NULL(env_def)) {
	env_def->name = name;
	return env_def;
    }
    else
      type_error(env_def, "ENV_DEF", "write_name_inEnvDef");
}

Public EnvDef* write_author(env_def, author)
     EnvDef* env_def;
     char* author;
{
    if (env_def == NULL)
      write_through_NULL("write_author");
    else if (is_env_def_non_NULL(env_def)) {
	env_def->author = author;
	return env_def;
    }
    else
      type_error(env_def, "ENV_DEF", "write_env_def");
}

Public EnvDef* write_date(env_def, date)
     EnvDef* env_def;
     char* date;
{
    if (env_def == NULL)
      write_through_NULL("write_date");
    else if (is_env_def_non_NULL(env_def)) {
	env_def->date = date;
	return env_def;
    }
    else
      type_error(env_def, "ENV_DEF", "write_date");
}

Public EnvDef* write_def_libs(env_def, def_libs)
     EnvDef* env_def;
     ObjList* def_libs;
{
    if (env_def == NULL)
      write_through_NULL("write_def_libs");
    else if (is_env_def_non_NULL(env_def))
      if (is_obj_list(def_libs, DEF_LIB)) {
	  env_def->def_libs = def_libs;
	  return env_def;
      }
      else
	type_error(def_libs, "OBJ_LIST of DEF_LIB", "write_def_libs");
    else
      type_error(env_def, "ENV_DEF", "write_def_libs");
}

Public EnvDef* insert_def_libs(env_def, def_libs)
     EnvDef* env_def;
     ObjList* def_libs;
{
    if (env_def == NULL)
      write_through_NULL("insert_def_libs");
    else if (is_env_def_non_NULL(env_def))
      if (env_def->def_libs == NULL) {
	  env_def->def_libs = def_libs;
	  return env_def;
      }
      else {
	  insert_list_to_obj_list(env_def->def_libs, def_libs);
	  return env_def;
      }
    else
      type_error(env_def, "ENV_DEF", "insert_def_libs");
}

Public EnvDef* add_def_libs(env_def, def_libs)
     EnvDef* env_def;
     ObjList* def_libs;
{
    if (env_def == NULL)
      write_through_NULL("add_def_libs");
    else if (is_env_def_non_NULL(env_def))
      if (env_def->def_libs == NULL) {
	  env_def->def_libs = def_libs;
	  return env_def;
      }
      else {
	  concat_list_to_obj_list(env_def->def_libs, def_libs);
	  return env_def;
      }
    else
      type_error(env_def, "ENV_DEF", "add_def_libs");
}

Public DefLib* car_of_def_libs(env_def)
     EnvDef* env_def;
{
    if (env_def == NULL || env_def->def_libs == NULL)
      return NULL;
    else
      return (DefLib*)extract_first_list_element(env_def->def_libs);
}

Public void delete_env_def(env_def)
     EnvDef* env_def;
{
    if (env_def != NULL)
      if (is_env_def_non_NULL(env_def)) {
	  if (env_def->name != NULL)
	    dotsrc_free(env_def->name);
	  if (env_def->author != NULL)
	    dotsrc_free(env_def->author);
	  if (env_def->date != NULL)
	    dotsrc_free(env_def->date);
	  delete_obj_list(env_def->def_libs);
	  dotsrc_free((char*)env_def);
      }
      else
	type_error(env_def, "ENV_DEF", "delete_env_def");
}

Public EnvDef* copy_env_def(env_def)
     EnvDef* env_def;
{
    if (env_def == NULL)
      return NULL;
    else if (is_env_def_non_NULL(env_def)) {
	EnvDef* e = (EnvDef*)dotsrc_malloc(sizeof(EnvDef));

	e->tag = ENV_DEF;
	e->name = dotsrc_strdup(env_def->name);
	e->author = dotsrc_strdup(env_def->author);
	e->date = dotsrc_strdup(env_def->date);
	e->def_libs = copy_obj_list(env_def->def_libs);
	return e;
    }
    else
      type_error(env_def, "ENV_DEF", "copy_env_def");
}

/*
  DEF_LIB
*/

Public Bool fis_def_lib(object)
     PseudoObject* object;
{
    return object == NULL || object->tag == DEF_LIB;
}

Public void decompose_def_lib(def_lib, lib_lab, lib_names)
     DefLib* def_lib;
     LibLab* lib_lab;
     ObjList** lib_names;
{
    if (def_lib == NULL) {
	*lib_lab = NULL;
	*lib_names = NULL;
    }
    else if (is_def_lib_non_NULL(def_lib)) {
	*lib_lab = def_lib->lib_lab;
	*lib_names = def_lib->lib_names;
    }
    else
      type_error(def_lib, "DEF_LIB", "decompose_def_lib");
}

Public LibLab read_lib_lab(def_lib)
     DefLib* def_lib;
{
    if (def_lib == NULL)
      return NULL;
    else if (is_def_lib_non_NULL(def_lib))
      return def_lib->lib_lab;
    else
      type_error(def_lib, "DEF_LIB", "read_lib_lab");
}

Public ObjList* read_lib_names(def_lib)
     DefLib* def_lib;
{
    if (def_lib == NULL)
      return NULL;
    else if (is_def_lib_non_NULL(def_lib))
      return def_lib->lib_names;
    else
      type_error(def_lib, "DEF_LIB", "read_lib_names");
}

Public DefLib* new_def_lib()
{
    DefLib* def_lib = (DefLib*)dotsrc_malloc(sizeof(DefLib));

    def_lib->tag = DEF_LIB;
    def_lib->lib_lab = NULL;
    def_lib->lib_names = NULL;
    return def_lib;
}

Public DefLib* set_def_lib(def_lib, lib_lab, lib_names)
     DefLib* def_lib;
     LibLab lib_lab;
     ObjList* lib_names;
{
    if (def_lib == NULL)
      write_through_NULL("set_lib_lab");
    if (! is_def_lib_non_NULL(def_lib))
      type_error(def_lib, "DEF_LIB", "set_lib_lab");
    if (! is_obj_list(lib_names, STRING))
      type_error(lib_names, "OBJ_LIST of STRING", "set_lib_lab");

    def_lib->lib_lab = lib_lab;
    def_lib->lib_names = lib_names;
    return def_lib;
}

Public DefLib* create_def_lib(lib_lab, lib_names)
     LibLab lib_lab;
     ObjList* lib_names;
{
    if (is_obj_list(lib_names, STRING)) {
	DefLib* def_lib = (DefLib*)dotsrc_malloc(sizeof(DefLib));

	def_lib->tag = DEF_LIB;
	def_lib->lib_lab = lib_lab;
	def_lib->lib_names = lib_names;
	return def_lib;
    }
    else
      type_error(lib_names, "OBJ_LIST of STRING", "create_def_lib");
}

Public DefLib* write_lib_lab(def_lib, lib_lab)
     DefLib* def_lib;
     LibLab lib_lab;
{
    if (def_lib == NULL)
      write_through_NULL("write_lib_lab");
    else if (is_def_lib_non_NULL(def_lib)) {
	def_lib->lib_lab = lib_lab;
	return def_lib;
    }
    else
      type_error(def_lib, "DEF_LIB", "write_lib_lab");
}

Public DefLib* write_lib_names(def_lib, lib_names)
     DefLib* def_lib;
     ObjList* lib_names;
{
    if (def_lib == NULL)
      write_through_NULL("write_lib_names");
    else if (is_def_lib_non_NULL(def_lib)) {
	def_lib->lib_names = lib_names;
	return def_lib;
    }
    else
      type_error(def_lib, "DEF_LIB", "write_lib_names");
}

Public DefLib* insert_lib_names(def_lib, lib_names)
     DefLib* def_lib;
     ObjList* lib_names;
{
    if (def_lib == NULL)
      write_through_NULL("insert_lib_names");
    else if (is_def_lib_non_NULL(def_lib))
      if (def_lib->lib_names == NULL) {
	  def_lib->lib_names = lib_names;
	  return def_lib;
      }
      else {
	  insert_list_to_obj_list(def_lib->lib_names, lib_names);
	  return def_lib;
      }
    else
      type_error(def_lib, "DEF_LIB", "insert_lib_names");
}

Public DefLib* add_lib_names(def_lib, lib_names)
     DefLib* def_lib;
     ObjList* lib_names;
{
    if (def_lib == NULL)
      write_through_NULL("add_lib_names");
    else if (is_def_lib_non_NULL(def_lib))
      if (def_lib->lib_names == NULL) {
	  def_lib->lib_names = lib_names;
	  return def_lib;
      }
      else {
	  concat_list_to_obj_list(def_lib->lib_names, lib_names);
	  return def_lib;
      }
    else
      type_error(def_lib, "DEF_LIB", "add_lib_names");
}

Public String* car_of_lib_names(def_lib)
     DefLib* def_lib;
{
    if (def_lib == NULL || def_lib->lib_names == NULL)
      return NULL;
    else
      return (String*)extract_first_list_element(def_lib->lib_names);
}

Public void delete_def_lib(def_lib)
     DefLib* def_lib;
{
    if (def_lib != NULL)
      if (is_def_lib_non_NULL(def_lib)) {
	  delete_obj_list(def_lib->lib_names);
	  dotsrc_free((char*)def_lib);
      }
      else
	type_error(def_lib, "DEF_LIB", "delete_def_lib");
}

Public DefLib* copy_def_lib(def_lib)
     DefLib* def_lib;
{
    if (def_lib == NULL)
      return NULL;
    else if (is_def_lib_non_NULL(def_lib)) {
	DefLib* d = (DefLib*)dotsrc_malloc(sizeof(DefLib));

	d->tag = DEF_LIB;
	d->lib_lab = def_lib->lib_lab;
	d->lib_names = copy_obj_list(def_lib->lib_names);
	return d;
    }
    else
      type_error(def_lib, "DEF_LIB", "copy_def_lib");
}

/*
  STRING
*/

Public Bool fis_string(object)
     PseudoObject* object;
{
    return object == NULL || object->tag == STRING;
}

Public void decompose_string(string, str_data)
     String* string;
     char** str_data;
{
    if (string == NULL) {
	*str_data = NULL;
    }
    else if (is_string_non_NULL(string)) {
	*str_data = string->str_data;
    }
    else
      type_error(string, "STRING", "decompose_string");
}

Public char* read_str_data(string)
     String* string;
{
    if (string == NULL)
      return NULL;
    else if (is_string_non_NULL(string))
      return string->str_data;
    else
      type_error(string, "STRING", "read_str_data");
}

Public String* new_string()
{
    String* string = (String*)dotsrc_malloc(sizeof(String));

    string->tag = STRING;
    string->str_data = NULL;
    return string;
}

Public String* set_string(string, str_data)
     String* string;
     char* str_data;
{
    if (string == NULL)
      write_through_NULL("set_string");
    if (! is_string_non_NULL(string))
      type_error(string, "STRING", "set_string");

    string->str_data = str_data;
    return string;
}

Public String* create_string(str_data)
     char* str_data;
{
    String* string = (String*)dotsrc_malloc(sizeof(String));

    string->tag = STRING;
    string->str_data = str_data;
    return string;
}

Public String* write_str_data(string, str_data)
     String* string;
     char* str_data;
{
    if (string == NULL)
      write_through_NULL("write_str_data");
    else if (is_string_non_NULL(string)) {
	string->str_data = str_data;
	return string;
    }
    else
      type_error(string, "STRING", "write_str_data");
}

Public void delete_string(string)
     String* string;
{
    if (string != NULL)
      if (is_string_non_NULL(string)) {
	  if (string->str_data != NULL)
	      dotsrc_free(string->str_data);
	  dotsrc_free((char*)string);
      }
      else
	type_error(string, "STRING", "delete_string");
}

Public String* copy_string(string)
     String* string;
{
    if (string == NULL)
      return NULL;
    else if (is_string_non_NULL(string)) {
	String *s = (String*)dotsrc_malloc(sizeof(String));

	s->tag = STRING;
	s->str_data = dotsrc_strdup(string->str_data);
	return s;
    }
    else
      type_error(string, "STRING", "copy_string");
}

/*
  EXP_DEF
*/

Public Bool fis_exp_def(object)
     PseudoObject* object;
{
    return object == NULL || object->tag == EXP_DEF;
}

Public void decompose_exp_def(exp_def, exps)
     ExpDef* exp_def;
     ObjList** exps;
{
    if (exp_def == NULL) {
	*exps = NULL;
    }
    else if (is_exp_def_non_NULL(exp_def)) {
	*exps = exp_def->exps;
    }
    else
      type_error(exp_def, "EXP_DEF", "decompose_exp_def");
}

Public ObjList* read_exps(exp_def)
     ExpDef* exp_def;
{
    if (exp_def == NULL)
      return NULL;
    else if (is_exp_def_non_NULL(exp_def))
      return exp_def->exps;
    else
      type_error(exp_def, "EXP_DEF", "read_exps");
}

Public ExpDef* new_exp_def()
{
    ExpDef* exp_def = (ExpDef*)dotsrc_malloc(sizeof(ExpDef));

    exp_def->tag = EXP_DEF;
    exp_def->exps = NULL;
    return exp_def;
}

Public ExpDef* set_exp_def(exp_def, exps)
     ExpDef* exp_def;
     ObjList* exps;
{
    if (exp_def == NULL)
      write_through_NULL("set_exp_def");
    if (! is_exp_def_non_NULL(exp_def))
      type_error(exp_def, "EXP_DEF", "set_exp_def");
    if (! is_obj_list(exps, EXP))
      type_error(exps, "EXPS", "set_exp_def");

    exp_def->exps = exps;
    return exp_def;
}

Public ExpDef* create_exp_def(exps)
     ObjList* exps;
{
    if (is_obj_list(exps, EXP)) {
	ExpDef* exp_def = (ExpDef*)dotsrc_malloc(sizeof(ExpDef));

	exp_def->tag = EXP_DEF;
	exp_def->exps = exps;
	return exp_def;
    }
    else
      type_error(exps, "OBJ_LIST of EXP", "create_exp_def");
}

Public ExpDef* write_exps(exp_def, exps)
     ExpDef* exp_def;
     ObjList* exps;
{
    if (exp_def == NULL)
      write_through_NULL("write_exps");
    else if (is_exp_def_non_NULL(exp_def)) {
	exp_def->exps = exps;
	return exp_def;
    }
    else
      type_error(exp_def, "EXP_DEF", "write_exps");
}

Public ExpDef* insert_exps(exp_def, exps)
     ExpDef* exp_def;
     ObjList* exps;
{
    if (exp_def == NULL)
      write_through_NULL("insert_exps");
    else if (is_exp_def_non_NULL(exp_def))
      if (exp_def->exps == NULL) {
	  exp_def->exps = exps;
	  return exp_def;
      }
      else {
	  insert_list_to_obj_list(exp_def->exps, exps);
	  return exp_def;
      }
    else
      type_error(exp_def, "EXP_DEF", "insert_exps");
}

Public ExpDef* add_exps(exp_def, exps)
     ExpDef* exp_def;
     ObjList* exps;
{
    if (exp_def == NULL)
      write_through_NULL("add_exps");
    else if (is_exp_def_non_NULL(exp_def))
      if (exp_def->exps == NULL) {
	  exp_def->exps = exps;
	  return exp_def;
      }
      else {
	  concat_list_to_obj_list(exp_def->exps, exps);
	  return exp_def;
      }
    else
      type_error(exp_def, "EXP_DEF", "add_exps");
}

Public Exp* car_of_exps(exp_def)
     ExpDef* exp_def;
{
    if (exp_def == NULL || exp_def->exps == NULL)
      return NULL;
    else
      return (Exp*)extract_first_list_element(exp_def->exps);
}

Public void delete_exp_def(exp_def)
     ExpDef* exp_def;
{
    if (exp_def != NULL)
      if (is_exp_def_non_NULL(exp_def)) {
	  delete_obj_list(exp_def->exps);
	  dotsrc_free((char*)exp_def);
      }
      else
	type_error(exp_def, "EXP_DEF", "delete_exp_def");
}

Public ExpDef* copy_exp_def(exp_def)
     ExpDef* exp_def;
{
    if (exp_def == NULL)
      return NULL;
    else if (is_exp_def_non_NULL(exp_def)) {
	ExpDef* e = (ExpDef*)dotsrc_malloc(sizeof(ExpDef));

	e->tag = EXP_DEF;
	e->exps = copy_obj_list(exp_def->exps);
	return e;
    }
    else
      type_error(exp_def, "EXP_DEF", "copy_exp_def");
}

/*
  EXP
*/

Public Bool fis_exp(object)
     PseudoObject* object;
{
    return object == NULL || object->tag == EXP;
}

Public void decompose_exp(exp, exp_name, o_term)
     Exp* exp;
     ExpName** exp_name;
     OTerm** o_term;
{
    if (exp == NULL) {
	*exp_name = NULL;
	*o_term = NULL;
    }
    else if (is_exp_non_NULL(exp)) {
	*exp_name = exp->exp_name;
	*o_term = exp->o_term;
    }
    else
      type_error(exp, "EXP", "decompose_exp");
}

Public ExpName* read_exp_name(exp)
     Exp* exp;
{
    if (exp == NULL)
      return NULL;
    else if (is_exp_non_NULL(exp))
      return exp->exp_name;
    else
      type_error(exp, "EXP", "read_exp_name");
}

Public OTerm* read_o_term_inExp(exp)
     Exp* exp;
{
    if (exp == NULL)
      return NULL;
    else if (is_exp_non_NULL(exp))
      return exp->o_term;
    else
      type_error(exp, "EXP", "read_o_term_inExp");
}

Public Exp* new_exp()
{
    Exp* exp = (Exp*)dotsrc_malloc(sizeof(Exp));

    exp->tag = EXP;
    exp->exp_name = NULL;
    exp->o_term = NULL;
    return exp;
}

Public Exp* set_exp(exp, exp_name, o_term)
     Exp* exp;
     ExpName* exp_name;
     OTerm* o_term;
{
    if (exp == NULL)
      write_through_NULL("set_exp");
    else if (is_exp_non_NULL(exp)) {
	exp->exp_name = exp_name;
	exp->o_term = o_term;
	return exp;
    }
    else
      type_error(exp, "EXP", "set_exp");
}

Public Exp* create_exp(exp_name, o_term)
     ExpName* exp_name;
     OTerm* o_term;
{
    if (! is_exp_name(exp_name))
      type_error(exp_name, "EXP_NAME", "create_exp");
    if (! is_o_term(o_term))
      type_error(exp_name, "EXP_NAME", "create_exp");

    {
	Exp* exp = (Exp*)dotsrc_malloc(sizeof(Exp));

	exp->tag = EXP;
	exp->exp_name = exp_name;
	exp->o_term = o_term;
	return exp;
    }
}

Public Exp* write_exp_name(exp, exp_name)
     Exp* exp;
     ExpName* exp_name;
{
    if (exp == NULL)
      write_through_NULL("write_exp_name");
    if (! is_exp_non_NULL(exp))
      type_error(exp, "EXP", "write_exp_name");
    if (! is_exp_name(exp_name))
      type_error(exp_name, "EXP_NAME", "write_exp_name");

    exp->exp_name = exp_name;
    return exp;
}

Public Exp* write_o_term_inExp(exp, o_term)
     Exp* exp;
     OTerm* o_term;
{
    if (exp == NULL)
      write_through_NULL("write_o_term_inExp");
    if (! is_exp_non_NULL(exp))
      type_error(exp, "EXP", "write_o_term_inExp");
    if (! is_o_term(o_term))
      type_error(o_term, "O_TERM", "write_o_term_inExp");

    exp->o_term = o_term;
    return exp;
}

Public void delete_exp(exp)
     Exp* exp;
{
    if (exp != NULL)
      if (is_exp_non_NULL(exp)) {
	  delete_exp_name(exp->exp_name);
	  delete_o_term(exp->o_term);
	  dotsrc_free((char*)exp);
      }
      else
	type_error(exp, "EXP", "delete_exp");
}

Public Exp* copy_exp(exp)
     Exp* exp;
{
    if (exp == NULL)
      return NULL;
    else if (is_exp_non_NULL(exp)) {
	Exp* e = (Exp*)dotsrc_malloc(sizeof(Exp));

	e->tag = EXP;
	e->exp_name = copy_exp_name(exp->exp_name);
	e->o_term = copy_o_term(exp->o_term);
	return e;
    }
    else
      type_error(exp, "EXP", "copy_exp");
}

/*
  OBJ_DEF
*/

Public Bool fis_obj_def(object)
     PseudoObject* object;
{
    return object == NULL || object->tag == OBJ_DEF;
}

Public void decompose_obj_def(obj_def, obj_subs)
     ObjDef* obj_def;
     ObjList** obj_subs;
{
    if (obj_def == NULL) {
	*obj_subs = NULL;
    }
    else if (is_obj_def_non_NULL(obj_def)) {
	*obj_subs = obj_def->obj_subs;
    }
    else
      type_error(obj_def, "OBJ_DEF", "decompose_obj_def");
}

Public ObjList* read_obj_subs(obj_def)
     ObjDef* obj_def;
{
    if (obj_def == NULL)
      return NULL;
    else if (is_obj_def_non_NULL(obj_def))
      return obj_def->obj_subs;
    else
      type_error(obj_def, "OBJ_DEF", "read_obj_subs");
}

Public ObjDef* new_obj_def()
{
    ObjDef* obj_def = (ObjDef*)dotsrc_malloc(sizeof(ObjDef));

    obj_def->tag = OBJ_DEF;
    obj_def->obj_subs = NULL;
    return obj_def;
}

Public ObjDef* set_obj_def(obj_def, obj_subs)
     ObjDef* obj_def;
     ObjList* obj_subs;
{
    if (obj_def == NULL)
      write_through_NULL("set_obj_def");
    if (! is_obj_def_non_NULL(obj_def))
      type_error(obj_def, "OBJ_DEF", "set_obj_def");
    if (! is_obj_list(obj_subs, OBJ_SUB))
      type_error(obj_subs, "OBJ_LIST of OBJ_SUB", "set_obj_def");

    obj_def->obj_subs = obj_subs;
    return obj_def;
}

Public ObjDef* create_obj_def(obj_subs)
     ObjList* obj_subs;
{
    if (is_obj_list(obj_subs, OBJ_SUB)) {
	ObjDef* obj_def = (ObjDef*)dotsrc_malloc(sizeof(ObjDef));

	obj_def->tag = OBJ_DEF;
	obj_def->obj_subs = obj_subs;
	return obj_def;
    }
    else
      type_error(obj_subs, "OBJ_LIST of OBJ_SUB", "create_obj_def");
}

Public ObjDef* write_obj_subs(obj_def, obj_subs)
     ObjDef* obj_def;
     ObjList* obj_subs;
{
    if (obj_def == NULL)
      write_through_NULL("write_obj_subs");
    else if (is_obj_list(obj_subs, OBJ_SUB)) {
	obj_def->obj_subs = obj_subs;
	return obj_def;
    }
    else
      type_error(obj_def, "OBJ_DEF", "write_obj_subs");
}

Public ObjDef* insert_obj_subs(obj_def, obj_subs)
     ObjDef* obj_def;
     ObjList* obj_subs;
{
    if (obj_def == NULL)
      write_through_NULL("insert_obj_subs");
    else if (is_obj_def_non_NULL(obj_def))
      if (obj_def->obj_subs == NULL) {
	  obj_def->obj_subs = obj_subs;
	  return obj_def;
      }
      else {
	  insert_list_to_obj_list(obj_def->obj_subs, obj_subs);
	  return obj_def;
      }
}

Public ObjDef* add_obj_subs(obj_def, obj_subs)
     ObjDef* obj_def;
     ObjList* obj_subs;
{
    if (obj_def == NULL)
      write_through_NULL("add_obj_subs");
    else if (is_obj_def_non_NULL(obj_def))
      if (obj_def->obj_subs == NULL) {
	  obj_def->obj_subs = obj_subs;
      }
      else {
	  concat_list_to_obj_list(obj_def->obj_subs, obj_subs);
	  return obj_def;
      }
    else
      type_error(obj_def, "OBJ_DEF", "add_obj_subs");
}

Public ObjSub* car_of_obj_subs(obj_def)
     ObjDef* obj_def;
{
    if (obj_def == NULL || obj_def->obj_subs == NULL)
      return NULL;
    else
      return (ObjSub*)extract_first_list_element(obj_def->obj_subs);
}

Public void delete_obj_def(obj_def)
     ObjDef* obj_def;
{
    if (obj_def != NULL)
      if (is_obj_def_non_NULL(obj_def)) {
	  delete_obj_list(obj_def->obj_subs);
	  dotsrc_free(obj_def);
      }
      else
	type_error(obj_def, "OBJ_DEF", "delete_obj_def");
}

Public ObjDef* copy_obj_def(obj_def)
     ObjDef* obj_def;
{
    if (obj_def == NULL)
      return NULL;
    else if (is_obj_def_non_NULL(obj_def)) {
	ObjDef* o = (ObjDef*)dotsrc_malloc(sizeof(ObjDef));

	o->tag = OBJ_DEF;
	o->obj_subs = copy_obj_list(obj_def->obj_subs);
	return o;
    }
    else
      type_error(obj_def, "OBJ_DEF", "copy_obj_def");
}

/*
  OBJ_SUB
*/

Public Bool fis_obj_sub(object)
     PseudoObject* object;
{
    return object == NULL || object->tag == OBJ_SUB;
}

Public void decompose_obj_sub(obj_sub, bobj1, bobj2)
     ObjSub* obj_sub;
     char **bobj1, **bobj2;
{
    if (obj_sub == NULL) {
	*bobj1 = NULL;
	*bobj2 = NULL;
    }
    else if (is_obj_sub_non_NULL(obj_sub)) {
	*bobj1 = obj_sub->bobj1;
	*bobj2 = obj_sub->bobj2;
    }
    else
      type_error(obj_sub, "OBJ_SUB", "decompose_obj_sub");
}

Public char* read_bobj1(obj_sub)
     ObjSub* obj_sub;
{
    if (obj_sub == NULL)
      return NULL;
    else
      return obj_sub->bobj1;
}

Public char* read_bobj2(obj_sub)
     ObjSub* obj_sub;
{
    if (obj_sub == NULL)
      return NULL;
    else
      return obj_sub->bobj2;
}

Public ObjSub* new_obj_sub()
{
    ObjSub* obj_sub = (ObjSub*)dotsrc_malloc(sizeof(ObjSub));

    obj_sub->tag = OBJ_SUB;
    obj_sub->bobj1 = NULL;
    obj_sub->bobj2 = NULL;
    return obj_sub;
}

Public ObjSub* set_obj_sub(obj_sub, bobj1, bobj2)
     ObjSub* obj_sub;
     char *bobj1, *bobj2;
{
    if (obj_sub == NULL)
      write_through_NULL("set_obj_sub");
    if (! is_obj_sub_non_NULL(obj_sub))
      type_error(obj_sub, "OBJ_SUB", "set_obj_sub");

    obj_sub->bobj1 = bobj1;
    obj_sub->bobj2 = bobj2;
    return obj_sub;
}

Public ObjSub* create_obj_sub(bobj1, bobj2)
     char *bobj1, *bobj2;
{
    ObjSub* obj_sub = (ObjSub*)dotsrc_malloc(sizeof(ObjSub));

    obj_sub->tag = OBJ_SUB;
    obj_sub->bobj1 = bobj1;
    obj_sub->bobj2 = bobj2;
    return obj_sub;
}

Public ObjSub* write_bobj1(obj_sub, bobj1)
     ObjSub* obj_sub;
     char* bobj1;
{
    if (obj_sub == NULL)
      write_through_NULL("write_bobj1");
    else if (is_obj_sub(obj_sub)) {
	obj_sub->bobj1 = bobj1;
	return obj_sub;
    }
    else
      type_error(obj_sub, "OBJ_SUB", "write_bobj1");
}

Public ObjSub* write_bobj2(obj_sub, bobj2)
     ObjSub* obj_sub;
     char* bobj2;
{
    if (obj_sub == NULL)
      write_through_NULL("write_bobj2");
    else if (is_obj_sub(obj_sub)) {
	obj_sub->bobj2 = bobj2;
	return obj_sub;
    }
    else
      type_error(obj_sub, "OBJ_SUB", "write_bobj2");
}

Public void delete_obj_sub(obj_sub)
     ObjSub* obj_sub;
{
    if (obj_sub != NULL)
      if (is_obj_sub(obj_sub)) {
	  if (obj_sub->bobj1 != NULL)
	    dotsrc_free(obj_sub->bobj1);
	  if (obj_sub->bobj2 != NULL)
	    dotsrc_free(obj_sub->bobj2);
	  dotsrc_free((char*)obj_sub);
      }
      else
	type_error(obj_sub, "OBJ_SUB", "delete_obj_sub");
}

Public ObjSub* copy_obj_sub(obj_sub)
     ObjSub *obj_sub;
{
    if (obj_sub == NULL)
      return NULL;
    else if (is_obj_sub_non_NULL(obj_sub)) {
	ObjSub* o = (ObjSub*)dotsrc_malloc(sizeof(ObjSub));

	o->tag = OBJ_SUB;
	o->bobj1 = dotsrc_strdup(obj_sub->bobj1);
	o->bobj2 = dotsrc_strdup(obj_sub->bobj2);
	return o;
    }
    else
      type_error(obj_sub, "OBJ_SUB", "copy_obj_sub");
}

/*
  OBJ_LIST
*/

/*
  fis_obj_list:
  function version of is_obj_list which checks if the object is an
OBJ_LIST and td is one of sub-types of whose element_tag.
*/

Public Bool fis_obj_list(object, td)
     PseudoObject* object;
     TypeDescriptor td;
{
    return object == NULL
           || (object->tag == OBJ_LIST
	       && is_type(td, ((ObjList*)object)->element_tag));
}

/*
  fis_any_obj_list:
  function version of is_any_obj_list which checks if the object is an
OBJ_LIST.
*/

Public Bool fis_any_obj_list(object)
     PseudoObject* object;
{
    return object == NULL || object->tag == OBJ_LIST;
}

/*
  fis_obj_list_non_NULL:
  function version of is_obj_list_non_NULL which is same as
is_obj_list except that is_obj_list_non_NULL don't check if object is
NULL pointer.
*/

Bool fis_obj_list_non_NULL(object, td)
     PseudoObject* object;
     TypeDescriptor td;
{
    return object->tag == OBJ_LIST
           && is_type(td, ((ObjList*)object)->element_tag);
}

/*
  new_obj_list:
  create new empty OBJ_LIST.
*/

Public ObjList* new_obj_list(td)
     TypeDescriptor td;
{
    ObjList* obj_list = (ObjList*)dotsrc_malloc(sizeof(ObjList));

    obj_list->tag = OBJ_LIST;
    obj_list->element_tag = td;
    obj_list->first = NULL;
    obj_list->last = NULL;
    return obj_list;
}

/*
  extract_first_list_element:
  remove first element from obj_list and return it.
*/

Public PseudoObject* extract_first_list_element(obj_list)
     ObjList* obj_list;
{
    if (obj_list == NULL)
      return NULL;
    else if (is_any_obj_list_non_NULL(obj_list)) {
	ObjElement* tmp = obj_list->first;

	if (tmp == NULL)
	  return NULL;
	else {
	    PseudoObject* data = tmp->element;
	    if ((obj_list->first = tmp->next) == NULL)
	      obj_list->last = NULL;
	    dotsrc_free((char*)tmp);
	    return data;
	}
    }
    else
      type_error(obj_list, "OBJ_LIST", "extract_first_list_element");
}

/*
  extract_last_list_element:
  remove last element from obj_list and return it
  CAUTION: extremely low speed
*/

Public PseudoObject* extract_last_list_element(obj_list)
     ObjList* obj_list;
{
    if (obj_list == NULL)
      return NULL;
    else if (is_any_obj_list_non_NULL(obj_list)) {
	ObjElement* tmp = obj_list->first;

	if (tmp == NULL)
	  return NULL;
	else if (tmp->next == NULL) {
	    PseudoObject* data = tmp->element;

	    obj_list->first = obj_list->last = NULL;
	    dotsrc_free((char*)tmp);
	    return data;
	}
	else {
	    PseudoObject* data = obj_list->last->element;

	    while (tmp->next != obj_list->last)
	      tmp = tmp->next;
	    dotsrc_free((char*)obj_list->last);
	    tmp->next = NULL;
	    obj_list->last = tmp;
	    return data;
	}
    }
    else
      type_error(obj_list, "OBJ_LIST", "extract_last_list_element");
}

/*
  cons_to_obj_list:
  insert a pseudo object into the top of an obj_list.
*/

Public ObjList* cons_to_obj_list(obj_list, object)
     ObjList* obj_list;
     PseudoObject* object;
{
    if (obj_list == NULL)
      write_through_NULL("cons_to_obj_list");
    else if (is_any_obj_list_non_NULL(obj_list))
      if (is_type(object->tag, obj_list->element_tag)) {
	  ObjElement* tmp = (ObjElement*)dotsrc_malloc(sizeof(ObjElement));

	  tmp->element = object;
	  tmp->next = obj_list->first;
	  obj_list->first = tmp;
	  if (obj_list->last == NULL)
	    obj_list->last = obj_list->first;
	  return obj_list;
      }
      else
	type_error(object, "type in OBJ_LIST", "cons_to_obj_list");
    else
      type_error(obj_list, "OBJ_LIST", "cons_to_obj_list");
}

/*
  append_to_obj_list:
  append a pseudo object after the tail of an obj_list.
*/

Public ObjList* append_to_obj_list(obj_list, object)
     ObjList* obj_list;
     PseudoObject* object;
{
    if (obj_list == NULL)
      write_through_NULL("append_to_obj_list");
    else if (is_any_obj_list_non_NULL(obj_list))
      if (is_type(object->tag, obj_list->element_tag)) {
	  ObjElement* tmp = (ObjElement*)dotsrc_malloc(sizeof(ObjElement));

	  tmp->element = object;
	  tmp->next = NULL;
	  if (obj_list->last == NULL) {
	      obj_list->first = tmp;
	      obj_list->last = tmp;
	  }
	  else {
	      obj_list->last->next = tmp;
	      obj_list->last = tmp;
	  }
	  return obj_list;
      }
      else
	type_error(object, "type in OBJ_LIST", "append_to_obj_list");
    else
      type_error(obj_list, "OBJ_LIST", "append_to_obj_list");
}

/*
  insert_list_to_obj_list:
  insert an OBJ_LIST into the top of another OBJ_LIST.
  former OBJ_LIST is eliminated.
*/

Public ObjList* insert_list_to_obj_list(obj_list1, obj_list2)
     ObjList *obj_list1, *obj_list2;
{
    if (obj_list1 == NULL)
      write_through_NULL("insert_list_to_obj_list");
    else if (obj_list2 == NULL) {
	dotsrc_free((char*)obj_list2);
	return obj_list1;
    }
    else if (is_type(obj_list2->tag, obj_list1->tag))
      if (obj_list2->first == NULL) {
	  dotsrc_free((char*)obj_list2);
	  return obj_list1;
      }
      else {
	  ObjElement *tmp = obj_list1->first;

	  obj_list1->first = obj_list2->first;
	  obj_list2->last->next = tmp;
	  if (obj_list1->last == NULL)
	    obj_list1->last = obj_list2->last;
	  dotsrc_free((char*)obj_list2);
	  return obj_list1;
      }
    else
      type_conflict(obj_list1->element_tag,
		    obj_list2->element_tag, "insert_list_to_obj_list");
}

/*
  concat_list_to_obj_list:
  concatenate an OBJ_LIST after the tail of another OBJ_LIST.
  former OBJ_LIST is eliminated.
*/

Public ObjList* concat_list_to_obj_list(obj_list1, obj_list2)
     ObjList *obj_list1, *obj_list2;
{
    if (obj_list1 == NULL)
      write_through_NULL("concat_list_to_obj_list");
    else if (obj_list2 == NULL) {
	dotsrc_free((char*)obj_list2);
	return obj_list1;
    }
    else if (is_type(obj_list2->tag, obj_list1->tag))
      if (obj_list2->first == NULL) {
	  dotsrc_free((char*)obj_list2);
	  return obj_list1;
      }
      else if (obj_list1->first == NULL) {
	  obj_list1->first = obj_list2->first;
	  obj_list1->last = obj_list2->last;
	  dotsrc_free((char*)obj_list2);
	  return obj_list1;
      }
      else {
	  ObjElement *tmp = obj_list1->last;

	  obj_list1->last->next = obj_list2->first;
	  obj_list1->last = obj_list2->last;
	  dotsrc_free((char*)obj_list2);
	  return obj_list1;
      }
    else
      type_conflict(obj_list1->element_tag,
		    obj_list2->element_tag, "concat_list_to_obj_list");
}

/*
  delete_obj_list:
  eliminate an OBJ_LIST with its all elements.
*/

Public void delete_obj_list(obj_list)
     ObjList* obj_list;
{
    if (obj_list != NULL) {
	ObjElement* tmp = obj_list->first;

	while (tmp != NULL) {
	    ObjElement* tmp2 = tmp->next;

	    delete_pseudo_object(tmp->element);
	    dotsrc_free(tmp);
	    tmp = tmp2;
	}
	dotsrc_free(obj_list);
    }
}

/*
  copy_obj_list:
  make a copy of an OBJ_LIST with its all elements and return it.
*/

Public ObjList* copy_obj_list(obj_list)
     ObjList* obj_list;
{
    if (obj_list == NULL)
      return NULL;
    else {
	ObjList* o = new_obj_list(obj_list->element_tag);
	ObjElement* p = obj_list->first;
	ObjElement**q = &o->first;
	ObjElement* tmp = NULL;

	while (p != NULL) {
	    *q = tmp = (ObjElement*)dotsrc_malloc(sizeof(ObjElement));
	    tmp->element = copy_pseudo_object(p->element);
	    q = &tmp->next;
	    p = p->next;
	}
	*q = NULL;
	o->last = tmp;
	return o;
    }
}

/*
  OBJ_ARRAY
*/

/*
  fis_obj_array:
  function version of is_obj_array which checks if the object is an
OBJ_ARRAY and td is one of sub-types of whose element_tag.
*/

Public Bool fis_obj_array(object, td)
     PseudoObject* object;
     TypeDescriptor td;
{
    return object == NULL
           || (object->tag == OBJ_ARRAY
	       && is_type(td, ((ObjArray*)object)->element_tag));
}

/*
  fis_any_obj_array:
  function version of is_any_obj_list which checks if the object ia an
OBJ_ARRAY.
*/

Public Bool fis_any_obj_array(object)
     PseudoObject* object;
{
    return object == NULL || object->tag == OBJ_ARRAY;
}

/*
  out_of_range:
  put index-out-of-range-for-array error message to stderr
*/

Local void out_of_range(obj_array, index, func_name)
     ObjArray* obj_array;
     int index;
     char* func_name;
{
    fprintf(stderr, "out of OBJ_ARRAY index %d in function %s\n",
	    index, func_name);
    exit(2);
}

/*
  fis_obj_array_non_NULL:
  function version of is_obj_array_non_NULL which is same as
is_obj_array except that is_obj_array_non_NULL don't check if object is
NULL pointer.
*/

Public Bool fis_obj_array_non_NULL(object, td)
     PseudoObject* object;
     TypeDescriptor td;
{
    return object->tag == OBJ_LIST
           && is_type(td, ((ObjArray*)object)->element_tag);
}

/*
  new_obj_array:
  create new empty n-sized OBJ_ARRAY.
*/

Public ObjArray* new_obj_array(td, n)
     TypeDescriptor td;
     size_t n;
{
    ObjArray* obj_array = (ObjArray*)dotsrc_malloc(sizeof(ObjArray));

    obj_array->tag = OBJ_ARRAY;
    obj_array->element_tag = td;
    obj_array->size = n;
    obj_array->elements
      = (PseudoObject**)dotsrc_calloc(n, sizeof(PseudoObject*));
    return obj_array;
}

/*
  compose_obj_array:
  create new OBJ_ARRAY and initilize it by an array of pseudo object.
  setting pseudo object array is done by pointer copy (not entity
  copy).
*/

Public ObjArray* compose_obj_array(td, n, array)
     TypeDescriptor td;
     int n;
     PseudoObject** array;
{
    ObjArray *obj_array;

    array_type_check(td, n, array);
    obj_array = (ObjArray*)dotsrc_malloc(sizeof(ObjArray));
    obj_array->tag = OBJ_ARRAY;
    obj_array->element_tag = td;
    obj_array->size = n;
    obj_array->elements = array;
    return obj_array;
}

/*
  new_empty_obj_array:
  create new empty OBJ_ARRAY
*/

Public ObjArray* new_empty_obj_array()
{
    ObjArray* obj_array = (ObjArray*)dotsrc_malloc(sizeof(ObjArray));

    obj_array->tag = OBJ_ARRAY;
    obj_array->element_tag = NULL;
    obj_array->size = 0;
    obj_array->elements = NULL;
    return obj_array;
}

/*
  array_size:
  return the size of an array.
*/

Public size_t array_size(obj_array)
     ObjArray* obj_array;
{
    if (obj_array == NULL)
      return 0;
    else if (is_any_obj_array(obj_array))
      return obj_array->size;
    else
      type_error(obj_array, "OBJ_ARRAY", "array_size");
}

/*
  change_array_size:
  change the size of an array
*/

Public ObjArray* change_array_size(obj_array, start, new_size)
     ObjArray* obj_array;
     int start, new_size;
{
    if (obj_array == NULL)
      write_through_NULL("change_array_size");
    else if (is_any_obj_array(obj_array))
      if (start == 0 && new_size <= obj_array->size) {
	  obj_array->size = new_size;
	  return obj_array;
      }
      else {
	  PseudoObject** new_array
	    = (PseudoObject**)dotsrc_malloc(sizeof(PseudoObject*));
	  int i, j;

	  for (i = start, j = 0; i < 0 && j < new_size; ++i, ++j) {
	      new_array[j] = NULL;
	  }
	  for (; i < obj_array->size && j < new_size; ++i, ++j) {
	      new_array[j] = obj_array->elements[i];
	  }
	  for (; j < new_size; ++j) {
	      new_array[j] = NULL;
	  }
	  obj_array->size = new_size;
	  obj_array->elements = new_array;
	  return obj_array;
      }
    else
      type_error(obj_array, "OBJ_ARRAY", "change_array_size");
}

/*
  change_array_size_deleting:
  change the size of an array with eliminating pseudo objects which
  turn to be not ablt to access through the OBJ_ARRAY.
*/

Public ObjArray* change_array_size_deleting(obj_array, start,
					    new_size)
     ObjArray* obj_array;
     int start, new_size;
{
    if (obj_array == NULL)
      write_through_NULL("change_array_size_deleting");
    else if (is_any_obj_array(obj_array))
      if (start == 0 && new_size <= obj_array->size) {
	  obj_array->size = new_size;
	  return obj_array;
      }
      else {
	  PseudoObject** new_array
	    = (PseudoObject**)dotsrc_malloc(sizeof(PseudoObject*));
	  int i, j;

	  for (i = 0; i < start; ++i) {
	      delete_pseudo_object(obj_array->elements[i]);
	  }
	  for (i = start, j = 0; i < 0 && j < new_size; ++i, ++j) {
	      new_array[j] = NULL;
	  }
	  for (; i < obj_array->size && j < new_size; ++i, ++j) {
	      new_array[j] = obj_array->elements[i];
	  }
	  for (; j < new_size; ++j) {
	      new_array[j] = NULL;
	  }
	  for (; i < obj_array->size; ++i) {
	      delete_pseudo_object(obj_array->elements[i]);
	  }
	  obj_array -> size = new_size;
	  obj_array -> elements = new_array;
	  return obj_array;
      }
    else
      type_error(obj_array, "OBJ_ARRAY", "change_array_size_deleting");
}

/*
  read_array_element:
  read an element of OBJ_ARRAY by an index.
*/

Public PseudoObject* read_array_element(obj_array, n)
     ObjArray* obj_array;
     int n;
{
    if (obj_array == NULL || n < 0 || n >= obj_array->size) {
	out_of_range(obj_array, n, "read_array_element");
    }
    else
      return obj_array->elements[n];
}

/*
  set_array_element:
  set an element of OBJ_ARRAY by an index
*/

Public ObjArray* set_array_element(obj_array, n, object)
     ObjArray* obj_array;
     int n;
     PseudoObject* object;
{
    if (obj_array == NULL)
      write_through_NULL("set_array_element");
    else if (n < 0 || n >= obj_array->size)
      out_of_range(obj_array, n, "set_array_element");
    else if (is_type(object->tag, obj_array->element_tag)) {
	obj_array->elements[n] = object;
	return obj_array;
    }
    else
      type_error(object, "type in OBJ_ARRAY", "set_array_element");
}

/*
  set_array_element_deleting:
  set an element of OBJ_ARRAY by index and eliminate the old one at
same location on the array.
*/

Public ObjArray* set_array_element_deleting(obj_array, n, object)
     ObjArray* obj_array;
     int n;
     PseudoObject* object;
{
    if (obj_array == NULL)
      write_through_NULL("set_array_element_deleting");
    else if (n < 0 || n >= obj_array->size)
      out_of_range(obj_array, n, "set_array_element_deleting");
    else if (is_type(object->tag, obj_array->element_tag)) {
	delete_pseudo_object(obj_array->elements[n]),
	obj_array->elements[n] = object;
	return obj_array;
    }
    else
      type_error(object,"type in OBJ_ARRAY", "set_array_element_deleting");
}

/*
  set_array_element_and_return_old:
  set an element of OBJ_ARRAY by index and return the old one at same
location on the array.
*/

Public PseudoObject* set_array_element_and_return_old(obj_array, n,
						      object)
     ObjArray* obj_array;
     int n;
     PseudoObject* object;
{
    if (obj_array == NULL)
      write_through_NULL("set_array_element_and_return_old");
    else if (n < 0 || n >= obj_array->size)
      out_of_range(obj_array, n, "set_array_element_and_return_old");
    else if (is_type(object->tag, obj_array->element_tag)) {
	PseudoObject* tmp = obj_array->elements[n];
	obj_array->elements[n] = object;
	return tmp;
    }
    else
      type_error(object, "type in OBJ_ARRAY",
		 "set_array_element_and_return_old");
}

/*
  delete_obj_array:
  eliminate an OBJ_ARRAY with its all elements.
*/

Public void delete_obj_array(obj_array)
     ObjArray* obj_array;
{
    if (obj_array != NULL) {
	int i;

	for (i = 0; i < obj_array->size; ++i) {
	    delete_pseudo_object(obj_array->elements[i]);
	}
	dotsrc_free(obj_array->elements);
	dotsrc_free(obj_array);
    }
}

/*
  copy_obj_array:
  make a acopy of an OBJ_ARRAY wit its all elements and return it.
*/

Public ObjArray* copy_obj_array(obj_array)
     ObjArray* obj_array;
{
    if (obj_array == NULL)
      return NULL;
    else {
	PseudoObject** elements
	  = (PseudoObject**)dotsrc_calloc(obj_array->size,
					  sizeof(PseudoObject*));
	ObjArray* o = compose_obj_array(obj_array->element_tag,
					obj_array->size, elements);
	int i;

	for (i = 0; i < obj_array->size; ++i)
	  elements[i] = copy_pseudo_object(obj_array->elements[i]);
	return o;
    }
}

/*
  list2array:
  convert OBJ_LIST to OBJ_ARRAY eliminating the OBJ_LIST.
*/

Local int count_size(obj_list)
     ObjList* obj_list;
{
    if (obj_list == NULL)
      return 0;
    else {
	ObjElement *o = obj_list->first;
	int n;

	for (n = 0; o != NULL; o = o->next)
	  ++n ;
	return n;
    }
}

Local PseudoObject** fill_array(obj_list, n)
     ObjList* obj_list;
     int n;
{
    PseudoObject** array
      = (PseudoObject**)dotsrc_calloc(n, sizeof(PseudoObject*));
    ObjElement* o = obj_list->first;
    int i;

    for (i = 0; i < n; ++ i) {
	ObjElement* tmp = o;

	array[i] = o->element;
	o = o->next;
	dotsrc_free((char*)tmp);
    }
    return array;
}

Public ObjArray* list2array(obj_list)
     ObjList* obj_list;
{
    if (obj_list == NULL)
      return NULL;
    else if (is_any_obj_list(obj_list)) {
	ObjArray* obj_array =
	  (ObjArray*)dotsrc_malloc(sizeof(ObjArray));

	obj_array->tag = OBJ_ARRAY;
	obj_array->element_tag = obj_list->element_tag;
	obj_array->size = count_size(obj_list);
	obj_array->elements = fill_array(obj_list, obj_array->size);
	dotsrc_free(obj_list);
	return obj_array;
    }
    else
      type_error(obj_list, "OBJ_LIST", "list2array");
}

/*
  array2list:
  convert OBJ_ARRAY to OBJ_LIST eliminatin the OBJ_ARRAY.
*/

Public ObjList* array2list(obj_array)
     ObjArray* obj_array;
{
    if (obj_array == NULL)
      return NULL;
    else if (is_any_obj_array(obj_array)) {
	ObjList* obj_list = (ObjList*)dotsrc_malloc(sizeof(ObjList));
	ObjElement **o, *p = NULL;
	int i;

	obj_list->tag = OBJ_LIST;
	obj_list->element_tag = obj_array->element_tag;
	o = &obj_list->first;
	for (i = 0; i < obj_array->size; ++ i) {
	    p = *o = (ObjElement*)dotsrc_malloc(sizeof(ObjElement));
	    p->element = obj_array->elements[i];
	    o = &p->next;
	}
	*o = NULL;
	obj_list->last = p;
	dotsrc_free((char*)obj_array);
	return obj_list;
    }
}
