Object subclass: #PSMedium
	instanceVariableNames: 'stream currentFont landscape height width '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Graphics-Postscript'!
PSMedium comment:
'I act as the output medium for PSGraphicsContext.  I play the same role that a DisplaySurface plays for GraphicsContext.

Instance variables:
stream			<WriteStream>		where the PostScript commands go
currentFont		<PSFont>			the current PostScript font'!


!PSMedium methodsFor: 'initialize-release'!

close
	stream close!

on: aStream
	"Set the output stream to aStream."
	height := 792.
	width := 612.
	landscape := false.
	stream := aStream.! !

!PSMedium methodsFor: 'accessing'!

landscape
	"This returns the instance variable"
	^landscape.!

landscape: aBoolean
	"This sets the instance variable"
	^landscape := aBoolean.! !

!PSMedium methodsFor: 'requests'!

background
	^ColorValue white.!

cr
	"Redirect cr  to the stream."

	stream cr.!

flush
	"Flush the stream"

	stream flush!

height
	"Answer the height of the postscript page"
	^height.
	"^landscape ifTrue: [612] ifFalse: [792]."
	" 792 = 11'' x 72 points"!

mapSerif: isSerif fixed: isFixed size: size italic: isItalic boldness: boldness 
	"Map a Smalltalk font family onto a PostScript face, and set the type size."

	| newFamily newFace |
	boldness > 0.5
		ifTrue: [newFace := 'Bold']
		ifFalse: [newFace := ''].
	isItalic ifTrue: [newFace := newFace , 'Italic'].
	isSerif
		ifTrue: [
				"newFace = '' ifTrue: [newFace := #Roman]]"newFamily := #Times]
		ifFalse: [isFixed
				ifTrue: [newFamily := #Courier]
				ifFalse: [newFamily := #Helvetica]].
	^Array with: (newFamily , newFace) asSymbol with: size!

put: aString
	"Put aString on the stream."

	stream nextPutAll: aString.!

putImage: anImage cmd: imageCmd 
	"Put an image on the medium"

	self scale: anImage width asFloat @ anImage height asFloat.
	self
		imageWidth: anImage width
		height: anImage height
		bits: anImage bits
		cmd: imageCmd!

setClippingRectangle: orig height: height width: width 
	"Make a rectangular path at the current origin, and use it as the 
	current clip path. Assume gsave and grestore are done externally. 
	
	Correction: Doesn't work, so do grestore and gsave to reset to the 
	initial configuration."

	
		stream nextPutAll:
"%%initclip"
'grestore gsave
newpath '.
	orig x asFloat printOn: stream.
	stream space.
	orig y asFloat printOn: stream.
	stream nextPutAll: ' moveto 0 '.
	height asFloat printOn: stream.
	stream nextPutAll: ' rlineto '.
	width asFloat printOn: stream.
	stream nextPutAll: ' 0 rlineto 0 '.
	height negated asFloat printOn: stream.
	stream nextPutAll: ' rlineto closepath clip newpath'; cr!

setFont: aFontStyle 
	"Let the medium know of a new font style."

	| newFont name size |
	newFont := self
				mapSerif: aFontStyle serif
				fixed: aFontStyle fixedWidth
				size: (72.0 "points per inch" * aFontStyle pixelSize / Screen default resolution y) rounded
				italic: aFontStyle italic
				boldness: aFontStyle boldness.
	name := newFont at: 1.
	size := newFont at: 2.
	(currentFont isNil or: [currentFont fullName ~= name or: [currentFont charSize ~= size]])
		ifTrue: 
			[currentFont := PSFont new family: (PSFontFamily named: name)
						size: size.
			currentFont charSize printOn: stream.
			stream nextPutAll: ' ('; nextPutAll: currentFont fullName; nextPutAll: ') '.
			stream nextPutAll: ' F'; cr]!

setLandscape
	"Rotate the page 90 degrees. Written by Tony White (arpw@bnr.ca)"
	landscape := true.
	stream nextPutAll: '/orientation landscape def'; cr.
	stream nextPutAll: 'setOrientation'; cr.!

setLimits: aPoint
	"This sets the width and height of the medium"
	height := aPoint y max: 792.
	width := aPoint x max: 612.!

setLineWidth: aNumber
	"Set line width for drawing"

	aNumber printOn: stream.
	stream nextPutAll: ' setlinewidth'; cr.!

show: string 
	"Emit the PostScript code to print a string of characters."

	| index length c code |
	stream nextPut: $(.
	index := 0.
	length := string size.
	[(index := index + 1) <= length]
		whileTrue: 
			[c := string at: index.
			c == $(
				ifTrue: [stream nextPutAll: '\050']
				ifFalse: [c == $)
						ifTrue: [stream nextPutAll: '\051']
						ifFalse: [c == $\
								ifTrue: [stream nextPutAll: '\\']
								ifFalse: [c asInteger > 127
									ifTrue:  
										[stream nextPutAll: ') show /'.
										code := currentFont encode: c.
										code = 65535
											ifTrue:	[code := 'space'].
										stream nextPutAll: code.
										stream nextPutAll: ' ISOshow ('] 
									ifFalse: [stream nextPut: c]]]]].
	stream nextPutAll: ') newShow'; cr!

space
	"Redirect space to the stream."

	stream space.!

translate: aPoint 
	"Translate PostScript coordinate system by aPoint"

	aPoint x printOn: stream.
	stream space.
	aPoint y printOn: stream.
	stream nextPutAll: ' translate'; cr!

width
	"Answer the width of the postscript page"
	^width.
	"^landscape ifTrue: [792] ifFalse: [612]."
	" 612 = 8.5'' x 72 points"! !

!PSMedium methodsFor: 'display surface compatibility'!

canBePaintedWith: aPaint
	"Answer whether aPaint is appropriate for me."
	^true! !

!PSMedium methodsFor: 'private'!

currentTransformation: a b: b c: c d: d tx: e ty: f 
	"Output the PostScript code to set the current graphics coordinate 
	transformation matrix."

	stream nextPut: $[.
	a asFloat printOn: stream.
	stream space.
	b asFloat printOn: stream.
	stream space.
	c asFloat printOn: stream.
	stream space.
	d asFloat printOn: stream.
	stream space.
	e asFloat printOn: stream.
	stream space.
	f asFloat printOn: stream.
	stream nextPutAll: ']'; cr!

imageWidth: width height: height bits: bits cmd: imageCmd
	"Emit the PostScript code needed to print a bitmap...  any needed 
	scaling,  translation,  etc.  is done externally."

	| hex byte bytesPerLine stride |
	hex := 'fedcba9876543210'.
	stream nextPutAll: ' /pix < '.
	stride := width + 31 // 32 * 4.	"Smalltalk image stride is 32 bits"
	bytesPerLine := width + 7 // 8.  "PostScript file stride is a byte"
	0 to: height - 1 do:
		[:y |
		0 to: bytesPerLine-1 do:
			[:x |
			byte := bits at: (y*stride + x) + 1.
			stream nextPut: (hex at: (byte bitShift: -4) + 1).
			stream nextPut: (hex at: (byte bitAnd: 15) + 1).].
		stream space].
	stream nextPutAll: '> def'; cr.
	width printOn: stream.
	stream space.
	height printOn: stream.
	imageCmd == #mask ifTrue: [ stream nextPutAll: ' false ' ] ifFalse: [stream nextPutAll: ' 1 '].
	self
		currentTransformation: width
		b: 0
		c: 0
		d: height "negated"
		tx: 0
		ty: 0 "height".
	stream nextPutAll: ' { pix } image'.
	imageCmd == #mask ifTrue: [stream nextPutAll: 'mask'].
	stream cr.!

scale: aPoint 
	"Emit the PostScript code to scale the coordinate system...  'aPoint' is 
	not really a point,  but rather the x and y scale factors."

	aPoint x asFloat printOn: stream.
	stream space.
	aPoint y asFloat printOn: stream.
	stream nextPutAll: ' scale'; cr! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

PSMedium class
	instanceVariableNames: ''!


!PSMedium class methodsFor: 'instance creation'!

on: aStream
	"Answer a PSMedium with output to aStream."

	^self new on: aStream! !

!PSMedium class methodsFor: 'requests'!

height
	"Answer the height of the postscript page"

	^792.   " 11'' x 72 points"!

width
	"Answer the width of the postscript page"

	^612.   " 8.5'' x 72 points"! !

Object subclass: #PSFont
	instanceVariableNames: 'family size '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Graphics-Postscript'!
PSFont comment:
'PSFont describes a font family, scaled to a particular size.

Instance Variables
	family	<PSFontFamily> Encodes the family, the character widths, and the encoding of the font
	size		<Integer> The height of an average upper case character'!


!PSFont methodsFor: 'initialize-release'!

family: fontFamily size: charSize

	family := fontFamily.
	size := charSize! !

!PSFont methodsFor: 'accessing'!

charSize

	^size!

encode: aCharacter

	^family encode: aCharacter!

fullName

	^family fullName!

underlinePosition

	^family underlinePosition * size!

underlineThickness

	^family underlineThickness * size!

widthOf: aCharacter

	^(family widths at: (self encode: aCharacter) ifAbsent: [family widths at: #space]) * size! !

Object subclass: #PSFontFamily
	instanceVariableNames: 'fullName widths underlinePosition underlineThickness '
	classVariableNames: 'Encodings PSFonts '
	poolDictionaries: ''
	category: 'Graphics-Postscript'!
PSFontFamily comment:
'I hold useful information from the font metric ("afm") files.  My metaclass contains code to parse the afm file and build instances.

Instance Variables:
	fullName			<String>				The name from the afm file.
	widths				<Dictionary>			The width info from  afm file -- assumes 1000 points in coordinate system.
	underlinePosition	<Number>			From the afm file.
	underlineThickness	<Number>			From the afm file.

Class Variables:
	Encodings			<Dictionary>			Maps character values to PostScript names for those characters
											Each character may have multiple names if different fonts refer to
											it by different names.
	PSFonts				<Dictionary>			Holds the font declarations.
'!


!PSFontFamily methodsFor: 'accessing'!

encode: aCharacter

	| names |
	names := Encodings at: aCharacter asInteger ifAbsent: [^65535].
	1 to: names size do:
		[:i |
		(widths includesKey: (names at: i))
			ifTrue:	[^names at: i]].
	^65535!

fullName
	"return the full name of the current font."

	^fullName!

fullName: string 
	"Set the current full name of the current font."

	fullName := string!

underlinePosition
	"Return where the underline goes in the current font."

	^underlinePosition!

underlinePosition: number 
	"Set where the underline goes in the current font."

	underlinePosition := number!

underlineThickness
	"Return how thick an underline is done in the current font."

	^underlineThickness!

underlineThickness: number 
	"Set how thick the underline should be in the current font.."

	underlineThickness := number!

widths
	"Return an array of the widths of all the characters in the current  
	font."

	^widths!

widths: array 
	"Set the widths of all the characters in the current font from the 
	contents of the given array."

	widths := array! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

PSFontFamily class
	instanceVariableNames: ''!
PSFontFamily class comment:
'Select the browser switch "instance" to see the comment'!


!PSFontFamily class methodsFor: 'class initialization'!

initialize
	"Process the default fonts for the LaserWrite-Plus."
	"PSFontFamily initialize."

	| dirList directoryNameString directoryName | 
	PSFonts := IdentityDictionary new: 50.
	[directoryNameString := self getDirName.
	directoryNameString isEmpty or: 
		[directoryName := directoryNameString asFilename.
		(Filename errorReporter inaccessibleSignal
				handle: [:ex | ex returnWith: true]
				do: [directoryName isDirectory not])
			or: [(dirList := directoryName filesMatching: '*AFM') isEmpty]]]
		whileTrue: 
			[(DialogView confirm: '"', directoryName asString, '" is missing, empty or not a directory.
				Try again?') ifFalse: [^self]].
	self processAdobeFiles: dirList.
	self processEncodings: (directoryName construct: 'PScodes')! !

!PSFontFamily class methodsFor: 'initialization'!

fixNameInternal: string 
	"Massage the font name,  getting rid of spaces.  Also,  map 'Oblique' 
	to 'Italic',  'medium' to '',  'Light' to '',  'Demi' to 'Bold',  'Roman'   
	to '',  for internal use... this is necessary to deal with some of the   
	Adobe special naming choices."

	| new foundIndex1 foundIndex2 c |
	foundIndex1 := string findString: 'Book' startingAt: 1.
	foundIndex2 := string findString: 'Bookman' startingAt: 1.
	foundIndex2 == 0 ifTrue: [foundIndex1 == 0 ifFalse: [string
				replaceFrom: foundIndex1
				to: foundIndex1 + 3
				with: '    ']].
	foundIndex1 := string findString: 'Demi' startingAt: 1.
	foundIndex1 == 0 ifFalse: [string
			replaceFrom: foundIndex1
			to: foundIndex1 + 3
			with: 'Bold'].
	foundIndex1 := string findString: 'Oblique' startingAt: 1.
	foundIndex1 == 0 ifFalse: [string
			replaceFrom: foundIndex1
			to: foundIndex1 + 6
			with: 'Italic '].
	foundIndex1 := string findString: 'Light' startingAt: 1.
	foundIndex1 == 0 ifFalse: [string
			replaceFrom: foundIndex1
			to: foundIndex1 + 4
			with: '     '].
	foundIndex1 := string findString: 'Medium' startingAt: 1.
	foundIndex1 == 0 ifFalse: [string
			replaceFrom: foundIndex1
			to: foundIndex1 + 5
			with: '      '].
	foundIndex1 := string findString: 'Roman' startingAt: 1.
	foundIndex1 == 0 ifFalse: [string
			replaceFrom: foundIndex1
			to: foundIndex1 + 4
			with: '     '].
	new := WriteStream on: (String new: 1).
	1 to: string size do: 
		[:i | 
		c := string at: i.
		c == 32 asCharacter ifFalse: [new nextPut: c]].
	^new contents!

getDirName
	"Prompt for and return the base directory name of the font metric files."

	^DialogView
		request: 'Where are the Font Metric files?'
		initialAnswer: ((Filename currentDirectory construct: 'utils')
							construct: 'FontMets') asString!

processAdobeFiles: aCollection
	"Load in the Adobe Font Metric (AFM) files,  and massage them into a 
	set of internal font descriptions..."

	| key f s i c width widths  named desc underlineThickness 
		underlinePosition  psName charName |
	aCollection
		do: 
			[:aName | 
			Transcript cr; show: 'Processing ' , aName.
			key := aName.
			widths := IdentityDictionary new: 300.
			f := aName asFilename.
			s := f contentsOfEntireFile.
			named := self extractFieldNamed: 'FullName' from: s.
			psName := self extractFieldNamed: 'FontName' from: s.
			key := (self fixNameInternal: named) asSymbol.
			underlineThickness := Number readFrom: (ReadStream on: (self 
extractFieldNamed: 'UnderlineThickness' from: s)).
			underlinePosition := Number readFrom: (ReadStream on: (self 
extractFieldNamed: 'UnderlinePosition' from: s)).
			i := s indexOfSubCollection: 'StartCharMetrics' startingAt: 1.
			i := s indexOfSubCollection: 'C ' startingAt: i.
			f := s readStream.
			f position: i.

			[f next = Character space ifFalse: [self afmReadError].
			[f next = $W] whileFalse.
			f next = $X ifFalse: [self afmReadError].
			f next = Character space ifFalse: [self afmReadError].
			width := Integer readFrom: f.
			[f next = $N] whileFalse.
			f next = Character space ifFalse: [self afmReadError].
			charName := (f upTo: Character space) asSymbol.
			widths at: charName put: width.
			f skipThrough: 13 asCharacter.
			c := f next.
			c = $C] whileTrue.
			f close.
			desc := (self new) fullName: psName;
						widths: widths; 
						underlineThickness: underlineThickness; 
						underlinePosition: underlinePosition.
			PSFonts at: key put: desc].
	^PSFonts!

processEncodings: fName

	| codes codeStream |
	codes := fName asFilename readStream fileIn.
	(codes class == Array and: [codes size even])
		ifFalse:	[self error: 'Character code file may be bad'].
	codeStream := codes readStream.
	Encodings := IdentityDictionary new.
	[codeStream atEnd]
		whileFalse:
			[| charName charVal |
			charName := codeStream next.
			charVal := codeStream next.
			(charName isString and: [charVal isInteger])
				ifTrue:	[charName := charName asSymbol.
						Encodings at: charVal
								put: ((Encodings
										at: charVal
										ifAbsent: [#()])
									copyWith: charName)]
				ifFalse:	[self error: 'Character code file may be bad']].! !

!PSFontFamily class methodsFor: 'accessing'!

named: aSymbol 
	"Return the font description corresponding to the given name."

	^PSFonts at: aSymbol! !

!PSFontFamily class methodsFor: 'private'!

afmReadError
	self error: 'Can''t parse font metrics file'!

extractFieldNamed: field from: string 
	"Find a named field in a string,  and return it."

	| i result |
	i := string indexOfSubCollection: field startingAt: 1.
	i := string indexOfSubCollection: ' ' startingAt: i.
	result := string copyFrom: i + 1 to: (string indexOfSubCollection: '
' startingAt: i)
					- 1.
	^result! !

GraphicsContext subclass: #PSGraphicsContext
	instanceVariableNames: 'currentFontStyle pageOffset scale '
	classVariableNames: 'CurrentContext PageOffset '
	poolDictionaries: ''
	category: 'Graphics-Postscript'!
PSGraphicsContext comment:
'I output PostScript commands representing the graphical objects displayed on me to a file (via a PSMedium).

There are several bugs. The worst is that the printed output is sometimes improperly clipped, which probably happens because the bounding boxes are incorrect. Also, if you print the System Workspace, you will see that some strings are mirror images of what they should be, and that extraneous spaces appear. (These are probably all simple Postscript problems, but I don''t know Postscript. -- Carl McConnell 7-21-93). 

If you make a copy of me, change my characteristics (i.e. color or font or offset or clipping bounds), and then go back to the original graphics context, I must generate Postscript to undo your change or the picture will be messed up. (An example of a place where this occurs is the method displayFromCharacter:to:startX:forTranslation:on: in ComposedText. ) To avoid this kind of problem, we keep the graphics context previously used in a class variable. At the beginning of each public display method, I check to see if I am the previously used graphics context. If not, I send my state information like line width, paint color, font, and clipping bounds to the medium to ensure the printer is in the proper state, and then store myself in the class variable.

This class is copyrighted by Luis Tavera and Thorr Einarsson. However, it may be freely distributed and modified with the condition that Prof. Ralph Johnson (johnson@cs.uiuc.edu) be sent a copy of the changes. You should also direct any comments to him.

Authors:
     Luis Tavera & Thorr Einarsson -- main authors
     Markus Geltz -- made modifications to produce encapsulated Postscript
     Carl McConnell -- fixed the copied graphics context problem

Instance variables:
	medium				<PSMedium>		where to output PostScript commands
	currentFontStyle		<FontDescription>	font being used

Class variables:
	CurrentContext		<PSGraphicsContext>	last instance used'!


!PSGraphicsContext methodsFor: 'initialize-release'!

close
	"Send postscript epilogue, then flush and close medium."

	self putCommand: #showpage.
	self flush.
	medium close.
	CurrentContext := nil!

initialize
	"This initializes the newly created instance"
	self pageOffset: self class pageOffset.
	self scale: (1 @ -1).!

initializeForMediumIn: anExtent 

	fontPolicy := Screen default defaultFontPolicy.

	lineWidth := 1.
	capStyle := GraphicsContext capButt.
	joinStyle := GraphicsContext joinMiter.
	offsetX := offsetY := 0.
	phaseX := phaseY := 0.
	anExtent isNil
		ifTrue: [self putCommand: #psComment]
		ifFalse: [self putCommand: #psCommentExtent: withArgs: (Array with: anExtent)].
	self putSetup.
	currentFontStyle := FontDescription default.
	self setFont.
	self paint: ColorValue black.
	medium put: 'gsave\' withCRs! !

!PSGraphicsContext methodsFor: 'accessing'!

clippingRectangle: aRectangleOrNil 
	"Set the clipping region to aRectangleOrNil. If aRectangleOrNil is nil, 
	no clipping occurs other than clipping to the bounds of the medium."

	super clippingRectangle: aRectangleOrNil.
	self setClippingRectangle!

font: aFont 
	"Set the default font that I use to draw strings."

	| thisFont changed underlined |
	changed := underlined := false.
	thisFont := (aFont isKindOf: FontDescription)
				ifTrue: [aFont]
				ifFalse: ["If this is a font description, no parsing is needed."
					aFont class parse: aFont name].
	currentFontStyle isNil
		ifTrue: 
			[currentFontStyle := thisFont.
			changed := true].
	thisFont pixelSize ~= currentFontStyle pixelSize
		ifTrue: 
			[currentFontStyle pixelSize: thisFont pixelSize.
			changed := true].
	thisFont italic ~= currentFontStyle italic
		ifTrue: 
			[currentFontStyle italic: thisFont italic.
			changed := true].
	thisFont serif ~= currentFontStyle serif
		ifTrue: 
			[currentFontStyle serif: thisFont serif.
			changed := true].
	thisFont fixedWidth ~= currentFontStyle fixedWidth
		ifTrue: 
			[currentFontStyle fixedWidth: thisFont fixedWidth.
			changed := true].
	thisFont boldness ~= currentFontStyle boldness
		ifTrue: 
			[currentFontStyle boldness: thisFont boldness.
			changed := true].
	underlined := thisFont underline isNil
				ifTrue: [false]
				ifFalse: [thisFont underline].
	underlined
		ifTrue: 
			[self startUnderlining.
			changed := true].
	changed ifTrue: [self setFont]!

lineWidth: width
	"Set the default width that I use to draw lines of unspecified width."

	super lineWidth: width.
	self setLineWidth!

pageOffset
	"This returns the instance variable"
	^pageOffset.!

pageOffset: aPoint
	"This sets the instance variable"
	^pageOffset := aPoint.!

scale
	"This returns the instance variable"
	^scale.!

scale: aPoint
	"This sets the instance variable"
	^scale := aPoint.! !

!PSGraphicsContext methodsFor: 'displaying'!

clear
	self establish.
	self putCommand: #erasePage.!

displayImage: anImage at: aPoint
	"Display the specified image, translated by aPoint."

	self establish.
	self putCommand: #gsave; putPoint: aPoint; putCommand: #translate.
	medium putImage: anImage cmd: #image.
	self putCommand: #grestore.!

displayLineFrom: startPoint to: endPoint 
	"Display a line between startPoint and endPoint."

	self establish.
	self putPoint: startPoint;	
		putCommand: #moveTo;
		putPoint: endPoint;	
		putCommand: #lineTo; 
		putCommand: #stroke.!

displayMappedString: aString from: startIndex to: endIndex at: aPoint withMap: map
	"Display the substring of aString from startIndex to endIndex. Place 
	the left end of the text baseline at aPoint. Use the default font and 
	paint."

	self establish.
	self putPoint: aPoint;	putCommand: #moveTo.
	medium show: (aString copyFrom: startIndex to: endIndex).!

displayMask: aMask at: aPoint 
	"Display the specified mask, translated by aPoint. Use the default 
	paint."

	self establish.
	self putCommand: #gsave; putPoint: aPoint; putCommand: #translate.
	medium putImage: aMask asImage cmd: #mask.
	self putCommand: #grestore.!

displayPolygon: pointCollection at: aPoint 
	"Fill the polygon whose vertices are specified by pointCollection, translated by aPoint, 
	with the current default paint."

	self establish.
	self putCommand: #newPath; putPoint: pointCollection first; putCommand: #moveTo.
	pointCollection keysAndValuesDo: 
			[:key :itm | key == 1 ifFalse: [self putPoint: (itm translatedBy: aPoint); putCommand: #lineTo]].
	self putCommand: #closePath; putCommand: #fill!

displayPolyline: pointCollection at: aPoint 
	"Display the polyline whose vertices are specified by pointCollection,
	translated by aPoint."

	self establish.
	self putCommand: #newPath; putPoint: (pointCollection first); putCommand: #moveTo.
	pointCollection keysAndValuesDo:
		[:key :itm | (key == 1) ifFalse: [ self putPoint: (itm translatedBy: aPoint); putCommand: #lineTo]].
	self putCommand: #closePath; putCommand: #stroke.!

displayString: aString from: startIndex to: endIndex at: aPoint 
	"Display the substring of aString from startIndex to endIndex. Place 
	the left end of the text baseline at aPoint. Use the default font and 
	paint."

	self establish.
	self putPoint: aPoint;	putCommand: #moveTo.
	medium show: (aString copyFrom: startIndex to: endIndex).! !

!PSGraphicsContext methodsFor: 'private'!

establish
	"If I am not the current context, then make myself the current 
	context, and establish any necessary state associated with me."

	| other |
	other := CurrentContext.
	other == self ifTrue: [^self].
	CurrentContext := self.
	other isNil ifTrue: ["If the other is nil, then there was no previous context, and the 
		state established when I was initialized should still be valid."
		^self].
	other clippingRectangleOrNil = self clippingRectangleOrNil ifFalse: [self setClippingRectangle].
	other font = self font ifFalse: [self setFont].
	other lineWidth = self lineWidth ifFalse: [self setLineWidth].
	other paint = self paint ifFalse: [self setPaint]!

putCommand: aCommand
	"put string for aCommand (operator or procedure) in medium"

	medium put: (PSGraphicsContext perform: aCommand) asString; cr!

putCommand: aCommand withArgs: collectionOfArgs 
	"Put string for aCommand (operator or procedure) in medium."

	medium put: (self class perform: aCommand withArguments: collectionOfArgs) asString; cr!

putNumber: aNumber 
	"put aNumber in medium"

	medium put: aNumber printString; space!

putPoint: aPoint 
	"put aPoint in medium as a string of two numbers ready to be placed on stack"

	medium put: (aPoint x + offsetX) asFloat printString; space; put: (aPoint y + offsetY) asFloat printString; space!

putScale: aPointOrNil
	"This writes out the scale to the file"

	| pt |
	pt := aPointOrNil isNil 
		ifTrue: [1 @ -1] 
		ifFalse: [aPointOrNil].
	self putPoint: pt.
	self scale: pt.
	self putCommand: #scale.!

putSetup
	"Put the Postscript Setup to the medium"

	self putCommand: #defInitialize.
	self putCommand: #defDrawEllipse.
	self putCommand: #defLandscape.
	self putCommand: #defTextHandling.
	self putCommand: #defRectPath.
	self putCommand: #defRectFill.
	self putCommand: #defRectStroke.!

putTranslation
	"Put the initial point to the medium"

	medium landscape 
		ifTrue: [self putPoint: (((medium width - medium height + self pageOffset x) / self scale x) @ (( self pageOffset y - medium width) / self scale y negated) )]
		ifFalse: [self putPoint: (((0+self pageOffset x) / self scale x)  @ ((self pageOffset y - medium height) / self scale y negated) )].
	self putCommand: #translate.!

setClippingRectangle
	"Set the PostScript clipping bounds."

	clipOriginX isNil
		ifTrue: [medium
				setClippingRectangle: 0 @ 0
				height: medium height
				width: medium width]
		ifFalse: [medium
				setClippingRectangle: clipOriginX @ clipOriginY
				height: clipHeight
				width: clipWidth]!

setFont
	"Let the medium know the new font style"

	medium setFont: currentFontStyle!

setLandscape
	"Rotate the page 90 degrees."
	medium setLandscape.!

setLimits: aPoint
	"This sets the height and width for the medium"
	medium setLimits: aPoint.!

setLineWidth
	medium setLineWidth: lineWidth!

setPaint
	paint installOn: self! !

!PSGraphicsContext methodsFor: 'private-primitives'!

primDisplayArcBBoxOrigin: originPoint extent: extentPoint startAngle: startAngle sweepAngle: sweepAngle 
	"Display an elliptical arc defined by the bounding rectangle of the 
	ellipse and by the starting angle and angle of sweep for the arc."

	| rec |
	self establish.
	rec := Rectangle origin: originPoint extent: extentPoint.
	self putNumber: startAngle; putNumber: sweepAngle; putPoint: rec center; 
		putNumber: (rec width asFloat / 2); putNumber: (rec height asFloat / 2); 
		putCommand: #drawEllipse; putCommand: #stroke.!

primDisplayRectangleOrigin: originPoint extent: extentPoint 
	"Display the rectangle with the top left corner extent. Use the 
	default paint."

	| rec |
	self establish.
	rec := Rectangle origin: originPoint extent: extentPoint.
	self putPoint: originPoint; putNumber: rec width; putNumber: rec height;
		putCommand: #rectFill.!

primDisplayRectangularBorderOrigin: originPoint extent: extentPoint 
	"Display the rectangular border with the specified top left corner 
	and extent. Use the default line width, cap style, join style, and paint."

	| rec |
	self establish.
	rec := Rectangle origin: originPoint extent: extentPoint.
	self putPoint: originPoint; putNumber: rec width; putNumber: rec height;
		putCommand: #rectStroke.!

primDisplayWedgeBBoxOrigin: originPoint extent: extentPoint startAngle: startAngle sweepAngle: sweepAngle 
	"Display an elliptical wedge defined by the bounding rectangle of the ellipse and by the
	 starting angle and angle of sweep for the arc."

	| rec |
	self establish.
	rec := Rectangle origin: originPoint extent: extentPoint.
	self putNumber: startAngle; putNumber: sweepAngle; putPoint: rec center; 
		putNumber: (rec width asFloat / 2); putNumber: (rec height asFloat / 2); 
		putCommand: #drawEllipse; putCommand: #fill.! !

!PSGraphicsContext methodsFor: 'private-initialize-release'!

setMedium: aDisplayMedium extent: anExtent 
	medium := aDisplayMedium.
	self initializeForMediumIn: anExtent!

setPaintToColor: aColorValue 
	"Set the default device paint to the specified ColorValue."

	self putNumber: aColorValue red; putNumber: aColorValue green; 
		putNumber: aColorValue blue;
		putCommand: #setRGBColor.!

setPaintToCoverage: aCoverageValue 
	"Set the default device paint to the specified CoverageValue."

	self error: 'This functionality is unimplemented'! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

PSGraphicsContext class
	instanceVariableNames: ''!


!PSGraphicsContext class methodsFor: 'class initialization'!

initialize
	"This initializes the class"
	"PSGraphicsContext initialize"
	self pageOffset: (20@20).
	CurrentContext := nil.! !

!PSGraphicsContext class methodsFor: 'instance creation'!

fileName: aString 
	^self fileName: aString extent: nil!

fileName: aString extent: anExtent 
	^self on: (Filename named: aString) writeStream extent: anExtent!

new
	^super new initialize.!

on: aStream
	^self on: aStream extent: nil!

on: aStream extent: anExtent 
	"Answer an instance that will write to the argument."

	^self new setMedium: (PSMedium on: aStream)
		extent: anExtent! !

!PSGraphicsContext class methodsFor: 'examples'!

windowExample
	"PSGraphicsContext windowExample"

	| file view gc |
	file := DialogView requestNewFileName: 'Name of file for Postscript:' default: 'view.ps'.
	file isEmpty ifTrue: [^self].
	view := ScheduledControllers activeController view.
	Cursor write
		showWhile: 
			[gc := self fileName: file extent: view extent.
			[view displayOn: gc]
				valueNowOrOnUnwindDo: [gc close]]! !

!PSGraphicsContext class methodsFor: 'constants'!

capButt
	"Answer the integer constant representing the butt cap style."
	^0!

capProjecting
	"Answer the integer constant representing the projecting cap style."
	^2!

capRound
	"Answer the integer constant representing the round cap style."
	^1!

joinBevel
	"Answer the integer constant representing the bevel join style."
	^2!

joinMiter
	"Answer the integer constant representing the miter join style."
	^0!

joinRound
	"Answer the integer constant representing the round join style."
	^1! !

!PSGraphicsContext class methodsFor: 'procedure definitions'!

defCTMsave
	"Answer the definition of a Postscript procedure that saves the current transformation matrix"

	^'/origCTM matrix currentmatrix def'.!

defDrawEllipse
	"Answer the definition of a Postscript procedure that draws an ellipse"

	^'/DrawEllipse		% stack startAngle sweepAngle ... -> --
{
  /m matrix currentmatrix def
  newpath
  4 2 roll translate scale
  0 0 1 5 3 roll arc
  closepath
  m setmatrix
} bind def'!

defErrorHandler
	"Answer the definition of a PostScript procedure that prints error messages on the printer"

	^'errordict begin
  /handleerror {
  systemdict begin
    initgraphics

    /Courier findfont 12 scalefont setfont
    72 720 moveto
    (Postscript problem:) show
    $error /errorname get
    =string cvs show
    72 700 moveto
    (Command: ) show
    $error /command get
    =string cvs show
    systemdict /showpage get exec
    end % systemdict
  } def
end
'.!

defInitialize
	"This defines variable setup material. Added by Tony White (arpw@bnr.ca)"
	^'
save /state exch def
initgraphics
/PPS 255 dict def PPS begin
/leftMargin 0 def
/rightMargin 0 def
/topMargin 0 def
/bottomMargin 0 def
/yDelta 16 def
/portrait 0 def
/landscape 1 def
/orientation portrait def
/doFooter false def
/doPageNumbers true def
/pageWidth 612.0 def
'.!

defLandscape
	"Add procedure that handles landscape specification. Added by Tony White (arpw@bnr.ca)"
	^'
/setOrientation
{
  orientation landscape eq
  {
    -90 rotate
    0 pageWidth sub 0 translate
  } if
} bind def
'.!

defRectFill
	"Answer the definition of a Postscript procedure that draws a filled rectangle"

	^'
/*RF {
  gsave newpath BuildRectPath fill grestore
} bind def
'!

defRectPath
	"Answer the definition of a Postscript procedure that builds a rectangular path"

	^'
/BuildRectPath {
  dup type dup /integertype eq exch /realtype eq or { %ifelse
    4 -2 roll moveto
    dup 0 exch rlineto exch 0 rlineto neg 0 exch rlineto closepath
  } {
    dup length 4 sub 0 exch 4 exch 
     {
        1 index exch 4 getinterval aload pop
        BuildRectPath
     } for
     pop
  } ifelse
} bind def
'.!

defRectStroke
	"Answer the definition of a Postscript procedure that draws a rectangle"

	^'
/*RS {
  gsave newpath BuildRectPath stroke grestore
} bind def
'!

defTextHandling
	"Last programmer: Markus Geltz (18 March 1993, 3:35:02 pm)" 
	"Creator: Markus Geltz (18 March 1993, 10:36:35 am)" 
	"Answer the definition of Postscript procedures that handle the text" 

^"%%initgraphics"
'
/F % Fast font setup.
{
findfont exch scalefont setfont
} bind def

/ISOshow
{
dup
/oldfont currentfont def
currentfont dup length dict /newdict exch def 
{1 index /FID ne
{newdict 3 1 roll put}
{pop pop}
ifelse
} forall
newdict /Encoding get 256 array copy
newdict exch /Encoding exch put
newdict /Encoding get exch 248 exch put
/TFNT newdict definefont pop
newdict setfont
(\370) newShow
oldfont setfont
} bind def

/newShow	% stack: text -> --
{
/m matrix currentmatrix def
1 -1 scale show
m setmatrix
} bind def
'!

psComment
	"Answer the Postscript prologue"

	^'%!!PS-Adobe-2.0 EPSF-1.2
%%Creator: Objectworks(R)\Smalltalk
%%BoundingBox: 203 384 366 512
%%EndComments'!

psCommentExtent: anExtent
	"Last programmer: Markus Geltz (22 March 1993, 1:28:25 pm)" 
	"Creator: Markus Geltz (22 March 1993, 1:28:19 pm)" 
	"Answer the Postscript prologue for Encapsulated Postscript" 

| xs ys |
xs := anExtent x.
ys := anExtent y.
^'%!!!!PS-Adobe-2.0 EPSF-1.2
%%Creator: Objectworks(R)\Smalltalk
%%BoundingBox: ' , 20 printString , ' ' , 772 printString , ' ' , (20 + xs) printString , ' ' , (772 - ys) printString , ' %%EndComments'! !

!PSGraphicsContext class methodsFor: 'constants-procedures'!

drawEllipse
	"Answer the drawellipse procedure in postscript"

	^'DrawEllipse'!

rectFill
	^'*RF'!

rectStroke
	^'*RS'! !

!PSGraphicsContext class methodsFor: 'constants-operators'!

closePath
	^'closepath'!

erasePage
	^'erasepage'!

fill
	^'fill'!

grestore
	^ 'grestore'.!

gsave
	^ 'gsave'.!

lineTo
	^'lineto'!

moveTo
	^'moveto'!

newPath
	^'newpath'!

scale
	^'scale'!

setRGBColor
	^'setrgbcolor'!

show
	^'show'!

showpage
	^'showpage'!

stroke
	^'stroke'!

translate
	^'translate'! !

!PSGraphicsContext class methodsFor: 'accessing'!

pageOffset
	"This returns the class variable"
	^PageOffset.!

pageOffset: aPoint
	"This sets the class variable"
	^PageOffset := aPoint.! !
PSGraphicsContext initialize!

PSFontFamily initialize!

