#include "geppetto.h"
#include "proto.h"

const int fitnessCases = 2048;

static bool addr[3], data[8];

static result *opNot P((int, const object **, void *));
static result *opAnd P((int, const object **, void *));
static result *opOr P((int, const object **, void *));
static result *opIf P((int, const object **, void *));
static objectList *multi11Terminals P((NOARGS));
static objectList *multi11Functions P((NOARGS));
static void *multi11CaseInitialize P((int));
static void multi11CaseFitness P((result *, int, int *, double *, double *,
				  void *));
static int multi11TerminateRun P((int, double, double));

static result *
opNot(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *rp;

  /* evaluate argument */
  rp = objectEval(argv[0], envp);
  if (!resultIsError(rp)) {

    /* make sure we got a boolean result */
    if (!resultIsBoolean(rp))
      resultSetError(rp, ErrorBadDataType);
    else {

      /* negate the result */
      resultSetBoolean(rp, !resultBoolean(rp));
    }
  }

  /* return result */
  return(rp);
}

static result *
opAnd(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r1, *r2;

  /* evaluate arguments */
  r1 = objectEval(argv[0], envp);
  if (resultIsError(r1))
    return(r1);
  r2 = objectEval(argv[1], envp);
  if (resultIsError(r2)) {
    resultFree(r1);
    return(r2);
  }

  /* make sure both results are valid */
  if (!resultIsBoolean(r1) || !resultIsBoolean(r2))
    resultSetError(r1, ErrorBadDataType);
  else
    resultSetBoolean(r1, resultBoolean(r1) & resultBoolean(r2));

  /* return the result */
  resultFree(r2);
  return(r1);
}

static result *
opOr(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r1, *r2;

  /* evaluate arguments */
  r1 = objectEval(argv[0], envp);
  if (resultIsError(r1))
    return(r1);
  r2 = objectEval(argv[1], envp);
  if (resultIsError(r2)) {
    resultFree(r1);
    return(r2);
  }

  /* make sure both results are valid */
  if (!resultIsBoolean(r1) || !resultIsBoolean(r2))
    resultSetError(r1, ErrorBadDataType);
  else
    resultSetBoolean(r1, resultBoolean(r1) | resultBoolean(r2));

  /* return the result */
  resultFree(r2);
  return(r1);
}

static result *
opIf(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *rp;
  bool bval;

  /* evaluate condition */
  rp = objectEval(argv[0], envp);
  if (!resultIsError(rp)) {

    /* make sure we got a boolean result */
    if (!resultIsBoolean(rp))
      resultSetError(rp, ErrorBadDataType);
    else {

      /* find out which branch we should take */
      bval = resultBoolean(rp);
      resultFree(rp);

      /* evaluate branch */
      rp = objectEval(argv[(bval ? 1 : 2)], envp);
    }
  }

  /* return result */
  return(rp);
}

static objectList *
multi11Terminals()
{
  objectList *list;
  int i;
  char name[3];
  variable *vp;

  list = objectListCreate(11);
  if (list) {

    /* set up address bits */
    name[0] = 'A';
    name[2] = 0;
    for (i = 0; i < 3; i++) {
      name[1] = '0' + i;

      vp = variableCreate(dtBoolean, name, &addr[i]);
      if (!vp || objectListAdd(list, vp)) {
	variableFree(vp);
	objectListFree(list);
	return(0);
      }
    }

    /* set up data bits */
    name[0] = 'D';
    name[2] = 0;
    for (i = 0; i < 8; i++) {
      name[1] = '0' + i;

      vp = variableCreate(dtBoolean, name, &data[i]);
      if (!vp || objectListAdd(list, vp)) {
	variableFree(vp);
	objectListFree(list);
	return(0);
      }
    }
  }

  return(list);

}

static objectList *
multi11Functions()
{
  objectList *list;
  operatorSrc *osp;

  list = objectListCreate(4);
  if (list) {

    osp = complexOperatorSrcCreate("NOT", opNot, dtBoolean, 1, 1, dtBoolean);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("AND", opAnd, dtBoolean, 2, 2, dtBoolean,
				   dtBoolean);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("OR", opOr, dtBoolean, 2, 2, dtBoolean,
				   dtBoolean);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("IF", opIf, dtBoolean, 3, 3, dtBoolean,
				   dtBoolean, dtBoolean);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }
  }

  return(list);
}

static void *
multi11CaseInitialize(fc)
int fc;
{
  int i, n;

  /* set address bits */
  for (i = 0, n = (fc >> 8) & 0x7; i < 3; i++, n >>= 1)
    addr[i] = (n & 0x1) == 1;

  /* set data bits */
  for (i = 0, n = fc & 0xff; i < 8; i++, n >>= 1)
    data[i] = (n & 0x1) == 1;

  return(0);
}

static void
multi11CaseFitness(rp, fc, hitp, rawp, stdp, envp)
result *rp;
int fc;
int *hitp;
double *rawp;
double *stdp;
void *envp;
{
  int i;

#ifdef DEBUG_MULTI11_NOT
  {
    charString *cstr;

    cstr = charStringCreate();
    charStringSet(cstr, "Result is");
    resultToString(rp, cstr);
    charStringPrint(cstr);
    charStringFree(cstr);
  }
#endif /* DEBUG_MULTI11 */

  /* the result should be a boolean */
  if (!resultIsBoolean(rp))
    *stdp += 1.0;
  else {

    i = (fc >> 8) & 0x7;

#ifdef DEBUG_MULTI11
    printf("Addr: %x, Data: %d%d%d%d%d%d%d%d, Computed: %d\n", i,
	   data[7], data[6], data[5], data[4], data[3], data[2], data[1],
	   data[0], resultBoolean(rp));
#endif /* DEBUG_MULTI11 */

    if ((data[i] != 0) == (resultBoolean(rp) != 0))
      *hitp += 1;
    else
      *stdp += 1.0;
  }
}

static int
multi11TerminateRun(hits, raw, std)
int hits;
double raw;
double std;
{
  return(hits == fitnessCases);
}

void
appInitialize(ip)
interface *ip;
{
  interfaceSetFitnessCases(ip, fitnessCases);
  interfaceSetTerminalList(ip, multi11Terminals());
  interfaceSetFunctionList(ip, multi11Functions());
  interfaceSetReturnTypes(ip, dtBoolean);
  interfaceCaseInitializeFunc(ip, multi11CaseInitialize);
  interfaceCaseFitnessFunc(ip, multi11CaseFitness);
  interfaceTerminateRunFunc(ip, multi11TerminateRun);
}

#ifdef DEBUG_MULTI11

#include "program.h"

int main P((int, char *[]));

int
main(argc, argv)
int argc;
char *argv[];
{
  interface *ip;
  void *envp;
  charString *cstr;
  program *pp;
  const char *src = "(IF A2 (IF A1 (IF A0 D7 D6) (IF A0 D5 D4)) (IF A1 (IF A0 D3 D2) (IF A0 D1 D0)))";
  int i = 1;

  ip = interfaceCreate(appInitialize);
  envp = interfaceCaseInitialize(ip, 0);

  /* create a charString for debugging */
  cstr = charStringCreate();

  /* dump terminal list */
  charStringSet(cstr, "Terminals: ");
  objectListToString(interfaceTerminalList(ip), cstr);
  charStringPrint(cstr);

  /* dump function list */
  charStringSet(cstr, "Functions: ");
  objectListToString(interfaceFunctionList(ip), cstr);
  charStringPrint(cstr);

  do {
    if (argc != 1)
      src = argv[i];
    pp = programParse(interfaceTerminalList(ip), interfaceFunctionList(ip),
		      src);
    if (pp == 0) {
      printf("Bad program '%s'!\n", src);
    } else {
      interfaceEval(ip, pp, 100, 0.0);
      programDump(pp, 1);
      programFree(pp);
    }
  } while (++i < argc);

  exit(0);
}
#endif /* DEBUG_MULTI11 */
