/*
 * Copyright (C) 1993 by Dave Glowacki
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

#include <malloc.h>
#include "interface.h"
#include "proto.h"

#ifndef NULL
#define NULL	0
#endif

interface *
interfaceCreate(initFunc)
initializeFunction *initFunc;
{
  interface *ip;

  /* create a new application interface */
  ip = (interface *)malloc(sizeof(interface));
  if (ip) {
    ip->cases = 0;
    ip->returntypes = dtAll;
    ip->tList = 0;
    ip->fList = 0;
    ip->current = 0;
    ip->environment = 0;
    ip->caseInit = 0;
    ip->caseTerminate = 0;
    ip->caseFitness = 0;
    ip->evalCleanup = 0;
    ip->terminateRun = 0;
    ip->destructor = 0;

    /* application-specific initialization */
    if (initFunc)
      (*initFunc)(ip);

    /* make sure the required things got initialized */
    if (ip->cases <= 0 || ip->tList == 0 || ip->fList == 0) {
      free(ip);
      ip = 0;
    }
  }

  return(ip);
}

void
interfaceEval(ip, pp, maxLoops, parsimony)
const interface *ip;
program *pp;
int maxLoops;
double parsimony;
{
  int i, l;
  void *envp = 0;
  int hits = 0;
  double raw = 0;
  double std = 0;
  result *rp;

  /* run through all the fitness cases */
  for (i = 0; i < ip->cases; i++) {

    /* set up environment for this case */
    if (ip->caseInit)
      envp = (*(ip->caseInit))(i);

    /* evaluate this case */
    if (ip->caseTerminate == NULL) {
      rp = programEval(pp, envp);	/* only need to run program once */
    } else {

      /* repeatedly evaluate the program */
      l = 0;
      rp = 0;
      do {
	if (rp)
	  resultFree(rp);
	rp = programEval(pp, envp);
      } while ((objectDataType(rp) != dtError) &&
	       !(*(ip->caseTerminate))(rp, envp, i) && (l++ < maxLoops));
    }

    /* evaluate the fitness of the result */
    if (ip->caseFitness)
      (*(ip->caseFitness))(rp, i, &hits, &raw, &std, envp);

    /* lose final result */
    if (rp)
      resultFree(rp);
  }

  /* clean up */
  if (ip->evalCleanup)
    ip->evalCleanup(envp);

  /* set number of hits for this program */
  programSetHits(pp, hits);

  /* modify standardized fitness by parsimony factor if desired */
  if (parsimony > 0.0)
    std += (programInternalNodes(pp) + programTerminalNodes(pp)) *
      parsimony;

  /* set final raw and standardized fitness */
  programSetRawFitness(pp, raw);
  programSetStandardizedFitness(pp, std);
}

void
interfaceFree(ip)
interface *ip;
{
  if (ip->destructor)
    (*(ip->destructor))();

  if (ip->tList)
    objectListFree(ip->tList);
  if (ip->fList)
    objectListFree(ip->fList);

  free(ip);
}

#ifdef DEBUG_INTERFACE

#include "constant.h"
#include "objectlist.h"
#include "operatrsrc.h"
#include "optrivial.h"

static objectList *appTerminals P((NOARGS));
static objectList *appFunctions P((NOARGS));
static void appInitialize P((interface *));
int main P((NOARGS));

static objectList *
appTerminals()
{
  objectList *list;
  int i;

  /* create list of terminals */
  list = objectListCreate(6);
  for (i = 1; i < 6; i++)
    if (objectListAdd(list, integerCreate(i))) {
      objectListFree(list);
      return(0);
    }
  return(list);
}

static objectList *
appFunctions()
{
  objectList *list;
  operatorSrc *osp;

  /* create list of functions */
  list = objectListCreate(2);

  osp = simpleOperatorSrcCreate("+", opAdd, 2);
  if (objectListAdd(list, osp)) {
    operatorSrcFree(osp);
    objectListFree(list);
    return(0);
  }

  osp = simpleOperatorSrcCreate("-", opSubtract, 2);
  if (objectListAdd(list, osp)) {
    operatorSrcFree(osp);
    objectListFree(list);
    return(0);
  }

  return(list);
}

static void
appInitialize(ip)
interface *ip;
{
  interfaceSetFitnessCases(ip, 1);
  interfaceSetTerminalList(ip, appTerminals());
  interfaceSetFunctionList(ip, appFunctions());
}

int
main()
{
  interface *ip;
  program *pp;

  /* create interface */
  ip = interfaceCreate(appInitialize);

  /* create a full program */
  pp = programCreate(interfaceFunctionList(ip), interfaceTerminalList(ip),
		     dtAll, 4, pcmFull);

  /* test fitness */
  interfaceEval(ip, pp, 100, 0.0);

  /* clean up */
  programFree(pp);
  interfaceFree(ip);

#ifdef KEEP_ALLOCATED_MEMORY
  constantFreeStack();
  resultFreeStack();
#endif /* KEEP_ALLOCATED_MEMORY */

  printf("INTERFACE works\n");
  exit(0);
}
#endif /* DEBUG_INTERFACE */
