/*
 * Copyright (C) 1993 by Dave Glowacki
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

#include <malloc.h>
#include <string.h>
#include "charstring.h"
#include "proto.h"

static int charStringExtend P((charString *, int));

charString *
charStringCreate()
{
  charString *cstr;

  cstr = (charString *)malloc(sizeof(charString));
  if (cstr) {
    cstr->buffer = 0;
    cstr->allocated = cstr->used = 0;
  }
  return(cstr);
}

static int
charStringExtend(cstr, len)
charString *cstr;
int len;
{
  /* try to get more space */
  if (cstr->allocated == 0) {
    cstr->buffer = (char *)malloc((unsigned )len);
    if (cstr->buffer) {
      cstr->allocated = len;
      cstr->buffer[0] = 0;
    }
  } else {
    cstr->buffer = (char *)realloc(cstr->buffer, (unsigned )len);
    cstr->allocated = len;
  }

  /* return success/failure */
  return(cstr->buffer == 0);
}

int
charStringNSet(cstr, ptr, len)
charString *cstr;
const char *ptr;
int len;
{
  char *rval;

  /* see if we need a bigger buffer */
  if (len+1 > cstr->allocated)
    if (charStringExtend(cstr, len+1))
      return(1);

  /* fail if there's still not enough space */
  if (len+1 > cstr->allocated)
    rval = 0;
  else {
    rval = strncpy(cstr->buffer, ptr, (unsigned )len);
    cstr->buffer[len] = 0;
    cstr->used = len;
  }

  return(rval == 0);
}

int
charStringNCatenate(cstr, ptr, len)
charString *cstr;
const char *ptr;
int len;
{
  char *rval;

  /* see if we need a bigger buffer */
  if (cstr->used+len+1 > cstr->allocated)
    if (charStringExtend(cstr, cstr->used+len+1))
      return(1);

  /* fail if there's still not enough space */
  if (cstr->used+len+1 > cstr->allocated)
    rval = 0;
  else {
    rval = strncpy(&(cstr->buffer[cstr->used]), ptr, (unsigned )len);
    cstr->buffer[cstr->used+len] = 0;
    cstr->used += len;
  }

  return(rval == 0);
}

int charStringRead(cstr, in)
charString *cstr;
FILE *in;
{
  int notDone = 1;
  int offset = 0;
  char *s = 0;

  /* get some space */
  if (!cstr->buffer)
    if (charStringExtend(cstr, 128))
      return(1);

  /* loop until we find a NEWLINE */
  while (notDone) {

    /* catch fgets() problems */
    if (fgets(&(cstr->buffer[offset]), cstr->allocated - offset, in) == NULL) {
      cstr->buffer[offset] = 0;
      cstr->used = (offset > 0 ? offset - 1 : 0);
      return(0);
    }

    /* search string */
    for (s = &(cstr->buffer[offset]);
	 notDone && s - cstr->buffer < cstr->allocated; s++)
      if (*s == '\n')
	notDone = 0;

    /* extend string if we need to read more */
    if (notDone) {

      /* remember how much we've already read */
      offset = cstr->allocated - 1;

      /* extend the buffer */
      if (charStringExtend(cstr, cstr->allocated << 1))
	return(1);
    }
  }

  /* remember how long the string is */
  if (s) {
    cstr->used = s - cstr->buffer;
    cstr->buffer[cstr->used] = 0;
  }
  return(0);
}

int
charStringWrite(cstr, out)
const charString *cstr;
FILE *out;
{
  const char *cp;

  if (cstr->buffer) {
    cp = cstr->buffer;
  } else
    cp = "";

  if (fputs(cp, out) == EOF)
    return(1);

  return(0);
}

int
charStringPrint(cstr)
const charString *cstr;
{
  int result;
  result = charStringWrite(cstr, stdout);
  if (result)
    return(result);

  if (putc('\n', stdout) == EOF)
    return(1);

  return(0);
}

void
charStringFree(cstr)
charString *cstr;
{
  if (cstr->buffer)
    free(cstr->buffer);
  free(cstr);
}

#ifdef DEBUG_CHARSTRING

int main P((NOARGS));
static void checkIO P((NOARGS));

static void
checkIO()
{
  FILE *in;
  charString *cstr;
  int stillGoing = 1;

  in = fopen("charstring.c", "r");
  if (!in) {
    fprintf(stderr, "Couldn't open 'charstring.c'\n");
    return;
  }

  cstr = charStringCreate();
  if (!cstr) {
    fprintf(stderr, "Couldn't create charString\n");
    return;
  }

  do {
    if (charStringRead(cstr, in)) {
      fprintf(stderr, "Read error\n");
      return;
    }

    if (charStringLength(cstr) <= 0)
      stillGoing = 0;
    else {
      if (charStringWrite(cstr, stdout)) {
	fprintf(stderr, "Write error\n");
	return;
      }
    }
  } while (stillGoing);

  charStringFree(cstr);
}

int
main()
{
  charString *s1;

  s1 = charStringCreate();
  charStringSet(s1, "String 1");
  charStringPrint(s1);
  charStringNSet(s1, "A longer buffer, but this part gets cut off", 15);
  charStringPrint(s1);
  charStringCatenate(s1, " that has stuff in it");
  charStringPrint(s1);
  charStringClear(s1);
  charStringNCatenate(s1, "abcdefghijklmnopqrstuvwxyz", 26);
  charStringPrint(s1);
  printf("Final string=\"%s\" (%d chars)\n", charStringBuffer(s1),
	 charStringLength(s1));

  checkIO();

  exit(0);
}
#endif /* DEBUG_CHARSTRING */
