/************************************************************************
 *                                                                      *
 *               PGA - Parallel Genetic Algorithm Testbed               *
 *                                                                      *
 *           "A Parallel Implementation of a genetic algorithm"         *
 *                                                                      *
 *                By Geoffrey H. Ballinger (geoff@ed.ac.uk),            *
 *                   Peter Ross (peter@ed.ac.uk)                        *
 *                                                                      *
 ************************************************************************/

/* 
   Copyright (C) 1993, Peter Ross and Geoffrey H. Ballinger.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
*/

#define TITLESTRING "PGA: parallel genetic algorithm testbed, version 2.4\n"

/*
 * Revision 2.4  93/6/29 pmr
 * Added `Royal road' problem, user-selectable block size (-errK, K int)
 * Fixed minor problem in fitprop_select(), copes with tiny/negative
 *   fitnesses (negative..??)
 * Fitprop_select() doesn't recalculate total fitness every time now;
 *   just upadted on insertion and in gen reproduction.
 * If saving to file, can save chromosomes too (in "file.chr")
 * Track chromosome parentage.
 *
 * Revision 2.3  93/6/18 pmr
 * Added one-point and uniform crossovers, (-C option)
 * First public (non-Edinburgh) release.
 *
 * Revision 2.2  93/6/2  pmr
 * Added 1-d knapsack problem, info from file `weights'
 * Modified screen layout a little
 *
 * Revision 2.1  93/3/18 pmr
 * Switch to char array for chromosomes, user-selectable length
 * Some code tidying
 *
 * Revision 2.0  92/2/27 pmr
 * Basic curses output
 * Added problem akin to binary F6 of Schaffer et al
 * Fixed problem with gen reproduction
 * Output to file too (append) if named
 * Fixed problem with sorting of chromosomes
 *
 * Revision 1.9  92/2/27 pmr
 * Some code and output tidying
 * Fixed bias problems
 *
 * Revision 1.8  90/11/23  10:24:33  ghb
 * Fitness is now a double.
 * Added De Jong's functions 1,2,3 and 5.
 * No longer display the best genotype.
 * Limit by evaluations, not generations.
 * Fixed bugs in generation based reproduction.
 * Adaptive mutation not allowed in generation based reproduction.
 * Fixed F5 using Genesis source as guide.
 * 
 * Revision 1.7  90/11/18  00:04:00  ghb
 * Deal in generations between migrations, not migrations per generation.
 * Fixed output bug.
 * Revision number in help screen.
 * 
 * Revision 1.6  90/11/14  11:51:49  ghb
 * Wrote replacement for drand48.
 * Adaptive mutation flag now works!
 * Added fitprop selection and gen reproduction.
 * Added user defined migration size.
 * 
 * Revision 1.5  90/11/13  10:16:32  ghb
 * Re-worked command line handleing.
 * Allow definable output file, operators, etc.
 * Removed "PGA: " from normal outputs.
 * Added help option.
 * 
 * Revision 1.4  90/11/12  08:56:48  ghb
 * Comments added.
 * 
 * Revision 1.3  90/11/09  13:30:42  ghb
 * Fixed Insertion to insert new best member.
 * It Works!
 * 
 * Revision 1.2  90/11/09  10:13:09  ghb
 * Shortened some variable names to make things more readable.
 * Limit number of generations, not number of evaluations.
 * Insertion, Selection, Reproduction, Crossover and Mutation!
 * 
 * Revision 1.1  90/11/07  21:27:02  ghb
 * Initial revision
 * 
*/


#include <stdio.h>            /* Include the IO and maths header files. */
#include <math.h>
#include <curses.h>
#include "pga.h"              /* screen layout configuration */

#ifdef hpux
#define random() lrand48()
#define srandom(seed) srand48((long) seed)
#endif hpux

typedef struct genodata {     /* genotype string plus reference count    */
  char *genotype;
  int refcount;
  int id;
  int parent1, parent2;
} GENODATA;

typedef struct chromosome {   /* A chromosome consists of a genotype and */
  GENODATA *gdp;               /* a fitness.                              */
  double fitness;
} CHROMOSOME;


void handle();                /* Prototypes. */
void report();
void free_pops();
int compchromo();
int read_weights_file();
double eval_max();
double eval_dj1();
double eval_dj2();
double eval_dj3();
double eval_dj4();
double eval_dj5();
double eval_bf6();
double eval_knap();
double eval_rrK();
void long_decode();
void double_decode();
void insert();
CHROMOSOME rank_select();
CHROMOSOME fitprop_select();
void one_reproduction();
void gen_reproduction();
CHROMOSOME one_pt_cross();
CHROMOSOME two_pt_cross();
CHROMOSOME uniform_cross();
CHROMOSOME mutate();
CHROMOSOME *init_pops();
double adaptive();
double nonadaptive();
void dump_chromos();
double drandom();

extern FILE *fopen();
extern int fscanf();
extern char *sprintf();
extern char *malloc();
extern char *strdup();

int numpop = 5,              /* Defaults */
    numchromo = 50,
    geno_size = 32,
    gens_stage = 100,
    repinterval = 10,
    isadaptive = FALSE,
    last_line = 24,
    rrK = 8,
    migint = 10,
   chromo_id = 1;
long *sticks;
long target;
FILE *output = (FILE *)NULL;
FILE *chromofile = (FILE *)NULL;
int chromoutput = FALSE;
char outfilename[256], chromofilename[256], *geno_size_string;
double mute_rate = 0.05,
       cross_rate = 0.6,
       bias = 1.5;
char *eval_name,
     *maxfitness,
     *cross_name,
     *select_name,
     *repro_name;
CHROMOSOME (*select)() = rank_select;
CHROMOSOME (*crossover)() = one_pt_cross;
void (*reproduction)() = one_reproduction;
void (*decode)() = long_decode;
double (*mrate)() = nonadaptive;
double (*evaluate)() = eval_max;
double *total_abs_fitness;

int evals;                   /* Globals */

/************************************************************************/
/* The main program.                                                    */
/************************************************************************/

main(argc,argv)
int argc;
char *argv[];
{
  int command,chromo,pop,gen,mignext,action,gen_total,fileoutput;
  double average;
  CHROMOSOME *pops,migchromo;


  srandom((unsigned)time(0));    /* Initialise the random number generator. */

  mignext=0;

  geno_size_string = (char *)malloc(20);
  sprintf(geno_size_string,"%d",geno_size);
  maxfitness = geno_size_string;
  eval_name = "max";
  cross_name = "two-point";
  select_name = "rank";
  repro_name = "one";  

  for (command=1; command<argc; command+=1) {
    handle(argv[command]);                    /* Handle any arguaments. */
  }

  if(geno_size > CRUCIAL_GENO_SIZE)
    decode = double_decode;
  if(evaluate == eval_knap) {
    sticks = (int *)calloc(geno_size,sizeof(long));
    if(!read_weights_file()) {
      fprintf(stderr,"No weights file accessible. It should hold\n");
      fprintf(stderr,"the target value (an integer) and up to\n");
      fprintf(stderr,"%d further integers. The aim is to find a\n", geno_size);
      fprintf(stderr,"subset with sum as close to the target as possible.\n");
      exit(1);
    }
  } else if(evaluate == eval_rrK) {
    maxfitness = (char *)malloc(10);
    sprintf(maxfitness,"%d", (geno_size / rrK) * rrK);
  }

  stdscr = initscr();
  last_line = tgetnum("li");
  if(last_line < 0) last_line = 24;
  

  fileoutput = (output == (FILE *)NULL)? FALSE : TRUE;

  mvprintw(POP_ROW,POP_COL,      "          Populations: %-6d",numpop);
  mvprintw(CPP_ROW,CPP_COL,      "  Chromosomes per pop: %-4d",numchromo);
  mvprintw(EVLIM_ROW,EVLIM_COL,  "Generations per stage: %-6d",gens_stage);
  mvprintw(REPINT_ROW,REPINT_COL,"   Reporting interval: %-6d",repinterval);
  mvprintw(MIG_ROW,MIG_COL,      "   Migration interval: %-6d",migint);
  mvprintw(CTYPE_ROW,CTYPE_COL,  "   Crossover type: %s", cross_name);
  if(!strcmp(repro_name,"one"))
    mvprintw(CROSS_ROW,CROSS_COL,"   Crossover rate: n/a");
  else
    mvprintw(CROSS_ROW,CROSS_COL,"   Crossover rate: %3.2f",cross_rate);
  if(!strcmp(select_name,"fitprop"))
    mvprintw(BIAS_ROW,BIAS_COL,  "   Selection bias: n/a");
  else
    mvprintw(BIAS_ROW,BIAS_COL,  "   Selection bias: %3.2f",bias);
  if(isadaptive)
    mvprintw(MUT_ROW,MUT_COL,    "    Mutation rate: adaptive");
  else
    mvprintw(MUT_ROW,MUT_COL,    "    Mutation rate: %3.2f",mute_rate);
  mvprintw(CRL_ROW,CRL_COL,      "Chromosome length: %d",geno_size);
  if(fileoutput)
    mvprintw(FNAME_ROW,FNAME_COL,"    Log file name: %s",outfilename);
  mvprintw(EVAL_ROW,EVAL_COL,    "Eval function: %s",eval_name);
  mvprintw(SELECT_ROW,SELECT_COL,"    Selection: %s", select_name);
  mvprintw(REPRO_ROW,REPRO_COL,  " Reproduction: %s",repro_name);
  mvprintw(POPTABLE_ROW,POPTABLE_COL,
               "Pop.......Average..........Best.(max = %s)",maxfitness);

  total_abs_fitness = (double *)malloc(numpop*sizeof(double));
  pops = init_pops(numpop,numchromo);
  gen_total = 1;
  evals=0;
  action = RESTART;

  if(fileoutput) {
    fprintf(output,   "\n\n===================================\n");
    fprintf(output,   "               eval = %s\n",eval_name);
    fprintf(output,   "        max-fitness = %s\n",maxfitness);
    fprintf(output,   "       reproduction = %s\n",repro_name);
    fprintf(output,   "             select = %s\n",select_name);
    if(!strcmp(select_name,"fitprop"))
      fprintf(output, "     selection-bias = n/a\n");
    else
      fprintf(output, "     selection-bias = %3.2f\n",bias);
    fprintf(output,   "  chromosome-length = %d\n",geno_size);
    fprintf(output,   "               pops = %d\n",numpop);
    fprintf(output,   "chromosomes-per-pop = %d\n",numchromo);
    fprintf(output,   " migration-interval = %d\n",migint);
    fprintf(output,   "     crossover-type = %s\n",cross_name);
    if(!strcmp(repro_name,"one"))
      fprintf(output, "     crossover-rate = n/a\n");
    else
      fprintf(output, "     crossover-rate = %3.2f\n",cross_rate);
    if(isadaptive)
      fprintf(output, "      mutation-rate = adaptive\n");
    else
      fprintf(output, "      mutation-rate = %3.2f\n", mute_rate);
    fprintf(output,   " reporting-interval = %d\n", repinterval);
    fprintf(output,   "\n");
  }

  mvprintw(GEN_ROW,GEN_COL,"Generation: ");
  clrtoeol();
  mvprintw(EVCOUNT_ROW,EVCOUNT_COL,"Evaluations so far: ");
  clrtoeol();
  refresh();
  report(0,fileoutput,pops);
  while(action != QUIT) {
      if(fileoutput)
        mvprintw(0,1,"(A)gain, (C)ontinue, (S)ave+continue, (Q)uit: ");
      else
        mvprintw(0,1,"(A)gain, (C)ontinue, (Q)uit: ");

      clrtoeol();
      refresh();
      do {
              switch(getch()) {
                    case 'a': case 'A':
                            action = RESTART;
                            free_pops(pops,numpop*numchromo);
                            gen_total = 1;
                            pops = init_pops(numpop,numchromo);
                            evals = 0;
                            move(GEN_ROW,GEN_COL+12);
                            clrtoeol();
                            move(EVCOUNT_ROW,EVCOUNT_COL+20);
                            clrtoeol();
                            refresh();
                            report(0,fileoutput,pops);
                            if(fileoutput)
                              fprintf(output,
                                      "\n\n------ re-starting ------\n\n");
                            break;
                    case 'c': case 'C':
                            action = CONTINUE;
                            break;
                    case 'q': case 'Q':
                            action = QUIT;
                            break;
		    case 's': case 'S':
                            if(fileoutput) {
                               action = CONTINUE;
                               dump_chromos(pops,gen_total-1);
                               break;
			    }
                    default:
                            action = ASK_AGAIN;
                            move(0,30);
                            clrtoeol();
                            refresh();
                            break;
              }
      } while(action == ASK_AGAIN);
      if(action == CONTINUE) {
        for(gen=1; gen <= gens_stage; gen++, gen_total++) {
          for(pop=0; pop<numpop; pop+=1) {            /* For each generation */
            reproduction(pops,pop*numchromo,numchromo);        /* Reproduce. */
          }
          if (numpop>1 && migint > 0 && gen%migint==0) {
            migchromo=select(pops,mignext*numchromo,numchromo);/* Migrate if */
            for(pop=0; pop<numpop; pop+=1) {                   /* necessary. */
              if(pop != mignext)               /* Don't insert to its source */
                insert(pops,migchromo,pop*numchromo,numchromo,0);
            }
            mignext=(mignext+1)%numpop;
          }
          if (gen_total%repinterval==0) {                      /* Report if  */
            report(gen_total,fileoutput,pops);                 /* necessary  */
  	  }
        }
      }      
  }
  if(fileoutput)
     fclose(output);  
  if(chromoutput)
     fclose(chromofile);
  free_pops(pops,numpop*numchromo);
  move(last_line-1,0);
  refresh();
  endwin();
  exit(0);
}


/************************************************************************/
/* This updates the screen and, if necessary, the output file           */
/************************************************************************/

void report(gen_total,fileoutput,pops)
int gen_total, fileoutput;
CHROMOSOME *pops;
{
  int pop, chromo;
  double average;

  mvprintw(GEN_ROW,GEN_COL+12,"%d",gen_total);         /* necessary */
  mvprintw(EVCOUNT_ROW,EVCOUNT_COL+20,"%d", evals);
  if(fileoutput) {
       fprintf(output, "\nGenerations=%d  Evaluations-so-far=%d\n",
                       gen_total,evals);
   }
   for(pop=0; pop<numpop; pop+=1) {               
      average=0.0;
      for(chromo=0; chromo<numchromo; chromo+=1) {
          average+=pops[pop*numchromo+chromo].fitness;
      }
      average/=(double)numchromo;
      mvprintw(POPTABLE_ROW+1+pop,POPTABLE_COL,
               "%-4d%c    %12.7f     %12.7f",
               pop, (fabs(average-pops[pop*numchromo].fitness)
                          < 1e-9)? '=' : ' ',
                          average,
                                     pops[pop*numchromo].fitness);
      if(fileoutput) {
          fprintf(output, "  %-4d    %12.7f     %12.7f\n",
                          pop,average,pops[pop*numchromo].fitness);
      }
      refresh();
  }
}


/************************************************************************/
/* This frees whole population                                          */
/************************************************************************/

void free_pops(whole_pop,num)
CHROMOSOME *whole_pop;
{
  int i;

  for(i=0; i<num; i++) {
    free(whole_pop[i].gdp->genotype);
    free(whole_pop[i].gdp);
  }
  free(whole_pop);
}

/************************************************************************/
/* This initialises a chromosome                                        */
/************************************************************************/

void init_genotype(gdp)
GENODATA *gdp;
{
  int i;
  long random();

  gdp->refcount = 1;
  gdp->id = chromo_id++;
  gdp->parent1 = gdp->parent2 = 0;
  gdp->genotype = (char *)malloc(geno_size+1);
  gdp->genotype[geno_size] = '\0';  /* for printing */
  for(i=0;i<geno_size;i++)
     gdp->genotype[i] = '0'+(random() & 1);
}

/************************************************************************/
/* This creates and randomly initialises the chromosome populations     */
/************************************************************************/

CHROMOSOME *init_pops(numpops,numchromo)
{
  int chromo,pop;
  CHROMOSOME *pops;
  GENODATA gd;

  pops=(CHROMOSOME *)malloc(numpop*numchromo*sizeof(CHROMOSOME));
  if(pops == (CHROMOSOME *)NULL) {
        mvprintw(POPTABLE_ROW+2,POPTABLE_COL,
                    "Cannot get enough memory - goodbye");
        move(last_line-1,0);
        refresh();
        exit(1);
  }

  for(pop=0;pop<numpops;pop++) total_abs_fitness[pop] = 0.0;

  for(chromo=0; chromo<numpops*numchromo; chromo+=1) {  /* Initialise the */
    pops[chromo].gdp = (GENODATA *)malloc(sizeof(GENODATA)); /* pops.     */
    init_genotype(pops[chromo].gdp);             
    pops[chromo].fitness=evaluate(pops[chromo].gdp->genotype);
    /* total_abs_fitness[n] is total fitness of population n */
    total_abs_fitness[chromo/numchromo] += fabs(pops[chromo].fitness);
  }

  for(pop=0; pop<numpops; pop+=1) {
    qsort((char *)((int)pops+(pop*numchromo*sizeof(CHROMOSOME))),
          numchromo, sizeof(CHROMOSOME), compchromo);
  }
  return(pops);
}

/***********************************************************************/
/* Help for the user                                                   */
/***********************************************************************/
void pga_help()
{
      fprintf(stderr,TITLESTRING);
      fprintf(stderr,"     -P<n>    Set number of populations. (5)\n");
      fprintf(stderr,"     -p<n>    Set number of chromosomes per population. (50)\n");
      fprintf(stderr,"     -n<n>    Set chromosome length. (32)\n");
      fprintf(stderr,"     -l<n>    Set # of generations per stage. (100)\n");
      fprintf(stderr,"     -i<n>    Set reporting interval in generations. (10)\n");
      fprintf(stderr,"     -M<n>    Interval between migrations. (10)\n");
      fprintf(stderr,"     -m<n>    Set mutation rate. (0.05)\n");
      fprintf(stderr,"     -c<n>    Set crossover rate (only for `gen'). (0.6)\n");
      fprintf(stderr,"     -b<n>    Set selection bias. (1.5)\n");
      fprintf(stderr,"     -a       Flip adaptive mutation flag (only for `one') (FALSE)\n");
      fprintf(stderr,"     -C<op>   Set crossover operator. (two)\n");
      fprintf(stderr,"     -s<op>   Set selection operator. (rank)\n");
      fprintf(stderr,"     -r<op>   Set reproduction operator. (one)\n");
      fprintf(stderr,"     -e<fn>   Set evaluation function. (max)\n");
      fprintf(stderr,"     -h       Display this information.\n");
      fprintf(stderr,"     <file>   Also log output in <file>. (none)\n\n");
      fprintf(stderr,"     Crossover operators ... one, two, uniform.\n");
      fprintf(stderr,"     Selection operators ... rank, fitprop.\n");
      fprintf(stderr,"     Reproduction operators ... one, gen.\n");
      fprintf(stderr,"     Evaluation functions ... max, dj1, dj2, dj3,\n");
      fprintf(stderr,"                              dj5, bf6, knap,\n");
      fprintf(stderr,"                              rrK (K=integer > 1).\n");
}


/************************************************************************/
/* This resets the defaults according to the argument 'arg'.            */
/************************************************************************/

void handle(arg)
char *arg;
{
  int newint;
  double newdouble;

  if (*arg=='-') {               /* Either a switch ...        */
    switch (*(arg+1)) {
    case 'P' :
      sscanf(arg,"-P%d",&newint);
      if (newint<1) {
        fprintf(stderr,"PGA: You must have at least 1 population.\n");
        exit(1);
      }
      else {
        numpop=newint;
      }
      break;
    case 'p' :
      sscanf(arg,"-p%d",&newint);
      if (newint<2) {
        fprintf(stderr,"PGA: You must have at least 2 chromosomes per population.\n");
        exit(1);
      }
      else {
        numchromo=newint;
      }
      break;
    case 'n' :
      sscanf(arg,"-n%d",&newint);
      if (newint<12) {
        fprintf(stderr,"PGA: Chromosome length must be at least 12.\n");
        exit(1);
      }
      else {
        geno_size=newint;
        sprintf(geno_size_string,"%d",geno_size);
      }
      break;
    case 'l' :
      sscanf(arg,"-l%d",&newint);
      if (newint<1) {
        fprintf(stderr,"PGA: Less than one generation per stage isn't sensible!\n");
        exit(1);
      }
      else {
        gens_stage=newint;
      }
      break;
    case 'i' :
      sscanf(arg,"-i%d",&newint);
      if (newint<1) {
        fprintf(stderr,"PGA: A report every %d generations isn't sensible!\n",newint);
        exit(1);
      }
      else {
        repinterval=newint;
      }
      break;
    case 'M' :
      sscanf(arg,"-M%d",&newint);
      if (newint<0) {
        fprintf(stderr,"PGA: A migration interval of %d doesn't make sense.\n",newint);
        exit(1);
      }
      else {
        migint=newint;
      }
      break;
    case 'a' :
      isadaptive=!isadaptive;
      if (isadaptive) mrate=adaptive;
      else mrate=nonadaptive;
      break;
    case 'm' :
      sscanf(arg,"-m%lf",&newdouble);
      if (newdouble<0 || newdouble>1) {
        fprintf(stderr,"PGA: The mutation rate must be between 0 and 1.\n");
        exit(1);
      }
      else {
        mute_rate=newdouble;
      }
      break;
    case 'c' :
      sscanf(arg,"-c%lf",&newdouble);
      if (newdouble<0 || newdouble>1) {
        fprintf(stderr,"PGA: The crossover rate must be between 0 and 1.\n");
        exit(1);
      }
      else {
        cross_rate=newdouble;
      }
      break;
    case 'b' :
      sscanf(arg,"-b%lf",&newdouble);
      if (newdouble<1.0 || newdouble>2.0) {
        fprintf(stderr,"PGA: The bias must be strictly between 1 and 2.\n");
        exit(1);
      }
      else {
        bias=newdouble;
      }
      break;
    case 's' :
      if (!strcmp(arg,"-srank")) {
        select = rank_select;
        select_name = "rank";
      }
      else if (!strcmp(arg,"-sfitprop")) {
        select = fitprop_select;
        select_name = "fitprop";
      }
      else {
        fprintf(stderr,"PGA: I don't know about %s selection.\n",arg+2);
        exit(1);
      }
      break;
    case 'r' :
      if (!strcmp(arg,"-rone")) {
        reproduction = one_reproduction;
        repro_name = "one";
      }
      else if (!strcmp(arg,"-rgen")) {
        reproduction = gen_reproduction;
        repro_name = "gen";
      }
      else {
        fprintf(stderr,"PGA: I don't know about %s reproduction.\n",arg+2);
        exit(1);
      }
      break;
    case 'C':
      if (!strcmp(arg,"-Cone")) {
        cross_name = "one-point";
        crossover = one_pt_cross;
      } else if (!strcmp(arg,"-Ctwo")) {
        cross_name = "two-point";
        crossover = two_pt_cross;
      } else if (!strcmp(arg,"-Cuniform")) {
        cross_name = "uniform";
        crossover = uniform_cross;
      }
      break;
    case 'e' :
      if (!strcmp(arg,"-emax")) {
        evaluate = eval_max;
        eval_name = "max";
        sprintf(geno_size_string,"%d",geno_size);
        maxfitness = geno_size_string;
      }
      else if (!strcmp(arg,"-edj1")) {
        evaluate = eval_dj1;
        eval_name = "dj1";
        maxfitness = "100";
      }
      else if (!strcmp(arg,"-edj2")) {
        evaluate = eval_dj2;
        eval_name = "dj2";
        maxfitness = "1000";
      }
      else if (!strcmp(arg,"-edj3")) {
        evaluate = eval_dj3;
        eval_name = "dj3";
        maxfitness = "55";
      }
      else if (!strcmp(arg,"-edj5")) {
        evaluate = eval_dj5;
        eval_name = "dj5";
        maxfitness = "approx 499.001997";
      }
      else if (!strcmp(arg,"-ebf6")) {
        evaluate = eval_bf6;
        eval_name = "bf6";
        maxfitness = "approx 0.994007";
      }
      else if (!strcmp(arg,"-eknap")) {
        evaluate = eval_knap;
        eval_name = "knap";
        maxfitness = "1.0";
      }
      else if (!strncmp(arg,"-err",4)) {
        evaluate = eval_rrK;
        eval_name = strdup(arg+2);
        /* NB, maxfitness set after arg processing, once geno_size known */
        rrK = atoi(arg+4);
        if(rrK < 2) {
           fprintf(stderr, "In rrK, K must be 2 or more; set to 8\n");
           rrK = 8;
        }
      }
      else {
        fprintf(stderr,"PGA: unknown evaluation function: %s\n",
                        arg+2);
        exit(1);
      }
      break;
    default :
      if (*(arg+1)!='h') fprintf(stderr,"Unknown argument %s.\n\n",arg);
      pga_help();
      exit(0);
    }
  }
  else {                         /* ... or an output filename. */
    output=fopen(arg,"a");
    if (output==NULL) {
      fprintf(stderr,"PGA: I can't open %s for writing.\n",arg);
      exit(1);
    }
    strcpy(outfilename,arg);
  }
}

/************************************************************************/
/* Dumps chromosomes to a file. Filename is outfilename.gen, eg foo.500 */
/************************************************************************/

void dump_chromos(pops,g)
CHROMOSOME *pops;
int g;
{
  int i,j;
  char filename[128];

  if(!chromoutput) {
     sprintf(chromofilename,"%s.chr",outfilename);
     chromofile = fopen(chromofilename,"a");
     chromoutput = (chromofile != (FILE *)NULL);
  }

  if(chromoutput) {
    for(i=0;i<numpop;i++) {
      for(j=0;j<numchromo;j++) {
        fprintf(chromofile,
                "%-3d %-5d %12.7f  %s  %5d %5d %5d\n",
                 i,   g,    pops[i*numchromo+j].fitness,
                                    pops[i*numchromo+j].gdp->genotype,
                                        pops[i*numchromo+j].gdp->id,
                                           pops[i*numchromo+j].gdp->parent1,
                                             pops[i*numchromo+j].gdp->parent2);
      }
      fprintf(chromofile,"\n\n");
    }
  }
}    

/************************************************************************/
/* Returns >0 if chromo2 is fitter than chromo1, <0 if chromo 1 is      */
/* fitter than chromo2, and 0 if they are equally fit. This is passed   */
/* to qsort as its comparison function.                                */
/************************************************************************/

int compchromo(chromo1,chromo2)
CHROMOSOME *chromo1,*chromo2;
{
  if((chromo2->fitness) > (chromo1->fitness))
     return(1);
  else if((chromo2->fitness) == (chromo1->fitness))
     return(0);
  else
     return(-1);
}


/************************************************************************/
/* This returns the number of bits set to one in 'genotype'. It is used */
/* to evaluate a chromosomes fitness.                                   */
/************************************************************************/

double eval_max(genotype)
char *genotype;
{
  int value, loop;

  evals+=1;

  value=0;
  for(loop=0; loop<geno_size; loop++) {       /* Check each bit in turn. */
    if (genotype[loop] == '1') value+=1;
  }

  return((double)value);
}


/************************************************************************/
/* De Jong's F1. Max fitness = 100                                      */
/************************************************************************/

double eval_dj1(genotype)
char *genotype;
{
  double x[3];

  evals+=1;

  decode(genotype,3,-5.12,5.12,x);

  return 100.0 - (x[0]*x[0]+x[1]*x[1]+x[2]*x[2]);
}


/************************************************************************/
/* De Jong's F2. Max fitness = 1000                                     */
/************************************************************************/

double eval_dj2(genotype)
char *genotype;
{
  double x[2];

  evals+=1;

  decode(genotype,2,-2.048,2.048,x);

  return 1000.0 - (100.0*(x[0]*x[0]-x[1])*(x[0]*x[0]-x[1]) + (1-x[0])*(1-x[0]));
}


/************************************************************************/
/* De Jong's F3. Max fitness = 55                                       */
/************************************************************************/

double eval_dj3(genotype)
char *genotype;
{
  double x[5];

  evals+=1;

  decode(genotype,5,-5.12,5.12,x);

  return 25.0 - (floor(x[0])+floor(x[1])+floor(x[2])+floor(x[3])+floor(x[4]));
}


/************************************************************************/
/* De Jong's F5. Max fitness = 500                                      */
/************************************************************************/

static int a[2][25] ={
        {
                -32, -16, 0, 16, 32, -32, -16, 0, 16, 32, -32, -16, 0, 16, 32,
                -32, -16, 0, 16, 32, -32, -16, 0, 16, 32        },
        {
                -32, -32, -32, -32, -32, -16, -16, -16, -16, -16,
                16, 16, 16, 16, 16, 32, 32, 32, 32, 32  }
};

double eval_dj5(genotype)
char *genotype;
{
  double x[2],total,lowtot,prod;
  int i,j,power;

  evals+=1;

  decode(genotype,2,-65.536,65.536,x);

  total=0.002;
  for(j=0; j<25; j+=1) {
    lowtot=1.0 + (double)j;
    for(i=0; i<2; i+=1) {
      prod=1.0;
      for(power=0; power<6; power+=1) {
        prod*=x[i]-a[i][j];
      }
      lowtot+=prod;
    }
    total+=1.0/lowtot;
  }

  return 500.0 - (1.0/total);
}

double eval_bf6(genotype)
char *genotype;
{
  double x[2], temp, temp2;

  evals+=1;

  decode(genotype,2,-100.0,100.0,x);

  temp = (x[1]*x[1] + x[0]*x[0]);
  temp2 = sin(sqrt(temp));
  
  return ( (temp2*temp2)/(1.0+0.001*temp*temp) );
}

double eval_knap(genotype)
char *genotype;
{
  int i;
  long sum = 0;
  double fit;

  evals += 1;

  for(i=0; i < geno_size; i++) {
    if(genotype[i]=='1') sum += sticks[i];
  }
  fit = 1.0/(1.0 + fabs((double)target - (double)sum));
  return(fit);
}

/* For knapsack problem, must read integers from file `weights' */
int read_weights_file()
{
    FILE *f;
    int i, r;
    long t;
    
    /* read target and sizes from file "weights" if it exists readable */
    if(((f=fopen("weights","r")) != (FILE *)NULL)
       && (fscanf(f,"%ld",&target) == 1)) {
          i = 0;
          while(i < geno_size && ((r = fscanf(f,"%ld", &t)) != EOF)) {
                if(r == 1) {
                    sticks[i++] = t;
                } else
                    /* got no number, so skip it as a string */
                    fscanf(f,"%*s");
          }
          fclose(f);
          return(1);
       }
    else return(0);
}

double eval_rrK(genotype)
char *genotype;
{
    int i = 0, score = 0, counter = 0;

    evals+=1;

    while(i < geno_size) {
      if(genotype[i] == '1') {
         i++;
         if(++counter == rrK) {
            counter = 0;
            score += rrK;
	 }
      } else {
         counter = 0;
         i = ((i/rrK)*rrK) + rrK;  /* skip to next block of rrK */
      }
    }
    return(score);
}

/************************************************************************/
/* This decodes the requested number of real parameters from the given  */
/* chromosome.                                                          */
/************************************************************************/

void long_decode(genotype,number,lower,upper,array)
char *genotype;
int number;
double lower,upper,*array;
{
  int loop,count,bits;
  long value;

  bits=(geno_size/number);
  
  for(loop=0; loop<number; loop+=1) {
    value=0;
    for(count=0; count<bits; count++) {
      value = 2*value + (int)(genotype[count+loop*bits] - '0');
    }
    array[loop]=((upper-lower)*((double)value)/pow(2.0,(double)bits))+lower;
  }
}  

void double_decode(genotype,number,lower,upper,array)
char *genotype;
int number;
double lower,upper,*array;
{
  int loop,count,bits;
  double value;

  bits=(geno_size/number);
  
  for(loop=0; loop<number; loop+=1) {
    value=0.0;
    for(count=0; count<bits; count++) {
      value = 2*value + (double)(genotype[count+loop*bits] - '0');
    }
    array[loop]=((upper-lower)*(value)/pow(2.0,(double)bits))+lower;
  }
}  


/************************************************************************/
/* This function inserts 'chromo' into the correct place (sorted by     */
/* fitness) within the subpopulation defined by 'base' and 'range' of   */
/* the populations array 'pops'.                                        */
/************************************************************************/

void insert(pops,chromo,base,range)
CHROMOSOME *pops,chromo;
int base,range;
{
  int current, pop;

  pop = base/numchromo;         /* which pop we're hitting */

                                /* Can it be inserted at all? */
  if (chromo.fitness>pops[base+range-1].fitness) {
    chromo.gdp->refcount++;
    if(pops[base+range-1].gdp->refcount == 1) {
      free(pops[base+range-1].gdp->genotype);
      free(pops[base+range-1].gdp);
    }
    current=base+range-2;       /* Start at the second last place. */
    do {                        /* If 'chromo' is fitter ... */
      if (chromo.fitness>pops[current].fitness) {
        pops[current+1]=pops[current]; /* ... then push it down one. */
      }
      else {
        total_abs_fitness[pop] += fabs(chromo.fitness) 
                                   - fabs(pops[current+1].fitness);        
        pops[current+1]=chromo;        /* ... else insert 'chromo' below */
        current=base;                  /* and finish.                    */
      }
      current-=1;               /* Work up the array until ... */
    } while (current>=base);    /* we get past the 'base'.     */
    if (chromo.fitness>pops[base].fitness) {
      total_abs_fitness[pop] += fabs(chromo.fitness) 
                                 - fabs(pops[base].fitness);
      pops[base]=chromo;        /* If chromo is the best put it in. */
    }
  } else {     /* It cannot be inserted, it is terrible */
      if(chromo.gdp->refcount==0) {
        free(chromo.gdp->genotype);
        free(chromo.gdp);
      }
  }
}


/************************************************************************/
/* This function performs rank based selection and is lifted from       */
/* Whitley's paper on GENITOR in icga 89. It selects from the specified */
/* subpopulation within 'pops'.                                         */
/************************************************************************/

CHROMOSOME rank_select(pops,base,range)
CHROMOSOME *pops;
int base,range;
{
  double value;

  if(bias<1.05)
     value=drandom()/bias;
  else {
     value=(bias-sqrt(bias*bias-4.0*(bias-1.0)*drandom()))/(2.0*(bias-1.0));
  }
  value=value*(double)range;

  return(pops[base+(int)value]);
}


/************************************************************************/
/* This function implements fitness proportionate selection. It treats  */
/* the fitness of the population as a "roulette wheel", rolls a ball on */
/* the wheel, and picks the chromosome within whose sector the ball     */
/* stops.                                                               */
/************************************************************************/

CHROMOSOME fitprop_select(pops,base,range)
CHROMOSOME *pops;
int base,range;
{
  int pop,loop;
  double where;

  pop = base/numchromo;  /* which pop we're working on */

  where=drandom()*total_abs_fitness[pop];  /* Random float in 0..total */

  for(loop=base; where>fabs(pops[loop].fitness); loop+=1)
    where-=fabs(pops[loop].fitness);

  return(pops[loop]);
}


/************************************************************************/
/* This function performs one-at-a-time reproduction on the specified   */
/* subpopulation. Adaptive mutation is used if activated.               */
/************************************************************************/

void one_reproduction(pops,base,range)
CHROMOSOME *pops;
int base,range;
{
  CHROMOSOME parent1,parent2,child;

  parent1=select(pops,base,range);         /* Select two parents. */
  parent2=select(pops,base,range);

  child=crossover(parent1,parent2);        /* Cross them to produce a child. */

  if (drandom()<=mrate(parent1,parent2))   /* Maybe mutate. */
    child=mutate(child);

  child.fitness=evaluate(child.gdp->genotype);  /* Evaluate the child. */

  insert(pops,child,base,range,1);         /* Insert the child into */
}                                          /* the population.       */


/************************************************************************/
/* This function performs generation based reproduction. A new          */
/* population is built by selecting members of the old one, possibly    */
/* crossing them with another, and possibly mutating them. This new     */
/* population is sorted and then used to replace the old population.    */
/************************************************************************/

void gen_reproduction(pops,base,range)
CHROMOSOME *pops;
int base,range;
{
  CHROMOSOME *newpop,parent1,parent2,child;
  int loop,reeval,pop;

  pop = base/numchromo;  /* which population we're reconstructing */

  newpop=(CHROMOSOME *)malloc(sizeof(CHROMOSOME)*range);

  for(loop=0; loop<range; loop+=1) {
    reeval=FALSE;
    parent1=select(pops,base,range);
    if (drandom()<=cross_rate) {
      reeval=TRUE;
      parent2=select(pops,base,range);
      child=crossover(parent1,parent2);
    }
    else {
      child=parent1;
    }
    if (drandom()<=mute_rate) {
      reeval=TRUE;
      child=mutate(child);
    }

    /* By this point, child may have refcount 0 if product of crossover */
    /* and/or of mutation; or may have refcount > 0 otherwise.          */
    /* Increment child's refcount, so it is 1 if completely new, >1     */
    /* if in old population too.                                        */
    child.gdp->refcount++;

    if (reeval) 
      child.fitness=evaluate(child.gdp->genotype);
    newpop[loop]=child;
  }

  qsort((char *)newpop,range,sizeof(CHROMOSOME),compchromo);

  total_abs_fitness[pop] = 0.0;
  for(loop=0; loop<range; loop+=1) {
    /* If old member is not wanted in new population or elsewhere, its */
    /* refcount will be 1. Decrement ref counts in old pop, ditch the  */
    /* unwanted. */
    if(pops[base+loop].gdp->refcount == 1) {
       free(pops[base+loop].gdp->genotype);
       free(pops[base+loop].gdp);
    } else
       pops[base+loop].gdp->refcount--;

    pops[base+loop]=newpop[loop];
    total_abs_fitness[pop] += fabs(newpop[loop].fitness);
  }

  free(newpop);
}


/************************************************************************/
/* This performs one point crossover on 'parent1' and 'parent2'.        */
/* Child is new (even though there may be identical genotypes elsewhere */
/* in the system) so has refcount 0; set to 1 only upon insertion       */
/************************************************************************/

CHROMOSOME one_pt_cross(parent1,parent2)
CHROMOSOME parent1,parent2;
{
  int chosen,i;
  CHROMOSOME child;

  chosen=random()%geno_size;        /* Generate crossover point randomly. */


                                   /* Create the child. */
  child.gdp = (GENODATA *)malloc(sizeof(GENODATA));
  child.gdp->refcount = 0;
  child.gdp->genotype = (char *)malloc(geno_size+1);
  child.gdp->genotype[geno_size]='\0';
  child.gdp->id = chromo_id++;
  child.gdp->parent1 = parent1.gdp->id;
  child.gdp->parent2 = parent2.gdp->id;

                                   /* Do the crossover */
  for(i=0; i<chosen; i++)
    child.gdp->genotype[i] = parent2.gdp->genotype[i];
  for(i=chosen; i<geno_size; i++)
    child.gdp->genotype[i] = parent1.gdp->genotype[i];

  return(child);
}

/************************************************************************/
/* This performs two point crossover on 'parent1' and 'parent2'.        */
/* Child is new (even though there may be identical genotypes elsewhere */
/* in the system) so has refcount 0; set to 1 only upon insertion       */
/************************************************************************/

CHROMOSOME two_pt_cross(parent1,parent2)
CHROMOSOME parent1,parent2;
{
  int first,last,i,swap;
  CHROMOSOME child;

  first=random()%geno_size;        /* Generate the two points randomly. */
  last=random()%geno_size;

  if (first>last) {                /* Make sure that they are in the */
    swap=first;                    /* right order.                   */
    first=last;
    last=swap;
  }

                                   /* Create the child. */
  child.gdp = (GENODATA *)malloc(sizeof(GENODATA));
  child.gdp->refcount = 0;
  child.gdp->genotype = (char *)malloc(geno_size+1);
  child.gdp->genotype[geno_size]='\0';
  child.gdp->id = chromo_id++;
  child.gdp->parent1 = parent1.gdp->id;
  child.gdp->parent2 = parent2.gdp->id;

                                   /* Do the crossover */
  for(i=0; i<first; i++)
    child.gdp->genotype[i] = parent2.gdp->genotype[i];
  for(i=first; i<last; i++)
    child.gdp->genotype[i] = parent1.gdp->genotype[i];
  for(i=last; i<geno_size; i++)
    child.gdp->genotype[i] = parent2.gdp->genotype[i];

  return(child);
}

/************************************************************************/
/* This performs uniform crossover on 'parent1' and 'parent2'.          */
/* Uses rand_bit() to cut down on expensive random number generation.   */
/* Child is new (even though there may be identical genotypes elsewhere */
/* in the system) so has refcount 0; set to 1 only upon insertion       */
/************************************************************************/

int rand_bit_src;
int rand_bit_bits = 0;
int rand_bit()
{
  if(rand_bit_bits == 0) {
    rand_bit_bits = 16;
    rand_bit_src = random() % 131072; /* Get 16+1 more random bits */
  }
  rand_bit_src = rand_bit_src >> 1;   /* Lose 1 bit */
  rand_bit_bits--;                    /* Number left after this call */
  return(rand_bit_src & 1);
}


CHROMOSOME uniform_cross(parent1,parent2)
CHROMOSOME parent1,parent2;
{
  int i;
  CHROMOSOME child;

  child.gdp = (GENODATA *)malloc(sizeof(GENODATA));
  child.gdp->refcount = 0;
  child.gdp->genotype = (char *)malloc(geno_size+1);
  child.gdp->genotype[geno_size]='\0';
  child.gdp->id = chromo_id++;
  child.gdp->parent1 = parent1.gdp->id;
  child.gdp->parent2 = parent2.gdp->id;

                                   /* Do the crossover */
  for(i=0; i<geno_size; i++)
    child.gdp->genotype[i] = 
          (rand_bit())? parent2.gdp->genotype[i] :
                        parent1.gdp->genotype[i] ;

  return(child);
}


/************************************************************************/
/* This flips a random bit in 'original'.                               */
/* If given chromo has non-zero refcount, then work on copy instead.    */
/* Refcounts get adjusted upon insertion, later.                        */
/************************************************************************/

CHROMOSOME mutate(original)
CHROMOSOME original;
{
  int place;
  CHROMOSOME newchromo;

  place=random()%geno_size;           /* A random position. */

  if(original.gdp->refcount > 0) {
     newchromo.gdp = (GENODATA *)malloc(sizeof(GENODATA));
     newchromo.gdp->refcount = 0;
     newchromo.gdp->genotype = (char *)malloc(geno_size+1);
     newchromo.gdp->genotype[geno_size]='\0';
     newchromo.gdp->id = chromo_id++;
     newchromo.gdp->parent1 = original.gdp->parent1;
     newchromo.gdp->parent2 = original.gdp->parent2;
     strncpy(newchromo.gdp->genotype,original.gdp->genotype,geno_size);
  } else
     newchromo = original;

  if(newchromo.gdp->genotype[place] == '0')
    newchromo.gdp->genotype[place] = '1';
  else
    newchromo.gdp->genotype[place] = '0';

  return(newchromo);
}


/************************************************************************/
/* This function returns the proportion of bits which are identical in  */
/* 'parent1' and 'parent2'. This is used as the adaptive mutation rate. */
/************************************************************************/

double adaptive(parent1,parent2)
CHROMOSOME parent1,parent2;
{
  int loop,count;
  double prob;

  count=0;
  for(loop=0; loop<geno_size; loop+=1) {
    if (parent1.gdp->genotype[loop] == parent2.gdp->genotype[loop]) count++;
  }
                        /* Proportion = observed / total. */
  prob=((double)count) / ((double)geno_size);

  return(prob);
}


/************************************************************************/
/* This function simply returns the fixed mutation rate.                */
/************************************************************************/

double nonadaptive(parent1,parent2)
CHROMOSOME parent1,parent2;
{
  return(mute_rate);
}


/************************************************************************/
/* This generates uniformly distributed random numbers from 0.0 to 1.0  */
/************************************************************************/

double drandom()
{
  return((double)random()/2147483647.0);
}

/* end of pga.c */
