#include <stdlib.h>
#include "rle.h"

static unsigned char *rl_src_buffer;
static unsigned char rl_escape;
static unsigned long rl_src;
static unsigned long rl_src_size;
static unsigned long rl_count;
static unsigned char rl_current;


#define COUNT_MAX	0x10000
#define LO(k)		((k) & (unsigned long)0xFF)
#define HI(k)		(((k) & (unsigned long)0xFF00)>>8)
#define UNLO(c)		(c)
#define UNHI(c)		((c) << 8)



/*
 *	RL-encode a buffer in-place
 *	returns:	no. of bytes after encoding
 */
unsigned long
rl_encode(buffer, size, escape)
unsigned char *buffer;		/* data to encode			*/
unsigned long size;		/* no. of bytes in buffer		*/
unsigned char escape;		/* esc. byte preceding a count		*/
{
    unsigned long src, dst;
    unsigned long count;
    unsigned char current;

    src = dst = 0;
    while (src < size) {
	count = 1;
	current = buffer[src];
	while (++src < size && buffer[src] == current && count < COUNT_MAX)
	    count++;
	if (count > 3 || current == escape) {
	    buffer[dst++] = escape;
	    buffer[dst++] = (char) LO(count);
	    buffer[dst++] = (char) HI(count);
	    buffer[dst++] = current;
	}
	else {
	    while (count--)
		buffer[dst++] = current;
	}
    }
    return dst;
}

/*
 *	RL-decode a buffer into another one
 *	returns:	no. of bytes after decoding
 *
 *	Caution: dst_buffer must be large enough to
 *	         hold the decoded data
 */
unsigned long
rl_decode(src_buffer, dst_buffer, src_size, escape)
unsigned char *src_buffer;	/* source buffer	*/
unsigned char *dst_buffer;	/* destination buffer	*/
unsigned long src_size;		/* no. of encoded bytes	*/
unsigned char escape;		/* escape code		*/
{
    unsigned long src, dst;
    unsigned long count;
    unsigned char current;

    src = dst = 0;
    while (src < src_size) {
	if (src_buffer[src] == escape) {
	    src++;
	    count = UNLO(src_buffer[src++]);
	    count |= UNHI(src_buffer[src++]);
	    current = src_buffer[src++];
	    while (count--)
		dst_buffer[dst++] = current;
	}
	else {
	    dst_buffer[dst++] = src_buffer[src++];
	}
    }
    return dst;
}

/*
 *	Determine the size needed for decoding
 */
unsigned long
rl_decode_length(src_buffer, src_size, escape)
unsigned char *src_buffer;	/* buffer		*/
unsigned long src_size;		/* size encoded		*/
unsigned char escape;		/* escape code		*/
{
    unsigned long src, dst;
    unsigned long count;

    src = dst = 0;
    while (src < src_size) {
	if (src_buffer[src] == escape) {
	    src++;
	    count = UNLO(src_buffer[src++]);
	    count |= UNHI(src_buffer[src++]);
	    src++;
	    dst += count;
	}
	else {
	    src++;
	    dst++;
	}
    }
    return (dst);
}				/* end rl_decode_length */


/*
 *	To avoid needing a buffer to hold the entire
 *	decoded data, buffers can be decoded in pieces
 */

/*
 *	Initialize for decoding in pieces
 */
void
rl_decode_init(src_buffer, src_size, escape)
unsigned char *src_buffer;	/* buffer		*/
unsigned long src_size;		/* size encoded		*/
unsigned char escape;		/* esc. code		*/
{
    rl_src_buffer = src_buffer;
    rl_src_size = src_size;
    rl_escape = escape;
    rl_src = (unsigned long)0;
    rl_count = (unsigned long)0;
}


/*
 *	Decode the next dst_size bytes from the buffer
 *	initialized before. If dst_buffer is NULL, dst_size
 *	bytes are skipped.
 *	returns:	no. of bytes written to dst_buffer
 */

unsigned long
rl_decode_next(dst_buffer, dst_size)
unsigned char *dst_buffer;	/* destination buffer	*/
unsigned long dst_size;		/* no. bytes to decode	*/
{
    unsigned long dst;

    dst = 0;
    while (rl_src <= rl_src_size && dst < dst_size) {
	while (rl_count && dst < dst_size) {
	    if (dst_buffer != (unsigned char *) 0)	/* skip */
		dst_buffer[dst] = rl_current;
	    dst++;
	    rl_count--;
	}
	if (rl_count == 0) {
	    if (rl_src_buffer[rl_src] == rl_escape) {
		rl_src++;
		rl_count  = UNLO(rl_src_buffer[rl_src++]);
		rl_count |= UNHI(rl_src_buffer[rl_src++]);
	    }
	    else {
		rl_count = 1;
	    }
	    rl_current = rl_src_buffer[rl_src++];
	}
    }
    return dst;
}
