/*
 *	pixmon - a PIXel oriented graphics MONitor
 *	(c) Joachim Sprave 1992
 */

/*
 *	C-lib headers
 */
#include <stdio.h>
#include <stdlib.h>

/*
 *	Various X headers
 */
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xos.h>
#include <X11/cursorfont.h>
#include <X11/Xutil.h>

/*
 *	Application headers
 */
#include "pixmon.h"
#include "defcmap.h"
#include "dither.h"
#include "rle.h"

void get_coor();
void get_colors();
void repaint();
void mapped();
void mouseinput();
void flushMap();
void dots2map();

/*
 *	Some X macros
 */
#define APPNAME			"pixmon"
#define APPCLASS		"PixMon"

/*
 *	Maximum number of colors used by PixMon
 */
#define MAX_COLORS		256

/*
 *	Scaling macros
 */ 
#define X_OM2WI(x)	((int)((x)*MyData.Xfactor+0.99))
#define Y_OM2WI(y)	((int)((y)*MyData.Yfactor+0.99))
#define X_WI2OM(x)	((int)((x)/MyData.Xfactor+0.5))
#define Y_WI2OM(y)	((int)((y)/MyData.Yfactor+0.5))

#define MIN(a,b)	((a)<(b)?(a):(b))
#define MAX(a,b)	((a)<(b)?(b):(a))


typedef struct {

	  int Om_width;
	  int Om_height;
	  int Wi_width;
	  int Wi_height;
	  int Wi_xpos;
	  int Wi_ypos;
	  float Scale;
	  float Xfactor;
	  float Yfactor;
	  char *Colormap;
	  Boolean Dither;
	  char *Butrep;
	  Pixel Fg, Bg;
	  char *Progname;
  }
ApplData;
typedef ApplData *ApplDataPtr;

/*
 *	Default values
 */
static int def_wi_width = 0;
static int def_wi_height = 0;
static int def_wi_xpos = 10;
static int def_wi_ypos = 10;
static int def_om_width = 100;
static int def_om_height = 100;
static float def_scale = 2.0;
static int no_of_colors = 2;

/*
 *	Global definitions
 */
static Pixmap stipple[N_PAT];
static Pixmap MyMap;
static unsigned long pixels[MAX_COLORS];
static Colormap cmap;

static int nargs;
static Arg wargs[32];
static Cursor WorkingCursor;
static Display *dpy;
static Window win;
static GC gc;
static GC cleargc;

static int isMapped = 0;
static int Ready2Exit = 0;

static Widget toplevel;
static Widget w;

/*
 *	buffer to hold the original dotmap
 */
static unsigned char *dotbuf;

/*
 *	buffer to read incoming packages
 */
static unsigned char *inpbuf;

/*
 * Application Data
 */
static ApplData MyData;

static IMGHDR fullhdr;

static XtResource application_resources[] =
{
	{"name", "Name", XtRString, sizeof(char *),
	 XtOffset(ApplDataPtr, Progname), XtRString, APPNAME},
	{"width", "Width", XtRInt, sizeof(int),
	 XtOffset(ApplDataPtr, Wi_width), XtRInt, (caddr_t) & def_wi_width},
	{"height", "Height", XtRInt, sizeof(int),
       XtOffset(ApplDataPtr, Wi_height), XtRInt, (caddr_t) & def_wi_height},
	{"xpos", "Xpos", XtRInt, sizeof(int),
	 XtOffset(ApplDataPtr, Wi_xpos), XtRInt, (caddr_t) & def_wi_xpos},
	{"ypos", "Ypos", XtRInt, sizeof(int),
	 XtOffset(ApplDataPtr, Wi_ypos), XtRInt, (caddr_t) & def_wi_ypos},
	{"scale", "Scale", XtRFloat, sizeof(float),
	 XtOffset(ApplDataPtr, Scale), XtRFloat, (caddr_t) & def_scale},
	{"dx", "DeltaX", XtRInt, sizeof(int),
	 XtOffset(ApplDataPtr, Om_width), XtRInt, (caddr_t) & def_om_width},
	{"dy", "DeltaY", XtRInt, sizeof(int),
       XtOffset(ApplDataPtr, Om_height), XtRInt, (caddr_t) & def_om_height},
	{"foreground", "Foreground", XtRPixel, sizeof(Pixel),
	 XtOffset(ApplDataPtr, Fg), XtRString, (caddr_t) "Black"},
	{"background", "Background", XtRPixel, sizeof(Pixel),
	 XtOffset(ApplDataPtr, Bg), XtRString, (caddr_t) "White"},
	{"cmap", "Cmap", XtRString, sizeof(char *),
	 XtOffset(ApplDataPtr, Colormap), XtRString, (caddr_t) "default"},
	{"dither", "Dither", XtRBoolean, sizeof(Boolean),
	 XtOffset(ApplDataPtr, Dither), XtRString, (caddr_t) "False"},
	{"butrep", "ButRep", XtRString, sizeof(char *),
	 XtOffset(ApplDataPtr, Butrep), XtRString, (caddr_t) 0},
};

static XrmOptionDescRec optionDescList[] =
{
	{"-width", "*width", XrmoptionSepArg, (caddr_t) & def_om_width},
	{"-height", "*height", XrmoptionSepArg, (caddr_t) & def_om_height},
	{"-scale", "*scale", XrmoptionSepArg, (caddr_t) & def_scale},
	{"-fg", "*foreground", XrmoptionSepArg, (caddr_t) NULL},
	{"-bg", "*background", XrmoptionSepArg, (caddr_t) NULL},
	{"-dx", "*dx", XrmoptionSepArg, (caddr_t) & def_om_width},
	{"-dy", "*dy", XrmoptionSepArg, (caddr_t) & def_om_height},
	{"-x", "*x", XrmoptionSepArg, (caddr_t) & def_wi_xpos},
	{"-y", "*y", XrmoptionSepArg, (caddr_t) & def_wi_ypos},
	{"-cmap", "*cmap", XrmoptionSepArg, (caddr_t) "default"},
	{"-dither", "*dither", XrmoptionNoArg, (caddr_t) "True"},
	{"-butrep", "*butrep", XrmoptionSepArg, (caddr_t) NULL},
};



int main(argc, argv)
int argc;
char *argv[];
{
	int i;
	XGCValues gcv;

	toplevel = XtInitialize(argv[0], APPCLASS,
	      optionDescList, XtNumber(optionDescList), & argc,
				argv);

	XtGetApplicationResources(toplevel, &MyData, application_resources,
	     XtNumber(application_resources), (ArgList) NULL, (Cardinal) 0);

	if (argc != 1) {
		(void) fprintf(stderr, "Usage: %s [Xt options]\n", argv[0]);
		exit(1);
	}
	fullhdr.x = fullhdr.y = 0;
	fullhdr.dx = MyData.Om_width;
	fullhdr.dy = MyData.Om_height;

	if (MyData.Wi_width == 0) {
		MyData.Wi_width = MyData.Om_width * MyData.Scale;
	}
	if (MyData.Wi_height == 0) {
		MyData.Wi_height = MyData.Om_height * MyData.Scale;
	}
	MyData.Xfactor = (float) MyData.Wi_width / MyData.Om_width;
	MyData.Yfactor = (float) MyData.Wi_height / MyData.Om_height;

	nargs = 0;
	XtSetArg(wargs[nargs], XtNwidth, MyData.Wi_width ? MyData.Wi_width : MyData.Om_width);
	nargs++;
	XtSetArg(wargs[nargs], XtNheight, MyData.Wi_height ? MyData.Wi_height : MyData.Om_height);
	nargs++;

	w = XtCreateManagedWidget(argv[0], widgetClass, toplevel,
				  wargs, XtNumber(wargs));

	dotbuf = (unsigned char *) calloc(MyData.Om_width * MyData.Om_height, sizeof(char));
	inpbuf = (unsigned char *) calloc(MyData.Om_width * MyData.Om_height, sizeof(char));

	if (dotbuf == NULL || inpbuf == NULL) {
		fprintf(stderr, "%s: cannot allocate enough memory...\n", MyData.Progname);
		exit(1);
	}

	XtAddEventHandler(w, (EventMask) (StructureNotifyMask | ExposureMask), False,
			  repaint, "repaint");
	XtAddEventHandler(w, (EventMask) ButtonPressMask, False,
			  mouseinput, "input_data");

	XtAddInput(fileno(stdin), XtInputReadMask, get_coor, w);

	XtRealizeWidget(toplevel);

	win = XtWindow(w);
	dpy = XtDisplay(w);

	cmap = DefaultColormap(dpy, DefaultScreen(dpy));

	MyMap = XCreatePixmap(dpy, win, MyData.Wi_width, MyData.Wi_height,
			      DefaultDepthOfScreen(XtScreen(w)));

	if (strcmp(MyData.Colormap, "raster") == 0) {
		no_of_colors = 2;
	} else {
		no_of_colors = DisplayCells(dpy, DefaultScreen(dpy));
	}

	WorkingCursor = XCreateFontCursor(dpy, XC_top_left_arrow);
	XDefineCursor(dpy, win, WorkingCursor);

	gcv.foreground = MyData.Fg;
	gcv.background = MyData.Bg;
	gcv.function = GXcopy;
	gc = XCreateGC(dpy, win, GCForeground | GCBackground
		       | GCFunction, &gcv);
	gcv.foreground = MyData.Bg;
	cleargc = XCreateGC(dpy, win, GCForeground | GCBackground
			    | GCFunction, &gcv);

	if (MyData.Dither) {
		no_of_colors = 2;
	}
	if (no_of_colors == 2) {
		XSetFillStyle(dpy, gc, FillOpaqueStippled);
		for (i = 0; i < N_PAT; i++) {
			stipple[i] = XCreateBitmapFromData(
						      dpy, win, fill_pat[i],
						     PAT_WIDTH, PAT_HEIGHT);
		}
	} else {
		if (no_of_colors > MAX_COLORS)
			no_of_colors = MAX_COLORS;
		get_colors();
	}
	dots2map(&fullhdr);
	flushMap(&fullhdr);

	XtMainLoop();
}



void
get_colors()
{
	int color, r, g, b;
	XColor xcolor;
	char cmapfile[256];
	char line[BUFSIZ];
	FILE *f;

	color = 0;
	f = fopen(MyData.Colormap, "r");
	if (f == NULL) {
		sprintf(cmapfile, "%s.cmap", MyData.Colormap);
		f = fopen(cmapfile, "r");
	}
	if (f != NULL) {
		while (color < MAX_COLORS
		       && fgets(line, BUFSIZ, f) != NULL) {
			if (*line != '#' && sscanf(line, "%d %d %d", &r, &g, &b) == 3) {
				rgb_default[color].red = r;
				rgb_default[color].green = g;
				rgb_default[color].blue = b;
				color++;
			}
		}
		fclose(f);
	}
	if (color < 256) {
		fprintf(stderr, "Color map '%s' contains %d colors.\n", cmapfile, color);
		fprintf(stderr, "PIXMON will use defaults for %d missing colors.\n", MAX_COLORS - color);
	}
	for (color = 0; color < MAX_COLORS; color++) {
		xcolor.red	= rgb_default[color].red << 8;
		xcolor.green	= rgb_default[color].green << 8;
		xcolor.blue	= rgb_default[color].blue << 8;
		xcolor.flags	= DoRed | DoGreen | DoBlue;
		XAllocColor(dpy, cmap, &xcolor);
		pixels[color]	= xcolor.pixel;
	}
}

void
setdots(x, y, dx, color)
int x, y, dx;
unsigned char color;
{


	if (no_of_colors > 2) {
		color = color % MAX_COLORS;
		XSetForeground(dpy, gc, pixels[color]);
	} else {
		color = color % N_PAT;
		XSetStipple(dpy, gc, stipple[color]);
	}
	XFillRectangle(dpy, MyMap, gc,
		       X_OM2WI(x), Y_OM2WI(y),
		       X_OM2WI(dx), Y_OM2WI(1));

}


void
dots2map(hdrp)
IMGHDR *hdrp;
{
	int x1, x, y, x2, y2;
	unsigned char *p, color;

	x2 = hdrp->x + hdrp->dx - 1;
	y2 = hdrp->y + hdrp->dy - 1;
	for (y = hdrp->y; y <= y2; y++) {
		p = dotbuf + (y * MyData.Om_width);
		x = hdrp->x;
		while (x <= x2) {
			x1 = x++;
			color = p[x1];
			while (x <= x2 && color == p[x])
				x++;
			setdots(x1, y, x - x1, color);
		}
	}
}

void
flushMap(hdrp)
IMGHDR *hdrp;
{
	XCopyArea(dpy, MyMap, win, gc,
		  X_OM2WI(hdrp->x), Y_OM2WI(hdrp->y),
		  X_OM2WI(hdrp->dx), Y_OM2WI(hdrp->dy),
		  X_OM2WI(hdrp->x), Y_OM2WI(hdrp->y));

}


unsigned short
swapbytes(x)
unsigned short x;
{
	unsigned short ret;

	ret = (x & 0x00FF) << 8;
	ret = ret | ((x & 0xFF00) >> 8);
	return ret;
}


void
get_coor(data, source, id)
caddr_t data;
int *source;
XtInputId id;
{
	int i;
	unsigned short magic, cigam;
	IMGHDR inp_hdr;
	int inp_hdr_size;
	int line;
	int size;

	unsigned char *dotptr, *inpptr;
	static XtWidgetGeometry xtgeom;
	static int dirty = 1;

	i = 50;
	magic = cigam = 0;
	while (fread(&magic, 1, sizeof(magic), stdin) == sizeof(magic)
	       && (magic != PIX_MAGIC && (cigam = swapbytes(magic)) != PIX_MAGIC) && --i);

	if (magic != PIX_MAGIC && cigam != PIX_MAGIC)
		return;

	size = fread(&inp_hdr, 1, sizeof(IMGHDR), stdin);

	if (cigam == PIX_MAGIC) {
		inp_hdr.type = swapbytes(inp_hdr.type);
		inp_hdr.x = swapbytes(inp_hdr.x);
		inp_hdr.y = swapbytes(inp_hdr.y);
		inp_hdr.dx = swapbytes(inp_hdr.dx);
		inp_hdr.dy = swapbytes(inp_hdr.dy);
		inp_hdr.sizelo = swapbytes(inp_hdr.sizelo);
		inp_hdr.sizehi = swapbytes(inp_hdr.sizehi);
	}
	inp_hdr_size = (inp_hdr.sizehi << (sizeof(short) * 8)) | inp_hdr.sizelo;

	if (size != sizeof(IMGHDR)) {
		fprintf(stderr, "%s: incomplete header received\n", MyData.Progname);
		return;
	}
	if (inp_hdr.type & PIX_KILL) {
		Ready2Exit = 1;
		return;
	}
	if (inp_hdr.type & PIX_DATA && inp_hdr.x + inp_hdr.dx > MyData.Om_width
	    || inp_hdr.y + inp_hdr.dy > MyData.Om_height) {
		fprintf(stderr, "%s: package doesn't fit (x=%d, y=%d, dx=%d, dy=%d --- ignored\n", 
			MyData.Progname, inp_hdr.x, inp_hdr.y, inp_hdr.dx, inp_hdr.dy);
		return;
	}
	/*
	 * Read all color values
	 */
	if (inp_hdr.type & PIX_DATA) {
		if (inp_hdr_size < inp_hdr.dx * inp_hdr.dy) {
			fread(inpbuf, sizeof(char), inp_hdr_size, stdin);

			rl_decode_init(inpbuf, inp_hdr_size, PIX_RLE);

			dotptr = dotbuf + inp_hdr.y * MyData.Om_width + inp_hdr.x;
			for (line = inp_hdr.y; line < inp_hdr.y + inp_hdr.dy; line++) {
				rl_decode_next(dotptr, inp_hdr.dx);
				dotptr += MyData.Om_width;
			}
		} else {
			fread(inpbuf, sizeof(char), inp_hdr_size, stdin);

			dotptr = dotbuf + inp_hdr.y * MyData.Om_width + inp_hdr.x;
			inpptr = inpbuf;

			for (line = inp_hdr.y; line < inp_hdr.y + inp_hdr.dy; line++) {
				memcpy(dotptr, inpptr, inp_hdr.dx);
				dotptr += MyData.Om_width;
				inpptr += inp_hdr.dx;
			}

		}
		dots2map(&inp_hdr);
		if (!(inp_hdr.type & PIX_FLUSH))
			dirty = 1;
	}
	if (inp_hdr.type & PIX_FLUSH) {
		if (dirty)
			flushMap(&fullhdr);
		else
			flushMap(&inp_hdr);
		dirty = 0;
	}
}




void
repaint(w, data, ev)
Widget w;
caddr_t data;
XEvent *ev;
{

	XEvent event;

	if (ev->type == MapNotify) {
		isMapped = TRUE;
		return;
	}
	if (ev->type == UnmapNotify) {
		isMapped = FALSE;
		return;
	}
	if (ev->type == ConfigureNotify) {

		MyData.Wi_width = ev->xconfigure.width;
		MyData.Wi_height = ev->xconfigure.height;
		MyData.Xfactor = MyData.Wi_width / (double) MyData.Om_width;
		MyData.Yfactor = MyData.Wi_height / (double) MyData.Om_height;

		XFreePixmap(dpy, MyMap);
		MyMap = XCreatePixmap(dpy, win, MyData.Wi_width, MyData.Wi_height,
				      DefaultDepthOfScreen(XtScreen(w)));

		dots2map(&fullhdr);

	} else if (ev->type == Expose) {

		if (!isMapped)
			return;
	}
	while (XCheckTypedEvent(dpy, Expose, &event));

	flushMap(&fullhdr);
}


void
mouseinput(w, data, ev)
Widget w;
caddr_t data;
XEvent *ev;
{
	int fd = 0;
	FILE *f;

	switch (ev->xany.type) {
	case ButtonPress:
		if (ev->xbutton.button == Button1) {
			if (Ready2Exit) {
				exit(0); /* brute force */
			}
		}
		break;
	default:
		break;
	}
}



/*** EOF ***/
