// -*- mode: c++; c-indent-level: 4; c++-member-init-indent: 8; comment-column: 45 -*-
// RAWGENE.HPP : Variable length genotype base class

// References: J.J. Merelo PhD; any GA book, like Goldberg, and Programming C++
//             Rules and recommendations, found somewhere on the net.


#ifndef __RAWGENE_HPP
#define __RAWGENE_HPP

#include "general.hpp"			     // General includes, and some defs


const unsigned char Wgts[9] = { 0,128,64,32,16,8,4,2,1 }; 
					     // 1st number does not mind
const unsigned char BITSINBYTE = 8;
const unsigned char LASTBYTEN = BITSINBYTE - 1;
const unsigned MAX8BITS = 256;



// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=---
// -=-=-=-=-=-=-=-=-=-=-=-=-- Class definition -=-=-=-=-=-=-=-=-=-=-=-=-=-=---
// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=---

class rawGene {
      
public:

    
    // several constructors, and one destructor

    rawGene( void ) {};			     // default constructor


    // more or less random constructors
    rawGene(  unsigned _len );
    rawGene( const char *inStr );
    rawGene( const char *_str, unsigned _length, float _mutationRate );
  
      
    // "binary" constructors, from two loving parents
    rawGene(const rawGene &rawGene1, 
	    const rawGene &rawGene2,
	    double xOverRate,		     // normal reproduction
	    double MutationRate );	     // with uniform xover

    rawGene(const rawGene &rawGene1, 
	    const rawGene &rawGene2, 
	    double xOverRate );		     // reproduction sans mutation

    rawGene(const rawGene &rawGene1, 
	    const rawGene &rawGene2, 
	    unsigned _startPoint, unsigned _endPoint, 
	    double _mutationRate);	     // 2-point crossover
      
    // "unary" constructors, from batchelor parent or from an amoeba
    rawGene(const rawGene &rawGene1, 
	    double MutationRate);
      
    // COPY constructor
    rawGene(const rawGene& _inRGen );

    // obviously, destructor
    virtual ~rawGene() {
      delete []str;
    };


// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- 
//   Common functions to all the "gene" hierarchy
// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

    // info functions
    virtual unsigned    getLength( void ) const { return length;} ;
                                                   // number of "genes"
   
    // conversion operator
    operator char*( void ) const{ return str;};
                  
    // overloaded operators
    const char          operator [] ( unsigned _idx ) const {
	return str[_idx];
    };	 

                                                            
    // Friend declaration
friend ostream&         operator << ( ostream& s, const rawGene & _inGen );
  
// Particular info function
    unsigned            getNumBytes( void ) const { return length; };

// -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=--      
protected:

    // definition of enumerated types and consts for the whole hierarchy
      
    void   checkNassign( unsigned _len);                        
    void   mutate( double _mutationRate);    // mutates with given mutationRate  
    // genetic operators
  
    void   changeBit(unsigned long chBit);   // Changes said bit

    void   swapBit(unsigned long swBit, 
		   const char* inStr);	     // Swaps bits with the incoming str

    void   swapChunkByte(unsigned char _scbStart,
			 unsigned char _scbEnd, 
			 unsigned _idxByte,  // Swaps some bits with the in
			 char _scbByte);     // byte
                                  
    void   swapChunk(unsigned long _swBitStart,
		     unsigned long _swBitEnd, // Swaps some bits with the in
		     const char *inStr );     // string

    void   transposeChunk(unsigned long _trBitStart,
			  unsigned long _trBitEnd,
			  unsigned long _lenBits );
				// interchanges bits from one position of the genome to the other
    // variable-length operators
    void   addChunkEnd( unsigned _size ); // adds a chunk of said size at
					  // end
    void   killChunk( unsigned _size, unsigned _pos );
					     // Eliminates some bits from _pos
					     // position
    void   dupChunk( unsigned _size, unsigned _pos );
					     // Duplicates some bits from that
					     // position

    // functions for splicing, including shift
    static void         shRight( char * _inStr, unsigned _pos, unsigned _len );
					     // shifts a string some bits to
					     // the right
    static void         shLeft( char *_inStr, unsigned _pos, unsigned _len);
					     // Idem, to the right

    void                splice(unsigned long _insertBit, 
			       const char* _inStr,
			       unsigned _numBits );
				// insert that many _numBits from the
				// string in _insertBit position

    void                spliceInto(unsigned long _insertBit, 
				   const char* _inStr,
				   unsigned long _insertBit2,
				   unsigned _numBits );
				// insert that many _numBits from the
				// string in _insertBit position, starting
				// by the _insertBit2 position

    // access to private variables
    const char*         getGenome( void ) const      { return str;} ;      
    unsigned            getStr( unsigned _idx ) const { return str[_idx];};
    
      
//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

private:    // instance variables
    char        *str;             // contains the chromosome itself
    unsigned    length;           // length of the genome, in bytes


};

typedef rawGene* pRawGene;


// friendly operators
// -=-=-=-=-=-=-=-=--
ostream& operator << ( ostream& s, const rawGene & _inGen );


#endif
