% TRUKCON2 Simulates fuzzy controlled truck.
%   This script is referred to in Chapter 5.
%
% USES: trukmod, trukplot

% Mark Beale 6-24-93
% Copyright (c) 1993 by PWS Publishing Company

% FUZZY SETS
% ==========

% HORIZONTAL POSITION: 5 shapes spaced by hand 
hposS = 0:5:100;
hposL = group([0   0  10  35],[30  40  50], ...
  [40  50  60],[50  60  70],[65 90 100 100]);
hposG = shapeval(hposL,hposS);

% TRUCK ANGLE: 7 Equally spaced triangles
deg = (pi/180);
angleS = [0:4:360]*deg;
angleL = eqshape(3,angleS,7); 
angleG = shapeval(angleL,angleS);

% STEERING ANGLE: 7 Equally spaced triangles
steeringS = [-30:1:30]*deg;
steeringL = eqshape(3,steeringS,7);
steeringG = shapeval(steeringL,steeringS);

% PLOT FUZZY SETS
% ===============

clc
disp('Hit any key to see Fuzzy Sets for Horizontal Position...')
pause
plotfs(hposS,'Fuzzy Sets for Horizontal Position', ...
  'Horizontal Position in meters')
plot(hposS,hposG)
hold off
pause

clc
disp('Hit any key to see Fuzzy Sets for Vertical Position...')
pause
plotfs(angleS/deg,'Fuzzy Sets for Truck Angle', ...
  'Truck Angle in Degrees')
plot(angleS/deg,angleG)
hold off
pause

clc
disp('Hit any key to see Fuzzy Sets for Steering Angle...')
pause
plotfs(steeringS/deg,'Fuzzy Sets for Steering Angle', ...
  'Steering Angle in Degrees')
plot(steeringS/deg,steeringG)
hold off
pause

% DEFINITION OF CONTROLLER
% ========================
% Decides steering angle based on truck horizontal position and angle:

% ANTECEDANT TABLE
A = table(1:5,1:7);

% CONSEQUENCE TABLE, Rows represent position, Columns represent angle
C = [5; 6; 6; 7; 7;
     3; 5; 6; 7; 7;
     2; 3; 6; 6; 7;
     2; 2; 4; 6; 6;
     1; 2; 2; 5; 6;
     1; 1; 2; 3; 5;
     1; 1; 2; 2; 3;];

% GROUP OF FUZZY SETS
Z = group(hposS,hposG,angleS,angleG,steeringS,steeringG);

% SIMULATION OF CONTROLLED TRUCK
% ==============================

% INITIAL STATE
s0 = [45; -60; 360*deg];

% Horizontal position = 50 meters
% Vertical position = -60 meters
% Truck angle = 360 degrees

% TIME
t0 = 0;
t1 = 200;
dt= 0.2;
T = (t0+dt):dt:t1;
steps = length(T);

S = zeros(3,steps);
U = zeros(1,steps);

clc
disp('Hit any key to see the controller at work...')
pause

s = s0;
trukplot(s,0)
title('Controlled Truck: Controller 2');

for i=1:steps

  % RE-EVALUATE CONTROLLER EVERY 5 TIME STEPS
  if rem(i,5) == 1
    u = frule(A,C,Z,[s(1); s(3)]);
  end

  % UPDATE STATE
  s = s + dt * trukmod(s,u);

  % PLOT EVERY 5 TIME STEPS
  if rem(i,10) == 1
    trukplot(s,u,'add')
  end

  % SAVE VALUES
  S(:,i) = s;
  U(i) = u;

  % FINISH IF HORIZONTAL POSITION >= 0
  if all(s(2) >= 0)
    steps = i;
    S = S(:,1:steps);
    U = U(1:steps);
    T = T(1:steps);
    t1 = t0+steps*dt;
    break
  end
end

% PLOT RESPONSES
% ==============

% HORIZONTAL POSITION RESPONSE
clc
disp('Hit any key to see the Horizontal Position Response...')
pause
plot([t0 T],[s0(1) S(1,:)])
hold on
plot([t0 t1],[50 50],':')
hold off
title('Horizontal Position Response')
xlabel('Time (Seconds)')
ylabel('Horizontal Position (Meters)')
pause

% VERTICAL POSITION RESPONSE
clc
disp('Hit any key to see the Vertical Position Response...')
pause
plot([t0 T],[s0(2) S(2,:)])
hold on
plot([t0 t1],[0 0],':')
hold off
title('Vertical Position Response')
xlabel('Time (Seconds)')
ylabel('Vertical Position (Meters)')
pause

% TRUCK ANGLE RESPONSE
clc
disp('Hit any key to see the Truck Angle Response...')
pause
plot([t0 T],[s0(3) S(3,:)]/deg)
hold on
plot([t0 t1],[180 180],':')
hold off
title('Truck Angle Response')
xlabel('Time (Seconds)')
ylabel('Truck Angle (Degrees)')
pause

% STEERING ANGLE RESPONSE
clc
disp('Hit any key to see the Steering Angle Response...')
pause
plot(T,U/deg)
hold on
plot([t0 t1],[0 0],':')
hold off
title('Steering Angle Response')
xlabel('Time (Seconds)')
ylabel('Steering Angle (Degrees)')
pause
clear
