function y = shapeval(v,s)
% SHAPEVAL Shape fuzzy set evalation.
%
% SHAPEVAL(L,S)
%   L - Group of shape limits.
%   S - Support values at which to evaluate shape.
% Returns grade.
%
% L can define any one of these shapes:
%   Singleton: L = [Position]
%   Rectangle: L = [Left Right]
%   Triangle:  L = [Left Peak Right]
%   Trapezoid: L = [Left L_Peak R_Peak Right]
%   Bump:      L = [Left L_Mid Peak R_Mid Right]
%   Flat Bump: L = [Left L_Mid L_Peak R_Peak R_Mid Right]
%
% SHAPEVAL(Z,S)
%   Z - Limits of GROUP of shapes.
%   S - Support values at which to evaluate shapes.
% Returns grades.
%
% EXAMPLE: z = group([1 2],[3],[4 5 6 7],[8 9 10]);
%          s = 0:.1:10;
%          g = shapeval(z,s);
%          plotfs(s,'Fuzzy Set Shapes')
%          plot(s,g)
%          hold off
%
% SEE ALSO: eqshape, fuzzyval

% Mark Beale 6-24-93
% Copyright (c) 1993 by PWS Publishing Company

[sr,sc] = size(s);

if ~isgroup(v)
  [vr,vc] = size(v);

  if vc == 1
    y = single(s,v);
  elseif vc == 2
    y = rectangl(s,v);
  elseif vc == 3
    y = triangle(s,v);
  elseif vc == 4
    y = trapzoid(s,v);
  elseif vc == 5
    y = bump(s,v);
  elseif vc == 6
    y = flatbump(s,v);
  else
    error('SHAPEVAL: Unrecognized shape.')
  end

else
  ns = ungroup(v);
  y = zeros(ns,sc);

  for i=1:ns
    vi = ungroup(v,i);
    [vr,vc] = size(vi);

    if vc == 1
      y(i,:) = single(s,vi);
    elseif vc == 2
      y(i,:) = rectangl(s,vi);
    elseif vc == 3
      y(i,:) = triangle(s,vi);
    elseif vc == 4
      y(i,:) = trapzoid(s,vi);
    elseif vc == 5
      y(i,:) = bump(s,vi);
    elseif vc == 6
      y(i,:) = flatbump(s,vi);
    else
      error('SHAPEVAL: Unrecognized shape.')
    end
  end
end
