/*
 * Copyright (c) 1992 The University of Utah and
 * the Center for Software Science (CSS).  All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the Center
 * for Software Science at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSS ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSS DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSS requests users of this software to return to css-dist@cs.utah.edu any
 * improvements that they make and grant CSS redistribution rights.
 *
 *      Utah $Hdr: trap_member.h 1.2 93/09/07$
 *      Author: Doug Orr, University of Utah CSS
 */

#ifndef	_trap_member_h_
#define	_trap_member_h_	 1

/*  trap_member:  "triangular" membership functions.  */

#include "geom_member.h"

template<class C> class trap_member : public geom_member<C> {
	C lx;		// base x
	C base1;	// base width
	C mx;		// cap x 
	C base2;	// cap width
	float h;	// height

	float s1;	// left triangle slope/y-int
	C b1;	
	float s2;	// right triangle slope/y-int
	C b2;

	/* cached copies of computed values */
	C a;							// area
	C c;       						// centroid
	C m;							// moment
	
	float			calc_area () const;
	float			calc_centroid () const;
	C			rx () const			{return lx+base1;}

    public:
	trap_member (C lx_, C mx1_, C mx2_, C rx_, float h, char * n_ = NULL);
	~trap_member ();

	degree 			is (const C value) const;
	C	 		output (const degree d) const;

	C			area () const			{return a;}
	C			centroid () const		{return c;}
	C			moment () const			{return m;}

	geom_member<C> * 	scale (const degree d) const;
	geom_member<C> * 	clip (const degree d) const;

};


template<class C>
trap_member<C>::trap_member (C lx_, C mx1_, C mx2_, C rx_, float h_, char * n_ = NULL) :
	lx (lx_),
	base1 (rx_-lx_),
	mx (mx1_),
	h (h_),
	base2 (mx2_-mx1_),
	geom_member<C> (NewStr (n_))
{

	/* calculate slopes and y-intercepts */
	s1 = (mx != lx) ?
		h / (mx - lx)
			: 0.0;		// infinity, really...
	
	s2 = (rx() != mx+base2)
		? h / (mx+base2 - rx())
			:  0.0;
	
	b1 = (C)(-s1 * lx);
	b2 = (C)(-s2 * rx());

	/* cache a copy of the area, centroid, and moment */
	a = (C)calc_area ();
	float cent = calc_centroid ();
	c = (C)cent;
	m = (C)(a * cent);
}

template<class C>
trap_member<C>::~trap_member ()
{
}



#endif	/* _trap_member_h_ */
