/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




#ifndef regel_h
#define regel_h

#include <OS/string.h>

#include "texte.h"
#include "lingvar.h"

/*
 * wegen dauernder Compiler-Warnings wird die Zeile 
 *           'enum RegelAusdruck { Praemisse, Konklusion };'
 * durch folgende Konstruktion ersetzt :
 * (wichtig : fuer die Werte, die bei 'define' benutzt werden, muss gelten :
 *            Praemisse = Konklusion - 1   )
 */
typedef int RegelAusdruck;
#define Praemisse   0
#define Konklusion  1

/*
 *------------------------------------------------------------------------------
 * Klasse : Bedingung
 * Zweck  : Definiert einen Ausdruck der Form : Variable = Fuzzy-Set
 *          Hierbei werden sowohl die Variable als auch das Fuzz-Set durch
 *          Strings dargestellt.
 *------------------------------------------------------------------------------
 */
class Bedingung
{
  public : Bedingung();
           Bedingung(String lingvar_name, String fuzzyset_name);
           Bedingung(const Bedingung& bedingung);

           String lingvar() const;
           void lingvar(String name);

           String fuzzyset() const;
           void fuzzyset(String name);

           Bedingung& operator=(const Bedingung&);
           boolean operator!=(const Bedingung&) const;
           boolean operator==(const Bedingung&) const;
  private: CopyString _lingvar_name,
                      _fuzzyset_name;
};

inline String Bedingung::lingvar() const { return _lingvar_name; }

inline void Bedingung::lingvar(String name) { _lingvar_name = name; }

inline String Bedingung::fuzzyset() const { return _fuzzyset_name; }

inline void Bedingung::fuzzyset(String name) { _fuzzyset_name = name; }


/*
 *------------------------------------------------------------------------------
 * Klasse : Regel
 * Zweck  : Definition einer linguistischen Regel.
 *          Die ling. Variablen und die Fuzzy-Sets werden dabei durch ihre
 *          Namen (also durch Strings) dargestellt. Zu diesem Zweck enthaelt
 *          eine Regel je ein Array fuer die 'Bedingungen' in Praemisse und
 *          Konklusion.
 *------------------------------------------------------------------------------
 */

class Regel
{
  public : Regel(int groesse_praemisse = 1, int groesse_konklusion = 1);
           Regel(const Regel& regel);
           ~Regel();

           Bedingung* bedingung(RegelAusdruck, int pos) const;
           void bedingung(RegelAusdruck,int pos, String var, String fuzzy);
           boolean enthalten(RegelAusdruck, Bedingung*) const;
           boolean enthalten(RegelAusdruck, String lingvar) const;
           int groesse(RegelAusdruck) const;

           String fuzzyset(RegelAusdruck, String lingvar);
           String lingvar(RegelAusdruck, String fuzzyset);

           Regel& operator=(const Regel&);
           boolean operator==(const Regel&);
  private: int _gr_praemisse,
               _gr_konklusion;
           Bedingung* _praemisse;
           Bedingung* _konklusion;
};

declarePtrList(RegelListe, Regel)

inline Regel::~Regel()
{
  delete[] _praemisse;
  delete[] _konklusion;
}


/*
 *------------------------------------------------------------------------------
 * Klasse : RegelBasis
 * Zweck  : Definiert eine Menge von Regeln.
 *          Die einzelnen Regeln werden hierbei mit Hilfe der Klasse 'PtrList'
 *          gespeichert.
 *------------------------------------------------------------------------------
 */
class RegelBasis
{
  public : RegelBasis() {};
           RegelBasis(const RegelBasis& regel_basis);
           ~RegelBasis();
           Regel* hol_regel(int nr) const;
           boolean del_regel(Regel* regel);
           boolean del_regel(int nr);
           void add_regel(Regel* regel);
           int anzahl_regeln() const;
           boolean such(Regel*, int& nr);
           RegelBasis& operator=(const RegelBasis&);
  private: RegelListe _regel_liste;
};

inline RegelBasis::RegelBasis(const RegelBasis& r_basis) { *this = r_basis; }

inline int RegelBasis::anzahl_regeln() const { return _regel_liste.count(); }

inline Regel* RegelBasis::hol_regel(int nr) const
{
  if(nr >= 0 && nr < anzahl_regeln())
    return _regel_liste.item(nr);
  else
    return nil;
}

#endif
