/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




#include <stdio.h>

#include <IV-look/kit.h>
#include <InterViews/layout.h>
#include <InterViews/action.h>
#include <InterViews/border.h>
#include <InterViews/monoglyph.h>

#include "texte.h"
#include "protokoll.h"

/*
 *------------------------------------------------------------------------------
 * Klasse : GrafWerteDisplay
 * Zweck  : Stellt Werte grafisch in einem Koordiantensystem dar. Dem
 *          Konstruktor wird eine Breite b uebergeben, die besagt, dass die
 *          letzten b Werte im Koordiantensystem dargestellt werden.
 *------------------------------------------------------------------------------
 */

class GrafWerteDisplay : public MonoGlyph
{
  public : GrafWerteDisplay(int breite = 20);
           ~GrafWerteDisplay();
           virtual void draw(Canvas*, const Allocation&) const;

           void display(float);
           void loeschen();
           void neue_skala(DimensionName, float min, float max);

  private: KooSystem* _grafik;
           Coord* _werte;               // alle Werte in einem Array merken
           int _breite,                 // Groesse des Arrays
               _pos;                    // erster freier Platz im Array
};

GrafWerteDisplay::GrafWerteDisplay(int breite)
{
  Coord mindest_hoehe = 150;

  _breite = breite;
  _grafik = new KooSystem(0, _breite, -1, 1);
  _werte  = new Coord[_breite];
  loeschen();

  LayoutKit* lkit = LayoutKit::instance();
  body(lkit->v_fixed_span(_grafik, mindest_hoehe));
}

GrafWerteDisplay::~GrafWerteDisplay() { delete[] _werte; }

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : GrafWerteDisplay::draw(Canvas*, Allocation&)
 * Parameter : Canvas*     - gesamter Bildschirmplatz
 *             Allocation& - zugewiesener Anteil des Bildschirmplatzes
 * Zweck : zeichnet den GrafWerteDisplay im zugewiesenen Bildschirmbereich.
 *------------------------------------------------------------------------------
 */
void GrafWerteDisplay::draw(Canvas* c, const Allocation& a) const
{
  MonoGlyph::draw(c, a);

  int start = (_pos == 0) ? _breite : _pos;
  for(int i = 0 ; i < _breite - 1; i++)
    if(--start == 0) {
      _grafik->linie(i, _werte[0], i + 1, _werte[_breite - 1]);
      start = _breite;
    } else
      _grafik->linie(i, _werte[start], i + 1, _werte[start - 1]);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : GrafWerteDisplay::display(float)
 * Parameter : float - darzustellender Wert
 * Zweck : zeichnet angebenen Wert ins Koordinatensystem.
 *------------------------------------------------------------------------------
 */
void GrafWerteDisplay::display(float wert)
{
  _werte[_pos] = wert;
  _pos         = (_pos + 1) % _breite;        // d.h. _pos laeuft immer rund
  _grafik->aktualisiere();
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : GrafWerteDisplay::loeschen()
 * Zweck : loescht alle Werte im Koordinatensystem.
 *------------------------------------------------------------------------------
 */
void GrafWerteDisplay::loeschen()
{
  for(int i = 0; i < _breite; _werte[i++] = 0);
  _pos = 0;
  _grafik->aktualisiere();
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : GrafWerteDisplay::neue_skala(DimensionName, float, float)
 * Parameter : DimensionName - welche Achse ?
 *             float min     - unterer Wert
 *             float max     - oberer Wert
 * Zweck : skaliert Achse mit gegebenem Wertebereich.
 *------------------------------------------------------------------------------
 */
void GrafWerteDisplay::neue_skala(DimensionName dim, float min, float max)
{
  _grafik->neue_skala(dim, min, max);
}


/*
 *------------------------------------------------------------------------------
 * Klasse : GrafWerteBrowser
 * Zweck  : Stellt die Fuzzy-Sets eines LingVarSatz'es grafisch dar. Dabei kann
 *          jeweils nur eine ling. Variable zur Zeit dargestellt werden.
 *          Die darstellbaren Variablen werden durch einen NfcStrBrowser zur
 *          Auswahl gestellt.
 *------------------------------------------------------------------------------
 */
class GrafWerteBrowser : public WerteBrowser
{
  public : GrafWerteBrowser(const LingVarSatz* steuer_var);
           virtual Glyph* glyph();
           virtual void eingabe_var(int var_nr, FuzzyTyp wert);
           virtual void ausgabe_var(int var_nr, FuzzyTyp wert);
           virtual void fehler_var(FuzzyTyp fehler);

  // lokale Funktionen
  private: void _init(const LingVarSatz* steuer_var);
           void _neuer_wert();

  // lokale Variablen
  private: const LingVarSatz* _steuer_var;
           GrafWerteDisplay* _var_display;    // zur Darstellung der Werte
           NfcStrBrowser* _display_tafel;     // zur Auswahl der mgl. Variablen
           int _display_var,                  // Nr. der ausgewaehlten Variable
               _anz_display_var,              // Anzahl der mgl. Variablen
               _ausgabevar_offset;            // Anzahl der Eingabe-Variablen
           Glyph* _glyph;
};

declareActionCallback(GrafWerteBrowser);
implementActionCallback(GrafWerteBrowser);

GrafWerteBrowser::GrafWerteBrowser(const LingVarSatz* steuer_var)
{
  _var_display = new GrafWerteDisplay;

  Action* cb = new ActionCallback(GrafWerteBrowser)(
                 this, &GrafWerteBrowser::_neuer_wert
               );
  _display_tafel = new NfcStrBrowser(MAXNAME, 5, cb);

  _init(steuer_var);

  Coord rand      = 10;
  LayoutKit* lkit = LayoutKit::instance();
  WidgetKit* wkit = WidgetKit::instance();
  NfcKit* nkit    = NfcKit::instance();

  _glyph = nkit->text_drauf(
             Txt(158),
             lkit->margin(
               lkit->vbox(
                 lkit->hmargin(
                   lkit->h_variable_span(
                     new Border(
                       lkit->vcenter(_display_tafel), wkit->foreground()
                     ), 0, 0
                   ), 0, fil, 0, 0, fil, 0
                 ), lkit->vglue(rand, 0, 0),
                 new Border(
                   lkit->vcenter(
                     lkit->hnatural(_var_display, 200)
                   ), wkit->foreground()
                 )
               ), rand, rand
             ), true
           );
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : GrafWerteBrowser::glyph()
 * Rueckgabewert : dem GrafWerteBrowser entsprechendes Grafikelement.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
Glyph* GrafWerteBrowser::glyph() { return _glyph; }

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : GrafWerteBrowser::eingabe_var(int, FuzzyTyp)
 * Parameter : int      - Nummer der Eingabe-Variable
 *             FuzzyTyp - darzustellender Wert
 * Zweck : Stellt Wert fuer angegebene Eingabe-Variable dar, falls diese
 *         gerade ausgewaehlt ist.
 *------------------------------------------------------------------------------
 */
void GrafWerteBrowser::eingabe_var(int var_nr, FuzzyTyp wert)
{
  if(_display_var == var_nr)
    _var_display->display(wert);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : GrafWerteBrowser::ausgabe_var(int, FuzzyTyp)
 * Parameter : int      - Nummer der Ausgabe-Variable
 *             FuzzyTyp - darzustellender Wert
 * Zweck : Stellt Wert fuer angegebene Ausgabe-Variable dar, falls diese
 *         gerade ausgewaehlt ist.
 *------------------------------------------------------------------------------
 */
void GrafWerteBrowser::ausgabe_var(int var_nr, FuzzyTyp wert)
{
  if(_display_var == var_nr + _ausgabevar_offset)
    _var_display->display(wert);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : GrafWerteBrowser::fehler_var(FuzzyTyp)
 * Parameter : FuzzyTyp - darzustellender Wert
 * Zweck : Stellt Wert fuer Fehler-Variable dar, falls diese gerade ausgewaehlt
 *         ist.
 *------------------------------------------------------------------------------
 */
void GrafWerteBrowser::fehler_var(FuzzyTyp fehler)
{
  if(_display_var == _anz_display_var - 1)
    _var_display->display(fehler);
}

/*
 *-------------------------- lokale Funktionen ---------------------------------
 */

void GrafWerteBrowser::_init(const LingVarSatz* steuer_var)
{
  _steuer_var = steuer_var;

  _anz_display_var = _steuer_var->anz_lingvars() + 1;
  _display_var     = _anz_display_var - 1;

  // Eintraege im Auswahlfeld loeschen
  while(_display_tafel->anzahl() > 0)
    _display_tafel->loeschen(0);

  // zuerst die Eingabe-Variablen ins Auswahlfeld ...
  int anzahl = _steuer_var->anz_lingvars(EingabeVar);
  for(int i = 0; i < anzahl; i++)
    _display_tafel->zur_auswahl(
      _steuer_var->hol_lingvar(EingabeVar, i)->name()
    );

  // ... dann die Ausgabe-Variablen ...
  anzahl = _steuer_var->anz_lingvars(AusgabeVar);
  for(i = 0; i < anzahl; i++)
    _display_tafel->zur_auswahl(
      _steuer_var->hol_lingvar(AusgabeVar, i)->name()
    );

  // ... und zuletzt die Fehler-Variable
  _display_tafel->zur_auswahl(FehlerVarName);
  _display_tafel->markieren(_display_var);

  _ausgabevar_offset = _steuer_var->anz_lingvars(EingabeVar);

  _display_tafel->aktualisiere();
}

void GrafWerteBrowser::_neuer_wert()
{
  int auswahl = _display_tafel->auswahl_nr();
  if(_display_var != auswahl) {
    _display_var = auswahl;
    _var_display->loeschen();
    if(_display_var == _anz_display_var - 1)

      // Fehler-Variable ausgewaehlt
      _var_display->neue_skala(Dimension_Y, -1, 1);
    else {
      LingVar* var;
      if(_display_var < _ausgabevar_offset)

        // Eingabe-Variable ausgewaehlt
        var = _steuer_var->hol_lingvar(EingabeVar, _display_var);
      else

        // Ausgabe-Variable ausgewaehlt
        var = _steuer_var->hol_lingvar(
                AusgabeVar, _display_var - _ausgabevar_offset
              );
      _var_display->neue_skala(Dimension_Y, var->min(), var->max());
    }
  }
}


/*
 *------------------------------------------------------------------------------
 * Klasse : NumWerteBrowser
 * Zweck  : Stellt die Fuzzy-Sets eines LingVarSatz'es numerisch dar. Dabei
 *          koennen, im Gegensatz zum grafischen Werte-Browser, alle Variablen
 *          gleichzeitig dargestellt werden.
 *------------------------------------------------------------------------------
 */

class NumWerteBrowser : public WerteBrowser
{
  public : NumWerteBrowser(const LingVarSatz* steuer_var);
           virtual ~NumWerteBrowser();
           virtual Glyph* glyph();
           virtual void eingabe_var(int var_nr, FuzzyTyp wert);
           virtual void ausgabe_var(int var_nr, FuzzyTyp wert);
           virtual void fehler_var(FuzzyTyp fehler);
  private: NfcLabel **_eingabe_display,
                    **_ausgabe_display,
                    *_fehler_display;
           int _anz_eingabe, _anz_ausgabe;
           Glyph* _glyph;
           char _buffer[50];
};

NumWerteBrowser::NumWerteBrowser(const LingVarSatz* steuer_var)
{
  Coord rand      = 10;
  Coord breite    = 70;        // Breite des Ausgabefeldes fuer Werte
  LayoutKit* lkit = LayoutKit::instance();
  WidgetKit* wkit = WidgetKit::instance();
  NfcKit* nkit    = NfcKit::instance();

  _anz_eingabe = steuer_var->anz_lingvars(EingabeVar);
  _anz_ausgabe = steuer_var->anz_lingvars(AusgabeVar);

  _eingabe_display = new NfcLabel*[_anz_eingabe];
  _ausgabe_display = new NfcLabel*[_anz_ausgabe];

  for(int i = 0; i < _anz_eingabe; i++)
    _eingabe_display[i] = new NfcLabel(LeerStr, wkit);
  for(i = 0; i < _anz_ausgabe; i++)
    _ausgabe_display[i] = new NfcLabel(LeerStr, wkit);
  _fehler_display = new NfcLabel(LeerStr, wkit);

  Glyph* basis = lkit->vbox();

  // Eingabe-Display erstellen
  for(i = 0; i < _anz_eingabe; i++)
    basis->append(
      lkit->hbox(
        lkit->hglue(rand, 0, 0),
        wkit->label(steuer_var->hol_lingvar(EingabeVar, i)->name().string()),
        lkit->hglue(rand, fil, 0),
        new Border(
          lkit->margin(lkit->hnatural(_eingabe_display[i], breite), 2),
          wkit->foreground()
        ), lkit->hglue(rand, 0, 0)
      )
    );

  // Ausgabe-Display erstellen
  for(i = 0; i < _anz_ausgabe; i++) 
    basis->append(
      lkit->hbox(
        lkit->hglue(rand, 0, 0),
        wkit->label(steuer_var->hol_lingvar(AusgabeVar, i)->name().string()),
        lkit->hglue(rand, fil, 0),
        new Border(
          lkit->margin(lkit->hnatural(_ausgabe_display[i], breite), 2),
          wkit->foreground()
        ), lkit->hglue(rand, 0, 0)
      )
    );

  // Fehler-Display erstellen
  basis->append(
      lkit->hbox(
        lkit->hglue(rand, 0, 0),
        wkit->label(FehlerVarName),
        lkit->hglue(rand, fil, 0),
        new Border(
          lkit->margin(lkit->hnatural(_fehler_display, breite), 2),
          wkit->foreground()
        ), lkit->hglue(rand, 0, 0)
      )
  );

  _glyph = nkit->text_drauf(Txt(158), basis, true);
}

NumWerteBrowser::~NumWerteBrowser()
{
  delete[] _eingabe_display;
  delete[] _ausgabe_display;
}

/*
 *------------------------------------------------------------------------------
 * Die folgenden Funktionen haben die selben Aufgaben, wie die entsprechenden
 * Funktionen vom 'GrafWerteBrowser'.
 *------------------------------------------------------------------------------
 */
Glyph* NumWerteBrowser::glyph() { return _glyph; }

void NumWerteBrowser::eingabe_var(int var_nr, FuzzyTyp wert)
{
  if(var_nr >= 0 && var_nr < _anz_eingabe) {
    sprintf(_buffer, "%g", wert);
    _eingabe_display[var_nr]->text(_buffer);
    _eingabe_display[var_nr]->aktualisiere();
  }
}

void NumWerteBrowser::ausgabe_var(int var_nr, FuzzyTyp wert)
{
  if(var_nr >= 0 && var_nr < _anz_ausgabe) {
    sprintf(_buffer, "%g", wert);
    _ausgabe_display[var_nr]->text(_buffer);
    _ausgabe_display[var_nr]->aktualisiere();
  }
}

void NumWerteBrowser::fehler_var(FuzzyTyp fehler)
{
  sprintf(_buffer, "%g", fehler);
  _fehler_display->text(_buffer);
  _fehler_display->aktualisiere();
}

/*
 *------------------------------------------------------------------------------
 *-------------------- Definition der Klasse : LingVarBrowser ------------------
 *------------------------------------------------------------------------------
 */

declareActionCallback(LingVarBrowser);
implementActionCallback(LingVarBrowser);

LingVarBrowser::LingVarBrowser()
{
  _lingvarsatz = nil;
  _merk_var_nr = -1;         // Nummer der momentan dargestellten Variablen
  _markiert    = -1;         // Nummer des momentan markierten Fuzzy-Sets

  Action* cb = new ActionCallback(LingVarBrowser)(
                 this, &LingVarBrowser::_var_auswahl
               );
  _var_tafel = new NfcStrBrowser(MAXNAME, 5, cb);
  _grafik    = new KooSystem(-1, 1, 0, 1);

  Coord rand      = 10;
  LayoutKit* lkit = LayoutKit::instance();
  WidgetKit* wkit = WidgetKit::instance();
  NfcKit* nkit    = NfcKit::instance();
  body(
    nkit->text_drauf(Txt(140),
      lkit->hbox(
        lkit->margin(
          new Border(
            lkit->h_variable_span(lkit->center(_var_tafel), 0, 0),
            wkit->foreground()
          ),
          rand, 0, 0, rand / 2, 0, 0,
          rand, fil, 0, rand, 0, 0
        ),
        lkit->margin(
          new Border(lkit->center(_grafik), wkit->foreground()),
          rand / 2, 0, 0, rand, 0, 0,
          rand, 0, 0, rand, 0, 0
        )
      ), true
    )
  );
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarBrowser::draw(Canvas*, Allocation&)
 * Parameter : Canvas*     - gesamter Bildschirmplatz
 *             Allocation& - zugewiesener Anteil des Bildschirmplatzes
 * Zweck : zeichnet Grafik im zugewiesenen Bildschirmbereich.
 *------------------------------------------------------------------------------
 */
void LingVarBrowser::draw(Canvas* c, const Allocation& a) const
{
  MonoGlyph::draw(c, a);
  _zeichne_fuzzysets();
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarBrowser::init(LingVarSatz*)
 * Parameter : LingVarSatz* - darzustellende ling. Variablen
 * Zweck : initialisiert die Grafik mit einem neuen Satz ling. Variablen.
 *------------------------------------------------------------------------------
 */
void LingVarBrowser::init(const LingVarSatz* lingvars)
{
  if(lingvars != nil && lingvars->anz_lingvars() > 0) {
    _lingvarsatz = lingvars;
    if(lingvars != nil) {
      init_var_tafel();
      if(_merk_var_nr == -1 || _merk_var_nr >= _lingvarsatz->anz_lingvars())
        zeig_var(_lingvarsatz->anz_lingvars() - 1);
      _var_tafel->markieren(_merk_var_nr);
      _grafik->aktualisiere();
    }
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarBrowser::zeig_var(int)
 * Parameter : int - Nummer einer Variablen
 * Zweck : stellt angegebene Variable im Koordinatensystem dar.
 *------------------------------------------------------------------------------
 */
void LingVarBrowser::zeig_var(int nr)
{
  if(_lingvarsatz == nil)
    return;

  LingVar* var = _lingvarsatz->hol_lingvar(nr);

  if(var != nil && nr != _merk_var_nr) {
    _grafik->neue_skala(Dimension_X, var->min(), var->max());
    _merk_var_nr = nr;
    _markiert    = -1;
  }
  if(_merk_var_nr != -1)
    _grafik->aktualisiere();
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarBrowser::zeig_fuz(Regel*)
 * Parameter : Regel* - irgendeine Regel
 * Zweck : Falls die gerade ausgwaehlte Variable in der Regel auftaucht, wird
 *         in der Grafik das Fuzzy-Set markiert, das im Zusammenhang mit der
 *         Variablen in der Regel genannt wird.
 *------------------------------------------------------------------------------
 */
void LingVarBrowser::zeig_fuz(Regel* regel)
{
  if(_merk_var_nr == -1 || _lingvarsatz == nil || regel == nil)
    return;

  LingVar* var = _lingvarsatz->hol_lingvar(_merk_var_nr);
  if(var->typ() == EingabeVar)
    _markiert = var->fuz_nr(regel->fuzzyset(Praemisse, var->name()));
  else
    _markiert = var->fuz_nr(regel->fuzzyset(Konklusion, var->name()));
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarBrowser::init_var_tafel()
 * Zweck : initialisiert die Tafel zur Auswahl einer ling. Variablen.
 *------------------------------------------------------------------------------
 */
void LingVarBrowser::init_var_tafel()
{
  if(_lingvarsatz == nil)
    return;

  while(_var_tafel->anzahl() > 0)
    _var_tafel->loeschen(0);
  for(int i = 0; i < _lingvarsatz->anz_lingvars(); i++)
    _var_tafel->zur_auswahl(_lingvarsatz->hol_lingvar(i)->name());
  _var_tafel->aktualisiere();
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarBrowser::aktualisiere()
 * Zweck : zeichnet die Grafik neu.
 *------------------------------------------------------------------------------
 */
void LingVarBrowser::aktualisiere() { _grafik->aktualisiere(); }

/*
 *-------------------------- loakle Funktionen ---------------------------------
 */

void LingVarBrowser::_var_auswahl() { zeig_var(_var_tafel->auswahl_nr()); }

void LingVarBrowser::_zeichne_fuzzysets() const
{
  if(_merk_var_nr == -1 || _lingvarsatz == nil)
    return;

  LingVar* var = _lingvarsatz->hol_lingvar(_merk_var_nr);
  int anzahl_fuz = var->anz_fuzzysets();
  for(int fuz_nr = 0; fuz_nr < anzahl_fuz; fuz_nr++) {

    FuzzySet* fuz = var->hol_fuzzyset(fuz_nr);
    int anzahl_int = fuz->anz_intervalle();
    for(int int_nr = 0; int_nr < anzahl_int; int_nr++) {

      FsIntervall* intervall = fuz->hol_intervall(int_nr);
      FuzzyTyp x1, y1, x2, y2;
      intervall->links(x1, y1);
      intervall->rechts(x2, y2);
      _grafik->linie(x1, y1, x2, y2);
      if(fuz_nr == _markiert)
        _grafik->zeig_linie(x1, y1, x2, y2);
    }
  }
}


/*
 *------------------------------------------------------------------------------
 *------------------- Definition der Klasse : DateiProtokoll -------------------
 *------------------------------------------------------------------------------
 */

DateiProtokoll::DateiProtokoll(const LingVarSatz* lingvars)
{
  _lingvarsatz = lingvars;

  String name("nicht_gefunden");

  // den Namen der Datei aus den Resourcen lesen
  WidgetKit::instance()->style()->find_attribute("protocol", name);

  // Datei erzeugen
  _datei = new ofstream(name.string());
  if(!_datei) {
    perror(Txt(216));
    _datei = nil;
  }
}

DateiProtokoll::~DateiProtokoll()
{
  if(_datei != nil)
    delete _datei;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : DateiProtokoll::fehlschlag(),
 *                   DateiProtokoll::messwert(int, FuzzyTyp),
 *                   DateiProtokoll::steuerwert(int, FuzzyTyp),
 *                   DateiProtokoll::fehlerwert(FuzzyTyp)
 *                   DateiProtokoll::steuer_var()
 * Zweck : Die jeweiligen Daten werden in die Protokoll-Datei geschrieben.
 *------------------------------------------------------------------------------
 */

void DateiProtokoll::fehlschlag()
{
  if(_datei != nil)
    *_datei << Txt(215) << '\n' << '\n';
}

void DateiProtokoll::messwert(int, FuzzyTyp mess)
{
  if(_datei != nil)
    *_datei << mess << '\t';
}

void DateiProtokoll::steuerwert(int, FuzzyTyp steuer)
{
  if(_datei != nil)
    *_datei << steuer << '\t';
}

void DateiProtokoll::fehlerwert(FuzzyTyp fehler)
{
  if(_datei != nil)
    *_datei << fehler << '\n';
}

void DateiProtokoll::start(float lern_faktor, const LingVarSatz* lingvars)
{
  if(_datei != nil) {
    *_datei << Txt(217) << lern_faktor << "\n" << Txt(218) << "\n";
    _schreib_lingvarsatz(lingvars);
    for(int i = 0; i < lingvars->anz_lingvars(EingabeVar); i++)
      *_datei << _lingvarsatz->hol_lingvar(EingabeVar, i)->name().string()
              << '\t';
    for(i = 0; i < lingvars->anz_lingvars(AusgabeVar); i++)
      *_datei << _lingvarsatz->hol_lingvar(AusgabeVar, i)->name().string()
              << '\t';
    *_datei << FehlerVarName << '\n';
  }
}

void DateiProtokoll::ende(long schritte, const LingVarSatz* lingvars)
{
  if(_datei != nil) {
    *_datei << Txt(219) << schritte << Txt(220) << "\n" << Txt(218) << "\n";
    _schreib_lingvarsatz(lingvars);
  }
}

void DateiProtokoll::_schreib_lingvarsatz(const LingVarSatz* lingvars)
{
  for(LingVarTyp typ = EingabeVar; typ <= AusgabeVar; typ++) {

    int anz_var = lingvars->anz_lingvars(typ);
    for(int var_nr = 0; var_nr < anz_var; var_nr++) {
      LingVar * var = lingvars->hol_lingvar(typ, var_nr);
      *_datei << "\t" << var->name().string() << " :\n";

      int anz_fuz = var->anz_fuzzysets();
      for(int fuz_nr = 0; fuz_nr < anz_fuz; fuz_nr++) {
        FuzzySet* fuz = var->hol_fuzzyset(fuz_nr);
        *_datei << "\t\t" << fuz->name().string() << " : ";

        int anz_lin = fuz->anz_intervalle();
        for(int lin_nr = 0; lin_nr < anz_lin; lin_nr++) {
          FsIntervall* linie = fuz->hol_intervall(lin_nr);
          FuzzyTyp x1, y1, x2, y2;
          linie->links(x1, y1);
          linie->rechts(x2, y2);
          if(y1 < y2)
            *_datei << " a = ("   << x1 << ", " << y1
                    << "), b = (" << x2 << ", " << y2 << ")";
          else
            *_datei << " a = ("   << x2 << ", " << y2
                    << "), b = (" << x1 << ", " << y1 << ")";
          if(lin_nr != anz_lin - 1)
            *_datei << " + ";
        }
        *_datei << "\n";
      }
    }
  }
}


/*
 *------------------------------------------------------------------------------
 *------------------ Definition der Klasse : NfcProtokoll ----------------------
 *------------------------------------------------------------------------------
 */

declareActionCallback(NfcProtokoll);
implementActionCallback(NfcProtokoll);

NfcProtokoll::NfcProtokoll(
                Wissensbank* wissensbank, ProtokollStruktur* prot_strukt
              )
{
  Coord rand      = 10;
  LayoutKit* lkit = LayoutKit::instance();
  WidgetKit* wkit = WidgetKit::instance();

  Glyph *regel_glyph, *lingvar_glyph, *werte_glyph;

  _gestoppt    = false;
  _wissensbank = wissensbank;

  // Regel-Repraesentation festlegen
  switch(prot_strukt->regel_rep) {
    case alsTafel :
      _regel_browser = new RegelTafel(nil, _wissensbank->steuer_var());
      regel_glyph    = _regel_browser->glyph();
      break;
    case alsTabelle :
      _regel_browser = new RegelTabelle(nil, _wissensbank->steuer_var());
      regel_glyph    = _regel_browser->glyph();
      break;
    default :
      regel_glyph    = lkit->hglue(0, fil, 0);
      _regel_browser = nil;
      break;
  }

  // Fuzzy-Set/Ling.Var. - Repraesentation festlegen
  switch(prot_strukt->lingvar_rep) {
    case alsGrafik :
      _var_browser  = new LingVarBrowser();
      lingvar_glyph = _var_browser;
      break;
    default :
      lingvar_glyph = lkit->vglue(0, fil, 0);
      _var_browser  = nil;
      break;
  }

  // Systemwerte-Repraesentation festlegen
  switch(prot_strukt->werte_rep) {
    case Numerisch :
      _werte_browser = new NumWerteBrowser(_wissensbank->steuer_var());
      werte_glyph    = _werte_browser->glyph();
      break;
    case Grafisch :
      _werte_browser = new GrafWerteBrowser(_wissensbank->steuer_var());
      werte_glyph    = _werte_browser->glyph();
      break;
    default :
      _werte_browser = nil;
      werte_glyph    = lkit->hglue(0, fil, 0);
      break;
  }

  // evtl. Protokoll-Datei anlegen
  if(prot_strukt->mit_prot_datei)
    _prot_datei = new DateiProtokoll(_wissensbank->steuer_var());
  else
    _prot_datei = nil;

  body(
    lkit->vbox(
      lkit->center(
        lkit->hbox(
          lkit->center(lkit->variable_span(regel_glyph, 0, 0)),
          lkit->hglue(rand, 0, 0),
          lkit->center(werte_glyph)
        )
      ), lkit->vglue(rand, 0, 0),
      lkit->center(lingvar_glyph)
    )
  );

  if(_var_browser != nil)
    _var_browser->init_var_tafel();
}

NfcProtokoll::~NfcProtokoll()
{
  if(_regel_browser != nil)
    delete _regel_browser;
  if(_prot_datei != nil)
    delete _prot_datei;
  if(_werte_browser != nil)
    delete _werte_browser;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcProtokoll::stop(boolean)
 * Parameter : boolean - stoppen oder wieder anstarten ?
 * Zweck : stoppen oder starten des Protokolls. Starten macht nur Sinn, wenn
 *         zuvor gestoppt wurde.
 *------------------------------------------------------------------------------
 */
void NfcProtokoll::stop(boolean teuv) { _gestoppt = teuv; }

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcProtokoll::init(LingVarSatz*)
 * Parameter : LingVarSatz* - linguistische Variablen
 * Zweck : initialisiert das Protokoll mit zu benutzenden ling. Variablen.
 *------------------------------------------------------------------------------
 */
void NfcProtokoll::init(const LingVarSatz* lingvars)
{
  if(_var_browser != nil)
    _var_browser->init(lingvars);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcProtokoll::init(RegelBasis*)
 * Parameter : RegelBasis* - Regeln
 * Zweck : initialisiert das Protokoll mit zu benutzenden Regeln.
 *------------------------------------------------------------------------------
 */
void NfcProtokoll::init(const RegelBasis* regeln)
{
  if(_regel_browser != nil)
    _regel_browser->zeig_regeln(regeln);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcProtokoll::aktua_lingvarsatz()
 * Zweck : nimmt zur Kenntnis, dass sich die Fuzzy-Sets der ling. Variablen
 *         geaendert haben und zeichnet sie daher neu.
 *------------------------------------------------------------------------------
 */
void NfcProtokoll::aktua_lingvarsatz()
{
  if(!_gestoppt && _var_browser != nil)
    _var_browser->aktualisiere();
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcProtokoll::zeig_regel(int)
 * Parameter : int - Nummer einer Regel
 * Zweck : markiert eine Regel. Ausserdem wird ein entsprechendes Fuzzy-Set
 *         markiert ( siehe LingVarBrowser::zeig_fuz(Regel*)).
 *------------------------------------------------------------------------------
 */
void NfcProtokoll::zeig_regel(int regel_nr)
{
  if(_gestoppt)
    return;

  if(_regel_browser != nil)
    _regel_browser->markiere(regel_nr);
  if(_var_browser != nil)
    _var_browser->zeig_fuz(_wissensbank->steuer_regeln()->hol_regel(regel_nr));
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcProtokoll::zeig_eingabe_var(int, FuzzyTyp)
 * Parameter : int      - Nummer einer Eingabe-Variablen
 *             FuzzyTyp - Wert
 * Zweck : protokolliert Wert der Eingabe-Variablen.
 *------------------------------------------------------------------------------
 */
void NfcProtokoll::zeig_eingabe_var(int var_nr, FuzzyTyp wert)
{
  if(!_gestoppt && _werte_browser != nil)
    _werte_browser->eingabe_var(var_nr, wert);
  if(_prot_datei != nil)
    _prot_datei->messwert(var_nr, wert);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcProtokoll::zeig_ausgabe_var(int, FuzzyTyp)
 * Parameter : int      - Nummer einer Ausgabe-Variablen
 *             FuzzyTyp - Wert
 * Zweck : protokolliert Wert der Ausgabe-Variablen.
 *------------------------------------------------------------------------------
 */
void NfcProtokoll::zeig_ausgabe_var(int var_nr, FuzzyTyp wert)
{
  if(!_gestoppt && _werte_browser != nil)
    _werte_browser->ausgabe_var(var_nr, wert);
  if(_prot_datei != nil)
    _prot_datei->steuerwert(var_nr, wert);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcProtokoll::zeig_fehler(FuzzyTyp)_
 * Parameter : FuzzyTyp - Fehler
 * Zweck : protokolliert Fehler.
 *------------------------------------------------------------------------------
 */
void NfcProtokoll::zeig_fehler(FuzzyTyp fehler)
{
  if(!_gestoppt && _werte_browser != nil)
    _werte_browser->fehler_var(fehler);
  if(_prot_datei != nil)
    _prot_datei->fehlerwert(fehler);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcProtokoll::fehlschlag()
 * Zweck : protokolliert das Fehlschlagen des Steuer-Vorgangs.
 *------------------------------------------------------------------------------
 */
void NfcProtokoll::fehlschlag()
{
  if(_prot_datei != nil)
    _prot_datei->fehlschlag();
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcProtokoll::datei_start(float, LingVarSatz*)
 * Parameter : float        - Lernfaktor
 *             LingVarSatz* - ling. Variablen
 * Zweck : Falls eine Protokoll-Datei angelegt werden soll, werden die entspr.
 *         Daten in diese abgelegt (wird zu Beginn eines neuen Controller-
 *         Aufrufs aufgerufen).
 *------------------------------------------------------------------------------
 */
void NfcProtokoll::datei_start(float lern_faktor, const LingVarSatz* lingvars)
{
  if(_prot_datei != nil)
    _prot_datei->start(lern_faktor, lingvars);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : NfcProtokoll::datei_ende(long, LingVarSatz*)
 * Parameter : float        - Anzahl der Schritte
 *             LingVarSatz* - ling. Variablen
 * Zweck : Falls eine Protokoll-Datei angelegt werden soll, werden die entspr.
 *         Daten in diese abgelegt (wird zu nach Beendigung eines Controller-
 *         Aufrufs aufgerufen).
 *------------------------------------------------------------------------------
 */
void NfcProtokoll::datei_ende(long anz_schritte, const LingVarSatz* lingvars)
{
  if(_prot_datei != nil)
    _prot_datei->ende(anz_schritte, lingvars);
}
