/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




/*
 *  Dieses Modul enthaelt verschiedene Klassen zur Definition einfacher
 *  Grafikelemente, die in fast allen Teilen des Programms benoetigt werden.
 */

#include <IV-look/kit.h>
#ifndef nfc_grafik_h
#define nfc_grafik_h

#include <IV-look/telltale.h>
#include <IV-look/fbrowser.h>
#include <IV-look/field.h>
#include <InterViews/window.h>
#include <InterViews/layout.h>
#include <InterViews/action.h>
#include <InterViews/event.h>
#include <InterViews/style.h>
#include <InterViews/scrbox.h>
#include <InterViews/patch.h>
#include <OS/string.h>

#include "global.h"
#include "lingvar.h"

#define MAXSTRING 200

// Definition einer Struktur zum Anfordern von Menues

struct MenueStruktur
{
  int eintrag;                   // Text des Menuepunktes
  Action* cb;                    // Callback-Funktion
  MenueStruktur* unter_menue;    // evtl.es Untermenue
  int typ;                       // Typ des Menuepunktes (siehe NfcKit::menue)
};


/*
 *------------------------------------------------------------------------------
 * Klasse : AusgabeFeld
 * Zweck  : Erzeugt ein Feld zur Ausgabe von Texten, z.B. fuer Fehlermeldungen.
 *          Setzt auf die Klasse TBScrollBox auf.
 *------------------------------------------------------------------------------
 */
class AusgabeFeld : public Patch
{
  public : AusgabeFeld(int spalten, int zeilen);
           ~AusgabeFeld();
           void ausgeben(String);
           void aktualisiere();
           GlyphIndex anzahl();
  private: TBScrollBox* _feld;
           int _max_zeilen;
};

inline AusgabeFeld::~AusgabeFeld() { Resource::unref(_feld); }

inline void AusgabeFeld::aktualisiere() { redraw(); }

inline GlyphIndex AusgabeFeld::anzahl() { return _feld->count(); }


/*
 *------------------------------------------------------------------------------
 * Klasse : NfcStrBrowser
 * Zweck  : Erzeugt ein Feld zur Auswahl von Texten, z.B. zur Auswahl vo Regeln.
 *          Setzt auf die Klasse FileBrowser auf.
 *------------------------------------------------------------------------------
 */
class NfcStrBrowser : public MonoGlyph
{
  public : NfcStrBrowser(int spalten, int zeilen,
                         Action* ok = nil, Action* abbruch = nil);
           ~NfcStrBrowser();
           void zur_auswahl(String);
           void loeschen(GlyphIndex nr);

           int auswahl_nr();
           void markieren(int nr);

           String ausgewaehlt();
           void markieren(String);

           GlyphIndex anzahl();
           void aktualisiere();
  private: FileBrowser* _auswahl;
           WidgetKit* _kit;
           StringList* _strings;
};

inline int NfcStrBrowser::auswahl_nr() { return _auswahl->selected(); }


/*
 *------------------------------------------------------------------------------
 * Klasse : NfcKit
 * Zweck  : Enthaelt verschiedene Funktionen zur Handhabung von Grafikelementen,
 *          die globale Verwendung finden.
 *------------------------------------------------------------------------------
 */
class NfcKit
{
  public : NfcKit();
           static NfcKit* instance();

           // Glyphs mit irgendwelchen Texten garnieren
           Glyph* text_drauf(const char* text, Glyph*, boolean mit_rand);
           Glyph* text_davor(const char* text, Glyph*, boolean mit_rand);

           // Menue erzeugen
           Menu* menue(MenueStruktur*, WidgetKit*, LayoutKit*);

  private: static NfcKit* _instance;
           Menu* _unter_menue(MenueStruktur*, int typ, WidgetKit*, LayoutKit*);
           LayoutKit* _layout_kit;
           WidgetKit* _widget_kit;
};


/*
 *------------------------------------------------------------------------------
 * Klasse : EingabeFeld
 * Zweck  : Erzeugt ein Feld zur Eingabe von Texten oder Zahlen. Setzt auf die
 *          Klasse FieldEditor auf.
 *------------------------------------------------------------------------------
 */
class EingabeFeld : public MonoGlyph
{
  public : EingabeFeld(int spalten, float def, Action* ok, Action* nook = nil);
           EingabeFeld(int spalten, String def, Action* ok, Action* nook = nil);
           ~EingabeFeld();
           void edit();
           boolean hol_wert(FuzzyTyp&);
           boolean hol_wert(int&);
           void setz_wert(float zahl);
           String text();
           void text(String);
           InputHandler* input_handler();
  private: void _init(int, String, Action* ok, Action* nook);
           void _ok_callback(FieldEditor*);
           void _nook_callback(FieldEditor*);
           String _lies_text();
           boolean _str2fuzzy(const char*, double&);
           void _float2str(float, char*);
           FieldEditor* _editor;
           Action *_ok_cb,
                  *_abbruch_cb;
};

declareActionCallback(EingabeFeld);
#define EingabeFeldCb(INST,PROC) ActionCallback(EingabeFeld)\
                                  (INST,&EingabeFeld::PROC)


/*
 *------------------------------------------------------------------------------
 * Klasse : NfcLabel
 * Zweck  : Erzeugt ein Label fuer Texte variabler Laenge. Die Unterschiede zu
 *          einem 'normalen' Label sind folgende :
 *                 - es wird kein Bildschirmplatz in x-Richtung angefordert,
 *                   und nur soviel Text dargestellt, wie dem NfcLabel zuge-
 *                   wiesen wird.
 *                 - der Text des Labels ist austauschbar.
 *                 - das Label kann zu jedem beliebigen Zeitpunkt neu gezeichnet
 *                   werden.
 *          Diese Klasse findet zum Beispiel als Tabellenfeld Verwendung.
 *------------------------------------------------------------------------------
 */
class NfcLabel : public Glyph
{
  public : NfcLabel(String text, WidgetKit*);
           virtual void request(Requisition&) const;
           virtual void allocate(Canvas*, const Allocation&, Extension&);
           virtual void draw(Canvas*, const Allocation&) const;
           void text(String text);
           void aktualisiere();
  private: void _init();
           Coord _breite;
           CopyString _text;              // darzustellender Text
           WidgetKit* _kit;
           Coord _chr_breite,             // Breite eines Zeichens
                 _chr_hoehe,              // Hoehe eines Zeichens
                 _darstellbar,            // Anzahl darstellbarer Zeichen
                 _offset;                 // Abstand zum rechten Rand

           // der Rest wird zum Aktualisieren benoetigt
           Canvas* _canvas;
           Coord _links, _rechts,
                 _unten, _oben;
};

inline void NfcLabel::aktualisiere()
{
  if(_canvas != nil)
     _canvas->damage(_links, _unten, _rechts, _oben);
}


/*
 *------------------------------------------------------------------------------
 * Klasse : AuswahlButton
 * Zweck  : Erzeugt ein Button, dass jeweils einen von mehreren moeglichen
 *          Texten darstellt und diesen Text nach jedem Anklicken wechselt.
 *          Der Button kann seine Texte dabei wahlweise horizontal oder
 *          vertikal darstellen. Baut auf auf die Klasse Deck.
 *------------------------------------------------------------------------------
 */
class AuswahlButton : public InputHandler
{
  public : AuswahlButton(WidgetKit*, Action*, DimensionName = Dimension_X);
           virtual void press(const Event&);
           void setz_callback(Action* callback) { _callback = callback; };
           void zur_auswahl(String);
           void auswahl(String);
           String auswahl();
           void auswahl_nr(int nr);
           int auswahl_nr();
  private: WidgetKit* _kit;
           Action* _callback;
           DimensionName _dimension;
           StringList _merk_strings;
           int _ausgewaehlt;
           Deck* _deck;
           Patch* _patch;
};

inline int AuswahlButton::auswahl_nr() { return _ausgewaehlt; };


/*
 *------------------------------------------------------------------------------
 * Klasse : NfcAction
 * Zweck  : Genau wie Action, nur dass vor dem Ausfuehren des Callbacks einer
 *          Variablen ein Wert zugewiesen wird. Adresse der Variablen und Wert
 *          sind dem Konstruktor mitzuteilen.
 *------------------------------------------------------------------------------
 */
class NfcAction : public Action
{
  public : NfcAction(int* var, int wert, Action* cb);
           virtual void execute();
  private: int* _var;
           int _wert;
           Action* _callback;
};


/*
 *------------------------------------------------------------------------------
 * Klasse : NfcFenster
 * Zweck  : Oeffnet ein TopLevelWindow zur Darstellung eines Glyph's.
 *------------------------------------------------------------------------------
 */
class NfcDialogFenster
{
  public : NfcDialogFenster(Glyph*);
           ~NfcDialogFenster();
           void position(Coord& x, Coord& y);
           boolean sichtbar();
           void einblenden(
                  Window*, String name, String icon_name, Coord x, Coord y
                );
           void ausblenden();
           void body(Glyph* inhalt);
           Window* window();
  private: Glyph* _inhalt;
           TopLevelWindow *_fenster;
           Coord _x_pos, _y_pos;
};


/*
 *------------------------------------------------------------------------------
 * Klasse : NfcAusgabe
 * Zweck  : Schreibt die uebergebenen Texte in ein 'AusgabeFeld'.
 *------------------------------------------------------------------------------
 */
class NfcAusgabe : public Ausgabe, public MonoGlyph
{
  public : NfcAusgabe(AusgabeFeld*);
           virtual void ausgeben(AusgabeTyp, String);
           virtual void ausgeben(AusgabeTyp, String, String);
           virtual void ausgeben(AusgabeTyp, String, String, String);
  private: AusgabeFeld* _feld;
           boolean _toggle;
};

#endif
