/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




#include <OS/math.h>

#include "lingvar.h"

implementPtrList(IntervallListe, FsIntervall)
implementPtrList(FuzzySetListe, FuzzySet)
implementPtrList(LingVarListe, LingVar)

/*
 *------------------------------------------------------------------------------
 *------------------- Definition der Klasse : 'FsIntervall' --------------------
 *------------------------------------------------------------------------------
 */

FsIntervall::FsIntervall()
{
  _x1 = _x2 = 0;
  _y1 = _y2 = 0;
  _links_drin = _rechts_drin = 1;
}

FsIntervall::FsIntervall(
               FuzzyTyp x1, FuzzyTyp y1, FuzzyTyp x2, FuzzyTyp y2,
               boolean links_drin, boolean rechts_drin
             )
{
  if(x1 == x2) {
    _x1 = _x2 = x1;
    _y1 = _y2 = y1;
  } else {
    if(x1 > x2) {
      _x1 = x2;
      _y1 = y2;
      _x2 = x1;
      _y2 = y1;
    } else {
      _x1 = x1;
      _y1 = y1;
      _x2 = x2;
      _y2 = y2;
    }
  }
  _links_drin  = links_drin;
  _rechts_drin = rechts_drin;
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FsIntervall::setz_links(FuzzyTyp, boolean)
 * Parameter : FuzzyTyp - neue x-Koordinate
 *             boolean  - ist neuer Punkt enthalten, oder nicht
 * Zweck : Definiert neue linke Grenze fuer das Intervall. Die x-Koordinate
 *         dieses Grenzpunktes wird uebergeben, die y-Koordinate berechnet.
 *------------------------------------------------------------------------------
 */
void FsIntervall::setz_links(FuzzyTyp x, boolean drin)
{
  FuzzyTyp merk_x;
  FuzzyTyp merk_y;

  if(x < _x2) {
    merk_x = x;
    merk_y = _ywert(x);
    _x1    = merk_x;
    _y1    = merk_y;
    _links_drin = drin;
  } else if(x == _x2) {
    merk_x = x;
    merk_y = _ywert(x);
    _x1 = _x2 = merk_x;
    _y1 = _y2 = merk_y;
    _links_drin = _rechts_drin = drin;
  }
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FsIntervall::setz_rechts(FuzzyTyp, boolean)
 * Zweck : genau wie 'FsIntervall::setz_links(FuzzyTyp, boolean)', nur fuer
 *         rechte Grenze.
 *------------------------------------------------------------------------------
 */
void FsIntervall::setz_rechts(FuzzyTyp x, boolean drin)
{
  FuzzyTyp merk_x;
  FuzzyTyp merk_y;

  if(x > _x1) {
    merk_x = x;
    merk_y = _ywert(x);
    _x2    = merk_x;
    _y2    = merk_y;
    _rechts_drin = drin;
  } else if(x == _x1) {
    merk_x = x;
    merk_y = _ywert(x);
    _x1 = _x2 = merk_x;
    _y1 = _y2 = merk_y;
    _links_drin = _rechts_drin = drin;
  }
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : boolean FsIntervall::enthalten(FuzzyTyp)
 * Parameter : FuzzyTyp - x-Koordinate
 * Zweck : gibt an, ob ein y-Wert fuer x-Koordinate defniert ist.
 *------------------------------------------------------------------------------
 */
boolean FsIntervall::enthalten(FuzzyTyp x)
{
  if((x == _x1 && _links_drin) || (x == _x2 && _rechts_drin))
    return true;
  if(x > _x1 && x < _x2)
    return true;
  return false;
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FuzzyTyp FsIntervall::wert(FuzzyTyp)
 * Parameter : FuzzyTyp - x-Koordinate
 * Rueckgabewert : zugehoeriger y-Wert
 * Zweck :  Liefert fuer eine gegebene x-Koordinate entsprechenden y-Wert,
 *          falls x-Koordinate im Intervall enthalten ist, sonst Null.
 *------------------------------------------------------------------------------
 */
FuzzyTyp FsIntervall::wert(FuzzyTyp x)
{
  if(enthalten(x) || x == _x1 || x == _x2)
    return _ywert(x);
  else
    return 0;
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : boolean FsIntervall::invertiere(FuzzyTyp y, FuzzyTyp& x)
 * Parameter : FuzzyTyp  - y-Koordinate
 *             FuzzyTyp& - x-Koordinate
 * Rueckgabewert : Erfolg der Aktion
 * Zweck : Bestimmt fuer gegebene y-Koordinate zugehoerige x-Koordinate.
 *------------------------------------------------------------------------------
 */
boolean FsIntervall::invertiere(FuzzyTyp y, FuzzyTyp& x)
{
  if(y > _y1 && y > _y2)
    return false;
  if(y < _y1 && y < _y2)
    return false;
  x = _xwert(y);
  return true;
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FsIntervall::breiter(
 *                     FuzzyTyp, FuzzyTyp, FuzzyTyp FuzzyTyp, FuzzyTyp
 *                   )
 * Parameter : FuzzyTyp grad   - Grad der Aktion
 *             FuzzyTyp min    - untere Definitionsgrenze
 *             FuzzyTyp max    - obere Definitionsgrenze
 *             FuzzyTyp min_br - minimale Breite
 *             FuzzyTyp max_br - maximale Breite
 * Zweck : Diese Funktion bestraft oder belohnt die einzelnen Intervalle mit
 *         mit bestimmten Grad. D.h. bei einem Grad > 0 werden die Intervalle
 *         belohnt, also breiter gemacht, bei einem Grad < 0 umgekehrt.
 *         Breiter machen heisst : der Punkt mit dem kleineren y-Wert wird
 *         vom anderen Punkt weggeschoben.
 *         Diese Funktion ist sinnvoll fuer ein Fussy-Set in der Praemisse
 *         einer Regel.
 *------------------------------------------------------------------------------
 */
void FsIntervall::breiter(
       FuzzyTyp grad,
       FuzzyTyp min,    FuzzyTyp max,
       FuzzyTyp min_br, FuzzyTyp max_br
     )
{
  if(Abs(_y1 - _y2) < 0.05)     // bei ca. waagerechten Linien nichts tun
    return;

  FuzzyTyp breite = _x2 - _x1;

  if(grad > 0) {

    // belohnen
    if(_y1 < _y2) {

      // beim Belohnen muss darauf geachtet weren, dass die Bereichsgrenzen
     //  nicht ueberschritten werden; ausserdem sollen die Punkte nicht ueber
    //   den Nullpunkt geschoben werden

      // _x1 verschieben
      if(_x1 >= 0)           //   0 /
        _x1 = Max(0, _x1 - grad * breite);
      else                   //   / 0
        _x1 = Max(min, _x1 - grad * breite);
      _x1 = Max(_x1, _x2 - max_br);
    } else {

      // _x2 verschieben
      if(_x2 > 0)            //   0 \
        _x2 = Min(max, _x2 + grad * breite);
      else                   //   \ 0
        _x2 = Min(0, _x2 + grad * breite);
      _x2 = Min(_x2, max_br + _x1);
    }
  } else {
    FuzzyTyp min_breite = Min(min_br, _x2 - _x1);

      // beim Bestrafen muss darauf geachtet werden, dass die Intervalle
     //  nicht zu steil werden

    // bestrafen
    if(_y1 < _y2)

      // _x1 verschieben
      _x1 = Min(_x2 - min_breite, _x1 - grad * breite);
    else

      // _x2 verschieben
      _x2 = Max(_x1 + min_breite, _x2 + grad * breite);
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FsIntervall::ausdehnen(
 *                     FuzzyTyp, FuzzyTyp, FuzzyTyp, FuzzyTyp
 *                   )
 * Parameter : FuzzyTyp xmin - minimaler x-Wert
 *             FuzzyTyp xmax - maximaler x-Wert
 *             FuzzyTyp ymin - minimaler y-Wert
 *             FuzzyTyp ymax - maximaler y-Wert
 * Zweck : Die Linie wird bis an die gegebenen Grenzen verlaengert (ausgedehnt).
 *------------------------------------------------------------------------------
 */
void FsIntervall::ausdehnen(FuzzyTyp xmin, FuzzyTyp xmax,
                            FuzzyTyp ymin, FuzzyTyp ymax)
{
  FuzzyTyp li = _ywert(xmin);
  FuzzyTyp re = _ywert(xmax);
  FuzzyTyp un = _xwert(ymin);
  FuzzyTyp ob = _xwert(ymax);

  // Endpunkte gehoeren dazu
  _links_drin = _rechts_drin = true;

  if(li != re && ob != un) {  // Linie ist weder senkrecht noch waagerecht

    // linke Grenze setzen
    if(li > ymin && li < ymax) {
      _x1 = xmin;
      _y1 = li;
    } else {
      _x1 = Min(ob, un);
      _y1 = (ob < un) ? ymax : ymin;
    }

    // rechte Grenze setzen
    if(re > ymin && re < ymax) {
      _x2 = xmax;
      _y2 = re;
    } else {
      _x2 = Max(ob, un);
      _y2 = (ob > un) ? ymax : ymin;
    }
  } else
    if(_x1 != _x2) {            // Linie ist waagerecht
      _x1 = xmin;
      _y1 = li;
      _x2 = xmax;
      _y2 = re;
    }
    // bei senkrechten Linien (Punkten) passiert nichts
}

/*
 *---------------------------- lokale Funktionen -------------------------------
 */

/*
 * Anmerkung : Geradelgleichung :  y = m * x + b
 *                                 m : Steigung;  b : y-Achsenabschnitt
 */

FuzzyTyp FsIntervall::_xwert(FuzzyTyp y)
{
  if(_y2 - _y1 == 0)
    return 0;            // Fehler; Linie ist waagerecht
  if(_x1 == _x2)
    return _x1;          // Linie ist senkrecht (Fuzzy-Singleton)
  FuzzyTyp m = (_y2 - _y1) / (_x2 - _x1);
  FuzzyTyp b =  _y1 - m * _x1;
  if(y == b)
    return 0;
  else
    return (y - b) / m;  // inverse Geradengleichung
}

FuzzyTyp FsIntervall::_ywert(FuzzyTyp x)
{
  if(_x2 - _x1 == 0)
    return _y1;
  FuzzyTyp m = (_y2 - _y1) / (_x2 - _x1);
  FuzzyTyp b = _y1 - m * _x1;
  return m * x + b;      // Geradengleichung
}

/*
 *------------------------------------------------------------------------------
 *------------------- Definition der Klasse : 'FuzzySet' -----------------------
 *------------------------------------------------------------------------------
 */

FuzzySet::FuzzySet()
{
  _min  = _max      = 0;
  _dehn = _schrumpf = 0;
  _name = LeerStr;
}

FuzzySet::FuzzySet(String name)
{
  _min   = _max      = 0;
  _dehn  = _schrumpf = 0;
  _name  = name;
}

FuzzySet::FuzzySet(const FuzzySet& fs) { *this  = fs; }

FuzzySet::~FuzzySet()
{
  while(_intervall_liste.count() > 0) {
    delete _intervall_liste.item(0);
    _intervall_liste.remove(0);
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FuzzyTyp FuzzySet::wert(FuzzyTyp)
 * Parameter : FuzzyTyp x - x-Koordinate
 * Rueckgabewert : Wert des Fuzzy-Sets an der Stelle x
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
FuzzyTyp FuzzySet::wert(FuzzyTyp x)
{
  for(int i = 0; i < anz_intervalle(); i++)
    if(hol_intervall(i)->enthalten(x))
      return hol_intervall(i)->wert(x);
  return 0;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FuzzyTyp FuzzySet::invertiere(FuzzyTyp)
 * Der Wert der inversen Funktion wird zurueckgeliefert; Auch wenn es mehrere
 * Werte gibt, wird nur der Erste, der gefunden wird, beachtet.
 *------------------------------------------------------------------------------
 */
FuzzyTyp FuzzySet::invertiere(FuzzyTyp y)
{
  FuzzyTyp x;
  for(int i = 0; i < anz_intervalle(); i++)
    if(hol_intervall(i)->invertiere(y, x))
      return x;
  return 0;
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FuzzySet::justiere(FuzzyTyp, boolean)
 * Parameter : FuzzyTyp - Grad der Aktion
 *             boolean  - steht das Fuzzy-Set in einer Praemisse ?
 * Zweck : Das Fuzzy-Set wird um den angegebenen Grad belohnt oder bestraft;
 *         d.h. es wird, jenachdem ob es in Praemisse oder Konklusion steht,
 *         breiter oder groesser gemacht (siehe entsprechende Methoden fuer
 *         FsIntervall).
 *------------------------------------------------------------------------------
 */
void FuzzySet::justiere(FuzzyTyp grad)
{
  if(anz_intervalle() == 1) {

     // Monotones Fuzzy-Set
     _intervall_liste.item(0)->breiter(grad, _min, _max, _schrumpf, _dehn);
  } else
    if(anz_intervalle() == 2) {

      // '3-Eck' Fuzzy-Set
      _intervall_liste.item(0)->breiter(grad, _min, _max, _schrumpf, _dehn);
      _intervall_liste.item(1)->breiter(grad, _min, _max, _schrumpf, _dehn);
    }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FuzzySet::add_intervall(FsIntervall)
 * Parameter : FsIntervall - irgendein Intervall
 * Zweck : Hinzufuegen eines Intervalls zum Fuzzt-Set.
 *------------------------------------------------------------------------------
 */
void FuzzySet::add_intervall(FsIntervall* iv)
{
  FuzzyTyp x1, x2;
  FuzzyTyp y1, y2;

  iv->links(x1, y1);
  iv->rechts(x2, y2);
  add_intervall(x1, y1, x2, y2, iv->enthalten(x1), iv->enthalten(x2));
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FuzzySet::add_intervall(
 *                     FuzzyTyp, FuzzyTyp, FuzzyTyp, FuzzyTyp, boolean, boolean
 *                   )
 * Parameter : FuzzyTyp x1, y1, x2, y2        - Endpunkte des neuen Intervalls
 *             boolean l_drin, boolean r_drin - gehoeren Endpunkte dazu ?
 * Zweck : Hinzufuegen eines Intervalls zum Fuzzt-Set.
 *------------------------------------------------------------------------------
 */
void FuzzySet::add_intervall(FuzzyTyp x1, FuzzyTyp y1, FuzzyTyp x2, FuzzyTyp y2,
                             boolean l_drin, boolean r_drin)
{
  if(x1 == x2)
    y2 = y1;
  if(x1 > x2)
    _loeschen(x2, x1);
  else
    _loeschen(x1, x2);
  if(y1 != 0 || y2 != 0) {
    FsIntervall* intervall = new FsIntervall(x1, y1, x2, y2, l_drin, r_drin);
    if(!intervall->leer())
      _intervall_liste.append(intervall);
    else
      delete intervall;
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FuzzySet::setz_grenzen(FuzzyTyp, FuzzyTyp)
 * Parameter : FuzzyTyp min - minimale Grenze
 *             FuzzyTyp max - maximale Grenze
 * Zweck : Minimale und maximale Grenzen werden gesetzt. Diese werden zum
 * Justieren des Fuzzy-Sets benoetigt.
 *------------------------------------------------------------------------------
 */
void FuzzySet::setz_grenzen(FuzzyTyp min, FuzzyTyp max)
{
  _min = min;
  _max = max;
  if(_dehn == 0) {
    _dehn     = _max - _min;      // Defaultwerte
    _schrumpf = _dehn / 10;
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FuzzySet::elastizitaet(FuzzyTyp, FuzzyTyp)
 * Parameter : FuzzyTyp schr - minimale Grenze
 *             FuzzyTyp dehn - maximale Grenze
 * Zweck : Grenzen setzen, in denen das Fuzzy-Set veraendert werden kann.
 *------------------------------------------------------------------------------
 */
void FuzzySet::elastizitaet(FuzzyTyp schr, FuzzyTyp dehn)
{
  _schrumpf = schr;
  _dehn     = dehn;
}

FuzzySet& FuzzySet::operator=(const FuzzySet& fs)
{
  _name     = fs.name();
  _min      = fs._min;
  _max      = fs._max;
  _schrumpf = fs._schrumpf;
  _dehn     = fs._dehn;

  while(anz_intervalle() > 0)
    del_intervall(0);
  for(int i = 0;  i < fs.anz_intervalle(); i++)
    add_intervall(fs.hol_intervall(i));
  return *this;
}

/*
 *-------------------------------lokale Funktionen------------------------------
 */

void FuzzySet::_loeschen(FuzzyTyp x1, FuzzyTyp x2)
{
  int i = 0;
  FsIntervall* intervall;
  while(i < anz_intervalle()) {
    intervall = hol_intervall(i);
    if(intervall->leer()) {
       del_intervall(i);
       i = 0;
    } else {
      if(intervall->enthalten(x1) && intervall->enthalten(x2)) {
        FsIntervall* neuer = new FsIntervall(*intervall);
        intervall->setz_rechts(x1, false);
        neuer->setz_links(x2, false);
        _intervall_liste.append(neuer);
        i = 0;
      } else {
        if(intervall->links() >= x1 && intervall->rechts() <= x2) {
          del_intervall(i);
          i = 0;
        } else {
          if(intervall->enthalten(x1)) {
            intervall->setz_rechts(x1, false);
            i = 0;
          } else {
            if(intervall->enthalten(x2)) {
              intervall->setz_links(x2, false);
              i = 0;
            } else i++;
          }
        }
      }
    }
  }
}

/*
 *------------------------------------------------------------------------------
 *------------------- Definition der Klasse : 'LingVar' ------------------------
 *------------------------------------------------------------------------------
 */

LingVar::LingVar()
{
  _typ  = EingabeVar;
  _name = LeerStr;
  _von  = -1;
  _bis  = 1;
}

LingVar::LingVar(String name)
{
  _typ  = EingabeVar;
  _name = name;
  _von  = -1;
  _bis  = 1;
}

LingVar::~LingVar()
{
  while(_fuzzyset_liste.count() > 0) {
    delete _fuzzyset_liste.item(0);
    _fuzzyset_liste.remove(0);
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVar::min(FuzzyTyp)
 * Parameter : FuzzyTyp - untere Definitionsgrenze
 * Rueckgabewert : Erfolg der Aktion
 * Zweck : Setzt neue untere Definitionsgrenze der ling. Variablen, wenn alle
 *         Fuzzy-Sets innerhalb dieser Grenze liegen.
 *------------------------------------------------------------------------------
 */
boolean LingVar::min(FuzzyTyp min)
{
  boolean ok = true;

  // zuerst testen, ob alle Fuzzysets im Definitionsbereich liegen
  for(int i = 0; i < anz_fuzzysets() && ok; i++) {
    FuzzySet* fuzzyset = hol_fuzzyset(i);
    for(int j = 0; j < fuzzyset->anz_intervalle() && ok; j++)
      ok = (min <= fuzzyset->hol_intervall(j)->links());
  }
  if(ok) {
    _von = min;

    // alle Fuzzy-Sets von neuen Grenzen in Kenntniss setzen
    int anzahl = anz_fuzzysets();
    for(i = 0; i < anzahl; i++)
      hol_fuzzyset(i)->setz_grenzen(_von, _bis);
  }
  return ok;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVar::max(FuzzyTyp)
 * Parameter : FuzzyTyp - obere Definitionsgrenze
 * Rueckgabewert : Erfolg der Aktion
 * Zweck : Komplement zu LingVar::min(FuzzyTyp)
 *------------------------------------------------------------------------------
 */
boolean LingVar::max(FuzzyTyp max)
{
  boolean ok = true;

  // zuerst testen, ob alle Fuzzysets im Definitionsbereich liegen
  for(int i = 0; i < anz_fuzzysets() && ok; i++) {
    FuzzySet* fuzzyset = hol_fuzzyset(i);
    for(int j = 0; j < fuzzyset->anz_intervalle() && ok; j++)
      ok = (max >= fuzzyset->hol_intervall(j)->rechts());
  }
  if(ok) {
    _bis = max;

    // alle Fuzzy-Sets von neuen Grenzen in Kenntniss setzen
    int anzahl = anz_fuzzysets();
    for(i = 0; i < anzahl; i++)
      hol_fuzzyset(i)->setz_grenzen(_von, _bis);
  }
  return ok;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVar::typ(LingVarTyp)
 * Parameter : LingVarTyp - neuer Typ
 * Zweck : Setzt neuen Typ fuer ling. Variable, wenn dies erlaubt ist. (es ist
 *         z.B. nicht erlaubt, das eine Ausgabe-Variable nicht-monotone
 *         Fuzzy-Sets enthaelt)
 *------------------------------------------------------------------------------
 */
void LingVar::typ(LingVarTyp typ) { _typ = typ; }

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVar::fuz_nr(String)
 * Parameter : String - Name eines Fuzzy-Sets
 * Rueckgabewert : Nummer des Fuzzy-Sets, wenn ein Fuzzy-Set mit entspr. Namen
 *                 in der ling. Variablen enthalten ist, sonst -1.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
int LingVar::fuz_nr(String name)
{
  for(int i = 0; i < anz_fuzzysets(); i++)
    if(name == hol_fuzzyset(i)->name())
      return i;
  return -1;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVar::hol_fuzzyset(String)
 * Parameter : String - Name eines Fuzzy-Sets
 * Rueckgabewert : Zeiger auf Fuzzy-Set, wenn ein Fuzzy-Set mit entspr. Namen
 *                 in der ling. Variablen enthalten ist, sonst nil.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
FuzzySet* LingVar::hol_fuzzyset(String name) const
{
  for(int i = 0; i < anz_fuzzysets(); i++)
    if(hol_fuzzyset(i)->name() == name)
      return hol_fuzzyset(i);
  return nil;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVar::del_fuzzyset(int)
 * Parameter : int - Nummer eines Fuzzy-Sets
 * Zweck : loescht Fuzzy-Set mit angegebener Nummer.
 *------------------------------------------------------------------------------
 */
void LingVar::del_fuzzyset(int nr)
{
  delete _fuzzyset_liste.item(nr);
  _fuzzyset_liste.remove(nr);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVar::del_fuzzyset(String)
 * Parameter : String - Name eines Fuzzy-Sets
 * Zweck : loescht Fuzzy-Set mit angegebenem Namen.
 *------------------------------------------------------------------------------
 */
void LingVar::del_fuzzyset(String name)
{
  int nr = -1;
  for(int i = 0; i < anz_fuzzysets(); i++)
    if(hol_fuzzyset(i)->name() == name)
      nr = i;
  if(nr != -1)
    del_fuzzyset(nr);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVar::add_fuzzyset(FuzzySet*)
 * Parameter : FuzzySet* - neues Fuzzy-Set
 * Rueckgabewert : Erfolg der Aktion.
 * Zweck : Nimmt neues Fuzzy-Set in der ling. Variablen auf, wenn dieses zur
 *         Variablen passt.
 *------------------------------------------------------------------------------
 */
boolean LingVar::add_fuzzyset(FuzzySet* fuzzyset)
{
  if(fuzzyset != nil) {

    // Bereichsgrenzen testen
    boolean ok = true;
    for(int i = 0; i < fuzzyset->anz_intervalle() && ok; i++)
      ok = (_von <= fuzzyset->hol_intervall(i)->links());
    for(i = 0; i < fuzzyset->anz_intervalle() && ok; i++)
      ok = (_bis >= fuzzyset->hol_intervall(i)->rechts());

    // falls ok -> eintragen
    if(ok) {
      FuzzySet* neuer = new FuzzySet(*fuzzyset);
      boolean gefunden = false;
      for(int i = 0; i < anz_fuzzysets(); i++)
        if(_fuzzyset_liste.item(i)->name() == fuzzyset->name()) {

          // altes Fuzzy-Set ueberschreiben
          del_fuzzyset(i);
          _fuzzyset_liste.insert(i, neuer);
          gefunden = true;
        }

      // neues Fuzzy-Set eintragen
      if(!gefunden)
        _fuzzyset_liste.append(neuer);

      // Grenzen werden im Fuzzy-Set zum justieren benoetigt
      neuer->setz_grenzen(_von, _bis);
    }
    return ok;
  } else
    return false;
}

LingVar& LingVar::operator=(const LingVar& lv)
{
  _name = lv.name();
  _typ  = lv.typ();
  _von  = lv.min();
  _bis  = lv.max();
  while(anz_fuzzysets() > 0)
    del_fuzzyset(0);
  for(int i = 0; i < lv.anz_fuzzysets(); i++)
    add_fuzzyset(lv.hol_fuzzyset(i));
  return *this;
}

/*
 *  ----------------- Definition der Klasse : 'LingVarSatz' --------------------
 */

LingVarSatz::~LingVarSatz()
{
  while(_lingvar_liste.count() > 0) {
    delete _lingvar_liste.item(0);
    _lingvar_liste.remove(0);
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarSatz::var_nr(LingVarTyp, String)
 * Parameter : LingVarTyp - Typ der Variablen
 *             String     - Name der Variablen
 * Rueckgabewert : Nummer der Variablen, wenn diese gefunden wurde, sonst -1.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
int LingVarSatz::var_nr(LingVarTyp typ, String name) const
{
  int anzahl = anz_lingvars(typ);
  for(int i = 0; i < anzahl; i++)
    if(hol_lingvar(typ, i)->name() == name)
      return i;
  return -1;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarSatz::var_nr(String)
 * Parameter : String - Name der Variablen
 * Rueckgabewert : Nummer der Variablen, wenn diese gefunden wurde, sonst -1.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
int LingVarSatz::var_nr(String name) const
{
  int anzahl = anz_lingvars();
  for(int i = 0; i < anzahl; i++)
    if(hol_lingvar(i)->name() == name)
      return i;
  return -1;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarSatz::hol_lingvar(LingVarTyp, int)
 * Parameter : LingVarTyp - Typ der Variablen
 *             int n      - Nummer der Variablen
 * Rueckgabewert : n'te Variablen mit angegebenem Typ, wenn diese gefunden
 *                 wurde, sonst nil.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
LingVar* LingVarSatz::hol_lingvar(LingVarTyp typ, int nr) const
{
  int zaehler = 0;
  int anzahl = anz_lingvars();
  for(int i = 0; i < anzahl; i++)
    if(hol_lingvar(i)->typ() == typ && zaehler++ == nr)
      return hol_lingvar(i);
  return nil;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarSatz::hol_lingvar(int)
 * Parameter : int n - Nummer der Variablen
 * Rueckgabewert : n'te Variablen, wenn diese gefunden wurde, sonst nil.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
LingVar* LingVarSatz::hol_lingvar(String name) const
{
  for(int i = 0; i < anz_lingvars(); i++)
    if(hol_lingvar(i)->name() == name)
      return hol_lingvar(i);
  return nil;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarSatz::anz_lingvars(LingVarTyp)
 * Parameter : LingVarTyp - Typ der Variablen
 * Rueckgabewert : Anzahl der Variablen mit angegebenem Typ.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
int LingVarSatz::anz_lingvars(LingVarTyp typ) const
{
  int zaehler = 0;
  for(int i = 0; i < anz_lingvars(); i++)
    if(hol_lingvar(i)->typ() == typ)
      zaehler++;
  return zaehler;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarSatz::del_lingvar(String)
 * Parameter : String - Name der ling. Variablen
 * Zweck : loescht ling. Variable mit entspr. Namen.
 *------------------------------------------------------------------------------
 */
void LingVarSatz::del_lingvar(String name)
{
  int nr = -1;
  for(int i = 0; nr == -1 && i < anz_lingvars(); i++)
    if(hol_lingvar(i)->name() == name)
      nr = i;
  if(nr != -1)
    del_lingvar(nr);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarSatz::add_lingvar(LingVar*)
 * Parameter : LingVar* - neue Variable
 * Zweck : fuegt neue Variable ein.
 *------------------------------------------------------------------------------
 */
void LingVarSatz::add_lingvar(LingVar* lv)
{
  LingVar* lingvar = new LingVar(*lv);
  boolean gefunden = false;
  for(int i = 0; i < anz_lingvars(); i++)
    if(_lingvar_liste.item(i)->name() == lingvar->name()) {
      del_lingvar(i);
      _lingvar_liste.insert(i, lingvar);
      gefunden = true;
    }
  if(!gefunden)
    _lingvar_liste.append(lingvar);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarSatz::tausche(int, int)
 * Parameter : int nr1, nr2 - Nummer irgendwelcher Variablen
 * Zweck : tauscht die ling. Variable in der Liste um. Die Reihenfolge der
 *         Variablen ist spaeter beim Einlesen der Messwerte, bzw. beim
 *         Ausgeben der Steuerwerte von Bedeutung und wird ueber diese Funktion
 *         manipuliert.
 *------------------------------------------------------------------------------
 */
void LingVarSatz::tausche(int nr1, int nr2)
{
  LingVar* elem1 = _lingvar_liste.item(nr1);
  LingVar* elem2 = _lingvar_liste.item(nr2);
  _lingvar_liste.remove(nr2);
  _lingvar_liste.insert(nr2, elem1);
  _lingvar_liste.remove(nr1);
  _lingvar_liste.insert(nr1, elem2);
}

LingVarSatz& LingVarSatz::operator=(const LingVarSatz& lvs)
{
  while(anz_lingvars() > 0)
    del_lingvar(0);
  for(int i = 0; i < lvs.anz_lingvars(); i++)
    add_lingvar(lvs.hol_lingvar(i));
  return *this;
}
