/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/



#include <stdio.h>
#include <string.h>
#include <fstream.h>

#include "datei.h"
#include "nfclook.h"

#define MAXDATEISTRING 30
#define ENDSTR '\n'
#define TRENN ' '
#define OPTIONAL '%'

/*
 *------------------------------------------------------------------------------
 *--------------------- Definition der Klasse DateiArbeit ----------------------
 *------------------------------------------------------------------------------
 */

declareFileChooserCallback(DateiArbeit);
implementFileChooserCallback(DateiArbeit);

DateiArbeit::DateiArbeit(Window* fenster, NfcAusgabe* ausgabe)
{
  _lv_extension = ".lv";
  _rg_extension = ".rg";
  _lv_filter    = "*.lv";
  _rg_filter    = "*.rg";

  Style* style = WidgetKit::instance()->style();
  style->attribute("*FileChooser*subcaption", Txt(88));      // "Dateiname :"
  style->attribute("*FileChooser*open", Txt(89));            // "Ok"
  style->attribute("*FileChooser*cancel", Txt(90));          // "Abbruch"
  style->attribute("*FileChooser*filter", "on");

  _basis_fenster = fenster;
  _ausgabe_feld  = ausgabe;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : DateiArbeit::speichern(boolean, LingVarSatz*),
 *                   DateiArbeit::speichern(boolean, RegelBasis*)
 *                   DateiArbeit::laden(boolean, LingVarSatz*),
 *                   DateiArbeit::laden(boolean, RegelBasis*)
 * Zweck : Speichern und Laden der ling. Variablen und Regeln. Es kann dabei
 *         jeweils angegeben werden, ob es sich um die Daten zur Steuerung oder
 *         zur Fehlerberechnung handelt. Anmwerkung : Beim Laden wird die
 *         Adresse angegeben, die zum Abspeicher der geledenen Daten benutzt
 *         wird.
 *------------------------------------------------------------------------------
 */

void DateiArbeit::speichern(boolean steuer_var, const LingVarSatz* lv)
{
  _steuer_daten = steuer_var;
  _quell_vars   = lv;
  _ausfuehren(LvSpeichern);
}

void DateiArbeit::speichern(boolean steuer_regeln, const RegelBasis* rb)
{
  _steuer_daten = steuer_regeln;
  _quell_regeln = rb;
  _ausfuehren(RgSpeichern);
}

void DateiArbeit::laden(boolean steuer_var, LingVarSatz* lv)
{
  _steuer_daten = steuer_var;
  _ziel_vars    = lv;
  _ausfuehren(LvLaden);
}

void DateiArbeit::laden(boolean steuer_regeln, RegelBasis* rb)
{
  _steuer_daten = steuer_regeln;
  _ziel_regeln  = rb;
  _ausfuehren(RgLaden);
}

/*
 *--------------------------- lokale Funktionen --------------------------------
 */

void DateiArbeit::_ausfuehren(Modus modus)
{
  _modus = modus;

  WidgetKit* wkit = WidgetKit::instance();
  Style* style = wkit->style();
  NullTerminatedString verzeichnis;
  style->find_attribute("DataDirectory", verzeichnis);

  switch(modus) {
    case LvSpeichern:
      wkit->style()->attribute("*FileChooser*filterPattern", _lv_filter);
      wkit->style()->attribute("*FileChooser*caption", Txt(91));
      break;
    case RgSpeichern:
      wkit->style()->attribute("*FileChooser*filterPattern", _rg_filter);
      wkit->style()->attribute("*FileChooser*caption", Txt(93));
      break;
    case LvLaden:
      wkit->style()->attribute("*FileChooser*filterPattern", _lv_filter);
      wkit->style()->attribute("*FileChooser*caption", Txt(92));
      break;
    case RgLaden:
      wkit->style()->attribute("*FileChooser*filterPattern", _rg_filter);
      wkit->style()->attribute("*FileChooser*caption", Txt(94));
      break;
  }


  FileChooser* fchooser = DialogKit::instance()->file_chooser(
                            verzeichnis.string(),
                            wkit->style(),
                            new FileChooserCallback(DateiArbeit)(
                              this,&DateiArbeit::_callback
                            )
                          );
  fchooser->post_for(_basis_fenster);
}

void DateiArbeit::_callback(FileChooser* fchooser, boolean ok)
{
  boolean erflog  = false;

  String ext;
  CopyString datei_name;

  if(ok) {
    if(_modus == LvSpeichern || _modus == LvLaden)
      ext = _lv_extension;
    else
      ext = _rg_extension;
    datei_name = _parse(*fchooser->selected(), ext);

    switch(_modus) {
      case LvSpeichern:                        // Daten speichern
      case RgSpeichern: {
        ofstream datei(datei_name.string());
        if(datei) {
          if(_modus == LvSpeichern) {
            if(datei << _quell_vars) {         // LingVarSatz speichern
              if(_steuer_daten)
                _ausgabe_feld->ausgeben(Meldung, Txt(97));
              else
                _ausgabe_feld->ausgeben(Meldung, Txt(181));
            } else                             // Fehler beim Speichern
              _ausgabe_feld->ausgeben(Fehler, Txt(99), datei_name, Txt(100));
          } else {                             // RegelBasis speichern
            if(datei << _quell_regeln) {
              if(_steuer_daten)
                _ausgabe_feld->ausgeben(Meldung, Txt(101));
              else
                _ausgabe_feld->ausgeben(Meldung, Txt(182));
            } else                             // Fehler beim Speichern
              _ausgabe_feld->ausgeben(Fehler, Txt(99), datei_name, Txt(100));
          }
        } else                                 // Fehler beim Oeffnen
          _ausgabe_feld->ausgeben(Fehler, Txt(95), datei_name, Txt(96));
      }
      break;
      case LvLaden:                            // Daten lesen
      case RgLaden: {
        ifstream datei(datei_name.string());
        if(datei) {
          if(_modus == LvLaden) {              // LingVarSatz laden
            LingVarSatz lvs;
            if(datei >> &lvs) {
              *_ziel_vars = lvs;
              if(_steuer_daten)
                _ausgabe_feld->ausgeben(Meldung, Txt(98));
              else
                _ausgabe_feld->ausgeben(Meldung, Txt(183));
            } else
              _ausgabe_feld->ausgeben(Fehler, Txt(99), datei_name, Txt(100));
          } else {                             // RegelBasis laden
            RegelBasis rgb;
            if(datei >> &rgb) {
              *_ziel_regeln = rgb;
              if(_steuer_daten)
                _ausgabe_feld->ausgeben(Meldung, Txt(102));
              else
                _ausgabe_feld->ausgeben(Meldung, Txt(184));
            } else
              _ausgabe_feld->ausgeben(Fehler, Txt(99), datei_name, Txt(100));
          }
        } else                                 // Fehler beim Oeffnen
          _ausgabe_feld->ausgeben(Fehler, Txt(95), datei_name, Txt(96));
      }
      break;
    }
  }
}

CopyString DateiArbeit::_parse(String auswahl, String extension)
{
  NullTerminatedString eingabe(auswahl);
  NullTerminatedString ext(extension);
  boolean ok = false;

  if(eingabe.length() > ext.length()) {
    String ende = eingabe.substr(eingabe.length() - ext.length(), ext.length());
    if(ende == ext)
      ok = true;
  }

  char* name = new char[eingabe.length() + ext.length()];
  strcpy(name, eingabe.string());
  if(!ok)
    strcat(name, ext.string());

  CopyString datei_name(name);
  delete[] name;
  return datei_name;
}


/*
 *------------------------------------------------------------------------------
 *--------- Definition der Ein/Ausgabe-Operatoren fuer ling. Variablen ---------
 * ( und was so dazugehoert )
 *------------------------------------------------------------------------------
 */

ostream& operator<<(ostream& os, FsIntervall* intervall)
{
  FuzzyTyp x1, y1, x2, y2;
  intervall->links(x1, y1);
  intervall->rechts(x2, y2);
  
  os << x1 << TRENN << y1 << TRENN << intervall->enthalten(x1) << TRENN;
  os << x2 << TRENN << y2 << TRENN << intervall->enthalten(x2) << TRENN;
  return os;
}

istream& operator>>(istream& is, FsIntervall* intervall)
{
  FuzzyTyp x1, y1, x2, y2;
  boolean l_drin, r_drin;

  is >> x1 >> y1 >> l_drin;
  is >> x2 >> y2 >> r_drin;
  if(is)
    *intervall = FsIntervall(x1, y1, x2, y2, l_drin, r_drin);
  return is;
}

ostream& operator<<(ostream& os, FuzzySet* fuzzy)
{
  NullTerminatedString name(fuzzy->name());
  os << name.string() << ENDSTR;

  /*
   * Anmerkung :
   * 'OPTIONAL' heisst, es folgen minimale und maximale Breite;
   * wird aus Kompatibilitaetsgruenden benutzt; es gibt auch Dateien mit
   * ling. Var. ohne diese beiden Werte fuer die Breite.
   */
  os << OPTIONAL << fuzzy->schrumpf() << TRENN << fuzzy->dehn() << TRENN;

  int anzahl = fuzzy->anz_intervalle();
  os << anzahl << TRENN;
  for(int i = 0; i < anzahl; i++)
    os << fuzzy->hol_intervall(i);
  return os;
}

istream& operator>>(istream& is, FuzzySet* fuzzy)
{
  FuzzySet fuz;
  char str[MAXDATEISTRING];
  char c;
  int anzahl = 0;;

  del_trenn(is);
  is.get(str, MAXDATEISTRING, ENDSTR);
  if(is.get(c) && c != ENDSTR)
    return is;
  fuz.name(str);

  // siehe Anmerkung oben
  if(is.peek() == OPTIONAL) {
    FuzzyTyp min_breite, max_breite;
    is >> c >> min_breite >> max_breite;
    fuz.elastizitaet(min_breite, max_breite);
  }

  is >> anzahl;
  for(int i = 0; i < anzahl; i++) {
    FsIntervall intervall;
    is >> &intervall;
    fuz.add_intervall(&intervall);
  }
  if(is)
    *fuzzy = fuz;
  return is;
}

ostream& operator<<(ostream& os, LingVar* lingvar)
{
  NullTerminatedString name(lingvar->name());
  os << name.string() << ENDSTR;
  os << lingvar->min() << TRENN << lingvar->max() << TRENN;
  if(lingvar->typ() == EingabeVar)
    os << 0 << TRENN;
  else
    os << 1 << TRENN;
  int anzahl = lingvar->anz_fuzzysets();
  os << anzahl << TRENN;
  for(int i = 0; i < anzahl; i++)
    os << lingvar->hol_fuzzyset(i);
  return os; 
}

istream& operator>>(istream& is, LingVar* lingvar)
{
  LingVar lv;
  char str[MAXDATEISTRING];
  char c;
  int anzahl = 0;

  del_trenn(is);
  is.get(str, MAXDATEISTRING, ENDSTR);
  if(is.get(c) && c != ENDSTR)
    return is;
  lv.name(str);

  FuzzyTyp min, max;
  int typ;
  is >> min >> max >> typ;
  lv.min(min);
  lv.max(max);
  if(typ == 0)
    lv.typ(EingabeVar);
  else
    lv.typ(AusgabeVar);

  is >> anzahl;
  for(int i = 0; i < anzahl; i++) {
    FuzzySet fuz;
    if(is >> &fuz)
      lv.add_fuzzyset(&fuz);
  }
  if(is)
    *lingvar = lv;
  return is;
}

ostream& operator<<(ostream& os, const LingVarSatz* lingvarsatz)
{
  int anzahl = lingvarsatz->anz_lingvars();
  os << anzahl << TRENN;
  for(int i = 0; i < anzahl; i++)
    os << lingvarsatz->hol_lingvar(i);
  return os;
}

istream& operator>>(istream& is, LingVarSatz* lingvarsatz)
{
  LingVar lingvar;
  int anzahl = 0;

  is >> anzahl;
  for(int i = 0; i < anzahl; i++)
    if(is >> &lingvar)
      lingvarsatz->add_lingvar(&lingvar);
  return is;
}


/*
 *------------------------------------------------------------------------------
 *-------------- Definition der Ein/Ausgabe-Operatoren fuer Regeln -------------
 * ( und was so dazugehoert )
 *------------------------------------------------------------------------------
 */

ostream& operator<<(ostream& os, const RegelBasis* regelbasis)
{
  int anzahl = regelbasis->anzahl_regeln();
  os << anzahl << TRENN;
  for(int i = 0; i < anzahl; i++)
    os << regelbasis->hol_regel(i);
  return os;
}

istream& operator>>(istream& is, RegelBasis* regelbasis)
{
  Regel regel;
  int anzahl = 0;

  is >> anzahl;
  for(int i = 0; i < anzahl; i++)
    if(is >> &regel)
      regelbasis->add_regel(&regel);
  return is;
}

ostream& operator<<(ostream& os, Regel* regel)
{
  int pra = regel->groesse(Praemisse);
  int kon = regel->groesse(Konklusion);
  os << pra << TRENN << kon << TRENN;
  for(RegelAusdruck ausdruck = Praemisse; ausdruck <= Konklusion; ausdruck++) {
    int anzahl = (ausdruck == Praemisse) ? pra : kon;
    for(int i = 0; i < anzahl; i++) {
      Bedingung* bed = regel->bedingung(ausdruck, i);
      os << bed->lingvar().string() << ENDSTR;
      os << bed->fuzzyset().string() << ENDSTR;
    }
  }
  return os;
}

istream& operator>>(istream& is, Regel* regel)
{
  char lv_str[MAXDATEISTRING];
  char fs_str[MAXDATEISTRING];
  char c;

  int pra, kon;
  is  >> pra >> kon;
  Regel neu(pra, kon);

  for(RegelAusdruck ausdruck = Praemisse; ausdruck <= Konklusion; ausdruck++) {
    int anzahl = (ausdruck == Praemisse) ? pra : kon;
    for(int i = 0; i < anzahl; i++) {

      del_trenn(is);
      is.get(lv_str, MAXDATEISTRING, ENDSTR); // Namen der ling. Variablen lesen
      if(is.get(c) && c != ENDSTR)
        return is;

      del_trenn(is);
      is.get(fs_str, MAXDATEISTRING, ENDSTR); // Namen des Fuzzy-Sets lesen
      if(is.get(c) && c != ENDSTR)
        return is;

      if(is)
        neu.bedingung(ausdruck, i, lv_str, fs_str);
    }
  }
  if(is)
    *regel = neu;
  return is;
}


// die folgende Funktion dient dazu, Trennzeichen aus dem Input-Stream
// zu entfernen; vor allem fuer das Einlesen von Strings

void del_trenn(istream& is)
{
  char c;
  do {
    is >> c;
  } while(is && c == TRENN);
  is.putback(c);
}
