/*
	infer3.c
	General-Purpose Fuzzy Inference Library Ver. 3.0
	(c) 1991 Numata Ko~iti
*/

#include "infer3.h"
#include <string.h>

#define A_ERR				(-2)
#define A_IG				(-1)
#define A_OFF				0
#define A_ON				(-1)

#define ELS_ERR				(-1)
#define ELS_OK				0

#define MAX_FUZZ_VAR_NAME_LENGTH	7
#define MAX_KEYWORD_LENGTH		7
#define MAX_LINE_LENGTH			80

#define OUT_NO				fzf->nInVar
#define SW_NO				(-1)


int EmptyLineSkip(char *Line, FILE *fp)
{
	do
		if (fgets(Line, MAX_LINE_LENGTH + 1, fp) == NULL)
			return (ELS_ERR);
	while (sscanf(Line, "%*s") == EOF);
	return (ELS_OK);
}


char Assign(const char *Symbol,
const char FuzzVarName[][MAX_N_FUZZ_VAR][MAX_FUZZ_VAR_NAME_LENGTH + 1],
int InVarNo, const FUZZ_INFO *fzf)
{
	int	i;

	if (InVarNo == SW_NO) {
		if (strcmp(Symbol, "ON") == 0)
			return (A_ON);
		if (strcmp(Symbol, "OFF") == 0)
			return (A_OFF);
	}
	else {
		for (i = 0; i < fzf->nFuzzVar[InVarNo]; i++)
			if (strcmp(Symbol, FuzzVarName[InVarNo][i]) == 0)
				return (i);
		if (strcmp(Symbol, "IG") == 0)
			return (A_IG);
	}
	return (A_ERR);
}


int ReadRule(FILE *fp, FUZZ_INFO *fzf)
{
	char	Line[MAX_LINE_LENGTH + 1], *Tok,
		FuzzVarName[MAX_N_IN_VAR + 1][MAX_N_FUZZ_VAR]
		[MAX_FUZZ_VAR_NAME_LENGTH + 1],
		KeyWord[MAX_N_IN_VAR + 2][MAX_KEYWORD_LENGTH + 1];
	int	i, j;

	if (EmptyLineSkip(Line, fp) == ELS_ERR)
		return (RR_SHORT_FILE);
	for (i = 0; ; i++) {
		sscanf(Line, "%7s", KeyWord[i]);
		if (strcmp(KeyWord[i], "SW") == 0) {
			fzf->nInVar = i - 1;
			break;
		}
		if (i >= MAX_N_IN_VAR + 1)
			return (RR_OVER_MAX);
		if (sscanf(Line, "%*s%f%f",
		&fzf->Interval[i][0], &fzf->Interval[i][1]) < 2)
			return (RR_SHORT_LINE);
		if (EmptyLineSkip(Line, fp) == ELS_ERR)
			return (RR_SHORT_FILE);
		for (j = 0; ; j++) {
			if (sscanf(Line, "%*s") == EOF) {
				fzf->nFuzzVar[i] = j;
				break;
			}
			if (j >= MAX_N_FUZZ_VAR)
				return (RR_OVER_MAX);
			if (sscanf(Line, "%7s%f%f%f%f", FuzzVarName[i][j],
			&fzf->Para[i][j][0], &fzf->Para[i][j][1],
			&fzf->Para[i][j][2], &fzf->Para[i][j][3]) < 5)
				return (RR_SHORT_LINE);
			if (fgets(Line, MAX_LINE_LENGTH + 1, fp) == NULL)
				return (RR_SHORT_FILE);
		}
		if (EmptyLineSkip(Line, fp) == ELS_ERR)
			return (RR_SHORT_FILE);
	}
	strtok(Line, "\n \t");
	for (i = 0; i < fzf->nInVar + 1; i++) {
		Tok = strtok(NULL, "\n \t");
		if (strcmp(Tok, KeyWord[i]) != 0)
			return (RR_UNDEFINED_KEYWORD);
	}
	if (EmptyLineSkip(Line, fp) == ELS_ERR)
		return (RR_SHORT_FILE);
	for (i = 0; ; i++) {
		if (sscanf(Line, "%*s") == EOF) {
			fzf->nRule = i;
			break;
		}
		if (i >= MAX_N_RULE)
			return (RR_OVER_MAX);
		Tok = strtok(Line, "\n \t");
		if ((fzf->Switch[i]
		= Assign(Tok, FuzzVarName, SW_NO, fzf)) == A_ERR)
			return (RR_UNDEFINED_SYMBOL);
		for (j = 0; j < fzf->nInVar + 1; j++) {
			if ((Tok = strtok(NULL, "\n \t")) == NULL)
				return (RR_SHORT_LINE);
			if ((fzf->FuzzVarNo[i][j]
			= Assign(Tok, FuzzVarName, j, fzf)) == A_ERR)
				return (RR_UNDEFINED_SYMBOL);
		}
		if (fgets(Line, MAX_LINE_LENGTH + 1, fp) == NULL)
			return (RR_SHORT_FILE);
	}
	if (EmptyLineSkip(Line, fp) == ELS_ERR)
		return (RR_SHORT_FILE);
	sscanf(Line, "%7s", KeyWord[0]);
	if (strcmp(KeyWord[0], "NEP") != 0)
		return (RR_UNDEFINED_KEYWORD);
	if (sscanf(Line, "%*s%d", &fzf->nEqualPart) < 1)
		return (RR_SHORT_LINE);
	return (RR_OK);
}


float Trapeze(float x, int InVarNo, char FuzzVarNo, float Height,
const FUZZ_INFO *fzf)
{
	if (x > fzf->Para[InVarNo][FuzzVarNo][0]
	&& x < fzf->Para[InVarNo][FuzzVarNo][1])
		return ((x - fzf->Para[InVarNo][FuzzVarNo][0]) * Height
		/ (fzf->Para[InVarNo][FuzzVarNo][1]
		- fzf->Para[InVarNo][FuzzVarNo][0]));
	if (x > fzf->Para[InVarNo][FuzzVarNo][2]
	&& x < fzf->Para[InVarNo][FuzzVarNo][3])
		return ((fzf->Para[InVarNo][FuzzVarNo][3] - x) * Height
		/ (fzf->Para[InVarNo][FuzzVarNo][3]
		- fzf->Para[InVarNo][FuzzVarNo][2]));
	if (x >= fzf->Para[InVarNo][FuzzVarNo][1]
	&& x <= fzf->Para[InVarNo][FuzzVarNo][2])
		return (Height);
	return (0.0);
}


float OutFunc(float x, const float Fit[], const FUZZ_INFO *fzf)
{
	float	Grade, MaxGrade = 0.0;
	int	i;

	for (i = 0; i < fzf->nRule; i++) {
		if (fzf->Switch[i] == A_OFF)
			continue;
		Grade
		= Trapeze(x, OUT_NO, fzf->FuzzVarNo[i][OUT_NO], Fit[i], fzf);
		if (Grade > MaxGrade)
			MaxGrade = Grade;
	}
	return (MaxGrade);
}


float Infer(const float InVar[], const FUZZ_INFO *fzf)
{
	float	Fit[MAX_N_RULE], Grade, fa, fb, h, s0, s1, x;
	int	i, j;

	for (i = 0; i < fzf->nRule; i++) {
		if (fzf->Switch[i] == A_OFF)
			continue;
		Fit[i] = 1.0;
		for (j = 0; j < fzf->nInVar; j++) {
			if (fzf->FuzzVarNo[i][j] == A_IG)
				continue;
			Grade
			= Trapeze(InVar[j], j, fzf->FuzzVarNo[i][j], 1.0, fzf);
			if (Grade < Fit[i])
				Fit[i] = Grade;
		}
	}

	fa = OutFunc(fzf->Interval[OUT_NO][0], Fit, fzf);
	fb = OutFunc(fzf->Interval[OUT_NO][1], Fit, fzf);
	s0 = (fa + fb) / 2.0;
	s1 = (fzf->Interval[OUT_NO][0] * fa + fzf->Interval[OUT_NO][1] * fb)
	/ 2.0;
	x = fzf->Interval[OUT_NO][0];
	h = (fzf->Interval[OUT_NO][1] - fzf->Interval[OUT_NO][0])
	/ fzf->nEqualPart;
	for (i = 1; i < fzf->nEqualPart; i++) {
		x += h;
		fa = OutFunc(x, Fit, fzf);
		s0 += fa;
		s1 += x * fa;
	}
	return (s1 / s0);
}




char uAssign(const char *Symbol,
const char FuzzVarName[][MAX_N_FUZZ_VAR][MAX_FUZZ_VAR_NAME_LENGTH + 1],
int InVarNo, const uFUZZ_INFO *fzf)
{
	int	i;

	if (InVarNo == SW_NO) {
		if (strcmp(Symbol, "ON") == 0)
			return (A_ON);
		if (strcmp(Symbol, "OFF") == 0)
			return (A_OFF);
	}
	else {
		for (i = 0; i < fzf->nFuzzVar[InVarNo]; i++)
			if (strcmp(Symbol, FuzzVarName[InVarNo][i]) == 0)
				return (i);
		if (strcmp(Symbol, "IG") == 0)
			return (A_IG);
	}
	return (A_ERR);
}


int uReadRule(FILE *fp, uFUZZ_INFO *fzf)
{
	char	Line[MAX_LINE_LENGTH + 1], *Tok,
		FuzzVarName[MAX_N_IN_VAR + 1][MAX_N_FUZZ_VAR]
		[MAX_FUZZ_VAR_NAME_LENGTH + 1],
		KeyWord[MAX_N_IN_VAR + 2][MAX_KEYWORD_LENGTH + 1];
	float	fPara[4];
	int	i, j, k;

	if (EmptyLineSkip(Line, fp) == ELS_ERR)
		return (RR_SHORT_FILE);
	for (i = 0; ; i++) {
		sscanf(Line, "%7s", KeyWord[i]);
		if (strcmp(KeyWord[i], "SW") == 0) {
			fzf->nInVar = i - 1;
			break;
		}
		if (i >= MAX_N_IN_VAR + 1)
			return (RR_OVER_MAX);
		if (sscanf(Line, "%*s%f%f",
		&fzf->Interval[i][0], &fzf->Interval[i][1]) < 2)
			return (RR_SHORT_LINE);
		if (EmptyLineSkip(Line, fp) == ELS_ERR)
			return (RR_SHORT_FILE);
		for (j = 0; ; j++) {
			if (sscanf(Line, "%*s") == EOF) {
				fzf->nFuzzVar[i] = j;
				break;
			}
			if (j >= MAX_N_FUZZ_VAR)
				return (RR_OVER_MAX);
			if (sscanf(Line, "%7s%f%f%f%f", FuzzVarName[i][j],
			&fPara[0], &fPara[1], &fPara[2], &fPara[3]) < 5)
				return (RR_SHORT_LINE);
			for (k = 0; k < 4; k++)
				if (fPara[k] < fzf->Interval[i][0])
					fzf->Para[i][j][k] = 0;
				else
					if (fPara[k] > fzf->Interval[i][1])
						fzf->Para[i][j][k] = 256;
					else
						fzf->Para[i][j][k]
						= 256.0 * (fPara[k]
						- fzf->Interval[i][0])
						/ (fzf->Interval[i][1]
						- fzf->Interval[i][0]) + 0.5;
			if (fgets(Line, MAX_LINE_LENGTH + 1, fp) == NULL)
				return (RR_SHORT_FILE);
		}
		if (EmptyLineSkip(Line, fp) == ELS_ERR)
			return (RR_SHORT_FILE);
	}
	strtok(Line, "\n \t");
	for (i = 0; i < fzf->nInVar + 1; i++) {
		Tok = strtok(NULL, "\n \t");
		if (strcmp(Tok, KeyWord[i]) != 0)
			return (RR_UNDEFINED_KEYWORD);
	}
	if (EmptyLineSkip(Line, fp) == ELS_ERR)
		return (RR_SHORT_FILE);
	for (i = 0; ; i++) {
		if (sscanf(Line, "%*s") == EOF) {
			fzf->nRule = i;
			break;
		}
		if (i >= MAX_N_RULE)
			return (RR_OVER_MAX);
		Tok = strtok(Line, "\n \t");
		if ((fzf->Switch[i]
		= uAssign(Tok, FuzzVarName, SW_NO, fzf)) == A_ERR)
			return (RR_UNDEFINED_SYMBOL);
		for (j = 0; j < fzf->nInVar + 1; j++) {
			if ((Tok = strtok(NULL, "\n \t")) == NULL)
				return (RR_SHORT_LINE);
			if ((fzf->FuzzVarNo[i][j]
			= uAssign(Tok, FuzzVarName, j, fzf)) == A_ERR)
				return (RR_UNDEFINED_SYMBOL);
		}
		if (fgets(Line, MAX_LINE_LENGTH + 1, fp) == NULL)
			return (RR_SHORT_FILE);
	}
	if (EmptyLineSkip(Line, fp) == ELS_ERR)
		return (RR_SHORT_FILE);
	sscanf(Line, "%7s", KeyWord[0]);
	if (strcmp(KeyWord[0], "NEP") != 0)
		return (RR_UNDEFINED_KEYWORD);
	if (sscanf(Line, "%*s%d", &fzf->nEqualPart) < 1)
		return (RR_SHORT_LINE);
	return (RR_OK);
}


unsigned int uTrapeze(unsigned int x, int InVarNo, char FuzzVarNo,
unsigned int Height, const uFUZZ_INFO *fzf)
{
	if (x > fzf->Para[InVarNo][FuzzVarNo][0]
	&& x < fzf->Para[InVarNo][FuzzVarNo][1])
		return ((x - fzf->Para[InVarNo][FuzzVarNo][0]) * Height
		/ (fzf->Para[InVarNo][FuzzVarNo][1]
		- fzf->Para[InVarNo][FuzzVarNo][0]));
	if (x > fzf->Para[InVarNo][FuzzVarNo][2]
	&& x < fzf->Para[InVarNo][FuzzVarNo][3])
		return ((fzf->Para[InVarNo][FuzzVarNo][3] - x) * Height
		/ (fzf->Para[InVarNo][FuzzVarNo][3]
		- fzf->Para[InVarNo][FuzzVarNo][2]));
	if (x >= fzf->Para[InVarNo][FuzzVarNo][1]
	&& x <= fzf->Para[InVarNo][FuzzVarNo][2])
		return (Height);
	return (0);
}


unsigned int uOutFunc(unsigned int x, const unsigned int Fit[],
const uFUZZ_INFO *fzf)
{
	int		i;
	unsigned int	Grade, MaxGrade = 0;

	for (i = 0; i < fzf->nRule; i++) {
		if (fzf->Switch[i] == A_OFF)
			continue;
		Grade
		= uTrapeze(x, OUT_NO, fzf->FuzzVarNo[i][OUT_NO], Fit[i], fzf);
		if (Grade > MaxGrade)
			MaxGrade = Grade;
	}
	return (MaxGrade);
}


float uInfer(const float InVar[], const uFUZZ_INFO *fzf)
{
	int		i, j;
	unsigned int	Fit[MAX_N_RULE], Grade, uInVar[MAX_N_IN_VAR],
			fa, fb, h, s0, x;
	unsigned long	s1;

	for (i = 0; i < fzf->nInVar; i++)
		uInVar[i] = 256.0 * (InVar[i] - fzf->Interval[i][0])
		/ (fzf->Interval[i][1] - fzf->Interval[i][0]) + 0.5;
	for (i = 0; i < fzf->nRule; i++) {
		if (fzf->Switch[i] == A_OFF)
			continue;
		Fit[i] = 255;
		for (j = 0; j < fzf->nInVar; j++) {
			if (fzf->FuzzVarNo[i][j] == A_IG)
				continue;
			Grade = uTrapeze(uInVar[j],
			j, fzf->FuzzVarNo[i][j], 255, fzf);
			if (Grade < Fit[i])
				Fit[i] = Grade;
		}
	}

	fa = uOutFunc(0, Fit, fzf);
	fb = uOutFunc(256, Fit, fzf);
	s0 = (fa + fb) / 2;
/*	s0 = (fa + fb) / 2.0 + 0.5;	*/
	s1 = 128 * fb;
	h = 256 / fzf->nEqualPart;
	for (x = h; x < 256; x += h) {
		fa = uOutFunc(x, Fit, fzf);
		s0 += fa;
		s1 += x * fa;
	}
	return ((fzf->Interval[OUT_NO][1] - fzf->Interval[OUT_NO][0])
	* s1 / (s0 * 256UL) + fzf->Interval[OUT_NO][0]);
}
