(**************************************************************************************)
(* module name: LingVarEditor                                                         *)
(* project: simulation of a fuzzy pathplanning algorithm                              *)
(* author: markus stadler, IIIC/8                                                     *)
(* first implementation: 6/8/91                                                       *)
(* last change:          7/12/91                                                      *)
(**************************************************************************************)

MODULE LingVarEditor;

FROM Fuzzy          IMPORT maxValPoints; 
FROM Menu           IMPORT MenuRes, SetMenu, GetMenuCmd, DeleteMenu, DisableMenu, 
                           EnableMenu, DisableItem, EnableItem, SetItem;
FROM Dialog         IMPORT Dialog, TextStyle, New, Dispose, Open, Close, UserAction,
                           Button, String, StaticString, RefreshItem;
FROM Conversions    IMPORT StringToReal, IntToString, RealToFixString;
FROM GraphicWindows IMPORT Window, OpenGraphicWindow, Clear, CloseGraphicWindow, 
                           SetMode, Area, WriteString, IdentifyPos, SetPen, MoveTo,
                           paint, invert, erase;
                    IMPORT FileSystem;
FROM FileUtil       IMPORT PutFileName, GetFileName;
FROM CursorMouse    IMPORT GetMouse, ML, MM, MR;
FROM String         IMPORT Append, AppendCh, Length;
FROM FuzzyScanner   IMPORT SymbolType, Descriptor, IdentName, OpenFile, CloseFile, Get,
                           LineNo, RealNumber, Identifier;
FROM MathLib        IMPORT Sin;
FROM Tools          IMPORT ShowString, wScreen, hScreen, ShowFile;
FROM System         IMPORT Allocate, Deallocate;
                    IMPORT Windows;             

(* definition of constants *)
                      
CONST
  
    maxFuzzyVars  =  64;  (* not same definition as in module fuzzy *)
    maxFuzzyVals  =  64;  (* not same definition as in module fuzzy *)
    maxValPerVar  =  24;  (* not same definition as in module fuzzy *)
    
    (* definition of items *)
    FileMenu   = 1;
    VarMenu    = 2;
    ValMenu    = 3;
    SelValMenu = 4;
    HelpMenu   = 5;
      
    LoadItem    = 1;
    SaveAsItem  = 2;
    ExitItem    = 4;
    
    VarInsertItem= 1;
    BoundaryItem = 2;
    VarCopyItem  = 3;
    VarPasteItem = 4;
    VarEraseItem = 5;
    
    ValInsertItem= 1;
    ValEditItem  = 2;
    ValInvertItem= 3;
    ValRotateItem= 4;
    ValCopyItem  = 5;
    ValPasteItem = 6;
    ValEraseItem = 8;

    
    VarMenuMainStr = "Variables|insert|set boundaries|copy|paste|erase|(-";
    ValMenuMainStr = "Values|insert|edit|invert|rotate|copy|paste|(-|erase";
    SelValMenuMainStr = "Select Values";
    
    firstVarList = 7;
    firstValList = 1;
    
    (* size and position of value diagram *)
    widthOfBar   = 4;
    leftV        = 50;
    bottomV      = 50;
    widthV       = widthOfBar*maxValPoints;
    heightV      = 200;
    
    HelpFile     = "LingVarHelp.HLP";

(* definition of types *)

TYPE 
                    (* fuzzy value consists of name and membership function *)
    FuzzyValType =  RECORD  
                      name:  IdentName;
                      mu:    ARRAY [0..maxValPoints-1] OF REAL;
                    END; (* RECORD *)
     
                    (* fuzzy variable consists of name, range, value-table *)
    LingVarType  =  POINTER TO LingVarDesc;
    LingVarDesc  =  RECORD
                      name:  IdentName;
                      lowerB, upperB:  REAL;
                      noVals: INTEGER;
                      vals:  ARRAY [0..maxValPerVar-1] OF FuzzyValType;
                    END; (* RECORD *) 

(* definition of global variables *)

VAR
    lingVars:  ARRAY [0..maxFuzzyVars-1] OF LingVarType;  (* table with all variables *)
    noVars:    INTEGER;       (* number of variables                                  *)
    curVar,                   (* currently selected variable, -1 if no var selected   *)
    curVal:    INTEGER;       (* currently selected value, -1 if no value is selected *)
    
    (* menu desciption strings *)
    VarMenuStr:    ARRAY [0..511] OF CHAR;
    SelValMenuStr: ARRAY [0..511] OF CHAR;
    
    enabledVarMenu,                (* status of VarMenu *)
    enabledValMenu,                (* status of ValMenu & SelValMenu *)
    changeVarMenu,                 (* TRUE whenever VarMenu has to be changed *)
    changeValMenu:  BOOLEAN;       (* TRUE whenever ValMenu has to be changed *)
    
    result:    MenuRes;            (* result of last selection in menus *)
    done:      BOOLEAN;            (* TRUE whenever menu command was selected *)
    
    window:    Window;             (* window with fuzzy values *)
    x,y:       INTEGER;            (* screen size, window pos and size *)
    windowW,
    windowH:   INTEGER;
    s:         BITSET;
    

    valClipboard:    FuzzyValType;      (* clipboars for variables and values *)
    varClipboard:    LingVarDesc;
    isValClipboard,
    isVarClipboard:  BOOLEAN;
    tempName:        IdentName;


(* utility procedures *)

PROCEDURE SetBoundaries; FORWARD;      (* forward declarations *)
PROCEDURE EditVal; FORWARD;
PROCEDURE DrawVal; FORWARD;
    
PROCEDURE Redraw(u: Window); (*--------------------------------------------------------*)
  (* simple redraw function assigned to window *)
BEGIN
  DrawVal;
END Redraw;

PROCEDURE SaveAs; (*-------------------------------------------------------------------*)
  (* ask for a filename and write variable description to the file *)
CONST
  EOL = 15C;
VAR
  fileName:  ARRAY [0..127] OF CHAR;
  done, 
  convOk:    BOOLEAN;
  f:         FileSystem.File;
  str:       ARRAY [0..31] OF CHAR;
  var, val,
  i, j:      INTEGER;
  
  PROCEDURE WriteString(s: ARRAY OF CHAR);
    (* write a string to file f *)
  VAR
    i: INTEGER;
  BEGIN
    FOR i:= 0 TO Length(s)-1 DO
      FileSystem.WriteChar(f, s[i]);
    END
  END WriteString;
  
BEGIN (* SaveAs *)
  (* get filename in dialog box *)
  fileName[0]:= 0C;
  PutFileName(fileName, done);
  
  IF done THEN
    FileSystem.Lookup(f, fileName, TRUE);
    FOR var:= 0 TO noVars-1 DO                     (* write all variable declarations *)
      IF lingVars[var]^.noVals>0 THEN                 (* at least one value necessary *)
        (* write header with name, range etc *)
        WriteString("LINGVAR "); WriteString(lingVars[var]^.name);
        WriteString(" ON [");
        RealToFixString(lingVars[var]^.lowerB, 5, 10, str, convOk); WriteString(str);
        WriteString(", ");
        RealToFixString(lingVars[var]^.upperB, 5, 10, str, convOk); WriteString(str);
        WriteString("] WITH"); FileSystem.WriteChar(f, EOL); (* new line *)
        
        FOR val:= 0 TO lingVars[var]^.noVals-1 DO                (* for all values do *)
          (* write header, name etc. no of valPoints is always written because *)
          (* of compatibility reasons with future versions *)
          WriteString("    VALUE "); WriteString(lingVars[var]^.vals[val].name);
          WriteString("("); IntToString(maxValPoints, 4, 4, 10, str, done); 
          WriteString(str); WriteString(")");
          WriteString(" IS"); FileSystem.WriteChar(f, EOL);
          (* write membership function *)
          FOR i:= 0 TO 7 DO
            WriteString("        ");
            FOR j:= 0 TO 7 DO
              RealToFixString(lingVars[var]^.vals[val].mu[8*i+j], 3, 7, str, convOk);
              WriteString(str); WriteString("  ");
            END;
            FileSystem.WriteChar(f, EOL);
          END
        END
      END;
      WriteString("END"); FileSystem.WriteChar(f, EOL);
    END
  END;
  (* close and store file *)
  FileSystem.Close(f);
END SaveAs;

PROCEDURE LoadVars; (*-----------------------------------------------------------------*)
  (* asks for a filename and loads the variable description file *)
VAR
  fileName:  ARRAY [0..127] OF CHAR;
  done, 
  error:     BOOLEAN;
  symbol:    SymbolType;
  currentD:  Descriptor;
  i:         INTEGER;
  
  (* the procedures ParseVarDecl and ParseValDecl are not documented here *)
  (* they are part of the parser of the module fuzzy and are documented   *)
  (* there in detail                                                      *)

  PROCEDURE ParseVarDecl; (*----------------------------------------------------------*)
  
    PROCEDURE ParseValDecl; (*--------------------------------------------------------*)
    VAR
      noPoints, i: INTEGER;
      mu2:         ARRAY [0..maxValPoints-1] OF REAL;
    BEGIN
      error:= symbol#ValueSy;
      IF NOT error THEN
        symbol:= Get(currentD);
        error:= symbol#IdentSy;
        IF NOT error AND (lingVars[curVar]^.noVals < maxFuzzyVals) THEN
          INC(lingVars[curVar]^.noVals); curVal:= lingVars[curVar]^.noVals-1;
          lingVars[curVar]^.vals[curVal].name:= Identifier;
          symbol:= Get(currentD);
          IF symbol= OpenParSy THEN
            symbol:= Get(currentD);
            error:= symbol#RealNoSy;
            IF NOT error THEN
              noPoints:= TRUNC(RealNumber);
              symbol:= Get(currentD);
              error:= symbol#CloseParSy;
              IF error THEN RETURN END;
              symbol:= Get(currentD);
            END
          ELSE
            noPoints:= maxValPoints;
          END;
          IF NOT error THEN
            error:= symbol#IsSy;
            IF NOT error THEN
              symbol:= Get(currentD);
              i:= 0;
              WHILE (symbol=RealNoSy) AND (i<noPoints) DO
                mu2[i]:= RealNumber;
                symbol:= Get(currentD);
                INC(i);
              END; (* WHILE *)
              error:= i<noPoints;
              IF NOT error THEN
                FOR i:= 0 TO maxValPoints-1 DO
                  lingVars[curVar]^.vals[curVal].mu[i]:=
                  mu2[TRUNC(FLOAT(i)/FLOAT(maxValPoints)*FLOAT(noPoints))];
                END; (* FOR *)
              END
            END
          END
        END
      END   
    END ParseValDecl;
    
  BEGIN
    error:= symbol#LingvarSy;
    IF NOT error THEN
      symbol:= Get(currentD);
      error:= symbol#IdentSy;
      IF NOT error AND (noVars<maxFuzzyVars) THEN
        INC(noVars); curVar:= noVars-1; curVal:= -1;
        Allocate(lingVars[curVar], SIZE(LingVarDesc));
        lingVars[curVar]^.name:= Identifier;
        lingVars[curVar]^.noVals:= 0;
        symbol:= Get(currentD);
        error:= symbol#OnSy;
        IF NOT error THEN
          symbol:= Get(currentD);
          error:= symbol#OpenBraceSy;
          IF NOT error THEN
            symbol:= Get(currentD);
            error:= symbol#RealNoSy;
            IF NOT error THEN
              lingVars[curVar]^.lowerB:= RealNumber;
              symbol:= Get(currentD);
              error:= symbol#CommaSy;
              IF NOT error THEN
                symbol:= Get(currentD);
                error:= symbol#RealNoSy;
                IF NOT error THEN
                  lingVars[curVar]^.upperB:= RealNumber;
                  symbol:= Get(currentD);
                  error:= symbol#CloseBraceSy;
                  IF NOT error THEN
                    symbol:= Get(currentD);
                    error:= symbol#WithSy;
                    IF NOT error THEN
                      symbol:= Get(currentD);
                      WHILE (symbol=ValueSy) AND NOT error DO
                        ParseValDecl;
                      END;
                      IF NOT error THEN
                        error:= symbol#EndSy;
                        IF NOT error THEN
                          symbol:= Get(currentD);
                        END
                      END
                    END
                  END
                END
              END
            END
          END
        END
      END
    END
  END ParseVarDecl;


BEGIN (* LoadVars *) (*---------------------------------------------------------------*)
  (* erase old declaration in memory *)
  FOR i:= 0 TO noVars-1 DO
    Deallocate(lingVars[i])
  END; (* FOR *)
  
  curVar:= -1; curVal:= -1; noVars:= 0; (* no var/val selected *)
  Clear(window);
  error:= FALSE;
  (* get filename and parse new declaration *)
  GetFileName(fileName, "", done);
  IF done THEN
    OpenFile(fileName, currentD, done);
    IF done THEN
      symbol:= Get(currentD);
      WHILE (symbol # EofSy) AND NOT error DO
        ParseVarDecl;
      END
    END;
    CloseFile(currentD);
  END;
  IF error THEN
    (* erase old declaration in memory *)
    FOR i:= 0 TO noVars-1 DO
      Deallocate(lingVars[i])
    END; (* FOR *)
    curVar:= -1; curVal:= -1; noVars:= 0;
    ShowString("errors occured; function aborted...")
  END;
  changeVarMenu:= TRUE; changeValMenu:= TRUE; (* menus have to be changed *)
  IF noVars>0 THEN
    curVar:= 0;
    IF lingVars[curVar]^.noVals>0 THEN
      curVal:= 0;
    END
  END;
  (* draw first value of first variable *)
  DrawVal;
END LoadVars;


PROCEDURE InsertVar; (*----------------------------------------------------------------*)
VAR
  d:                     Dialog;
  t, ok, cancel,item, i: INTEGER;
  pressed:               BOOLEAN;
  v:                     IdentName;
BEGIN
  (* build up dialog box *)
  New(d, 20, hScreen-200, 426, 120);
  StaticString(d, "Enter Name of LingVar:", 10, 80, Plain);
  v[0]:= 0C;  (* erase string *)
  String(d, t, v, 10, 60, 390);  (* read name *)
  Button(d, ok, TRUE, 90, 10, "OK");
  Button(d, cancel, FALSE, 210, 10, "CANCEL");
  Open(d);
  (* read entries *)
  REPEAT
    pressed:= UserAction(d, item)
  UNTIL pressed AND ((item=ok) OR (item=cancel));
  (* terminate string in front of first blank *)
  i:= 0;
  WHILE (i<32) AND (v[i]#" ") DO
    INC(i)
  END; (* WHILE *)
  IF i<32 THEN v[i]:= 0C END;
  
  Close(d);  (* close dialog box *)
  Dispose(d);
  (* if name valid and ok-button pressed and still room for more variables *)
  IF (Length(v)>0) AND (item=ok) AND (noVars<maxFuzzyVars) THEN  
    (* allocate new variable *)
    INC(noVars);
    Allocate(lingVars[noVars-1], SIZE(LingVarDesc));
    lingVars[noVars-1]^.name:= v;
    lingVars[noVars-1]^.noVals:= 0;
    lingVars[noVars-1]^.lowerB:= 0.0;
    lingVars[noVars-1]^.upperB:= 1.0;
    curVar:= noVars-1;
    curVal:= -1;
    DrawVal;
    changeVarMenu:= TRUE; changeValMenu:= TRUE;
    SetBoundaries;
  END; (* IF *)
END InsertVar;


PROCEDURE SetBoundaries; (*-----------------------------------------------------------*)
(* opens a dialog box and asks for the range of the variable *)
VAR
  d: Dialog;
  t1, t2, b, item, i: INTEGER;
  v1, v2: ARRAY [0..20] OF CHAR;
  ok, done, convOk: BOOLEAN;
  value: REAL;
BEGIN
  IF curVar>= 0 THEN
    (* build up dialog box *)
    New(d, 20, hScreen-200, 300, 160);
    StaticString(d, "Enter range:", 10, 120, Plain);
    RealToFixString(lingVars[curVar]^.lowerB, 4, 7, v1, convOk);
    RealToFixString(lingVars[curVar]^.upperB, 4, 7, v2, convOk);
    StaticString(d, "lower:", 10, 90, Plain);
    String(d, t1, v1, 60, 90, 200);
    StaticString(d, "upper:", 10, 60, Plain);
    String(d, t2, v2, 60, 60, 200);
    Button(d, b, TRUE, 100, 20, "OK");
    ok:= FALSE;
    Open(d);
    REPEAT
      IF UserAction(d, item) AND (item=b) THEN
        StringToReal(v1, 0, value, ok);
        IF ok THEN
          lingVars[curVar]^.lowerB:= value;
          StringToReal(v2, 0, value, ok);
          IF ok THEN
            lingVars[curVar]^.upperB:= value;
          ELSE
            RealToFixString(lingVars[curVar]^.upperB, 4, 7,  v2, convOk);
            RefreshItem(d, t2);
          END;
        ELSE
          RealToFixString(lingVars[curVar]^.lowerB, 4, 7, v1, convOk);
          RefreshItem(d, t1);
        END;
      END;
    UNTIL ok;  (* until entry is valid *)
    Close(d);
    Dispose(d);
    DrawVal;
  END; (* IF *)
END SetBoundaries;

PROCEDURE InsertVal; (*--------------------------------------------------------------*)
(* opens a dialog box to insert a new value *)
VAR
  str:                    ARRAY [0..127] OF CHAR;
  d:                      Dialog;
  t, ok, cancel, item, i: INTEGER;
  v:                      IdentName;
  pressed:                BOOLEAN;
 BEGIN
  IF curVar>= 0 THEN
    (* build up dialog box *)
    New(d, 20, hScreen-200, 426, 120);
    StaticString(d, "Enter Name of value:", 10, 80, Plain);
    v[0]:= 0C;
    String(d, t, v, 10, 60, 390);
    Button(d, ok, TRUE, 90, 10, "OK");
    Button(d, cancel, FALSE, 210, 10, "CANCEL");
    Open(d);
    
    REPEAT
      pressed:= UserAction(d, item) 
    UNTIL pressed AND ((item=ok) OR (item=cancel));
    
    (* terminate string in front of first blank *)
    i:= 0;
    WHILE (i<32) AND (v[i]#" ") DO
      INC(i)
    END; (* WHILE *)
    IF i<32 THEN v[i]:= 0C END;
    Close(d);
    Dispose(d);
    
    (* if name is valid, ok pressed and still enough room for the value: *)
    IF (Length(v)>0) AND (lingVars[curVar]^.noVals<maxValPerVar) AND (item=ok) THEN
      INC(lingVars[curVar]^.noVals);
      curVal:= lingVars[curVar]^.noVals-1;
      lingVars[curVar]^.vals[curVal].name:= v;
      FOR i:= 0 TO maxValPoints-1 DO
        lingVars[curVar]^.vals[curVal].mu[i]:= 0.0;
      END; (* FOR *)
      EditVal;  (* definition of the membership function *)
    END; (* IF *)
  END; (* IF *)
  changeValMenu:= TRUE;
END InsertVal;

PROCEDURE DrawVal; (*----------------------------------------------------------------*)
(* draws the current value, if there is one, in the window *)
VAR
  str:       ARRAY [0..127] OF CHAR;
  x:         INTEGER;
  convOk:    BOOLEAN;
BEGIN
  Clear(window);
  
  IF (curVar>=0) AND (curVal>=0) THEN
    (* write title, i.e. name of variable and name of value *)
    SetMode(window, paint);
    SetPen(window, 20, bottomV+heightV+35);
    str[0]:= 0C; Append(str, "Linguistic variable: ");
    Append(str, lingVars[curVar]^.name);
    WriteString(window, str);
    
    SetPen(window, 20, bottomV+heightV+20);
    str[0]:= 0C; Append(str, "Linguistic value:    ");
    Append(str, lingVars[curVar]^.vals[curVal].name);
    WriteString(window, str);

    (* draw border of membership diagram *)
    SetPen(window, leftV, bottomV);
    MoveTo(window, leftV+widthV+1, bottomV);
    MoveTo(window, leftV+widthV+1, bottomV+heightV+1);
    MoveTo(window, leftV,  bottomV+heightV+1);
    MoveTo(window, leftV, bottomV);
    
    (* draw values in window *)
    FOR x:= 0 TO maxValPoints-1 DO
      Area(window, 3, leftV+1+x*widthOfBar, bottomV, widthOfBar,
                   TRUNC(lingVars[curVar]^.vals[curVal].mu[x]*FLOAT(heightV)));
    END; (* FOR *)
    
    (* write scales etc. *)
    SetPen(window, leftV-15, bottomV-20);
    RealToFixString(lingVars[curVar]^.lowerB,2,8,str,convOk); WriteString(window, str);
    SetPen(window, leftV+widthV-15, bottomV-20);
    RealToFixString(lingVars[curVar]^.upperB,2,8,str,convOk); WriteString(window, str);
    SetPen(window, leftV-20, bottomV);         WriteString(window, "0.0");
    SetPen(window, leftV-20, bottomV+heightV); WriteString(window, "1.0");
  END; (* IF *)
END DrawVal;

PROCEDURE EditVal; (*----------------------------------------------------------------*)
(* lets the user choose between to kinds of editing to modify the membership function *)
CONST
  Pi  = 3.141592654;
VAR
  dialog:       Dialog;
  man, shape,
  cancel, item: INTEGER;

  str:          ARRAY [0..127] OF CHAR;
  x, y,                                 (* vars to hold mouse positions etc *)
  xprev, yprev: INTEGER;      
  s:            BITSET;
  u:            Window;
  
  a:            ARRAY [0..3] OF REAL;   (* holds points of shape *)
  ir:           REAL;                   (* indices *)
  i:            INTEGER;
  convOk:       BOOLEAN;
BEGIN
  IF (curVar>=0) AND (curVal>=0) THEN
    DrawVal;
    (* draw dialog box *)
    New(dialog,20, hScreen-200, 400, 120);
    StaticString(dialog, "Choose edit mode!", 10, 80, Plain);
    Button(dialog, man, TRUE,     20, 40, "manual");
    Button(dialog, shape, FALSE,  150, 40, "shape");
    Button(dialog, cancel, FALSE, 280, 40, "cancel");
    Open(dialog);
    REPEAT 
    UNTIL UserAction(dialog, item) AND ((item=man) OR (item=shape) OR (item=cancel));
    Close(dialog);
    Dispose(dialog);
    
    IF item= man THEN                                  (* manul mode *)
      xprev:= -1; yprev:= -1;
      REPEAT
        (* get mouse positon and calc offset in window *)
        GetMouse(s, x, y);
        u:= Windows.UpWindow(x, y);
        IF u#Windows.Background THEN
          IdentifyPos(u, x, y);
        END; (* IF *)
        IF u=window THEN
          x:= (x-leftV-1) DIV widthOfBar;
          y:= y-bottomV-1;
          IF y>heightV THEN y:= heightV END;
          IF y<0   THEN y:= 0   END;
          
          IF (x>=0) AND (x<maxValPoints) THEN
            (* show coordinates *)
            SetPen(window, leftV+widthV DIV 2 - 25, bottomV-20);
            RealToFixString(lingVars[curVar]^.lowerB+
                        (lingVars[curVar]^.upperB-lingVars[curVar]^.lowerB)*
                        FLOAT(x)/FLOAT(maxValPoints-1), 2, 8, str, convOk);
            Append(str, "     ");
            WriteString(window, str);
            SetPen(window, 2, 150);
            RealToFixString(FLOAT(y)/200.0, 3, 7, str, convOk);
            WriteString(window, str);
            
            (* if mouse position has changed, redraw bar *)
            IF ((x#xprev) OR (y#yprev)) AND (ML IN s) THEN
              lingVars[curVar]^.vals[curVal].mu[x]:= FLOAT(y)/FLOAT(heightV);
              xprev:= x; yprev:= y;
              Area(window, 0, leftV+1+x*widthOfBar, bottomV, widthOfBar, heightV);
              Area(window, 3, leftV+1+x*widthOfBar, bottomV, widthOfBar, y);
            END; (* IF *)
          END; (* IF *)
        END; (* IF *)  
      UNTIL (ML IN s) AND (u#window);(* repeat until button clicked outside of window *)
      
    ELSIF item=shape THEN      (* shape mode *)
      (* read all for points for the shape-declaration *)
      FOR i:= 0 TO 3 DO
        str[0]:= 0C; AppendCh(str, CHR(49+i)); Append(str, ". value of shape");
        SetPen(window, 150, 5); WriteString(window, str);
        (* get mouse position of next "click" *)
        REPEAT
          GetMouse(s, x, y);
          u:= Windows.UpWindow(x, y);
          IF u#Windows.Background THEN
            IdentifyPos(u, x, y);
          END; (* IF *)
          IF u=window THEN
            x:= (x-leftV-1) DIV widthOfBar;
            IF x<0 THEN x:= 0 END;
            IF x> maxValPoints-1 THEN x:= maxValPoints END;
            (* show coordinates *)
            SetPen(window, leftV+widthV DIV 2 - 25, bottomV-20);
            RealToFixString(lingVars[curVar]^.lowerB+
                        (lingVars[curVar]^.upperB-lingVars[curVar]^.lowerB)*
                        FLOAT(x)/FLOAT(maxValPoints), 2, 8, str, convOk);
            Append(str, "     ");
            WriteString(window, str);
          END; (* IF *)  
        UNTIL (ML IN s);
        a[i]:= FLOAT(x)/FLOAT(maxValPoints-1);
        IF (i>0) AND (a[i]<a[i-1]) THEN a[i]:= a[i-1] END;
        REPEAT GetMouse(s, x, y) UNTIL NOT (ML IN s);
      END; (* FOR *)
      
      (* calculate membership function with shape-declaration *)
      FOR i:= 0 TO maxValPoints-1 DO
        ir:= FLOAT(i)/FLOAT(maxValPoints-1);
        IF ir<a[0] THEN 
          lingVars[curVar]^.vals[curVal].mu[i]:= 0.0
        ELSIF (ir>=a[0]) AND (ir<a[1]) THEN
          lingVars[curVar]^.vals[curVal].mu[i]:=
          0.5+0.5*Sin(Pi/(a[1]-a[0])*(ir-0.5*(a[0]+a[1])));
        ELSIF (ir>=a[1]) AND (ir<a[2]) THEN
          lingVars[curVar]^.vals[curVal].mu[i]:= 1.0;
        ELSIF (ir>=a[2]) AND (ir<a[3]) THEN
          lingVars[curVar]^.vals[curVal].mu[i]:=
          0.5-0.5*Sin(Pi/(a[3]-a[2])*(ir-0.5*(a[3]+a[2])));
        ELSE
          lingVars[curVar]^.vals[curVal].mu[i]:= 0.0
        END
      END;
      DrawVal;
    END; (* IF *)
  END; (* IF *)
END EditVal;

PROCEDURE InvertVal; (*--------------------------------------------------------------*)
  (* inverts the current value; val -> NOT(val) = 1.0 - val *)
VAR
  i: INTEGER;
BEGIN
  IF (curVar>=0) AND (curVal>=0) THEN
    FOR i:= 0 TO maxValPoints-1 DO
      lingVars[curVar]^.vals[curVal].mu[i]:= 1.0-lingVars[curVar]^.vals[curVal].mu[i]
    END;
    DrawVal;
  END
END InvertVal;

PROCEDURE RotateVal; (*---------------------------------------------------------------*)
(* lets user rotate a value *)
VAR
  d:               Dialog;
  t, b, item, i:   INTEGER;
  v:               ARRAY [0..20] OF CHAR;
  ok, done, 
  convOk:          BOOLEAN;
  value:           REAL;
  dist:            INTEGER;
  fval:            FuzzyValType;
  
BEGIN
  IF (curVar>=0) AND (curVal>=0) THEN
    (* open dialog box *)
    New(d, 20, hScreen-200, 300, 160);
    StaticString(d, "distance to ratate:", 10, 120, Plain);
    v[0]:= 0C;
    String(d, t, v, 60, 80, 200);
    Button(d, b, TRUE, 100, 20, "OK");
    ok:= FALSE;
    Open(d);
    REPEAT
      IF UserAction(d, item) AND (item=b) THEN  (* if ok pressed then: *)
        StringToReal(v, 0, value, ok);
        IF ok AND (ABS(value)<lingVars[curVar]^.upperB-lingVars[curVar]^.lowerB) THEN
          dist:= TRUNC(value/(lingVars[curVar]^.upperB-lingVars[curVar]^.lowerB)
                        *FLOAT(maxValPoints));
          FOR i:= 0 TO maxValPoints-1 DO
            fval.mu[(maxValPoints+i+dist) MOD maxValPoints]:= 
                                      lingVars[curVar]^.vals[curVal].mu[i];
          END; (* FOR *)
          lingVars[curVar]^.vals[curVal].mu:= fval.mu;
          
        ELSE
          RealToFixString(0.0, 4, 8, v, convOk);
          RefreshItem(d, t);
        END;
      END;
    UNTIL ok;
    Close(d);
    Dispose(d);
    DrawVal;
  END; (* IF *)
END RotateVal;


PROCEDURE EraseVar; (*-------------------------------------------------------------*)
  (* erase, after confirmation, the current variable *)
VAR
  d:    Dialog;
  yes ,no, item: INTEGER;
  string: ARRAY [0..127] OF CHAR;
BEGIN
  IF (curVar>=0) AND (noVars>0) THEN
    (* open dialog box *)
    New(d, 20, hScreen-200, 400, 120);
    string[0]:= 0C;
    Append(string, "Erase fuzzy variable '");
    Append(string, lingVars[curVar]^.name);
    Append(string, "' ?");
    StaticString(d, string , 10, 80, Plain);
    Button(d, yes, FALSE, 50, 40, "YES");
    Button(d, no, TRUE, 200, 40, "NO");
    Open(d);
    REPEAT 
    UNTIL UserAction(d, item) AND ((item=yes) OR (item=no));
    (* if answer is yes, delete variable *)
    IF item=yes THEN
      Deallocate(lingVars[curVar]);
      lingVars[curVar]:= lingVars[noVars-1];
      DEC(noVars);
      changeVarMenu:= TRUE;
    END; (* IF *)
    Close(d);
    Dispose(d);
    curVar:= -1; curVal:= -1; DrawVal;
  END; (* IF *)
END EraseVar;

PROCEDURE EraseVal; (*----------------------------------------------------------------*)
  (*erase, after confirmation, the current value *)
VAR
  d:    Dialog;
  yes ,no, item: INTEGER;
  string: ARRAY [0..127] OF CHAR;
BEGIN
  IF (curVar>=0) AND (curVal>=0) THEN
    (* open dialog box *)
    New(d, 20, hScreen-200, 400, 120);
    string[0]:= 0C;
    Append(string, "Erase fuzzy value '");
    Append(string, lingVars[curVar]^.vals[curVal].name);
    Append(string, "' ?");
    StaticString(d, string , 10, 80, Plain);
    Button(d, yes, FALSE, 50, 40, "YES");
    Button(d, no, TRUE, 200, 40, "NO");
    Open(d);
    REPEAT
    UNTIL UserAction(d, item) AND ((item=yes) OR (item=no));
    IF item=yes THEN  (* if answer is yes, delete value *)
      WITH lingVars[curVar]^ DO
        vals[curVal].name:= vals[noVals-1].name;
        vals[curVal].mu  := vals[noVals-1].mu;
        DEC(noVals);
        IF noVals>0 THEN curVal:= 0 ELSE curVal:= -1 END;
        DrawVal;
        changeValMenu:= TRUE;
      END
    END;
    Close(d);
    Dispose(d);
  END
END EraseVal;


(*------------------------------------------------------------------------------------*)

VAR i: INTEGER;

BEGIN (* LingVarEditor *)
  
  (* install the menus *)
  SetMenu(FileMenu, "File|Load Declaration|Save As|(-|Exit"  );
  SetMenu(VarMenu,  VarMenuMainStr);
  SetMenu(ValMenu,  ValMenuMainStr); DisableMenu(ValMenu);
  SetMenu(SelValMenu, SelValMenuMainStr); DisableMenu(SelValMenu);
  enabledValMenu:= FALSE;
  SetMenu(HelpMenu, "Help|Show Help");
                            
  noVars:= 0;
  curVar:= -1;              (* no variable selected *)
  curVal:= -1;              (* no value selected    *)
  isValClipboard:= FALSE;
  isVarClipboard:= FALSE;
  
  
  (* calc size and position of LingVarWindow *)
  windowW:= 2*leftV+widthV;
  windowH:= bottomV+heightV+80;
  OpenGraphicWindow(window, 10, hScreen-windowH-50,
                            windowW, windowH, "LingVar Editor", Redraw);

  (* main loop *)
   
  LOOP
    GetMenuCmd(result, done);
    IF done THEN
      Windows.PlaceOnTop(window);  (* make window visible *)
      CASE result.menuID OF
        FileMenu: CASE result.menuCmd OF
                    SaveAsItem:    SaveAs;
                  | LoadItem:      LoadVars;
                  | ExitItem:      EXIT;
                  END; (* CASE *)
      | VarMenu:  CASE result.menuCmd OF
                  | VarInsertItem:    InsertVar; 
                  | BoundaryItem:     SetBoundaries;
                  | VarCopyItem:      IF curVar>= 0 THEN
                                        varClipboard:= lingVars[curVar]^;
                                        isVarClipboard:= TRUE;
                                      END; (* IF *)
                  | VarPasteItem:     IF (curVar>=0) AND isVarClipboard THEN
                                        (* copy all values to current variable *)
                                        tempName:= lingVars[curVar]^.name;
                                        lingVars[curVar]^:= varClipboard;
                                        lingVars[curVar]^.name:= tempName;
                                      END; (* IF *)
                                      DrawVal;
                  | VarEraseItem:     EraseVar;
                  ELSE
                    (* select a variable *)
                    curVar:= result.menuCmd-firstVarList;
                    IF lingVars[curVar]^.noVals>0 THEN curVal:= 0 END;
                    DrawVal;
                    changeValMenu:= TRUE;
                  END; (* CASE *)
      | ValMenu:  CASE result.menuCmd OF
                  | ValInsertItem: InsertVal;
                  | ValEditItem:   EditVal;
                  | ValInvertItem: InvertVal;
                  | ValRotateItem: RotateVal;
                  | ValCopyItem:   IF (curVar>=0) AND (curVal>=0) THEN
                                     valClipboard:= lingVars[curVar]^.vals[curVal];
                                     isValClipboard:= TRUE;
                                   END; (* IF *)
                  | ValPasteItem:  IF (curVar>=0) AND (curVal>=0) AND isValClipboard THEN
                                     (* copy clipboard to current value *)
                                     tempName:= lingVars[curVar]^.vals[curVal].name;
                                     lingVars[curVar]^.vals[curVal]:= valClipboard;
                                     lingVars[curVar]^.vals[curVal].name:= tempName;
                                     changeValMenu:= TRUE;
                                   END; (* IF *)
                                   DrawVal;
                  | ValEraseItem:  EraseVal;
                  ELSE
                    (* should not occur - error *)
                  END; (* CASE *)
      | SelValMenu: (* select and (re-)draw value *)
                    curVal:= result.menuCmd-firstValList;
                    DrawVal;
      | HelpMenu:   (* show help file *)
                    ShowFile(HelpFile);
      END; (* CASE *)
    END; (* IF *)
    
    (* recalculate menus for selection *)
    
    IF changeVarMenu THEN
      VarMenuStr[0]:= 0C;
      Append(VarMenuStr, VarMenuMainStr);  (* header and main commands *)
      (* append variables to select *)
      FOR i:= 0 TO noVars-1 DO
        AppendCh(VarMenuStr, "|"); Append(VarMenuStr, lingVars[i]^.name);
      END; (* FOR *);
      SetMenu(VarMenu, VarMenuStr);
      changeVarMenu:= FALSE;
    END; (* IF *)
    
    IF changeValMenu THEN
      SelValMenuStr[0]:= 0C;
      Append(SelValMenuStr, SelValMenuMainStr); (* header and main commands *)
      IF curVar>=0 THEN
        (* append values to select *)
        FOR i:= 0 TO lingVars[curVar]^.noVals-1 DO
          AppendCh(SelValMenuStr, "|");
          Append(SelValMenuStr, lingVars[curVar]^.vals[i].name);
        END; (* FOR *);
      END; (* IF *)
      SetMenu(SelValMenu, SelValMenuStr);
      IF NOT enabledValMenu THEN DisableMenu(ValMenu); DisableMenu(SelValMenu); END;
      changeValMenu:= FALSE;
    END; (* IF *)
    
    IF curVar>= 0 THEN
      IF NOT enabledValMenu THEN
        EnableMenu(ValMenu); EnableMenu(SelValMenu); enabledValMenu:= TRUE;
      END
    ELSE
      IF enabledValMenu THEN
        DisableMenu(ValMenu); enabledValMenu:= FALSE;
      END
    END;
    
  END;(* LOOP *)
  
  CloseGraphicWindow(window);
  
  DeleteMenu(SelValMenu);
  DeleteMenu(ValMenu);
  DeleteMenu(VarMenu);
  DeleteMenu(FileMenu);
  
END LingVarEditor.
    
    
  