(************************************************************************************)
(* module name: FuzzyScanner                                                        *)
(* project: simulation of a fuzzy pathplanning algorithm                            *)(* author: markus stadler, IIIC/8                                                   *)
(* first implementation: 5/31/91                                                    *)
(* last change:          6/5/91                                                     *)
(************************************************************************************)

IMPLEMENTATION MODULE FuzzyScanner;

FROM FileSystem IMPORT File, Response, Lookup, Close, ReadChar;
FROM String     IMPORT Equal;
FROM System     IMPORT Allocate, Deallocate;
FROM SYSTEM     IMPORT TSIZE;

(* definition of constants *)

CONST    EOL = 15C;
         NUL = 0C;

(* definition of types *)

TYPE
  Descriptor     = POINTER TO DescriptorDesc;
  DescriptorDesc = RECORD
                     f:       File;
                     lineNo:  INTEGER;
                     comment: INTEGER;
                     ch:      CHAR;
                   END;

(* definition of global variables *)
      
(* utility procedures *)

PROCEDURE nextCh(d: Descriptor); (* reads the next character -------------------------*)
BEGIN
  ReadChar(d^.f, d^.ch);
  IF d^.f.eof THEN d^.ch:= NUL END;
END nextCh;

PROCEDURE ReadIdent(d: Descriptor);  (* reads an identifier --------------------------*)
  (* first character of identifier is already in 'ch' *)
VAR
  i: INTEGER;
BEGIN
  i:= 0;
  WHILE ((i<maxIdentLen) AND 
         (((CAP(d^.ch)>="A") AND (CAP(d^.ch)<="Z")) OR (d^.ch="_") OR (d^.ch="$") OR
            ((d^.ch>="0") AND (d^.ch<="9")))) DO
    Identifier[i]:= d^.ch;
    INC(i);
    nextCh(d);
  END; (* WHILE *)
  IF i<maxIdentLen-1 THEN Identifier[i]:= NUL END;
END ReadIdent;

PROCEDURE ReadRealNo(d: Descriptor); (* reads a real number --------------------------*)
  (* first digit of number is already in ch *)
VAR
  factor:  REAL;
  wasDot:  BOOLEAN;
BEGIN
  wasDot:= FALSE;
  RealNumber:= 0.0;
  factor:= 1.0;
  WHILE (((d^.ch>="0") AND (d^.ch<="9")) OR ((d^.ch=".") AND NOT wasDot)) DO
    IF d^.ch="." THEN wasDot:= TRUE (* is ok, really! *)
    ELSIF wasDot THEN
      factor:= factor/10.0;
      RealNumber:= RealNumber+FLOAT(ORD(d^.ch)-ORD("0"))*factor;
    ELSE
      RealNumber:= RealNumber*10.0+FLOAT(ORD(d^.ch)-ORD("0"));
    END; (* IF *)
    nextCh(d);
  END; (* WHILE *)
END ReadRealNo;

PROCEDURE Comment(d: Descriptor); (* reads until the end of a comment ----------------*)
BEGIN
  INC(d^.comment);  (* next level in nested comments *)
  LOOP
    nextCh(d);
    IF d^.ch=EOL THEN INC(d^.lineNo) END;
    IF d^.ch=NUL THEN d^.comment:= 0; EXIT END;
    IF d^.ch="(" THEN
      nextCh(d);
      IF d^.ch="*" THEN Comment(d) END;
    END;
    IF d^.ch="*" THEN
      nextCh(d);
      IF d^.ch=")" THEN
        nextCh(d); EXIT
      END
    END;
  END; (* LOOP *)
  DEC(d^.comment);
END Comment; 

(* exported procedures *)

PROCEDURE OpenFile(fileName: ARRAY OF CHAR; VAR desc: Descriptor; VAR ok: BOOLEAN); (*-*)
  (* opens the file 'fileName' to read definition of fuzzy database *)
VAR f: File;
BEGIN
  Lookup(f, fileName, FALSE);
  ok:= (f.res = done);
  IF ok THEN
    Allocate(desc, TSIZE(DescriptorDesc));
    desc^.f:= f;
    desc^.lineNo:= 1;
    desc^.comment:= 0;
    nextCh(desc)
  ELSE
    Close(f);    (* even when Lookup wasn't successful, close file *)
  END;
END OpenFile;
  
PROCEDURE CloseFile(d: Descriptor); (*------------------------------------------------*)
  (* closes the file opened by OpenFile before *)
BEGIN
  Close(d^.f);   
  Deallocate(d);  (* free memory of descriptor *)
END CloseFile;

PROCEDURE Get(d: Descriptor):SymbolType; (*--------------------------------------------*)
  (* returns the next symbol found, if EOF is reached, EofSy is returned *)
VAR
  s: SymbolType;
BEGIN
  (* jump over blanks, cr & linefeeds *)
  WHILE ((d^.ch>NUL) AND (d^.ch<=" ")) DO
    IF d^.ch=EOL THEN INC(d^.lineNo) END;
    nextCh(d);
  END; (* WHILE *)
  
  CASE d^.ch OF
    NUL     :  s:= EofSy;
  | "["     :  s:= OpenBraceSy; nextCh(d);
  | "]"     :  s:= CloseBraceSy; nextCh(d);
  | ";"     :  s:= SemicolonSy; nextCh(d);
  | ","     :  s:= CommaSy; nextCh(d);
  | "."     :  s:= PeriodSy; nextCh(d);
  | ":"     :  nextCh(d);
               IF d^.ch="=" THEN
                 s:= BecomesSy; nextCh(d);
               ELSE
                 s:= ColonSy
               END; (* IF *)
  | "("     :  nextCh(d);
               IF d^.ch="*" THEN
                 Comment(d);  (* jump over comment *)
                 IF d^.comment<0 THEN
                   s:= ErrorSy
                 ELSE
                   s:= Get(d);
                 END
               ELSE
                 s:= OpenParSy;
               END;
  | ")"     :  s:= CloseParSy; nextCh(d);
  | "0".."9":  ReadRealNo(d); s:= RealNoSy;
  | "-"     :  nextCh(d);  (* could be negative number or error *)
               IF ("0"<=d^.ch) AND (d^.ch<="9") THEN
                 ReadRealNo(d); RealNumber:= -RealNumber; s:= RealNoSy;
               ELSE
                 s:= ErrorSy; nextCh(d);
               END; (* IF *)
  | "a".."z",
    "A".."Z",
    "_", "$":  ReadIdent(d);
               (* check for following identifiers *)
               IF    Equal(Identifier, "LINGVAR"  ) THEN s:= LingvarSy
               ELSIF Equal(Identifier, "FUZZYVARS") THEN s:= FuzzyVarsSy
               ELSIF Equal(Identifier, "IN"       ) THEN s:= InSy
               ELSIF Equal(Identifier, "ON"       ) THEN s:= OnSy
               ELSIF Equal(Identifier, "WITH"     ) THEN s:= WithSy
               ELSIF Equal(Identifier, "END"      ) THEN s:= EndSy
               ELSIF Equal(Identifier, "VALUE"    ) THEN s:= ValueSy
               ELSIF Equal(Identifier, "IS"       ) THEN s:= IsSy
               ELSIF Equal(Identifier, "IF"       ) THEN s:= IfSy
               ELSIF Equal(Identifier, "THEN"     ) THEN s:= ThenSy
               ELSIF Equal(Identifier, "AND"      ) THEN s:= AndSy
               ELSIF Equal(Identifier, "OR"       ) THEN s:= OrSy
               ELSIF Equal(Identifier, "NOT"      ) THEN s:= NotSy
               ELSE  s:= IdentSy
               END; (* IF *)
  ELSE 
               s:= ErrorSy; nextCh(d);
  END; (* CASE *)
  RETURN s
END Get;

PROCEDURE LineNo(d: Descriptor):INTEGER; (*-------------------------------------------*)
  (* returns the current linenumber; can be used to add linenumber to *)
  (* error messages found by the parser                               *)
BEGIN
  RETURN d^.lineNo
END LineNo;

END FuzzyScanner.

