/* $RCSfile: flie.c,v $ $Date: 1993/08/05 03:51:54 $ $Revision: 1.45 $  */
/*                                                                      */
/* adapted from the original flie of Institute of Robotics, ETH, Zurich */
/*                                                                      */
/* by G.K. Egan + A. Sekercioglu,                                       */    
/* The Laboratory for Concurrent Computing Systems,                     */  
/* Swinburne University of Technology, Melbourne, Australia             */
/* for correspondence: Ahmet Sekercioglu ( yas@swin.oz.au )             */
/*                                                                      */
/* notes:                                                               */
/* 1. This version of flie only handles trapezoidal, triangular and     */
/*    rectangular ( crisp ! ) fuzzy sets.                               */
/* 2. Parser error recovery is very primitive.                          */
/*                                                                      */

#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <sys/types.h>
#include <math.h>

#include "hlnfs.h"

#define extern /*extern*/
#include "flie.h"
#undef extern

#define MAX_IDENT_LENGTH   32
#define MAX_TABLE          512
#define MAX_STRING_LENGTH  32
#define MAX_LINE_LENGTH    512

typedef char Identifier[MAX_IDENT_LENGTH];
typedef char String[MAX_STRING_LENGTH];

typedef enum {
  LingVarSy, FuzzyVarsSy, InSy, OnSy, WithSy, EndSy, OpenBraceSy,
  CloseBraceSy, OpenParSy, CloseParSy, ClassSy, IsSy, IfSy, ThenSy, AndSy,
  OrSy, NotSy, SemicolonSy, CommaSy, ColonSy, SlashSy, PeriodSy, BecomesSy,
  RealNoSy, IdentSy, EofSy, ErrorSy
} SymbolType;

typedef enum {
  OpAND, OpOR, OpNOT, OpIS
} OperatorType;

typedef struct OperatorDesc {
  OperatorType OpType;
  union {
    struct {
      struct OperatorDesc *LeftOp;
      struct OperatorDesc *RightOp;
    } BinExp;
    struct OperatorDesc *OnlyOp;
    struct {
      int ExpVar;        /* If operator type is "OpIS" then these fields */
      int ExpClass;      /* contain the indexes of "expression variable" */
                          /* and "fuzzy class" of the this variable       */
                          /* in the "VariableArray" and the "ClassArray"  */
                          /* respectively.                                */
    } Term;
  } ExpTree;                   
} OperatorDesc;

typedef struct RuleRec {
  boolean Alive;
  int    RuleVar;            /* Index of the output variable in the      */
                              /* RuleArray.                               */
  int    RuleClass;          /* Index of the fuzzy class of the output   */
                              /* variable in the ClassArray.              */
  double  RuleStrength;       /* Current confidence level of the rule as  */
                              /* calculated by the "ApplyRules" routine.  */
  double  AdaptationWeight;   /* Used only by adaptive version. Initia-   */
                              /* lized to 1.0 and current value is        */
                              /* updated by the Decision Unit dynamically.*/
                              /* always has a value in interval [0.0,2.0] */
  OperatorDesc *Expression;
} RuleRec;

typedef RuleRec RuleArray[MAX_RULES];

typedef struct TableRec {
  Identifier Name;
  enum {
    IsVarName, IsClassName
  } Kind;
  int ClassVarName;
  int ClassNo;
} TableRec;

typedef TableRec TableArray[MAX_TABLE];

/* prototypes of static functions:  */
static int P_eof( FILE *f );
static int P_eoln( FILE *f );
static void WriteIdent( FILE *FN, char *Ident);
static int Length( char *s);
static boolean Equal( char *Id, char *t );
static void report_error( char *s_ );
static void Nextch( FILE *FN );
static void ReadIdent( FILE *FN);
static void ReadRealNo( FILE *FN );
static void Comment( FILE *FN );
static SymbolType Get( FILE *FN );
static void ComputeAreaAndMoment( ClassRec *CurrClass, int NoSyn);
static void Synchronize( FILE *FN, SymbolType s );
static void ParseClassDecl( FILE *FN, int VarName, 
			      ClassRec *Class,int *NoClasses );
static void ParseVarDecl( FILE *FN );
static void ParseVars( FILE *FN );
static boolean FindClassNoForVar( char *Ident_, int VarName, int *ClassNo);
static OperatorDesc *ParseLogicExpr( FILE *FN );
static void ParseRule( FILE *FN );
static void SimplifyRuleBase();
static void ShowRule( FILE *FN, int RuleNo);
static void Explain( FILE *FN);
static void Fuzzify( double Inp, ClassRec *class );
static void FuzzifyInputVariables();
static double CalcLogicExpr( OperatorDesc *expr );
static void ApplyRules();
static void DeFuzzifyOutputVariables();

/* function-block wide global declarations: */
static char rcs_id[]="$Id: flie.c,v 1.45 1993/08/05 03:51:54 yas Exp $";
static RuleArray Rules;
static TableArray Table;
static boolean IsUpper[256];
static boolean IsAlpha[256];
static boolean IsBlank[256];
static boolean IsDigit[256];
static int tx;
static double RealNumber;
static Identifier Id;
static SymbolType Symbol;
static char Line[MAX_LINE_LENGTH];
static int ll;
static int cc;
static int CommentDepth;
static int LineNo;
static char ch;
static boolean ParsingOK;


static int P_eof( f ) 
     FILE *f; 
{ 

  /* Check if at end of file, using Pascal "eof" semantics.  End-of-file   */
  /* for stdin is broken; remove the special case for it to be broken in a */
  /* different way.                                                        */

  register int ch;

  if ( feof( f ) )
    return 1;
  if ( f == stdin )
    return 0;    /* not safe to look-ahead on the keyboard! */
  ch = getc(f);
  if ( ch == EOF )
    return 1;
  ungetc( ch, f );
  return 0;
}


static int P_eoln( f )
     FILE *f;
{
  
  /* Check if at end of line (or end of entire file). */

  register int ch;
  
  ch = getc(f);
  if (ch == EOF)
    return 1;
  ungetc(ch, f);
  return (ch == '\n');
}


static void WriteIdent( FN, Ident )
     FILE *FN;
     char *Ident;
{
  int i;

  i = 1;
  while ( !IsBlank[Ident[i - 1]] ) {
    putc( Ident[i - 1], FN );
    i++;
  } /* while */
}  /* WriteIdent */


static int Length( s )
     char *s;
{
  int i;

  i = MAX_STRING_LENGTH;
  while (s[i - 1] == ' ')
    i--;
  return i;
}  /* Length */


static boolean Equal( Id, t )
     char *Id;
     char *t;
{
  int i;
  boolean Different;

  Different = FALSE;
  i = 1;
  do {
    if (Id[i - 1] != t[i - 1])
      Different = TRUE;
    else
      i++;
  } while ( !( Different || 
               i == MAX_IDENT_LENGTH ||
	       ( Id[i - 1] == ' ' && t[i - 1] == ' ' ) 
	     ) 
	  );
  return ( !Different );
}  /* Equal */

static void ReportError(s_)
char *s_;
{
  String s;
  int i, FORLIM;

  memcpy(s, s_, sizeof(String));
  fprintf(stderr, "%6ld : ", LineNo);
  FORLIM = ll;
  for (i = 0; i < FORLIM; i++)
    putc(Line[i], stderr);
  fprintf(stderr, "\n%*c^", (int)(cc + 9), ' ');
  FORLIM = Length(s);
  for (i = 0; i < FORLIM; i++)
    putc(s[i], stderr);
  putc('\n', stderr);
  ParsingOK = FALSE;
}  /* ReportError */


static void Nextch(FN)
FILE *FN;
{
  if (cc < ll) {
    ch = Line[cc - 1];
    cc++;
    return;
  }
  if (P_eof(FN)) {
    ch = ' ';
    return;
  }
  ll = 0;
  cc = 1;
  while (!P_eoln(FN)) {
    ll++;
    Line[ll - 1] = getc(FN);
    if (Line[ll - 1] == '\n')
      Line[ll - 1] = ' ';
  }
  ll++;
  Line[ll - 1] = ' ';
  if (!P_eof(FN)) {
    fscanf(FN, "%*[^\n]");
    getc(FN);
  }
  LineNo++;
  ch = Line[0];
  cc = 2;
}  /* Nextch */


static void ReadIdent(FN)
FILE *FN;
{
  int i;

  i = 1;
  while (i <= MAX_IDENT_LENGTH &&
	 (IsAlpha[ch] || IsDigit[ch] || ch == '_' || ch == '$')) {
    if (IsUpper[ch])
      ch = ((ch) -'A'+'a');
    Id[i - 1] = ch;
    i++;
    Nextch(FN);
  }
  if (i < MAX_IDENT_LENGTH)
    Id[i - 1] = ' ';
}  /* ReadIdent */


static void ReadRealNo(FN)
FILE *FN;
{
  double factor;
  boolean wasDot;

  wasDot = FALSE;
  RealNumber = 0.0;
  factor = 1.0;
  while ((IsDigit[ch] || ch == '.') && !wasDot) {
    if (ch == '.')
      wasDot = TRUE;
    else if (wasDot) {
      factor /= 10.0;
      RealNumber += (ch - '0') * factor;
    } else
      RealNumber = RealNumber * 10.0 + ch - '0';
    Nextch(FN);
  }
}  /* ReadRealNo */


static void Comment(FN)
FILE *FN;
{
  boolean Finish;

  CommentDepth++;
  Finish = FALSE;
  do {
    Nextch(FN);
    if (ch == '(') {
      Nextch(FN);
      if (ch == '*')
	Comment(FN);
    }
    if (ch == '*') {
      Nextch(FN);
      if (ch == ')') {
	Nextch(FN);
	Finish = TRUE;
      }
    }
  } while (!Finish);
  CommentDepth--;
}  /* Comment */


static SymbolType Get(FN)
FILE *FN;
{
  SymbolType s;

  while (IsBlank[ch] & (!P_eof(FN)))
    Nextch(FN);
  if (P_eof(FN)) {
    s = EofSy;
    return s;
  }
  if (IsAlpha[ch] || ch == '_' || ch == '$') {
    ReadIdent(FN);
    if (Equal(Id, "is                              ")) {
      s = IsSy;
      return s;
    }
    if (Equal(Id, "if                              ")) {
      s = IfSy;
      return s;
    }
    if (Equal(Id, "then                            ")) {
      s = ThenSy;
      return s;
    }
    if (Equal(Id, "and                             ")) {
      s = AndSy;
      return s;
    }
    if (Equal(Id, "or                              ")) {
      s = OrSy;
      return s;
    }
    if (Equal(Id, "not                             ")) {
      s = NotSy;
      return s;
    }
    if (Equal(Id, "lingvar                         ")) {
      s = LingVarSy;
      return s;
    }
    if (Equal(Id, "fuzzyvars                       ")) {
      s = FuzzyVarsSy;
      return s;
    }
    if (Equal(Id, "in                              ")) {
      s = InSy;
      return s;
    }
    if (Equal(Id, "class                           ")) {
      s = ClassSy;
      return s;
    }
    if (Equal(Id, "on                              ")) {
      s = OnSy;
      return s;
    }
    if (Equal(Id, "with                            ")) {
      s = WithSy;
      return s;
    }
    if (Equal(Id, "end                             "))
      s = EndSy;
    else
      s = IdentSy;
    return s;
  }
  switch (ch) {

  case '[':
    s = OpenBraceSy;
    Nextch(FN);
    break;

  case ']':
    s = CloseBraceSy;
    Nextch(FN);
    break;

  case ';':
    s = SemicolonSy;
    Nextch(FN);
    break;

  case ',':
    s = CommaSy;
    Nextch(FN);
    break;

  case '.':
    s = PeriodSy;
    Nextch(FN);
    break;

  case '/':
    s = SlashSy;
    break;

  case ':':
    Nextch(FN);
    if (ch == '=') {
      s = BecomesSy;
      Nextch(FN);
    } else
      s = ColonSy;
    break;

  case '(':
    Nextch(FN);
    if (ch == '*') {
      Comment(FN);
      if (CommentDepth < 0)
	s = ErrorSy;
      else
	s = Get(FN);
    } else
      s = OpenParSy;
    break;

  case ')':
    s = CloseParSy;
    Nextch(FN);
    break;

  case '0':
  case '1':
  case '2':
  case '3':
  case '4':
  case '5':
  case '6':
  case '7':
  case '8':
  case '9':
    ReadRealNo(FN);
    s = RealNoSy;
    break;

  case '-':
    Nextch(FN);
    if (IsDigit[ch]) {
      ReadRealNo(FN);
      RealNumber = -RealNumber;
      s = RealNoSy;
    } else
      s = ErrorSy;
    break;
  }
  return s;
}  /* Get */

boolean FoundVar(VarIdent_, N)
char *VarIdent_;
int *N;
{
  Identifier VarIdent;
  boolean Found;

  memcpy(VarIdent, VarIdent_, sizeof(Identifier));
  *N = NoVars - 1;
  Found = FALSE;
  while (*N >= 0 && !Found) {
    if (Equal(VarIdent, Table[Variables[*N].VarName].Name))
      Found = TRUE;
    else
      (*N)--;
  }
  return Found;
}  /* FoundVar */

static void Synchronise(FN, s, m)
FILE *FN;
SymbolType s;
char *m;
{
  ReportError(m);
  while (Symbol != s && Symbol != EofSy)
    Symbol = Get(FN);
}  /* Synchronise */

static double ComputeClassArea( Synonym )
     SynonymRec *Synonym;
{
  double class_area;

  /* note : height of the trapezoid is 1.0 */
  /* first triangular region: */
  class_area = ( fabs( Synonym->d - Synonym->c ) ) * 0.5;
  /* rectangular region: */
  class_area = class_area + ( fabs( Synonym->c - Synonym->b ) );
  /* second triangular region: */
  class_area = class_area + 
    ( fabs( Synonym->b - Synonym->a ) ) * 0.5;
  return( class_area );
} /* ClassArea */


static double ComputeClassYMoment( Synonym )
     SynonymRec *Synonym;
{
  double class_y_moment;

  /* note : 1. height of the trapezoid is 1.0                              */
  /*        2. Whenever c = d and a = b,   */
  /*           the special case must be handled to avoid division by zero. */

  /* first triangular region: */
  if ( Synonym->c == Synonym->d ) {
    class_y_moment = 0;
  }
  else {
    class_y_moment = ( ( pow( Synonym->d, (double) 3.0 ) -
                         pow( Synonym->c, (double) 3.0 ) 
                       ) /
                       ( 3 * ( Synonym->c - Synonym->d ) )
                     ) +
                     ( ( ( -( Synonym->c ) ) / 
                       ( Synonym->c - Synonym->d ) ) + 1 
                     ) * 0.5 *
                     ( pow( Synonym->d, (double) 2.0 ) -
		       pow( Synonym->c, (double) 2.0 ) 
                     );
  } /* if */
  /* rectangular region: */
  if ( Synonym->b != Synonym->c ) {
    /* This if statement is only for saving the computation time, */
    /* there is no danger of division by zero.                    */
    class_y_moment = class_y_moment + ( pow( Synonym->c, (double) 2.0 ) -
	         		        pow( Synonym->b, (double) 2.0 ) 
                                      ) * 0.5;
  } /* if */
  /* second triangular region: */
  if ( Synonym->a != Synonym->b ) {
    class_y_moment = class_y_moment + 
	               ( ( ( pow( Synonym->b, (double) 3.0 ) -
		             pow( Synonym->a, (double) 3.0 ) 
		           ) /
		           ( -3 * ( Synonym->a - Synonym->b ) )
		         ) +
                         ( Synonym->a / ( 2 * ( Synonym->a - 
                                                      Synonym->b
			  	   	            )
                                              ) *
                           ( pow( Synonym->b, (double) 2.0 ) -
                             pow( Synonym->a, (double) 2.0 ) 
                           )
		         )
                       );
  } /* if */
  return( class_y_moment );
} /* ClassYMoment */

static void ComputeAreaAndMoment(CurrClass, NoSyn)
ClassRec *CurrClass;
int NoSyn;
{
  int s;
  double a0, a1, a2;
  SynonymRec *CurrSyn;

  for (s = 0; s < NoSyn; s++) {
    CurrSyn = &CurrClass->Synonym[s];
#ifdef GKE
    a0 = (CurrSyn->b - CurrSyn->a) / 2.0;
    a1 = CurrSyn->c - CurrSyn->b;
    a2 = (CurrSyn->d - CurrSyn->c) / 2.0;
    CurrSyn->ClassArea = a0 + a1 + a2;
    CurrSyn->ClassMoment = a0 * (CurrSyn->a + (CurrSyn->b - CurrSyn->a) * 2.0 / 3.0) +
			 a1 * (CurrSyn->b + CurrSyn->c) / 2.0 +
			 a2 * (CurrSyn->c + (CurrSyn->d - CurrSyn->c) / 3.0);
#else
    CurrSyn->ClassArea = ComputeClassArea(CurrSyn);
    CurrSyn->ClassMoment = ComputeClassYMoment(CurrSyn);
#endif
  }
}  /* ComputeAreaAndMoment */


static void ParseClassDecl(FN, VarName, Class, NoClasses)
FILE *FN;
int VarName;
ClassRec *Class;
int *NoClasses;
{
  int i;
  TableRec *TableEntry;
  ClassRec *CurrClass;
  int FORLIM;
  SynonymRec *CurrSyn;

  if (Symbol != ClassSy) {
    Synchronise(FN, ClassSy, "CLASS expected                  ");
    return;
  }
  Symbol = Get(FN);
  if (Symbol != IdentSy) {
    Synchronise(FN, ClassSy, "Identifier expected             ");
    return;
  }
  if (*NoClasses >= MAX_CLASSES || tx >= MAX_TABLE) {
    Synchronise(FN, ClassSy, "too many fuzzy classes defined  ");
    return;
  }
  (*NoClasses)++;
  tx++;
  TableEntry = &Table[tx];
  CurrClass = &Class[*NoClasses - 1];
  memcpy(TableEntry->Name, Id, sizeof(Identifier));
  TableEntry->Kind = IsClassName;
  TableEntry->ClassVarName = VarName;
  TableEntry->ClassNo = *NoClasses - 1;
  CurrClass->ClassName = tx;
  CurrClass->CurrClassStrength = 0.0;
  CurrClass->NewClassStrength = 0.0;
  CurrClass->Mentioned = FALSE;
  Symbol = Get(FN);
  if (Symbol == OpenParSy) {
    Symbol = Get(FN);
    if (Symbol == RealNoSy) {
      CurrClass->NoInflexions = (int)RealNumber;
      Symbol = Get(FN);
      if (Symbol == CloseParSy)
	Symbol = Get(FN);
      else
	ReportError(") expected                      ");
    } else {
      ReportError("Number expected                 ");
      CurrClass->NoInflexions = 4;
      Symbol = Get(FN);
    }
  } else
    CurrClass->NoInflexions = 4;
  if (Symbol != IsSy) {
    Synchronise(FN, ClassSy, "IS or ( expected                ");
    return;
  }
  Symbol = Get(FN);
  CurrClass->NoSynonyms = CurrClass->NoInflexions / 4;
  FORLIM = CurrClass->NoSynonyms;
  for (i = 0; i < FORLIM; i++) {
    CurrSyn = &CurrClass->Synonym[i];
    CurrSyn->a = RealNumber;
    Symbol = Get(FN);
    CurrSyn->b = RealNumber;
    Symbol = Get(FN);
    CurrSyn->c = RealNumber;
    Symbol = Get(FN);
    CurrSyn->d = RealNumber;
    Symbol = Get(FN);
  }
  CurrClass->DominantRule = NULL_RULE;
  CurrClass->NewClassStrength = 0.0;
  CurrClass->CurrClassStrength = 0.0;
  CurrClass->NewSyn = 0;
  CurrClass->CurrSyn = 0;
  ComputeAreaAndMoment(&Class[*NoClasses - 1], (int)CurrClass->NoSynonyms);
}  /* ParseClassDecl */

static void ParseVarDecl(FN)
FILE *FN;
{
  TableRec *TableEntry;
  VarRec *CurrVar;



  if (Symbol != LingVarSy) {
    Synchronise(FN, LingVarSy, "LINGVAR expected                ");
    return;
  }
  if (tx >= MAX_TABLE) {
    Synchronise(FN, LingVarSy, "to many identifiers             ");
    return;
  }
  if (NoVars >= MAX_VARIABLES) {
    Synchronise(FN, LingVarSy, "to many fuzzy variables         ");
    return;
  }
  tx++;
  NoVars++;
  Symbol = Get(FN);
  if (Symbol != IdentSy) {
    Synchronise(FN, LingVarSy, "Identifier expected             ");
    return;
  }
  TableEntry = &Table[tx];
  CurrVar = &Variables[NoVars - 1];
  TableEntry->Kind = IsVarName;
  memcpy(TableEntry->Name, Id, sizeof(Identifier));
  CurrVar->VarName = tx;
  CurrVar->NoClasses = 0;
  CurrVar->CurrVal = 0.0;
  CurrVar->CurrVal = 0.0;
  CurrVar->VarKind = InputVar;
  Symbol = Get(FN);
  if (Symbol != OnSy) {
    Synchronise(FN, LingVarSy, "ON expected                     ");
    return;
  }
  Symbol = Get(FN);
  if (Symbol != OpenBraceSy) {
    Synchronise(FN, LingVarSy, "[ expected                      ");
    return;
  }
  Symbol = Get(FN);
  if (Symbol != RealNoSy) {
    Synchronise(FN, LingVarSy, "Number expected                 ");
    return;
  }
  CurrVar->VarLowerB = RealNumber;
  Symbol = Get(FN);
  if (Symbol == CommaSy)
    Symbol = Get(FN);
  if (Symbol != RealNoSy) {
    Synchronise(FN, LingVarSy, "Number expected                 ");
    return;
  }
  CurrVar->VarUpperB = RealNumber;
  CurrVar->FailSafe = (CurrVar->VarUpperB + CurrVar->VarLowerB) * 0.5;
  CurrVar->MaxSlewRate = CurrVar->VarUpperB - CurrVar->VarLowerB;
  Symbol = Get(FN);
  if (Symbol != CloseBraceSy) {
    Synchronise(FN, LingVarSy, "] expected                      ");
    return;
  }
  Symbol = Get(FN);
  if (Symbol != WithSy) {
    Synchronise(FN, LingVarSy, "WITH expected                   ");
    return;
  }
  Symbol = Get(FN);
  while (Symbol == ClassSy)
    ParseClassDecl(FN, CurrVar->VarName, CurrVar->Class, &CurrVar->NoClasses);
  if (Symbol == EndSy)
    Symbol = Get(FN);
  else
    Synchronise(FN, LingVarSy, "end expected                    ");
}  /* ParseVarDecl */


static void ParseVars(FN)
FILE *FN;
{
  while (Symbol == LingVarSy)
    ParseVarDecl(FN);
}  /* ParseVars */


static boolean FoundClassNoForVar(Ident_, VarName, ClassNo)
char *Ident_;
int VarName, *ClassNo;
{
  Identifier Ident;
  boolean Found;
  int Name;

  memcpy(Ident, Ident_, sizeof(Identifier));
  Found = FALSE;
  Name = tx;
  while (Name >= 0 && !Found) {
    if (Equal(Ident, Table[Name].Name) &&
	Table[Name].ClassVarName == Variables[VarName].VarName)
      Found = TRUE;
    else
      Name--;
  }
  if (Found)
    *ClassNo = Table[Name].ClassNo;
  else
    *ClassNo = 0;
  return Found;
}  /* FoundClassNoForVar */


static OperatorDesc *ParseLogicExpr(FN)
FILE *FN;
{
  OperatorDesc *Result, *Op;

  Op = (OperatorDesc *)malloc(sizeof(OperatorDesc));
  if (Symbol == IdentSy) {
    Op->OpType = OpIS;
    if (!FoundVar(Id, &Op->ExpTree.Term.ExpVar)) {
      ReportError("variable not defined            ");
      free(Op);
      Result = NULL;
    } else {
      Symbol = Get(FN);
      if (Symbol == IsSy) {
	Symbol = Get(FN);
	if (Symbol == IdentSy) {
	  if (!FoundClassNoForVar(Id, Op->ExpTree.Term.ExpVar, &Op->ExpTree.Term.ExpClass)) {
	    WriteIdent(stderr,
		       Table[Variables[Op->ExpTree.Term.ExpVar].VarName].Name);
	    putc(' ', stderr);
	    WriteIdent(stderr, Id);
	    putc('\n', stderr);
	    ReportError("class not defined for variable  ");
	    free(Op);
	    Result = NULL;
	  } else
	    Variables[Op->ExpTree.Term.ExpVar].Class[Op->ExpTree.Term.ExpClass].Mentioned = TRUE;
	} else {
	  ReportError("Identifier expected             ");
	  free(Op);
	  Result = NULL;
	}
      } else {
	ReportError("IS expected                     ");
	free(Op);
	Result = NULL;
      }
    }
  } else if (Symbol == NotSy) {
    Op->OpType = OpNOT;
    Symbol = Get(FN);
    if (Symbol == OpenParSy) {
      Symbol = Get(FN);
      Op->ExpTree.OnlyOp = ParseLogicExpr(FN);
      if (Op->ExpTree.OnlyOp == NULL) {
	free(Op);
	Result = NULL;
      } else if (Symbol != CloseParSy) {
	ReportError(") expected                      ");
	Result = NULL;
      }
    } else {
      ReportError("( expected                      ");
      Result = NULL;
    }
  } else if (Symbol == OpenParSy) {
    Symbol = Get(FN);
    Op->ExpTree.BinExp.LeftOp = ParseLogicExpr(FN);
    if (Op->ExpTree.BinExp.LeftOp == NULL) {
      free(Op);
      Result = NULL;
    } else if (Symbol == CloseParSy) {
      Symbol = Get(FN);
      if (Symbol == OrSy)
	Op->OpType = OpOR;
      else if (Symbol == AndSy)
	Op->OpType = OpAND;
      else if (Symbol == ThenSy) {
	ReportError("and or or expected              ");
	free(Op);
	Result = NULL;
      }
      Symbol = Get(FN);
      if (Symbol == OpenParSy) {
	Symbol = Get(FN);
	Op->ExpTree.BinExp.RightOp = ParseLogicExpr(FN);
	if (Op->ExpTree.BinExp.RightOp == NULL) {
	  free(Op);
	  Result = NULL;
	} else if (Symbol != CloseParSy) {
	  ReportError(") expected                      ");
	  Result = NULL;
	}
      } else {
	ReportError("( expected                      ");
	Result = NULL;
      }
    } else {
      ReportError(") expected                      ");
      Result = NULL;
    }
  } else {
    free(Op);
    Result = NULL;
  }
  Symbol = Get(FN);
  return Op;
}  /* ParseLogicExpr */


static void ParseRule(FN)
FILE *FN;
{
  OperatorDesc *Expr;
  RuleRec *CurrRule;

  if (NoRules >= MAX_RULES) {
    Synchronise(FN, IfSy, "too many rules defined          ");
    return;
  }
  if (Symbol != IfSy) {
    Synchronise(FN, IfSy, "if expected                     ");
    return;
  }
  Symbol = Get(FN);
  Expr = ParseLogicExpr(FN);
  if (Expr == NULL) {
    Synchronise(FN, IfSy, "logical expression invalid      ");
    return;
  }
  NoRules++;
  CurrRule = &Rules[NoRules - 1];
  CurrRule->Alive = TRUE;
  CurrRule->AdaptationWeight = 1.0;
  CurrRule->Expression = Expr;
  if (Symbol != ThenSy) {
    Synchronise(FN, IfSy, "then expected                   ");
    return;
  }
  Symbol = Get(FN);
  if (Symbol != IdentSy) {
    Synchronise(FN, IfSy, "Identifier expected             ");
    return;
  }
  if (!FoundVar(Id, &CurrRule->RuleVar)) {
    Synchronise(FN, IfSy, "variable not declared           ");
    return;
  }
  Symbol = Get(FN);
  if (Symbol != BecomesSy) {
    Synchronise(FN, IfSy, ":= expected                     ");
    return;
  }
  Symbol = Get(FN);
  if (Symbol != IdentSy) {
    Synchronise(FN, IfSy, "Identifier expected             ");
    return;
  }
  if (!FoundClassNoForVar(Id, CurrRule->RuleVar, &CurrRule->RuleClass)) {
    Synchronise(FN, IfSy, "class not declared for variable ");
    return;
  }
  Variables[CurrRule->RuleVar].VarKind = OutputVar;
  Variables[CurrRule->RuleVar].Class[CurrRule->RuleClass].Mentioned = TRUE;
  Symbol = Get(FN);
  if (Symbol == SemicolonSy)
    Symbol = Get(FN);
  else
    Synchronise(FN, IfSy, "; expected                      ");
}  /* ParseRule */


static void CheckRuleBase(FN)
FILE *FN;
{
  int v, c, FORLIM;
  VarRec *CurrVar;
  int FORLIM1;
  ClassRec *CurrClass;

  fprintf(FN,
    "The following classes (where listed) were not mentioned in any rule\n");
  FORLIM = NoVars;
  for (v = 0; v < FORLIM; v++) {
    CurrVar = &Variables[v];
    fprintf(FN, "\nlingvar: ");
    WriteIdent(FN, Table[CurrVar->VarName].Name);
    putc('\n', FN);
    FORLIM1 = CurrVar->NoClasses;
    for (c = 0; c < FORLIM1; c++) {
      CurrClass = &CurrVar->Class[c];
      if (!CurrClass->Mentioned) {
	putc('\t', FN);
	WriteIdent(FN, Table[CurrClass->ClassName].Name);
	putc('\n', FN);
      }
    }
  }
}  /* CheckRuleBase */


static void SimplifyRuleBase()
{
  int i, j;
  OperatorDesc *Expr;

  i = 0;
  while (i < NoRules) {
    j = i + 1;
    while (j < NoRules) {
      if (Rules[i].RuleVar != Rules[j].RuleVar ||
	  Rules[i].RuleClass != Rules[j].RuleClass) {
	j++;
	continue;
      }
      Expr = (OperatorDesc *)malloc(sizeof(OperatorDesc));
      Expr->OpType = OpOR;
      Expr->ExpTree.BinExp.LeftOp = Rules[i].Expression;
      Expr->ExpTree.BinExp.RightOp = Rules[j].Expression;
      Rules[i].Expression = Expr;
      Rules[j] = Rules[NoRules - 1];
      NoRules--;
    }
    i++;
  }
}  /* SimplifyRuleBase */


boolean LoadRuleBase(FN)
FILE *FN;
{
  int c;
  short TEMP;
  int i;

  for (TEMP = '\0'; TEMP <= 255; TEMP++) {
    c = TEMP;
    IsAlpha[c] = FALSE;
    IsDigit[c] = FALSE;
    IsUpper[c] = FALSE;
  }
  for (c = '\0'; c <= ' '; c++)
    IsBlank[c] = TRUE;
  for (c = 'A'; c <= 'Z'; c++) {
    IsAlpha[c] = TRUE;
    IsUpper[c] = TRUE;
  }
  for (c = '0'; c <= '9'; c++)
    IsDigit[c] = TRUE;
  for (c = 'a'; c <= 'z'; c++)
    IsAlpha[c] = TRUE;
  cc = 1;
  ll = cc;
  ch = ' ';
  tx = -1;
  NoRules = 0;
  NoVars = 0;
  LineNo = 0;
  CommentDepth = 0;
  ParsingOK = TRUE;
  Symbol = Get(FN);
  ParseVars(FN);
  while (Symbol == IfSy)
    ParseRule(FN);
  if (Symbol != EndSy)
    ReportError("end expected                    ");
  if (ParsingOK)
    CheckRuleBase(stderr);
  /* determine the number of input variables...*/
  NoInputVars = 0;
  for ( i = 0 ; i < NoVars ; i++ ) {
    if ( Variables[i].VarKind == InputVar ) {
      NoInputVars++;
    } /* if */
  } /* for */

  return( ParsingOK);
}  /* LoadRuleBase */


static void ShowLogicExpr(FN, Expr)
FILE *FN;
OperatorDesc *Expr;
{
  putc('(', FN);
  if (Expr->OpType == OpIS) {
    WriteIdent(FN, Table[Variables[Expr->ExpTree.Term.ExpVar].VarName].Name);
    fprintf(FN, " is ");
    WriteIdent(FN,
      Table[Variables[Expr->ExpTree.Term.ExpVar].Class[Expr->ExpTree.Term.ExpClass].ClassName].
      Name);
    fprintf(FN, " [%3.1f]",
	    Variables[Expr->ExpTree.Term.ExpVar].Class[Expr->ExpTree.Term.ExpClass].
	    CurrClassStrength);
  } else if (Expr->OpType == OpAND) {
    ShowLogicExpr(FN, Expr->ExpTree.BinExp.LeftOp);
    fprintf(FN, "\n\t\tand ");
    ShowLogicExpr(FN, Expr->ExpTree.BinExp.RightOp);
  } else if (Expr->OpType == OpOR) {
    ShowLogicExpr(FN, Expr->ExpTree.BinExp.LeftOp);
    fprintf(FN, " or ");
    ShowLogicExpr(FN, Expr->ExpTree.BinExp.RightOp);
  } else {
    fprintf(FN, " not ");
    ShowLogicExpr(FN, Expr->ExpTree.OnlyOp);
  }
  putc(')', FN);
}  /* ShowLogicExpr */


static void ShowRule(FN, RuleNo)
FILE *FN;
int RuleNo;
{
  RuleRec *CurrRule;
  VarRec *CurrVar;

  CurrRule = &Rules[RuleNo];
  CurrVar = &Variables[CurrRule->RuleVar];
  fprintf(FN, "Rule%3ld: if ", RuleNo);
  ShowLogicExpr(FN, CurrRule->Expression);
  fprintf(FN, " then ");
  WriteIdent(FN, Table[CurrVar->VarName].Name);
  fprintf(FN, " is ");
  WriteIdent(FN, Table[CurrVar->Class[CurrRule->RuleClass].ClassName].Name);
  fprintf(FN, " [%4.2f]*{%4.2f}={%4.2f}\n", 
         CurrRule->RuleStrength, 
         CurrRule->AdaptationWeight,
         CurrRule->RuleStrength*CurrRule->AdaptationWeight);
}  /* ShowRule */


static void Explain(FN)
FILE *FN;
{
  int c;
  int o, FORLIM;
  VarRec *CurrVar;
  int FORLIM1;
  ClassRec *CurrClass;

  FORLIM = NoVars;
  for (o = 0; o < FORLIM; o++) {
    CurrVar = &Variables[o];
    if (CurrVar->VarKind == OutputVar) {
      fprintf(FN, "The explanation for ");
      WriteIdent(FN, Table[CurrVar->VarName].Name);
      fprintf(FN, " being set to %8.2f with confidence %5.2f is:\n",
	      CurrVar->CurrVal, CurrVar->CurrValConf);
      FORLIM1 = CurrVar->NoClasses;
      for (c = 0; c < FORLIM1; c++) {
	CurrClass = &CurrVar->Class[c];
	if (CurrClass->DominantRule != NULL_RULE) {
	  putc(' ', FN);
	  ShowRule(stdout, CurrClass->DominantRule);
	}
      }
    }
  }
}  /* Explain */


static void Fuzzify(Inp, CurrClass)
double Inp;
ClassRec *CurrClass;
{
  double VarStrength;
  int s, FORLIM;
  SynonymRec *CurrSyn;

  VarStrength = 0.0;
  FORLIM = CurrClass->NoSynonyms;
  for (s = 0; s < FORLIM; s++) {
    CurrSyn = &CurrClass->Synonym[s];
    if (Inp < CurrSyn->a || Inp > CurrSyn->d)
      VarStrength = 0.0;
    else if (Inp < CurrSyn->b)
      VarStrength = (Inp - CurrSyn->a) / (CurrSyn->b - CurrSyn->a);
    else if (Inp > CurrSyn->c)
      VarStrength = 1.0 + (CurrSyn->c - Inp) / (CurrSyn->d - CurrSyn->c);
    else
      VarStrength = 1.0;
    if (VarStrength > CurrClass->NewClassStrength) {
      CurrClass->NewClassStrength = VarStrength;
      CurrClass->NewSyn = s;
    }
  }
}  /* Fuzzify */

static double Window(lb, ub, Val)
double lb, ub, Val;
{
  if (Val < lb)
    return lb;
  else if (Val > ub)
    return ub;
  else
    return Val;
}  /* Window */

static void FuzzifyVariables()
{
  int v;
  int c;
  double WindowVal;
  int FORLIM;
  VarRec *CurrVar;
  int FORLIM1;

  FORLIM = NoVars;
  for (v = 0; v < FORLIM; v++) {
    CurrVar = &Variables[v];
    if (CurrVar->VarKind == InputVar) {
      WindowVal = Window(CurrVar->VarLowerB, CurrVar->VarUpperB, CurrVar->CurrVal);
      FORLIM1 = CurrVar->NoClasses;
      for (c = 0; c < FORLIM1; c++)
	Fuzzify(WindowVal, &CurrVar->Class[c]);
    }
  }
}  /* FuzzifyVariables */

static double CalcLogicExpr(Expr)
OperatorDesc *Expr;
{
  double a, b;

  if (Expr->OpType == OpIS)
    return (Variables[Expr->ExpTree.Term.ExpVar].Class[Expr->ExpTree.Term.ExpClass].
	    CurrClassStrength);
  else if (Expr->OpType == OpAND) {
    a = CalcLogicExpr(Expr->ExpTree.BinExp.LeftOp);
    b = CalcLogicExpr(Expr->ExpTree.BinExp.RightOp);
    if (a < b)
      return a;
    else
      return b;
  } else if (Expr->OpType == OpOR) {
    a = CalcLogicExpr(Expr->ExpTree.BinExp.LeftOp);
    b = CalcLogicExpr(Expr->ExpTree.BinExp.RightOp);
    if (a > b)
      return a;
    else
      return b;
  } else {
    return (1.0 - CalcLogicExpr(Expr->ExpTree.OnlyOp));

  }
}  /* CalcLogicExpr */

static void ApplyRules()
{
  int r;
  RuleRec *CurrRule;
  ClassRec *CurrClass;

  for (r = 0; r < NoRules; r++) {
    CurrRule = &Rules[r];
    if (CurrRule->Alive) {
      CurrClass = &Variables[CurrRule->RuleVar].Class[CurrRule->RuleClass];
      CurrRule->RuleStrength = CalcLogicExpr(CurrRule->Expression);
      if ((CurrRule->RuleStrength*CurrRule->AdaptationWeight) > CurrClass->NewClassStrength) {
	CurrClass->DominantRule = r;
	CurrClass->NewClassStrength = CurrRule->RuleStrength*CurrRule->AdaptationWeight;
      }
    }
  }
}  /* ApplyRules */

static double SlewLimitedVal(CurrVal, NewVal, MaxSlewRate)
double CurrVal, NewVal, MaxSlewRate;
{
  /*if ( fabs(CurrVal - NewVal) > MaxSlewRate )
    return ( CurrVal + (CurrVal - NewVal) / fabs(CurrVal - NewVal) * MaxSlewRate);
  else*/
    return NewVal;
}  /* SlewLimitedVal */




static void DeFuzzifyVariables()
{
  int v, c;
  VarRec *CurrVar;
  RuleRec *CurrRule;
  ClassRec *CurrClass;
  SynonymRec *CurrSyn;

  for (v = 0; v < NoVars; v++) {
    CurrVar = &Variables[v];
    if (CurrVar->VarKind == OutputVar) {
      CurrVar->TotalAreas = 0.0;
      CurrVar->TotalMoments = 0.0;
      for (c = 0; c < CurrVar->NoClasses; c++) {
	CurrClass = &CurrVar->Class[c];
	CurrSyn = &CurrClass->Synonym[CurrClass->NewSyn];
	CurrVar->TotalAreas += 
	  CurrSyn->ClassArea * CurrClass->NewClassStrength;
	CurrVar->TotalMoments += 
	  CurrSyn->ClassMoment * CurrClass->NewClassStrength;
        if (CurrClass->DominantRule != NULL_RULE) {
	  CurrRule = &Rules[CurrClass->DominantRule];
	  if (CurrRule->RuleStrength > CurrVar->CurrValConf)
	    CurrVar->CurrValConf = 
	      CurrRule->RuleStrength*CurrRule->AdaptationWeight;
	}
      }
      if (fabs(CurrVar->TotalAreas) > INDECISION_THRESHOLD)
	CurrVar->CurrVal = 
	  SlewLimitedVal( CurrVar->CurrVal,
	                  CurrVar->TotalMoments / CurrVar->TotalAreas, 
			  CurrVar->MaxSlewRate );
      else {
	fprintf( stderr,
	  "INDECISION PANIC: input fuzzy set strengths are too small to defuzzify variable [");
	WriteIdent(stderr, Table[CurrVar->VarName].Name);
	fprintf(stderr, "]\n");
/*	CurrVar->CurrVal = CurrVar->FailSafe;*/
      }
    }
  }
}  /* DeFuzzifyVariables */




static void UpdateClasses()
{
  int v, c;
  VarRec *CurrVar;
  ClassRec *CurrClass;

  for (v = 0; v < NoVars; v++) {
    CurrVar = &Variables[v];
    for (c = 0; c < CurrVar->NoClasses; c++) {
      CurrClass = &CurrVar->Class[c];
      CurrClass->CurrClassStrength = CurrClass->NewClassStrength;
      CurrClass->CurrSyn = CurrClass->NewSyn;
      CurrClass->NewClassStrength = 0.0;
      CurrClass->DominantRule = NULL_RULE;
    }
  }
}  /* UpdateClasses */


void FuzzyInferenceEngine()
{
  FuzzifyVariables();
  UpdateClasses();
  ApplyRules();
  DeFuzzifyVariables();
  if (GiveExplanation) {
    Explain(stdout);
  }
}  /* FuzzyInferenceEngine */

extern void DumpCurrentValues()
{
  int n;
  VarRec *CurrVar;

# ifdef DEBUG
    printf( "DumpCurrentValues...\n" );
# endif

  printf( "\n@@" );
  for ( n = 0 ; n < NoVars ; n++ ) {
    CurrVar = &Variables[n];
    printf( "%5.1f\t", CurrVar->CurrVal );
  } /* for */
  printf( "\n" );
} /* DumpCurrentValues */


extern void UpdateAdaptationWeights( v )
     vector *v;
{
  int i;

# ifdef DEBUG
  printf( "UpdateAdaptationWeights...\n" );
# endif
  for ( i= 0 ; i < v->size ; i++ ) {
    Rules[i].AdaptationWeight = v->elements[i];
#   ifdef DEBUG 
    printf( "rule no: %d\t AdaptationWeight: %5.1f\n", i, 
	   Rules[i].AdaptationWeight );
#   endif
  } /* for */
} /* UpdateAdaptationWeights */
