// -*- C++ -*-

#include <string.h>
#include <iostream.h>
#include <strstream.h>

#include <CNCL/QueueFIFO.h>
#include <CNCL/EventScheduler.h>
#include <CNCL/FiboG.h>
#include <CNCL/NegExp.h>
#include <CNCL/Uniform.h>
#include <CNCL/Moments.h>
#include <CNCL/LREF.h>

#include "Job.h"


enum { EV_JOB, EV_TIMER_G, EV_TIMER_S }; // CNEvent types for M/M/1 simulation


class Server : public CNEventHandler
{
private:
    Job *job;		// Served job
    CNQueueFIFO queue;	// CNQueue
    CNRandom &rnd_b;	// Distribution of service time b
    CNMoments t_w;	// Evaluation tau_w with moments
    CNLREF    t_w_lre;  // Evaluation tau_w with LRE algorithm
    enum { ST_WAITING, ST_SERVING };
    ready;              // flag for LRE evaluation
    ofstream lre_outfile;
    
public:
    virtual void event_handler(const CNEvent *ev);

    void print_results();
    void eval_job(Job *job);

    
    Server(CNRandom &rnd, char* fn)
	: rnd_b(rnd), job(NIL), t_w("tau_w"),
	  t_w_lre(0.01,0.99,0.01,100,CNLRE::LIN,0,"LRE-evaluation"),
	  lre_outfile(fn)
    {
	state(ST_WAITING);
	ready = FALSE;
    }
};


class Generator : public CNEventHandler
{
private:
    CNRandom &rnd_a;	// Distribution of arrival time a
    Server *server;	// Connected queue/server
    CNMoments t_a;
    double value;
 
public:
    virtual void event_handler(const CNEvent *ev);

    void print_results();

    Generator(CNRandom &rnd, Server *serv) : rnd_a(rnd), server(serv),
	      t_a("tau_a") {}
};


void Generator::event_handler(const CNEvent *ev)
{
    // Incoming event -> generate new Job
    send_now(new CNEvent(EV_JOB, server, new Job));
    // CNRandom delay
    value = rnd_a();
    send_delay(new CNEvent(EV_TIMER_G), value);
    t_a.put(value);
}


void Generator::print_results()
{
    cout << t_a;
}


void Server::event_handler(const CNEvent *ev)
{

    switch(state())
    {
    case ST_SERVING:
	switch(ev->type())
	{
	case EV_JOB:
	    // Incoming job, put into queue
	    {
		Job *job;
		job = (Job *)ev->object();
		job->in = now();
		queue.put(job);
	    }
	    break;
	    
	case EV_TIMER_S:
	    // Timer event, service time run down
	    job->out = now();
	    // Evaluate job
	    eval_job(job);
	    delete job;
	    job = NIL;
            if (ready == TRUE) scheduler()->stop();
	    // Get new job from queue
	    if(!queue.empty())
	    {
		job = (Job *)queue.get();
		job->start = now();
		// CNRandom service time
		send_delay(new CNEvent(EV_TIMER_S), rnd_b());
		state(ST_SERVING);
	    }
	    else
	    {
		state(ST_WAITING);
	    }
	    break;
	    
	default:
	    error("mm1: ", "illegal event in state ST_SERVING");
	    break;
	}

	break;
	
    case ST_WAITING:
	switch(ev->type())
	{
	case EV_JOB:
	    // Incoming job
	    job = (Job *)ev->object();
	    job->in    = now();
	    job->start = now();
	    // CNRandom service time
	    send_delay(new CNEvent(EV_TIMER_S), rnd_b());
	    state(ST_SERVING);
	    break;

	default:
	    error("mm1: ", "illegal event in state ST_WAITING");
	    break;
	}
	
	break;
	
    default:
	error("mm1: ", "illegal state");
	break;
	
    }
}



void Server::eval_job(Job *job)
{

    t_w.put(job->start - job->in);
    t_w_lre.put(job->start - job->in);

    if (t_w_lre.status() == CNStatistics::END) ready = TRUE;
}


void Server::print_results()
{
    cout << t_w;
    lre_outfile << t_w_lre;
}

 

main()
{

   CNRNG   *rng  = new CNFiboG;
   char file_name[20] = "";
   ostrstream strm(file_name, 20);
   memset(file_name, 0, 20);

// *************************************************************************
// Parameters of distribution functions for G/G/1 - FIFO simulation
// Set as required for the excercises

    double lambda = 0.9;
    double mu = 1.0;
    double rho = lambda/mu;

// Problem #2  M/M/1 simulation and LRE evaluation

    CNNegExp rnd_a(1.0/lambda, rng);
    CNNegExp rnd_b(1.0/mu, rng);
    double c_B2 = 1.0; // due to neg.exp. distributed service times 
    strm << "mm1" << "." << lambda << "." << mu << ".lre";

// Problem #3  G/G/1 simulation

//    CNNegExp  rnd_a(1.0/lambda, rng);
//    CNUniform rnd_b(0.0,2.0, rng); // mu = 1.0 
//    double c_B2 = 1.0/3.0; // due to uniform distributed service times 
//    strm << "mg1.lre";

//    CNUniform rnd_a(0.0,4.0, rng); // lambda = 0.5 
//    CNNegExp  rnd_b(1.0/mu, rng);
//    double c_B2 = 1.0; // due to neg.exp. distributed service times 
//    strm << "gm1.lre";

//    CNUniform rnd_a(0.0,4.0, rng); // lambda = 0.5 
//    CNUniform rnd_b(0.0,2.0, rng); // mu = 1.0 
//    double c_B2 = 1.0/3.0; // due to uniform distributed service times 
//    strm << "gg1.lre";

    double T_w = rho*(1 + c_B2)/(2*(1 - rho));
    double T_a = 1.0/lambda;

// *************************************************************************

    Server         server(rnd_b,file_name);
    Generator      generator(rnd_a, &server);
    CNEventScheduler scheduler;
    
    scheduler.start(new CNEvent(EV_TIMER_G, &generator, 0));

    generator.print_results(); 
    server.print_results();
    
    cout << endl;
    cout << "You will find the results of the LRE evaluation in file " << file_name << endl;

}
