// -*- C++ -*-

#include <string.h>
#include <iostream.h>

#include <CNCL/QueueFIFO.h>
#include <CNCL/EventScheduler.h>
#include <CNCL/FiboG.h>
#include <CNCL/NegExp.h>
#include <CNCL/Uniform.h>
#include <CNCL/Moments.h>
#include <CNCL/LREF.h>

#include <CNCL/EZD.h>
#include <CNCL/EZDQueue.h>
#include <CNCL/EZDServer.h>
#include <CNCL/EZDWindow.h>
#include <CNCL/EZDPushButton.h>
#include <CNCL/EZDText.h>

#include "Job.h"


enum { NJOBS=100000 };

enum { EV_JOB, EV_TIMER_G, EV_TIMER_S }; // CNEvent types for M/M/1 simulation


class Server : public CNEventHandler
{
private:
    Job *job;		// Served job
    CNQueueFIFO queue;	// CNQueue
    CNRandom &rnd_b;	// Distribution of service time b
    CNMoments t_w;	// Evaluation tau_w
    enum { ST_WAITING, ST_SERVING };
    long n;             // Counter for jobs
    EZDQueue  ezdq;
    EZDServer ezds;
    EZDText   ezdl;
    EZDText   ezdmwm;
    EZDText   ezdvw;
    EZDText   ezdsim;
    EZDText   ezdjob;
    step_flag;
    
public:
    virtual void event_handler(const CNEvent *ev);

    void print_results();
    void eval_job(Job *job);

    void ezd_input();
    
    Server(CNRandom &rnd)
	: rnd_b(rnd), job(NIL), t_w("tau_w"),
	  ezdq(100, 170, 200, 30, 30), ezds(-1, 170, 30),
	  ezdl("l", 70, 250, "", "-adobe-courier-*-r-normal-*-12-*-*-*-*-*-*-*"),
	  ezdsim("Sim", 50, 70, ""), ezdjob("Jobs", 50, 90, ""),
	  n(0), ezdmwm("mwm", 270, 300, ""), ezdvw("vw", 270, 320, "")
    {
	step_flag = TRUE;
	state(ST_WAITING);
	ezds.x( ezdq.right() );
	ezdq.redraw();
	ezdq.length(0);
	ezdq.color("DarkOrchid1");
	ezds.color("Green");
	ezdl.set_text_val ("Queue length :", 0);
	ezdmwm.set_text_val("T_w mean (sim)        : ", 0);
	ezdvw.set_text_val("T_w coeff. of var.    : ", 0);
	ezdjob.set_text_val("# of jobs : ", n);
    }
};



void Server::ezd_input()
{
    if(step_flag || EZD::test_event())
    {
	CNString in;
	in = EZD::event();
	if(strstr(in, " EXIT "))
	    scheduler()->stop();
	if(strstr(in, " STEP "))
	    step_flag = TRUE;
	if(strstr(in, " RUN "))
	    step_flag = FALSE;
    }
}


class Generator : public CNEventHandler
{
private:
    CNRandom &rnd_a;	// Distribution of arrival time a
    Server *server;	// Connected queue/server
    CNMoments t_a;
    double value;
    EZDText ezdmam;
 
public:
    virtual void event_handler(const CNEvent *ev);

    void print_results();

    Generator(CNRandom &rnd, Server *serv) : rnd_a(rnd), server(serv),
          t_a("tau_a"), ezdmam("mam", 70, 300, "")
    {
	ezdmam.set_text_val("T_a mean (sim) : ", 0);
    }
};


void Generator::event_handler(const CNEvent *ev)
{
    // Incoming event -> generate new Job
    send_now(new CNEvent(EV_JOB, server, new Job));
    // CNRandom delay
    value = rnd_a();
    send_delay(new CNEvent(EV_TIMER_G), value);
    t_a.put(value);
    ezdmam.set_text_val("T_a mean (sim) : ", t_a.mean());
}


void Generator::print_results()
{
    cout << t_a;
}


void Server::event_handler(const CNEvent *ev)
{

    ezdsim.set_text_val("SimTime   : ", scheduler()->time());

    ezd_input();
    
    switch(state())
    {
    case ST_SERVING:
	switch(ev->type())
	{
	case EV_JOB:
	    // Incoming job, put into queue
	    {
		Job *job;
		job = (Job *)ev->object();
		job->in = now();
		queue.put(job);
		ezdq.length( queue.length() * 5 );
		ezdl.set_text_val("Queue length :", queue.length() );
	    }
	    break;
	    
	case EV_TIMER_S:
	    // Timer event, service time run down
	    job->out = now();
	    // Evaluate job
	    eval_job(job);
	    delete job;
	    job = NIL;

	    n++;
	    ezdjob.set_text_val("# of jobs : ", n);

	    // Get new job from queue
	    if(!queue.empty())
	    {
		job = (Job *)queue.get();
		ezdq.length( queue.length() * 5 );
		ezdl.set_text_val("Queue length :", queue.length() );
		job->start = now();
		// CNRandom service time
		send_delay(new CNEvent(EV_TIMER_S), rnd_b());
		ezds.color("Red");
		state(ST_SERVING);
	    }
	    else
	    {
		ezds.color("Green");
		state(ST_WAITING);
	    }
	    break;
	    
	default:
	    error("mm1: ", "illegal event in state ST_SERVING");
	    break;
	}

	break;
	
    case ST_WAITING:
	switch(ev->type())
	{
	case EV_JOB:
	    // Incoming job
	    job = (Job *)ev->object();
	    job->in    = now();
	    job->start = now();
	    // CNRandom service time
	    send_delay(new CNEvent(EV_TIMER_S), rnd_b());
	    ezds.color("Red");
	    state(ST_SERVING);
	    break;

	default:
	    error("mm1: ", "illegal event in state ST_WAITING");
	    break;
	}
	
	break;
	
    default:
	error("mm1: ", "illegal state");
	break;
	
    }
}



void Server::eval_job(Job *job)
{
// Moments evaluation

    t_w.put(job->start - job->in);

    ezdmwm.set_text_val("T_w mean (sim)        : ", t_w.mean());
    ezdvw.set_text_val("T_w coeff. of var.    : ", t_w.relative_deviation());
}


void Server::print_results()
{
    cout << t_w;
}

 

main()
{

   CNRNG   *rng  = new CNFiboG;

// *************************************************************************
// Parameters of distribution functions for G/G/1 - FIFO simulation
// Set as required for the excercises

    double lambda = 0.9;
    double mu = 1.0;
    double rho = lambda/mu;

// Problem #1  M/M/1 simulation

    CNNegExp rnd_a(1.0/lambda, rng);
    CNNegExp rnd_b(1.0/mu, rng);
    double c_B2 = 1; // due to neg.exp. distributed service times 


// Problem #3  G/G/1 simulation

//    CNNegExp  rnd_a(1.0/lambda, rng);
//    CNUniform rnd_b(0.0,2.0, rng); // mu = 1.0 
//    double c_B2 = 1.0/3.0; // due to uniform distributed service times 

//    CNUniform rnd_a(0.0,4.0, rng); // lambda = 0.5 
//    CNNegExp  rnd_b(1.0/mu, rng);
//    double c_B2 = 1.0; // due to neg.exp. distributed service times 

//    CNUniform rnd_a(0.0,4.0, rng); // lambda = 0.5 
//    CNUniform rnd_b(0.0,2.0, rng); // mu = 1.0 
//    double c_B2 = 1.0/3.0; // due to uniform distributed service times 

    double T_w = rho*(1 + c_B2)/(2*(1 - rho));
    double T_a = 1.0/lambda;

// *************************************************************************

    EZDWindow win("G/G/1",500, 400);
    EZDDrawing draw;
    win.overlay(&draw);

    EZD::draw_text(0, 0, 500, 50, "center center", "G/G/1 Simulation",
		   "", "-*-times-bold-r-*-*-40-*-*-*-*-*-*-*");
    EZD::draw_text(0, 70, 320, 20, "right center", "Press button:", "", "");
    
    EZDPushButton step("STEP", 329, 70, 50, 20, "Step");
    EZDPushButton run ("RUN",  389, 70, 50, 20, "Run");
    EZDPushButton exit("EXIT", 449, 70, 50, 20, "Exit");
    EZDText ezdmwt("mwt", 270, 280, "");
    EZDText ezdmat("mat", 70, 280, "");
    EZDText ezdload("load", 270, 250, "",
		    "-adobe-courier-*-r-normal-*-12-*-*-*-*-*-*-*");
    ezdmwt.set_text_val("T_w mean M/G/1 (theo) : ", T_w);
    ezdmat.set_text_val("T_a mean (theo): ", T_a);
    ezdload.set_text_val("Load:  ", rho);
    
    Server         server(rnd_b);
    Generator      generator(rnd_a, &server);
    CNEventScheduler scheduler;
    
    scheduler.start(new CNEvent(EV_TIMER_G, &generator, 0));

    generator.print_results(); 
    server.print_results();
    
}
