/* file: UTIL.PL {writel etc., tracing, history, describe, strategy, show} */
/*                          *************
                               M I K E
                            *************
               Micro Interpreter for Knowledge Engineering
                  {written in Edinburgh-syntax Prolog}

MIKE: Copyright (c) 1989, 1990 The Open University (U.K.)

MIKE is intended for educational purposes, and may not
be sold as or incorporated in a commercial product without
written permission from: The Copyrights Officer, Open University,
Milton Keynes MK7 6AA, U.K.

The Open University accepts no responsibility for any legal or other
consequences which may arise directly or indirectly as a result of the
use of all or parts of the contents of this program.

This software accompanies Open University Study Pack PD624, 'KNOWLEDGE
ENGINEERING'.  Complete sets of study pack materials may be obtained from:

                      Learning Materials Sales Office
                      The Open University
                      P.O. Box 188
                      Milton Keynes MK7 6DH, U.K.

                      Tel: [+44] (908) 653338
                      Fax: [+44] (908) 653744
*/
/* Utilities for MIKE */
/* mainly tracing and formatting information */

/* If you are porting MIKE to another PROLOG then pay particular attention
   to this file.  If the PROLOG to which you are porting has the built in
   primitive 'append' then you may well have to comment the definition of
   it in this file.  It is assumed that the PROLOG you are using has a
   built-in definition of 'abolish'.  If you are porting to LPA MacProlog
   then also remove the definition of 'kill' below.  */
/* This file is divided into three main sections:
     1.  simple utilities, such as 'pd624 member'
     2.  tracing facilities
     3.  the utilities 'describe', 'strategy', and 'show'
*/
/* =================== (1) SIMPLE UTILITIES ============================= */

kill(_). /* for compatibility with LPA MacPROLOG, dummy definition needed */

/* ---  'pd624 member'(X, [X|_]). is to avoid any name clashes with existing
         definitions of member */

'pd624 member'(X,[X|_]).
'pd624 member'(X, [_|Xs]) :-
	'pd624 member'(X, Xs).

do_just_once(X) :-
	call(X),
	!.			

append([],L,L).
append([H|L1],L2,[H|L3]):-
	append(L1,L2,L3).

/* definition of union and intersection (commented out), just in case...

union([],Ys,Ys).
union([X|Xs], Ys, Zs) :-
   member(X,Ys),
   !,
   union(Xs,Ys,Zs).
union([X|Xs], Ys, [X|Zs]) :-
   union(Xs,Ys,Zs).

intersection([],Ys,[]).
intersection([X|Xs], Ys, [X|Zs]) :-
   member(X,Ys),
   !,
   intersection(Xs,Ys,Zs).
intersection([X|Xs],Ys,Zs) :-
   intersection(Xs,Ys,Zs).

*/


'pd624 subset'([],X):- nonvar(X).
'pd624 subset'([H|T],Target):-
	'pd624 member'(H,Target),
	'pd624 subset'(T,Target).

'pd624 & member'(A,A).
'pd624 & member'(A,A & _).
'pd624 & member'(A,_ & B):-
	'pd624 & member'(A,B).

'pd624 list length'([],0).
'pd624 list length'([A|List],Length):-
   'pd624 list length'(List,Length1),
   Length is Length1 + 1.

/* 'pd624 length with disjunct check' sees if there is a disjunct in the
   pattern.  If there is it will take the first disjunct (in left to right
   sequence) and it will compute the specificity in terms of the specificity
   of either side of the disjunct and then choose the highest.  All other
   disjuncts will be ignored */
'pd624 length with disjunct check'(A or B,L):-
       'pd624 length'(A,L1),
       'pd624 length'(B,L2),
       (L1 >= L2,L1 = L;L2 = L),!.
'pd624 length with disjunct check'(A,B):-
       'pd624 length'(A,B).

'pd624 length'(A or B,N):-
   'pd624 length'(A,N1),
   'pd624 length'(B,N2),
   N is N1 + N2.
'pd624 length'(_ & T,N):-
    'pd624 length'(T,N1),
    N is 1 + N1.
'pd624 length'(A,1).

/* A tailor-made 'quicksort' for triples of the form (S,I,T),
   (these are the three arguments of enabled, used elsewhere in
   this file).
   S (Switch) is an integer, and triples need to be sorted
   into ascending numerical order.  */
'pd624 sort'([],[]).
'pd624 sort'([(S,I,T)|SITs],Sorted) :-
       'pd624 split'(SITs,S,Los,His),
       'pd624 sort'(Los, SortedLos),
       'pd624 sort'(His, SortedHis),
       append(SortedLos, [(S,I,T)|SortedHis], Sorted).

'pd624 split'([],_,[],[]).
'pd624 split'([(S,I,T)|SITs],Crit,[(S,I,T)|Los],His) :-
       S < Crit,
       'pd624 split'(SITs,Crit,Los,His).
'pd624 split'([(S,I,T)|SITs],Crit,Los,[(S,I,T)|His]) :-
       S >= Crit,
       'pd624 split'(SITs,Crit,Los,His).

/* ---- writel and other output routines -------------- */
writel([]).
writel([(rule Name forward if Ifs then Thens)|Rest]):-
    write('rule '),write(Name),write(' forward '),nl,tab(6), write(' if '),
    nl,tab11_write(Ifs),nl,tab(6),write(' then '),nl,
    tab11_write(Thens),write('. '),nl,
    writel(Rest),!.
writel([(rule Name backward if Ifs then Thens)|Rest]):-
    write('rule '),write(Name),write(' backward '),nl,tab(6), write(' if '),
    nl,tab11_write(Ifs),nl,tab(6),write(' then '),nl,
    tab11_write(Thens),write('. '),nl,
    writel(Rest),!.
writel([nl|R]):-
    nl, writel(R).
writel([t/Tab|Rest]):-
    tab(Tab),
    writel(Rest).
writel([&|Rest]):-
	  write(' & '),
		writel(Rest).
writel(A:[H|[]]):-
 tab(6),write(A),write(' : '),write(' ['),
 write(H),write(']').
writel(A:[H|T]):-
	tab(6),write(A),write(' : '),write(' ['),
 write(H),write(','),nl,
	write1(T), write(']').
writel(A:B):-
	tab(6),write(A),write(' : '),write(B).
writel([H|T]):-
    write(H),nl,
    writel(T).
writel((A,B)):- /* conjunct, but for MIKE this means 'with' Body */
	writel(A), write(','),nl,
	writel(B).
writel(A):-
	tab(12),write(A),nl.

write1([X]) :-
	tab(20),write(X).
write1([]).
write1([X|[]]):-
 tab(20),write(X).
write1([H|T]):-
	tab(20),write(H),write(','),
	nl,write1(T).

conj_write((A&B)) :- write(A),write(' & '),conj_write(B), !.
conj_write(X) :- write(X).

tab11_write((A or B)):-
    !,
    tab11_write(A),     nl,
    tab(6),write('or '), nl,
    tab11_write(B).
tab11_write((H & T)):-
    !,
    tab(11),write(H),write(' & '),nl,tab11_write(T).
tab11_write(H):-
    tab(11),write(H).

'pd624 write'([]).
'pd624 write'([nl|B]):-!, nl,'pd624 write'(B).
'pd624 write'([tab(A)|B]):- !,tab(A),'pd624 write'(B).
'pd624 write'([t/A|B]):- !,tab(A),'pd624 write'(B).
'pd624 write'([A|B]):-!, write(A),'pd624 write'(B).
'pd624 write'([nl]):-nl.
'pd624 write'([tab(L)]):-tab(L).
'pd624 write'([t/L]):-tab(L).
'pd624 write'([A]):- write(A).

'pd624 pretty list'([]).
'pd624 pretty list'([nl|B]):- !,nl,'pd624 pretty list'(B).
'pd624 pretty list'([tab(A)|B]):- !,tab(A),'pd624 pretty list'(B).
'pd624 pretty list'([t/A|B]):- !,tab(A),'pd624 pretty list'(B).
'pd624 pretty list'([A|B]):- !,tab(1),write(A),'pd624 pretty list'(B).
'pd624 pretty list'([nl]):-nl.
'pd624 pretty list'([tab(L)]):-tab(L).
'pd624 pretty list'([t/L]):-tab(L).
'pd624 pretty list'([A]):- tab(1),write(A).

/* ========================== (2)  T R A C I N G ======================= */
/* tracing is normally called with no arguments, in which case it prompts
   the user with a menu of choices.  With a single integer argument,
   just that option is 'toggled' (i.e. turned from off to on, or on to off).
   Optional syntax is: tracing([N1,N2,N3,...]), where N1 etc. are integers
   from 1 to 10 specifying the number of the tracing option you wish
   to 'toggle' */

tracing([]).
tracing([X|Xs]) :-  /* list of integers expected */
  tracing(X),
  tracing(Xs).
tracing(X):- change_options(X),   /* single integer expected */
 enabled(I,T,X),
 write_options([(X,I,T)]).

tracing:-            /* this is the normal usage */
   display_tracing_options,
	  !,
   'pd624 write'(['Type the numbers of the option you wish to change',nl,
      ' eg. 1,2,3,4. and then a FULL STOP',nl,
      'Or quit. to exit without altering the settings',nl]),
    write('==> '),
    read(Input),
    change_options(Input),
    display_tracing_options,
    'pd624 write'(['Type',nl,'  ?- show symbols.',nl,
                  'for a reminder of what the tracing symbols mean.',nl]).


display_tracing_options :-
    findall((S,I,T),(enabled(I,T,S), \+ 'pd624 member'(S,[11,12,13,14,15]))
               ,Newoptions_unsorted),
    'pd624 sort'(Newoptions_unsorted,Newoptions),
    write_options(Newoptions),nl.

change_options((A,B)):- /* conjunction of options? deal with head then tail */
	 !,
  change_options(A),
		change_options(B).
change_options(9):-   /* options 13,14,15 are 'yoked' with number 9 */
	enabled(A,' disabled ',9),
	reverse_option(A,enable),
  change_options((13,14,15)), !.
change_options(9):-
 enabled(A,' enabled ',9),
	reverse_option(A,disable),
 change_options((13,14,15)), !.
change_options(A):-           /* normal case: singleton option */
	enabled(Name,' disabled ',A),
	reverse_option(Name,enable),
 !.
change_options(A):-
	enabled(Name,' enabled ',A),
	reverse_option(Name,disable),
 !.
change_options(quit):- !. /* normal way to bail out of tracing options... */
change_options(q):- !.    /* but we also allow 'q', 'exit', 'e', 'ok', 'halt' */
change_options(exit):- !. /* as un-documented alternatives */
change_options(e):- !.
change_options(ok):- !.
change_options(halt):- !.
change_options(A) :-
    writel([A,' is an illegal option',nl,'legal options are numbers that',
                'appear in the tracing menu. ']),!.


reverse_option(Name,disable):-
    retract(enabled(Name,_,A)),
    assert(enabled(Name,' disabled ',A)),
    !.
reverse_option(Name,enable):-
    retract(enabled(Name,_,A)),
    assert(enabled(Name,' enabled ',A)),
    !.
reverse_option(A,_):-
    writel([A,' is an illegal option',nl,'legal options are numbers that',
                'appear in the tracing menu. ']),!.

turn_off_option(A):-  /* like change, but unconditionally turns off or
                         else leaves it alone if it was already off */
	enabled(Name,' enabled ',A),
	reverse_option(Name,disable).
turn_off_option(_).

write_options([]).
write_options([(Index,Item,S)|T]):-
    write_plus_or_minus(S,Index),write(Index),write(': '),write(Item),
    write(' is currently'),write(S),write('. '),nl,
    write_options(T).
write_options([A|B]):-
	write('ERROR: from write options'),
	write(A),nl,
	write_options(B).

write_plus_or_minus(' disabled ', N) :-
   write('-'), (N < 10, write(' ') ; true), !.

write_plus_or_minus(_, N) :-
   write('+'), (N < 10, write(' '); true), !.

when_enabled(P for List):-
    enabled(P,' enabled ',_ignore_the_index), /* flag enabled? */
    !,
    P for List.    /* then call P (e.g. 'show outcome for backward chain..') */
                   /* if P can be re-satisfied on backtracking, this is fine */
when_enabled(P for List) :-
    enabled(P,' disabled ',_), !. /* i.e. do nothing if flag disabled */
when_enabled(X) :-
   writel(['Warning: when_enabled/1 has been passed an unexpected argument:',
          X,nl,'  Only the authorised tracing flags are allowed!',
   'These are the ones displayed when you type:  ?- tracing.',
   '[Succeeding anyway, which may cause extra solutions to be found!]']).

/* enabled/3 is just a database of flags, using the following 3 arguments:
   1  The name of the option
   2  Its current state, either ' enabled ' or ' disabled '
   3  An integer indicating its position in the tracing multiple-choice menu.
*/
enabled('show conflict set',' disabled ',1).
enabled('show refractoriness',' disabled ',2).
enabled('show specificity',' disabled ',3).
enabled('show recency',' disabled ',4).
enabled('show new working memory elements or frame changes',' disabled ',5).
enabled('show chosen rule',' disabled ',6).
enabled('show backward chaining',' disabled ',7).
enabled('show outcome of backward chaining',' disabled ',8).
enabled('show single stepping',' disabled ',9).
enabled('show history on request',' enabled ',10).

/* the next five options are set internally at run-time, and are not
   meant to be settable by the user!! */

enabled('show individual LHS in', ' disabled ',11).
enabled('show individual LHS out', ' disabled ',12).
enabled('show single stepping in', ' disabled ',13).
enabled('show single stepping out', ' disabled ',14).
enabled('show conflict winner', ' disabled ',15).

'show outcome of backward chaining' for P/Depth:-  /* success */
      enabled('show backward chaining',' enabled ',7),
     	write('<- '),tab(Depth),write('+ '),conj_write(P),nl.
'show outcome of backward chaining' for P/Depth:-  /* only come here on retry */
	     write('<- '),tab(Depth), write('^ '),conj_write(P),nl,
      !,
      fail.  /* because we need to propagate failure back to older sibling */
'show backward chaining' for P/Depth:-  /* goal invocation */
  do_just_once((Depth = 0, nl ; true)), /* extra newline only for first call */
  write('<- '),tab(Depth),write('? '),conj_write(P),nl.
'show conflict set' for P:-
    nl,write('Conflict Set is: '),nl,
    writel(P).
'show refractoriness' for P :-
    nl,write('Refractoriness filter threw out the following rule: '),nl,
    writel(P).
'show specificity' for P:-
    nl,write('Conflict set AFTER specificity filter is: '), nl,
    writel(P).
'show new working memory elements or frame changes' for P :-
    nl,write('New working memory elements or frame changes are: '), nl,
    writel(P).
'show recency' for P:-
    nl,write('Conflict set AFTER recency filter is: '), nl,
    writel(P).
'show chosen rule' for P:-
    nl,write('Chosen rule is: '), nl,
    writel(P).

'show individual LHS in' for X :-
              write('-LHS-> '),write('? '),write(X),
              pd624_read_loop,
              !.

'show individual LHS out' for X :-
              write('-LHS-> '),write('+ '), write(X),nl,
              !.

'show single stepping in' for X :-
              turn_off_option(11),   /* kill creeping in, regardless */
              turn_off_option(12),  /* kill creeping out, regardless */
              nl,write('-> ? '),write(X),ulnl,
              pd624_read_loop,
              !.      /* because embedded within a findall */

'show single stepping out' for X :-
              write('   +'),   /* many instantiations may win here! */
              !.

'show conflict winner' for X:-
              nl,write('-> * '),write(X),write(' ************'),
              ulnl,
              pd624_read_loop,
              !.

'show history on request' for [RuleName,Symbol] :-
      'pd624 current cycle is'(CycleNum),
       assertz('pd624 fc_history'(RuleName,CycleNum,Symbol)),
       !.

/* ------------ Forward chaining history display ------------------------ */
history(_,_) :-
  enabled('show history on request',' disabled ',_),
  !,
  writel(['To see the history of execution in tabular form, you must FIRST',
          'ensure that the tracing option called ''show history on request'' has been enabled',
          '(this is necessary because the history has to be stored during execution).',
          'To enable the relevant option, you can either type',
          '    ?- tracing(10).',
          'or else',
          '    ?- tracing.',
          'and then respond appropriately to the menu of choices.',nl,
          'When you have set your options correctly, you can then reinvoke',
          '    ?- fc.',
          'and then type',
          '    ?- show history.',
          'when execution has completed.']).

history(beginning, end) :-  /* default case passed in from ?- show history. */
   'pd624 current cycle is'(SoFar),
   SoFar > 45,
   'pd624 write'([
     'The highest cycle number reached on the preceding run was: ',SoFar,nl,
 'You can display the history for any contiguous group of up to 45 cycles',nl,
 'by entering a modified version of the show history command.',nl,
 'For example, to see cycles 30 to 55, say, you would type in', nl,
 '  ?- show history/30-55.',nl]).

history(beginning, end) :-   /* constants are deliberate 'flags' */
   'pd624 current cycle is'(SoFar),
   SoFar < 46,
   SoFar > 0,
   history(1,SoFar),  /* this invokes the 'legitimate' output below */
   !.

history(beginning, end) :-   /* constants are deliberate 'flags' */
   'pd624 current cycle is'(0),
   nl,
   write('No history to show you yet!  Try: ?- fc.'),
   nl,
   !.

history(Lo,Hi) :-
   integer(Lo),
   integer(Hi),
   Diff is Hi - Lo,
   Diff > 44,
   'pd624 current cycle is'(SoFar),
   'pd624 write'(['Sorry, you can only display 45 cycles at a time.',nl,
 'Try   ?- show history.  to see the first 45 cycles,',nl,
 'or    ?- show history/40-85. to see cycles 40 to 85, etc.',nl,
 'The highest cycle number reached on the preceding run was: ',SoFar,nl]),
   !,
   fail.

history(Lo, Hi) :-
   integer(Lo),
   integer(Hi),
   !,
   'pd624 write'([nl,'  RULE NAME                           CYCLE NUMBER(',
                 Lo,'-',Hi,')',nl]),
   'pd624 generate listofnums'(Lo,Hi,WholeList),
   tab(30),posh_dots(WholeList),nl,
   findall(Name,((rule Name forward if X then Y)),AllNames),
   'pd624 fc_history display'(AllNames,WholeList).

history(A,B) :-
   'pd624 write'(['Sorry, only integer values are allowed, e.g.', nl,
                  '  ?- show history/40-85. ', nl]),
   !,
   fail.

posh_dots([]).
posh_dots([N|Ns]) :-
   posh_symbol(N,Sym),
   write(Sym),
   posh_dots(Ns).

posh_symbol(N, Int) :-
  0 is N mod 10,  /* multiple of 10?  then use integer from 1 to 9  */
  !,
  posh_truncate(N, Int).

posh_symbol(N, ':') :-  /* use : for multiples of 5, e.g.  ....:....1....: */
  0 is N mod 5,
  !.

posh_symbol(N, '.').  /* default case... just use a dot */

posh_truncate(N, Int) :-
    Temp is N//10,
    Int is Temp mod 10.

'pd624 fc_history display'([Rule|Rules],NumList) :-
    'pd624 show name'(Rule),
    'pd624 fc_history gimme one line'(Rule, NumList),
    'pd624 fc_history display'(Rules,NumList).

'pd624 fc_history display'([],_) :- nl.  /* termination */

'pd624 show name'(RuleName) :-
       'pd624 string length'(RuleName,Len),
       'pd624 maybe truncate name'(RuleName,Len).

'pd624 maybe truncate name'(Name,Len) :-
       Len > 30,                /* very long name? */
       write(Name),nl,tab(30).  /* then insert <CR>, tab across */

'pd624 maybe truncate name'(Name,Len) :-
       Len =< 30,              /* name length < 30 chars?  OK... */
       Remainder is 30 - Len,  /* cause fc_history stuff starts at column 30 */
       write(Name), tab(Remainder). /* write it out, tab the rest */

'pd624 string length'(Atom, Length) :-
       integer(Atom),       /* can only happen if you use rule name like 1 */
       !,
       'pd624 power_of_10'(Atom,Power),  /* e.g. 124 is 3 (3 digits) */
       Length is Power + 1 .

/* COMPATIBILITY NOTE: THE CLAUSE WHICH FOLLOWS IS SPECIFIC TO THE
   VERSION OF PROLOG SUPPLIED WITH PD624.  AN ALTERNATIVE DEFINITION
   IS GIVEN A FEW LINES FURTHER ON. */

  'pd624 string length'(Atom,Length):-
        name(Atom,String),
        list(List,String),
        'pd624 list length'(List,Length).

/* COMPATIBILITY NOTE ... THE FOLLOWING CODE CAN BE USED WITH
   MOST EDINBURGH SYNTAX PROLOG DIALECTS, TO REPLACE THE IMMEDIATELY
   PRECEDING CLAUSE. REARRANGE COMMENT BRACKETS ACCORDINGLY. */

/*
'pd624 string length'(Atom, Length) :-
       name(Atom, List),
       'pd624 list length'(List, Length).
  */

'pd624 power_of_10'(X,0) :- X < 10, !.
'pd624 power_of_10'(X,1) :- X < 100, !.
'pd624 power_of_10'(X,2) :- X < 1000, !.
'pd624 power_of_10'(X,3) :- X < 10000, !.
'pd624 power_of_10'(X,4) :- X < 100000, !.
'pd624 power_of_10'(X,5). /* This means that a rule named 123456789 will
                             have a fixed string length of 6.  This
                             will cause ?- show history. to print a
                             slightly messed up chart.  The solution
                             is to use numbers < 999999 for rule names! */


'pd624 fc_history gimme one line'(Rule,[]) :- nl.
'pd624 fc_history gimme one line'(Rule,[Num|Rest]) :-
        /* if you have a symbol stored, write it out, else write ' ' */
        'pd624 get best symbol'(Rule,Num,Sym),
        write(Sym),
        'pd624 fc_history gimme one line'(Rule,Rest).

'pd624 get best symbol'(Rule,Num,'*') :-   /* strict priority sequence */
        'pd624 fc_history'(Rule,Num,'*'),
        !.

'pd624 get best symbol'(Rule,Num,'+') :-
       'pd624 fc_history'(Rule,Num,'+'),
       !.

'pd624 get best symbol'(Rule,Num,' '). /* no symbol, use blank */

'pd624 generate listofnums'(X,Hi,[0]) :-
       X > Hi,
       write('Sorry, can only work with an ascending sequence of integers.'),
       nl,
       !,
       fail.

'pd624 generate listofnums'(Hi,Hi,[Hi]) :- !.
'pd624 generate listofnums'(Lo,Hi,[Lo|Rest]) :-
       Next is Lo + 1,
       'pd624 generate listofnums'(Next,Hi,Rest).


/* ------------ handler for user input to single-stepper -------------- */

pd624_flag(dummy).  /* for MacProlog-like dialects & POPLOG, requiring 1 */

pd624_read_loop :-
               not pd624_flag(unleashed),
               get0(Char),
               ((not Char = 13, get0(NextChar)) ; true),
               pd624_deal_with(Char), !.

pd624_read_loop :-
               pd624_flag(unleashed).

ulnl :-  /* unleashed new line */
     pd624_flag(unleashed),
     nl.

ulnl.

pd624_deal_with(97) :-  /* a for abort (does not really... ) */
                   change_options(9),
                   add halt. /* i.e. switch off stepper */
pd624_deal_with(110) :- /* n for no-tracing */
                   change_options(9). /* i.e. switch off stepper */
pd624_deal_with(13).   /* <CR> */
pd624_deal_with(98) :-  /* b for break */
                   repeat,
                   nl,write('MIKE ?- '),
                   read(INPUT),
                   (INPUT = quit;
                    (do_just_once(call(INPUT)),
                     do_just_once((write(INPUT);write(no))),
                     fail) ).

pd624_deal_with(99) :- /* c for creep through left hand side conditions */
         change_options(11),
         change_options(12). /* toggles LHS creeping */

pd624_deal_with(117) :-  /* u for unleash */
                    assert(pd624_flag(unleashed)).


pd624_deal_with(HelpChar) :-
                    (HelpChar = 63 ; HelpChar = 104),
                    nl,nl,
                    write('a(bort at end of the current interpreter cycle)'),nl,
                    write('b(reak until quit)'),nl,
                    write('c(reep through left-hand-side conditions)'),nl,
                    write('h(elp)'),nl,
                    write('n(o more tracing)'),nl,
                    write('u(nleash)'),nl,
                    write('<CR> = step'),nl,
                    pd624_read_loop.

pd624_deal_with(_).



/* ======= (3)  D E S C R I B E ,  S T R A T E G Y , and  S H O W ====== */
describe A:-
	(A  instance_of Object with Body),
	write(A instance_of Object),
	write(' with '),
	nl,writel(Body), write('.'), nl.
describe A:-
	(A  subclass_of Object with Body),
	write(A subclass_of Object),
	write(' with '),
	nl,writel(Body), write('.'), nl.
describe A:-
 (rule A forward if Ifs then Thens),
 writel([(rule A forward if Ifs then Thens)]).
describe A:-
 (rule A backward if Ifs then Thens),
 writel([(rule A backward if Ifs then Thens)]).

strategy menu:-
	current_conflict_resolution_strategy(List),
	writel(['The current conflict resolution strategy is ',List,

	       nl,'To change it, type the numbers that correspond to',
	    'the ordering that you want from the following menu',
	' e.g. if you want the ordering to be specificity, recency, refractoriness',
	'then type 3,2,1.  <remember the FULL STOP!>',
	'1 - refractoriness','2 - recency','3 - specificity']),
	write('==>'),read(P),sort_out_options(P,List1),
	retract(current_conflict_resolution_strategy(_)),
	assert(current_conflict_resolution_strategy(List1)),
	writel(['Ok, the new strategy is now ',List1,nl]).

strategy List:-
	retract(current_conflict_resolution_strategy(P)),
	assert(current_conflict_resolution_strategy(List)).
	
sort_out_options((A,B),[H|T]):-  !,
	'pd624 member'((A,H),[(1,refractoriness),(2,recency),(3,specificity)]),
	sort_out_options(B,T).

sort_out_options(A,[H]):-
	'pd624 member'((A,H),[(1,refractoriness),(2,recency),(3,specificity)]).

/* ---------------- the show facility ------------------------- */

show history/Lo-Hi :-
     history(Lo,Hi).

show history:-
     history(beginning, end). /* this provides default output (see above) */

show wm :-
     wm.
show rules:-
    'pd624 write'(['The currently loaded ruleset is ',
    nl,'the following : ',nl]),
    assert('wm counter'(0)),
    ((rule X forward if _ then _);
     (rule X backward if _ then _)),
    do_just_once(('pd624 write'([t/5,X,nl]),
                 retract('wm counter'(P)),
                 New is P + 1,
                 assert('wm counter'(New)))),
    fail.
show rules:-
    retract('wm counter'(Number)),
    'pd624 write'([nl,'A total of ',Number,
     ' rules were found.',nl]).

show frames:-
    'pd624 write'(['The currently loaded frames are ',
    nl,'the following : ',nl]),
    assert('wm counter'(0)),
    ((A subclass_of _ with _X);
     (A instance_of _ with _)),
    do_just_once(('pd624 write'([t/5,A,nl]),
                  retract('wm counter'(P)),
                  New is P + 1,
                  assert('wm counter'(New)))),
    fail.
show frames:-
    retract('wm counter'(Number)),
    'pd624 write'([nl,'A total of ',Number,
     ' frames were found.',nl]).

show symbols :-
   writel([
'SINGLE-STEP SYMBOL  |     MEANING (ASSUMES TRACING OPTION 9 IS SELECTED)',
'->                     Forward chaining taking place',
'-> ? <rule-name>       Considering this rule on forward chaining cycle',
'   +                   The above (just-considered) rule enters conflict set',
'   +   +   +   +       Four instantiations of above rule enter conflict set',
'-> * <rule-name>       This rule alone has been selected for firing',
'-LHS-> ? <pattern>     Considering this Left-Hand-Side pattern',
'-LHS-> + <pattern>     This Left-Hand-Side pattern matched successfully',
'<-                     Backward chaining taking place',
'<- ? <pattern>         Trying to deduce this pattern',
'<- + <pattern>         Pattern deduced successfully',
'<- - <pattern>         Failed to deduce this pattern',
'<- ^ <pattern>         Backtrack to find alternative proof for pattern',
'When the single-step tracer pauses you can type one of the following letters:',
' a(bort at end of the current interpreter cycle) - soon bails out',
' b(reak until quit) - invokes Prolog interpreter until you type ?- quit.',
' c(reep through left-hand-side conditions) - fine-grained -LHS-> trace',
' h(elp) - reminder of these symbols.   ?  has the same effect.',
' n(o more tracing) - suppresses extensive printout.',
' u(nleash) - no more pausing at each step, let loose extensive printout',
' <CR> = step through each rule (or LHS condition) as encountered.' ]).

