/*
 * wx_cmds.cc
 * wxWindows command processing
 *
 */

#include "wx.h"
#include "wx_hash.h"
#include "wx_types.h"
#include "wx_ipc.h"

#include <iostream.h>
#include <ctype.h>
#include <stdlib.h>

#include "wx_cmds.h"

#ifdef USE_CLIPS
#include "clips.h"
#endif

// Record what frames and dialogs are used so we can clear
// them up if necessary
wxList wxClipsFrameList;

/*
 * Mechanism for getting CLIPS callbacks to return a value
 *
 */

int clipsReturnType = clipsUNKNOWN;
char *clipsReturnString = NULL;
long clipsReturnLong = 0;
double clipsReturnDouble = 0.0;
extern "C" int clipsEchoOn;

/*
 * The argument to an externally called function is passed to this
 * function, which stores the appropriate return value and type
 * for the external program to fetch.
 */

#ifdef USE_CLIPS
void clipsReturnResult()
{
  DATA_OBJECT temp;
  RtnUnknown(1, &temp);

  switch (GetType(temp))
  {
    case INTEGER:
    {
      clipsReturnLong = DOToLong(temp);
      clipsReturnType = clipsLONG;
      break;
    }
    case STRING:
    case SYMBOL:
    {
      if (clipsReturnString)
        delete[] clipsReturnString;

      clipsReturnString = copystring(DOToString(temp));
      clipsReturnType = clipsSTRING;
      break;
    }
    case FLOAT:
    {
      clipsReturnDouble = DOToDouble(temp);
      clipsReturnType = clipsDOUBLE;
      break;
    }
    default:
    {
      clipsReturnType = clipsUNKNOWN;
    }
  }
}
#endif

/*
 * Get platform:
 *  "Windows 3.1", "XView" or "Motif"
 *
 */

char *wxcGetPlatform(void)
{
#ifdef wx_motif
  return "Motif";
#endif
#ifdef wx_xview
  return "XView";
#endif
#ifdef wx_msw
  return "Windows 3.1";
#endif
}

#ifdef USE_CLIPS
char *clipsGetPlatform()
{
  return (char *)AddSymbol(wxcGetPlatform());
}
#endif

/************************************************************************
 * CLEARING-UP FUNCTIONS
 *
 ************************************************************************/

void wxcCleanWindows(void)
{
  wxNode *node = wxClipsFrameList.First();
  while (node)
  {
    wxWindow *win = (wxWindow *)node->Data();
    win->Show(FALSE);
    delete win;
    node = wxClipsFrameList.First();
  }
}

#ifdef USE_CLIPS
void clipsCleanWindows()
{
  if (ArgCountCheck("clipsCleanWindows", EXACTLY, 0) == -1)
    return;

  wxcCleanWindows();
}
#endif

/************************************************************************
 * SPECIAL-PURPOSE CLIPS WINDOWS/WIDGETS
 *
 ************************************************************************/

class wxClipsCallbacks: public wxList
{
 public:
  wxClipsCallbacks(void):wxList(wxKEY_STRING)
    {}
  ~wxClipsCallbacks(void)
   {
#ifdef USE_CLIPS
     wxNode *node = First();
     while (node)
     {
       delete[] (char *)node->Data();
       node = node->Next();
     }
#endif
   }
};

// Dialog box
class wxClipsDialogBox: public wxDialogBox
{
 public:
  wxClipsDialogBox(wxWindow *parent, char *title,
              Bool modal = FALSE,
              int x = -1, int y = -1,
              int width = -1, int height = -1, long style = 0):
   wxDialogBox(parent, title, modal, x, y, width, height, style)
   {
     wxClipsFrameList.Append(this);
   }
                   
  ~wxClipsDialogBox()
    {
      wxClipsCallbacks *data;
      if (data = (wxClipsCallbacks *)GetClientData())
        delete data;
      wxDeleteThingInTable((long)this);
      wxClipsFrameList.DeleteObject(this);
    }
  void OnSize(int w, int h);
};

void wxClipsDialogBox::OnSize(int w, int h)
  {
    wxClipsCallbacks *callbacks = (wxClipsCallbacks *)GetClientData();
    if (callbacks)
    {
      wxNode *node = callbacks->Find("OnSize");
      if (node)
      {
        wxClipsFunction func = (wxClipsFunction)node->Data();
#ifdef USE_CLIPS
        char buf[100];
        sprintf(buf, "(%s %ld %d %d)", func, (long)this, w, h);
        clipsEchoOn = FALSE;
        RouteCommand(buf);
        clipsEchoOn = TRUE;
#else
        (* (void (*)(long, long, long)) func) ((long)this, w, h);
#endif
      }
    }
    else
      wxDialogBox::OnSize(w, h);
  }


// Panel
class wxClipsPanel: public wxPanel
{
 public:
  wxClipsPanel(wxFrame *frame,
              int x = -1, int y = -1,
              int width = -1, int height = -1, long style = 0):
   wxPanel(frame, x, y, width, height, style)
    { }
                   
  ~wxClipsPanel()
    {
      wxClipsCallbacks *data;
      if (data = (wxClipsCallbacks *)GetClientData())
        delete data;
      wxDeleteThingInTable((long)this);
    }

  void OnSize(int w, int h);
};

void wxClipsPanel::OnSize(int w, int h)
  {
    wxClipsCallbacks *callbacks = (wxClipsCallbacks *)GetClientData();
    if (callbacks)
    {
      wxNode *node = callbacks->Find("OnSize");
      if (node)
      {
        wxClipsFunction func = (wxClipsFunction)node->Data();
#ifdef USE_CLIPS
        char buf[100];
        sprintf(buf, "(%s %ld %d %d)", func, (long)this, w, h);
        clipsEchoOn = FALSE;
        RouteCommand(buf);
        clipsEchoOn = TRUE;
#else
        (* (void (*)(long, long, long)) func) ((long)this, w, h);
#endif
      }
    }
    else
      wxPanel::OnSize(w, h);
  }

// Canvas
class wxClipsCanvas: public wxCanvas
{
 public:
  wxClipsCanvas(wxFrame *frame,
              int x = -1, int y = -1,
              int width = -1, int height = -1, long style = 0):
   wxCanvas(frame, x, y, width, height, style)
   {
     wxCanvasDC *dc = GetDC();
     wxStoreThingInTable((long)dc, dc, wxTYPE_DC_CANVAS);
   }
                   
  ~wxClipsCanvas()
    {
      wxClipsCallbacks *data;
      if (data = (wxClipsCallbacks *)GetClientData())
        delete data;
      wxDeleteThingInTable((long)this);
      wxCanvasDC *dc = GetDC();
      wxDeleteThingInTable((long)dc);
    }

  void OnSize(int w, int h);

  void OnChar(wxKeyEvent& event);
  void OnEvent(wxMouseEvent& event);
  void OnPaint(void);
};

void wxClipsCanvas::OnSize(int w, int h)
  {
    wxClipsCallbacks *callbacks = (wxClipsCallbacks *)GetClientData();
    if (callbacks)
    {
      wxNode *node = callbacks->Find("OnSize");
      if (node)
      {
        wxClipsFunction func = (wxClipsFunction)node->Data();
#ifdef USE_CLIPS
        char buf[100];
        sprintf(buf, "(%s %ld %d %d)", func, (long)this, w, h);
        clipsEchoOn = FALSE;
        RouteCommand(buf);
        clipsEchoOn = TRUE;
#else
        (* (void (*)(long, long, long)) func) ((long)this, w, h);
#endif
      }
    }
//    wxCanvas::OnSize(w, h);
  }

void wxClipsCanvas::OnChar(wxKeyEvent& event)
  {
    wxClipsCallbacks *callbacks = (wxClipsCallbacks *)GetClientData();
    if (callbacks)
    {
      wxNode *node = callbacks->Find("OnChar");
      if (node)
      {
        wxStoreThingInTable((long)&event, &event, wxTYPE_KEY_EVENT);

        wxClipsFunction func = (wxClipsFunction)node->Data();
#ifdef USE_CLIPS
        char buf[100];
        sprintf(buf, "(%s %ld %ld %ld)", func, (long)this, event.KeyCode(), (long)&event);
        clipsEchoOn = FALSE;
        RouteCommand(buf);
        clipsEchoOn = TRUE;
#else
        (* (void (*)(long, long, long)) func) ((long)this, event.KeyCode(), (long)&event);
#endif
        wxDeleteThingInTable((long)&event);
      }
      else wxCanvas::OnChar(event);
    }
    else
      wxCanvas::OnChar(event);
  }


void wxClipsCanvas::OnEvent(wxMouseEvent& event)
  {
    wxClipsCallbacks *callbacks = (wxClipsCallbacks *)GetClientData();
    if (callbacks)
    {
      wxNode *node = callbacks->Find("OnEvent");
      if (node)
      {
        wxStoreThingInTable((long)&event, &event, wxTYPE_MOUSE_EVENT);
        wxClipsFunction func = (wxClipsFunction)node->Data();
#ifdef USE_CLIPS
        char buf[100];
        sprintf(buf, "(%s %ld %ld)", func, (long)this, (long)&event);
        clipsEchoOn = FALSE;
        RouteCommand(buf);
        clipsEchoOn = TRUE;
#else
        (* (void (*)(long, long)) func) ((long)this, (long)&event);
#endif        
        wxDeleteThingInTable((long)&event);
      }
    }
  }

void wxClipsCanvas::OnPaint(void)
  {
    wxClipsCallbacks *callbacks = (wxClipsCallbacks *)GetClientData();
    if (callbacks)
    {
      wxNode *node = callbacks->Find("OnPaint");
      if (node)
      {
        wxClipsFunction func = (wxClipsFunction)node->Data();
#ifdef USE_CLIPS
        char buf[100];
        sprintf(buf, "(%s %ld)", func, (long)this);
        clipsEchoOn = FALSE;
        RouteCommand(buf);
        clipsEchoOn = TRUE;
#else
        (* (void (*)(long)) func) ((long)this);
#endif
      }
    }
  }

// Pen
class wxClipsPen: public wxPen
{
 public:
  wxClipsPen(wxColour& col, int width, int style):
   wxPen(col, width, style)
   {
   }
                   
  ~wxClipsPen()
   {
      wxDeleteThingInTable((long)this);
   }
};

// Text subwindow
class wxClipsTextWindow: public wxTextWindow
{
 public:
  wxClipsTextWindow(wxFrame *frame,
              int x = -1, int y = -1,
              int width = -1, int height = -1, long style = 0):
   wxTextWindow(frame, x, y, width, height, style)
    { }
                   
  ~wxClipsTextWindow()
    {
      wxClipsCallbacks *data;
      if (data = (wxClipsCallbacks *)GetClientData())
        delete data;
      wxDeleteThingInTable((long)this);
    }

  void OnSize(int w, int h);
};

void wxClipsTextWindow::OnSize(int w, int h)
  {
    wxClipsCallbacks *callbacks = (wxClipsCallbacks *)GetClientData();
    if (callbacks)
    {
      wxNode *node = callbacks->Find("OnSize");
      if (node)
      {
        wxClipsFunction func = (wxClipsFunction)node->Data();
#ifdef USE_CLIPS
        char buf[100];
        sprintf(buf, "(%s %ld %d %d)", func, (long)this, w, h);
        clipsEchoOn = FALSE;
        RouteCommand(buf);
        clipsEchoOn = TRUE;
#else
        (* (void (*)(long, long, long)) func) ((long)this, w, h);
#endif
      }
      else wxTextWindow::OnSize(w, h);
    }
    else
      wxTextWindow::OnSize(w, h);
  }

// Frame
class wxClipsFrame: public wxFrame
{
 public:
  wxClipsFrame(wxFrame *parent, char *title,
          int x=-1, int y=-1, int width=-1, int height=-1,
          long style = wxSDI | wxDEFAULT_FRAME):wxFrame(parent, title, x, y, width, height, style)
  {
    wxClipsFrameList.Append(this);
  }

  ~wxClipsFrame()
    {
      wxClipsCallbacks *data;
      if (data = (wxClipsCallbacks *)GetClientData())
        delete data;
      wxDeleteThingInTable((long)this);
      wxClipsFrameList.DeleteObject(this);
    }

  void OnSize(int w, int h);
  void OnMenuCommand(int id);
  Bool OnClose(void);
};

void wxClipsFrame::OnSize(int w, int h)
  {
    wxClipsCallbacks *callbacks = (wxClipsCallbacks *)GetClientData();
    if (callbacks)
    {
      wxNode *node = callbacks->Find("OnSize");
      if (node)
      {
        wxClipsFunction func = (wxClipsFunction)node->Data();
#ifdef USE_CLIPS
        char buf[100];
        sprintf(buf, "(%s %ld %d %d)", func, (long)this, w, h);
        clipsEchoOn = FALSE;
        RouteCommand(buf);
        clipsEchoOn = TRUE;
#else
        (* (void (*)(long, long, long)) func) ((long)this, w, h);
#endif
      }
      else wxFrame::OnSize(w, h);
    }
    else
      wxFrame::OnSize(w, h);
  }

void wxClipsFrame::OnMenuCommand(int id)
  {
    wxClipsCallbacks *callbacks = (wxClipsCallbacks *)GetClientData();
    if (callbacks)
    {
      wxNode *node = callbacks->Find("OnMenuCommand");
      if (node)
      {
        wxClipsFunction func = (wxClipsFunction)node->Data();
#ifdef USE_CLIPS
        char buf[100];
        sprintf(buf, "(%s %ld %d)", func, (long)this, (long)id);
        clipsEchoOn = FALSE;
        RouteCommand(buf);
        clipsEchoOn = TRUE;
#else
        (* (void (*)(long, long)) func) ((long)this, (long)id);
#endif
      }
    }
    else
      wxFrame::OnMenuCommand(id);
  }

Bool wxClipsFrame::OnClose(void)
  {
    wxClipsCallbacks *callbacks = (wxClipsCallbacks *)GetClientData();
    Bool ans = FALSE;
    if (callbacks)
    {
      wxNode *node = callbacks->Find("OnClose");
      if (node)
      {
        wxClipsFunction func = (wxClipsFunction )node->Data();
#ifdef USE_CLIPS
        char buf[200];
        clipsReturnType = clipsUNKNOWN;
        sprintf(buf, "(return-result (%s %ld))", func, (long)this);
        clipsEchoOn = FALSE;
        RouteCommand(buf);
        clipsEchoOn = TRUE;
#else
        clipsReturnLong = (* (long (*)(long)) func) ((long)this);
        clipsReturnType = clipsLONG;
#endif        
        if (clipsReturnType != clipsLONG)
	{
          ClipsError << "OnClose callback for frame should return an integer (1 or 0)!\n";
	}
        else ans = (Bool)clipsReturnLong;
      }
    }
    else ans = wxFrame::OnClose();
    return ans;
  }

// Button
class wxClipsButton: public wxButton
{
 public:
  wxClipsFunction clipsFunc;
  wxClipsButton(wxPanel *panel, wxClipsFunction clipsFuncName, wxFunction func, char *label, int x = -1, int y = -1,
           int width = -1, int height = -1, long style = 0):
    wxButton(panel, func, label, x, y, width, height, style)
 {
#ifdef USE_CLIPS
    if (clipsFuncName)
      clipsFunc = copystring(clipsFuncName);
    else clipsFunc = NULL;
#else
    clipsFunc = clipsFuncName;
#endif
 }
  wxClipsButton(wxPanel *panel, wxClipsFunction clipsFuncName, wxFunction func, wxBitmap *bitmap, int x = -1, int y = -1,
           int width = -1, int height = -1, long style = 0):
   wxButton(panel, func, bitmap, x, y, width, height, style)
 {
#ifdef USE_CLIPS
    if (clipsFuncName)
      clipsFunc = copystring(clipsFuncName);
    else clipsFunc = NULL;
#else
    clipsFunc = clipsFuncName;
#endif
 }
 ~wxClipsButton(void)
 {
#ifdef USE_CLIPS
   if (clipsFunc)
     delete[] clipsFunc;
#endif
   wxDeleteThingInTable((long)this);
 }
};

// Message
class wxClipsMessage: public wxMessage
{
 public:
  wxClipsMessage(wxPanel *panel, char *label, int x = -1, int y = -1, long style = 0):
    wxMessage(panel, label, x, y, style)
 {
 }
 ~wxClipsMessage(void)
 {
   wxDeleteThingInTable((long)this);
 }
};

// Text
class wxClipsText: public wxText
{
 public:
  wxClipsFunction clipsFunc;
  wxClipsText(wxPanel *panel, wxClipsFunction clipsFuncName, wxFunction func, char *label, char *value = "",
         int x = -1, int y = -1, int width = -1, int height = -1, long style = 0):
    wxText(panel, func, label, value, x, y, width, height, style)
 {
#ifdef USE_CLIPS
    if (clipsFuncName)
      clipsFunc = copystring(clipsFuncName);
    else clipsFunc = NULL;
#else
    clipsFunc = clipsFuncName;
#endif
 }

 ~wxClipsText(void)
 {
#ifdef USE_CLIPS
   if (clipsFunc)
     delete[] clipsFunc;
#endif
   wxDeleteThingInTable((long)this);
 }
};

// Multi-line Text
class wxClipsMultiText: public wxMultiText
{
 public:
  wxClipsFunction clipsFunc;
  wxClipsMultiText(wxPanel *panel, wxClipsFunction clipsFuncName, wxFunction func, char *label, char *value = "",
         int x = -1, int y = -1, int width = -1, int height = -1, long style = 0):
    wxMultiText(panel, func, label, value, x, y, width, height, style)
 {
#ifdef USE_CLIPS
    if (clipsFuncName)
      clipsFunc = copystring(clipsFuncName);
    else clipsFunc = NULL;
#else
    clipsFunc = clipsFuncName;
#endif
 }

 ~wxClipsMultiText(void)
 {
#ifdef USE_CLIPS
   if (clipsFunc)
     delete[] clipsFunc;
#endif
   wxDeleteThingInTable((long)this);
 }
};

// Check box
class wxClipsCheckBox: public wxCheckBox
{
 public:
  wxClipsFunction clipsFunc;
  wxClipsCheckBox(wxPanel *panel, wxClipsFunction clipsFuncName, wxFunction func, char *label,
         int x = -1, int y = -1, int width = -1, int height = -1, long style = 0):
    wxCheckBox(panel, func, label, x, y, width, height, style)
 {
#ifdef USE_CLIPS
    if (clipsFuncName)
      clipsFunc = copystring(clipsFuncName);
    else clipsFunc = NULL;
#else
    clipsFunc = clipsFuncName;
#endif
 }

 ~wxClipsCheckBox(void)
 {
#ifdef USE_CLIPS
   if (clipsFunc)
     delete[] clipsFunc;
#endif
   wxDeleteThingInTable((long)this);
 }
};

// Slider
class wxClipsSlider: public wxSlider
{
 public:
  wxClipsFunction clipsFunc;
  wxClipsSlider(wxPanel *panel, wxClipsFunction clipsFuncName, wxFunction func, char *label,
         int value, int min_value, int max_value, int width, int x = -1, int y = -1, long style = 0):
    wxSlider(panel, func, label, value, min_value, max_value, width, x, y, style)
 {
#ifdef USE_CLIPS
    if (clipsFuncName)
      clipsFunc = copystring(clipsFuncName);
    else clipsFunc = NULL;
#else
    clipsFunc = clipsFuncName;
#endif
 }

 ~wxClipsSlider(void)
 {
#ifdef USE_CLIPS
   if (clipsFunc)
     delete[] clipsFunc;
#endif
   wxDeleteThingInTable((long)this);
 }
};

// List box
class wxClipsListBox: public wxListBox
{
 public:
  wxClipsFunction clipsFunc;
  wxClipsListBox(wxPanel *panel, wxClipsFunction clipsFuncName, wxFunction func, char *label,
                 Bool multiple = FALSE, int x = -1, int y = -1, int width = -1, int height = -1,
                 long style = 0):
    wxListBox(panel, func, label, multiple, x, y, width, height, 0, NULL, style)
 {
#ifdef USE_CLIPS
    if (clipsFuncName)
      clipsFunc = copystring(clipsFuncName);
    else clipsFunc = NULL;
#else
    clipsFunc = clipsFuncName;
#endif
 }

 ~wxClipsListBox(void)
 {
#ifdef USE_CLIPS
   if (clipsFunc)
     delete[] clipsFunc;
#endif
   wxDeleteThingInTable((long)this);
 }
};

// Choice
class wxClipsChoice: public wxChoice
{
 public:
  wxClipsFunction clipsFunc;
  wxClipsChoice(wxPanel *panel, wxClipsFunction clipsFuncName, wxFunction func, char *label,
                 int x = -1, int y = -1, int width = -1, int height = -1, int n = 0, char **choices = NULL, long style = 0):
    wxChoice(panel, func, label, x, y, width, height, n, choices, style)
 {
#ifdef USE_CLIPS
    if (clipsFuncName)
      clipsFunc = copystring(clipsFuncName);
    else clipsFunc = NULL;
#else
    clipsFunc = clipsFuncName;
#endif
 }

 ~wxClipsChoice(void)
 {
#ifdef USE_CLIPS
   if (clipsFunc)
     delete[] clipsFunc;
#endif
   wxDeleteThingInTable((long)this);
 }
};

class wxClipsMenuBar: public wxMenuBar
{
 public:
  wxClipsMenuBar(void)
    { }
  ~wxClipsMenuBar(void)
  {
     wxDeleteThingInTable((long)this);
  }
};

class wxClipsMenu: public wxMenu
{
 public:
  wxClipsMenu(char *title):wxMenu(title)
    { }
  ~wxClipsMenu(void)
  {
     wxDeleteThingInTable((long)this);
  }
};

/************************************************************************
 * DDE Classes
 *
 ************************************************************************/

// wxConnection
class wxClipsConnection: public wxConnection
{
 public:
  wxClipsFunction clipsFunc;
  wxClipsConnection(wxClipsFunction clipsFuncName)
  {
#ifdef USE_CLIPS
     if (clipsFuncName)
       clipsFunc = copystring(clipsFuncName);
     else clipsFunc = NULL;
#else
    clipsFunc = clipsFuncName;
#endif
  }

 ~wxClipsConnection(void)
 {
#ifdef USE_CLIPS
   if (clipsFunc)
     delete[] clipsFunc;
#endif
   wxDeleteThingInTable((long)this);
 }

 Bool OnAdvise(char *topic, char *item, char *data, int size, int format);

 Bool OnExecute(char *topic, char *data, int size, int format);

 Bool OnPoke(char *topic, char *item, char *data, int size, int format);

 char *OnRequest(char *topic, char *item);

 Bool OnStartAdvise(char *topic, char *item);

 Bool OnStopAdvise(char *topic, char *item);
};

Bool wxClipsConnection::OnAdvise(char *topic, char *item, char *data, int size, int format)
 {
   if (clipsFunc)
   {
#ifdef USE_CLIPS
      char buf[400];
      sprintf(buf, "(return-result (%s %ld OnAdvise %s %s %s))", clipsFunc, (long)this, topic, item, data);
      clipsEchoOn = FALSE;
      RouteCommand(buf);
      clipsEchoOn = TRUE;
#else
#endif
      if (clipsReturnType != clipsLONG)
      {
        ClipsError << "Connection OnAdvise callback function should return an integer (0 or 1).\n";
        return FALSE;
      }
      else 
        return (Bool)clipsReturnLong;
   }
   else return FALSE;
 }

Bool wxClipsConnection::OnExecute(char *topic, char *data, int size, int format)
 {
   if (clipsFunc)
   {
#ifdef USE_CLIPS
      char buf[400];
      sprintf(buf, "(return-result (%s %ld OnExecute %s %s %s))", clipsFunc, (long)this, topic, "no item", data);
      clipsEchoOn = FALSE;
      RouteCommand(buf);
      clipsEchoOn = TRUE;
#else
#endif
      if (clipsReturnType != clipsLONG)
      {
        ClipsError << "Connection OnExecute callback function should return an integer (0 or 1).\n";
        return FALSE;
      }
      else 
        return (Bool)clipsReturnLong;
   }
   else return FALSE;
 }

Bool wxClipsConnection::OnPoke(char *topic, char *item, char *data, int size, int format)
 {
   if (clipsFunc)
   {
#ifdef USE_CLIPS
      char buf[400];
      sprintf(buf, "(return-result (%s %ld OnPoke %s %s %s))", clipsFunc, (long)this, topic, item, data);
      clipsEchoOn = FALSE;
      RouteCommand(buf);
      clipsEchoOn = TRUE;
#else
#endif
      if (clipsReturnType != clipsLONG)
      {
        ClipsError << "Connection OnPoke callback function should return an integer (0 or 1).\n";
        return FALSE;
      }
      else 
        return (Bool)clipsReturnLong;
   }
   else return FALSE;
 }

char *wxClipsConnection::OnRequest(char *topic, char *item)
 {
   if (clipsFunc)
   {
#ifdef USE_CLIPS
      char buf[400];
      sprintf(buf, "(return-result (%s %ld OnRequest %s %s %s))", clipsFunc, (long)this, topic, item, "no data");
      clipsEchoOn = FALSE;
      RouteCommand(buf);
      clipsEchoOn = TRUE;
#else
#endif
      if (clipsReturnType != clipsSTRING)
      {
        ClipsError << "Connection OnRequest callback function should return a string.\n";
        return NULL;
      }
      else 
        return clipsReturnString;
   }
   else return NULL;
 }

Bool wxClipsConnection::OnStartAdvise(char *topic, char *item)
 {
   if (clipsFunc)
   {
#ifdef USE_CLIPS
      char buf[400];
      sprintf(buf, "(return-result (%s %ld OnStartAdvise %s %s %s))", clipsFunc, (long)this, topic, item, "no data");
      clipsEchoOn = FALSE;
      RouteCommand(buf);
      clipsEchoOn = TRUE;
#else
#endif
      if (clipsReturnType != clipsLONG)
      {
        ClipsError << "Connection OnStartAdvise callback function should return an integer (0 or 1).\n";
        return FALSE;
      }
      else 
        return (Bool)clipsReturnLong;
   }
   else return FALSE;
 }

Bool wxClipsConnection::OnStopAdvise(char *topic, char *item)
 {
   if (clipsFunc)
   {
#ifdef USE_CLIPS
      char buf[400];
      sprintf(buf, "(return-result (%s %ld OnStopAdvise %s %s %s))", clipsFunc, (long)this, topic, item, "no data");
      clipsEchoOn = FALSE;
      RouteCommand(buf);
      clipsEchoOn = TRUE;
#else
#endif
      if (clipsReturnType != clipsLONG)
      {
        ClipsError << "Connection OnStopAdvise callback function should return an integer (0 or 1).\n";
        return FALSE;
      }
      else 
        return (Bool)clipsReturnLong;
   }
   else return FALSE;
 }

// wxServer
class wxClipsServer: public wxServer
{
 public:
  wxClipsFunction clipsFunc;
  wxClipsServer(wxClipsFunction clipsFuncName)
  {
#ifdef USE_CLIPS
     if (clipsFuncName)
       clipsFunc = copystring(clipsFuncName);
     else clipsFunc = NULL;
#else
    clipsFunc = clipsFuncName;
#endif
  }

 ~wxClipsServer(void)
 {
#ifdef USE_CLIPS
   if (clipsFunc)
     delete[] clipsFunc;
#endif
   wxDeleteThingInTable((long)this);
 }

 wxConnection *OnAcceptConnection(char *topic);
};

wxConnection *wxClipsServer::OnAcceptConnection(char *topic)
 {
   if (clipsFunc)
   {
#ifdef USE_CLIPS
      char buf[400];
      sprintf(buf, "(return-result (%s %ld %s))", clipsFunc, (long)this, topic);
      clipsEchoOn = FALSE;
      RouteCommand(buf);
      clipsEchoOn = TRUE;
      if (clipsReturnType != clipsSTRING)
      {
        ClipsError << "Server OnAcceptConnection callback function should return a function name string.\n";
        return NULL;
      }
      else if (clipsReturnString)
      {
        if (strcmp(clipsReturnString, "") == 0)
          return NULL;
        else
        {
          wxClipsConnection *connection = new wxClipsConnection(clipsReturnString);
          wxStoreThingInTable((long)connection, connection, wxTYPE_DDE_CONNECTION);
          return connection;
        }
      }
      else return NULL;
#else
      return NULL;
#endif
   }
   else return NULL;
 }

// wxClient
class wxClipsClient: public wxClient
{
 public:
  wxClipsFunction clipsFunc;
  wxClipsClient(wxClipsFunction clipsFuncName)
  {
#ifdef USE_CLIPS
     if (clipsFuncName)
       clipsFunc = copystring(clipsFuncName);
     else clipsFunc = NULL;
#else
     clipsFunc = clipsFuncName;
#endif
  }

 ~wxClipsClient(void)
 {
#ifdef USE_CLIPS
   if (clipsFunc)
     delete[] clipsFunc;
#endif
   wxDeleteThingInTable((long)this);
 }

 wxConnection *OnMakeConnection(void)
 {
   if (clipsFunc)
   {
      wxClipsConnection *connection = new wxClipsConnection(clipsFunc);
      wxStoreThingInTable((long)connection, connection, wxTYPE_DDE_CONNECTION);
      return connection;
   }
   else return NULL;
 }
};

/************************************************************************
 * GENERIC WINDOW FUNCTIONS
 *
 ************************************************************************/


// Delete window
long wxcWindowDelete(long id)
{
  wxWindow *win = (wxWindow *)wxGetTypedObject(id, wxTYPE_WINDOW);
  if (win)
  {
    delete win;
    wxDeleteThingInTable(id);
    return 1;
  }
  else
  {
    ClipsError << "Error in window-delete: could not find window " << id << ".\n";
    return 0;
  }
}

// Delete window
#ifdef USE_CLIPS
long clipsWindowDelete()
{
  if (ArgCountCheck("clipsWindowDelete", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcWindowDelete(id);
}
#endif

// Show window
long wxcWindowShow(long id, Bool show)
{
  wxWindow *win = (wxWindow *)wxGetTypedObject(id, wxTYPE_WINDOW);
  if (win)
  {
    win->Show(show);
    return 1;
  }
  else
  {
    ClipsError << "Error in window-show: could not find window " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsWindowShow()
{
  if (ArgCountCheck("clipsWindowShow", AT_LEAST, 1) == -1)
    return 0;

  long id = RtnLong(1);
  Bool show = TRUE;
  if (RtnArgCount() > 1)
    show = (Bool)RtnLong(2);
  return wxcWindowShow(id, show);
}
#endif

// Get window parent
long wxcWindowGetParent(long id)
{
  wxWindow *win = (wxWindow *)wxGetTypedObject(id, wxTYPE_WINDOW);
  if (win)
  {
    wxWindow *parent = win->GetParent();
    if (parent)
      return (long)parent;
    else
      return 0;
  }
  else
  {
    ClipsError << "Error in window-get-parent: could not find window " << id << ".\n";
    return 0;
  }
}

// Get window parent
#ifdef USE_CLIPS
long clipsWindowGetParent()
{
  if (ArgCountCheck("clipsWindowGetParent", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);

  return wxcWindowGetParent(id);
}
#endif

// Add a callback to a window
long wxcWindowAddCallback(long id, char *event_type, wxClipsFunction callback)
{
  if (!event_type || !callback)
    return 0;

  wxWindow *win = (wxWindow *)wxGetTypedObject(id, wxTYPE_WINDOW);
  if (win)
  {
    wxClipsCallbacks *data = (wxClipsCallbacks *)win->GetClientData();
    if (!data)
    {
      data = new wxClipsCallbacks;
      win->SetClientData((char *)data);
    }
#ifdef USE_CLIPS
    data->Append(copystring(event_type), (wxObject *)copystring(callback));
#else
    data->Append(copystring(event_type), (wxObject *)callback);
#endif    
    return 1;
  }
  else
  {
    ClipsError << "Error in window-add-callback: could not find window " << id << ".\n";
    return 0;
  }
}

// Add a callback to a window
#ifdef USE_CLIPS
long clipsWindowAddCallback()
{
  if (ArgCountCheck("wxcWindowAddCallback", EXACTLY, 3) == -1)
    return 0;

  long id = RtnLong(1);

  char *event_type = RtnLexeme(2);
  char *callback = RtnLexeme(3);

  return wxcWindowAddCallback(id, event_type, callback);
}
#endif

// Get window width
long wxcWindowGetWidth(long id)
{
  wxWindow *win = (wxWindow *)wxGetTypedObject(id, wxTYPE_WINDOW);
  if (win)
  {
    int w, h;
    win->GetSize(&w, &h);
    return w;
  }
  else
  {
    ClipsError << "Error in window-get-width: could not find window " << id << ".\n";
    return 0;
  }
}

// Get window width
#ifdef USE_CLIPS
long clipsWindowGetWidth()
{
  if (ArgCountCheck("clipsWindowGetWidth", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcWindowGetWidth(id);
}
#endif

// Get window height
long wxcWindowGetHeight(long id)
{
  wxWindow *win = (wxWindow *)wxGetTypedObject(id, wxTYPE_WINDOW);
  if (win)
  {
    int w, h;
    win->GetSize(&w, &h);
    return h;
  }
  else
  {
    ClipsError << "Error in window-get-height: could not find window " << id << ".\n";
    return 0;
  }
}

// Get window height
#ifdef USE_CLIPS
long clipsWindowGetHeight()
{
  if (ArgCountCheck("clipsWindowGetHeight", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcWindowGetHeight(id);
}
#endif

// Get window client width
long wxcWindowGetClientWidth(long id)
{
  wxWindow *win = (wxWindow *)wxGetTypedObject(id, wxTYPE_WINDOW);
  if (win)
  {
    int w, h;
    win->GetClientSize(&w, &h);
    return w;
  }
  else
  {
    ClipsError << "Error in window-get-client-width: could not find window " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsWindowGetClientWidth()
{
  if (ArgCountCheck("clipsWindowGetClientWidth", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcWindowGetClientWidth(id);
}
#endif

// Get window client height
long wxcWindowGetClientHeight(long id)
{
  wxWindow *win = (wxWindow *)wxGetTypedObject(id, wxTYPE_WINDOW);
  if (win)
  {
    int w, h;
    win->GetClientSize(&w, &h);
    return h;
  }
  else
  {
    ClipsError << "Error in window-get-client-height: could not find window " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsWindowGetClientHeight()
{
  if (ArgCountCheck("clipsWindowGetClientHeight", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcWindowGetClientHeight(id);
}
#endif

// Get window x position
long wxcWindowGetX(long id)
{
  wxWindow *win = (wxWindow *)wxGetTypedObject(id, wxTYPE_WINDOW);
  if (win)
  {
    int x, y;
    win->GetPosition(&x, &y);
    return x;
  }
  else
  {
    ClipsError << "Error in window-get-x: could not find window " << id << ".\n";
    return 0;
  }
}

// Get window x position
#ifdef USE_CLIPS
long clipsWindowGetX()
{
  if (ArgCountCheck("clipsWindowGetX", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcWindowGetX(id);
}
#endif

// Get window y position
long wxcWindowGetY(long id)
{
  wxWindow *win = (wxWindow *)wxGetTypedObject(id, wxTYPE_WINDOW);
  if (win)
  {
    int x, y;
    win->GetPosition(&x, &y);
    return y;
  }
  else
  {
    ClipsError << "Error in window-get-y: could not find window " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsWindowGetY()
{
  if (ArgCountCheck("clipsWindowGetY", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcWindowGetY(id);
}
#endif

// Set size
long wxcWindowSetSize(long id, long x, long y, long w, long h)
{
  wxWindow *win = (wxWindow *)wxGetTypedObject(id, wxTYPE_WINDOW);
  if (win)
  {
    win->SetSize((int)x, (int)y, (int)w, (int)h);
    return 1;
  }
  else
  {
    ClipsError << "Error in window-set-size: could not find window " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsWindowSetSize()
{
  if (ArgCountCheck("clipsWindowSetSize", EXACTLY, 5) == -1)
    return 0;

  long id = RtnLong(1);
  int x = (int) RtnLong(2);
  int y = (int) RtnLong(3);
  int w = (int) RtnLong(4);
  int h = (int) RtnLong(5);
  return wxcWindowSetSize(id, x, y, w, h);
}
#endif

// Set client size
long wxcWindowSetClientSize(long id, long w, long h)
{
  wxWindow *win = (wxWindow *)wxGetTypedObject(id, wxTYPE_WINDOW);
  if (win)
  {
    win->SetClientSize((int)w, (int)h);
    return 1;
  }
  else
  {
    ClipsError << "Error in window-set-client-size: could not find window " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsWindowSetClientSize()
{
  if (ArgCountCheck("clipsWindowSetClientSize", EXACTLY, 3) == -1)
    return 0;

  long id = RtnLong(1);
  int w = (int) RtnLong(2);
  int h = (int) RtnLong(3);

  return wxcWindowSetClientSize(id, w, h);
}
#endif

// Set focus
long wxcWindowSetFocus(long id)
{
  wxWindow *win = (wxWindow *)wxGetTypedObject(id, wxTYPE_WINDOW);
  if (win)
  {
    win->SetFocus();
    return 1;
  }
  else
  {
    ClipsError << "Error in window-set-focus: could not find window " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsWindowSetFocus()
{
  if (ArgCountCheck("clipsWindowSetFocus", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcWindowSetFocus(id);
}
#endif

// Fit
long wxcWindowFit(long id)
{
  wxWindow *win = (wxWindow *)wxGetTypedObject(id, wxTYPE_WINDOW);
  if (win)
  {
    win->Fit();
    return 1;
  }
  else
  {
    ClipsError << "Error in window-fit: could not find window " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsWindowFit()
{
  if (ArgCountCheck("clipsWindowFit", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcWindowFit(id);
}
#endif

// Centre
long wxcWindowCentre(long id, char *stringDir)
{
  int direction = wxBOTH;
  if (stringDir)
  {
    if (strcmp(stringDir, "wxBOTH") == 0)
      direction = wxBOTH;
    else if (strcmp(stringDir, "wxHORIZONTAL") == 0)
      direction = wxHORIZONTAL;
    else if (strcmp(stringDir, "wxVERTICAL") == 0)
      direction = wxVERTICAL;
    else
    {
      ClipsError << "Error in window-centre: unrecognised direction " << direction << "\n.";
      ClipsError << "Should be wxHORIZONTAL, wxVERTICAL or wxBOTH.\n";
      return 0;
    }
  }

  wxWindow *win = (wxWindow *)wxGetTypedObject(id, wxTYPE_WINDOW);
  if (win)
  {
    win->Centre(direction);
    return 1;
  }
  else
  {
    ClipsError << "Error in window-centre: could not find window " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsWindowCentre()
{
  if (ArgCountCheck("clipsWindowCentre", AT_LEAST, 1) == -1)
    return 0;

  int no_args = RtnArgCount();
  long id = RtnLong(1);
  int direction = wxHORIZONTAL;
  char *stringDir = "wxBOTH";
  if (no_args > 1)
    stringDir = RtnLexeme(2);

  return wxcWindowCentre(id, stringDir);
}
#endif

/************************************************************************
 * GENERIC PANEL ITEM FUNCTIONS
 *
 ************************************************************************/

// Set this button to be the default
long wxcPanelItemSetDefault(long id)
{
  wxButton *item = (wxButton *)wxGetTypedObject(id, wxTYPE_BUTTON);
  if (item)
  {
    item->SetDefault();
    return 1;
  }
  else
  {
    ClipsError << "Error in panel-item-set-default: could not find item " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsPanelItemSetDefault()
{
  if (ArgCountCheck("clipsPanelItemSetDefault", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);

  return wxcPanelItemSetDefault(id);
}
#endif

// Set label
long wxcPanelItemSetLabel(long id, char *label)
{
  if (!label)
    return 0;

  wxItem *item = (wxItem *)wxGetTypedObject(id, wxTYPE_ITEM);
  if (item)
  {
    item->SetLabel(label);
    return 1;
  }
  else
  {
    ClipsError << "Error in panel-item-set-label: could not find item " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsPanelItemSetLabel()
{
  if (ArgCountCheck("clipsPanelItemSetLabel", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  char *label = RtnLexeme(2);
  return wxcPanelItemSetLabel(id, label);
}
#endif

// Get label
char *wxcPanelItemGetLabel(long id)
{
  wxItem *item = (wxItem *)wxGetTypedObject(id, wxTYPE_ITEM);
  if (item)
  {
    char *label = item->GetLabel();
    return label;
  }
  else
  {
    ClipsError << "Error in panel-item-get-label: could not find item " << id << ".\n";
    return NULL;
  }
}

#ifdef USE_CLIPS
char *clipsPanelItemGetLabel()
{
  if (ArgCountCheck("clipsPanelItemSetLabel", EXACTLY, 1) == -1)
    return (char *)AddSymbol("");

  long id = RtnLong(1);
  char *ans = wxcPanelItemGetLabel(id);
  if (ans)
    return ans;
  else
    return (char *)AddSymbol("");
}
#endif

/************************************************************************
 * PANEL ITEM CALLBACKS
 *
 ************************************************************************/

// Event within scope of a callback (if nested, can only rely on
// getting event info immediately after function is called.)
wxCommandEvent *wxCurrentEvent = NULL;

void wxClipsButtonCallback(wxClipsButton& item, wxCommandEvent& event)
{
  wxCurrentEvent = &event;
  if (item.clipsFunc)
  {
#ifdef USE_CLIPS
    char buf[100];
    sprintf(buf, "(%s %ld)", item.clipsFunc, &item);
    clipsEchoOn = FALSE;
    RouteCommand(buf);
    clipsEchoOn = TRUE;
#else
    (* (void (*)(long)) item.clipsFunc) ((long)&item);
#endif
  }
  wxCurrentEvent = NULL;
}

void wxClipsTextCallback(wxClipsText& item, wxCommandEvent& event)
{
  wxCurrentEvent = &event;
  if (item.clipsFunc)
  {
#ifdef USE_CLIPS
    char buf[100];
    sprintf(buf, "(%s %ld)", item.clipsFunc, &item);
    clipsEchoOn = FALSE;
    RouteCommand(buf);
    clipsEchoOn = TRUE;
#else
    (* (void (*)(long)) item.clipsFunc) ((long)&item);
#endif
  }
  wxCurrentEvent = NULL;
}

void wxClipsMultiTextCallback(wxClipsMultiText& item, wxCommandEvent& event)
{
  wxCurrentEvent = &event;
  if (item.clipsFunc)
  {
#ifdef USE_CLIPS
    char buf[100];
    sprintf(buf, "(%s %ld)", item.clipsFunc, &item);
    clipsEchoOn = FALSE;
    RouteCommand(buf);
    clipsEchoOn = TRUE;
#else
    (* (void (*)(long)) item.clipsFunc) ((long)&item);
#endif
  }
  wxCurrentEvent = NULL;
}

void wxClipsListBoxCallback(wxClipsListBox& item, wxCommandEvent& event)
{
  wxCurrentEvent = &event;
  if (item.clipsFunc)
  {
#ifdef USE_CLIPS
    char buf[100];
    sprintf(buf, "(%s %ld)", item.clipsFunc, &item);
    clipsEchoOn = FALSE;
    RouteCommand(buf);
    clipsEchoOn = TRUE;
#else
    (* (void (*)(long)) item.clipsFunc) ((long)&item);
#endif
  }
  wxCurrentEvent = NULL;
}

void wxClipsChoiceCallback(wxClipsChoice& item, wxCommandEvent& event)
{
  wxCurrentEvent = &event;
  if (item.clipsFunc)
  {
#ifdef USE_CLIPS
    char buf[100];
    sprintf(buf, "(%s %ld)", item.clipsFunc, &item);
    clipsEchoOn = FALSE;
    RouteCommand(buf);
    clipsEchoOn = TRUE;
#else
    (* (void (*)(long)) item.clipsFunc) ((long)&item);
#endif
  }
  wxCurrentEvent = NULL;
}

void wxClipsCheckBoxCallback(wxClipsCheckBox& item, wxCommandEvent& event)
{
  wxCurrentEvent = &event;
  if (item.clipsFunc)
  {
#ifdef USE_CLIPS
    char buf[100];
    sprintf(buf, "(%s %ld)", item.clipsFunc, &item);
    clipsEchoOn = FALSE;
    RouteCommand(buf);
    clipsEchoOn = TRUE;
#else
    (* (void (*)(long)) item.clipsFunc) ((long)&item);
#endif
  }
  wxCurrentEvent = NULL;
}

void wxClipsSliderCallback(wxClipsSlider& item, wxCommandEvent& event)
{
  wxCurrentEvent = &event;
  if (item.clipsFunc)
  {
#ifdef USE_CLIPS
    char buf[100];
    sprintf(buf, "(%s %ld)", item.clipsFunc, &item);
    clipsEchoOn = FALSE;
    RouteCommand(buf);
    clipsEchoOn = TRUE;
#else
    (* (void (*)(long)) item.clipsFunc) ((long)&item);
#endif
  }
  wxCurrentEvent = NULL;
}


/************************************************************************
 * DIALOG AND PANEL ITEM FUNCTIONS
 *
 ************************************************************************/

// Create a new dialog box
long wxcDialogBoxCreate(long parent_id, char *title, long modal,
  long x, long y, long width, long height, char *styleString)
{
  wxWindow *parent = NULL;

  if (parent_id != 0)
  {
    parent = (wxWindow *)wxGetTypedObject(parent_id, wxTYPE_WINDOW);
    if (!parent)
    {
      ClipsError << "Error in dialog-box-create: could not find parent window " << parent_id << ".\n";
      return 0;
    }
  }
  long style = wxClipsParseBitList(styleString);

  wxClipsDialogBox *dialog_box = new wxClipsDialogBox(parent, title, (Bool)modal,
       (int)x, (int)y, (int)width, (int)height, style);
  wxStoreThingInTable((long)dialog_box, dialog_box, wxTYPE_DIALOG_BOX);
  return (long)dialog_box;
}

#ifdef USE_CLIPS
long clipsDialogBoxCreate()
{
  if (ArgCountCheck("clipsDialogBoxCreate", AT_LEAST, 2) == -1)
    return 0;

  int no_args = RtnArgCount();

  long parent_id = RtnLong(1);
  char *title = RtnLexeme(2);

  if (!title)
    return -1;

  // Set up defaults
  Bool modal = TRUE;
  int x = -1;
  int y = -1;
  int width = -1;
  int height = -1;
  char *style = "wxABSOLUTE_POSITIONING | wxSYSTEM_MENU | wxTHICK_FRAME";

  if (no_args > 2)
    modal = (Bool)RtnLong(3);

  if (no_args > 3)
    x = (int) RtnLong(4);

  if (no_args > 4)
    y = (int) RtnLong(5);

  if (no_args > 5)
    width = (int) RtnLong(6);

  if (no_args > 6)
    height = (int) RtnLong(7);

  if (no_args > 7)
    style = RtnLexeme(8);

  return wxcDialogBoxCreate(parent_id, title, modal, x, y, width, height, style);
}
#endif

// Create a new panel
long wxcPanelCreate(long parent_id, long x, long y, long width, long height, char *styleString)
{
  wxFrame *frame = NULL;

  if (parent_id != 0)
  {
    frame = (wxFrame *)wxGetTypedObject(parent_id, wxTYPE_FRAME);
    if (!frame)
    {
      ClipsError << "Error in panel-create: could not find parent frame " << parent_id << ".\n";
      return 0;
    }
  }
  long style = wxClipsParseBitList(styleString);

  wxClipsPanel *panel = new wxClipsPanel(frame, (int)x, (int)y, (int)width, (int)height, style);
  wxStoreThingInTable((long)panel, panel, wxTYPE_PANEL);
  return (long)panel;
}

#ifdef USE_CLIPS
long clipsPanelCreate()
{
  if (ArgCountCheck("clipsPanelCreate", AT_LEAST, 1) == -1)
    return 0;

  int no_args = RtnArgCount();

  long parent_id = RtnLong(1);

  // Set up defaults
  int x = -1;
  int y = -1;
  int width = -1;
  int height = -1;
  char *style = "wxABSOLUTE_POSITIONING";

  if (no_args > 1)
    x = (int) RtnLong(2);

  if (no_args > 2)
    y = (int) RtnLong(3);

  if (no_args > 3)
    width = (int) RtnLong(4);

  if (no_args > 4)
    height = (int) RtnLong(5);

  if (no_args > 5)
    style = RtnLexeme(6);

  return wxcPanelCreate(parent_id, x, y, width, height, style);
}
#endif

// Set the label position
long wxcPanelSetLabelPosition(long id, char *position)
{
  wxPanel *panel = (wxPanel *)wxGetTypedObject(id, wxTYPE_PANEL);
  if (panel)
  {
    int pos;
    if (strcmp(position, "wxVERTICAL") == 0)
      pos = wxVERTICAL;
    else if (strcmp(position, "wxHORIZONTAL") == 0)
      pos = wxHORIZONTAL;
    else
    {
      ClipsError << "Error in panel-set-label-position: argument should be wxVERTICAL or wxHORIZONTAL.\n";
      return 0;
    }
    panel->SetLabelPosition(pos);
    return 1;
  }
  else
  {
    ClipsError << "Error in panel-set-label-position: could not find panel " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsPanelSetLabelPosition()
{
  if (ArgCountCheck("clipsPanelSetLabelPosition", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  char *position = RtnLexeme(2);
  if (!position)
    return 0;
  return wxcPanelSetLabelPosition(id, position);
}
#endif

// Set the panel label font
long wxcPanelSetLabelFont(long id, long fontId)
{
  wxPanel *panel = (wxPanel *)wxGetTypedObject(id, wxTYPE_PANEL);
  if (panel)
  {
    wxFont *font = (wxFont *)wxGetTypedObject(fontId, wxTYPE_FONT);
    if (font)
    {
      panel->SetLabelFont(font);
      return 1;
    }
    else
    {
      ClipsError << "Error in panel-set-label-font: could not find font " << fontId << ".\n";
      return 0;
    }
  }
  else
  {
    ClipsError << "Error in panel-set-label-font: could not find panel " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsPanelSetLabelFont()
{
  if (ArgCountCheck("clipsPanelSetLabelFont", EXACTLY, 2) == -1)
    return 0;

  return wxcPanelSetLabelFont(RtnLong(1), RtnLong(2));
}
#endif

// Set the panel button font
long wxcPanelSetButtonFont(long id, long fontId)
{
  wxPanel *panel = (wxPanel *)wxGetTypedObject(id, wxTYPE_PANEL);
  if (panel)
  {
    wxFont *font = (wxFont *)wxGetTypedObject(fontId, wxTYPE_FONT);
    if (font)
    {
      panel->SetButtonFont(font);
      return 1;
    }
    else
    {
      ClipsError << "Error in panel-set-button-font: could not find font " << fontId << ".\n";
      return 0;
    }
  }
  else
  {
    ClipsError << "Error in panel-set-button-font: could not find panel " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsPanelSetButtonFont()
{
  if (ArgCountCheck("clipsPanelSetButtonFont", EXACTLY, 2) == -1)
    return 0;

  return wxcPanelSetButtonFont(RtnLong(1), RtnLong(2));
}
#endif

// New line
long wxcPanelNewLine(long id)
{
  wxPanel *panel = (wxPanel *)wxGetTypedObject(id, wxTYPE_PANEL);
  if (panel)
  {
    panel->NewLine();
    return 1;
  }
  else
  {
    ClipsError << "Error in panel-new-line: not find panel " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsPanelNewLine()
{
  if (ArgCountCheck("clipsPanelNewLine", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcPanelNewLine(id);
}
#endif

// Create a new canvas
long wxcCanvasCreate(long parent_id, long x, long y, long width, long height, char *styleString)
{
  wxFrame *frame = NULL;

  if (parent_id != 0)
  {
    frame = (wxFrame *)wxGetTypedObject(parent_id, wxTYPE_FRAME);
    if (!frame)
    {
      ClipsError << "Error in canvas-create: could not find parent frame " << parent_id << ".\n";
      return 0;
    }
  }
  long style = wxClipsParseBitList(styleString);

  wxClipsCanvas *canvas = new wxClipsCanvas(frame, (int)x, (int)y, (int)width, (int)height, (int)style);
  wxStoreThingInTable((long)canvas, canvas, wxTYPE_CANVAS);
  return (long)canvas;
}

#ifdef USE_CLIPS
long clipsCanvasCreate()
{
  if (ArgCountCheck("clipsCanvasCreate", AT_LEAST, 1) == -1)
    return 0;

  int no_args = RtnArgCount();

  long parent_id = RtnLong(1);

  // Set up defaults
  int x = -1;
  int y = -1;
  int width = -1;
  int height = -1;
  char *style = "wxRETAINED";

  if (no_args > 1)
    x = (int) RtnLong(2);

  if (no_args > 2)
    y = (int) RtnLong(3);

  if (no_args > 3)
    width = (int) RtnLong(4);

  if (no_args > 4)
    height = (int) RtnLong(5);

  if (no_args > 5)
    style = RtnLexeme(6);

  return wxcCanvasCreate(parent_id, x, y, width, height, style);
}
#endif

// Set the canvas scrollbars
long wxcCanvasSetScrollbars(long id, long horiz, long vert,
  long x_length, long y_length, long x_page, long y_page)
{
  wxCanvas *canvas = (wxCanvas *)wxGetTypedObject(id, wxTYPE_CANVAS);
  if (canvas)
  {
    canvas->SetScrollbars((int)horiz, (int)vert,
                          (int)x_length, (int)y_length, (int)x_page, (int)y_page);
    return 1;
  }
  else
  {
    ClipsError << "Error in canvas-set-scrollbars: could not find canvas " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsCanvasSetScrollbars()
{
  if (ArgCountCheck("clipsCanvasSetScrollbars", EXACTLY, 7) == -1)
    return 0;

  long id = RtnLong(1);
  long horiz = RtnLong(2);
  long vert = RtnLong(3);
  long x_length = RtnLong(4);
  long y_length = RtnLong(5);
  long x_page = RtnLong(6);
  long y_page = RtnLong(7);
  return wxcCanvasSetScrollbars(id, horiz, vert, x_length, y_length,
     x_page, y_page);
}
#endif

// Set the canvas scrollbars
long wxcCanvasScroll(long id, long x_pos, long y_pos)
{
  wxCanvas *canvas = (wxCanvas *)wxGetTypedObject(id, wxTYPE_CANVAS);
  if (canvas)
  {
    canvas->Scroll((int)x_pos, (int)y_pos);
    return 1;
  }
  else
  {
    ClipsError << "Error in canvas-scroll: could not find canvas " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsCanvasScroll()
{
  if (ArgCountCheck("clipsCanvasScroll", EXACTLY, 3) == -1)
    return 0;

  long id = RtnLong(1);
  long x_pos = RtnLong(2);
  long y_pos = RtnLong(3);

  return wxcCanvasScroll(id, x_pos, y_pos);
}
#endif

// Get canvas's device context
long wxcCanvasGetDC(long id)
{
  wxCanvas *canvas = (wxCanvas *)wxGetTypedObject(id, wxTYPE_CANVAS);
  if (canvas)
  {
    return (long)canvas->GetDC();
  }
  else
  {
    ClipsError << "Error in canvas-get-dc: could not find canvas " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsCanvasGetDC()
{
  if (ArgCountCheck("clipsCanvasGetDC", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);

  return wxcCanvasGetDC(id);
}
#endif

/*
 * Device context functions
 */

long wxcDCSetPen(long id, long penId)
{
  wxDC *dc = (wxDC *)wxGetTypedObject(id, wxTYPE_DC);
  if (dc)
  {
    wxPen *pen = (wxPen *)wxGetTypedObject(penId, wxTYPE_PEN);
    if (pen)
    {
      dc->SetPen(pen);
      return 1;
    }
    else
    {
      ClipsError << "Error in dc-set-pen: could not find pen " << penId << ".\n";
      return 0;
    }
  }
  else
  {
    ClipsError << "Error in dc-set-pen: could not find dc " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsDCSetPen()
{
  if (ArgCountCheck("clipsDCSetPen", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  long penId = RtnLong(2);

  return wxcDCSetPen(id, penId);
}
#endif

long wxcDCSetBrush(long id, long brushId)
{
  wxDC *dc = (wxDC *)wxGetTypedObject(id, wxTYPE_DC);
  if (dc)
  {
    wxBrush *brush = (wxBrush *)wxGetTypedObject(brushId, wxTYPE_BRUSH);
    if (brush)
    {
      dc->SetBrush(brush);
      return 1;
    }
    else
    {
      ClipsError << "Error in dc-set-brush: could not find brush " << brushId << ".\n";
      return 0;
    }
  }
  else
  {
    ClipsError << "Error in dc-set-brush: could not find dc " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsDCSetBrush()
{
  if (ArgCountCheck("clipsDCSetBrush", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  long brushId = RtnLong(2);

  return wxcDCSetBrush(id, brushId);
}
#endif

long wxcDCSetFont(long id, long fontId)
{
  wxDC *dc = (wxDC *)wxGetTypedObject(id, wxTYPE_DC);
  if (dc)
  {
    wxFont *font = (wxFont *)wxGetTypedObject(fontId, wxTYPE_FONT);
    if (font)
    {
      dc->SetFont(font);
      return 1;
    }
    else
    {
      ClipsError << "Error in dc-set-font: could not find font " << fontId << ".\n";
      return 0;
    }
  }
  else
  {
    ClipsError << "Error in dc-set-font: could not find dc " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsDCSetFont()
{
  if (ArgCountCheck("clipsDCSetFont", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  long fontId = RtnLong(2);

  return wxcDCSetFont(id, fontId);
}
#endif

long wxcDCSetTextForeground(long id, char *colour)
{
  if (!colour)
    return 0;

  wxDC *dc = (wxDC *)wxGetTypedObject(id, wxTYPE_DC);
  if (dc)
  {
    wxColour *col = wxTheColourDatabase->FindColour(colour);
    if (col)
    {
      dc->SetTextForeground(col);
      return 1;
    }
    else
    {
      ClipsError << "Error in dc-set-text-foreground: could not find colour " << colour << ".\n";
      return 0;
    }
  }
  else
  {
    ClipsError << "Error in dc-set-text-foreground: could not find dc " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsDCSetTextForeground()
{
  if (ArgCountCheck("clipsDCSetTextForeground", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  char *colour = RtnLexeme(2);
  return wxcDCSetTextForeground(id, colour);
}
#endif

long wxcDCSetTextBackground(long id, char *colour)
{
  if (!colour)
    return 0;

  wxDC *dc = (wxDC *)wxGetTypedObject(id, wxTYPE_DC);
  if (dc)
  {
    wxColour *col = wxTheColourDatabase->FindColour(colour);
    if (col)
    {
      dc->SetTextBackground(col);
      return 1;
    }
    else
    {
      ClipsError << "Error in dc-set-text-background: could not find colour " << colour << ".\n";
      return 0;
    }
  }
  else
  {
    ClipsError << "Error in dc-set-text-background: could not find dc " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsDCSetTextBackground()
{
  if (ArgCountCheck("clipsDCSetTextBackground", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  char *colour = RtnLexeme(2);
  return wxcDCSetTextBackground(id, colour);
}
#endif

long wxcDCSetLogicalFunction(long id, char *op)
{
  if (!op)
    return 0;

  wxDC *dc = (wxDC *)wxGetTypedObject(id, wxTYPE_DC);
  if (dc)
  {
    int log_op = 0;
    
    if (strcmp(op, "wxCOPY") == 0)
      log_op = wxCOPY;
    else if (strcmp(op, "wxXOR") == 0)
      log_op = wxXOR;
    else if (strcmp(op, "wxINVERT") == 0)
      log_op = wxINVERT;
    else if (strcmp(op, "wxOR_REVERSE") == 0)
      log_op = wxOR_REVERSE;
    else if (strcmp(op, "wxAND_REVERSE") == 0)
      log_op = wxAND_REVERSE;
    else
    {
      ClipsError << "Error in dc-set-logical-function: unrecognized function " << op << "\n";
      ClipsError << "Function must be one of wxCOPY, wxXOR, wxINVERT, wxOR_REVERSE, wxAND_REVERSE.\n";
      return 0;
    }
    dc->SetLogicalFunction(log_op);
    return 1;
  }
  else
  {
    ClipsError << "Error in dc-set-logical-function: could not find dc " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsDCSetLogicalFunction()
{
  if (ArgCountCheck("clipsDCSetLogicalFunction", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  char *op = RtnLexeme(2);
  return wxcDCSetLogicalFunction(id, op);
}
#endif

long wxcDCBlit(long id, float xdest, float ydest, float width, float height, long sourceId,
  float xsrc, float ysrc, char *logicalFuncString)
{
  wxDC *dc = (wxDC *)wxGetTypedObject(id, wxTYPE_DC);
  wxDC *sourceDC = (wxDC *)wxGetTypedObject(sourceId, wxTYPE_DC);
  if (dc && sourceDC)
  {
    long op = wxClipsParseBitList(logicalFuncString);
    dc->Blit(xdest, ydest, width, height, (wxCanvasDC *)sourceDC, xsrc, ysrc, (int)op);
    return 1;
  }
  else
  {
    ClipsError << "Error in dc-blit: could not find one or both device contexts, " << id << " or " << sourceId << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsDCBlit()
{
  if (ArgCountCheck("clipsDCBlit", AT_LEAST, 8) == -1)
    return 0;

  long id = RtnLong(1);
  float xdest = (float)RtnDouble(2);
  float ydest = (float)RtnDouble(3);
  float width = (float)RtnDouble(4);
  float height = (float)RtnDouble(5);
  long sourceId = RtnLong(6);
  float xsrc = (float)RtnDouble(7);
  float ysrc = (float)RtnDouble(8);
  char *logicalOpString = "wxCOPY";
  if (RtnArgCount() > 8)
    logicalOpString = RtnLexeme(9);
  return wxcDCBlit(id, xdest, ydest, width, height, sourceId, xsrc, ysrc, logicalOpString);
}
#endif
 
long wxcDCDrawLine(long id, float x1, float y1, float x2, float y2)
{
  wxDC *dc = (wxDC *)wxGetTypedObject(id, wxTYPE_DC);
  if (dc)
  {
    dc->DrawLine(x1, y1, x2, y2);
    return 1;
  }
  else
  {
    ClipsError << "Error in dc-draw-line: could not find dc " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsDCDrawLine()
{
  if (ArgCountCheck("clipsDCDrawLine", EXACTLY, 5) == -1)
    return 0;

  long id = RtnLong(1);
  float x1 = (float)RtnDouble(2);
  float y1 = (float)RtnDouble(3);
  float x2 = (float)RtnDouble(4);
  float y2 = (float)RtnDouble(5);
  return wxcDCDrawLine(id, x1, y1, x2, y2);
}
#endif

long wxcDCDrawPoint(long id, float x1, float y1)
{
  wxDC *dc = (wxDC *)wxGetTypedObject(id, wxTYPE_DC);
  if (dc)
  {
    dc->DrawPoint(x1, y1);
    return 1;
  }
  else
  {
    ClipsError << "Error in dc-draw-point: could not find dc " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsDCDrawPoint()
{
  if (ArgCountCheck("clipsDCDrawPoint", EXACTLY, 3) == -1)
    return 0;

  long id = RtnLong(1);
  float x1 = (float)RtnDouble(2);
  float y1 = (float)RtnDouble(3);
  return wxcDCDrawPoint(id, x1, y1);
}
#endif

long wxcDCDrawRectangle(long id, float x1, float y1, float x2, float y2)
{
  wxDC *dc = (wxDC *)wxGetTypedObject(id, wxTYPE_DC);
  if (dc)
  {
    dc->DrawRectangle(x1, y1, x2, y2);
    return 1;
  }
  else
  {
    ClipsError << "Error in dc-draw-rectangle: could not find dc " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsDCDrawRectangle()
{
  if (ArgCountCheck("clipsDCDrawRectangle", EXACTLY, 5) == -1)
    return 0;

  long id = RtnLong(1);
  float x1 = (float)RtnDouble(2);
  float y1 = (float)RtnDouble(3);
  float x2 = (float)RtnDouble(4);
  float y2 = (float)RtnDouble(5);

  return wxcDCDrawRectangle(id, x1, y1, x2, y2);
}
#endif

long wxcDCDrawRoundedRectangle(long id, float x1, float y1, float x2, float y2,
  float radius)
{
  wxDC *dc = (wxDC *)wxGetTypedObject(id, wxTYPE_DC);
  if (dc)
  {
    dc->DrawRoundedRectangle(x1, y1, x2, y2, radius);
    return 1;
  }
  else
  {
    ClipsError << "Error in dc-draw-rounded-rectangle: could not find dc " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsDCDrawRoundedRectangle()
{
  if (ArgCountCheck("clipsDCDrawRoundedRectangle", AT_LEAST, 5) == -1)
    return 0;

  long id = RtnLong(1);
  float x1 = (float)RtnDouble(2);
  float y1 = (float)RtnDouble(3);
  float x2 = (float)RtnDouble(4);
  float y2 = (float)RtnDouble(5);
  float radius = 20;
  if (RtnArgCount() > 5)
    radius = (float)RtnDouble(6);
  return wxcDCDrawRoundedRectangle(id, x1, y1, x2, y2, radius);
}
#endif

long wxcDCDrawEllipse(long id, float x, float y, float width, float height)
{
  wxDC *dc = (wxDC *)wxGetTypedObject(id, wxTYPE_DC);
  if (dc)
  {
    dc->DrawEllipse(x, y, width, height);
    return 1;
  }
  else
  {
    ClipsError << "Error in dc-draw-ellipse: could not find dc " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsDCDrawEllipse()
{
  if (ArgCountCheck("clipsDCDrawEllipse", EXACTLY, 5) == -1)
    return 0;

  long id = RtnLong(1);
  float x = (float)RtnDouble(2);
  float y = (float)RtnDouble(3);
  float width = (float)RtnDouble(4);
  float height = (float)RtnDouble(5);

  return wxcDCDrawEllipse(id, x, y, width, height);
}
#endif

#ifdef USE_CLIPS
long clipsDCDrawSpline()
{
  DATA_OBJECT arg;

  if (ArgCountCheck("clipsDCDrawSpline", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);

  wxDC *dc = (wxDC *)wxGetTypedObject(id, wxTYPE_DC);
  if (dc)
  {
    RtnUnknown(2, &arg);

    if (GetType(arg) != MULTIFIELD)
    {
      ClipsError << "Error in dc-draw-spline: 2nd argument must be a multifield of floating-point numbers.\n";
      return 0;
    }

    int start = GetpDOBegin(&arg);
    int end = GetpDOEnd(&arg);

    VOID *multifieldPtr = ClipsGetValue(arg);

    int i;
    wxList pointList;
    for (i = start; i <= end; i += 2)
    {
      double x, y;
      if ((GetMFType(multifieldPtr, i) == FLOAT) || (GetMFType(multifieldPtr, i) == INTEGER))
        x = ValueToDouble(GetMFValue(multifieldPtr, i));
      else
      {
        ClipsError << "Error in dc-draw-spline: multifield element " << i-start+1 << " must be a number.\n";
        return 0;
      }
      if (((i+1) <= end) && ((GetMFType(multifieldPtr, i+1) == FLOAT) || (GetMFType(multifieldPtr, i+1) == INTEGER)))
        y = ValueToDouble(GetMFValue(multifieldPtr, i+1));
      else
      {
        ClipsError << "Error in dc-draw-spline: multifield element " << i-start+2 << " must be a number.\n";
        return 0;
      }
#ifndef BE_PARANOID
      pointList.Append(new wxPoint((float)x, (float)y));
#else
      wxPoint *new_pt = new wxPoint ;
      new_pt->x = (float)x ;
      new_pt->y = (float)y ;
      pointList.Append(new_pt) ;
#endif
    }
    dc->DrawSpline(&pointList);
    wxNode *node = pointList.First();
    while (node)
    {
      wxPoint *point = (wxPoint *)node->Data();
      delete point;
      node = node->Next();
    }
    return 1;
  }
  else
  {
    ClipsError << "Error in dc-draw-spline: could not find dc " << id << ".\n";
    return 0;
  }
}
#endif

#ifdef USE_CLIPS
long clipsDCDrawPolygon()
{
  DATA_OBJECT arg;

  if (ArgCountCheck("clipsDCDrawPolygon", AT_LEAST, 2) == -1)
    return 0;

  float x_offset = 0.0;
  float y_offset = 0.0;
  
  long id = RtnLong(1);
  int no_args = RtnArgCount();
  if (no_args > 2)
    x_offset = (float)RtnDouble(3);
  if (no_args > 3)
    y_offset = (float)RtnDouble(4);

  wxDC *dc = (wxDC *)wxGetTypedObject(id, wxTYPE_DC);
  if (dc)
  {
    RtnUnknown(2, &arg);

    if (GetType(arg) != MULTIFIELD)
    {
      ClipsError << "Error in dc-draw-polygon: 2nd argument must be a multifield of floating-point numbers.\n";
      return 0;
    }

    int start = GetpDOBegin(&arg);
    int end = GetpDOEnd(&arg);

    VOID *multifieldPtr = ClipsGetValue(arg);

    int i;
    wxList pointList;
    for (i = start; i <= end; i += 2)
    {
      double x, y;
      if ((GetMFType(multifieldPtr, i) == FLOAT) || (GetMFType(multifieldPtr, i) == INTEGER))
        x = ValueToDouble(GetMFValue(multifieldPtr, i));
      else
      {
        ClipsError << "Error in dc-draw-polygon: multifield element " << i-start+1 << " must be a number.\n";
        return 0;
      }
      if (((i+1) <= end) && ((GetMFType(multifieldPtr, i+1) == FLOAT) || (GetMFType(multifieldPtr, i+1) == INTEGER)))
        y = ValueToDouble(GetMFValue(multifieldPtr, i+1));
      else
      {
        ClipsError << "Error in dc-draw-polygon: multifield element " << i-start+2 << " must be a number.\n";
        return 0;
      }
#ifndef BE_PARANOID
      pointList.Append(new wxPoint((float)x, (float)y));
#else
      wxPoint *new_pt = new wxPoint ;
      new_pt->x = (float)x ;
      new_pt->y = (float)y ;
      pointList.Append(new_pt) ;
#endif
    }
    dc->DrawPolygon(&pointList);
    wxNode *node = pointList.First();
    while (node)
    {
      wxPoint *point = (wxPoint *)node->Data();
      delete point;
      node = node->Next();
    }
    return 1;
  }
  else
  {
    ClipsError << "Error in dc-draw-polygon: could not find dc " << id << ".\n";
    return 0;
  }
}
#endif

#ifdef USE_CLIPS
long clipsDCDrawLines()
{
  DATA_OBJECT arg;

  if (ArgCountCheck("clipsDCDrawLines", AT_LEAST, 2) == -1)
    return 0;

  float x_offset = 0.0;
  float y_offset = 0.0;
  
  long id = RtnLong(1);
  int no_args = RtnArgCount();
  if (no_args > 2)
    x_offset = (float)RtnDouble(3);
  if (no_args > 3)
    y_offset = (float)RtnDouble(4);

  wxDC *dc = (wxDC *)wxGetTypedObject(id, wxTYPE_DC);
  if (dc)
  {
    RtnUnknown(2, &arg);

    if (GetType(arg) != MULTIFIELD)
    {
      ClipsError << "Error in dc-draw-lines: 2nd argument must be a multifield of floating-point numbers.\n";
      return 0;
    }

    int start = GetpDOBegin(&arg);
    int end = GetpDOEnd(&arg);

    VOID *multifieldPtr = ClipsGetValue(arg);

    int i;
    wxList pointList;
    for (i = start; i <= end; i += 2)
    {
      double x, y;
      if ((GetMFType(multifieldPtr, i) == FLOAT) || (GetMFType(multifieldPtr, i) == INTEGER))
        x = ValueToDouble(GetMFValue(multifieldPtr, i));
      else
      {
        ClipsError << "Error in dc-draw-lines: multifield element " << i-start+1 << " must be a number.\n";
        return 0;
      }
      if (((i+1) <= end) && ((GetMFType(multifieldPtr, i+1) == FLOAT) || (GetMFType(multifieldPtr, i+1) == INTEGER)))
        y = ValueToDouble(GetMFValue(multifieldPtr, i+1));
      else
      {
        ClipsError << "Error in dc-draw-lines: multifield element " << i-start+2 << " must be a number.\n";
        return 0;
      }
#ifndef BE_PARANOID
      pointList.Append(new wxPoint((float)x, (float)y));
#else
      wxPoint *new_pt = new wxPoint ;
      new_pt->x = (float)x ;
      new_pt->y = (float)y ;
      pointList.Append(new_pt) ;
#endif
    }
    dc->DrawPolygon(&pointList);
    wxNode *node = pointList.First();
    while (node)
    {
      wxPoint *point = (wxPoint *)node->Data();
      delete point;
      node = node->Next();
    }
    return 1;
  }
  else
  {
    ClipsError << "Error in dc-draw-lines: could not find dc " << id << ".\n";
    return 0;
  }
}
#endif

long wxcDCDrawText(long id, char *text, float x1, float y1)
{
  if (!text)
    return 0;

  wxDC *dc = (wxDC *)wxGetTypedObject(id, wxTYPE_DC);
  if (dc)
  {
    dc->DrawText(text, x1, y1);
    return 1;
  }
  else
  {
    ClipsError << "Error in dc-draw-text: could not find dc " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsDCDrawText()
{
  if (ArgCountCheck("clipsDCDrawText", EXACTLY, 4) == -1)
    return 0;

  long id = RtnLong(1);
  char *text = RtnLexeme(2);
  float x1 = (float)RtnDouble(3);
  float y1 = (float)RtnDouble(4);

  return wxcDCDrawText(id, text, x1, y1);
}
#endif

long wxcDCSetClippingRegion(long id, float x1, float y1, float x2, float y2)
{
  wxDC *dc = (wxDC *)wxGetTypedObject(id, wxTYPE_DC);
  if (dc)
  {
    dc->SetClippingRegion(x1, y1, x2, y2);
    return 1;
  }
  else
  {
    ClipsError << "Error in dc-set-clipping-region: could not find dc " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsDCSetClippingRegion()
{
  if (ArgCountCheck("clipsDCSetClippingRegion", EXACTLY, 5) == -1)
    return 0;

  long id = RtnLong(1);
  float x1 = (float)RtnDouble(2);
  float y1 = (float)RtnDouble(3);
  float x2 = (float)RtnDouble(4);
  float y2 = (float)RtnDouble(5);

  return wxcDCSetClippingRegion(id, x1, y1, x2, y2);
}
#endif

long wxcDCDestroyClippingRegion(long id)
{
  wxDC *dc = (wxDC *)wxGetTypedObject(id, wxTYPE_DC);
  if (dc)
  {
    dc->DestroyClippingRegion();
    return 1;
  }
  else
  {
    ClipsError << "Error in dc-destroy-clipping-region: could not find dc " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsDCDestroyClippingRegion()
{
  if (ArgCountCheck("clipsDCDestroyClippingRegion", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcDCDestroyClippingRegion(id);
}
#endif

long wxcDCClear(long id)
{
  wxDC *dc = (wxDC *)wxGetTypedObject(id, wxTYPE_DC);
  if (dc)
  {
    dc->Clear();
    return 1;
  }
  else
  {
    ClipsError << "Error in dc-clear: could not find dc " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsDCClear()
{
  if (ArgCountCheck("clipsDCClear", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcDCClear(id);
}
#endif

// Create a memory device context
long wxcMemoryDCCreate(void)
{
  wxMemoryDC *dc = new wxMemoryDC;
  wxStoreThingInTable((long)dc, dc, wxTYPE_DC_MEMORY);
  return (long)dc;
}

#ifdef USE_CLIPS
long clipsMemoryDCCreate()
{
  if (ArgCountCheck("clipsMemoryDCCreate", EXACTLY, 0) == -1)
    return 0;

  return wxcMemoryDCCreate();
}
#endif

// Delete a memory device context
long wxcMemoryDCDelete(long id)
{
  wxMemoryDC *dc = (wxMemoryDC *)wxGetTypedObject(id, wxTYPE_DC_MEMORY);
  if (dc)
  {
    wxDeleteThingInTable(id);
    delete dc;
    return 1;
  }
  else
  {
    ClipsError << "Error in memory-dc-delete: could not find memory device context " << id << ".\n";
  }
  return 0;
}

#ifdef USE_CLIPS
long clipsMemoryDCDelete()
{
  if (ArgCountCheck("clipsMemoryDCDelete", EXACTLY, 1) == -1)
    return 0;

  return wxcMemoryDCDelete(RtnLong(1));
}
#endif

// Select a bitmap into a memory device context
long wxcMemoryDCSelectObject(long id, long bitmapId)
{
  wxMemoryDC *dc = (wxMemoryDC *)wxGetTypedObject(id, wxTYPE_DC_MEMORY);
  if (dc)
  {
    wxBitmap *bitmap = (wxBitmap *)wxGetTypedObject(bitmapId, wxTYPE_BITMAP);
    if (bitmap)
    {
      dc->SelectObject(bitmap);
      return 1;
    }
    else
      ClipsError << "Error in memory-dc-select-object: could not find bitmap " << bitmapId << ".\n";
  }
  else
  {
    ClipsError << "Error in memory-dc-select-object: could not find memory device context " << id << ".\n";
  }
  return 0;
}

#ifdef USE_CLIPS
long clipsMemoryDCSelectObject()
{
  if (ArgCountCheck("clipsMemoryDCSelectObject", EXACTLY, 2) == -1)
    return 0;

  return wxcMemoryDCSelectObject(RtnLong(1), RtnLong(2));
}
#endif

/*
 * Event accessors
 */

long wxcMouseEventIsButton(long id)
{
  wxMouseEvent *event = (wxMouseEvent *)wxGetTypedObject(id, wxTYPE_MOUSE_EVENT);
  if (event)
  {
    return event->IsButton();
  }
  else
  {
    ClipsError << "Error in mouse-event-is-button: could not find event " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMouseEventIsButton()
{
  if (ArgCountCheck("clipsMouseEventIsButton", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcMouseEventIsButton(id);
}
#endif

long wxcMouseEventButtonDown(long id)
{
  wxMouseEvent *event = (wxMouseEvent *)wxGetTypedObject(id, wxTYPE_MOUSE_EVENT);
  if (event)
  {
    return event->ButtonDown();
  }
  else
  {
    ClipsError << "Error in mouse-event-button-down: could not find event " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMouseEventButtonDown()
{
  if (ArgCountCheck("clipsMouseEventButtonDown", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcMouseEventButtonDown(id);
}
#endif

long wxcMouseEventControlDown(long id)
{
  wxMouseEvent *event = (wxMouseEvent *)wxGetTypedObject(id, wxTYPE_MOUSE_EVENT);
  if (event)
  {
    return event->ControlDown();
  }
  else
  {
    ClipsError << "Error in mouse-event-control-down: could not find event " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMouseEventControlDown()
{
  if (ArgCountCheck("clipsMouseEventControlDown", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcMouseEventControlDown(id);
}
#endif

long wxcMouseEventShiftDown(long id)
{
  wxMouseEvent *event = (wxMouseEvent *)wxGetTypedObject(id, wxTYPE_MOUSE_EVENT);
  if (event)
  {
    return event->ShiftDown();
  }
  else
  {
    ClipsError << "Error in mouse-event-shift-down: could not find event " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMouseEventShiftDown()
{
  if (ArgCountCheck("clipsMouseEventShiftDown", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcMouseEventShiftDown(id);
}
#endif

long wxcMouseEventButton(long id, long butt)
{
  wxMouseEvent *event = (wxMouseEvent *)wxGetTypedObject(id, wxTYPE_MOUSE_EVENT);
  if (event)
  {
    return event->Button((int)butt);
  }
  else
  {
    ClipsError << "Error in mouse-event-button: could not find event " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMouseEventButton()
{
  if (ArgCountCheck("clipsMouseEventButton", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  long butt = RtnLong(2);
  return wxcMouseEventButton(id, butt);
}
#endif

long wxcMouseEventLeftDown(long id)
{
  wxMouseEvent *event = (wxMouseEvent *)wxGetTypedObject(id, wxTYPE_MOUSE_EVENT);
  if (event)
  {
    return event->LeftDown();
  }
  else
  {
    ClipsError << "Error in mouse-event-left-down: could not find event " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMouseEventLeftDown()
{
  if (ArgCountCheck("clipsMouseEventLeftDown", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcMouseEventLeftDown(id);
}
#endif

long wxcMouseEventMiddleDown(long id)
{
  wxMouseEvent *event = (wxMouseEvent *)wxGetTypedObject(id, wxTYPE_MOUSE_EVENT);
  if (event)
  {
    return event->MiddleDown();
  }
  else
  {
    ClipsError << "Error in mouse-event-middle-down: could not find event " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMouseEventMiddleDown()
{
  if (ArgCountCheck("clipsMouseEventMiddleDown", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcMouseEventMiddleDown(id);
}
#endif

long wxcMouseEventRightDown(long id)
{
  wxMouseEvent *event = (wxMouseEvent *)wxGetTypedObject(id, wxTYPE_MOUSE_EVENT);
  if (event)
  {
    return event->RightDown();
  }
  else
  {
    ClipsError << "Error in mouse-event-right-down: could not find event " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMouseEventRightDown()
{
  if (ArgCountCheck("clipsMouseEventRightDown", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcMouseEventRightDown(id);
}
#endif

long wxcMouseEventLeftUp(long id)
{
  wxMouseEvent *event = (wxMouseEvent *)wxGetTypedObject(id, wxTYPE_MOUSE_EVENT);
  if (event)
  {
    return event->LeftUp();
  }
  else
  {
    ClipsError << "Error in mouse-event-left-up: could not find event " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMouseEventLeftUp()
{
  if (ArgCountCheck("clipsMouseEventLeftUp", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcMouseEventLeftUp(id);
}
#endif

long wxcMouseEventMiddleUp(long id)
{
  wxMouseEvent *event = (wxMouseEvent *)wxGetTypedObject(id, wxTYPE_MOUSE_EVENT);
  if (event)
  {
    return event->MiddleUp();
  }
  else
  {
    ClipsError << "Error in mouse-event-middle-up: could not find event " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMouseEventMiddleUp()
{
  if (ArgCountCheck("clipsMouseEventMiddleUp", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcMouseEventMiddleUp(id);
}
#endif

long wxcMouseEventRightUp(long id)
{
  wxMouseEvent *event = (wxMouseEvent *)wxGetTypedObject(id, wxTYPE_MOUSE_EVENT);
  if (event)
  {
    return event->RightUp();
  }
  else
  {
    ClipsError << "Error in mouse-event-right-up: could not find event " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMouseEventRightUp()
{
  if (ArgCountCheck("clipsMouseEventRightUp", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcMouseEventRightUp(id);
}
#endif

long wxcMouseEventDragging(long id)
{
  wxMouseEvent *event = (wxMouseEvent *)wxGetTypedObject(id, wxTYPE_MOUSE_EVENT);
  if (event)
  {
    return event->Dragging();
  }
  else
  {
    ClipsError << "Error in mouse-event-dragging: could not find event " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMouseEventDragging()
{
  if (ArgCountCheck("clipsMouseEventDragging", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcMouseEventDragging(id);
}
#endif

double wxcMouseEventPositionX(long id)
{
  wxMouseEvent *event = (wxMouseEvent *)wxGetTypedObject(id, wxTYPE_MOUSE_EVENT);
  if (event)
  {
    float x, y;
    event->Position(&x, &y);
    return (double)x;
  }
  else
  {
    ClipsError << "Error in mouse-event-position-x: could not find event " << id << ".\n";
    return 0.0;
  }
}

#ifdef USE_CLIPS
double clipsMouseEventPositionX()
{
  if (ArgCountCheck("clipsMouseEventPositionX", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcMouseEventPositionX(id);
}
#endif

double wxcMouseEventPositionY(long id)
{
  wxMouseEvent *event = (wxMouseEvent *)wxGetTypedObject(id, wxTYPE_MOUSE_EVENT);
  if (event)
  {
    float x, y;
    event->Position(&x, &y);
    return (double)y;
  }
  else
  {
    ClipsError << "Error in mouse-event-position-y: could not find event " << id << ".\n";
    return 0.0;
  }
}

#ifdef USE_CLIPS
double clipsMouseEventPositionY()
{
  if (ArgCountCheck("clipsMouseEventPositionY", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcMouseEventPositionY(id);
}
#endif

// Create a new text window
long wxcTextWindowCreate(long parent_id, long x, long y, long width, long height,
  char *styleString)
{
  wxFrame *frame = NULL;

  if (parent_id != 0)
  {
    frame = (wxFrame *)wxGetTypedObject(parent_id, wxTYPE_FRAME);
    if (!frame)
    {
      ClipsError << "Error in text-window-create: could not find parent frame " << parent_id << ".\n";
      return 0;
    }
  }
  long style = wxClipsParseBitList(styleString);

  wxClipsTextWindow *text_window =
    new wxClipsTextWindow(frame, (int)x, (int)y, (int)width, (int)height, style);
  wxStoreThingInTable((long)text_window, text_window, wxTYPE_TEXT_WINDOW);
  return (long)text_window;
}

#ifdef USE_CLIPS
long clipsTextWindowCreate()
{
  if (ArgCountCheck("clipsTextWindowCreate", AT_LEAST, 1) == -1)
    return 0;

  int no_args = RtnArgCount();

  long parent_id = RtnLong(1);

  // Set up defaults
  int x = -1;
  int y = -1;
  int width = -1;
  int height = -1;
  char *style = "";

  if (no_args > 1)
    x = (int) RtnLong(2);

  if (no_args > 2)
    y = (int) RtnLong(3);

  if (no_args > 3)
    width = (int) RtnLong(4);

  if (no_args > 4)
    height = (int) RtnLong(5);

  if (no_args > 5)
    style = RtnLexeme(6);

  return wxcTextWindowCreate(parent_id, x, y, width, height, style);
}
#endif

// Load a file
long wxcTextWindowLoadFile(long id, char *filename)
{
  if (!filename)
    return 0;

  wxTextWindow *win = (wxTextWindow *)wxGetTypedObject(id, wxTYPE_TEXT_WINDOW);
  if (win)
  {
    return (long)win->LoadFile(filename);
  }
  else
  {
    ClipsError << "Error in text-window-load-file: could not find text window " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsTextWindowLoadFile()
{
  if (ArgCountCheck("clipsTextWindowLoadFile", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  char *filename = RtnLexeme(2);
  return wxcTextWindowLoadFile(id, filename);
}
#endif

// Save a file
long wxcTextWindowSaveFile(long id, char *filename)
{
  if (!filename)
    return 0;

  wxTextWindow *win = (wxTextWindow *)wxGetTypedObject(id, wxTYPE_TEXT_WINDOW);
  if (win)
  {
    return (long)win->SaveFile(filename);
  }
  else
  {
    ClipsError << "Error in text-window-save-file: could not find text window " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsTextWindowSaveFile()
{
  if (ArgCountCheck("clipsTextWindowSaveFile", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  char *filename = RtnLexeme(2);
  return wxcTextWindowSaveFile(id, filename);
}
#endif

// Write text to the window
void wxcTextWindowWrite(long id, long val)
{
  wxTextWindow *win = (wxTextWindow *)wxGetTypedObject(id, wxTYPE_TEXT_WINDOW);
  if (win)
    *win << val;
  else
  {
    ClipsError << "Error in text-window-write: could not find text window " << id << ".\n";
    return;
  }
}

void wxcTextWindowWrite(long id, char *val)
{
  wxTextWindow *win = (wxTextWindow *)wxGetTypedObject(id, wxTYPE_TEXT_WINDOW);
  if (win)
    *win << val;
  else
  {
    ClipsError << "Error in text-window-write: could not find text window " << id << ".\n";
    return;
  }
}

void wxcTextWindowWrite(long id, double val)
{
  wxTextWindow *win = (wxTextWindow *)wxGetTypedObject(id, wxTYPE_TEXT_WINDOW);
  if (win)
    *win << val;
  else
  {
    ClipsError << "Error in text-window-write: could not find text window " << id << ".\n";
    return;
  }
}

#ifdef USE_CLIPS
VOID clipsTextWindowWrite()
{
  if (ArgCountCheck("clipsTextWindowWrite", AT_LEAST, 1) == -1)
    return;

  long id = RtnLong(1);

  DATA_OBJECT temp;

  for (int i = 2; i <= RtnArgCount(); i++)
  {
    RtnUnknown(i, &temp);

    switch (GetType(temp))
    {
      case INTEGER:
      {
        wxcTextWindowWrite(id, (long)DOToLong(temp));
        break;
      }
      case STRING:
      case SYMBOL:
      {
        wxcTextWindowWrite(id, DOToString(temp));
        break;
      }
      case FLOAT:
      {
        wxcTextWindowWrite(id, DOToDouble(temp));
        break;
      }
      default:
      {
        ClipsError << "Error in text-window-write: unrecognized type.\n";
        break;
      }
    }
  }
}
#endif

// Return TRUE if modified
long wxcTextWindowModified(long id)
{
  wxTextWindow *win = (wxTextWindow *)wxGetTypedObject(id, wxTYPE_TEXT_WINDOW);
  if (win)
  {
    return (long)win->Modified();
  }
  else
  {
    ClipsError << "Error in text-window-modified: could not find text window " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsTextWindowModified()
{
  if (ArgCountCheck("clipsTextWindowModifed", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcTextWindowModified(id);
}
#endif

long wxcTextWindowClear(long id)
{
  wxTextWindow *win = (wxTextWindow *)wxGetTypedObject(id, wxTYPE_TEXT_WINDOW);
  if (win)
  {
    win->Clear();
    return 1;
  }
  else
  {
    ClipsError << "Error in text-window-clear: could not find text window " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsTextWindowClear()
{
  if (ArgCountCheck("clipsTextWindowClear", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcTextWindowClear(id);
}
#endif

// Return TRUE if modified
void wxcTextWindowDiscardEdits(long id)
{
  wxTextWindow *win = (wxTextWindow *)wxGetTypedObject(id, wxTYPE_TEXT_WINDOW);
  if (win)
  {
    win->DiscardEdits();
    return;
  }
  else
  {
    ClipsError << "Error in text-window-discard-edits: could not find text window " << id << ".\n";
    return;
  }
}

#ifdef USE_CLIPS
VOID clipsTextWindowDiscardEdits()
{
  if (ArgCountCheck("clipsTextWindowDiscardEdits", EXACTLY, 1) == -1)
    return;

  long id = RtnLong(1);
  wxcTextWindowDiscardEdits(id);
}
#endif

// Create a button
long wxcButtonCreate(long parent_id, wxClipsFunction func, char *label,
  long x, long y, long width, long height, char *styleString)
{
#ifdef USE_CLIPS
  if (func && (strlen(func) == 0))
    func = NULL;
#endif

  if (!label)
    return -1;

  wxPanel *panel = (wxPanel *)wxGetTypedObject(parent_id, wxTYPE_PANEL);
  if (panel)
  {
    long style = wxClipsParseBitList(styleString);
    wxClipsButton *button = new wxClipsButton(panel, func, (wxFunction)wxClipsButtonCallback,
     label, (int)x, (int)y, (int)width, (int)height, style);
    wxStoreThingInTable((long)button, button, wxTYPE_BUTTON);
    return (long)button;
  }
  else
  {
    ClipsError << "Error in button-create: could not find parent panel " << parent_id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsButtonCreate()
{
  if (ArgCountCheck("clipsButtonCreate", AT_LEAST, 3) == -1)
    return 0;

  int no_args = RtnArgCount();

  long parent_id = RtnLong(1);
  char *func = RtnLexeme(2);

  if (func && (strlen(func) == 0))
    func = NULL;

  char *label = RtnLexeme(3);
  if (!label)
    return -1;

  // Set up defaults
  int x = -1;
  int y = -1;
  int width = -1;
  int height = -1;
  char *style = "";

  if (no_args > 3)
    x = (int) RtnLong(4);

  if (no_args > 4)
    y = (int) RtnLong(5);

  if (no_args > 5)
    width = (int) RtnLong(6);

  if (no_args > 6)
    height = (int) RtnLong(7);

  if (no_args > 7)
    style = RtnLexeme(8);

  return wxcButtonCreate(parent_id, func, label, x, y, width, height, style);
}
#endif

// Create a button from a bitmap
long wxcButtonCreateFromBitmap(long parent_id, wxClipsFunction func, long bitmapId,
  long x, long y, long width, long height, char *styleString)
{
#if (defined(wx_msw) && !defined(FAFA_LIB))
  ClipsError << "Error in button-create-from-bitmap:\nBitmap buttons not supported in this version.\n";
  return -1;
#else
  if (func && (strlen(func) == 0))
    func = NULL;
#endif
  wxBitmap *bitmap = (wxBitmap *)wxGetTypedObject(bitmapId, wxTYPE_BITMAP);

  if (!bitmap)
  {
    ClipsError << "Error in button-create-from-bitmap: could not find bitmap " << bitmapId << ".\n";
    return -1;
  }

  wxPanel *panel = (wxPanel *)wxGetTypedObject(parent_id, wxTYPE_PANEL);
  if (panel)
  {
    long style = wxClipsParseBitList(styleString);
    wxClipsButton *button = new wxClipsButton(panel, func, (wxFunction)wxClipsButtonCallback,
     bitmap, (int)x, (int)y, (int)width, (int)height, style);
    wxStoreThingInTable((long)button, button, wxTYPE_BUTTON);
    return (long)button;
  }
  else
  {
    ClipsError << "Error in button-create-from-bitmap: could not find parent panel " << parent_id << ".\n";
    return -1;
  }
}

#ifdef USE_CLIPS
long clipsButtonCreateFromBitmap()
{
  if (ArgCountCheck("clipsButtonCreateFromBitmap", AT_LEAST, 3) == -1)
    return 0;

  int no_args = RtnArgCount();

  long parent_id = RtnLong(1);
  char *func = RtnLexeme(2);

  if (func && (strlen(func) == 0))
    func = NULL;

  long bitmapId = RtnLong(3);

  // Set up defaults
  int x = -1;
  int y = -1;
  int width = -1;
  int height = -1;
  char *style = "";

  if (no_args > 3)
    x = (int) RtnLong(4);

  if (no_args > 4)
    y = (int) RtnLong(5);

  if (no_args > 5)
    width = (int) RtnLong(6);

  if (no_args > 6)
    height = (int) RtnLong(7);

  if (no_args > 7)
    style = RtnLexeme(8);

  return wxcButtonCreateFromBitmap(parent_id, func, bitmapId, x, y, width, height, style);
}
#endif

// Create a text widget
long wxcTextCreate(long parent_id, wxClipsFunction func, char *label, char *value,
  int x, int y, int width, int height, char *styleString)
{
#ifdef USE_CLIPS
  if (func && (strlen(func) == 0))
    func = NULL;
#endif

  if (!label)
    return -1;
  if (!value)
    return -1;

  wxPanel *panel = (wxPanel *)wxGetTypedObject(parent_id, wxTYPE_PANEL);
  if (panel)
  {
    long style = wxClipsParseBitList(styleString);
    wxClipsText *text = new wxClipsText(panel, func, (wxFunction)wxClipsTextCallback,
                   label, value, x, y, width, height, style);
    wxStoreThingInTable((long)text, text, wxTYPE_TEXT);
    return (long)text;
  }
  else
  {
    ClipsError << "Error in text-create: could not find parent panel " << parent_id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsTextCreate()
{
  if (ArgCountCheck("clipsTextCreate", AT_LEAST, 3) == -1)
    return 0;

  int no_args = RtnArgCount();

  long parent_id = RtnLong(1);
  char *func = RtnLexeme(2);

  if (func && (strlen(func) == 0))
    func = NULL;

  char *label = RtnLexeme(3);
  if (!label)
    return -1;

  // Set up defaults
  int x = -1;
  int y = -1;
  int width = -1;
  int height = -1;
  char *value = "";
  char *style = "";

  if (no_args > 3)
    value = RtnLexeme(4);
  if (!value)
    return -1;

  if (no_args > 4)
    x = (int) RtnLong(5);

  if (no_args > 5)
    y = (int) RtnLong(6);

  if (no_args > 6)
    width = (int) RtnLong(7);

  if (no_args > 7)
    height = (int) RtnLong(8);

  if (no_args > 8)
    style = RtnLexeme(9);
    
  return wxcTextCreate(parent_id, func, label, value, x, y, width, height, style);
}
#endif

// Set value
long wxcTextSetValue(long id, char *val)
{
  if (!val)
    return 0;

  wxText *item = (wxText *)wxGetTypedObject(id, wxTYPE_TEXT);
  if (item)
  {
    item->SetValue(val);
    return 1;
  }
  else
  {
    ClipsError << "Error in text-set-value: could not find text item " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsTextSetValue()
{
  if (ArgCountCheck("clipsTextSetValue", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  char *val = RtnLexeme(2);
  return wxcTextSetValue(id, val);
}
#endif

// Get value
char *wxcTextGetValue(long id)
{
  wxText *item = (wxText *)wxGetTypedObject(id, wxTYPE_TEXT);
  if (item)
  {
    char *val = item->GetValue();
    return val;
  }
  else
  {
    ClipsError << "Error in text-get-value: could not find text item " << id << ".\n";
    return NULL;
  }
}

#ifdef USE_CLIPS
char *clipsTextGetValue()
{
  if (ArgCountCheck("clipsTextGetValue", EXACTLY, 1) == -1)
    return (char *)AddSymbol("");

  long id = RtnLong(1);
  char *ans = wxcTextGetValue(id);
  if (ans)
    return (char *)AddSymbol(ans);
  else
    return (char *)AddSymbol("");
}
#endif

// Create a multi-line text widget
long wxcMultiTextCreate(long parent_id, wxClipsFunction func, char *label, char *value,
  long x, long y, long width, long height, char *styleString)
{
#ifdef USE_CLIPS
  if (func && (strlen(func) == 0))
    func = NULL;
#endif
  if (!label)
    return -1;
  if (!value)
    return -1;
  wxPanel *panel = (wxPanel *)wxGetTypedObject(parent_id, wxTYPE_PANEL);
  if (panel)
  {
    long style = wxClipsParseBitList(styleString);
    wxClipsMultiText *text = new wxClipsMultiText(panel, func, (wxFunction)wxClipsMultiTextCallback, label, value,
      (int)x, (int)y, (int)width, (int)height, style);
    wxStoreThingInTable((long)text, text, wxTYPE_MULTI_TEXT);
    return (long)text;
  }
  else
  {
    ClipsError << "Error in text-create: could not find parent panel " << parent_id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMultiTextCreate()
{
  if (ArgCountCheck("clipsMultiTextCreate", AT_LEAST, 3) == -1)
    return 0;

  int no_args = RtnArgCount();

  long parent_id = RtnLong(1);
  char *func = RtnLexeme(2);
  char *label = RtnLexeme(3);
  // Set up defaults
  int x = -1;
  int y = -1;
  int width = -1;
  int height = -1;
  char *value = "";
  char *style = "";

  if (no_args > 3)
    value = RtnLexeme(4);
  if (!value)
    return -1;

  if (no_args > 4)
    x = (int) RtnLong(5);

  if (no_args > 5)
    y = (int) RtnLong(6);

  if (no_args > 6)
    width = (int) RtnLong(7);

  if (no_args > 7)
    height = (int) RtnLong(8);

  if (no_args > 8)
    style = RtnLexeme(9);

  return wxcMultiTextCreate(parent_id, func, label, value, x, y, width, height, style);
}
#endif

// Set value
long wxcMultiTextSetValue(long id, char *val)
{
  if (!val)
    return 0;

  wxMultiText *item = (wxMultiText *)wxGetTypedObject(id, wxTYPE_MULTI_TEXT);
  if (item)
  {
    item->SetValue(val);
    return 1;
  }
  else
  {
    ClipsError << "Error in multi-text-set-value: could not find text item " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMultiTextSetValue()
{
  if (ArgCountCheck("clipsMultiTextSetValue", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  char *val = RtnLexeme(2);
  return wxcMultiTextSetValue(id, val);
}
#endif

// Get value
char *wxcMultiTextGetValue(long id)
{
  wxMultiText *item = (wxMultiText *)wxGetTypedObject(id, wxTYPE_MULTI_TEXT);
  if (item)
  {
    char *val = item->GetValue();
    return val;
  }
  else
  {
    ClipsError << "Error in multi-text-get-value: could not find text item " << id << ".\n";
    return NULL;
  }
}

#ifdef USE_CLIPS
char *clipsMultiTextGetValue()
{
  if (ArgCountCheck("clipsMultiTextGetValue", EXACTLY, 1) == -1)
    return (char *)AddSymbol("");

  long id = RtnLong(1);
  char *ans = wxcMultiTextGetValue(id);
  if (ans)
    return (char *)AddSymbol(ans);
  else
    return (char *)AddSymbol("");
}
#endif

// Create a check box widget
long wxcCheckBoxCreate(long parent_id, wxClipsFunction func, char *label,
  long x, long y, long width, long height, char *styleString)
{
#ifdef USE_CLIPS
  if (func && (strlen(func) == 0))
    func = NULL;
#endif
  if (!label)
    return -1;

  wxPanel *panel = (wxPanel *)wxGetTypedObject(parent_id, wxTYPE_PANEL);
  if (panel)
  {
    long style = wxClipsParseBitList(styleString);
    wxClipsCheckBox *box = new wxClipsCheckBox(panel, func, (wxFunction)wxClipsCheckBoxCallback, label,
      (int)(int)x, (int)y, (int)width, (int)height, style);
    wxStoreThingInTable((long)box, box, wxTYPE_CHECK_BOX);
    return (long)box;
  }
  else
  {
    ClipsError << "Error in check-box-create: could not find parent panel " << parent_id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsCheckBoxCreate()
{
  if (ArgCountCheck("clipsCheckBoxCreate", AT_LEAST, 2) == -1)
    return 0;

  int no_args = RtnArgCount();

  long parent_id = RtnLong(1);
  char *func = RtnLexeme(2);
  char *label = RtnLexeme(3);
  // Set up defaults
  int x = -1;
  int y = -1;
  int width = -1;
  int height = -1;
  char *style = "";

  if (no_args > 3)
    x = (int) RtnLong(4);

  if (no_args > 4)
    y = (int) RtnLong(5);

  if (no_args > 5)
    width = (int) RtnLong(6);

  if (no_args > 6)
    height = (int) RtnLong(7);

  if (no_args > 7)
    style = RtnLexeme(8);

  return wxcCheckBoxCreate(parent_id, func, label, x, y, width, height, style);
}
#endif

// Set value
long wxcCheckBoxSetValue(long id, long val)
{
  wxCheckBox *item = (wxCheckBox *)wxGetTypedObject(id, wxTYPE_CHECK_BOX);
  if (item)
  {
    item->SetValue((Bool)val);
    return 1;
  }
  else
  {
    ClipsError << "Error in check-box-set-value: could not find check box " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsCheckBoxSetValue()
{
  if (ArgCountCheck("clipsCheckBoxSetValue", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  Bool val = (Bool)RtnLong(2);
  return wxcCheckBoxSetValue(id, val);
}
#endif

// Get value
long wxcCheckBoxGetValue(long id)
{
  wxCheckBox *item = (wxCheckBox *)wxGetTypedObject(id, wxTYPE_CHECK_BOX);
  if (item)
  {
    Bool val = item->GetValue();
    return val;
  }
  else
  {
    ClipsError << "Error in check-box-get-value: could not find check box " << id << ".\n";
    return -1;
  }
}

#ifdef USE_CLIPS
long clipsCheckBoxGetValue()
{
  if (ArgCountCheck("clipsCheckBoxGetValue", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcCheckBoxGetValue(id);
}
#endif

// Create a slider widget
long wxcSliderCreate(long parent_id, wxClipsFunction func, char *label,
  long value, long min_value, long max_value, long width, long x, long y, char *styleString)
{
#ifdef USE_CLIPS
  if (func && (strlen(func) == 0))
    func = NULL;
#endif
  if (!label)
    return -1;

  wxPanel *panel = (wxPanel *)wxGetTypedObject(parent_id, wxTYPE_PANEL);
  if (panel)
  {
    long style = wxClipsParseBitList(styleString);
    wxClipsSlider *slider = new wxClipsSlider(panel, func, (wxFunction)wxClipsSliderCallback, label,
    (int)value, (int)min_value, (int)max_value, (int)width, (int)x, (int)y, style);
    wxStoreThingInTable((long)slider, slider, wxTYPE_SLIDER);
    return (long)slider;
  }
  else
  {
    ClipsError << "Error in slider-create: could not find parent panel " << parent_id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsSliderCreate()
{
  if (ArgCountCheck("clipsSliderCreate", AT_LEAST, 7) == -1)
    return 0;

  int no_args = RtnArgCount();

  long parent_id = RtnLong(1);
  char *func = RtnLexeme(2);
  char *label = RtnLexeme(3);
  int value = (int)RtnLong(4);
  int min_value = (int)RtnLong(5);
  int max_value = (int)RtnLong(6);
  int width = (int)RtnLong(7);
  char *style = "";

  // Set up defaults
  int x = -1;
  int y = -1;

  if (no_args > 7)
    x = (int) RtnLong(8);

  if (no_args > 8)
    y = (int) RtnLong(9);

  if (no_args > 9)
    style = RtnLexeme(10);

  return wxcSliderCreate(parent_id, func, label, value, min_value, max_value,
    width, x, y, style);
}
#endif

// Set value
long wxcSliderSetValue(long id, long val)
{
  wxSlider *item = (wxSlider *)wxGetTypedObject(id, wxTYPE_SLIDER);
  if (item)
  {
    item->SetValue((int)val);
    return 1;
  }
  else
  {
    ClipsError << "Error in slider-set-value: could not find slider " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsSliderSetValue()
{
  if (ArgCountCheck("clipsSliderSetValue", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  long val = RtnLong(2);
  return wxcSliderSetValue(id, val);
}
#endif

// Get value
long wxcSliderGetValue(long id)
{
  wxSlider *item = (wxSlider *)wxGetTypedObject(id, wxTYPE_SLIDER);
  if (item)
  {
    int val = item->GetValue();
    return (long)val;
  }
  else
  {
    ClipsError << "Error in slider-get-value: could not find slider " << id << ".\n";
    return -1;
  }
}

#ifdef USE_CLIPS
long clipsSliderGetValue()
{
  if (ArgCountCheck("clipsSliderGetValue", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcSliderGetValue(id);
}
#endif

// Create a message widget
long wxcMessageCreate(long parent_id, char *label, long x, long y, char *styleString)
{
  if (!label)
    return -1;
  wxPanel *panel = (wxPanel *)wxGetTypedObject(parent_id, wxTYPE_PANEL);
  if (panel)
  {
    long style = wxClipsParseBitList(styleString);
    wxClipsMessage *mess = new wxClipsMessage(panel, label, (int)x, (int)y, style);
    wxStoreThingInTable((long)mess, mess, wxTYPE_MESSAGE);
    return (long)mess;
  }
  else
  {
    ClipsError << "Error in message-create: could not find parent panel " << parent_id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMessageCreate()
{
  if (ArgCountCheck("clipsMessageCreate", AT_LEAST, 2) == -1)
    return 0;

  int no_args = RtnArgCount();

  long parent_id = RtnLong(1);
  char *label = RtnLexeme(2);

  // Set up defaults
  int x = -1;
  int y = -1;
  char *style = "";

  if (no_args > 2)
    x = (int) RtnLong(3);

  if (no_args > 3)
    y = (int) RtnLong(4);

  if (no_args > 4)
    style = RtnLexeme(5);

  return wxcMessageCreate(parent_id, label, x, y, style);
}
#endif

// Create a listbox widget
long wxcListBoxCreate(long parent_id, wxClipsFunction func, char *label, long multiple,
  long x, long y, long width, long height, char *styleString)
{
#ifdef USE_CLIPS
  if (func && (strlen(func) == 0))
    func = NULL;
#endif
  if (!label)
    return -1;

  wxPanel *panel = (wxPanel *)wxGetTypedObject(parent_id, wxTYPE_PANEL);
  if (panel)
  {
    long style = wxClipsParseBitList(styleString);
    wxClipsListBox *item = new wxClipsListBox(panel, func, (wxFunction)wxClipsListBoxCallback, label,
     (Bool)multiple, (int)x, (int)y, (int)width, (int)height, style);
    wxStoreThingInTable((long)item, item, wxTYPE_LIST_BOX);
    return (long)item;
  }
  else
  {
    ClipsError << "Error in list-box-create: could not find parent panel " << parent_id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsListBoxCreate()
{
  if (ArgCountCheck("clipsListBoxCreate", AT_LEAST, 3) == -1)
    return 0;

  int no_args = RtnArgCount();

  long parent_id = RtnLong(1);
  char *func = RtnLexeme(2);
  char *label = RtnLexeme(3);
  // Set up defaults
  Bool multiple = FALSE;
  int x = -1;
  int y = -1;
  int width = -1;
  int height = -1;
  char *style = "";

  if (no_args > 3)
    multiple = (Bool)RtnLong(4);

  if (no_args > 4)
    x = (int) RtnLong(5);

  if (no_args > 5)
    y = (int) RtnLong(6);

  if (no_args > 6)
    width = (int) RtnLong(7);

  if (no_args > 7)
    height = (int) RtnLong(8);

  if (no_args > 8)
    style = RtnLexeme(9);

  return wxcListBoxCreate(parent_id, func, label, multiple, x, y, width, height, style);
}
#endif

// Append a string (and optional client data)
long wxcListBoxAppend(long id, char *item_string, char *client_data)
{
  wxListBox *item = (wxListBox *)wxGetTypedObject(id, wxTYPE_LIST_BOX);
  if (item)
  {
    item->Append(item_string, client_data);
    return 1;
  }
  else
  {
    ClipsError << "Error in list-box-append: could not find list box " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsListBoxAppend()
{
  if (ArgCountCheck("clipsListBoxAppend", AT_LEAST, 2) == -1)
    return 0;

  int no_args = RtnArgCount();

  long id = RtnLong(1);
  char *item_string = RtnLexeme(2);
  char *client_data = NULL;
  if (no_args > 2)
    client_data = RtnLexeme(3);

  return wxcListBoxAppend(id, item_string, client_data);
}
#endif

// Find a string and return ID
long wxcListBoxFindString(long id, char *item_string)
{
  if (!item_string)
    return -1;

  wxListBox *item = (wxListBox *)wxGetTypedObject(id, wxTYPE_LIST_BOX);
  if (item)
  {
    return item->FindString(item_string);
  }
  else
  {
    ClipsError << "Error in list-box-find-string: could not find list box " << id << ".\n";
    return -1;
  }
}

#ifdef USE_CLIPS
long clipsListBoxFindString()
{
  if (ArgCountCheck("clipsListBoxFindString", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  char *item_string = RtnLexeme(2);

  return wxcListBoxFindString(id, item_string);
}
#endif

// Clear
long wxcListBoxClear(long id)
{
  wxListBox *item = (wxListBox *)wxGetTypedObject(id, wxTYPE_LIST_BOX);
  if (item)
  {
    item->Clear();
    return 1;
  }
  else
  {
    ClipsError << "Error in list-box-clear: could not find list box " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsListBoxClear()
{
  if (ArgCountCheck("clipsListBoxClear", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcListBoxClear(id);
}
#endif

// Set the current selection
long wxcListBoxSetSelection(long id, long sel)
{
  wxListBox *item = (wxListBox *)wxGetTypedObject(id, wxTYPE_LIST_BOX);
  if (item)
  {
    item->SetSelection((int)sel);
    return 1;
  }
  else
  {
    ClipsError << "Error in list-box-set-selection: could not find list box " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsListBoxSetSelection()
{
  if (ArgCountCheck("clipsListBoxSetSelection", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  long sel = RtnLong(2);
  return wxcListBoxSetSelection(id, sel);
}
#endif

// Deselect an item
long wxcListBoxDeselect(long id, long sel)
{
  wxListBox *item = (wxListBox *)wxGetTypedObject(id, wxTYPE_LIST_BOX);
  if (item)
  {
    item->Deselect((int)sel);
    return 1;
  }
  else
  {
    ClipsError << "Error in list-box-deselect: could not find list box " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsListBoxDeselect()
{
  if (ArgCountCheck("clipsListBoxDeselect", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  long sel = RtnLong(2);
  return wxcListBoxDeselect(id, sel);
}
#endif

// Get the current selection
long wxcListBoxGetSelection(long id)
{
  wxListBox *item = (wxListBox *)wxGetTypedObject(id, wxTYPE_LIST_BOX);
  if (item)
  {
    return item->GetSelection();
  }
  else
  {
    ClipsError << "Error in list-box-get-selection: could not find list box " << id << ".\n";
    return -1;
  }
}

#ifdef USE_CLIPS
long clipsListBoxGetSelection()
{
  if (ArgCountCheck("clipsListBoxGetSelection", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcListBoxGetSelection(id);
}
#endif

// Get string selection
char *wxcListBoxGetStringSelection(long id)
{
  wxListBox *item = (wxListBox *)wxGetTypedObject(id, wxTYPE_LIST_BOX);
  if (item)
  {
    char *s = item->GetStringSelection();
    return s;
  }
  else
  {
    ClipsError << "Error in list-box-get-string-selection: could not find list box " << id << ".\n";
    return NULL;
  }
}

#ifdef USE_CLIPS
char *clipsListBoxGetStringSelection()
{
  if (ArgCountCheck("clipsListBoxGetStringSelection", EXACTLY, 1) == -1)
    return (char *)AddSymbol("");

  long id = RtnLong(1);
  char *ans = wxcListBoxGetStringSelection(id);
  if (ans)
    return (char *)AddSymbol(ans);
  else
    return (char *)AddSymbol("");
}
#endif

// Set string selection
long wxcListBoxSetStringSelection(long id, char *sel)
{
  if (!sel)
    return 0;

  wxListBox *item = (wxListBox *)wxGetTypedObject(id, wxTYPE_LIST_BOX);
  if (item)
  {
    return item->SetStringSelection(sel);
  }
  else
  {
    ClipsError << "Error in list-box-set-string-selection: could not find list box " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsListBoxSetStringSelection()
{
  if (ArgCountCheck("clipsListBoxSetStringSelection", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  char *sel = RtnLexeme(2);
  return wxcListBoxSetStringSelection(id, sel);
}
#endif

// Get number of items
long wxcListBoxNumber(long id)
{
  wxListBox *item = (wxListBox *)wxGetTypedObject(id, wxTYPE_LIST_BOX);
  if (item)
  {
    return item->Number();
  }
  else
  {
    ClipsError << "Error in list-box-number: could not find list box " << id << ".\n";
    return -1;
  }
}

#ifdef USE_CLIPS
long clipsListBoxNumber()
{
  if (ArgCountCheck("clipsListBoxSetNumber", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcListBoxNumber(id);
}
#endif

// Delete one item
long wxcListBoxDelete(long id, long n)
{
  wxListBox *item = (wxListBox *)wxGetTypedObject(id, wxTYPE_LIST_BOX);
  if (item)
  {
    item->Delete((int)n);
    return 1;
  }
  else
  {
    ClipsError << "Error in list-box-delete: could not find list box " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsListBoxDelete()
{
  if (ArgCountCheck("clipsListBoxSetNumber", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  long n = RtnLong(2);
  return wxcListBoxDelete(id, n);
}
#endif

// Get client data for an item
char *wxcListBoxGetClientData(long id, long sel)
{
  wxListBox *item = (wxListBox *)wxGetTypedObject(id, wxTYPE_LIST_BOX);
  if (item)
  {
    char *s = item->GetClientData((int)sel);
    return s;
  }
  else
  {
    ClipsError << "Error in list-box-get-client-data: could not find list box " << id << ".\n";
    return NULL;
  }
}

#ifdef USE_CLIPS
char *clipsListBoxGetClientData()
{
  if (ArgCountCheck("clipsListBoxGetClientData", EXACTLY, 2) == -1)
    return (char *)AddSymbol("");

  long id = RtnLong(1);
  long sel = RtnLong(2);
  char *ans = wxcListBoxGetClientData(id, sel);
  if (ans)
    return (char *)AddSymbol(ans);
  else
    return (char *)AddSymbol("");
}
#endif

// Get string for integer ID
char *wxcListBoxGetString(long id, long sel)
{
  wxListBox *item = (wxListBox *)wxGetTypedObject(id, wxTYPE_LIST_BOX);
  if (item)
  {
    char *s = item->GetString((int)sel);
    return s;
  }
  else
  {
    ClipsError << "Error in list-box-get-string: could not find list box " << id << ".\n";
    return NULL;
  }
}

#ifdef USE_CLIPS
char *clipsListBoxGetString()
{
  if (ArgCountCheck("clipsListBoxGetString", EXACTLY, 2) == -1)
    return (char *)AddSymbol("");

  long id = RtnLong(1);
  long sel = RtnLong(2);
  char *ans = wxcListBoxGetString(id, sel);
  if (ans)
    return (char *)AddSymbol(ans);
  else
    return (char *)AddSymbol("");
}
#endif

// Get selections
int *wxListBoxSelections = NULL;
int wxListBoxNSelections = 0;
int wxListBoxCurrentSelection;

long wxcListBoxGetFirstSelection(long id)
{
  wxListBox *item = (wxListBox *)wxGetTypedObject(id, wxTYPE_LIST_BOX);
  if (item)
  {
    wxListBoxNSelections = item->GetSelections(&wxListBoxSelections);
    wxListBoxCurrentSelection = 1;
    if (wxListBoxNSelections > 0)
      return wxListBoxSelections[0];
    else
      return -1;
  }
  else
  {
    ClipsError << "Error in list-box-get-first-selection: could not find list box " << id << ".\n";
    return -1;
  }
}

#ifdef USE_CLIPS
long clipsListBoxGetFirstSelection()
{
  if (ArgCountCheck("clipsListBoxGetFirstSelection", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcListBoxGetFirstSelection(id);
}
#endif

long wxcListBoxGetNextSelection(void)
{
  if (wxListBoxCurrentSelection >= wxListBoxNSelections)
    return -1;

  wxListBoxCurrentSelection ++;

  return wxListBoxSelections[wxListBoxCurrentSelection - 1];
}

#ifdef USE_CLIPS
long clipsListBoxGetNextSelection()
{
  if (ArgCountCheck("clipsListBoxGetNextSelection", EXACTLY, 0) == -1)
    return 0;
  return wxcListBoxGetNextSelection();
}
#endif

// Generate choice widget creator, used only indirectly by C-only or CLIPS-only
// functions. This is so we can have a generic function for all languages.
// Create a choice widget
long wxcChoiceCreate(long parent_id, wxClipsFunction func, char *label, long x, long y,
  long width, long height, int n, char **choices, char *styleString)
{
#ifdef USE_CLIPS
  if (func && (strlen(func) == 0))
    func = NULL;
#endif
  if (!label)
    return -1;
  wxPanel *panel = (wxPanel *)wxGetTypedObject(parent_id, wxTYPE_PANEL);
  if (panel)
  {
    long style = wxClipsParseBitList(styleString);
    wxClipsChoice *item = new wxClipsChoice(panel, func, (wxFunction)wxClipsChoiceCallback, label,
     (int)x, (int)y, (int)width, (int)height, n, choices, style);
    wxStoreThingInTable((long)item, item, wxTYPE_CHOICE);
    return (long)item;
  }
  else
  {
    ClipsError << "Error in choice-create: could not find parent panel " << parent_id << ".\n";
    return 0;
  }
}

#ifndef USE_CLIPS
// Create a choice widget
long wxcChoiceCreate(long parent_id, wxClipsFunction func, char *label, long x, long y,
  long width, long height)
{
  return wxcChoiceCreate(parent_id, func, label, x, y, width, height, 0, NULL
}

// Create a choice widget
long wxcChoiceCreate(long parent_id, wxClipsFunction func, char *label, long x, long y,
  long width, long height, ClipsExpr& choices, char *styleString)
{
  if (choices.Type() != ClipsList)
    return NULL;

  int n = 0;
  char **choiceStrings = new char *[choices.Number()];
  ClipsExpr *expr = choices.GetFirst();
  while (expr)
  {
    if (expr->Type() == ClipsString || expr->Type() == ClipsWord)
    {
      choiceStrings[n] = expr->StringValue();
      n ++;
    }
    else if (expr->Type() == ClipsStringClass)
    {
      choiceStrings[n] = expr->StringClassValue().GetData();
      n ++;
    }
    expr = expr->GetNext();
  }

  if (!label)
    return -1;

  // Use generic, char * array version above
  long retValue = wxcChoiceCreate(parent_id, func, label, x, y, width, height, n, choiceStrings, styleString);
  delete[] choiceStrings;
  return retValue;
}
#endif

#ifdef USE_CLIPS
long clipsChoiceCreate()
{
  if (ArgCountCheck("clipsChoiceCreate", AT_LEAST, 3) == -1)
    return 0;

  int no_args = RtnArgCount();

  long parent_id = RtnLong(1);
  char *func = RtnLexeme(2);
  char *label = RtnLexeme(3);
  // Set up defaults
  int x = -1;
  int y = -1;
  int width = -1;
  int height = -1;
  char *style = "";
  DATA_OBJECT arg;

  if (no_args > 3)
    x = (int) RtnLong(4);

  if (no_args > 4)
    y = (int) RtnLong(5);

  if (no_args > 5)
    width = (int) RtnLong(6);

  if (no_args > 6)
    height = (int) RtnLong(7);

  int n = 0;
  char **choices = NULL;

  if (no_args > 7)
  {
    // Extract a list of strings
    RtnUnknown(8, &arg);

    if (GetType(arg) != MULTIFIELD)
    {
      ClipsError << "Error in choice-create: 8th argument must be a multifield of strings.\n";
      return -1;
    }
    int start = GetpDOBegin(&arg);
    int end = GetpDOEnd(&arg);
    n = end-start + 1;

    VOID *multifieldPtr = ClipsGetValue(arg);

    choices = new char *[n];

    for (int i = start; i <= end; i++)
    {
      if ((GetMFType(multifieldPtr, i) == STRING) || (GetMFType(multifieldPtr, i) == SYMBOL))
      {
        choices[i-start] = ValueToString(GetMFValue(multifieldPtr, i));
      }
      else
      {
        ClipsError << "Error in get-choice: 8th argument, multifield element " << i-start+1 << " must be a string or symbol.\n";
        delete[] choices;
        return -1;
      }
    }
  }

  if (no_args > 8)
    style = RtnLexeme(9);

  long retValue = wxcChoiceCreate(parent_id, func, label, x, y, width, height, n, choices, style);
  if (choices)
    delete[] choices;
  return retValue;
}
#endif

// Append a string
long wxcChoiceAppend(long id, char *item_string)
{
  if (!item_string)
    return 0;

  wxChoice *item = (wxChoice *)wxGetTypedObject(id, wxTYPE_CHOICE);
  if (item)
  {
    item->Append(item_string);
    return 1;
  }
  else
  {
    ClipsError << "Error in choice-append: could not find choice " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsChoiceAppend()
{
  if (ArgCountCheck("clipsChoiceAppend", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  char *item_string = RtnLexeme(2);
  return wxcChoiceAppend(id, item_string);
}
#endif

// Find a string and return ID
long wxcChoiceFindString(long id, char *item_string)
{
  if (!item_string)
    return -1;

  wxChoice *item = (wxChoice *)wxGetTypedObject(id, wxTYPE_CHOICE);
  if (item)
  {
    return item->FindString(item_string);
  }
  else
  {
    ClipsError << "Error in choice-find-string: could not find choice " << id << ".\n";
    return -1;
  }
}

#ifdef USE_CLIPS
long clipsChoiceFindString()
{
  if (ArgCountCheck("clipsChoiceFindString", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  char *item_string = RtnLexeme(2);
  return wxcChoiceFindString(id, item_string);
}
#endif

// Clear
long wxcChoiceClear(long id)
{
  wxChoice *item = (wxChoice *)wxGetTypedObject(id, wxTYPE_CHOICE);
  if (item)
  {
    item->Clear();
    return 1;
  }
  else
  {
    ClipsError << "Error in choice-clear: could not find choice " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsChoiceClear()
{
  if (ArgCountCheck("clipsChoiceClear", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcChoiceClear(id);
}
#endif

// Set the current selection
long wxcChoiceSetSelection(long id, long sel)
{
  wxChoice *item = (wxChoice *)wxGetTypedObject(id, wxTYPE_CHOICE);
  if (item)
  {
    item->SetSelection((int)sel);
    return 1;
  }
  else
  {
    ClipsError << "Error in choice-set-selection: could not find choice " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsChoiceSetSelection()
{
  if (ArgCountCheck("clipsChoiceSetSelection", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  long sel = RtnLong(2);

  return wxcChoiceSetSelection(id, sel);
}
#endif

// Get the current selection
long wxcChoiceGetSelection(long id)
{
  wxChoice *item = (wxChoice *)wxGetTypedObject(id, wxTYPE_CHOICE);
  if (item)
  {
    return item->GetSelection();
  }
  else
  {
    ClipsError << "Error in choice-get-selection: could not find choice " << id << ".\n";
    return -1;
  }
}

#ifdef USE_CLIPS
long clipsChoiceGetSelection()
{
  if (ArgCountCheck("clipsChoiceGetSelection", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcChoiceGetSelection(id);
}
#endif

// Get string selection
char *wxcChoiceGetStringSelection(long id)
{
  wxChoice *item = (wxChoice *)wxGetTypedObject(id, wxTYPE_CHOICE);
  if (item)
  {
    char *s = item->GetStringSelection();
    return s;
  }
  else
  {
    ClipsError << "Error in choice-get-string-selection: could not find choice " << id << ".\n";
    return NULL;
  }
}

#ifdef USE_CLIPS
char *clipsChoiceGetStringSelection()
{
  if (ArgCountCheck("clipsChoiceGetStringSelection", EXACTLY, 1) == -1)
    return (char *)AddSymbol("");

  long id = RtnLong(1);
  char *ans = wxcChoiceGetStringSelection(id);
  if (ans)
    return (char *)AddSymbol(ans);
  else
    return (char *)AddSymbol("");
}
#endif

// Set string selection
long wxcChoiceSetStringSelection(long id, char *sel)
{
  if (!sel)
    return 0;

  wxChoice *item = (wxChoice *)wxGetTypedObject(id, wxTYPE_CHOICE);
  if (item)
  {
    return item->SetStringSelection(sel);
  }
  else
  {
    ClipsError << "Error in choice-set-string-selection: could not find choice " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsChoiceSetStringSelection()
{
  if (ArgCountCheck("clipsChoiceSetStringSelection", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  char *sel = RtnLexeme(2);
  return wxcChoiceSetStringSelection(id, sel);
}
#endif

// Get string for integer ID
char *wxcChoiceGetString(long id, long sel)
{
  wxChoice *item = (wxChoice *)wxGetTypedObject(id, wxTYPE_CHOICE);
  if (item)
  {
    char *s = item->GetString((int)sel);
    return s;
  }
  else
  {
    ClipsError << "Error in choice-get-string: could not find choice " << id << ".\n";
    return NULL;
  }
}

#ifdef USE_CLIPS
char *clipsChoiceGetString()
{
  if (ArgCountCheck("clipsChoiceGetString", EXACTLY, 2) == -1)
    return (char *)AddSymbol("");

  long id = RtnLong(1);
  long sel = RtnLong(2);
  char *ans = wxcChoiceGetString(id, sel);
  if (ans)
    return (char *)AddSymbol(ans);
  else
    return (char *)AddSymbol("");
}
#endif

// Create a menu
long wxcMenuCreate(char *title)
{
  wxClipsMenu *menu = new wxClipsMenu(title);
  wxStoreThingInTable((long)menu, menu, wxTYPE_MENU);
  return (long)menu;
}

#ifdef USE_CLIPS
long clipsMenuCreate()
{
  if (ArgCountCheck("clipsMenuCreate", AT_LEAST, 0) == -1)
    return 0;

  int no_args = RtnArgCount();

  // Set up defaults
  char *title = NULL;
  if (no_args > 0)
    title = RtnLexeme(1);

  return wxcMenuCreate(title);
}
#endif

// Append a string or submenu
long wxcMenuAppend(long id, long item_id, char *menu_string, long submenu_id)
{
  if (!menu_string)
    return 0;

  wxMenu *menu = (wxMenu *)wxGetTypedObject(id, wxTYPE_MENU);
  if (!menu)
  {
    ClipsError << "Error in menu-append: could not find menu " << id << ".\n";
    return 0;
  }

  if (submenu_id > -1)
  {
    wxMenu *submenu = (wxMenu *)wxGetTypedObject(submenu_id, wxTYPE_MENU);
    if (submenu)
    {
      menu->Append((int)item_id, menu_string, submenu);
      return 1;
    }
    else
    {
      ClipsError << "Error in menu-append: could not find submenu " << submenu_id << "\n";
      return 0;
    }
  }
  else
  {
    menu->Append((int)item_id, menu_string);
    return 1;
  }
}

#ifdef USE_CLIPS
long clipsMenuAppend()
{
  if (ArgCountCheck("clipsMenuAppend", AT_LEAST, 3) == -1)
    return 0;

  int no_args = RtnArgCount();

  long id = RtnLong(1);
  long item_id = RtnLong(2);
  char *menu_string = RtnLexeme(3);
  long submenu_id = -1;
  if (no_args > 3)
    submenu_id = RtnLong(4);
  return wxcMenuAppend(id, item_id, menu_string, submenu_id);
}
#endif

// Append a separator
long wxcMenuAppendSeparator(long id)
{
  wxMenu *menu = (wxMenu *)wxGetTypedObject(id, wxTYPE_MENU);
  if (menu)
  {
    menu->AppendSeparator();
    return 1;
  }
  else
  {
    ClipsError << "Error in menu-append-separator: could not find menu " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMenuAppendSeparator()
{
  if (ArgCountCheck("clipsMenuAppendSeparator", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcMenuAppendSeparator(id);
}
#endif

// Enable an item
long wxcMenuEnable(long id, long item_id, long enabled)
{
  wxMenu *menu = (wxMenu *)wxGetTypedObject(id, wxTYPE_MENU);
  if (menu)
  {
    menu->Enable((int)item_id, (Bool)enabled);
    return 1;
  }
  else
  {
    ClipsError << "Error in menu-enable: could not find menu " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMenuEnable()
{
  if (ArgCountCheck("clipsMenuEnable", EXACTLY, 3) == -1)
    return 0;

  long id = RtnLong(1);
  long item_id = RtnLong(2);
  long enabled = RtnLong(3);
  return wxcMenuEnable(id, item_id, enabled);
}
#endif

// Check an item
long wxcMenuCheck(long id, long item_id, long checked)
{
  wxMenu *menu = (wxMenu *)wxGetTypedObject(id, wxTYPE_MENU);
  if (menu)
  {
    menu->Check((int)item_id, (Bool)checked);
    return 1;
  }
  else
  {
    ClipsError << "Error in menu-check: could not find menu " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMenuCheck()
{
  if (ArgCountCheck("clipsMenuCheck", EXACTLY, 3) == -1)
    return 0;

  long id = RtnLong(1);
  long item_id = RtnLong(2);
  long checked = RtnLong(3);

  return wxcMenuCheck(id, item_id, checked);
}
#endif

// Create a menu bar
long wxcMenuBarCreate(void)
{
  wxClipsMenuBar *menu_bar = new wxClipsMenuBar;
  wxStoreThingInTable((long)menu_bar, menu_bar, wxTYPE_MENU_BAR);
  return (long)menu_bar;
}

#ifdef USE_CLIPS
long clipsMenuBarCreate()
{
  if (ArgCountCheck("clipsMenuBarCreate", EXACTLY, 0) == -1)
    return 0;
  return wxcMenuBarCreate();
}
#endif

// Append a menu
long wxcMenuBarAppend(long id, long menu_id, char *menu_string)
{
  if (!menu_string)
    return 0;
  wxMenuBar *menu_bar = (wxMenuBar *)wxGetTypedObject(id, wxTYPE_MENU_BAR);
  if (menu_bar)
  {
    wxMenu *menu = (wxMenu *)wxGetTypedObject(menu_id, wxTYPE_MENU);
    if (menu)
    {
      menu_bar->Append(menu, menu_string);
      return 1;
    }
    else
    {
      ClipsError << "Error in menu-bar-append: could not find menu " << menu_id << ".\n";
      return 0;
    }
  }
  else
  {
    ClipsError << "Error in menu-bar-append: could not find menu bar " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMenuBarAppend()
{
  if (ArgCountCheck("clipsMenuBarAppend", EXACTLY, 3) == -1)
    return 0;

  long id = RtnLong(1);
  long menu_id = RtnLong(2);
  char *menu_string = RtnLexeme(3);
  return wxcMenuBarAppend(id, menu_id, menu_string);
}
#endif

// Enable an item
long wxcMenuBarEnable(long id, long item_id, long enabled)
{
  wxMenuBar *menu_bar = (wxMenuBar *)wxGetTypedObject(id, wxTYPE_MENU_BAR);
  if (menu_bar)
  {
    menu_bar->Enable((int)item_id, (Bool)enabled);
    return 1;
  }
  else
  {
    ClipsError << "Error in menu-bar-enable: could not find menu bar " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMenuBarEnable()
{
  if (ArgCountCheck("clipsMenuBarEnable", EXACTLY, 3) == -1)
    return 0;

  long id = RtnLong(1);
  long item_id = RtnLong(2);
  long enabled = RtnLong(3);
  return wxcMenuBarEnable(id, item_id, enabled);
}
#endif

// Check an item
long wxcMenuBarCheck(long id, long item_id, long checked)
{
  wxMenuBar *menu_bar = (wxMenuBar *)wxGetTypedObject(id, wxTYPE_MENU_BAR);
  if (menu_bar)
  {
    menu_bar->Check((int)item_id, (Bool)checked);
    return 1;
  }
  else
  {
    ClipsError << "Error in menu-bar-check: could not find menu bar " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsMenuBarCheck()
{
  if (ArgCountCheck("clipsMenuBarCheck", EXACTLY, 3) == -1)
    return 0;

  long id = RtnLong(1);
  long item_id = RtnLong(2);
  long checked = RtnLong(3);
  return wxcMenuBarCheck(id, item_id, checked);
}
#endif

/************************************************************************
 * FRAME FUNCTIONS
 *
 ************************************************************************/

// Create a new frame
long wxcFrameCreate(long parent_id, char *title, long x, long y,
  long width, long height, char *styleString)
{
/*
  int style = wxSDI | wxDEFAULT_FRAME;

  if (s)
  {
    if (strcmp(s, "wxMDI_PARENT") == 0)
      style = wxMDI_PARENT;
    else if (strcmp(s, "wxSDI") == 0)
      style = wxSDI;
    else if (strcmp(s, "wxMDI_CHILD") == 0)
      style = wxMDI_CHILD;
  }
*/
  long style = wxClipsParseBitList(styleString);
  wxFrame *parent = NULL;

  if (parent_id != 0)
  {
    parent = (wxFrame *)wxGetTypedObject(parent_id, wxTYPE_FRAME);
    if (!parent)
    {
      ClipsError << "Error in frame-create: could not find parent frame " << parent_id << ".\n";
      return 0;
    }
  }

  wxClipsFrame *new_frame = new wxClipsFrame(parent, title, (int)x, (int)y,
   (int)width, (int)height, style);
  wxStoreThingInTable((long)new_frame, new_frame, wxTYPE_FRAME);
  return (long)new_frame;
}

#ifdef USE_CLIPS
long clipsFrameCreate()
{
  if (ArgCountCheck("clipsFrameCreate", AT_LEAST, 2) == -1)
    return 0;

  int no_args = RtnArgCount();

  long parent_id = RtnLong(1);
  char *title = RtnLexeme(2);

  // Set up defaults

  int x = -1;
  int y = -1;
  int width = -1;
  int height = -1;

  if (no_args > 2)
    x = (int) RtnLong(3);

  if (no_args > 3)
    y = (int) RtnLong(4);

  if (no_args > 4)
    width = (int) RtnLong(5);

  if (no_args > 5)
    height = (int) RtnLong(6);

  char *style = "wxSDI | wxDEFAULT_FRAME";
  if (no_args > 6)
    style = RtnLexeme(7);
  return wxcFrameCreate(parent_id, title, x, y, width, height, style);
}
#endif

// Set the menu bar
long wxcFrameSetMenuBar(long id, long menubar_id)
{
  wxFrame *frame = (wxFrame *)wxGetTypedObject(id, wxTYPE_FRAME);
  if (frame)
  {
    wxMenuBar *menu_bar = (wxMenuBar *)wxGetTypedObject(menubar_id, wxTYPE_MENU_BAR);
    if (menu_bar)
    {
      frame->SetMenuBar(menu_bar);
      return 1;
    }
    else
    {
      ClipsError << "Error in frame-set-menu-bar: could not find menu bar " << menubar_id << ".\n";
      return 0;
    }
  }
  else
  {
    ClipsError << "Error in frame-set-menu-bar: could not find frame " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsFrameSetMenuBar()
{
  if (ArgCountCheck("clipsFrameSetMenuBar", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  long menubar_id = RtnLong(2);
  return wxcFrameSetMenuBar(id, menubar_id);
}
#endif

// Create the frame's status line
long wxcFrameCreateStatusLine(long id)
{
  wxFrame *frame = (wxFrame *)wxGetTypedObject(id, wxTYPE_FRAME);
  if (frame)
  {
    frame->CreateStatusLine();
    return 1;
  }
  else
  {
    ClipsError << "Error in frame-create-status-line: could not find frame " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsFrameCreateStatusLine()
{
  if (ArgCountCheck("clipsFrameCreateStatusLine", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcFrameCreateStatusLine(id);
}
#endif

// Set the status line text
long wxcFrameSetStatusText(long id, char *text)
{
  if (!text)
    return 0;

  wxFrame *frame = (wxFrame *)wxGetTypedObject(id, wxTYPE_FRAME);
  if (frame)
  {
    frame->SetStatusText(text);
    return 1;
  }
  else
  {
    ClipsError << "Error in frame-set-status-text: could not find frame " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsFrameSetStatusText()
{
  if (ArgCountCheck("clipsFrameSetStatusText", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  char *text = RtnLexeme(2);
  return wxcFrameSetStatusText(id, text);
}
#endif

// Set the icon
long wxcFrameSetIcon(long id, long iconId)
{
  wxFrame *frame = (wxFrame *)wxGetTypedObject(id, wxTYPE_FRAME);
  if (frame)
  {
    wxIcon *icon = (wxIcon *)wxGetTypedObject(iconId, wxTYPE_ICON);
    if (icon)
    {
      frame->SetIcon(icon);
      return 1;
    }
    else
    {
      ClipsError << "Error in frame-set-icon: could not find icon " << iconId << ".\n";
      return 0;
    }
  }
  else
  {
    ClipsError << "Error in frame-set-icon: could not find frame " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsFrameSetIcon()
{
  if (ArgCountCheck("clipsFrameSetIcon", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  long iconId = RtnLong(2);
  return wxcFrameSetIcon(id, iconId);
}
#endif

// Iconimize/restore the frame
long wxcFrameIconize(long id, long iconize)
{
  wxFrame *frame = (wxFrame *)wxGetTypedObject(id, wxTYPE_FRAME);
  if (frame)
  {
    frame->Iconize((Bool)iconize);
    return 1;
  }
  else
  {
    ClipsError << "Error in frame-iconize: could not find frame " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsFrameIconize()
{
  if (ArgCountCheck("clipsFrameIconize", AT_LEAST, 2) == -1)
    return 0;

  long id = RtnLong(1);
  long iconize = TRUE;

  if (RtnArgCount() > 1)
    iconize = RtnLong(2);
  return wxcFrameIconize(id, iconize);
}
#endif

/************************************************************************
 * FUNCTIONS TO ACCESS HELP SYSTEM
 *
 ************************************************************************/

/*
extern wxHelpInstance *HelpInstance;

long clipsHelpLoadFile()
{
  if (ArgCountCheck("clipsHelpLoadFile", EXACTLY, 1) == -1)
    return 0;
  char *filename = RtnLexeme(1);
  if (!filename)
    return 0;
  return HelpInstance->LoadFile(filename);
}

long clipsHelpDisplayContents()
{
  if (ArgCountCheck("clipsHelpDisplayContents", EXACTLY, 0) == -1)
    return 0;
  return HelpInstance->DisplayContents();
}

long clipsHelpDisplaySection()
{
  if (ArgCountCheck("clipsHelpDisplaySection", EXACTLY, 1) == -1)
    return 0;
  return HelpInstance->DisplaySection(RtnLong(1));
}

long clipsHelpDisplayBlock()
{
  if (ArgCountCheck("clipsHelpDisplayBlock", EXACTLY, 1) == -1)
    return 0;
  return HelpInstance->DisplayBlock(RtnLong(1));
}

long clipsHelpKeywordSearch()
{
  if (ArgCountCheck("clipsHelpKeywordSearch", EXACTLY, 1) == -1)
    return 0;
  char *keyword = RtnLexeme(1);
  if (!keyword)
    return 0;
  return HelpInstance->KeywordSearch(keyword);
}

*/

/************************************************************************
 * INPUT/OUTPUT AND USER INTERFACE FUNCTIONS
 *
 ************************************************************************/

/*
 * Implement reading strings and symbols
 *
 */

/*
char *clipsRead(void)
{
  char *s = wxGetTextFromUser("Enter text");
  if (s)
    return (char *)AddSymbol(s);
  return (char *)AddSymbol("");
}
*/

#ifdef USE_CLIPS
char *clipsReadString(void)
{
  char *s = wxGetTextFromUser("Enter text");
  if (s)
    return (char *)AddSymbol(s);
  return (char *)AddSymbol("");
}
#endif

/*
 * Pass optional message and optional default string
 *
 */

char *wxcGetTextFromUser(char *message, char *default_string,
  long centre)
{
  char *s = wxGetTextFromUser(message, "Enter text", default_string, NULL, -1, -1, (Bool)centre);
  return s;
}

#ifdef USE_CLIPS
char *clipsGetTextFromUser(void)
{
  char *message = "Enter a string";
  char *default_string = "";
  Bool centre = TRUE;

  if (ArgCountCheck("clipsGetTextFromUser", AT_LEAST, 1) != -1)
    message = RtnLexeme(1);

  if (!message)
    return (char *)AddSymbol("");

  int no_args = RtnArgCount();

  if (no_args > 1)
    default_string = RtnLexeme(2);
  if (!default_string)
    return (char *)AddSymbol("");

  if (no_args > 2)
    centre = (Bool)RtnLong(3);

  char *s = wxcGetTextFromUser(message, default_string, centre);
  if (s)
    return (char *)AddSymbol(s);
  else
    return (char *)AddSymbol("");
}
#endif

/*
 * Convert string to number
 *
 */

double wxcStringToFloat(char *str)
{
  if (!str)
    return 0.0;
  return atof(str);
}

#ifdef USE_CLIPS
double clipsStringToFloat(void)
{
  if (ArgCountCheck("clipsStringToFloat", EXACTLY, 1) == -1)
  {
    ClipsError << "Error in string-to-float: one string argument required.\n";
    return 0.0;
  }

  char *str = RtnLexeme(1);
  return wxcStringToFloat(str);
}
#endif

long wxcStringToLong(char *str)
{
  if (!str)
    return 0;
  return atol(str);
}

#ifdef USE_CLIPS
long clipsStringToLong(void)
{
  if (ArgCountCheck("clipsStringToLong", EXACTLY, 1) == -1)
  {
    ClipsError << "Error in string-to-long: one string argument required.\n";
    return 0;
  }

  char *str = RtnLexeme(1);
  return wxcStringToLong(str);
}
#endif

/*
 * Convert number to string
 *
 */

char *wxcFloatToString(double l)
{
  static char buf[50];
  sprintf(buf, "%.4f", 1);
  return buf;
}

#ifdef USE_CLIPS
char *clipsFloatToString(void)
{
  if (ArgCountCheck("clipsFloatToString", EXACTLY, 1) == -1)
  {
    ClipsError << "Error in float-to-string: one floating point argument required.\n";
    return (char *)AddSymbol("");
  }
  return (char *)AddSymbol(wxcFloatToString(RtnDouble(1)));
}
#endif

char *wxcLongToString(long l)
{
  static char buf[50];
  sprintf(buf, "%ld", 1);
  return buf;
}

#ifdef USE_CLIPS
char *clipsLongToString(void)
{
  if (ArgCountCheck("clipsLongToString", EXACTLY, 1) == -1)
  {
    ClipsError << "Error in long-to-string: one integer argument required.\n";
    return (char *)AddSymbol("");
  }
  return (char *)AddSymbol(wxcFloatToString(RtnLong(1)));
}
#endif

/*
 * Convert string to symbol
 *
 */

char *wxcStringToSymbol(char *s)
{
  return s;
}

#ifdef USE_CLIPS
char *clipsStringToSymbol(void)
{
  if (ArgCountCheck("clipsStringToSymbol", EXACTLY, 1) == -1)
  {
    ClipsError << "Error in string-to-symbol: one string argument required.\n";
    return (char *)AddSymbol("");
  }

  char *str = RtnLexeme(1);
  if (!str)
    return (char *)AddSymbol("");
  return (char *)AddSymbol(str);
}
#endif

/*
 * Convert symbol to string
 *
 */

char *wxcSymbolToString(char *s)
{
  return s;
}

#ifdef USE_CLIPS
char *clipsSymbolToString(void)
{
  if (ArgCountCheck("clipsSymbolToString", EXACTLY, 1) == -1)
  {
    ClipsError << "Error in symbol-to-string: one symbol argument required.\n";
    return (char *)AddSymbol("");
  }

  char *str = RtnLexeme(1);
  if (!str)
    return (char *)AddSymbol("");
  return (char *)AddSymbol(str);
}
#endif

/*
 * Convert a list of strings or symbols to a sorted list of strings
 *
 */

#ifdef USE_CLIPS
VOID clipsStringSort(DATA_OBJECT_PTR returnValuePtr)
{
  DATA_OBJECT arg;

  if (ArgCountCheck("clipsStringSort", AT_LEAST, 1) == -1)
  {
    ClipsError << "Error in string-sort: 1st argument must be a multifield of strings or symbols.\n";
    SetMultifieldErrorValue(returnValuePtr);
    return;
  }

  RtnUnknown(1, &arg);

  if (GetType(arg) != MULTIFIELD)
  {
    ClipsError << "Error in string-sort: 1st argument must be a multifield of strings or symbols.\n";
    SetMultifieldErrorValue(returnValuePtr);
    return;
  }

  int start = GetpDOBegin(&arg);
  int end = GetpDOEnd(&arg);
  int no_strings = end-start + 1;

  VOID *multifieldPtr = ClipsGetValue(arg);
  wxStringList stringList;

  int i;
  for (i = start; i <= end; i++)
  {
    if ((GetMFType(multifieldPtr, i) == STRING) || (GetMFType(multifieldPtr, i) == STRING))
      stringList.Add(ValueToString(GetMFValue(multifieldPtr, i)));
    else
    {
      ClipsError << "Error in string-sort: multifield element " << i-start+1 << " must be a string or symbol.\n";
      SetMultifieldErrorValue(returnValuePtr);
      return;
    }
  }
  stringList.Sort();

  VOID *newMultifield = CreateMultifield(no_strings);
  for (i = start; i <= end; i++)
  {
    wxNode *node = stringList.Nth(i-start);
    if (node)
    {
      char *s = (char *)node->Data();

      SetMFType(newMultifield, i, STRING);
      SetMFValue(newMultifield, i, AddSymbol(s));
    }
  }
  SetpType(returnValuePtr, MULTIFIELD);
  SetpValue(returnValuePtr, newMultifield);
  SetpDOBegin(returnValuePtr,start);
  SetpDOEnd(returnValuePtr,end);
  return;
}
#endif

/*
 * Ask user to choose from a menu of strings (message followed by a
 * variable number of string arguments)
 *
 */

#ifndef USE_CLIPS
char *wxcGetChoice(char *message, ClipsExpr& choices, long centre, long parentId)
{
  if (choices.Type() != ClipsList)
    return NULL;

  int n = 0;
  char **choiceStrings = new char *[choices.Number()];
  ClipsExpr *expr = choices.GetFirst();
  while (expr)
  {
    if (expr->Type() == ClipsString || expr->Type() == ClipsWord)
    {
      choiceStrings[n] = expr->StringValue();
      n ++;
    }
    else if (expr->Type() == ClipsStringClass)
    {
      choiceStrings[n] = expr->StringClassValue().GetData();
      n ++;
    }
    expr = expr->GetNext();
  }
  wxWindow *parent = NULL;
  if (parentId > 0)
  {
    parent = (wxWindow *)wxGetTypedObject(parentId, wxTYPE_WINDOW);
    if (!parent)
    {
      ClipsError << "Error in file-selector: could not find parent window " << parentId << ".\n";
      delete[] choiceStrings;
      return NULL;
    }
  }

  char *s = wxGetSingleChoice(message, "Menu", n, choiceStrings, window, -1, -1, (Bool)centre);
  delete[] choiceStrings;
  return s;
}
#endif

#ifdef USE_CLIPS
char *clipsGetChoice(void)
{
  DATA_OBJECT arg;
  Bool centre = TRUE;
  long parentId = 0;

  if (ArgCountCheck("clipsGetChoice", AT_LEAST, 2) == -1)
    return (char *)AddSymbol("");

  int no_args = RtnArgCount();
  if (no_args > 2)
    centre = (Bool)RtnLong(3);
  if (no_args > 3)
    parentId = RtnLong(4);

  RtnUnknown(2, &arg);

  if (GetType(arg) != MULTIFIELD)
  {
    ClipsError << "Error in get-choice: 2nd argument must be a multifield of strings.\n";
    return (char *)AddSymbol("");
  }

  char *message = RtnLexeme(1);
  if (!message)
    return (char *)AddSymbol("");

  int start = GetpDOBegin(&arg);
  int end = GetpDOEnd(&arg);
  int no_strings = end-start + 1;

  VOID *multifieldPtr = ClipsGetValue(arg);

  char **choices = new char *[no_strings];

  for (int i = start; i <= end; i++)
  {
    if ((GetMFType(multifieldPtr, i) == STRING) || (GetMFType(multifieldPtr, i) == SYMBOL))
      choices[i-start] = ValueToString(GetMFValue(multifieldPtr, i));
    else
    {
      ClipsError << "Error in get-choice: 2nd argument, multifield element " << i-start+1 << " must be a string or symbol.\n";
      delete choices;
      return (char *)AddSymbol("");
    }
  }

  wxWindow *parent = NULL;
  if (parentId > 0)
  {
    parent = (wxWindow *)wxGetTypedObject(parentId, wxTYPE_WINDOW);
    if (!parent)
    {
      ClipsError << "Error in file-selector: could not find parent window " << parentId << ".\n";
      return NULL;
    }
  }

  char *s = wxGetSingleChoice(message, "Menu", no_strings, choices, parent, -1, -1, centre);
  delete choices;

  if (s)
    return (char *)AddSymbol(s);
  return (char *)AddSymbol("");
}
#endif

/*
 * Pass message, optional flags = "OK", "OK-CANCEL", "YES-NO" or "YES-NO-CANCEL"
 *
 */

char *wxcMessageBox(char *message, char *flags, long centre, long parentId)
{
  int wx_flags = wxOK;
  if (strcmp(flags, "OK") == 0)
    wx_flags = wxOK;
  else if (strcmp(flags, "OK-CANCEL") == 0)
    wx_flags = wxOK | wxCANCEL;
  else if (strcmp(flags, "YES-NO") == 0)
    wx_flags = wxYES_NO;
  else if (strcmp(flags, "YES-NO-CANCEL") == 0)
    wx_flags = wxYES_NO | wxCANCEL;

  if (centre)
    wx_flags |= wxCENTRE;

  wxWindow *parent = NULL;
  if (parentId > 0)
  {
    parent = (wxWindow *)wxGetTypedObject(parentId, wxTYPE_WINDOW);
    if (!parent)
    {
      ClipsError << "Error in message-box: could not find parent window " << parentId << ".\n";
      return NULL;
    }
  }

  int res = wxMessageBox(message, "Message", wx_flags, parent);
  char *ans = "ERROR";
  switch (res)
  {  
    case wxCANCEL: ans = "CANCEL"; break;
    case wxOK:     ans = "OK"; break;
    case wxYES:    ans = "YES"; break;
    case wxNO:     ans = "NO"; break;
  }

  return ans;
}

#ifdef USE_CLIPS
char *clipsMessageBox(void)
{
  int no_args = RtnArgCount();
  if (ArgCountCheck("clipsMessageBox", AT_LEAST, 1) == -1)
    return (char *)AddSymbol("");

  char *message = RtnLexeme(1);
  if (!message)
    return (char *)AddSymbol("");

  char *flags = "OK";
  Bool centre = TRUE;
  long parentId = 0;

  if (no_args > 1)
    flags = RtnLexeme(2);
  if (!flags)
    return (char *)AddSymbol("");

  if (no_args > 2)
    centre = (Bool)RtnLong(3);
  if (no_args > 3)
    parentId = RtnLong(4);

  char *s = wxcMessageBox(message, flags, centre, parentId);
  if (!s)
    return (char *)AddSymbol("");
  else
    return (char *)AddSymbol(s);
}
#endif

/*
 * Pass optional message, default path, default file, default extension, default
 * wildcard.
 *
 */

char *wxcFileSelector(char *message,
  char *def_path, char *def_file, char *def_ext, char *def_wildcard, long parentId)
{
  wxWindow *parent = NULL;
  if (parentId > 0)
  {
    parent = (wxWindow *)wxGetTypedObject(parentId, wxTYPE_WINDOW);
    if (!parent)
    {
      ClipsError << "Error in file-selector: could not find parent window " << parentId << ".\n";
      return NULL;
    }
  }

  char *s = wxFileSelector(message, def_path, def_file, def_ext, def_wildcard, 0, parent);
  return s;
}

#ifdef USE_CLIPS
char *clipsFileSelector(void)
{
  char *message = "Select a file";
  char *def_path = NULL;
  char *def_file = NULL;
  char *def_ext = NULL;
  char *def_wildcard = "*.*";
  long parentId = 0;

  int no_args = RtnArgCount();
  if (no_args > 0)
    message = RtnLexeme(1);
  if (!message)
    return (char *)AddSymbol("");

  if (no_args > 1)
    def_path = RtnLexeme(2);
  if (no_args > 2)
    def_file = RtnLexeme(3);
  if (no_args > 3)
    def_ext = RtnLexeme(4);
  if (no_args > 4)
    def_wildcard = RtnLexeme(5);
  if (!def_wildcard)
    return (char *)AddSymbol("");
  if (no_args > 5)
    parentId = RtnLong(6);

  char *s = wxcFileSelector(message, def_path, def_file, def_ext, def_wildcard, parentId);

  if (s)
    return (char *)AddSymbol(s);
  return (char *)AddSymbol("");
}
#endif

/*
 * Yield to windowing system (if appropriate -- i.e. Windows only)
 *
 */

long wxcYield(void)
{
  wxYield();
  return 1;
}

#ifdef USE_CLIPS
long clipsYield(void)
{
  return wxcYield();
}
#endif

/*
 * Sleep for nSecs seconds
 *
 */

long wxcSleep(long nSecs)
{
  wxSleep((int)nSecs);
  return 1;
}

#ifdef USE_CLIPS
long clipsSleep(void)
{
  if (ArgCountCheck("clipsSleep", EXACTLY, 1) == -1)
    return 0;
  return wxcSleep(RtnLong(1));
}
#endif

/* Have to implement new batch command since CLIPS assumes using
 * the main loop
 */
#ifdef USE_CLIPS
void wxExecuteClipsFile(char *filename)
{
  if (FileExists(filename))
  {
    ExecuteBatchFile(filename);
  }
#ifdef wx_x
  else cerr << "Cannot find CLIPS file " << filename << ".\n";
#endif
}
#endif

#ifdef USE_CLIPS
void clipsBatch(void)
{
  if (ArgCountCheck("clipsBatch", EXACTLY, 1) == -1)
    return;
  char *filename = RtnLexeme(1);
  if (!filename)
    return;
  wxExecuteClipsFile(filename);
}
#endif

/*
 * DDE functions
 *
 */

#ifdef USE_CLIPS
long clipsClientCreate(void)
{
//  int no_args = RtnArgCount();
  if (ArgCountCheck("clipsClientCreate", EXACTLY, 1) == -1)
    return 0;

  char *func = RtnLexeme(1);
  if (!func)
    return 0;

  wxClipsClient *client = new wxClipsClient(func);
  wxStoreThingInTable((long)client, client, wxTYPE_DDE_CLIENT);
  return (long)client;
}
#endif

/*
 * Returns an integer representing the connection, or zero if failed.
 *
 */
 
#ifdef USE_CLIPS
long clipsClientMakeConnection(void)
{
//  int no_args = RtnArgCount();
  if (ArgCountCheck("clipsClientMakeConnection", EXACTLY, 4) == -1)
    return 0;

  long id = RtnLong(1);
  char *host = RtnLexeme(2);
  char *service = RtnLexeme(3);
  char *topic = RtnLexeme(4);

  if (!host || !service || !topic)
    return 0;

  wxClipsClient *client = (wxClipsClient *)wxGetTypedObject(id, wxTYPE_DDE_CLIENT);
  if (!client)
  {
    ClipsError << "Error in client-make-connection: could not find client " << id << ".\n";
    return 0;
  }

  wxConnection *connection = client->MakeConnection(host, service, topic);
  return (long)connection;
}
#endif

#ifdef USE_CLIPS
long clipsServerCreate(void)
{
//  int no_args = RtnArgCount();
  if (ArgCountCheck("clipsServerCreate", EXACTLY, 1) == -1)
    return 0;

  char *func = RtnLexeme(1);
  if (!func)
    return 0;

  wxClipsServer *server = new wxClipsServer(func);
  wxStoreThingInTable((long)server, server, wxTYPE_DDE_SERVER);
  return (long)server;
}
#endif

/*
 * Connection
 */
 
#ifdef USE_CLIPS
long clipsConnectionAdvise(void)
{
//  int no_args = RtnArgCount();
  if (ArgCountCheck("clipsConnectionAdvise", EXACTLY, 3) == -1)
    return 0;

  long id = RtnLong(1);
  char *item = RtnLexeme(2);
  char *data = RtnLexeme(3);

  if (!item || !data)
    return 0;

  wxClipsConnection *connection = (wxClipsConnection *)wxGetTypedObject(id, wxTYPE_DDE_CONNECTION);
  if (!connection)
  {
    ClipsError << "Error in connection-advise: could not find connection " << id << ".\n";
    return 0;
  }

  return (long)connection->Advise(item, data);
}
#endif

#ifdef USE_CLIPS
long clipsConnectionExecute(void)
{
//  int no_args = RtnArgCount();
  if (ArgCountCheck("clipsConnectionExecute", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  char *data = RtnLexeme(2);

  if (!data)
    return 0;

  wxClipsConnection *connection = (wxClipsConnection *)wxGetTypedObject(id, wxTYPE_DDE_CONNECTION);
  if (!connection)
  {
    ClipsError << "Error in connection-execute: could not find connection " << id << ".\n";
    return 0;
  }

  return (long)connection->Execute(data);
}
#endif

#ifdef USE_CLIPS
long clipsConnectionDisconnect(void)
{
//  int no_args = RtnArgCount();
  if (ArgCountCheck("clipsConnectionDisconnect", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);

  wxClipsConnection *connection = (wxClipsConnection *)wxGetTypedObject(id, wxTYPE_DDE_CONNECTION);
  if (!connection)
  {
    ClipsError << "Error in connection-disconnect: could not find connection " << id << ".\n";
    return 0;
  }

  return (long)connection->Disconnect();
}
#endif

#ifdef USE_CLIPS
long clipsConnectionPoke(void)
{
//  int no_args = RtnArgCount();
  if (ArgCountCheck("clipsConnectionPoke", EXACTLY, 3) == -1)
    return 0;

  long id = RtnLong(1);
  char *item = RtnLexeme(2);
  char *data = RtnLexeme(3);

  if (!item || !data)
    return 0;

  wxClipsConnection *connection = (wxClipsConnection *)wxGetTypedObject(id, wxTYPE_DDE_CONNECTION);
  if (!connection)
  {
    ClipsError << "Error in connection-poke: could not find connection " << id << ".\n";
    return 0;
  }

  return (long)connection->Poke(item, data);
}
#endif

#ifdef USE_CLIPS
char *clipsConnectionRequest(void)
{
//  int no_args = RtnArgCount();
  if (ArgCountCheck("clipsConnectionRequest", EXACTLY, 2) == -1)
    return (char *)AddSymbol("");

  long id = RtnLong(1);
  char *item = RtnLexeme(2);

  if (!item)
    return (char *)AddSymbol("");

  wxClipsConnection *connection = (wxClipsConnection *)wxGetTypedObject(id, wxTYPE_DDE_CONNECTION);
  if (!connection)
  {
    ClipsError << "Error in connection-request: could not find connection " << id << ".\n";
    return (char *)AddSymbol("");
  }

  char *data = connection->Request(item);
  if (data)
    return (char *)AddSymbol(data);
  else
    return (char *)AddSymbol("");
}
#endif

#ifdef USE_CLIPS
long clipsConnectionStartAdvise(void)
{
//  int no_args = RtnArgCount();
  if (ArgCountCheck("clipsConnectionStartAdvise", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  char *item = RtnLexeme(2);

  if (!item)
    return 0;

  wxClipsConnection *connection = (wxClipsConnection *)wxGetTypedObject(id, wxTYPE_DDE_CONNECTION);
  if (!connection)
  {
    ClipsError << "Error in connection-start-advise: could not find connection " << id << ".\n";
    return 0;
  }

  return (long)connection->StartAdvise(item);
}
#endif

#ifdef USE_CLIPS
long clipsConnectionStopAdvise(void)
{
//  int no_args = RtnArgCount();
  if (ArgCountCheck("clipsConnectionStopAdvise", EXACTLY, 2) == -1)
    return 0;

  long id = RtnLong(1);
  char *item = RtnLexeme(2);

  if (!item)
    return 0;

  wxClipsConnection *connection = (wxClipsConnection *)wxGetTypedObject(id, wxTYPE_DDE_CONNECTION);
  if (!connection)
  {
    ClipsError << "Error in connection-stop-advise: could not find connection " << id << ".\n";
    return 0;
  }

  return (long)connection->StopAdvise(item);
}
#endif

#ifdef USE_CLIPS
/*
 * Dummy 'declare' for C++ type specification.
 */
 
void clipsDeclare(void)
{
}
#endif

/*************************************************************/
/* UserFunctions:  The function which informs CLIPS of any   */
/*   user defined functions.  In the default case, there are */
/*   no user defined functions.  To define functions, either */
/*   this function must be replaced by a function with the   */
/*   same name within this file, or this function can be     */
/*   deleted from this file and included in another file.    */
/*   User defined functions may be included in this file or  */
/*   other files.                                            */
/*   Example of redefined UserFunctions:                     */
/*     UserFunctions()                                       */
/*       {                                                   */
/*        DefineFunction("fun1",'i',fun1,"fun1");            */
/*        DefineFunction("other",'f',other,"other");         */
/*       }                                                   */
/*************************************************************/

#ifdef USE_CLIPS
extern void wxGDIUserFunctions();

void wxUserFunctions()
{
  wxGDIUserFunctions();
  
  // Miscellaneous functions
  DefineFunction("batch", 'v', PTIF clipsBatch, "clipsBatch");
  DefineFunction("file-selector", 's', PTIF clipsFileSelector, "clipsFileSelector");
  DefineFunction("float-to-string", 's', PTIF clipsFloatToString, "clipsFloatToString");
  DefineFunction("long-to-string", 's', PTIF clipsLongToString, "clipsLongToString");
  DefineFunction("get-choice", 's', PTIF clipsGetChoice, "clipsGetChoice");
  DefineFunction("get-text-from-user", 's', PTIF clipsGetTextFromUser, "clipsGetTextFromUser");
  DefineFunction("get-platform", 's', PTIF clipsGetPlatform, "clipsGetPlatform");
  DefineFunction("message-box", 'w', PTIF clipsMessageBox, "clipsMessageBox");
  DefineFunction("read-string", 's', PTIF clipsReadString, "clipsReadString");
  DefineFunction("string-to-float", 'd', PTIF clipsStringToFloat, "clipsStringToFloat");
  DefineFunction("string-to-long", 'l', PTIF clipsStringToLong, "clipsStringToLong");
  DefineFunction("string-to-symbol", 'w', PTIF clipsStringToSymbol, "clipsStringToSymbol");
  DefineFunction("symbol-to-string", 's', PTIF clipsSymbolToString, "clipsSymbolToString");
  DefineFunction("yield", 'l', PTIF clipsYield, "clipsYield");
  DefineFunction("sleep", 'l', PTIF clipsSleep, "clipsSleep");

  // Window functions
  DefineFunction("button-create", 'l', PTIF clipsButtonCreate, "clipsButtonCreate");
  DefineFunction("button-create-from-bitmap", 'l', PTIF clipsButtonCreateFromBitmap, "clipsButtonCreateFromBitmap");

  DefineFunction("canvas-create", 'l', PTIF clipsCanvasCreate, "clipsCanvasCreate");
  DefineFunction("canvas-set-scrollbars", 'l', PTIF clipsCanvasSetScrollbars, "clipsCanvasSetScrollbars");
  DefineFunction("canvas-scroll", 'l', PTIF clipsCanvasScroll, "clipsCanvasScroll");
  DefineFunction("canvas-get-dc", 'l', PTIF clipsCanvasGetDC, "clipsCanvasGetDC");

  DefineFunction("check-box-create", 'l', PTIF clipsCheckBoxCreate, "clipsCheckBoxCreate");
  DefineFunction("check-box-set-value", 'l', PTIF clipsCheckBoxSetValue, "clipsCheckBoxSetValue");
  DefineFunction("check-box-get-value", 'l', PTIF clipsCheckBoxGetValue, "clipsCheckBoxGetValue");

  DefineFunction("choice-create", 'l', PTIF clipsChoiceCreate, "clipsChoiceCreate");
  DefineFunction("choice-append", 'l', PTIF clipsChoiceAppend, "clipsChoiceAppend");
  DefineFunction("choice-find-string", 'l', PTIF clipsChoiceFindString, "clipsChoiceFindString");
  DefineFunction("choice-clear", 'l', PTIF clipsChoiceClear, "clipsChoiceClear");
  DefineFunction("choice-set-selection", 'l', PTIF clipsChoiceSetSelection, "clipsChoiceSetSelection");
  DefineFunction("choice-get-string-selection", 's', PTIF clipsChoiceGetStringSelection, "clipsChoiceGetStringSelection");
  DefineFunction("choice-get-selection", 'l', PTIF clipsChoiceGetSelection, "clipsChoiceGetSelection");
  DefineFunction("choice-set-string-selection", 'l', PTIF clipsChoiceSetStringSelection, "clipsChoiceSetStringSelection");
  DefineFunction("choice-get-string", 's', PTIF clipsChoiceGetString, "clipsChoiceGetString");

  DefineFunction("clean-windows", 'v', PTIF clipsCleanWindows, "clipsCleanWindows");

  DefineFunction("dialog-box-create", 'l', PTIF clipsDialogBoxCreate, "clipsDialogBoxCreate");

  DefineFunction("dc-blit", 'l', PTIF clipsDCBlit, "clipsDCBlit");
  DefineFunction("dc-clear", 'l', PTIF clipsDCClear, "clipsDCClear");
  DefineFunction("dc-destroy-clipping-region", 'l', PTIF clipsDCDestroyClippingRegion, "clipsDCDestroyClippingRegion");
  DefineFunction("dc-draw-ellipse", 'l', PTIF clipsDCDrawEllipse, "clipsDCDrawEllipse");
  DefineFunction("dc-draw-line", 'l', PTIF clipsDCDrawLine, "clipsDCDrawLine");
  DefineFunction("dc-draw-lines", 'l', PTIF clipsDCDrawLines, "clipsDCDrawLines");
  DefineFunction("dc-draw-point", 'l', PTIF clipsDCDrawPoint, "clipsDCDrawPoint");
  DefineFunction("dc-draw-polygon", 'l', PTIF clipsDCDrawPolygon, "clipsDCDrawPolygon");
  DefineFunction("dc-draw-spline", 'l', PTIF clipsDCDrawSpline, "clipsDCDrawSpline");
  DefineFunction("dc-draw-rectangle", 'l', PTIF clipsDCDrawRectangle, "clipsDCDrawRectangle");
  DefineFunction("dc-draw-rounded-rectangle", 'l', PTIF clipsDCDrawRoundedRectangle, "clipsDCDrawRoundedRectangle");
  DefineFunction("dc-draw-text", 'l', PTIF clipsDCDrawText, "clipsDCDrawText");
  DefineFunction("dc-set-clipping-region", 'l', PTIF clipsDCSetClippingRegion, "clipsDCSetClippingRegion");
  DefineFunction("dc-set-pen", 'l', PTIF clipsDCSetPen, "clipsDCSetPen");
  DefineFunction("dc-set-logical-function", 'l', PTIF clipsDCSetLogicalFunction, "clipsDCSetLogicalFunction");
  DefineFunction("dc-set-text-foreground", 'l', PTIF clipsDCSetTextForeground, "clipsDCSetTextForeground");
  DefineFunction("dc-set-text-background", 'l', PTIF clipsDCSetTextBackground, "clipsDCSetTextBackground");
  DefineFunction("dc-set-brush", 'l', PTIF clipsDCSetBrush, "clipsDCSetBrush");
  DefineFunction("dc-set-font", 'l', PTIF clipsDCSetFont, "clipsDCSetFont");

  DefineFunction("memory-dc-create", 'l', PTIF clipsMemoryDCCreate, "clipsMemoryDCCreate");
  DefineFunction("memory-dc-delete", 'l', PTIF clipsMemoryDCDelete, "clipsMemoryDCDelete");
  DefineFunction("memory-dc-select-object", 'l', PTIF clipsMemoryDCSelectObject, "clipsMemoryDCSelectObject");

  DefineFunction("mouse-event-is-button", 'l', PTIF clipsMouseEventIsButton, "clipsMouseEventIsButton");
  DefineFunction("mouse-event-button-down", 'l', PTIF clipsMouseEventButtonDown, "clipsMouseEventButtonDown");
  DefineFunction("mouse-event-control-down", 'l', PTIF clipsMouseEventControlDown, "clipsMouseEventControlDown");
  DefineFunction("mouse-event-shift-down", 'l', PTIF clipsMouseEventShiftDown, "clipsMouseEventShiftDown");
  DefineFunction("mouse-event-button", 'l', PTIF clipsMouseEventButton, "clipsMouseEventButton");
  DefineFunction("mouse-event-left-down", 'l', PTIF clipsMouseEventLeftDown, "clipsMouseEventLeftDown");
  DefineFunction("mouse-event-middle-down", 'l', PTIF clipsMouseEventMiddleDown, "clipsMouseEventMiddleDown");
  DefineFunction("mouse-event-right-down", 'l', PTIF clipsMouseEventRightDown, "clipsMouseEventRightDown");
  DefineFunction("mouse-event-left-up", 'l', PTIF clipsMouseEventLeftUp, "clipsMouseEventLeftUp");
  DefineFunction("mouse-event-middle-up", 'l', PTIF clipsMouseEventMiddleUp, "clipsMouseEventMiddleUp");
  DefineFunction("mouse-event-right-up", 'l', PTIF clipsMouseEventRightUp, "clipsMouseEventRightUp");
  DefineFunction("mouse-event-dragging", 'l', PTIF clipsMouseEventDragging, "clipsMouseEventDragging");
  DefineFunction("mouse-event-position-x", 'd', PTIF clipsMouseEventPositionX, "clipsMouseEventPositionX");
  DefineFunction("mouse-event-position-y", 'd', PTIF clipsMouseEventPositionY, "clipsMouseEventPositionY");

  DefineFunction("frame-create", 'l', PTIF clipsFrameCreate, "clipsFrameCreate");
  DefineFunction("frame-create-status-line", 'l', PTIF clipsFrameCreateStatusLine, "clipsFrameCreateStatusLine");
  DefineFunction("frame-iconize", 'l', PTIF clipsFrameIconize, "clipsFrameIconize");
  DefineFunction("frame-set-icon", 'l', PTIF clipsFrameSetIcon, "clipsFrameSetIcon");
  DefineFunction("frame-set-menu-bar", 'l', PTIF clipsFrameSetMenuBar, "clipsFrameSetMenuBar");
  DefineFunction("frame-set-status-text", 'l', PTIF clipsFrameSetStatusText, "clipsFrameSetStatusText");

  DefineFunction("list-box-create", 'l', PTIF clipsListBoxCreate, "clipsListBoxCreate");
  DefineFunction("list-box-append", 'l', PTIF clipsListBoxAppend, "clipsListBoxAppend");
  DefineFunction("list-box-find-string", 'l', PTIF clipsListBoxFindString, "clipsListBoxFindString");
  DefineFunction("list-box-clear", 'l', PTIF clipsListBoxClear, "clipsListBoxClear");
  DefineFunction("list-box-set-selection", 'l', PTIF clipsListBoxSetSelection, "clipsListBoxSetSelection");
  DefineFunction("list-box-get-string-selection", 's', PTIF clipsListBoxGetStringSelection, "clipsListBoxGetStringSelection");
  DefineFunction("list-box-get-selection", 'l', PTIF clipsListBoxGetSelection, "clipsListBoxGetSelection");
  DefineFunction("list-box-set-string-selection", 'l', PTIF clipsListBoxSetStringSelection, "clipsListBoxSetStringSelection");
  DefineFunction("list-box-number", 'l', PTIF clipsListBoxNumber, "clipsListBoxNumber");
  DefineFunction("list-box-delete", 'l', PTIF clipsListBoxDelete, "clipsListBoxDelete");
  DefineFunction("list-box-get-string", 's', PTIF clipsListBoxGetString, "clipsListBoxGetString");
  DefineFunction("list-box-get-first-selection", 'l', PTIF clipsListBoxGetFirstSelection, "clipsListBoxGetFirstSelection");
  DefineFunction("list-box-get-next-selection", 'l', PTIF clipsListBoxGetNextSelection, "clipsListBoxGetNextSelection");

  DefineFunction("menu-create", 'l', PTIF clipsMenuCreate, "clipsMenuCreate");
  DefineFunction("menu-append", 'l', PTIF clipsMenuAppend, "clipsMenuAppend");
  DefineFunction("menu-append-separator", 'l', PTIF clipsMenuAppendSeparator, "clipsMenuAppendSeparator");
  DefineFunction("menu-check", 'l', PTIF clipsMenuCheck, "clipsMenuCheck");
  DefineFunction("menu-enable", 'l', PTIF clipsMenuEnable, "clipsMenuEnable");

  DefineFunction("menu-bar-create", 'l', PTIF clipsMenuBarCreate, "clipsMenuBarCreate");
  DefineFunction("menu-bar-append", 'l', PTIF clipsMenuBarAppend, "clipsMenuBarAppend");
  DefineFunction("menu-bar-check", 'l', PTIF clipsMenuBarCheck, "clipsMenuBarCheck");
  DefineFunction("menu-bar-enable", 'l', PTIF clipsMenuBarEnable, "clipsMenuBarEnable");

  DefineFunction("message-create", 'l', PTIF clipsMessageCreate, "clipsMessageCreate");

  DefineFunction("multi-text-create", 'l', PTIF clipsMultiTextCreate, "clipsMultiTextCreate");
  DefineFunction("multi-text-set-value", 'l', PTIF clipsMultiTextSetValue, "clipsMultiTextSetValue");
  DefineFunction("multi-text-get-value", 's', PTIF clipsMultiTextGetValue, "clipsMultiTextGetValue");

  DefineFunction("panel-create", 'l', PTIF clipsPanelCreate, "clipsPanelCreate");
  DefineFunction("panel-set-label-position", 'l', PTIF clipsPanelSetLabelPosition, "clipsPanelSetLabelPosition");
  DefineFunction("panel-set-label-font", 'l', PTIF clipsPanelSetLabelFont, "clipsPanelSetLabelFont");
  DefineFunction("panel-set-button-font", 'l', PTIF clipsPanelSetButtonFont, "clipsPanelSetButtonFont");
  DefineFunction("panel-new-line", 'l', PTIF clipsPanelNewLine, "clipsPanelNewLine");

  DefineFunction("panel-item-set-default", 'l', PTIF clipsPanelItemSetDefault, "clipsPanelItemSetDefault");
  DefineFunction("panel-item-set-label", 'l', PTIF clipsPanelItemSetLabel, "clipsPanelItemSetLabel");
  DefineFunction("panel-item-get-label", 's', PTIF clipsPanelItemGetLabel, "clipsPanelItemGetLabel");

  DefineFunction("return-result", 'v', PTIF clipsReturnResult, "clipsReturnResult");

  DefineFunction("slider-create", 'l', PTIF clipsSliderCreate, "clipsSliderCreate");
  DefineFunction("slider-set-value", 'l', PTIF clipsSliderSetValue, "clipsSliderSetValue");
  DefineFunction("slider-get-value", 'l', PTIF clipsSliderGetValue, "clipsSliderGetValue");

  DefineFunction("string-sort", 'm', PTIF clipsStringSort, "clipsStringSort");

  DefineFunction("text-create", 'l', PTIF clipsTextCreate, "clipsTextCreate");
  DefineFunction("text-set-value", 'l', PTIF clipsTextSetValue, "clipsTextSetValue");
  DefineFunction("text-get-value", 's', PTIF clipsTextGetValue, "clipsTextGetValue");

  DefineFunction("text-window-clear", 'l', PTIF clipsTextWindowClear, "clipsTextWindowClear");
  DefineFunction("text-window-create", 'l', PTIF clipsTextWindowCreate, "clipsTextWindowCreate");
  DefineFunction("text-window-load-file", 'l', PTIF clipsTextWindowLoadFile, "clipsTextWindowLoadFile");
  DefineFunction("text-window-save-file", 'l', PTIF clipsTextWindowSaveFile, "clipsTextWindowSaveFile");
  DefineFunction("text-window-modified", 'l', PTIF clipsTextWindowModified, "clipsTextWindowModified");
  DefineFunction("text-window-discard-edits", 'v', PTIF clipsTextWindowDiscardEdits, "clipsTextWindowDiscardEdits");
  DefineFunction("text-window-write", 'v', PTIF clipsTextWindowWrite, "clipsTextWindowWrite");

  DefineFunction("window-add-callback", 'l', PTIF clipsWindowAddCallback, "clipsWindowAddCallback");
  DefineFunction("window-delete", 'l', PTIF clipsWindowDelete, "clipsWindowDelete");
  DefineFunction("window-get-parent", 'l', PTIF clipsWindowGetParent, "clipsWindowGetParent");
  DefineFunction("window-show", 'l', PTIF clipsWindowShow, "clipsWindowShow");

  DefineFunction("window-get-x", 'l', PTIF clipsWindowGetX, "clipsWindowGetX");
  DefineFunction("window-get-y", 'l', PTIF clipsWindowGetY, "clipsWindowGetY");
  DefineFunction("window-get-width", 'l', PTIF clipsWindowGetWidth, "clipsWindowGetWidth");
  DefineFunction("window-get-height", 'l', PTIF clipsWindowGetHeight, "clipsWindowGetHeight");
  DefineFunction("window-get-client-width", 'l', PTIF clipsWindowGetClientWidth, "clipsWindowGetClientWidth");
  DefineFunction("window-get-client-height", 'l', PTIF clipsWindowGetClientHeight, "clipsWindowGetClientHeight");
  DefineFunction("window-set-size", 'l', PTIF clipsWindowSetSize, "clipsWindowSetSize");
  DefineFunction("window-set-client-size", 'l', PTIF clipsWindowSetClientSize, "clipsWindowSetClientSize");
  DefineFunction("window-set-focus", 'l', PTIF clipsWindowSetFocus, "clipsWindowSetFocus");
  DefineFunction("window-centre", 'l', PTIF clipsWindowCentre, "clipsWindowCentre");
  DefineFunction("window-fit", 'l', PTIF clipsWindowFit, "clipsWindowFit");

  /*
   * DDE functions
   *
   */
  DefineFunction("client-create", 'l', PTIF clipsClientCreate, "clipsClientCreate");
  DefineFunction("client-make-connection", 'l', PTIF clipsClientMakeConnection, "clipsClientMakeConnection");

  DefineFunction("server-create", 'l', PTIF clipsServerCreate, "clipsServerCreate");

  DefineFunction("connection-advise", 'l', PTIF clipsConnectionAdvise, "clipsConnectionAdvise");
  DefineFunction("connection-execute", 'l', PTIF clipsConnectionExecute, "clipsConnectionExecute");
  DefineFunction("connection-disconnect", 'l', PTIF clipsConnectionDisconnect, "clipsConnectionDisconnect");
  DefineFunction("connection-poke", 'l', PTIF clipsConnectionPoke, "clipsConnectionPoke");
  DefineFunction("connection-request", 's', PTIF clipsConnectionRequest, "clipsConnectionRequest");
  DefineFunction("connection-start-advise", 'l', PTIF clipsConnectionStartAdvise, "clipsConnectionStartAdvise");
  DefineFunction("connection-stop-advise", 'l', PTIF clipsConnectionStopAdvise, "clipsConnectionStopAdvise");
   
  /*
   * C++ compatibility functions
   *
   */
   
  DefineFunction("declare", 'v', PTIF clipsDeclare, "clipsDeclare");

/*
  DefineFunction("help-load-file", 'l', PTIF clipsHelpLoadFile, "clipsHelpLoadFile");
  DefineFunction("help-display-contents", 'l', PTIF clipsHelpDisplayContents, "clipsHelpDisplayContents");
  DefineFunction("help-display-section", 'l', PTIF clipsHelpDisplaySection, "clipsHelpDisplaySection");
  DefineFunction("help-display-block", 'l', PTIF clipsHelpDisplayBlock, "clipsHelpDisplayBlock");
  DefineFunction("help-keyword-search", 'l', PTIF clipsHelpKeywordSearch, "clipsHelpKeywordSearch");
*/
}
#endif
