/*
 * wxcutils.cc
 * wxCLIPS utility functions
 *
 */

#include "wx.h"
#include "wx_hash.h"
#include "wx_types.h"
#include "wx_ipc.h"

#include <iostream.h>
#include <ctype.h>
#include <stdlib.h>

#include "wx_cmds.h"

// Used to add an object plus type to the hash table
class wxTypedObject: public wxObject
{
 public:
  WXTYPE type;
  wxObject *object;
};

#ifdef USE_CLIPS
#include "clips.h"
#endif

ClipsErrorClass ClipsError;
wxHashTable wxClipsObjects(wxKEY_INTEGER);

ClipsErrorClass& ClipsErrorClass::operator << (char *s)
{
  ClipsErrorFunction(s);
  return *this;
}

ClipsErrorClass& ClipsErrorClass::operator << (char c)
{
  char buf[2];
  buf[0] = c;
  buf[1] = 0;

  ClipsErrorFunction(buf);
  return *this;
}

ClipsErrorClass& ClipsErrorClass::operator << (int i)
{
  char buf[20];
  sprintf(buf, "%d", i);

  ClipsErrorFunction(buf);
  return *this;
}

ClipsErrorClass& ClipsErrorClass::operator << (long i)
{
  char buf[20];
  sprintf(buf, "%ld", i);

  ClipsErrorFunction(buf);
  return *this;
}

ClipsErrorClass& ClipsErrorClass::operator << (double d)
{
  char buf[20];
  sprintf(buf, "%.2f", d);

  ClipsErrorFunction(buf);
  return *this;
}

ClipsErrorClass& ClipsErrorClass::operator << (float f)
{
  char buf[20];
  sprintf(buf, "%.2f", f);

  ClipsErrorFunction(buf);
  return *this;
}

#ifdef USE_CLIPS
void wxInitClips(void)
{
  InitializeCLIPS();
}
#endif

/*
 * Storage/retrieval functions.
 */

void wxStoreThingInTable(long item_id, wxObject *object, WXTYPE type)
{
#ifdef wx_x
  char buf[100];
  wxTypedObject *old_obj = (wxTypedObject *)wxClipsObjects.Get(item_id);
  if (old_obj)
  {
    sprintf(buf, "Warning: hash table clash,  item = %ld, old type = %d, new type = %d", item_id, old_obj->type, type);
    return;
  }
#endif
  wxTypedObject *obj = new wxTypedObject;
  obj->object = object;
  obj->type = type;
  wxClipsObjects.Put(item_id, obj);
}

void wxDeleteThingInTable(long item_id)
{
  wxTypedObject *obj = (wxTypedObject *)wxClipsObjects.Delete(item_id);
  if (obj)
    delete obj;
}

wxObject *wxGetThingFromTable(long item_id, WXTYPE *type)
{
  wxTypedObject *obj = (wxTypedObject *)wxClipsObjects.Get(item_id);
  if (!obj)
    return NULL;

  if (type)
    *type = obj->type;
  return obj->object;
}

wxObject *wxGetTypedObject(long objectId,
                           WXTYPE expectedType)
{
  WXTYPE type;
  wxObject *obj = wxGetThingFromTable(objectId, &type);
  if (!obj)
    return NULL;
  
  if (!wxSubType(type, expectedType))
  {
    char buf[300];
    sprintf(buf, "Type error: %s found where %s expected.\n",
            wxGetTypeName(type), wxGetTypeName(expectedType));
    ClipsError << buf;
    return NULL;
  }
  return obj;
}

/*
 * Style flag parsing. Want to be able to pass C-style
 * bitwise-or'ed flags to CLIPS functions, using strings
 * instead of bit lists, e.g.
 * "wxSYSTEM_MENU | wxBORDER"
 */

char *wxClipsParseWord(char *s, int *i)
{
  if (!s)
    return NULL;

  static char buf[150];
  int len = strlen(s);
  int j = 0;
  int ii = *i;
  while ((ii < len) && (isalpha(s[ii]) || (s[ii] == '_')))
  {
    buf[j] = s[ii];
    j ++;
    ii ++;
  }
  buf[j] = 0;

  // Eat whitespace and conjunction characters
  while ((ii < len) &&
         ((s[ii] == ' ') || (s[ii] == '|') || (s[ii] == ',')))
  {
    ii ++;
  }
  *i = ii;
  if (j == 0)
    return NULL;
  else
    return buf;
}

struct wxClipsBitListStruct
{
  char *word;
  long bits;
};

static wxClipsBitListStruct wxClipsBitListTable[] =
{
  { "wxSINGLE", wxSINGLE },
  { "wxMULTIPLE", wxMULTIPLE },
  { "wxEXTENDED", wxEXTENDED },
  { "wxNEEDED_SB", wxNEEDED_SB },
  { "wxALWAYS_SB", wxALWAYS_SB },
  { "wxVSCROLL", wxVSCROLL },
  { "wxHSCROLL", wxHSCROLL },
  { "wxCAPTION", wxCAPTION },
  { "wxABSOLUTE_POSITIONING", wxABSOLUTE_POSITIONING },
  { "wxSTAY_ON_TOP", wxSTAY_ON_TOP},
  { "wxICONIZE", wxICONIZE},
  { "wxMINIMIZE", wxICONIZE},
  { "wxMAXIMIZE", wxMAXIMIZE},
  { "wxSDI", wxSDI},
  { "wxMDI_PARENT", wxMDI_PARENT},
  { "wxMDI_CHILD", wxMDI_CHILD},
  { "wxTHICK_FRAME", wxTHICK_FRAME},
  { "wxSYSTEM_MENU", wxSYSTEM_MENU},
  { "wxMINIMIZE_BOX", wxMINIMIZE_BOX},
  { "wxMAXIMIZE_BOX", wxMAXIMIZE_BOX},
  { "wxRESIZE_BOX", wxRESIZE_BOX},
  { "wxDEFAULT_FRAME", wxDEFAULT_FRAME},
  { "wxBORDER", wxBORDER},
  { "wxRETAINED", wxRETAINED},
  { "wxEDITABLE", wxEDITABLE},
  { "wxNATIVE_IMPL", wxNATIVE_IMPL},
  { "wxEXTENDED_IMPL", wxEXTENDED_IMPL},
  { "wxBACKINGSTORE", wxBACKINGSTORE},
  { "wxFLAT", wxFLAT},
  { "wxMOTIF_RESIZE", wxMOTIF_RESIZE},
  { "wxXVIEW_CMD", wxXVIEW_CMD},
  { "wxFIXED_LENGTH", wxFIXED_LENGTH},

  // Enhanced dialog (not yet implemented in wxCLIPS)
  { "wxBOTTOM_COMMANDS", wxBOTTOM_COMMANDS},
  { "wxRIGHT_COMMANDS", wxRIGHT_COMMANDS},
  { "wxSTATUS_FOOTER", wxSTATUS_FOOTER},
  { "wxNO_STATUS_FOOTER", wxNO_STATUS_FOOTER},
  { "wxNO_CANCEL_BUTTON", wxNO_CANCEL_BUTTON},
  { "wxCANCEL_BUTTON_FIRST", wxCANCEL_BUTTON_FIRST},
  { "wxCANCEL_BUTTON_SECOND", wxCANCEL_BUTTON_SECOND},
  { "wxCANCEL_BUTTON_LAST", wxCANCEL_BUTTON_LAST},
  { "wxCANCEL_BUTTON_FIRST", wxCANCEL_BUTTON_FIRST},
  { "wxCANCEL_BUTTON_FIRST", wxCANCEL_BUTTON_FIRST},
  { "wxENH_DEFAULT", wxENH_DEFAULT},
  { "wxCOLOURED", wxCOLOURED},

  // Text font families
  { "wxDEFAULT", wxDEFAULT},
  { "wxDECORATIVE", wxDECORATIVE},
  { "wxROMAN", wxROMAN},
  { "wxSCRIPT", wxSCRIPT},
  { "wxSWISS", wxSWISS},
  { "wxMODERN", wxMODERN},
  { "wxTELETYPE", wxTELETYPE},
  { "wxVARIABLE", wxVARIABLE},
  { "wxFIXED", wxFIXED},
  { "wxNORMAL", wxNORMAL},
  { "wxLIGHT", wxLIGHT},
  { "wxBOLD", wxBOLD},
  { "wxITALIC", wxITALIC},
  { "wxSLANT", wxSLANT},
  { "wxSOLID", wxSOLID},
  { "wxDOT", wxDOT},
  { "wxLONG_DASH", wxLONG_DASH},
  { "wxSHORT_DASH", wxSHORT_DASH},
  { "wxDOT_DASH", wxDOT_DASH},
  { "wxUSER_DASH", wxUSER_DASH},
  { "wxTRANSPARENT", wxTRANSPARENT},
  { "wxSTIPPLE", wxSTIPPLE},
  { "wxBDIAGONAL_HATCH", wxBDIAGONAL_HATCH},
  { "wxCROSSDIAG_HATCH", wxCROSSDIAG_HATCH},
  { "wxFDIAGONAL_HATCH", wxFDIAGONAL_HATCH},
  { "wxCROSS_HATCH", wxCROSS_HATCH},
  { "wxHORIZONTAL_HATCH", wxHORIZONTAL_HATCH},
  { "wxVERTICAL_HATCH", wxVERTICAL_HATCH},
  { "wxJOIN_BEVEL", wxJOIN_BEVEL},
  { "wxJOIN_MITER", wxJOIN_MITER},
  { "wxJOIN_ROUND", wxJOIN_ROUND},
  { "wxCAP_ROUND", wxCAP_ROUND},
  { "wxCAP_PROJECTING", wxCAP_PROJECTING},
  { "wxCAP_BUTT", wxCAP_BUTT},

  // Logical ops
  { "wxCLEAR", wxCLEAR},
  { "wxXOR", wxXOR},
  { "wxINVERT", wxINVERT},
  { "wxOR_REVERSE", wxOR_REVERSE},
  { "wxAND_REVERSE", wxAND_REVERSE},
  { "wxCOPY", wxCOPY},
  { "wxAND", wxAND},
  { "wxAND_INVERT", wxAND_INVERT},
  { "wxNO_OP", wxNO_OP},
  { "wxNOR", wxNOR},
  { "wxEQUIV", wxEQUIV},
  { "wxSRC_INVERT", wxSRC_INVERT},
  { "wxOR_INVERT", wxOR_INVERT},
  { "wxNAND", wxNAND},
  { "wxOR", wxOR},
  { "wxSET", wxSET},

  { "wxFLOOD_SURFACE", wxFLOOD_SURFACE},
  { "wxFLOOD_BORDER", wxFLOOD_BORDER},
  { "wxODDEVEN_RULE", wxODDEVEN_RULE},
  { "wxWINDING_RULE", wxWINDING_RULE},
  { "wxHORIZONTAL", wxHORIZONTAL},
  { "wxVERTICAL", wxVERTICAL},
  { "wxBOTH", wxBOTH},
  { "wxCENTER_FRAME", wxCENTER_FRAME},
  { "wxOK", wxOK},
  { "wxYES_NO", wxYES_NO},
  { "wxCANCEL", wxCANCEL},
  { "wxYES", wxYES},
  { "wxNO", wxNO},
  { "wxICON_EXCLAMATION", wxICON_EXCLAMATION},
  { "wxICON_HAND", wxICON_HAND},
  { "wxICON_QUESTION", wxICON_QUESTION},
  { "wxICON_INFORMATION", wxICON_INFORMATION},
  { "wxICON_STOP", wxICON_STOP},
  { "wxICON_ASTERISK", wxICON_ASTERISK},
  { "wxICON_MASK", wxICON_MASK},
  { "wxCENTRE", wxCENTRE},
  { "wxCENTER", wxCENTRE}
};

static int wxClipsBitListCount = (sizeof(wxClipsBitListTable)/sizeof(wxClipsBitListStruct));

long wxClipsParseBitList(char *bitListString)
{
  int i = 0;
  char *word;
  long bitList = 0;
  while (word = wxClipsParseWord(bitListString, &i))
  {
    Bool found = FALSE;
    for (int j = 0; j < wxClipsBitListCount; j++)
      if (strcmp(wxClipsBitListTable[j].word, word) == 0)
      {
        bitList |= wxClipsBitListTable[j].bits;
        found = TRUE;
        break;
      }
    if (!found)
    {
      ClipsError << "Unrecognised flag " << word << "\n";
    }
  }
  return bitList;
}

