   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*             CLIPS Version 5.10  07/17/91            */
   /*                                                     */
   /*                    SYMBOL MODULE                    */
   /*******************************************************/

/*************************************************************/
/* Purpose:                                                  */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Gary D. Riley                                        */
/*                                                           */
/* Contributing Programmer(s):                               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/

#define _SYMBOL_SOURCE_

#include <stdio.h>
#define _CLIPS_STDIO_
#include <string.h>

#include "setup.h"
#include "constant.h"
#include "symbol.h"
#include "clipsmem.h"
#include "router.h"
#include "utility.h"
#include "evaluatn.h"
    
#define AVERAGE_STRING_SIZE 10
#define NUMBER_OF_LONGS_FOR_HASH 25

/**********************************************************/
/* EPHEMERON STRUCTURE: Data structure used to keep track */
/*   of ephemeral symbols, floats, and integers.          */
/*                                                        */
/*   associatedValue: Contains a pointer to the storage   */
/*   structure for the symbol, float, or integer which is */
/*   ephemeral.                                           */
/*                                                        */
/*   next: Contains a pointer to the next ephemeral item  */
/*   in a list of ephemeral items.                        */
/**********************************************************/
struct ephemeron
  {
   VOID *associatedValue;
   struct ephemeron *next;
  };

/***************************************/
/* LOCAL INTERNAL FUNCTION DEFINITIONS */
/***************************************/

#if ANSI_COMPILER  
   static VOID                    RemoveSymbol(SYMBOL_HN *);
   static VOID                    RemoveFloat(FLOAT_HN *);
   static VOID                    RemoveInteger(INTEGER_HN *);
   static VOID                    AddEphemeralSymbol(SYMBOL_HN *);
   static VOID                    AddEphemeralFloat(FLOAT_HN *);
   static VOID                    AddEphemeralInteger(INTEGER_HN *); 
   static VOID                    RemoveEphemeralSymbols(void);
   static VOID                    RemoveEphemeralFloats(void);
   static VOID                    RemoveEphemeralIntegers(void);
#else
   static VOID                    RemoveSymbol();
   static VOID                    RemoveFloat();
   static VOID                    RemoveInteger();
   static VOID                    AddEphemeralSymbol();
   static VOID                    AddEphemeralFloat();
   static VOID                    AddEphemeralInteger(); 
   static VOID                    RemoveEphemeralSymbols();
   static VOID                    RemoveEphemeralFloats();
   static VOID                    RemoveEphemeralIntegers();
#endif

/****************************************/
/* GLOBAL INTERNAL VARIABLE DEFINITIONS */
/****************************************/

   globle VOID               *CLIPSTrueSymbol;
   globle VOID               *CLIPSFalseSymbol;
   
/***************************************/
/* LOCAL INTERNAL VARIABLE DEFINITIONS */
/***************************************/

   static SYMBOL_HN         **SymbolTable;
   static FLOAT_HN          **FloatTable;
   static INTEGER_HN        **IntegerTable; 
   static struct ephemeron   *EphemeralSymbolList = NULL;
   static struct ephemeron   *EphemeralFloatList = NULL;
   static struct ephemeron   *EphemeralIntegerList = NULL;
   static char               *FalseSymbol = "FALSE";
   static char               *TrueSymbol = "TRUE";

/*####################################################################*/
/*####################################################################*/
/*###                                                              ###*/
/*###                      HASHING FUNCTIONS                       ###*/
/*###                                                              ###*/
/*####################################################################*/
/*####################################################################*/
  
/********************************************************************/
/* AddSymbol:  Searches for the string in the symbol table. If the  */ 
/*   string is already in the symbol table, then the address of the */
/*   string's location in the symbol table is returned.  Otherwise, */
/*   the string is added to the symbol table and then the address   */
/*   of the string's location in the symbol table is returned.      */
/********************************************************************/
globle VOID *AddSymbol(str)
   char *str;
   {
    int tally, length;
    SYMBOL_HN *past = NULL, *peek;
    
    /*====================================*/
    /* Get the hash value for the string. */
    /*====================================*/

    if (str == NULL)
      {
       CLIPSSystemError("SYMBOL",1);
       ExitCLIPS(5);
      }

    tally = HashSymbol(str,SYMBOL_HASH_SIZE);
    peek = SymbolTable[tally];

    /*====================================================*/
    /* If the symbol table entry for the string is empty, */
    /* then place the string at that location, and        */
    /* return the address of the string.                  */
    /*====================================================*/

    if (peek == NULL)
      {
       peek = get_struct(symbolHashNode); 
       SymbolTable[tally] = peek;
       length = strlen(str) + 1;
       peek->contents = (char *) gm2(length);
       peek->next = NULL;
       peek->bucket = tally;
       peek->count = 0;
       peek->relatedSymbol = NULL;
       strcpy(peek->contents,str);
       AddEphemeralSymbol(peek);
       peek->depth = CurrentEvaluationDepth;
       return((VOID *) peek);
      }

    /*==================================================*/
    /* Search for the string in the list of entries for */ 
    /* this symbol table location.  If the string is    */
    /* found, then return the address of the string.    */
    /*==================================================*/

    while (peek != NULL)
      {
       if (strcmp(str,peek->contents) == 0)
         { return((VOID *) peek); }
       past = peek;
       peek = peek->next;
      }

    /*==================================================*/
    /* Add the string at the end of the list of entries */
    /* for this symbol table location.  Return the      */
    /* address of the string.                           */
    /*==================================================*/

    past->next = get_struct(symbolHashNode);
    peek = past->next;
    length = strlen(str) + 1;
    peek->contents = (char *) gm2(length);
    peek->next = NULL;
    peek->bucket = tally;
    peek->count = 0;
    peek->relatedSymbol = NULL;
    strcpy(peek->contents,str);
    
    AddEphemeralSymbol(peek);
    peek->depth = CurrentEvaluationDepth;
    return((VOID *) peek);
   }
   
/****************************************************************/
/* FindSymbol:  Searches for the string in the symbol table and */
/*   returns a pointer to it if found, otherwise returns NULL.  */
/****************************************************************/
globle SYMBOL_HN *FindSymbol(str)
   char *str;
   {
    int tally;
    SYMBOL_HN *peek;

    tally = HashSymbol(str,SYMBOL_HASH_SIZE);
    peek = SymbolTable[tally];

    while (peek != NULL)
      {
       if (strcmp(str,peek->contents) == 0) return(peek);
       peek = peek->next;
      }

    return(NULL);
   }

/*******************************************************************/
/* AddDouble:  Searches for the double in the hash table. If the   */ 
/*   double is already in the hash table, then the address of the  */
/*   double is returned.  Otherwise, the double is hashed into the */
/*   table and the address of the double is also returned.         */
/*******************************************************************/
globle VOID *AddDouble(number)
   double number;
   {
    int tally;
    FLOAT_HN *past = NULL, *peek;
    
    /*====================================*/
    /* Get the hash value for the double. */
    /*====================================*/

    tally = HashFloat(number,FLOAT_HASH_SIZE);
    peek = FloatTable[tally];

    /*==================================================*/
    /* If the hash table entry for the double is empty, */
    /* then place the double at that location, and      */
    /* return the address of the double.                */
    /*==================================================*/

    if (peek == NULL)
      {
       peek = get_struct(floatHashNode); 
       FloatTable[tally] = peek;
       peek->contents = number;
       peek->next = NULL;
       peek->bucket = tally;
       peek->count = 0;
       AddEphemeralFloat(peek);
       peek->depth = CurrentEvaluationDepth;
       return((VOID *) peek);
      }

    /*==================================================*/
    /* Search for the double in the list of entries for */ 
    /* this hash location.  If the double is found,     */
    /* then return the address of the double.           */
    /*==================================================*/

    while (peek != NULL)
      {
       if (number == peek->contents)
         { return((VOID *) peek); }
       past = peek;
       peek = peek->next;
      }

    /*==================================================*/
    /* Add the double at the end of the list of entries */
    /* for this hash location.  Return the address of   */
    /* the double.                                      */
    /*==================================================*/

    past->next = get_struct(floatHashNode);
    peek = past->next;
    peek->contents = number;
    peek->next = NULL;
    peek->bucket = tally;
    peek->count = 0;

    AddEphemeralFloat(peek);
    peek->depth = CurrentEvaluationDepth;
    return((VOID *) peek);
   }
  
/****************************************************************/
/* AddLong:  Searches for the long in the hash table. If the    */ 
/*   long is already in the hash table, then the address of the */
/*   long is returned.  Otherwise, the long is hashed into the  */
/*   table and the address of the long is also returned.        */
/****************************************************************/
globle VOID *AddLong(number)
   long int number;
   {
    int tally;
    INTEGER_HN *past = NULL, *peek;
    
    /*==================================*/
    /* Get the hash value for the long. */
    /*==================================*/

    tally = HashInteger(number,INTEGER_HASH_SIZE);
    peek = IntegerTable[tally];

    /*================================================*/
    /* If the hash table entry for the long is empty, */
    /* then place the long at that location, and      */
    /* return the address of the long.                */
    /*================================================*/

    if (peek == NULL)
      {
       peek = get_struct(integerHashNode); 
       IntegerTable[tally] = peek;
       peek->contents = number;
       peek->next = NULL;
       peek->bucket = tally;
       peek->count = 0;
       AddEphemeralInteger(peek);
       peek->depth = CurrentEvaluationDepth;
       return((VOID *) peek);
      }

    /*================================================*/
    /* Search for the long in the list of entries for */ 
    /* this hash location.  If the long is found,     */
    /* then return the address of the long.           */
    /*================================================*/

    while (peek != NULL)
      {
       if (number == peek->contents)
         { return((VOID *) peek); }
       past = peek;
       peek = peek->next;
      }

    /*================================================*/
    /* Add the long at the end of the list of entries */
    /* for this hash location.  Return the address of */
    /* the long.                                      */
    /*================================================*/

    past->next = get_struct(integerHashNode);
    peek = past->next;
    peek->contents = number;
    peek->next = NULL;
    peek->bucket = tally;
    peek->count = 0;

    AddEphemeralInteger(peek);
    peek->depth = CurrentEvaluationDepth;
    return((VOID *) peek);
   }
 
/********************************************************************/
/* InitializeAtomTables: Initializes the SymbolTable, IntegerTable, */
/*   and FloatTable. It also initializes the CLIPSTrueSymbol and    */
/*   CLIPSFalseSymbol.                                              */
/********************************************************************/
globle VOID InitializeAtomTables()
   {
    int i;

    SymbolTable = (SYMBOL_HN **) 
                   gm2((int) sizeof (SYMBOL_HN *) * SYMBOL_HASH_SIZE);
                   
    FloatTable = (FLOAT_HN **) 
                   gm2((int) sizeof (FLOAT_HN *) * FLOAT_HASH_SIZE);
                   
    IntegerTable = (INTEGER_HN **) 
                    gm2((int) sizeof (INTEGER_HN *) * INTEGER_HASH_SIZE);
      
    for (i = 0; i < SYMBOL_HASH_SIZE; i++) SymbolTable[i] = NULL;
    for (i = 0; i < FLOAT_HASH_SIZE; i++) FloatTable[i] = NULL;
    for (i = 0; i < INTEGER_HASH_SIZE; i++) IntegerTable[i] = NULL;
    
    CLIPSTrueSymbol = AddSymbol(TrueSymbol);
    IncrementSymbolCount(CLIPSTrueSymbol);
    CLIPSFalseSymbol = AddSymbol(FalseSymbol);
    IncrementSymbolCount(CLIPSFalseSymbol);
   }
  
/***************************************************/
/* HashSymbol: Computes a hash value for a symbol. */
/***************************************************/
globle int HashSymbol(word,range)
  char *word;
  int range;
  {
   register int k,j,i;
   register int length;
   int tally;
   unsigned long count = 0L,tmpLong;
   char *tmpPtr;
   
   tmpPtr = (char *) &tmpLong;
   
   /*===============================================*/
   /* Count the number of characters in the symbol. */
   /*===============================================*/
   
   for (length = 0; word[length]; length++);
   
   /*================================================================ */
   /* Add up the first part of the word as unsigned long int values.  */
   /*================================================================ */
   
   length = length / sizeof(unsigned long);
   for (i = 0 , j = 0 ; i < length; i++) 
     {
      for (k = 0 ; k < sizeof(unsigned long) ; k++ , j++)
        tmpPtr[k] = word[j];
      count += tmpLong;
     }
      
   /*============================================*/
   /* Add the remaining characters to the count. */
   /*============================================*/
   
   for (word = (char *) &word[j]; *word; word++) count += (unsigned long) *word;
   
   /*========================*/
   /* Return the hash value. */
   /*========================*/
   
   tally = (int) (count % range);
   if (tally < 0) return(-tally);
   
   return(tally); 
  }
  
/***************************************************/
/* HashFloat:  Computes a hash value for an float. */
/***************************************************/
globle int HashFloat(number,range)
  double number;
  int range;
  {
   union
     {
      double fv;
      unsigned long int liv;
     } fis;
   unsigned long count = 0;
   int tally;
   
   fis.liv = 0;
   fis.fv = number;
   count = fis.liv;
   
   tally = (int) (count % range);
   
   if (tally < 0) return(-tally);
   
   return(tally); 
  }
  
/******************************************************/
/* HashInteger: Computes a hash value for an integer. */
/******************************************************/
globle int HashInteger(number,range)
  long int number;
  int range;
  {
   int tally;

   tally = (int) (number % range);
   
   if (tally < 0) return(-tally);
   
   return(tally); 
  }

/*****************************************************************************/
/* IncrementSymbolCount: Increments the count value for a SymbolTable entry. */
/*****************************************************************************/
globle VOID IncrementSymbolCount(hash_loc)
  SYMBOL_HN *hash_loc;
  {
   if (hash_loc->count < 0)
     {
      CLIPSSystemError("SYMBOL",2);
      ExitCLIPS(5);
     }

   hash_loc->count++;
  }
  
/***************************************************************************/
/* IncrementFloatCount: Increments the count value for a FloatTable entry. */
/***************************************************************************/
globle VOID IncrementFloatCount(hash_loc)
  FLOAT_HN *hash_loc;
  {
   if (hash_loc->count < 0)
     {
      CLIPSSystemError("SYMBOL",3);
      ExitCLIPS(5);
     }

   hash_loc->count++;
  }
  
/*******************************************************************************/
/* IncrementIntegerCount: Increments the count value for a IntegerTable entry. */
/*******************************************************************************/
globle VOID IncrementIntegerCount(hash_loc)
  INTEGER_HN *hash_loc;
  {
   if (hash_loc->count < 0)
     {
      CLIPSSystemError("SYMBOL",4);
      ExitCLIPS(5);
     }

   hash_loc->count++;
  }

/*****************************************************************************/
/* DecrementSymbolCount: Decrements the count value for a SymbolTable entry. */
/*   Adds the symbol to the EphemeralSymbolList if the count becomes zero.   */ 
/*****************************************************************************/
globle VOID DecrementSymbolCount(hash_loc)
  SYMBOL_HN *hash_loc;
  {
   if (hash_loc->count < 0)
     {
      CLIPSSystemError("SYMBOL",5);
      ExitCLIPS(5);
     }

   if (hash_loc->count == 0)
     {
      CLIPSSystemError("SYMBOL",6);
      ExitCLIPS(5);
     }

   hash_loc->count--;

   if (hash_loc->count != 0) return;

   if (hash_loc->markedEphemeral == CLIPS_FALSE) AddEphemeralSymbol(hash_loc);

   return;
  }
  
/***************************************************************************/
/* DecrementFloatCount: Decrements the count value for a FloatTable entry. */
/*   Adds the float to the EphemeralFloatList if the count becomes zero.   */ 
/***************************************************************************/
globle VOID DecrementFloatCount(hash_loc)
  FLOAT_HN *hash_loc;
  {
   if (hash_loc->count <= 0)
     {
      CLIPSSystemError("SYMBOL",7);
      ExitCLIPS(5);
     }

   hash_loc->count--;

   if (hash_loc->count != 0) return;

   if (hash_loc->markedEphemeral == CLIPS_FALSE) AddEphemeralFloat(hash_loc);

   return;
  }
  
/*******************************************************************************/
/* DecrementIntegerCount: Decrements the count value for a IntegerTable entry. */
/*   Adds the integer to the EphemeralIntegerList if the count becomes zero.   */  
/*******************************************************************************/
globle VOID DecrementIntegerCount(hash_loc)
  INTEGER_HN *hash_loc;
  {
   if (hash_loc->count <= 0)
     {
      CLIPSSystemError("SYMBOL",8);
      ExitCLIPS(5);
     }

   hash_loc->count--;

   if (hash_loc->count != 0) return;

   if (hash_loc->markedEphemeral == CLIPS_FALSE)  AddEphemeralInteger(hash_loc);

   return;
  }


/********************************************************/
/* RemoveSymbol: Removes a symbol from the SymbolTable. */ 
/********************************************************/
static VOID RemoveSymbol(hash_loc)
  SYMBOL_HN *hash_loc;
  {
   SYMBOL_HN *prev_hash, *check_hash;

   prev_hash = NULL;
   check_hash = SymbolTable[hash_loc->bucket];

   while (check_hash != hash_loc)
     {
      prev_hash = check_hash;
      check_hash = check_hash->next;

      if (check_hash == NULL)
        {
         CLIPSSystemError("SYMBOL",9);
         ExitCLIPS(5);
        }
     }
     
   if (prev_hash == NULL)
     { SymbolTable[hash_loc->bucket] = hash_loc->next; }
   else
     { prev_hash->next = check_hash->next; }
   
   rm(hash_loc->contents,(int) strlen(hash_loc->contents) + 1);
   rtn_struct(symbolHashNode,hash_loc);
  }
  
/*****************************************************/
/* RemoveFloat: Removes a float from the FloatTable. */ 
/*****************************************************/
static VOID RemoveFloat(hash_loc)
  FLOAT_HN *hash_loc;
  {
   FLOAT_HN *prev_hash, *check_hash;

   prev_hash = NULL;
   check_hash = FloatTable[hash_loc->bucket];

   while (check_hash != hash_loc)
     {
      prev_hash = check_hash;
      check_hash = check_hash->next;

      if (check_hash == NULL)
        {
         CLIPSSystemError("SYMBOL",10);
         ExitCLIPS(5);
        }
     }
     
   if (prev_hash == NULL)
     { FloatTable[hash_loc->bucket] = hash_loc->next; }
   else
     { prev_hash->next = check_hash->next; }
   
   rtn_struct(floatHashNode,hash_loc);
  }
  
/************************************************************/
/* RemoveInteger: Removes an integer from the IntegerTable. */
/************************************************************/
static VOID RemoveInteger(hash_loc)
  INTEGER_HN *hash_loc;
  {
   INTEGER_HN *prev_hash, *check_hash;

   prev_hash = NULL;
   check_hash = IntegerTable[hash_loc->bucket];

   while (check_hash != hash_loc)
     {
      prev_hash = check_hash;
      check_hash = check_hash->next;

      if (check_hash == NULL)
        {
         CLIPSSystemError("SYMBOL",11);
         ExitCLIPS(5);
        }
     }
     
   if (prev_hash == NULL)
     { IntegerTable[hash_loc->bucket] = hash_loc->next; }
   else
     { prev_hash->next = check_hash->next; }
   
   rtn_struct(integerHashNode,hash_loc);
  }
  
/**************************************************************/
/* AddEphemeralSymbol: Adds a SymbolTable entry to the list   */
/*   of ephemeral symbols.  These locations have a zero count */
/*   indicating that no structure is using the symbol.        */
/**************************************************************/
static VOID AddEphemeralSymbol(theSymbol)
  SYMBOL_HN *theSymbol;  
  {
   struct ephemeron *temp;
     
   if (theSymbol->count != 0)
     {
      CLIPSSystemError("SYMBOL",12);
      ExitCLIPS(5);
     }

   theSymbol->markedEphemeral = CLIPS_TRUE;
   
   temp = get_struct(ephemeron);
   temp->associatedValue = (VOID *) theSymbol;
   temp->next = EphemeralSymbolList;
/*
   if (CurrentEvaluationDepth >= 0) theSymbol->depth = CurrentEvaluationDepth;
   else theSymbol->depth = 0;
*/
   EphemeralSymbolList = temp;

   EphemeralItemCount++;
   EphemeralItemSize += sizeof(struct ephemeron) + sizeof(SYMBOL_HN) + 
                        AVERAGE_STRING_SIZE;
  }
  
/*************************************************************/
/* AddEphemeralFloat: Adds a FloatTable entry to the list    */
/*   of ephemeral floats.  These locations have a zero count */
/*   indicating that no structure is using the float.        */
/*************************************************************/
static VOID AddEphemeralFloat(theFloat)
  FLOAT_HN *theFloat;  
  {
   struct ephemeron *temp;
     
   if (theFloat->count != 0)
     {
      CLIPSSystemError("SYMBOL",13);
      ExitCLIPS(5);
     }

   theFloat->markedEphemeral = CLIPS_TRUE;
   
   temp = get_struct(ephemeron);
   temp->associatedValue = (VOID *) theFloat;
   temp->next = EphemeralFloatList;
/*
   if (CurrentEvaluationDepth >= 0) theFloat->depth = CurrentEvaluationDepth;
   else theFloat->depth = 0;
*/
   EphemeralFloatList = temp;

   EphemeralItemCount++;
   EphemeralItemSize += sizeof(struct ephemeron) + sizeof(FLOAT_HN);
  }
  
/***************************************************************/
/* AddEphemeralInteger: Adds an IntegerTable entry to the list */
/*   of ephemeral integers.  These locations have a zero count */
/*   indicating that no structure is using the integer.        */
/***************************************************************/
static VOID AddEphemeralInteger(theInteger)
  INTEGER_HN *theInteger;  
  {
   struct ephemeron *temp;
     
   if (theInteger->count != 0)
     {
      CLIPSSystemError("SYMBOL",14);
      ExitCLIPS(5);
     }

   theInteger->markedEphemeral = CLIPS_TRUE;
   
   temp = get_struct(ephemeron);
   temp->associatedValue = (VOID *) theInteger;
   temp->next = EphemeralIntegerList;
/*
   if (CurrentEvaluationDepth >= 0) theInteger->depth = CurrentEvaluationDepth;
   else theInteger->depth = 0;
*/
   EphemeralIntegerList = temp;

   EphemeralItemCount++;
   EphemeralItemSize += sizeof(struct ephemeron) + sizeof(INTEGER_HN);
  }

/*************************************************************/
/* RemoveEphemeralAtoms: Causes the removal of all ephemeral */
/*   symbols, integers, and floats, that still have a count  */
/*   value of zero, from their respective storage tables.    */
/*************************************************************/
globle VOID RemoveEphemeralAtoms() 
  {
   RemoveEphemeralSymbols();
   RemoveEphemeralFloats();
   RemoveEphemeralIntegers();
  }
  
/**************************************************************/
/* RemoveEphemeralSymbols: Removes symbols from the ephemeral */
/*   symbol list that have a count of zero and were placed on */
/*   the list at a higher level than the current evaluation   */
/*   depth. Since symbols are ordered in the list in          */
/*   descending order, the removal process can end when a     */
/*   depth is reached less than the current evaluation depth. */
/*   Because ephemeral symbols can be "pulled" up through an  */
/*   evaluation depth, this routine needs to check through    */
/*   both the previous and current evaluation depth.          */
/**************************************************************/  
static VOID RemoveEphemeralSymbols()
  {
   struct ephemeron *edPtr, *lastPtr = NULL, *nextPtr;
   
   edPtr = EphemeralSymbolList;
   while (edPtr != NULL)
     {
      /*======================================================*/
      /* Check through previous and current evaluation depth  */
      /* because these symbols can be interspersed, otherwise */
      /* symbols are stored in descending evaluation depth.   */
      /*======================================================*/
      
      nextPtr = edPtr->next;
      
      /*==================================================*/
      /* Remove any symbols that have a count of zero and */
      /* were added to the ephemeral list at a higher     */
      /* evaluation depth.                                */
      /*==================================================*/
      
      if ((((SYMBOL_HN *) edPtr->associatedValue)->count == 0) &&
          (((SYMBOL_HN *) edPtr->associatedValue)->depth > CurrentEvaluationDepth))
        { 
         RemoveSymbol((SYMBOL_HN *) edPtr->associatedValue);
         rtn_struct(ephemeron,edPtr);
         if (lastPtr == NULL) EphemeralSymbolList = nextPtr;
         else lastPtr->next = nextPtr;
         EphemeralItemCount--;
         EphemeralItemSize -= sizeof(struct ephemeron) + sizeof(SYMBOL_HN) + 
                              AVERAGE_STRING_SIZE;
        }
        
      /*=======================================*/
      /* Remove ephemeral status of any symbol */
      /* with a count greater than zero.       */
      /*=======================================*/
                 
      else if (((SYMBOL_HN *) edPtr->associatedValue)->count > 0) 
        {
         ((SYMBOL_HN *) edPtr->associatedValue)->markedEphemeral = CLIPS_FALSE; 
         rtn_struct(ephemeron,edPtr);
         if (lastPtr == NULL) EphemeralSymbolList = nextPtr;
         else lastPtr->next = nextPtr;
         EphemeralItemCount--;
         EphemeralItemSize -= sizeof(struct ephemeron) + sizeof(SYMBOL_HN) + 
                              AVERAGE_STRING_SIZE;
        }
        
      /*==================================================*/
      /* Otherwise keep the symbol in the ephemeral list. */
      /*==================================================*/
      
      else
        { lastPtr = edPtr; }
        
      edPtr = nextPtr;
     }
  }
  
/*****************************************************/
/* RemoveEphemeralFloats: Similar in nature to the   */
/*   function RemoveEphemeralSymbols, but it removes */
/*   ephemeral floating point numbers.               */
/*****************************************************/  
static VOID RemoveEphemeralFloats()
  {
   struct ephemeron *edPtr, *lastPtr = NULL, *nextPtr;
   
   edPtr = EphemeralFloatList;
   while (edPtr != NULL)
     { 
      nextPtr = edPtr->next;
      
      if ((((FLOAT_HN *) edPtr->associatedValue)->count == 0) &&
          (((FLOAT_HN *) edPtr->associatedValue)->depth > CurrentEvaluationDepth))
        { 
         RemoveFloat((FLOAT_HN *) edPtr->associatedValue);
         rtn_struct(ephemeron,edPtr);
         if (lastPtr == NULL) EphemeralFloatList = nextPtr;
         else lastPtr->next = nextPtr;
         EphemeralItemCount--;
         EphemeralItemSize -= sizeof(struct ephemeron) + sizeof(FLOAT_HN);
        }
      else if (((FLOAT_HN *) edPtr->associatedValue)->count > 0) 
        {
         ((FLOAT_HN *) edPtr->associatedValue)->markedEphemeral = CLIPS_FALSE; 
         rtn_struct(ephemeron,edPtr);
         if (lastPtr == NULL) EphemeralFloatList = nextPtr;
         else lastPtr->next = nextPtr;
         EphemeralItemCount--;
         EphemeralItemSize -= sizeof(struct ephemeron) + sizeof(FLOAT_HN);
        }      
      else
        { lastPtr = edPtr; }
        
      edPtr = nextPtr;
     }
  }
  
/*****************************************************/
/* RemoveEphemeralIntegers: Similar in nature to the */
/*   function RemoveEphemeralSymbols, but it removes */
/*   ephemeral integers.                             */
/*****************************************************/  
static VOID RemoveEphemeralIntegers()
  {
   struct ephemeron *edPtr, *lastPtr = NULL, *nextPtr;
   
   edPtr = EphemeralIntegerList;
   while (edPtr != NULL)
     {
      nextPtr = edPtr->next;
      
      if ((((INTEGER_HN *) edPtr->associatedValue)->count == 0) &&
          (((INTEGER_HN *) edPtr->associatedValue)->depth > CurrentEvaluationDepth))
        { 
         RemoveInteger((INTEGER_HN *) edPtr->associatedValue);
         rtn_struct(ephemeron,edPtr);
         if (lastPtr == NULL) EphemeralIntegerList = nextPtr;
         else lastPtr->next = nextPtr;
         EphemeralItemCount--;
         EphemeralItemSize -= sizeof(struct ephemeron) + sizeof(INTEGER_HN);
        }
      else if (((INTEGER_HN *) edPtr->associatedValue)->count > 0) 
        {
         ((INTEGER_HN *) edPtr->associatedValue)->markedEphemeral = CLIPS_FALSE; 
         rtn_struct(ephemeron,edPtr);
         if (lastPtr == NULL) EphemeralIntegerList = nextPtr;
         else lastPtr->next = nextPtr;
         EphemeralItemCount--;
         EphemeralItemSize -= sizeof(struct ephemeron) + sizeof(INTEGER_HN);
        }      
      else
        { lastPtr = edPtr; }
        
      edPtr = nextPtr;
     }
  }

/********************/
/* GetSymbolTable   */
/********************/
globle SYMBOL_HN **GetSymbolTable()
  {
   return(SymbolTable);
  }

/********************/
/* SetSymbolTable */
/********************/
globle VOID SetSymbolTable(value)
  SYMBOL_HN **value;
  {
   SymbolTable = value;
  }

/********************/
/* GetFloatTable   */
/********************/
globle FLOAT_HN **GetFloatTable()
  {
   return(FloatTable);
  }

/********************/
/* SetFloatTable */
/********************/
globle VOID SetFloatTable(value)
  FLOAT_HN **value;
  {
   FloatTable = value;
  }

/********************/
/* GetIntegerTable   */
/********************/
globle INTEGER_HN **GetIntegerTable()
  {
   return(IntegerTable);
  }

/********************/
/* SetIntegerTable */
/********************/
globle VOID SetIntegerTable(value)
  INTEGER_HN **value;
  {
   IntegerTable = value;
  }
  
/**************************/
/* RefreshBooleanSymbols: */
/***************************/
globle VOID RefreshBooleanSymbols()
  {
   CLIPSTrueSymbol = (VOID *) FindSymbol(TrueSymbol);
   CLIPSFalseSymbol = (VOID *) FindSymbol(FalseSymbol);
  }
  
/***************************************/
/* FindSymbolMatches:                  */
/***************************************/
globle struct symbolMatch *FindSymbolMatches(searchString,numberOfMatches)
  char *searchString;
  int *numberOfMatches;
  {
   struct symbolMatch *reply = NULL, *temp;
   struct symbolHashNode *hashPtr = NULL;
   int searchLength;
   
   searchLength = strlen(searchString);
   *numberOfMatches = 0;
   
   while ((hashPtr = GetNextSymbolMatch(searchString,searchLength,hashPtr)) != NULL)
     {
      *numberOfMatches = *numberOfMatches + 1;
      temp = get_struct(symbolMatch);
      temp->match = hashPtr;
      temp->next = reply;
      reply = temp;
     }
     
   return(reply);
  }

/***************************************/
/* ReturnSymbolMatches:                */
/***************************************/
globle VOID ReturnSymbolMatches(listOfMatches)
  struct symbolMatch *listOfMatches;
  {
   struct symbolMatch *temp;
   
   while (listOfMatches != NULL)
     {
      temp = listOfMatches->next;
      rtn_struct(symbolMatch,listOfMatches);
      listOfMatches = temp;
     }
  }
 
/***************************************/
/* GetNextSymbolMatch:                 */
/***************************************/
globle SYMBOL_HN *GetNextSymbolMatch(searchString,searchLength,prevSymbol)
  char *searchString;
  int searchLength;
  SYMBOL_HN *prevSymbol;
  {
   register int i;
   SYMBOL_HN *hashPtr;
   
   if (prevSymbol == NULL)
     {
      i = 0;
      hashPtr = SymbolTable[0];
     }
   else
     {
      i = prevSymbol->bucket;
      hashPtr = prevSymbol->next;
     }
     
   while (1)
     {
      while (hashPtr != NULL)
        { 
         if ((hashPtr->contents[0] != '(') &&
             (! hashPtr->markedEphemeral) &&
             (strncmp(searchString,hashPtr->contents,searchLength) == 0))
           return(hashPtr);           
         hashPtr = hashPtr->next;
        }
      if (++i >= SYMBOL_HASH_SIZE)
        break;
      hashPtr = SymbolTable[i];
     }
   return(NULL);
  }


