
;;;======================================================
;;;   Wine Expert Sample Problem
;;;
;;;     WINEX: The WINe EXpert system.
;;;     This example selects an appropriate wine
;;;     to drink with a meal.
;;;
;;;     CLIPS Version 5.1 Example
;;;     Modified to run in wxCLIPS
;;;
;;;     To execute, merely load, reset and run.
;;;======================================================

;;****************
;;* DEFFUNCTIONS *
;;****************

(deffunction ask-question (?question $?allowed-values)
   (bind ?ans (get-text-from-user ?question))
   (bind ?answer none)
   (if (eq ?ans "") then (halt)
    else
    (bind ?answer (string-to-symbol (lowcase ?ans)))
    (while (and (not (member ?answer ?allowed-values)) (neq ?ans "")) do
      (bind ?ans (get-text-from-user ?question))
      (bind ?answer (string-to-symbol (lowcase ?ans)))
    )
   )
    ?answer)

;;*****************
;;* INITIAL STATE *
;;*****************

(deffacts valid-combinations ""
  (combine best-body)
  (combine best-color)
  (combine best-sweetness)
  (combine recommended-body)
  (combine recommended-color)
  (combine recommended-sweetness)
  (combine wine))

;;*******************************
;;* CHOOSE WINE QUALITIES RULES *
;;*******************************

(defrule choose-body-for-spicy-sauce ""
  (phase choose-qualities)
  (has-sauce yes)
  (sauce spicy)
  =>
  (assert (best-body full 100 =(gensym))))

(defrule choose-body-for-delicate-taste "" 
  (phase choose-qualities)
  (tastiness delicate)
  =>
  (assert (best-body light 100 =(gensym))))

(defrule choose-body-for-average-taste ""
  (phase choose-qualities)
  (tastiness average)
  =>
  (assert (best-body light 30 =(gensym)))
  (assert (best-body medium 60 =(gensym)))
  (assert (best-body full 30 =(gensym))))

(defrule choose-body-for-strong-taste ""
  (phase choose-qualities)
  (tastiness strong)
  =>
  (assert (best-body medium 40 =(gensym)))
  (assert (best-body full 80 =(gensym))))

(defrule choose-body-for-cream-sauce ""
  (phase choose-qualities)
  (has-sauce yes)
  (sauce cream)
  =>
  (assert (best-body medium 40 =(gensym)))
  (assert (best-body full 60 =(gensym))))

(defrule choose-color-for-meat ""
  (phase choose-qualities)
  (main-component meat)
  (has-veal no)
  =>
  (assert (best-color red 90 =(gensym))))

(defrule choose-color-for-poultry ""
  (phase choose-qualities)
  (main-component poultry)
  (has-turkey no)
  =>
  (assert (best-color white 90 =(gensym)))
  (assert (best-color red 30 =(gensym))))

(defrule choose-color-for-fish ""
  (phase choose-qualities)
  (main-component fish)
  =>
  (assert (best-color white 100 =(gensym))))

(defrule choose-color-for-tomato-sauce ""
  (phase choose-qualities)
  (not (main-component fish))
  (has-sauce yes)
  (sauce tomato)
  =>
  (assert (best-color red 100 =(gensym))))

(defrule choose-color-for-turkey ""
  (phase choose-qualities)
  (main-component poultry)
  (has-turkey yes)
  =>
  (assert (best-color red 80 =(gensym)))
  (assert (best-color white 50 =(gensym))))

(defrule choose-color-for-cream-sauce ""
  (phase choose-qualities)
  (main-component unknown)
  (has-sauce yes)
  (sauce cream)
  =>
  (assert (best-color white 40 =(gensym))))

(defrule choose-color-for-sweet-sauce ""
  (phase choose-qualities)
  (has-sauce yes)
  (sauce sweet)
  =>
  (assert (best-sweetness sweet 90 =(gensym)))
  (assert (best-sweetness medium 40 =(gensym))))

(defrule spicy-sauce-is-spicy-feature ""
  (phase choose-qualities)
  (has-sauce yes)
  (sauce spicy)
  =>
  (assert (feature spiciness)))

(defrule best-body-always-recommended ""
  (phase recommend-qualities)
  (best-body ?body ?per ?)
  =>
  (assert (recommended-body ?body ?per =(gensym))))

(defrule preferred-body-may-be-recommended ""
  (phase recommend-qualities)
  (preferred-body ?body)
  (best-body ?body ?per ?)
  =>
  (assert (recommended-body ?body =(/ (* 20 ?per) 100) =(gensym))))

(defrule recommend-medium-body-1 ""
  (phase recommend-qualities)
  (preferred-body light)
  (best-body full ?per ?)
  =>
  (assert (recommended-body medium ?per =(gensym))))

(defrule recommend-medium-body-2 ""
  (phase recommend-qualities)
  (preferred-body full)
  (best-body light ?per ?)
  =>
  (assert (recommended-body medium ?per =(gensym))))

(defrule best-color-always-recommended ""
  (phase recommend-qualities)
  (best-color ?color ?per ?)
  =>
  (assert (recommended-color ?color ?per =(gensym))))

(defrule preferred-color-may-be-recommended
  (phase recommend-qualities)
  (preferred-color ?color)
  (best-color ?color ?per ?)
  =>
  (assert (recommended-color ?color =(/ (* 20 ?per) 100) =(gensym))))

(defrule preferred-color-is-unknown ""
  (phase recommend-qualities)
  (preferred-color unknown)
  =>
  (assert (recommended-color white 50 =(gensym)))
  (assert (recommended-color red 50 =(gensym))))

(defrule best-sweetness-always-recommended ""
  (phase recommend-qualities)
  (best-sweetness ?sweet ?per ?)
  =>
  (assert (recommended-sweetness ?sweet ?per =(gensym))))

(defrule preferred-sweetness-may-be-recommended ""
  (phase recommend-qualities)
  (best-sweetness ?sweet ?per ?)
  (preferred-sweetness ?sweet)
  =>
  (assert (recommended-sweetness ?sweet =(/ (* 20 ?per) 100) =(gensym))))

(defrule recommend-medium-sweetness-1 ""
  (phase recommend-qualities)
  (best-sweetness sweet ?per ?)
  (preferred-sweetness dry)
  =>
  (assert (recommended-sweetness medium ?per =(gensym))))

(defrule recommend-medium-sweetness-2 ""
  (phase recommend-qualities)
  (best-sweetness dry ?per ?)
  (preferred-sweetness sweet)
  =>
  (assert (recommended-sweetness medium ?per =(gensym))))

;;*************************************
;;* DEFAULT QUALITIES SELECTION RULES *
;;*************************************

(defrule use-prefered-body-if-no-best-body ""
  (phase default-qualities)
  (preferred-body ?body&~unknown)
  (not (best-body ? ? ?))
  =>
  (assert (recommended-body ?body 100 =(gensym))))    

(defrule use-medium-body-if-no-best-body ""
  (phase default-qualities)
  (not (best-body ? ? ?))
  =>
  (assert (recommended-body medium 100 =(gensym))))

(defrule use-preferred-color-if-no-best-color ""
  (phase default-qualities)
  (preferred-color ?color&~unknown)
  (not (best-color ? ? ?))
  =>
  (assert (recommended-color ?color 100 =(gensym))))

(defrule use-medium-sweetness-if-preference-unknown ""
  (phase default-qualities)
  (not (best-sweetness ? ? ?))
  (preferred-sweetness unknown)
  =>
  (assert (recommended-sweetness medium 100 =(gensym))))

(defrule use-preferred-sweetness-if-no-best-sweetness ""
  (phase default-qualities)
  (not (best-sweetness ? ? ?))
  (preferred-sweetness ?sweet&~unknown)
  =>
  (assert (recommended-sweetness ?sweet 100 =(gensym))))

;;************************
;;* WINE SELECTION RULES *
;;************************

(defrule recommend-gamay ""
  (phase select-wines)
  (recommended-color red ?per1 ?)
  (recommended-body medium ?per2 ?)
  (or (recommended-sweetness medium ?per3 ?)
      (recommended-sweetness sweet ?per3 ?))
  =>
  (assert (wine Gamay =(min ?per1 ?per2 ?per3) =(gensym))))

(defrule recommend-chablis ""
  (phase select-wines)
  (recommended-color white ?per1 ?)
  (recommended-body light ?per2 ?)
  (recommended-sweetness dry ?per3 ?)
  =>
  (assert (wine Chablis =(min ?per1 ?per2 ?per3) =(gensym))))

(defrule recommend-sauvignon-blanc ""
  (phase select-wines)
  (recommended-color white ?per1 ?)
  (recommended-body medium ?per2 ?)
  (recommended-sweetness dry ?per3 ?)
  =>
  (assert (wine Sauvignon-Blanc =(min ?per1 ?per2 ?per3) =(gensym))))

(defrule recommend-chardonnay ""
  (phase select-wines)
  (recommended-color white ?per1 ?)
  (or (recommended-body  medium ?per2 ?)
      (recommended-body  full ?per2 ?))
  (or (recommended-sweetness medium ?per3 ?)
      (recommended-sweetness dry ?per3 ?))
  =>
  (assert (wine Chardonnay =(min ?per1 ?per2 ?per3) =(gensym))))

(defrule recommend-soave ""
  (phase select-wines)
  (recommended-color white ?per1 ?)
  (recommended-body light ?per2 ?)
  (or (recommended-sweetness medium  ?per3 ?)
      (recommended-sweetness dry ?per3 ?))
  =>
  (assert (wine Soave =(min ?per1 ?per2 ?per3) =(gensym))))

(defrule recommend-riesling ""
  (phase select-wines)
  (recommended-color white ?per1 ?)
  (or (recommended-body light ?per2 ?)
      (recommended-body medium ?per2 ?))
  (or (recommended-sweetness medium ?per3 ?)
      (recommended-sweetness sweet ?per3 ?))
  =>
  (assert (wine Riesling =(min ?per1 ?per2 ?per3) =(gensym))))

(defrule recommend-geverztraminer ""
  (phase select-wines)
  (recommended-color white ?per1 ?)
  (recommended-body full ?per2 ?)
  (feature spiciness)
  =>
  (assert (wine Geverztraminer =(min ?per1 ?per2) =(gensym))))

(defrule recommend-chenin-blanc ""
  (phase select-wines)
  (recommended-color white ?per1 ?)
  (recommended-body light ?per2 ?)
  (or (recommended-sweetness medium ?per3 ?)
      (recommended-sweetness sweet ?per3 ?))
  =>
  (assert (wine Chenin-Blanc =(min ?per1 ?per2 ?per3) =(gensym))))

(defrule recommend-valpolicella ""
  (phase select-wines)
  (recommended-color red ?per1 ?)
  (recommended-body light ?per2 ?)
  =>
  (assert (wine Valpolicella =(min ?per1 ?per2) =(gensym))))

(defrule recommend-zinfandel-and-cabernet-sauvignon ""
  (phase select-wines)
  (recommended-color red ?per1 ?)
  (or (recommended-sweetness medium ?per2 ?)
      (recommended-sweetness dry  ?per2 ?))
  =>
  (assert (wine Cabernet-Sauvignon =(min ?per1 ?per2) =(gensym)))
  (assert (wine Zinfandel =(min ?per1 ?per2) =(gensym))))

(defrule recommend-pinot-noir ""
  (phase select-wines)
  (recommended-color red ?per1 ?)
  (recommended-body medium ?per2 ?)
  (recommended-sweetness medium ?per3 ?)
  =>
  (assert (wine Pinot-Noir =(min ?per1 ?per2 ?per3) =(gensym))))

(defrule recommend-burgundy ""
  (phase select-wines)
  (recommended-color red ?per1 ?)
  (recommended-body full ?per2 ?)
  =>
  (assert (wine Burgundy =(min ?per1 ?per2) =(gensym))))

;;***************
;;* QUERY RULES *
;;***************

(defrule question-1 ""
  ?rem <- (ask-question)
  (not (main-component ?))
  =>
  (retract ?rem)
  (bind ?response
     (ask-question "Is the main component of the meal meat, fish, or poultry? "
                    meat fish poultry unknown))
  (assert (main-component ?response)))

(defrule question-2 ""
  ?rem <- (ask-question)
  (main-component poultry)
  (not (has-turkey ?))
  =>
  (retract ?rem)
  (bind ?response
     (ask-question "Does the meal have turkey in it? "
                    yes no unknown))
  (assert (has-turkey ?response)))

(defrule question-3 ""
  ?rem <- (ask-question)
  (main-component meat)
  (not (has-veal ?))
  =>
  (retract ?rem)
  (bind ?response
     (ask-question "Does the meal have veal in it? "
                    yes no unknown))
  (assert (has-veal ?response)))

(defrule question-4 ""  
  ?rem <- (ask-question)
  (not (has-sauce ?))
  =>
  (retract ?rem)
  (bind ?response
     (ask-question "Does the meal have a sauce on it? "
                    yes no unknown))
  (assert (has-sauce ?response)))

(defrule question-5 ""
  ?rem <- (ask-question)
  (has-sauce yes)
  (not (sauce ?))
  =>
  (retract ?rem)  
  (bind ?response
     (ask-question "Is the sauce for the meal spicy, sweet, cream, or tomato? "
                    sauce spicy sweet cream tomato unknown))
  (assert (sauce ?response)))

(defrule question-6 ""
  ?rem <- (ask-question)
  (not (tastiness ?))
  =>
  (retract ?rem)
  (bind ?response
     (ask-question "Is the flavor of the meal delicate, average, or strong? "
                    delicate average strong unknown))
  (assert (tastiness ?response)))

(defrule question-7 ""
  ?rem <- (ask-question)
  (not (preferred-body ?))
  =>
  (retract ?rem)
  (bind ?response
     (ask-question "Do you generally prefer light, medium, or full bodied wines? "
                    light medium full unknown))
  (assert (preferred-body ?response)))

(defrule question-8 ""
  ?rem <- (ask-question)
  (not (preferred-color ?))
  =>
  (retract ?rem)
  (bind ?response
     (ask-question "Do you generally prefer red or white wines? "
                    red white unknown))
  (assert (preferred-color ?response)))

(defrule question-9 ""
  ?rem <- (ask-question)
  (not (preferred-sweetness ?))
  =>
  (retract ?rem)
  (bind ?response
     (ask-question "Do you generally prefer dry, medium, or sweet wines? "
                    dry medium sweet unknown))
  (assert (preferred-sweetness ?response))) 

(defrule ask-another-question ""
  (not (ask-question))
  =>
  (assert (ask-question)))

;;*****************************
;;* PRINT SELECTED WINE RULES *
;;*****************************

(defrule print-wine ""
  (phase print-wines)
  ?rem <- (wine ?name ?per ?)		  
  (not (wine ?name1 ?per1&:(> ?per1 ?per) ?))
  =>
  (retract ?rem)
  (format t " %-24s %2d%%%n" ?name ?per))

(defrule end-spaces ""
   (phase print-wines)
   (not (wine ? ? ?))
   =>
   (printout t t))

;;*******************************
;;* ELIMINATE POOR CHOICES RULE *
;;*******************************

(defrule remove-poor-wine-choices ""
  (phase remove-poor-choices)
  ?rem <- (wine ? ?per ?)
  (test (< ?per 20))
  =>
  (retract ?rem))

;;****************************
;;* COMBINE CERTAINTIES RULE *
;;****************************

(defrule combine-certainties ""
  (declare (salience 10000))
  (combine ?rel)
  ?rem1 <- (?rel ?val ?per1 ?sym1)
  ?rem2 <- (?rel ?val ?per2 ?sym2&~?sym1)
  =>
  (retract ?rem1 ?rem2)
  (assert (?rel ?val
		=(/ (- (* 100 (+ ?per1 ?per2)) (* ?per1 ?per2)) 100)
		=(gensym))))
		
;;**************************************
;;* PHASE CONTROL RULES                *
;;*   PHASE 0: Ask Questions           *
;;*   PHASE 1: Choose Best Qualities   *
;;*   PHASE 2: Choose Recommended      *
;;*            Qualities               *
;;*   PHASE 3: Check for Default       *
;;*            Recommended Qualities   *
;;*   PHASE 4: Select Wines based on   *
;;*            Recommended Qualities   *
;;*   PHASE 5: Remove Wine Selections  *
;;*            with Low Certainties    *
;;*   PHASE 6: Display Wine Selections *
;;**************************************

(defrule change-to-phase-1 ""
   (declare (salience -10))
   =>
   (assert (phase choose-qualities)))

(defrule change-to-phase-2 ""
   (declare (salience -10))
   ?phase <- (phase choose-qualities)
   =>
   (retract ?phase)
   (assert (phase recommend-qualities)))

(defrule change-to-phase-3 ""
   (declare (salience -10))
   ?phase <- (phase recommend-qualities)
   =>
   (retract ?phase)
   (assert (phase default-qualities)))

(defrule change-to-phase-4 ""
   (declare (salience -10))
   ?phase <- (phase default-qualities)
   =>
   (retract ?phase)
   (assert (phase select-wines)))

(defrule change-to-phase-5 ""
   (declare (salience -10))
   ?phase <- (phase select-wines)
   =>
   (retract ?phase)
   (assert (phase remove-poor-choices)))

(defrule change-to-phase-6 ""
   (declare (salience -10))
   ?phase <- (phase remove-poor-choices)
   =>
   (retract ?phase)
   (printout t t)
   (printout t "        SELECTED WINES" t t)
   (printout t " WINE                  CERTAINTY" t)
   (printout t " -------------------------------" t)
   (assert (phase print-wines)))
