/*
 * gdi.cc
 * wxWindows command processing: GDI objects
 *
 */

#include "wx.h"
#include "wx_hash.h"
#include "wx_types.h"

#ifdef wx_msw
#if LOADABLE_BITMAPS
#include "dib.h"
#endif
#if LOADABLE_ICONS
#include "wxcurico.h"
#endif
#endif

#include <iostream.h>
#include <ctype.h>
#include <stdlib.h>

#include "wx_cmds.h"

extern wxObject *wxGetTypedObject(long objectId, WXTYPE expectedType);
extern void wxStoreThingInTable(long item_id, wxObject *object, WXTYPE type);
extern void wxDeleteThingInTable(long item_id);
extern wxObject *wxGetThingFromTable(long item_id, WXTYPE *type);
extern void wxCleanWindows(void);

#ifdef USE_CLIPS
#include "clips.h"
#endif

extern ClipsErrorClass ClipsError;

// Pen
class wxClipsPen: public wxPen
{
 public:
  wxClipsPen(wxColour& col, int width, int style):
   wxPen(col, width, style)
   {
   }
                   
  ~wxClipsPen()
   {
      wxDeleteThingInTable((long)this);
   }
};

// Brush
class wxClipsBrush: public wxBrush
{
 public:
  wxClipsBrush(wxColour& col, int style):
   wxBrush(col, style)
   {
   }
                   
  ~wxClipsBrush()
   {
      wxDeleteThingInTable((long)this);
   }
};

// Font
class wxClipsFont: public wxFont
{
 public:
  wxClipsFont(int pointSize, int family, int size, int height):
   wxFont(pointSize, family, size, height)
   {
   }
                   
  ~wxClipsFont()
   {
      wxDeleteThingInTable((long)this);
   }
};

// Icon
class wxClipsIcon: public wxIcon
{
 public:
  wxClipsIcon(char *file):
   wxIcon(file)
   {
   }
                   
  ~wxClipsIcon()
   {
      wxDeleteThingInTable((long)this);
   }
};

// Create a new pen
long wxcPenCreate(char *colour, long width, char *style)
{
  if (!colour || !style)
    return -1;

  wxColour *col = wxTheColourDatabase->FindColour(colour);
  int sty = 0;
  if (!col)
  {
    ClipsError << "Error in pen-create: could not find colour " << colour << ".\n";
    return 0;
  }
  if (strcmp(style, "wxSOLID") == 0)
    sty = wxSOLID;
  else if (strcmp(style, "wxDOT") == 0)
    sty = wxDOT;
  else if (strcmp(style, "wxLONG_DASH") == 0)
    sty = wxLONG_DASH;
  else if (strcmp(style, "wxSHORT_DASH") == 0)
    sty = wxSHORT_DASH;
  else if (strcmp(style, "wxDOT_DASH") == 0)
    sty = wxDOT_DASH;
  else if (strcmp(style, "wxTRANSPARENT") == 0)
    sty = wxTRANSPARENT;
  else
  {
    ClipsError << "Error in pen-create: unrecognized style " << style << "\n";
    ClipsError << "Style must be one of wxSOLID, wxDOT, wxLONG_DASH, wxSHORT_DASH, wxDOT_DASH, wxTRANSPARENT.\n";
    return 0;
  }

  wxClipsPen *pen = new wxClipsPen(*col, (int)width, sty);
  wxStoreThingInTable((long)pen, pen, wxTYPE_PEN);
  return (long)pen;
}

#ifdef USE_CLIPS
long clipsPenCreate()
{
  if (ArgCountCheck("clipsPenCreate", EXACTLY, 3) == -1)
    return 0;

  char *colour = RtnLexeme(1);
  long width = RtnLong(2);
  char *style = RtnLexeme(3);

  return wxcPenCreate(colour, width, style);
}
#endif

// Delete pen
long wxcPenDelete(long id)
{
  wxPen *pen = (wxPen *)wxGetTypedObject(id, wxTYPE_PEN);
  if (pen)
  {
    delete pen;
    return 1;
  }
  else
  {
    ClipsError << "Error in pen-delete: could not find pen " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsPenDelete()
{
  if (ArgCountCheck("clipsPenDelete", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcPenDelete(id);
}
#endif

// Create a new brush
long wxcBrushCreate(char *colour, char *style)
{
  if (!colour || !style)
    return -1;

  wxColour *col = wxTheColourDatabase->FindColour(colour);
  int sty = 0;
  if (!col)
  {
    ClipsError << "Error in brush-create: could not find colour " << colour << ".\n";
    return 0;
  }
  if (strcmp(style, "wxSOLID") == 0)
    sty = wxSOLID;
  else if (strcmp(style, "wxTRANSPARENT") == 0)
    sty = wxTRANSPARENT;
  else
  {
    ClipsError << "Error in brush-create: unrecognized style " << style << "\n";
    ClipsError << "Style must be one of wxSOLID, wxTRANSPARENT.\n";
    return 0;
  }

  wxClipsBrush *brush = new wxClipsBrush(*col, sty);
  wxStoreThingInTable((long)brush, brush, wxTYPE_BRUSH);
  return (long)brush;
}

#ifdef USE_CLIPS
long clipsBrushCreate()
{
  if (ArgCountCheck("clipsBrushCreate", EXACTLY, 2) == -1)
    return 0;

  char *colour = RtnLexeme(1);
  char *style = RtnLexeme(2);
  return wxcBrushCreate(colour, style);
}
#endif

// Delete pen
long wxcBrushDelete(long id)
{
  wxBrush *brush = (wxBrush *)wxGetTypedObject(id, wxTYPE_BRUSH);
  if (brush)
  {
    delete brush;
    return 1;
  }
  else
  {
    ClipsError << "Error in brush-delete: could not find brush " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsBrushDelete()
{
  if (ArgCountCheck("clipsBrushDelete", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcBrushDelete(id);
}
#endif

// Create a new font
long wxcFontCreate(long pointSize, char *family, char *style, char *weight,
  long underlined)
{
  int wxStyle = 0;
  int wxFamily = 0;
  int wxWeight = 0;
  if (!family || !style || !weight)
    return -1;

  if (strcmp(family, "wxROMAN") == 0)
    wxFamily = wxROMAN;
  else if (strcmp(family, "wxSCRIPT") == 0)
    wxFamily = wxSCRIPT;
  else if (strcmp(family, "wxDECORATIVE") == 0)
    wxFamily = wxDECORATIVE;
  else if (strcmp(family, "wxSWISS") == 0)
    wxFamily = wxSWISS;
  else if (strcmp(family, "wxMODERN") == 0)
    wxFamily = wxMODERN;
  else if (strcmp(family, "wxDEFAULT") == 0)
    wxFamily = wxMODERN;
  else
  {
    ClipsError << "Error in font-create: unrecognized family " << family << ".\n";
    ClipsError << "Family must be one of wxDECORATIVE, wxDEFAULT, wxMODERN, wxROMAN, wxSCRIPT, wxSWISS.\n";
    return 0;
  }

  if (strcmp(weight, "wxNORMAL") == 0)
    wxStyle = wxNORMAL;
  else if (strcmp(weight, "wxBOLD") == 0)
    wxStyle = wxBOLD;
  else if (strcmp(weight, "wxLIGHT") == 0)
    wxStyle = wxLIGHT;
  else
  {
    ClipsError << "Error in font-create: unrecognized weight " << weight << ".\n";
    ClipsError << "Weight must be one of wxNORMAL, wxBOLD, wxLIGHT.\n";
    return 0;
  }

  if (strcmp(style, "wxNORMAL") == 0)
    wxStyle = wxNORMAL;
  else if (strcmp(style, "wxITALIC") == 0)
    wxStyle = wxITALIC;
  else
  {
    ClipsError << "Error in font-create: unrecognized style " << style << ".\n";
    ClipsError << "Style must be one of wxNORMAL, wxITALIC.\n";
    return 0;
  }

  wxClipsFont *font = new wxClipsFont((int)pointSize, wxFamily, wxStyle, wxWeight);
  wxStoreThingInTable((long)font, font, wxTYPE_FONT);
  return (long)font;
}

#ifdef USE_CLIPS
long clipsFontCreate()
{
  if (ArgCountCheck("clipsFontCreate", EXACTLY, 5) == -1)
    return 0;

  int pointSize = (int)RtnLong(1);
  char *family = RtnLexeme(2);
  char *style = RtnLexeme(3);
  char *weight  = RtnLexeme(4);
  Bool underlined = (Bool)RtnLong(5);
  return wxcFontCreate(pointSize, family, style, weight, underlined);
}
#endif

// Delete font
long wxcFontDelete(long id)
{
  wxFont *font = (wxFont *)wxGetTypedObject(id, wxTYPE_FONT);
  if (font)
  {
    delete font;
    return 1;
  }
  else
  {
    ClipsError << "Error in font-delete: could not find font " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsFontDelete()
{
  if (ArgCountCheck("clipsFontDelete", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcFontDelete(id);
}
#endif

// Create a new bitmap
long wxcBitmapCreate(int width, int height, int depth)
{
  wxBitmap *bitmap = new wxBitmap(width, height, depth);
  wxStoreThingInTable((long)bitmap, bitmap, wxTYPE_BITMAP);
  return (long)bitmap;
}

#ifdef USE_CLIPS
long clipsBitmapCreate()
{
  if (ArgCountCheck("clipsBitmapCreate", AT_LEAST, 2) == -1)
    return 0;

  int depth = -1;
  if (RtnArgCount() > 2)
    depth = (int)RtnLong(3);
  return wxcBitmapCreate((int)RtnLong(1), (int)RtnLong(2), depth);
}
#endif

// Load a bitmap from a file (.XBM for X, .BMP for Windows)
long wxcBitmapLoadFromFile(char *f)
{
#if (defined(wx_msw) && !LOADABLE_BITMAPS)
  ClipsError << "Error in bitmap-load-from-file: not implemented in this version.";
  return -1;
#else
  if (FileExists(f))
  {
#ifdef wx_x
    wxBitmap *bitmap = new wxBitmap(f);
#else
    wxBitmap *bitmap = wxLoadBitmap(f);
#endif
    if (!bitmap)
    {
      ClipsError << "Error in bitmap-load-from-file: error while loading bitmap " << f << ".\n";
      return -1;
    }

    wxStoreThingInTable((long)bitmap, bitmap, wxTYPE_BITMAP);
    return (long)bitmap;
  }
  else
  {
    ClipsError << "Error in bitmap-load-from-file: could not find bitmap file " << f << ".\n";
    return -1;
  }
  return -1;
#endif
}

#ifdef USE_CLIPS
long clipsBitmapLoadFromFile()
{
  if (ArgCountCheck("clipsBitmapLoadFromFile", EXACTLY, 1) == -1)
    return 0;

  return wxcBitmapLoadFromFile(RtnLexeme(1));
}
#endif

// Delete bitmap
long wxcBitmapDelete(long id)
{
  wxBitmap *bitmap = (wxBitmap *)wxGetTypedObject(id, wxTYPE_BITMAP);
  if (bitmap)
  {
    wxDeleteThingInTable(id);
    delete bitmap;
    return 1;
  }
  else
  {
    ClipsError << "Error in bitmap-delete: could not find bitmap " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsBitmapDelete()
{
  if (ArgCountCheck("clipsBitmapDelete", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcBitmapDelete(id);
}
#endif

// Get the bitmap's width
long wxcBitmapGetWidth(long id)
{
  wxBitmap *bitmap = (wxBitmap *)wxGetTypedObject(id, wxTYPE_BITMAP);
  if (bitmap)
  {
    return bitmap->GetWidth();
  }
  else
  {
    ClipsError << "Error in bitmap-get-width: could not find bitmap " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsBitmapGetWidth()
{
  if (ArgCountCheck("clipsBitmapGetWidth", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcBitmapGetWidth(id);
}
#endif

// Get the bitmap's height
long wxcBitmapGetHeight(long id)
{
  wxBitmap *bitmap = (wxBitmap *)wxGetTypedObject(id, wxTYPE_BITMAP);
  if (bitmap)
  {
    return bitmap->GetHeight();
  }
  else
  {
    ClipsError << "Error in bitmap-get-height: could not find bitmap " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsBitmapGetHeight()
{
  if (ArgCountCheck("clipsBitmapGetHeight", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcBitmapGetHeight(id);
}
#endif

// Create a new icon
long wxcIconCreate(char *file)
{
  if (!file)
    return -1;

  wxClipsIcon *icon = new wxClipsIcon(file);
  wxStoreThingInTable((long)icon, icon, wxTYPE_ICON);
  return (long)icon;
}

#ifdef USE_CLIPS
long clipsIconCreate()
{
  if (ArgCountCheck("clipsIconCreate", EXACTLY, 1) == -1)
    return 0;

  char *file = RtnLexeme(1);
  return wxcIconCreate(file);
}
#endif

// Load an icon from a file (.XBM for X, .ICO for Windows)
long wxcIconLoadFromFile(char *f)
{
#if (defined(wx_msw) && !LOADABLE_ICONS)
  ClipsError << "Error in icon-load-from-file: not implemented in this version.";
  return -1;
#else
  if (FileExists(f))
  {
#ifdef wx_x
    wxIcon *icon = new wxIcon(f);
#else
    wxIconEx *icon = new wxIconEx(f);
#endif
    wxStoreThingInTable((long)icon, icon, wxTYPE_ICON);
    return (long)icon;
  }
  else
  {
    ClipsError << "Error in icon-load-from-file: could not find icon file " << f << ".\n";
    return 0;
  }
  return 0;
#endif
}

#ifdef USE_CLIPS
long clipsIconLoadFromFile()
{
  if (ArgCountCheck("clipsIconLoadFromFile", EXACTLY, 1) == -1)
    return 0;

  return wxcIconLoadFromFile(RtnLexeme(1));
}
#endif

// Delete bitmap
long wxcIconDelete(long id)
{
  wxIcon *icon = (wxIcon *)wxGetTypedObject(id, wxTYPE_ICON);
  if (icon)
  {
    wxDeleteThingInTable(id);
    delete icon;
    return 1;
  }
  else
  {
    ClipsError << "Error in icon-delete: could not find icon " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsIconDelete()
{
  if (ArgCountCheck("clipsIconDelete", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcIconDelete(id);
}
#endif

// Get the icon's width
long wxcIconGetWidth(long id)
{
  wxIcon *icon = (wxIcon *)wxGetTypedObject(id, wxTYPE_ICON);
  if (icon)
  {
    return icon->GetWidth();
  }
  else
  {
    ClipsError << "Error in icon-get-width: could not find icon " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsIconGetWidth()
{
  if (ArgCountCheck("clipsIconGetWidth", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcIconGetWidth(id);
}
#endif

// Get the icon's height
long wxcIconGetHeight(long id)
{
  wxIcon *icon = (wxIcon *)wxGetTypedObject(id, wxTYPE_ICON);
  if (icon)
  {
    return icon->GetHeight();
  }
  else
  {
    ClipsError << "Error in icon-get-height: could not find icon " << id << ".\n";
    return 0;
  }
}

#ifdef USE_CLIPS
long clipsIconGetHeight()
{
  if (ArgCountCheck("clipsIconGetHeight", EXACTLY, 1) == -1)
    return 0;

  long id = RtnLong(1);
  return wxcIconGetHeight(id);
}
#endif


#ifdef USE_CLIPS
void wxGDIUserFunctions()
{
  DefineFunction("bitmap-create", 'l', PTIF clipsBitmapCreate, "clipsBitmapCreate");
  DefineFunction("bitmap-load-from-file", 'l', PTIF clipsBitmapLoadFromFile, "clipsBitmapLoadFromFile");
  DefineFunction("bitmap-delete", 'l', PTIF clipsBitmapDelete, "clipsBitmapDelete");
  DefineFunction("bitmap-get-width", 'l', PTIF clipsBitmapGetWidth, "clipsBitmapGetWidth");
  DefineFunction("bitmap-get-height", 'l', PTIF clipsBitmapGetHeight, "clipsBitmapGetHeight");

  DefineFunction("icon-create", 'l', PTIF clipsIconCreate, "clipsIconCreate");
  DefineFunction("icon-load-from-file", 'l', PTIF clipsIconLoadFromFile, "clipsIconLoadFromFile");
  DefineFunction("icon-delete", 'l', PTIF clipsIconDelete, "clipsIconDelete");
  DefineFunction("icon-get-width", 'l', PTIF clipsIconGetWidth, "clipsIconGetWidth");
  DefineFunction("icon-get-height", 'l', PTIF clipsIconGetHeight, "clipsIconGetHeight");

  DefineFunction("brush-create", 'l', PTIF clipsBrushCreate, "clipsBrushCreate");
  DefineFunction("brush-delete", 'l', PTIF clipsBrushDelete, "clipsBrushDelete");

  DefineFunction("font-create", 'l', PTIF clipsFontCreate, "clipsFontCreate");
  DefineFunction("font-delete", 'l', PTIF clipsFontDelete, "clipsFontDelete");

  DefineFunction("pen-create", 'l', PTIF clipsPenCreate, "clipsPenCreate");
  DefineFunction("pen-delete", 'l', PTIF clipsPenDelete, "clipsPenDelete");
}
#endif
