   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*             CLIPS Version 5.10  07/17/91            */
   /*                                                     */
   /*                   SYSPRED MODULE                    */
   /*******************************************************/

/*************************************************************/
/* Purpose:                                                  */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Gary D. Riley                                        */
/*                                                           */
/* Contributing Programmer(s):                               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/

#define _SYSPRED_SOURCE_

#include <stdio.h>
#define _CLIPS_STDIO_

#include "setup.h"

#include "expressn.h"
#include "evaluatn.h"
#include "router.h"
#include "reteutil.h"

/***************************************/
/* LOCAL INTERNAL FUNCTION DEFINITIONS */
/***************************************/

#if ANSI_COMPILER  
   VOID                    PredicateFunctionDefinitions(void);
   BOOLEAN                 EqFunction(void);
   BOOLEAN                 NeqFunction(void);
   BOOLEAN                 StringpFunction(void);
   BOOLEAN                 WordpFunction(void);
   BOOLEAN                 LexemepFunction(void);
   BOOLEAN                 NumberpFunction(void);
   BOOLEAN                 FloatpFunction(void);
   BOOLEAN                 IntegerpFunction(void);
   BOOLEAN                 MultifieldpFunction(void);
   BOOLEAN                 PointerpFunction(void);
   BOOLEAN                 NotFunction(void);
   BOOLEAN                 AndFunction(void);
   BOOLEAN                 OrFunction(void);
   VOID                    AdditionFunction(DATA_OBJECT_PTR);
   VOID                    MultiplicationFunction(DATA_OBJECT_PTR);
   VOID                    SubtractionFunction(DATA_OBJECT_PTR);
   VOID                    DivisionFunction(DATA_OBJECT_PTR);
   long                    DivFunction(void);
   BOOLEAN                 LessThanOrEqualFunction(void);
   BOOLEAN                 GreaterThanOrEqualFunction(void);
   BOOLEAN                 LessThanFunction(void);
   BOOLEAN                 GreaterThanFunction(void);  
   BOOLEAN                 NumericEqualFunction(void);
   BOOLEAN                 NumericNotEqualFunction(void);
   BOOLEAN                 OddpFunction(void);
   BOOLEAN                 EvenpFunction(void);
   static BOOLEAN          SegmentsEqual(struct element *,struct element *,int,int);
   BOOLEAN                 SetAutoFloatDividendCommand(void);
   BOOLEAN                 GetAutoFloatDividendCommand(void);
   BOOLEAN                 GetAutoFloatDividend(void);
   BOOLEAN                 SetAutoFloatDividend(int);
#else
   VOID                    PredicateFunctionDefinitions();
   BOOLEAN                 EqFunction();
   BOOLEAN                 NeqFunction();
   BOOLEAN                 StringpFunction();
   BOOLEAN                 WordpFunction();
   BOOLEAN                 LexemepFunction();
   BOOLEAN                 NumberpFunction();
   BOOLEAN                 FloatpFunction();
   BOOLEAN                 IntegerpFunction();
   BOOLEAN                 MultifieldpFunction();
   BOOLEAN                 PointerpFunction();
   BOOLEAN                 NotFunction();
   BOOLEAN                 AndFunction();
   BOOLEAN                 OrFunction();
   VOID                    AdditionFunction();
   VOID                    MultiplicationFunction();
   VOID                    SubtractionFunction();
   VOID                    DivisionFunction();
   long                    DivFunction();
   BOOLEAN                 LessThanOrEqualFunction();
   BOOLEAN                 GreaterThanOrEqualFunction();
   BOOLEAN                 LessThanFunction();
   BOOLEAN                 GreaterThanFunction();  
   BOOLEAN                 NumericEqualFunction();
   BOOLEAN                 NumericNotEqualFunction();
   BOOLEAN                 OddpFunction();
   BOOLEAN                 EvenpFunction();
   static BOOLEAN          SegmentsEqual();
   BOOLEAN                 SetAutoFloatDividendCommand();
   BOOLEAN                 GetAutoFloatDividendCommand();
   BOOLEAN                 GetAutoFloatDividend();
   BOOLEAN                 SetAutoFloatDividend();
#endif

/***************************************/
/* LOCAL INTERNAL VARIABLE DEFINITIONS */
/***************************************/

   static BOOLEAN          AutoFloatDividend = CLIPS_TRUE;

#if ! RUN_TIME
/**************************************************/
/* PredicateFunctionDefinitions: Defines standard */  
/*   math and predicate functions.                */
/**************************************************/
globle VOID PredicateFunctionDefinitions()
  {
   DefineFunction("not",      'b', NotFunction,                "NotFunction");
   DefineFunction("and",      'b', AndFunction,                "AndFunction");
   DefineFunction("or",       'b', OrFunction,                 "OrFunction");
   
   DefineFunction("+",        'u', PTIF AdditionFunction,           "AdditionFunction");
   DefineFunction("*",        'u', PTIF MultiplicationFunction,     "MultiplicationFunction");
   DefineFunction("-",        'u', PTIF SubtractionFunction,        "SubtractionFunction");
   DefineFunction("/",        'u', PTIF DivisionFunction,           "DivisionFunction");
   DefineFunction("div",      'l', PTIF DivFunction,                "DivFunction");
   
   DefineFunction("eq",       'b', EqFunction,                 "EqFunction");
   DefineFunction("neq",      'b', NeqFunction,                "NeqFunction");
   
   DefineFunction("<=",       'b', LessThanOrEqualFunction,    "LessThanOrEqualFunction");
   DefineFunction(">=",       'b', GreaterThanOrEqualFunction, "GreaterThanOrEqualFunction");
   DefineFunction("<",        'b', LessThanFunction,           "LessThanFunction");
   DefineFunction(">",        'b', GreaterThanFunction,        "GreaterThanFunction");
   DefineFunction("=",        'b', NumericEqualFunction,       "NumericEqualFunction");
   DefineFunction("!=",       'b', NumericNotEqualFunction,    "NumericNotEqualFunction");
   DefineFunction("<>",       'b', NumericNotEqualFunction,    "NumericNotEqualFunction");

   DefineFunction("wordp",    'b', WordpFunction,              "WordpFunction");
   DefineFunction("symbolp",  'b', WordpFunction,              "WordpFunction");
   DefineFunction("stringp",  'b', StringpFunction,            "StringpFunction");
   DefineFunction("lexemep",  'b', LexemepFunction,            "LexemepFunction");
   DefineFunction("numberp",  'b', NumberpFunction,            "NumberpFunction");
   DefineFunction("integerp", 'b', IntegerpFunction,           "IntegerpFunction");
   DefineFunction("floatp",   'b', FloatpFunction,             "FloatpFunction"); 
   DefineFunction("oddp",     'b', OddpFunction,               "OddpFunction");
   DefineFunction("evenp",    'b', EvenpFunction,              "EvenpFunction");
   DefineFunction("multifieldp",'b', MultifieldpFunction,          "MultifieldpFunction");
   DefineFunction("sequencep",'b', MultifieldpFunction,          "MultifieldpFunction");
   DefineFunction("pointerp", 'b', PointerpFunction,           "PointerpFunction");
   
   DefineFunction("set-auto-float-dividend", 'b', 
                  SetAutoFloatDividendCommand,  "SetAutoFloatDividendCommand");
   DefineFunction("get-auto-float-dividend", 'b', 
                  GetAutoFloatDividendCommand,  "GetAutoFloatDividendCommand");
  }
#endif

/****************************************/
/* EqFunction:                               */
/****************************************/
globle BOOLEAN EqFunction()
  {
   DATA_OBJECT item, next_item;
   ELEMENT_PTR elem_a, elem_b = NULL;
   int num_a, i, a_extent, b_extent = 0;

   num_a = RtnArgCount();

   if (num_a == 0)
     { return(CLIPS_FALSE); }
   
   RtnUnknown(1,&item);
   
   if (GetType(item) == MULTIFIELD)
     {
      elem_b = GetMFPtr((struct fact *) ClipsGetValue(item),GetDOBegin(item));
      b_extent = GetDOLength(item); 
     }

   for (i = 2 ; i <= num_a ; i++)
     {
      RtnUnknown(i,&next_item);
      if (GetType(next_item) != GetType(item))
        { return(CLIPS_FALSE); }
      
      if (GetType(next_item) == MULTIFIELD)
        {
         elem_a = GetMFPtr((struct fact *) ClipsGetValue(next_item),GetDOBegin(next_item));
         a_extent = GetDOLength(next_item);

         if (SegmentsEqual(elem_a,elem_b,a_extent,b_extent) == CLIPS_FALSE)
           { return(CLIPS_FALSE); }
        }
      else if (next_item.value != item.value)
        { return(CLIPS_FALSE); }        
     }
   
   return(CLIPS_TRUE);
  }

/****************************************/
/* NeqFunction:                              */
/****************************************/
globle BOOLEAN NeqFunction()
  {
   DATA_OBJECT item, next_item;
   ELEMENT_PTR elem_a, elem_b = NULL;
   int num_a, i, a_extent, b_extent = 0;
   
   num_a = RtnArgCount();

   if (num_a == 0)
     { return(CLIPS_FALSE); }
   
   RtnUnknown(1,&item);
   if (GetType(item) == MULTIFIELD)
     {
      elem_b = GetMFPtr((struct fact *) ClipsGetValue(item),GetDOBegin(item));
      b_extent = GetDOLength(item); 
     }

   for (i = 2 ; i <= num_a ; i++)
     {
      RtnUnknown(i,&next_item);
      if (GetType(next_item) != GetType(item))
        { /* skip this one */ }
      
      else if (next_item.type == MULTIFIELD)
        {
         elem_a = GetMFPtr((struct fact *) ClipsGetValue(next_item),GetDOBegin(next_item));
         a_extent = GetDOLength(next_item);

         if (SegmentsEqual(elem_a,elem_b,a_extent,b_extent) == CLIPS_TRUE)
           { return(CLIPS_FALSE); }
        }
      else if (next_item.value == item.value)
        { return(CLIPS_FALSE); }        
     }
   
   return(CLIPS_TRUE);
  }


/**************************************************/
/* SegmentsEqual: determines if two segments are equal. */
/**************************************************/
static BOOLEAN SegmentsEqual(elem_a,elem_b,a_extent,b_extent)
  ELEMENT_PTR elem_a, elem_b;
  int a_extent, b_extent;
  {
   if (a_extent != b_extent)
     { return(CLIPS_FALSE); }  

   while (a_extent != 0)
     {
      if (elem_a->type != elem_b->type)
        { return(CLIPS_FALSE); }
        
      if (elem_a->value != elem_b->value)
        { return(CLIPS_FALSE); }

      a_extent--;

      if (a_extent > 0) 
        {
         elem_a++;
         elem_b++;
        }
     }
   return(CLIPS_TRUE);
  }

/****************************************/
/* StringpFunction:                             */
/****************************************/
globle BOOLEAN StringpFunction()
  {
   DATA_OBJECT item;
   
   if (ArgCountCheck("stringp",EXACTLY,1) == -1) return(CLIPS_FALSE);
   
   RtnUnknown(1,&item);

   if (GetType(item) == STRING)
     { return(CLIPS_TRUE); }
   else
     { return(CLIPS_FALSE); }
  }

/**************************************/
/* WordpFunction:                             */
/**************************************/
globle BOOLEAN WordpFunction()
  {
   DATA_OBJECT item;
   
   if (ArgCountCheck("symbolp",EXACTLY,1) == -1) return(CLIPS_FALSE);
   
   RtnUnknown(1,&item);

   if (GetType(item) == SYMBOL)
     { return(CLIPS_TRUE); }
   else
     { return(CLIPS_FALSE); }
  }
  
/****************************************/
/* LexemepFunction                      */
/****************************************/
globle BOOLEAN LexemepFunction()
  {
   DATA_OBJECT item;
   
   if (ArgCountCheck("lexemep",EXACTLY,1) == -1) return(CLIPS_FALSE);
   
   RtnUnknown(1,&item);

   if ((GetType(item) == SYMBOL) || (GetType(item) == STRING))
     { return(CLIPS_TRUE); }
   else
     { return(CLIPS_FALSE); }
  }
  
  
/****************************************/
/* NumberpFunction                              */
/****************************************/
globle BOOLEAN NumberpFunction()
  {
   DATA_OBJECT item;
   
   if (ArgCountCheck("numberp",EXACTLY,1) == -1) return(CLIPS_FALSE);
   
   RtnUnknown(1,&item);

   if ((GetType(item) == FLOAT) || (GetType(item) == INTEGER))
     { return(CLIPS_TRUE); }
   else
     { return(CLIPS_FALSE); }
  }
  
/****************************************/
/* FloatpFunction                       */
/****************************************/
globle BOOLEAN FloatpFunction()
  {
   DATA_OBJECT item;
   
   if (ArgCountCheck("floatp",EXACTLY,1) == -1) return(CLIPS_FALSE);
   
   RtnUnknown(1,&item);

   if (GetType(item) == FLOAT) 
     { return(CLIPS_TRUE); }
   else
     { return(CLIPS_FALSE); }
  }
 
    
/****************************************/
/* IntegerpFunction:                               */
/****************************************/
globle BOOLEAN IntegerpFunction()
  {
   DATA_OBJECT valstruct;

   if (ArgCountCheck("integerp",EXACTLY,1) == -1) return(CLIPS_FALSE);

   RtnUnknown(1,&valstruct);

   if (GetType(valstruct) != INTEGER) return(CLIPS_FALSE);
     
   return(CLIPS_TRUE);
  } 
  
/****************************************/
/* MultifieldpFunction:                           */
/****************************************/
globle BOOLEAN MultifieldpFunction()
  {
   DATA_OBJECT valstruct;

   if (ArgCountCheck("multifieldp",EXACTLY,1) == -1) return(CLIPS_FALSE);

   RtnUnknown(1,&valstruct);

   if (GetType(valstruct) != MULTIFIELD) return(CLIPS_FALSE);
     
   return(CLIPS_TRUE);
  } 
  
/****************************************/
/* PointerpFunction:                           */
/****************************************/
globle BOOLEAN PointerpFunction()
  {
   DATA_OBJECT valstruct;

   if (ArgCountCheck("pointerp",EXACTLY,1) == -1) return(CLIPS_FALSE);

   RtnUnknown(1,&valstruct);

   if (GetType(valstruct) != EXTERNAL_ADDRESS) return(CLIPS_FALSE);
     
   return(CLIPS_TRUE);
  } 

/****************************************/
/* NotFunction                          */
/****************************************/
globle BOOLEAN NotFunction()
  {
   EXPR_PTR test_ptr;
   DATA_OBJECT result;

   test_ptr = GetFirstArgument();
   if (test_ptr == NULL) { return(CLIPS_FALSE); }
   
   if (EvaluateExpression(test_ptr,&result)) return(CLIPS_FALSE);
   
   if ((result.value == (VOID *) CLIPSFalseSymbol) && (result.type == SYMBOL))
     { return(CLIPS_TRUE); }
  
   return(CLIPS_FALSE);
  }

/****************************************/
/* AndFunction                          */
/****************************************/
globle BOOLEAN AndFunction()
  {
   EXPR_PTR test_ptr;
   DATA_OBJECT result;

   test_ptr = GetFirstArgument();

   while (test_ptr != NULL)
     {        
      if (EvaluateExpression(test_ptr,&result)) return(CLIPS_FALSE);
      if ((result.value == (VOID *) CLIPSFalseSymbol) && (result.type == SYMBOL))
        { return(CLIPS_FALSE); }
        
      test_ptr = GetNextArgument(test_ptr);
     }
   
   return(CLIPS_TRUE);
  }

/****************************************/
/* OrFunction                                   */
/****************************************/
globle BOOLEAN OrFunction()
  {
   EXPR_PTR test_ptr;
   DATA_OBJECT result;

   test_ptr = GetFirstArgument();

   while (test_ptr != NULL)
     {
      if (EvaluateExpression(test_ptr,&result)) return(CLIPS_FALSE);
      
      if ((result.value != (VOID *) CLIPSFalseSymbol) || (result.type != SYMBOL))
        { return(CLIPS_TRUE); }
    
      test_ptr = GetNextArgument(test_ptr);
     }
   
   return(CLIPS_FALSE);
  }

/****************************************/
/* AdditionFunction                     */
/****************************************/
globle VOID AdditionFunction(result)
  DATA_OBJECT_PTR result;
  {
   double ftotal = 0.0;
   long ltotal = 0L;
   BOOLEAN useFloatTotal = CLIPS_FALSE;
   EXPR_PTR test_ptr;
   DATA_OBJECT rv;
   int pos = 1;

   test_ptr = GetFirstArgument();

   while (test_ptr != NULL)
     {
      if (! GetNumericArgument(test_ptr,"+",&rv,useFloatTotal,pos)) test_ptr = NULL;
      else test_ptr = GetNextArgument(test_ptr);
      
      if (useFloatTotal)
        { ftotal += ValueToDouble(rv.value); }
      else
        {
         if (rv.type == INTEGER)
           { ltotal += ValueToLong(rv.value); }
         else
           {
            ftotal = (double) ltotal + ValueToDouble(rv.value);
            useFloatTotal = CLIPS_TRUE;
           }
        }
        
      pos++;
     }
   
   if (useFloatTotal)
     {
      result->type = FLOAT;
      result->value = (VOID *) AddDouble(ftotal);
     }
   else
     {
      result->type = INTEGER;
      result->value = (VOID *) AddLong(ltotal);
     }
  } 

/****************************************/
/* MultiplicationFunction                             */
/****************************************/
globle VOID MultiplicationFunction(result)
  DATA_OBJECT_PTR result;
  {
   double ftotal = 1.0;
   long ltotal = 1L;
   BOOLEAN useFloatTotal = CLIPS_FALSE;
   EXPR_PTR test_ptr;
   DATA_OBJECT rv;
   int pos = 1;

   test_ptr = GetFirstArgument();

   while (test_ptr != NULL)
     {
      if (! GetNumericArgument(test_ptr,"*",&rv,useFloatTotal,pos)) test_ptr = NULL;
      else test_ptr = GetNextArgument(test_ptr);
      
      if (useFloatTotal)
        { ftotal *= ValueToDouble(rv.value); }
      else
        {
         if (rv.type == INTEGER)
           { ltotal *= ValueToLong(rv.value); }
         else
           {
            ftotal = (double) ltotal * ValueToDouble(rv.value);
            useFloatTotal = CLIPS_TRUE;
           }
        }
      pos++;
     }
   
   if (useFloatTotal)
     {
      result->type = FLOAT;
      result->value = (VOID *) AddDouble(ftotal);
     }
   else
     {
      result->type = INTEGER;
      result->value = (VOID *) AddLong(ltotal);
     }
  } 

/****************************************/
/* SubtractionFunction                             */
/****************************************/
globle VOID SubtractionFunction(result)
  DATA_OBJECT_PTR result;
  {
   double ftotal = 0.0;
   long ltotal = 0L;
   BOOLEAN useFloatTotal = CLIPS_FALSE;
   EXPR_PTR test_ptr;
   DATA_OBJECT rv;
   int pos = 1;
   
   /*===============================================*/
   /* Get the first argument. The number which will */
   /* be divided by all subsequent numbers.         */
   /*===============================================*/
   
   test_ptr = GetFirstArgument();
   if (test_ptr != NULL)
     {
      if (! GetNumericArgument(test_ptr,"-",&rv,useFloatTotal,pos)) test_ptr = NULL;
      else test_ptr = GetNextArgument(test_ptr);
      
      if (rv.type == INTEGER)
        { ltotal = ValueToLong(rv.value); }
      else
        {
         ftotal = ValueToDouble(rv.value);
         useFloatTotal = CLIPS_TRUE;
        }
      pos++;
     }

   /*===================================*/
   /* Subtract the remaining arguments. */
   /*===================================*/
   
   while (test_ptr != NULL)
     {
      if (! GetNumericArgument(test_ptr,"-",&rv,useFloatTotal,pos)) test_ptr = NULL;
      else test_ptr = GetNextArgument(test_ptr);
      
      if (useFloatTotal)
        { ftotal -= ValueToDouble(rv.value); }
      else
        {
         if (rv.type == INTEGER)
           { ltotal -= ValueToLong(rv.value); }
         else
           {
            ftotal = (double) ltotal - ValueToDouble(rv.value);
            useFloatTotal = CLIPS_TRUE;
           }
        }
      pos++;
     }
   
   if (useFloatTotal)
     {
      result->type = FLOAT;
      result->value = (VOID *) AddDouble(ftotal);
     }
   else
     {
      result->type = INTEGER;
      result->value = (VOID *) AddLong(ltotal);
     }
  } 


/****************************************/
/* DivisionFunction                     */
/****************************************/
globle VOID DivisionFunction(result)
  DATA_OBJECT_PTR result;
  {
   double ftotal = 1.0;
   long ltotal = 1L;
   BOOLEAN useFloatTotal = AutoFloatDividend;
   EXPR_PTR test_ptr;
   DATA_OBJECT rv;
   int pos = 1;

   /*===============================================*/
   /* Get the first argument. The number which will */
   /* be divided by all subsequent numbers.         */
   /*===============================================*/
   
   test_ptr = GetFirstArgument();
   if (test_ptr != NULL)
     {
      if (! GetNumericArgument(test_ptr,"/",&rv,useFloatTotal,pos)) test_ptr = NULL;
      else test_ptr = GetNextArgument(test_ptr);
      
      if (rv.type == INTEGER)
        { ltotal = ValueToLong(rv.value); }
      else
        {
         ftotal = ValueToDouble(rv.value);
         useFloatTotal = CLIPS_TRUE;
        }
      pos++;
     }
     
   /*====================================*/
   /* Divide by the remaining arguments. */
   /*====================================*/
   
   while (test_ptr != NULL)
     {
      if (! GetNumericArgument(test_ptr,"/",&rv,useFloatTotal,pos)) test_ptr = NULL;
      else test_ptr = GetNextArgument(test_ptr);
      
      if ((rv.type == INTEGER) ? (ValueToLong(rv.value) == 0L) : 
                                 ((rv.type == FLOAT) ? ValueToDouble(rv.value) == 0.0 : CLIPS_FALSE))
        {
         PrintCLIPS(WERROR,"Attempt to divide by zero in / function\n");
         SetHaltExecution(CLIPS_TRUE);
         SetEvaluationError(CLIPS_TRUE);
         result->type = FLOAT;
         result->value = (VOID *) AddDouble(1.0);
         return;
        }
      
      if (useFloatTotal)
        { ftotal /= ValueToDouble(rv.value); }
      else
        {
         if (rv.type == INTEGER)
           { ltotal /= ValueToLong(rv.value); }
         else
           {
            ftotal = (double) ltotal / ValueToDouble(rv.value);
            useFloatTotal = CLIPS_TRUE;
           }
        }
      pos++;
     }
   
   if (useFloatTotal)
     {
      result->type = FLOAT;
      result->value = (VOID *) AddDouble(ftotal);
     }
   else
     {
      result->type = INTEGER;
      result->value = (VOID *) AddLong(ltotal);
     }
  } 
  
/****************************************/
/* DivFunction                          */
/****************************************/
globle long DivFunction()
  {
   long total = 1L;
   EXPR_PTR test_ptr;
   DATA_OBJECT rv;
   int pos = 1;

   /*===============================================*/
   /* Get the first argument. The number which will */
   /* be divided by all subsequent numbers.         */
   /*===============================================*/
   
   test_ptr = GetFirstArgument();
   if (test_ptr != NULL)
     {
      if (! GetNumericArgument(test_ptr,"div",&rv,CLIPS_FALSE,pos)) test_ptr = NULL;
      else test_ptr = GetNextArgument(test_ptr);
      
      if (rv.type == INTEGER)
        { total = ValueToLong(rv.value); }
      else
        { total = (long) ValueToDouble(rv.value); }
      pos++;
     }
     
   /*====================================*/
   /* Divide by the remaining arguments. */
   /*====================================*/
   
   while (test_ptr != NULL)
     {
      if (! GetNumericArgument(test_ptr,"div",&rv,CLIPS_FALSE,pos)) test_ptr = NULL;
      else test_ptr = GetNextArgument(test_ptr);
      
      if ((rv.type == INTEGER) ? (ValueToLong(rv.value) == 0L) : 
                                 ((rv.type == FLOAT) ? ((long) ValueToDouble(rv.value)) == 0L : CLIPS_FALSE))
        {
         PrintCLIPS(WERROR,"Attempt to divide by zero in div function\n");
         SetHaltExecution(CLIPS_TRUE);
         SetEvaluationError(CLIPS_TRUE);
         return(1L);
        }
      
      if (rv.type == INTEGER)
        { total /= ValueToLong(rv.value); }
      else
        { total = total / (long) ValueToDouble(rv.value); }
        
      pos++;
     }
   
   return(total);
  } 
  
/****************************************/
/* LessThanOrEqualFunction              */
/****************************************/ 
globle BOOLEAN LessThanOrEqualFunction()
  {
   EXPR_PTR test_ptr;
   DATA_OBJECT rv1, rv2;
   int pos = 1;

   test_ptr = GetFirstArgument();

   if (test_ptr == NULL) { return(CLIPS_TRUE); }
   if (! GetNumericArgument(test_ptr,"<=",&rv1,CLIPS_FALSE,pos)) return(CLIPS_FALSE);
   test_ptr = GetNextArgument(test_ptr);
   if (test_ptr == NULL) { return(CLIPS_TRUE); }
   pos++;
   
   while (test_ptr != NULL)
     {
      if (! GetNumericArgument(test_ptr,"<=",&rv2,CLIPS_FALSE,pos)) return(CLIPS_FALSE);
      if (rv1.type == INTEGER)
        {
         if (rv2.type == INTEGER)
           {
            if (ValueToLong(rv1.value) > ValueToLong(rv2.value))
              { return(CLIPS_FALSE); }
           }
         else
           { 
            if ((double) ValueToLong(rv1.value) > ValueToDouble(rv2.value))
              { return(CLIPS_FALSE); }
           }
        }
      else
        {
         if (rv2.type == INTEGER)
           {
            if (ValueToDouble(rv1.value) > (double) ValueToLong(rv2.value))
              { return(CLIPS_FALSE); }
           }
         else
           { 
            if (ValueToDouble(rv1.value) > ValueToDouble(rv2.value))
              { return(CLIPS_FALSE); }
           }
        }
      
      rv1.type = rv2.type;
      rv1.value = rv2.value;
      test_ptr = GetNextArgument(test_ptr);
      pos++;
     }
   
   return(CLIPS_TRUE);
  } 

/****************************************/
/* GreaterThanOrEqualFunction                */
/****************************************/
globle BOOLEAN GreaterThanOrEqualFunction()
  {
   EXPR_PTR test_ptr;
   DATA_OBJECT rv1, rv2;
   int pos = 1;

   test_ptr = GetFirstArgument();

   if (test_ptr == NULL) { return(CLIPS_TRUE); }
   if (! GetNumericArgument(test_ptr,">=",&rv1,CLIPS_FALSE,pos)) return(CLIPS_FALSE);
   test_ptr = GetNextArgument(test_ptr);
   if (test_ptr == NULL) { return(CLIPS_TRUE); }
   pos++;
   
   while (test_ptr != NULL)
     {
      if (! GetNumericArgument(test_ptr,">=",&rv2,CLIPS_FALSE,pos)) return(CLIPS_FALSE);
      if (rv1.type == INTEGER)
        {
         if (rv2.type == INTEGER)
           {
            if (ValueToLong(rv1.value) < ValueToLong(rv2.value))
              { return(CLIPS_FALSE); }
           }
         else
           { 
            if ((double) ValueToLong(rv1.value) < ValueToDouble(rv2.value))
              { return(CLIPS_FALSE); }
           }
        }
      else
        {
         if (rv2.type == INTEGER)
           {
            if (ValueToDouble(rv1.value) < (double) ValueToLong(rv2.value))
              { return(CLIPS_FALSE); }
           }
         else
           { 
            if (ValueToDouble(rv1.value) < ValueToDouble(rv2.value))
              { return(CLIPS_FALSE); }
           }
        }
      
      rv1.type = rv2.type;
      rv1.value = rv2.value;
      test_ptr = GetNextArgument(test_ptr);
      pos++;
     }
   
   return(CLIPS_TRUE);
  } 

/****************************************/
/* LessThanFunction                            */
/****************************************/
globle BOOLEAN LessThanFunction()
  {
   EXPR_PTR test_ptr;
   DATA_OBJECT rv1, rv2;
   int pos = 1;

   test_ptr = GetFirstArgument();

   if (test_ptr == NULL) { return(CLIPS_TRUE); }
   if (! GetNumericArgument(test_ptr,"<",&rv1,CLIPS_FALSE,pos)) return(CLIPS_FALSE);
   test_ptr = GetNextArgument(test_ptr);
   if (test_ptr == NULL) { return(CLIPS_TRUE); }
   pos++;
   
   while (test_ptr != NULL)
     {
      if (! GetNumericArgument(test_ptr,"<",&rv2,CLIPS_FALSE,pos)) return(CLIPS_FALSE);
      if (rv1.type == INTEGER)
        {
         if (rv2.type == INTEGER)
           {
            if (ValueToLong(rv1.value) >= ValueToLong(rv2.value))
              { return(CLIPS_FALSE); }
           }
         else
           { 
            if ((double) ValueToLong(rv1.value) >= ValueToDouble(rv2.value))
              { return(CLIPS_FALSE); }
           }
        }
      else
        {
         if (rv2.type == INTEGER)
           {
            if (ValueToDouble(rv1.value) >= (double) ValueToLong(rv2.value))
              { return(CLIPS_FALSE); }
           }
         else
           { 
            if (ValueToDouble(rv1.value) >= ValueToDouble(rv2.value))
              { return(CLIPS_FALSE); }
           }
        }
      
      rv1.type = rv2.type;
      rv1.value = rv2.value;
      test_ptr = GetNextArgument(test_ptr);
      pos++;
     }
   
   return(CLIPS_TRUE);
  }
 
/****************************************/
/* GreaterThanFunction                         */
/****************************************/
globle BOOLEAN GreaterThanFunction()
  {
   EXPR_PTR test_ptr;
   DATA_OBJECT rv1, rv2;
   int pos = 1;

   test_ptr = GetFirstArgument();

   if (test_ptr == NULL) { return(CLIPS_TRUE); }
   if (! GetNumericArgument(test_ptr,">",&rv1,CLIPS_FALSE,pos)) return(CLIPS_FALSE);
   test_ptr = GetNextArgument(test_ptr);
   if (test_ptr == NULL) { return(CLIPS_TRUE); }
   pos++;
   
   while (test_ptr != NULL)
     {
      if (! GetNumericArgument(test_ptr,">",&rv2,CLIPS_FALSE,pos)) return(CLIPS_FALSE);
      if (rv1.type == INTEGER)
        {
         if (rv2.type == INTEGER)
           {
            if (ValueToLong(rv1.value) <= ValueToLong(rv2.value))
              { return(CLIPS_FALSE); }
           }
         else
           { 
            if ((double) ValueToLong(rv1.value) <= ValueToDouble(rv2.value))
              { return(CLIPS_FALSE); }
           }
        }
      else
        {
         if (rv2.type == INTEGER)
           {
            if (ValueToDouble(rv1.value) <= (double) ValueToLong(rv2.value))
              { return(CLIPS_FALSE); }
           }
         else
           { 
            if (ValueToDouble(rv1.value) <= ValueToDouble(rv2.value))
              { return(CLIPS_FALSE); }
           }
        }
      
      rv1.type = rv2.type;
      rv1.value = rv2.value;
      test_ptr = GetNextArgument(test_ptr);
      pos++;
     }
   
   return(CLIPS_TRUE);
  } 
  
/****************************************/
/* NumericEqualFunction                      */
/****************************************/
globle BOOLEAN NumericEqualFunction()
  {
   EXPR_PTR test_ptr;
   DATA_OBJECT rv1, rv2;
   int pos = 1;

   test_ptr = GetFirstArgument();

   if (test_ptr == NULL) { return(CLIPS_TRUE); }
   if (! GetNumericArgument(test_ptr,"=",&rv1,CLIPS_FALSE,pos)) return(CLIPS_FALSE);
   test_ptr = GetNextArgument(test_ptr);
   if (test_ptr == NULL) { return(CLIPS_TRUE); }
   
   while (test_ptr != NULL)
     {
      if (! GetNumericArgument(test_ptr,"=",&rv2,CLIPS_FALSE,pos)) return(CLIPS_FALSE);
      if (rv1.type == INTEGER)
        {
         if (rv2.type == INTEGER)
           {
            if (ValueToLong(rv1.value) != ValueToLong(rv2.value))
              { return(CLIPS_FALSE); }
           }
         else
           { 
            if ((double) ValueToLong(rv1.value) != ValueToDouble(rv2.value))
              { return(CLIPS_FALSE); }
           }
        }
      else
        {
         if (rv2.type == INTEGER)
           {
            if (ValueToDouble(rv1.value) != (double) ValueToLong(rv2.value))
              { return(CLIPS_FALSE); }
           }
         else
           { 
            if (ValueToDouble(rv1.value) != ValueToDouble(rv2.value))
              { return(CLIPS_FALSE); }
           }
        }
      
      test_ptr = GetNextArgument(test_ptr);
      pos++;
     }
   
   return(CLIPS_TRUE);
  } 

 
/****************************************/
/* NumericNotEqualFunction                    */
/****************************************/
globle BOOLEAN NumericNotEqualFunction()
  {
   EXPR_PTR test_ptr;
   DATA_OBJECT rv1, rv2;
   int pos = 1;

   test_ptr = GetFirstArgument();

   if (test_ptr == NULL) { return(CLIPS_TRUE); }
   if (! GetNumericArgument(test_ptr,"<>",&rv1,CLIPS_FALSE,pos)) return(CLIPS_FALSE);
   test_ptr = GetNextArgument(test_ptr);
   if (test_ptr == NULL) { return(CLIPS_TRUE); }
   pos++;
   
   while (test_ptr != NULL)
     {
      if (! GetNumericArgument(test_ptr,"<>",&rv2,CLIPS_FALSE,pos)) return(CLIPS_FALSE);
      if (rv1.type == INTEGER)
        {
         if (rv2.type == INTEGER)
           {
            if (ValueToLong(rv1.value) == ValueToLong(rv2.value))
              { return(CLIPS_FALSE); }
           }
         else
           { 
            if ((double) ValueToLong(rv1.value) == ValueToDouble(rv2.value))
              { return(CLIPS_FALSE); }
           }
        }
      else
        {
         if (rv2.type == INTEGER)
           {
            if (ValueToDouble(rv1.value) == (double) ValueToLong(rv2.value))
              { return(CLIPS_FALSE); }
           }
         else
           { 
            if (ValueToDouble(rv1.value) == ValueToDouble(rv2.value))
              { return(CLIPS_FALSE); }
           }
        }
      
      test_ptr = GetNextArgument(test_ptr);
      pos++;
     }
   
   return(CLIPS_TRUE);
  } 
   
/****************************************/
/* OddpFunction:                                */
/****************************************/
globle BOOLEAN OddpFunction()
  {
   DATA_OBJECT valstruct;
   long num, halfnum;

   if (ArgCountCheck("oddp",EXACTLY,1) == -1) return(CLIPS_FALSE);
   if (ArgTypeCheck("oddp",1,INTEGER,&valstruct) == CLIPS_FALSE) return(CLIPS_FALSE);
   
   num = DOToLong(valstruct);

   halfnum = (num / 2) * 2;
   if (num == halfnum) return(CLIPS_FALSE);
   
   return(CLIPS_TRUE);
  }
  
/****************************************/
/* EvenpFunction:                               */
/****************************************/
globle BOOLEAN EvenpFunction()
  {
   DATA_OBJECT valstruct;
   long num, halfnum;

   if (ArgCountCheck("evenp",EXACTLY,1) == -1) return(CLIPS_FALSE);
   if (ArgTypeCheck("evenp",1,INTEGER,&valstruct) == CLIPS_FALSE) return(CLIPS_FALSE);
   
   num = DOToLong(valstruct);

   halfnum = (num / 2) * 2;
   if (num != halfnum) return(CLIPS_FALSE);
   
   return(CLIPS_TRUE);
  }

/************************************************************/
/* SetAutoFloatDividendCommand:                        */
/************************************************************/
globle int SetAutoFloatDividendCommand()
  {
   int oldValue;
   DATA_OBJECT arg_ptr;
   
   oldValue = AutoFloatDividend;
   
   if (ArgCountCheck("set-auto-float-dividend",EXACTLY,1) == -1) 
     { return(oldValue); }
     
   RtnUnknown(1,&arg_ptr);
     
   if ((arg_ptr.value == (VOID *) CLIPSFalseSymbol) && (arg_ptr.type == SYMBOL))
     { AutoFloatDividend = CLIPS_FALSE; }
   else
     { AutoFloatDividend = CLIPS_TRUE; }
     
   return(oldValue);
  }
  
/************************************************************/
/* GetAutoFloatDividendCommand:                        */
/************************************************************/
globle int GetAutoFloatDividendCommand()
  {   
   ArgCountCheck("get-auto-float-dividend",EXACTLY,0);
     
   return(AutoFloatDividend);
  }
  
/************************************************************/
/* GetAutoFloatDividend:                        */
/************************************************************/
globle BOOLEAN GetAutoFloatDividend()
  {
   return(AutoFloatDividend);
  }

/***************************************************/
/* SetAutoFloatDividend:                           */
/***************************************************/
globle BOOLEAN SetAutoFloatDividend(value)
  int value;
  { 
   int ov;
   
   ov = AutoFloatDividend;
   AutoFloatDividend = value;
   return(ov);
  }
