   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*             CLIPS Version 5.10  07/17/91            */
   /*                                                     */
   /*                    STRINGS MODULE                   */
   /*******************************************************/

/*************************************************************/
/* Purpose:                                                  */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Gary D. Riley                                        */
/*                                                           */
/* Contributing Programmer(s):                               */
/*      Barry Cameron                                        */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/

#define _STRING_SOURCE_
   
#include "setup.h"

#if STRING_FUNCTIONS

#include <stdio.h>
#define _CLIPS_STDIO_
#include <ctype.h>
#include <string.h>

#include "constant.h"
#include "clipsmem.h"
#include "utility.h"
#include "expressn.h"
#include "evaluatn.h"
#include "scanner.h"
#include "constrct.h"
#include "router.h"
#include "spclform.h"

#if DEFRULE_CONSTRUCT
#include "defrule.h"
#endif

/****************************************/
/* GLOBAL INTERNAL FUNCTION DEFINITIONS */
/****************************************/
 
#if ANSI_COMPILER  
   VOID                           StringFunctionDefinitions(void);
   VOID                           StrCatFunction(DATA_OBJECT_PTR);
   VOID                           SymCatFunction(DATA_OBJECT_PTR);
   VOID                           StrOrSymCatFunction(DATA_OBJECT_PTR ,int);
   long int                       StrLengthFunction(void);
   VOID                           UpcaseFunction(DATA_OBJECT_PTR);
   VOID                           LowcaseFunction(DATA_OBJECT_PTR);
   long int                       StrCompareFunction(void);
   VOID                          *SubStringFunction(void);
   VOID                           StrIndexFunction(DATA_OBJECT_PTR);
   VOID                           EvalFunction(DATA_OBJECT_PTR);
   int                            BuildFunction(void);
#else
   VOID                           StringFunctionDefinitions();
   VOID                           StrCatFunction();
   VOID                           SymCatFunction();
   VOID                           StrOrSymCatFunction();
   long int                       StrLengthFunction();
   VOID                           UpcaseFunction();
   VOID                           LowcaseFunction();
   long int                       StrCompareFunction();
   VOID                          *SubStringFunction();
   VOID                           StrIndexFunction();
   VOID                           EvalFunction();
   int                            BuildFunction();
#endif

#if ! RUN_TIME
/********************************************/
/* StringFunctionDefinitions:               */
/********************************************/
globle VOID StringFunctionDefinitions()
  {
   DefineFunction("str-cat",     'u', PTIF StrCatFunction,      "StrCatFunction");
   DefineFunction("sym-cat",     'u', PTIF SymCatFunction,      "SymCatFunction");
   DefineFunction("str-length",  'l', PTIF StrLengthFunction,   "StrLengthFunction");
   DefineFunction("str-compare", 'l', PTIF StrCompareFunction,  "StrCompareFunction");
   DefineFunction("upcase",      'u', PTIF UpcaseFunction,      "UpcaseFunction");
   DefineFunction("lowcase",     'u', PTIF LowcaseFunction,     "LowcaseFunction");
   DefineFunction("sub-string",  's', PTIF SubStringFunction,   "SubStringFunction");
   DefineFunction("str-index",   'u', PTIF StrIndexFunction,    "StrIndexFunction");
   DefineFunction("eval",        'u', PTIF EvalFunction,        "EvalFunction");
   DefineFunction("build",       'b', PTIF BuildFunction,       "BuildFunction");
  }
#endif
 
/*****************************************************************/
/* StrCatFunction: Concatenates a series of primitive data types */
/*   together yielding a string.                                 */
/*   Syntax: (str-cat <value-1> ... <value-n>)                   */
/*****************************************************************/
globle VOID StrCatFunction(cat_value)
  DATA_OBJECT_PTR cat_value;
  {
   StrOrSymCatFunction(cat_value,STRING);
  }
  
/*****************************************************************/
/* SymCatFunction: Concatenates a series of primitive data types */
/*   together yielding a symbol.                                 */
/*   Syntax: (str-cat <value-1> ... <value-n>)                   */
/*****************************************************************/
globle VOID SymCatFunction(cat_value)
  DATA_OBJECT_PTR cat_value;
  {
   StrOrSymCatFunction(cat_value,SYMBOL);
  }
  
/****************************************/
/* StrOrSymCatFunction:                      */
/****************************************/
globle VOID StrOrSymCatFunction(cat_value,type)
  DATA_OBJECT_PTR cat_value;
  int type;
  {
   DATA_OBJECT arg_ptr;
   int numa, i, total, j;
   char *str_mem, **str_arr;
   VOID *hash_ptr;
   char *functionName;
   
   SetpType(cat_value,type);
   SetpValue(cat_value,(VOID *) AddSymbol(""));
   
   if (type == STRING) functionName = "str-cat";
   else functionName = "sym-cat";
   
   /*============================================================*/
   /* Functions str-cat and sym-cat expect exactly one argument. */
   /*============================================================*/
   
   if ((numa = ArgCountCheck(functionName,AT_LEAST,1)) == -1)
     { return; }
     
   str_arr = (char **) gm1((int) sizeof(char *) * numa);

   total = 1;
   for (i = 1 ; i <= numa ; i++)
     {
      RtnUnknown(i,&arg_ptr);
      if ((GetType(arg_ptr) == STRING) || 
#if OBJECT_SYSTEM
          (GetType(arg_ptr) == INSTANCE_NAME) ||
#endif
          (GetType(arg_ptr) == SYMBOL))
        { str_arr[i - 1] = ValueToString(ClipsGetValue(arg_ptr)); }
      else if (GetType(arg_ptr) == FLOAT)
        {
         hash_ptr = AddSymbol(FloatToString(ValueToDouble(ClipsGetValue(arg_ptr))));
         str_arr[i - 1] = ValueToString(hash_ptr);
        }
      else if (GetType(arg_ptr) == INTEGER)
        {
         hash_ptr = AddSymbol(LongIntegerToString(ValueToLong(ClipsGetValue(arg_ptr))));
         str_arr[i - 1] = ValueToString(hash_ptr);
        }
      else
        {
         ExpectedTypeError(functionName,i,"string, instance name, symbol, float, or integer");
         rm(str_arr,(int) sizeof(char *) * numa);
         SetHaltExecution(CLIPS_TRUE);
         SetEvaluationError(CLIPS_TRUE);
         SetpValue(cat_value,(VOID *) AddSymbol(""));
         return;
        }
        
      total += strlen(str_arr[i - 1]);
     } 

   str_mem = (char *) gm2 (((int) sizeof(char) * total));

   j = 0;
   for (i = 0 ; i < numa ; i++)
     {
      sprintf(&str_mem[j],"%s",str_arr[i]);
      j += strlen(str_arr[i]);
     }

   SetpValue(cat_value,(VOID *) AddSymbol(str_mem));
   rm(str_mem,(int) sizeof(char) * total);
   rm(str_arr,(int) sizeof(char *) * numa);

   return;
  }

/***************************************************************/
/* StrLengthFunction:  Returns the length of a STRING or SYMBOL. */
/*   Syntax: (str-length <string-or-symbol>)                     */
/***************************************************************/
globle long int StrLengthFunction()
  {
   DATA_OBJECT val1;

   /*===================================================*/
   /* Function str_length expects exactly one argument. */
   /*===================================================*/
   
   if (ArgCountCheck("str-length",EXACTLY,1) == -1)
     { return(-1L); }

   /*================================================*/
   /* The argument should be of type SYMBOL or STRING. */
   /*================================================*/
   
   if (ArgTypeCheck("str-length",1,SYMBOL_OR_STRING,&val1) == CLIPS_FALSE)
     { return(-1L); }
     
   /*==========================================*/
   /* Return the length of the string or symbol. */
   /*==========================================*/
   
   return( (long) strlen(DOToString(val1)));
  }
  
/******************************************************/
/* UpcaseFunction:  Returns a STRING or SYMBOL in uppercase. */
/*   Syntax: (upcase <string-or-symbol>)                */
/******************************************************/
globle VOID UpcaseFunction(rv)
  DATA_OBJECT_PTR rv;
  {
   DATA_OBJECT val1;
   int i, slen;
   char *osptr, *nsptr;

   /*===================================================*/
   /* Function upcase expects exactly one argument. */
   /*===================================================*/
   
   if (ArgCountCheck("upcase",EXACTLY,1) == -1)
     { 
      SetpType(rv,STRING);
      SetpValue(rv,(VOID *) AddSymbol(""));
      return;
     }

   /*================================================*/
   /* The argument should be of type SYMBOL or STRING. */
   /*================================================*/
   
   if (ArgTypeCheck("upcase",1,SYMBOL_OR_STRING,&val1) == CLIPS_FALSE)
     { 
      SetpType(rv,STRING);
      SetpValue(rv,(VOID *) AddSymbol(""));
      return;
     }
   
   osptr = DOToString(val1);
   slen = strlen(osptr) + 1;
   nsptr = (char *) gm2(slen);
   
   for (i = 0  ; i < slen ; i++)
     {
      if (islower(osptr[i]))
        { nsptr[i] = toupper(osptr[i]); }
      else 
        { nsptr[i] = osptr[i]; }
     }
   
   /*===============================*/
   /* Return the uppercased string. */
   /*===============================*/
   
   SetpType(rv,GetType(val1));
   SetpValue(rv,(VOID *) AddSymbol(nsptr));
   rm(nsptr,slen);
   return;
  }
  
/******************************************************/
/* LowcaseFunction:  Returns a STRING or SYMBOL in lowercase. */
/*   Syntax: (lowcase <string-or-symbol>)               */
/******************************************************/
globle VOID LowcaseFunction(rv)
  DATA_OBJECT_PTR rv;
  {
   DATA_OBJECT val1;
   int i, slen;
   char *osptr, *nsptr;

   /*===================================================*/
   /* Function lowcase expects exactly one argument. */
   /*===================================================*/
   
   if (ArgCountCheck("lowcase",EXACTLY,1) == -1)
     { 
      SetpType(rv,STRING);
      SetpValue(rv,(VOID *) AddSymbol(""));
      return;
     }

   /*================================================*/
   /* The argument should be of type SYMBOL or STRING. */
   /*================================================*/
   
   if (ArgTypeCheck("lowcase",1,SYMBOL_OR_STRING,&val1) == CLIPS_FALSE)
     { 
      SetpType(rv,STRING);
      SetpValue(rv,(VOID *) AddSymbol(""));
      return;
     }

   osptr = DOToString(val1);
   slen = strlen(osptr) + 1;
   nsptr = (char *) gm2(slen);
   
   for (i = 0  ; i < slen ; i++)
     {
      if (isupper(osptr[i]))
        { nsptr[i] = tolower(osptr[i]); }
      else 
        { nsptr[i] = osptr[i]; }
     }
   
   /*===============================*/
   /* Return the lowercased string. */
   /*===============================*/
   
   SetpType(rv,GetType(val1));
   SetpValue(rv,(VOID *) AddSymbol(nsptr));
   rm(nsptr,slen);
   return;
  }
  
/************************************************************/
/* StrCompareFunction:  Compares two strings.               */
/*   Syntax: (str_compare <string-1> <string-2> [<length>]) */
/*   Returns 0 is <string-1> and <string-2> are equal, < 0  */
/*   if <string-1> is less than <string-2>, and > 0 if      */
/*   <string-1> is greater than <string-2>. This function   */
/*   based on the C functions strcmp and strncmp.           */
/************************************************************/
globle long int StrCompareFunction()
  {
   int nargs, length;
   DATA_OBJECT val1, val2, val3;
   long rv;

   /*=======================================================*/
   /* Function str-compare expects either 2 or 3 arguments. */
   /*=======================================================*/
   
   if (ArgCountCheck("str-compare",AT_LEAST,2) == -1)
     { return(0L); }
     
   if ((nargs = ArgCountCheck("str-compare",NO_MORE_THAN,3)) == -1)
     { return(0L); }

   /*===========================================================*/
   /* The first two arguments should be of type SYMBOL or STRING. */
   /*===========================================================*/
    
   if (ArgTypeCheck("str-compare",1,SYMBOL_OR_STRING,&val1) == CLIPS_FALSE)
     { return(0L); }
   
   if (ArgTypeCheck("str-compare",2,SYMBOL_OR_STRING,&val2) == CLIPS_FALSE)
     { return(0L); }
     
   /*===================================================*/
   /* Compare the strings. Use the 3rd argument for the */
   /* maximum length of comparison, if it is provided.  */         
   /*===================================================*/
   
   if (nargs == 3)
     {
      if (ArgTypeCheck("str-compare",3,INTEGER,&val3) == CLIPS_FALSE)
        { return(0L); }
        
      length = CoerceToInteger(GetType(val3),ClipsGetValue(val3));
      rv = strncmp(DOToString(val1),DOToString(val2),length);
     }
   else
     { rv = strcmp(DOToString(val1),DOToString(val2)); }
   
   if (rv < 0) rv = -1;
   else if (rv > 0) rv = 1;
   return(rv);
  }
       
/*****************************************************/
/* SubStringFunction:  Returns a portion of a string */
/*   and returns the pointer to a new string.        */
/*****************************************************/
globle VOID *SubStringFunction()
  {
   DATA_OBJECT val_ptr;
   char *tmp_str, *ret_str;
   int start, end, i, j;
   VOID *rv;

   /*===================================*/
   /* Check and retrieve the arguments. */
   /*===================================*/
   
   if (ArgCountCheck("sub-string",EXACTLY,3) == -1)
     { return((VOID *) AddSymbol("")); }

   if (ArgTypeCheck("sub-string",1,INTEGER,&val_ptr) == CLIPS_FALSE)
     { return((VOID *) AddSymbol("")); }

   start = CoerceToInteger(val_ptr.type,val_ptr.value) - 1;

   if (ArgTypeCheck("sub-string",2,INTEGER,&val_ptr) == CLIPS_FALSE) 
     {  return((VOID *) AddSymbol("")); }

   end = CoerceToInteger(val_ptr.type,val_ptr.value) - 1;

   if (ArgTypeCheck("sub-string",3,SYMBOL_OR_STRING,&val_ptr) == CLIPS_FALSE) 
     { return((VOID *) AddSymbol("")); }  

   /*================================================*/
   /* If parameters are out of range return an error */
   /*================================================*/
      
   if (start < 0) start = 0;
   if (end > (int) strlen(DOToString(val_ptr)))
     { end = strlen(DOToString(val_ptr)); }

   /*==================================*/
   /* If the start is greater than the */
   /* end, return a null string.       */
   /*==================================*/
      
   if (start > end)
     { return((VOID *) AddSymbol("")); }
   
   /*=============================================*/
   /* Otherwise, allocate the string and copy the */
   /* designated portion of the old string to the */
   /* new string.                                 */
   /*=============================================*/
   
   else
     {
      ret_str = (char *) gm2(end - start +2);  /* (end - start) inclusive + EOS */
      tmp_str = DOToString(val_ptr);
      for(j=0, i=start;i <= end; i++, j++)
        { *(ret_str+j) = *(tmp_str+i); }
      *(ret_str+j) = '\0';
     } 
     
   /*========================*/
   /* Return the new string. */
   /*========================*/

   rv = (VOID *) AddSymbol(ret_str);
   rm(ret_str,end - start + 2);
   return(rv);
  }

/**********************************************************/
/* StrIndexFunction: Returns the position of the first string in */
/*   the second. If string is not found, 0 is returned.   */
/**********************************************************/
globle VOID StrIndexFunction(result)
  DATA_OBJECT_PTR result;
  {
   DATA_OBJECT val_ptr1, val_ptr2;
   char *strg1, *strg2;
   int i, j;

   result->type = SYMBOL;
   result->value = (VOID *) CLIPSFalseSymbol;
   
   /*===================================*/
   /* Check and retrieve the arguments. */
   /*===================================*/
   
   if (ArgCountCheck("str-index",EXACTLY,2) == -1) return;

   if (ArgTypeCheck("str-index",1,SYMBOL_OR_STRING,&val_ptr1) == CLIPS_FALSE) return;

   if (ArgTypeCheck("str-index",2,SYMBOL_OR_STRING,&val_ptr2) == CLIPS_FALSE) return;
 
   strg1 = DOToString(val_ptr1);
   strg2 = DOToString(val_ptr2);
 
   /*=================================*/
   /* Find the position in string2 of */
   /* string1 (counting from 1).      */
   /*=================================*/

   if (strlen(strg1) == 0) 
     {
      result->type = INTEGER;
      result->value = (VOID *) AddLong((long) strlen(strg2) + 1L);
      return;
     }
   
   for (i=1; *strg2; i++, strg2++)
     {
      for (j=0; *(strg1+j) && *(strg1+j) == *(strg2+j); j++);
             
      if (*(strg1+j) == '\0') 
        {
         result->type = INTEGER;
         result->value = (VOID *) AddLong((long) i);
         return;
        }
     }
 
   return;
  }

/****************************************/
/* EVAL:  Evaluates a STRING or SYMBOL. */
/*   Syntax: (eval <string-or-symbol>)  */
/****************************************/
#if (! RUN_TIME) && (! BLOAD_ONLY)
globle VOID EvalFunction(rv)
  DATA_OBJECT_PTR rv;
  {
   DATA_OBJECT val1;
   struct expr *top;
   int ov;
   static int depth = 0;
   char logicalNameBuffer[20];
   struct BindInfo *oldBinds;

   /*===================================================*/
   /* Function eval expects exactly one argument. */
   /*===================================================*/
 
   if (ArgCountCheck("eval",EXACTLY,1) == -1)
     {
      SetpType(rv,SYMBOL);
      SetpValue(rv,(VOID *) CLIPSFalseSymbol); 
      return; 
     }

   /*================================================*/
   /* The argument should be of type SYMBOL or STRING. */
   /*================================================*/
   
   if (ArgTypeCheck("eval",1,SYMBOL_OR_STRING,&val1) == CLIPS_FALSE)
     { 
      SetpType(rv,SYMBOL);
      SetpValue(rv,(VOID *) CLIPSFalseSymbol);
      return;
     }
   
   /*======================================================*/
   /* Evaluate the string. Create a different logical name */
   /* for use each time the eval function is called.       */                   
   /*======================================================*/
   
   depth++;
   sprintf(logicalNameBuffer,"Eval-%d",depth);
   if (OpenStringSource(logicalNameBuffer,DOToString(val1),0) == 0)
     {
      SetpType(rv,SYMBOL);
      SetpValue(rv,(VOID *) CLIPSFalseSymbol);
      depth--;
      return; 
     }
      
   ov = GetPPBufferStatus();
   SetPPBufferStatus(CLIPS_FALSE);
   oldBinds = GetParsedBindNames();
   SetParsedBindNames(NULL);
   
   top = ParseAtomOrExpression(logicalNameBuffer);
   
   SetPPBufferStatus(ov);
   ClearParsedBindNames();
   SetParsedBindNames(oldBinds); 
   
   if (top == NULL)
     { 
      CloseStringSource(logicalNameBuffer);
      SetpType(rv,SYMBOL);
      SetpValue(rv,(VOID *) CLIPSFalseSymbol);
      depth--;
      return; 
     }

   if (ExpressionContainsVariables(top,CLIPS_FALSE))
     {
      PrintCLIPS(WERROR,"Some variables could not be accessed by the eval function\n");
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      SetpType(rv,SYMBOL);
      SetpValue(rv,(VOID *) CLIPSFalseSymbol);
     }
   else
     { EvaluateExpression(top,rv); }
 
   depth--;
   ReturnExpression(top);
   CloseStringSource(logicalNameBuffer);
  }
#else
globle VOID EvalFunction(rv)
  DATA_OBJECT_PTR rv;
  {   
   PrintCLIPS(WERROR,"Function eval does not work in run time modules\n");
   SetpType(rv,SYMBOL);
   SetpValue(rv,(VOID *) CLIPSFalseSymbol);
  }
#endif

/*****************************************/
/* BUILD:  Evaluates a STRING or SYMBOL. */
/*   Syntax: (build <string-or-symbol>)  */
/*****************************************/
#if (! RUN_TIME) && (! BLOAD_ONLY)
globle int BuildFunction()
  {
   char *command_name;
   struct token com_tkn;
   DATA_OBJECT val1;
   int errorFlag;

   /*===================================================*/
   /* Function eval expects exactly one argument. */
   /*===================================================*/
 
   if (ArgCountCheck("build",EXACTLY,1) == -1) return(CLIPS_FALSE);

   /*================================================*/
   /* The argument should be of type SYMBOL or STRING. */
   /*================================================*/
   
   if (ArgTypeCheck("build",1,SYMBOL_OR_STRING,&val1) == CLIPS_FALSE)
     { return(CLIPS_FALSE); }
   
   /*=====================================================*/
   /* No additions during an assert or retract operation. */
   /*=====================================================*/

#if DEFRULE_CONSTRUCT
   if (GetRuleDeletions() == CLIPS_FALSE) return(CLIPS_FALSE);
#endif

   /*======================*/
   /* Evaluate the string. */
   /*======================*/
   
   if (OpenStringSource("build",DOToString(val1),0) == 0)
     { return(CLIPS_FALSE); }
      
   GetToken("build",&com_tkn);
     
   if (com_tkn.type != LPAREN)
     {
      CloseStringSource("build");
      return(CLIPS_FALSE);
     }
   
   GetToken("build",&com_tkn);
   if (com_tkn.type != SYMBOL) 
     {
      CloseStringSource("build");
      return(CLIPS_FALSE);
     }
     
   command_name = ValueToString(com_tkn.value);
   
   errorFlag = ParseConstruct(command_name,"build");
   
   CloseStringSource("build");
   
   if (errorFlag == 1)
     { 
      PrintCLIPS(WERROR,"\nERROR:\n");
      PrintInChunks(WERROR,GetPPBuffer());
      PrintCLIPS(WERROR,"\n");
     }
        
   DestroyPPBuffer();
      
   if (errorFlag == 0) return(CLIPS_TRUE);
   
   return(CLIPS_FALSE);
  }
#else
globle int BuildFunction()
  {   
   PrintCLIPS(WERROR,"Function build does not work in run time modules\n");
   return(CLIPS_FALSE);
  }
#endif
#endif
