   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*             CLIPS Version 5.10  07/17/91            */
   /*                                                     */
   /*                   MULTIVAR MODULE                   */
   /*******************************************************/

/*************************************************************/
/* Purpose:                                                  */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Gary D. Riley                                        */
/*      Barry Cameron                                        */
/*      Brian Donnell                                        */
/*                                                           */
/* Contributing Programmer(s):                               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/

#define _MULTIVAR_SOURCE_

#include "setup.h"

#if MULTIFIELD_FUNCTIONS || OBJECT_SYSTEM

#include <stdio.h>
#define _CLIPS_STDIO_
#include <string.h>

#include "clipsmem.h"
#include "facts.h"
#include "utility.h"
#include "evaluatn.h"
#include "router.h"
#include "multivar.h"

#if OBJECT_SYSTEM
#include "object.h"
#endif

#if ANSI_COMPILER 
#if ! RUN_TIME
   VOID                           MultifieldFunctionDefinitions(void);
#endif
#if MULTIFIELD_FUNCTIONS
   VOID                           MvDeleteFunction(DATA_OBJECT_PTR);
   VOID                           MvReplaceFunction(DATA_OBJECT_PTR);
   VOID                           MvAppendFunction(DATA_OBJECT_PTR);
   VOID                           StrExplodeFunction(DATA_OBJECT_PTR);
   VOID                          *StrImplodeFunction(void);
   VOID                           MvSubseqFunction(DATA_OBJECT_PTR);
   VOID                           NthFunction(DATA_OBJECT_PTR);
   BOOLEAN                        SubsetFunction(void);
   VOID                           MemberFunction(DATA_OBJECT_PTR);
   static int                     FindItemInSegment(int,VOID *,DATA_OBJECT_PTR);
#endif
   static VOID                    MVRangeError(int,int,int,char *);
#else
#if MULTIFIELD_FUNCTIONS
#if ! RUN_TIME
   VOID                           MultifieldFunctionDefinitions();
#endif
   VOID                           MvDeleteFunction();
   VOID                           MvReplaceFunction();
   VOID                           MvAppendFunction();
   VOID                           StrExplodeFunction();
   VOID                          *StrImplodeFunction();
   VOID                           MvSubseqFunction();
   VOID                           NthFunction();
   BOOLEAN                        SubsetFunction();
   VOID                           MemberFunction();
   static int                     FindItemInSegment();
#endif
   static VOID                    MVRangeError();
#endif

#endif

#if MULTIFIELD_FUNCTIONS

#if ! RUN_TIME
/*********************************************/
/* MultifieldFunctionDefinitions:            */
/*********************************************/
globle VOID MultifieldFunctionDefinitions()
  {
   DefineFunction("mv-subseq",   'm', PTIF MvSubseqFunction,   "MvSubseqFunction");
   DefineFunction("mv-delete",   'm', PTIF MvDeleteFunction,   "MvDeleteFunction");
   DefineFunction("delete$",     'm', PTIF MvDeleteFunction,   "MvDeleteFunction");
   DefineFunction("mv-replace",  'm', PTIF MvReplaceFunction,  "MvReplaceFunction");
   DefineFunction("mv-append",   'm', PTIF MvAppendFunction,   "MvAppendFunction");
   DefineFunction("create$",     'm', PTIF MvAppendFunction,   "MvAppendFunction");
   DefineFunction("str-explode", 'm', PTIF StrExplodeFunction, "StrExplodeFunction");
   DefineFunction("str-implode", 's', PTIF StrImplodeFunction, "StrImplodeFunction");
   DefineFunction("nth",         'u', PTIF NthFunction,        "NthFunction");
   DefineFunction("nth$",        'u', PTIF NthFunction,        "NthFunction");
   DefineFunction("member",      'u', PTIF MemberFunction,     "MemberFunction");
   DefineFunction("member$",     'u', PTIF MemberFunction,     "MemberFunction");
   DefineFunction("subset",      'b', PTIF SubsetFunction,     "SubsetFunction");
   DefineFunction("subsetp",     'b', PTIF SubsetFunction,     "SubsetFunction"); 
  }
#endif

/*****************************************************/
/* MvDeleteFunction:  Delete function for segments.  */
/*   Note: delete does have to create a new segment. */
/*****************************************************/
globle VOID MvDeleteFunction(seg_value)
  DATA_OBJECT_PTR seg_value;
  {
   DATA_OBJECT val_ptr1, val_ptr2;
   int n;

   if (ArgCountCheck("mv-delete",EXACTLY,2) == -1)
     {
      SetMultifieldErrorValue(seg_value);
      return;
     }

   if ((ArgTypeCheck("mv-delete",1,INTEGER,&val_ptr1) == CLIPS_FALSE) ||
       (ArgTypeCheck("mv-delete",2,MULTIFIELD,&val_ptr2) == CLIPS_FALSE))
     {
      SetEvaluationError(CLIPS_TRUE);
      SetMultifieldErrorValue(seg_value);
      return;
     }

   n = (int) DOToLong(val_ptr1);
   if (DeleteMultiValueField(seg_value,&val_ptr2,n,n,"mv-delete") == CLIPS_FALSE)
     {
      SetEvaluationError(CLIPS_TRUE);
      SetMultifieldErrorValue(seg_value);
      return;
     }
   
   return;
  }

/***********************************************/
/* MvReplaceFunction:  Replace function for segments. */
/***********************************************/
globle VOID MvReplaceFunction(seg_value)
  DATA_OBJECT_PTR seg_value;
  {
   DATA_OBJECT val_ptr1, val_ptr2, val_ptr3;
   SEGMENT seg_ptr, orig_ptr;
   int start, end, i, j, n;

   if (ArgCountCheck("mv-replace",EXACTLY,3) == -1)
     {
      SetMultifieldErrorValue(seg_value);
      return;
     }

   if ((ArgTypeCheck("mv-replace",1,INTEGER,&val_ptr1) == CLIPS_FALSE) ||
       (ArgTypeCheck("mv-replace",2,MULTIFIELD,&val_ptr2) == CLIPS_FALSE))
     {
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      SetMultifieldErrorValue(seg_value);
      return;
     }

   RtnUnknown(3,&val_ptr3);
   if ((GetType(val_ptr3) != INTEGER) &&
       (GetType(val_ptr3) != FLOAT) &&
       (GetType(val_ptr3) != SYMBOL) &&
       (GetType(val_ptr3) != EXTERNAL_ADDRESS) &&
#if OBJECT_SYSTEM
       (GetType(val_ptr3) != INSTANCE_NAME) &&
       (GetType(val_ptr3) != INSTANCE) &&
#endif
       (GetType(val_ptr3) != STRING))
     {
      ExpectedTypeError("mv-replace",3,"primitive data type");
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      SetMultifieldErrorValue(seg_value);
      return;
     }
   /*=====================================================*/
   /* Get the field number to be deleted and check to see */
   /* that it exists within the segment boundaries.       */
   /*=====================================================*/

   n = (int) DOToLong(val_ptr1);
   if ((n > GetDOLength(val_ptr2)) || (n < 1))
     {
      SetpType(seg_value,MULTIFIELD);
      SetpDOBegin(seg_value,GetDOBegin(val_ptr2));
      SetpDOEnd(seg_value,GetDOEnd(val_ptr2));
      SetpValue(seg_value,ClipsGetValue(val_ptr2));
      return;
     }
 
   /*=================================================*/
   /* Get a new segment with length one less than the */
   /* original.                                       */ 
   /*=================================================*/

   start = GetDOBegin(val_ptr2);
   end = GetDOEnd(val_ptr2);
   seg_ptr = (struct fact *) CreateMultifield((end - start) + 1);
   orig_ptr = (SEGMENT) GetpValue(&val_ptr2);

   /*=======================================*/
   /* Copy all values from the old segment. */
   /*=======================================*/

   for (i = start, j = 1; i <= end; i++, j++)
     {
      SetMFType(seg_ptr,j,(GetMFType(orig_ptr,i)));
      SetMFValue(seg_ptr,j,(GetMFValue(orig_ptr,i)));
     }
     
   /*=========================================*/
   /* Replace the value from the old segment. */
   /*=========================================*/

   SetMFType(seg_ptr,n,(GetType(val_ptr3)));
   SetMFValue(seg_ptr,n,(ClipsGetValue(val_ptr3)));

   /*=========================*/
   /* Return the new segment. */
   /*=========================*/

   SetpType(seg_value,MULTIFIELD);
   SetpDOBegin(seg_value,1);
   SetpDOEnd(seg_value,(end - start) + 1);
   SetpValue(seg_value,(VOID *) seg_ptr);
   return;
  }
  
/*********************************************/
/* MvAppendFunction:  Append function for segments. */
/*********************************************/
globle VOID MvAppendFunction(seg_value)
  DATA_OBJECT_PTR seg_value;
  {
   DATA_OBJECT val_ptr;
   DATA_OBJECT_PTR val_arr;
   SEGMENT seg_ptr, orig_ptr;
   int start, end, i, j, k, seg_size, num_a;

   num_a = RtnArgCount(); 

   /*=========================================*/
   /* If no arguments are given return a NULL */
   /* multifield variable.                    */ 
   /*=========================================*/

   if (num_a == 0)
     {
       SetpType(seg_value,MULTIFIELD);
       SetpDOBegin(seg_value,1);
       SetpDOEnd(seg_value,0);
       seg_ptr = (struct fact *) CreateMultifield(0);
       SetpValue(seg_value,(VOID *) seg_ptr);
       return;
     }

   else
     {

      /*========================================*/
      /* Get a new segment with length equal to */
      /* the total length of all the arguments. */
      /*========================================*/

      val_arr = (DATA_OBJECT_PTR) gm1((int) sizeof(DATA_OBJECT) * num_a);
      seg_size = 0;
      for(i = 1; i <= num_a; i++) 
        {
         RtnUnknown(i,&val_ptr);
         SetpType(val_arr+i-1,GetType(val_ptr));
         if(GetType(val_ptr) == MULTIFIELD)
           {
            SetpValue(val_arr+i-1,GetpValue(&val_ptr));
            start = GetDOBegin(val_ptr);
            end = GetDOEnd(val_ptr);
           }
         else
           {
            SetpValue(val_arr+i-1,ClipsGetValue(val_ptr));
            start = end = -1;
           }
           
         seg_size += end - start + 1;
         SetpDOBegin(val_arr+i-1,start);
         SetpDOEnd(val_arr+i-1,end);
        }
      seg_ptr = (struct fact *) CreateMultifield(seg_size);
   
      /*========================================*/
      /* Copy each argument into new segment.  */
      /*========================================*/
   
      for(k=0,j=1; k < num_a;k++) 
        {
         if (GetpType(val_arr+k) == MULTIFIELD)
           {
            start = GetpDOBegin(val_arr+k);
            end = GetpDOEnd(val_arr+k);
            orig_ptr = (SEGMENT) GetpValue(val_arr+k);
            for(i=start; i< end + 1; i++,j++)
              {
               SetMFType(seg_ptr,j,(GetMFType(orig_ptr,i)));
               SetMFValue(seg_ptr,j,(GetMFValue(orig_ptr,i)));
              }
           }
         else
           {
            SetMFType(seg_ptr,j,(GetpType(val_arr+k)));
            SetMFValue(seg_ptr,j,(GetpValue(val_arr+k)));
            j++; 
           }
        }
         
      /*=========================*/
      /* Return the new segment. */
      /*=========================*/
   
      SetpType(seg_value,MULTIFIELD);
      SetpDOBegin(seg_value,1);
      SetpDOEnd(seg_value,seg_size);
      SetpValue(seg_value,(VOID *) seg_ptr);
      rm(val_arr,(int) sizeof(DATA_OBJECT) * num_a);
      return;
     }
  }

/*********************************************/
/* StrExplodeFunction:  Explodes a string to a      */
/*   segment variable and returns the new variable.      */
/*********************************************/
globle VOID StrExplodeFunction(str_value)
  DATA_OBJECT_PTR str_value;
  {
   DATA_OBJECT val_ptr;
   SEGMENT seg_ptr, StringToFact();
   int end;

  if (ArgCountCheck("str-explode",EXACTLY,1) == -1)
    {
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      SetMultifieldErrorValue(str_value);
      return;
     }

  if (ArgTypeCheck("str-explode",1,STRING,&val_ptr) == CLIPS_FALSE)
    {
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      SetMultifieldErrorValue(str_value);
      return;
     }

  seg_ptr = StringToMultifield(DOToString(val_ptr));
  if (seg_ptr == NULL)
    {
     seg_ptr = (SEGMENT) CreateMultifield(0);
     end = 0;
    }
  else
    { end = GetMFLength(seg_ptr); }

  /*=========================*/
  /* Return the new segment. */
  /*=========================*/

  SetpType(str_value,MULTIFIELD);
  SetpDOBegin(str_value,1);
  SetpDOEnd(str_value,end);
  SetpValue(str_value,(VOID *) seg_ptr);
  return;
  } 

/****************************************************/
/* StrImplodeFunction:  Implodes a segment variable */
/*   to a string and returns the string.            */
/****************************************************/
globle VOID *StrImplodeFunction()
  {
   DATA_OBJECT val_ptr;
   int strsize = 0;
   int i, j;
   char *tmp_str, *ret_str;
   SEGMENT segPtr;
   VOID *rv;
   
   if (ArgCountCheck("str-implode",EXACTLY,1) == -1)
     { return(AddSymbol("")); }   

   if (ArgTypeCheck("str-implode",1,MULTIFIELD,&val_ptr) == CLIPS_FALSE)
     { return(AddSymbol("")); }   

   /*===================================================*/
   /* Determine the size of the string to be allocated. */
   /*===================================================*/

   segPtr = (SEGMENT) ClipsGetValue(val_ptr);
   for (i = GetDOBegin(val_ptr) ; i <= GetDOEnd(val_ptr) ; i++)
     {
      if (GetMFType(segPtr,i) == FLOAT)
        {
         tmp_str = FloatToString(ValueToDouble(GetMFValue(segPtr,i)));
         strsize += strlen(tmp_str) + 1;
        }
      else if (GetMFType(segPtr,i) == INTEGER)
        {
         tmp_str = LongIntegerToString(ValueToLong(GetMFValue(segPtr,i)));
         strsize += strlen(tmp_str) + 1;
        }
      else if (GetMFType(segPtr,i) == STRING)
        {  
         strsize += strlen(ValueToString(GetMFValue(segPtr,i))) + 3;
         tmp_str = ValueToString(GetMFValue(segPtr,i));
         while(*tmp_str)
           {
            if(*tmp_str == '"') 
              { strsize++; }
            tmp_str++;
           }
        }
#if OBJECT_SYSTEM
      else if (GetMFType(segPtr,i) == INSTANCE_NAME)
        { strsize += strlen(ValueToString(GetMFValue(segPtr,i))) + 3; }
      else if (GetMFType(segPtr,i) == INSTANCE)
        { strsize += strlen(ValueToString(((INSTANCE_TYPE *) 
                            GetMFValue(segPtr,i))->name)) + 3; }
#endif       
      else
        { strsize += strlen(ValueToString(GetMFValue(segPtr,i))) + 1; }
     }

   /*=============================================*/
   /* Allocate the string and copy all components */
   /* of the MULTIFIELD variable to it.             */
   /*=============================================*/

   if (strsize == 0) return(AddSymbol(""));
   ret_str = (char *) gm2(strsize);
   for(j=0, i=GetDOBegin(val_ptr); i <= GetDOEnd(val_ptr) ; i++)
     {

      /*============================*/
      /* Convert numbers to strings */
      /*============================*/

      if (GetMFType(segPtr,i) == FLOAT)
        {
         tmp_str = FloatToString(ValueToDouble(GetMFValue(segPtr,i)));
         while(*tmp_str)
           {
            *(ret_str+j) = *tmp_str;
            j++, tmp_str++;
           }
        }
      else if (GetMFType(segPtr,i) == INTEGER)
        {
         tmp_str = LongIntegerToString(ValueToLong(GetMFValue(segPtr,i)));
         while(*tmp_str)
           {
            *(ret_str+j) = *tmp_str;
            j++, tmp_str++;
           }
        }

      /*=======================================*/
      /* Enclose strings in quotes and preceed */
      /* imbedded quotes with a backslash      */
      /*=======================================*/

      else if (GetMFType(segPtr,i) == STRING)
        {
         tmp_str = ValueToString(GetMFValue(segPtr,i));
         *(ret_str+j) = '"';
         j++;
         while(*tmp_str)
           {
            if(*tmp_str == '"')
              {
               *(ret_str+j) = '\\';
               j++;
              } 
            *(ret_str+j) = *tmp_str;
            j++, tmp_str++;
           }
         *(ret_str+j) = '"';
         j++;
        }
#if OBJECT_SYSTEM
      else if (GetMFType(segPtr,i) == INSTANCE_NAME)
        {
         tmp_str = ValueToString(GetMFValue(segPtr,i));
         *(ret_str + j++) = '[';
         while(*tmp_str)
           {
            *(ret_str+j) = *tmp_str;
            j++, tmp_str++;
           }
         *(ret_str + j++) = ']';
        }
      else if (GetMFType(segPtr,i) == INSTANCE)
        {
         tmp_str = ValueToString(((INSTANCE_TYPE *) GetMFValue(segPtr,i))->name);
         *(ret_str + j++) = '[';
         while(*tmp_str)
           {
            *(ret_str+j) = *tmp_str;
            j++, tmp_str++;
           }
         *(ret_str + j++) = ']';
        }
#endif       
      else
        {
         tmp_str = ValueToString(GetMFValue(segPtr,i));
         while(*tmp_str)
           {
            *(ret_str+j) = *tmp_str;
            j++, tmp_str++;
           }
         }
      *(ret_str+j) = ' ';
      j++;
     }
   *(ret_str+j-1) = '\0';

   /*====================*/
   /* Return the string. */
   /*====================*/
   
   rv = AddSymbol(ret_str);
   rm(ret_str,strsize);
   return(rv);
  }
  
/****************************************************/
/* MvSubseqFunction: Subsequence function for segments. */
/****************************************************/
globle VOID MvSubseqFunction(sub_value)
  DATA_OBJECT_PTR sub_value;
  {
   DATA_OBJECT val_ptr;
   int start, end, length;
   
   /*========================================*/
   /* Check for correct number of arguments. */
   /*========================================*/
   
   if (ArgCountCheck("mv-subseq",EXACTLY,3) == -1)
     { 
      SetMultifieldErrorValue(sub_value);
      return; 
     }
     
   /*=============================================*/
   /* Get range arguments. If they are not within */
   /* appropriate ranges, return a null segment.  */  
   /*=============================================*/
   
   if (ArgTypeCheck("mv-subseq",1,INTEGER,&val_ptr) == CLIPS_FALSE)
     { 
      SetMultifieldErrorValue(sub_value);
      return; 
     }
   start = (int) DOToLong(val_ptr);
   
   if (ArgTypeCheck("mv-subseq",2,INTEGER,&val_ptr) == CLIPS_FALSE)
     { 
      SetMultifieldErrorValue(sub_value);
      return; 
     }
   end = (int) (int) DOToLong(val_ptr);
   
   if ((end < 1) || (end < start))
     { 
      SetMultifieldErrorValue(sub_value);
      return; 
     }

   /*==================================================*/
   /* Get the segment to be subdivided. Adjust lengths */  
   /* to conform to segment boundaries.                */
   /*==================================================*/
   
   if (ArgTypeCheck("mv-subseq",3,MULTIFIELD,&val_ptr) == CLIPS_FALSE)
     { 
      SetMultifieldErrorValue(sub_value);
      return; 
     }
     
   length = GetDOLength(val_ptr);
   if (start > length) 
     {
      SetMultifieldErrorValue(sub_value); 
      return;
     }
   if (end > length) end = length;
   if (start < 1) start = 1;
   
   /*=========================*/
   /* Return the new segment. */
   /*=========================*/
   
   SetpType(sub_value,MULTIFIELD);
   SetpValue(sub_value,ClipsGetValue(val_ptr));
   SetpDOEnd(sub_value,GetDOBegin(val_ptr) + end - 1);
   SetpDOBegin(sub_value,GetDOBegin(val_ptr) + start - 1);
  }
  
/****************************************/
/* NthFunction:                              */
/****************************************/
globle VOID NthFunction(nth_value)
  DATA_OBJECT_PTR nth_value;
  {
   DATA_OBJECT val_ptr1, val_ptr2;
   SEGMENT elm_ptr;
   int n;

   if (ArgCountCheck("nth",EXACTLY,2) == -1)
     {
      SetpType(nth_value,SYMBOL);
      SetpValue(nth_value,(VOID *) AddSymbol("nil"));
      return;
     }

   if ((ArgTypeCheck("nth",1,INTEGER,&val_ptr1) == CLIPS_FALSE) ||
       (ArgTypeCheck("nth",2,MULTIFIELD,&val_ptr2) == CLIPS_FALSE))
     {
      SetpType(nth_value,SYMBOL);
      SetpValue(nth_value,(VOID *) AddSymbol("nil"));
      return;
     }

   n = (int) DOToLong(val_ptr1);
   if ((n > GetDOLength(val_ptr2)) || (n < 1))
     {
      SetpType(nth_value,SYMBOL);
      SetpValue(nth_value,(VOID *) AddSymbol("nil"));
      return;
     }

   elm_ptr = (SEGMENT) ClipsGetValue(val_ptr2);
   SetpType(nth_value,GetMFType(elm_ptr,n + GetDOBegin(val_ptr2) - 1));
   SetpValue(nth_value,GetMFValue(elm_ptr,n + GetDOBegin(val_ptr2) - 1));
  }

/* ------------------------------------------------------------------
 *    SubsetFunction: 
 *               This function compares two multi-field variables
 *               to see if the first is a subset of the second. It
 *               does not consider order. 
 *
 *    INPUTS:    Two arguments via CLIPS stack. First is the sublist
 *               multi-field variable, the second is the list to be
 *               compared to. Both should be of type MULTIFIELD.
 *
 *    OUTPUTS:   One floating point number, 1.0 if the first list
 *               is a subset of the second, else 0.0 if it is not.
 *
 *    NOTES:     This function is called from CLIPS with the subset
 *               command. Repeated values in the sublist must also
 *               be repeated in the main list.
 * ------------------------------------------------------------------
 */

globle BOOLEAN SubsetFunction()
  {
   DATA_OBJECT item1, item2;
   int i;
   
   if (ArgCountCheck("subset",EXACTLY,2) == -1)
     return(CLIPS_FALSE);
     
   if (ArgTypeCheck("subset",1,MULTIFIELD,&item1) == CLIPS_FALSE)
     return(CLIPS_FALSE);
     
   if (ArgTypeCheck("subset",2,MULTIFIELD,&item2) == CLIPS_FALSE)
     return(CLIPS_FALSE);
   
   for (i = GetDOBegin(item1) ; i <= GetDOEnd(item1) ; i++)
     {
      if (FindItemInSegment(GetMFType((SEGMENT) ClipsGetValue(item1),i),
                            GetMFValue((SEGMENT) ClipsGetValue(item1),i),&item2) == 0)
        { return(CLIPS_FALSE); }
     }
   
   return(CLIPS_TRUE);
  }

/***************************************/
/* MemberFunction:                          */
/***************************************/
globle VOID MemberFunction(result)
  DATA_OBJECT_PTR result;
  {
   DATA_OBJECT item1, item2;
   int pos;

   result->type = SYMBOL;
   result->value = (VOID *) CLIPSFalseSymbol;
      
   if (ArgCountCheck("member",EXACTLY,2) == -1) return;
   
   RtnUnknown(1,&item1);
   if ((GetType(item1) != SYMBOL) && 
       (GetType(item1) != STRING) &&
       (GetType(item1) != INTEGER) &&
       (GetType(item1) != EXTERNAL_ADDRESS) &&
#if OBJECT_SYSTEM
       (GetType(item1) != INSTANCE_NAME) &&
       (GetType(item1) != INSTANCE) &&
#endif
       (GetType(item1) != FLOAT))
     {
      ExpectedTypeError("member",1,"primitive data type");
      SetEvaluationError(CLIPS_TRUE);
      return;
     }

   if (ArgTypeCheck("member",2,MULTIFIELD,&item2) == CLIPS_FALSE) return;
     
   pos = FindItemInSegment(item1.type,item1.value,&item2);
   
   if (pos != 0L)
     {
      result->type = INTEGER;
      result->value = (VOID *) AddLong((long) pos);
     }
  }
  
/***************************************/
/* FindItemInSegment:                  */
/***************************************/
static int FindItemInSegment(searchType,searchValue,val_ptr)
  int searchType;
  VOID *searchValue;
  DATA_OBJECT_PTR val_ptr;
  {
   int mul_length, i;
   
   mul_length = GetpDOLength(val_ptr);
   for (i = 0 ; i < mul_length ; i++)
     { 
      if ((searchValue == GetMFValue((SEGMENT) GetpValue(val_ptr),i + GetpDOBegin(val_ptr))) &&
          (searchType == GetMFType((SEGMENT) GetpValue(val_ptr),i + GetpDOBegin(val_ptr))))
        return(i+1); 
     }
        
   return(CLIPS_FALSE);
  }

#endif

#if OBJECT_SYSTEM || MULTIFIELD_FUNCTIONS

/**************************************************************************
  NAME         : ReplaceMultiValueField
  DESCRIPTION  : Performs a replace on the src multi-field value
                   storing the results in the dst multi-field value
  INPUTS       : 1) The destination value buffer
                 2) The source value (can be NULL)
                 3) Beginning of index range
                 4) End of range
                 5) The new field value
  RETURNS      : CLIPS_TRUE if successful, CLIPS_FALSE otherwise
  SIDE EFFECTS : Allocates and sets a ephemeral segment (even if new
                   number of fields is 0)
                 Src value segment is not changed
  NOTES        : index is NOT guaranteed to be valid
                 src is guaranteed to be a multi-field variable or NULL
 **************************************************************************/
globle int ReplaceMultiValueField(dst,src,rb,re,field,funcName)
  DATA_OBJECT *dst,*src,*field;
  int rb,re;
  char *funcName;
  {
   register int i,j,k;
   register ELEMENT_PTR deptr,septr;
   int srclen,dstlen;
   
   srclen = (src != NULL) ? (src->end - src->begin + 1) : 0;
   if ((re < rb) ||
       (rb < 1) || (re < 1) ||
       (rb > srclen) || (re > srclen))
     {
      MVRangeError(rb,re,srclen,funcName);
      return(CLIPS_FALSE);
     }
   rb = src->begin + rb - 1;
   re = src->begin + re - 1;
   if (field->type == MULTIFIELD)
     dstlen = srclen + GetpDOLength(field) - (re-rb+1);
   else
     dstlen = srclen + 1 - (re-rb+1);
   dst->type = MULTIFIELD;
   dst->begin = 0;
   dst->value = CreateMultifield(dstlen);
   dst->end = dstlen-1;
   for (i = 0 , j = src->begin ; j < rb ; i++ , j++)
     {
      deptr = &((struct fact *) dst->value)->atoms[i];
      septr = &((struct fact *) src->value)->atoms[j];
      deptr->type = septr->type;
      deptr->value = septr->value;
     }
   if (field->type != MULTIFIELD)
     {
      deptr = &((struct fact *) dst->value)->atoms[rb-1];
      deptr->type = field->type;
      deptr->value = field->value;
      i++;
     }
   else
     {
      for (k = field->begin ; k <= field->end ; k++ , i++)
        {
         deptr = &((struct fact *) dst->value)->atoms[i];
         septr = &((struct fact *) field->value)->atoms[k];
         deptr->type = septr->type;
         deptr->value = septr->value;
        }
     }
   while (j < re)
     j++;
   for (j++ ; i < dstlen ; i++ , j++)
     {
      deptr = &((struct fact *) dst->value)->atoms[i];
      septr = &((struct fact *) src->value)->atoms[j];
      deptr->type = septr->type;
      deptr->value = septr->value;
     }
   return(CLIPS_TRUE);
  }

/**************************************************************************
  NAME         : InsertMultiValueField
  DESCRIPTION  : Performs an insert on the src multi-field value
                   storing the results in the dst multi-field value
  INPUTS       : 1) The destination value buffer
                 2) The source value (can be NULL)
                 3) The index for the change
                 4) The new field value
  RETURNS      : CLIPS_TRUE if successful, CLIPS_FALSE otherwise
  SIDE EFFECTS : Allocates and sets a ephemeral segment (even if new
                   number of fields is 0)
                 Src value segment is not changed
  NOTES        : index is NOT guaranteed to be valid
                 src is guaranteed to be a multi-field variable or NULL
 **************************************************************************/
globle int InsertMultiValueField(dst,src,index,field,funcName)
  DATA_OBJECT *dst,*src,*field;
  int index;
  char *funcName;
  {
   register int i,j,k;
   register ELEMENT_PTR deptr,septr;
   int srclen,dstlen;
   
   srclen = (src != NULL) ? (src->end - src->begin + 1) : 0;
   if ((index < 1) || (index > (srclen+1)))
     {
      MVRangeError(index,index,srclen+1,funcName);
      return(CLIPS_FALSE);
     }
   dst->type = MULTIFIELD;
   dst->begin = 0;
   if (src == NULL)
     {
      if (field->type == MULTIFIELD)
        {
         DuplicateSegment(dst,field);
         AddToSegmentList((SEGMENT) dst->value);
        }
      else
        {
         dst->value = CreateMultifield(0);
         dst->end = 0;
         deptr = &((struct fact *) dst->value)->atoms[0];
         deptr->type = field->type;
         deptr->value = field->value;
        }
      return(CLIPS_TRUE);
     }
   dstlen = (field->type == MULTIFIELD) ? GetpDOLength(field) + srclen : srclen + 1;
   dst->value = CreateMultifield(dstlen);
   dst->end = dstlen-1;
   index--;
   for (i = 0 , j = src->begin ; j < index ; i++ , j++)
     {
      deptr = &((struct fact *) dst->value)->atoms[i];
      septr = &((struct fact *) src->value)->atoms[j];
      deptr->type = septr->type;
      deptr->value = septr->value;
     }
   if (field->type != MULTIFIELD)
     {
      deptr = &((struct fact *) dst->value)->atoms[index];
      deptr->type = field->type;
      deptr->value = field->value;
      i++;
     }
   else
     {
      for (k = field->begin ; k <= field->end ; k++ , i++)
        {
         deptr = &((struct fact *) dst->value)->atoms[i];
         septr = &((struct fact *) field->value)->atoms[k];
         deptr->type = septr->type;
         deptr->value = septr->value;
        }
     }
   for ( ; j <= src->end ; i++ , j++)
     {
      deptr = &((struct fact *) dst->value)->atoms[i];
      septr = &((struct fact *) src->value)->atoms[j];
      deptr->type = septr->type;
      deptr->value = septr->value;
     }
   return(CLIPS_TRUE);
  }
  
/*******************************************************
  NAME         : MVRangeError
  DESCRIPTION  : Prints out an error messages for index
                   out-of-range errors in multi-field
                   access functions
  INPUTS       : 1) The bad range start
                 2) The bad range end
                 3) The max end of the range (min is
                     assumed to be 1)
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : None
 ******************************************************/
static VOID MVRangeError(brb,bre,max,funcName)
  int brb,bre,max;
  char *funcName;
  {
   PrintCLIPS(WERROR,"Multifield index ");
   if (brb == bre)
     PrintLongInteger(WERROR,(long) brb);
   else
     {
      PrintCLIPS(WERROR,"range ");
      PrintLongInteger(WERROR,(long) brb);
      PrintCLIPS(WERROR,"..");
      PrintLongInteger(WERROR,(long) bre);
     }
   PrintCLIPS(WERROR," out of range 1..");
   PrintLongInteger(WERROR,(long) max);
   if (funcName != NULL)
     {
      PrintCLIPS(WERROR," in function ");
      PrintCLIPS(WERROR,funcName);
     }
   PrintCLIPS(WERROR,".\n");
  }
  
/**************************************************************************
  NAME         : DeleteMultiValueField
  DESCRIPTION  : Performs a modify on the src multi-field value
                   storing the results in the dst multi-field value
  INPUTS       : 1) The destination value buffer
                 2) The source value (can be NULL)
                 3) The beginning index for deletion
                 4) The ending index for deletion
  RETURNS      : CLIPS_TRUE if successful, CLIPS_FALSE otherwise
  SIDE EFFECTS : Allocates and sets a ephemeral segment (even if new
                   number of fields is 0)
                 Src value segment is not changed
  NOTES        : index is NOT guaranteed to be valid
                 src is guaranteed to be a multi-field variable or NULL
 **************************************************************************/
globle int DeleteMultiValueField(dst,src,rb,re,funcName)
  DATA_OBJECT *dst,*src;
  int rb,re;
  char *funcName;
  {
   register int i,j;
   register ELEMENT_PTR deptr,septr;
   int srclen,dstlen;
   
   srclen = (src != NULL) ? (src->end - src->begin + 1) : 0;
   if ((re < rb) ||
       (rb < 1) || (re < 1) ||
       (rb > srclen) || (re > srclen))
     {
      MVRangeError(rb,re,srclen,funcName);
      return(CLIPS_FALSE);
     }
   dst->type = MULTIFIELD;
   dst->begin = 0;
   if (srclen == 0)
    {
     dst->value = CreateMultifield(0);
     dst->end = -1;
     return(CLIPS_TRUE);
    }
   rb = src->begin + rb -1;
   re = src->begin + re -1;
   dstlen = srclen-(re-rb+1);
   dst->end = dstlen-1;
   dst->value = CreateMultifield(dstlen);
   for (i = 0 , j = src->begin ; j < rb ; i++ , j++)
     {
      deptr = &((struct fact *) dst->value)->atoms[i];
      septr = &((struct fact *) src->value)->atoms[j];
      deptr->type = septr->type;
      deptr->value = septr->value;
     }
   while (j < re)
     j++;
   for (j++ ; i <= dst->end ; j++ , i++)
     {
      deptr = &((struct fact *) dst->value)->atoms[i];
      septr = &((struct fact *) src->value)->atoms[j];
      deptr->type = septr->type;
      deptr->value = septr->value;
     }
   return(CLIPS_TRUE);
  }
  
#endif


