   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*               CLIPS Version 5.10  07/17/91          */
   /*                                                     */
   /*                                                     */
   /*******************************************************/

/*************************************************************/
/* Purpose: CLIPS Generic Functions Interface Routines       */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Brian L. Donnell                                     */
/*                                                           */
/* Contributing Programmer(s):                               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/
   
/* =========================================
   *****************************************
               EXTERNAL DEFINITIONS
   =========================================
   ***************************************** */
#include "setup.h"

#if DEFGENERIC_CONSTRUCT

#if OBJECT_SYSTEM
#include "classcom.h"
#include "classfun.h"
#include "inscom.h"
#else
#if ANSI_COMPILER
#include <string.h>
#endif

#define OBJECT_TYPE_NAME "OBJECT"

#endif

#include "clipsmem.h"
#include "constant.h"
#include "constrct.h"
#if DEFGLOBAL_CONSTRUCT
#include "defglobl.h"
#endif
#include "facts.h"
#include "generate.h"

#if BLOAD || BLOAD_ONLY || BLOAD_AND_BSAVE
#include "bload.h"
#include "genrcbin.h"
#endif

#if CONSTRUCT_COMPILER && (! RUN_TIME)
#include "genrccmp.h"
#endif

#include "genrcfun.h"
#include "router.h"
#include "scanner.h"
#include "parsutil.h"
#include "spclform.h"
#include "utility.h"

#if DEFFUNCTION_CONSTRUCT
#include "deffnctn.h"
#endif

#define _GENRCCOM_SOURCE_
#include "genrccom.h"

/* =========================================
   *****************************************
                   CONSTANTS
   =========================================
   ***************************************** */
   
/* =========================================
   *****************************************
               MACROS AND TYPES
   =========================================
   ***************************************** */
   
/* =========================================
   *****************************************
      INTERNALLY VISIBLE FUNCTION HEADERS
   =========================================
   ***************************************** */
#if ANSI_COMPILER

#if (! BLOAD_ONLY) && (! RUN_TIME)

static int ParseDefgeneric(char *);
static int ParseDefmethod(char *);
static SYMBOL_HN *ParseMethodName(char *);
static SYMBOL_HN *ParseMethodNameAndIndex(char *,unsigned *);
static int ParseParameters(char *,EXPRESSION **,SYMBOL_HN **);
static RESTRICTION *ParseRestriction(char *);
static int DuplicateParameters(EXPRESSION *,EXPRESSION **,SYMBOL_HN *);
static VOID DeleteTempRestricts(EXPRESSION *);
static EXPRESSION *AddParameter(EXPRESSION *,EXPRESSION *,SYMBOL_HN *,RESTRICTION *);
static EXPRESSION *ValidType(SYMBOL_HN *);
static VOID PackRestrictionTypes(RESTRICTION *,EXPRESSION *);
static int ReplaceParameters(EXPRESSION *,EXPRESSION *,SYMBOL_HN *);
static int FindParameter(SYMBOL_HN *,EXPRESSION *,SYMBOL_HN *);
static VOID SaveDefgenerics(char *);
static VOID SaveDefmethods(char *);
static int RemoveGeneric(GENERIC_FUNC *);
static VOID RemoveGenericMethod(GENERIC_FUNC *,int);

#endif

static VOID GrabGenericWildargs(DATA_OBJECT *,int);
static GENERIC_FUNC *CheckGenericExists(char *,char *);
static int CheckMethodExists(char *,GENERIC_FUNC *,int);

#else

#if (! BLOAD_ONLY) && (! RUN_TIME)

static int ParseDefgeneric();
static int ParseDefmethod();
static SYMBOL_HN *ParseMethodName();
static SYMBOL_HN *ParseMethodNameAndIndex();
static int ParseParameters();
static RESTRICTION *ParseRestriction();
static int DuplicateParameters();
static VOID DeleteTempRestricts();
static EXPRESSION *AddParameter();
static EXPRESSION *ValidType();
static VOID PackRestrictionTypes();
static int ReplaceParameters();
static int FindParameter();
static VOID SaveDefgenerics();
static VOID SaveDefmethods();
static int RemoveGeneric();
static VOID RemoveGenericMethod();

#endif

static VOID GrabGenericWildargs();
static GENERIC_FUNC *CheckGenericExists();
static int CheckMethodExists();

#endif
      
/* =========================================
   *****************************************
      INTERNALLY VISIBLE GLOBAL VARIABLES
   =========================================
   ***************************************** */
#if (! BLOAD_ONLY) && (! RUN_TIME)
static struct token GenericInputToken;
#endif

/* =========================================
   *****************************************
          EXTERNALLY VISIBLE FUNCTIONS
   =========================================
   ***************************************** */
   
/***********************************************************
  NAME         : SetupGenericFunctions
  DESCRIPTION  : Initializes all generic function
                   data structures, constructs and functions
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : Generic function CLIPS functions set up
  NOTES        : None
 ***********************************************************/
globle VOID SetupGenericFunctions()
  {
#if ! RUN_TIME

#if BLOAD || BLOAD_ONLY || BLOAD_AND_BSAVE
   SetupGenericsBload();
#endif

#if CONSTRUCT_COMPILER
   SetupGenericsCompiler();
#endif

#if ! BLOAD_ONLY
   AddConstruct("defgeneric",PTIF ParseDefgeneric);
   AddConstruct("defmethod",PTIF ParseDefmethod);
   
  /* ================================================================
     Make sure defmethods are cleared last, for other constructs may
       be using them and need to be cleared first
       
     Need to be cleared in two stages so that mutually dependent
       constructs (like classes) can be cleared
     ================================================================ */
   AddSaveFunction("defgeneric",PTIF SaveDefgenerics,1000);
   AddSaveFunction("defmethod",PTIF SaveDefmethods,-1000);
   AddClearFunction("defmethod",(VOID (*)()) ClearDefmethods,-500);
   AddClearFunction("defgeneric",(VOID (*)()) ClearDefgenerics,-2000);
   DefineFunction("undefgeneric",'v',PTIF CmdUndefgeneric,"CmdUndefgeneric");
   DefineFunction("undefmethod",'v',PTIF CmdUndefmethod,"CmdUndefmethod");
#endif

   DefineFunction("(gnrc-runknown)",'u',PTIF RtnGenericUnknown,"RtnGenericUnknown"); 
   DefineFunction("(gnrc-bind)",'u',PTIF GetGenericBind,"GetGenericBind"); 
   DefineFunction("(gnrc-wildargs)",'u',PTIF GetGenericWildargs,"GetGenericWildargs");
   
   DefineFunction("call-next-method",'u',PTIF CallNextMethod,"CallNextMethod");
   DefineFunction("next-methodp",'b',PTIF NextMethodP,"NextMethodP");

#if DEBUGGING_FUNCTIONS
   DefineFunction("ppdefgeneric",'v',PTIF PPDefgeneric,"PPDefgeneric");
   DefineFunction("list-defgenerics",'v',PTIF CmdListDefgenerics,"CmdListDefgenerics");
   DefineFunction("ppdefmethod",'v',PTIF PPDefmethod,"PPDefmethod");
   DefineFunction("list-defmethods",'v',PTIF CmdListDefmethods,"CmdListDefmethods");
   DefineFunction("preview-generic",'v',PTIF PreviewGeneric,"PreviewGeneric");
#endif
   
#if OBJECT_SYSTEM
   DefineFunction("type",'u',PTIF GetInstanceClassCmd,"GetInstanceClassCmd");
#else
   DefineFunction("type",'u',PTIF TypeOf,"TypeOf");
#endif

#endif

#if DEBUGGING_FUNCTIONS
   AddWatchItem("generic-functions",&WatchGenerics,34);
   AddWatchItem("methods",&WatchMethods,33);
#endif
  }
  
/***************************************************
  NAME         : GetDefgenericName
  DESCRIPTION  : Gets the name of a gneric function
  INPUTS       : Pointer to a generic function
  RETURNS      : Name-string of generic function
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
globle char *GetDefgenericName(ptr)
  VOID *ptr;
  {
   return(ValueToString(((GENERIC_FUNC *) ptr)->name));
  }
  
/************************************************
  NAME         : FindDefgeneric
  DESCRIPTION  : Looks up a generic function
  INPUTS       : The name-string of the generic
  RETURNS      : The address of the generic, NULL
                   if not found
  SIDE EFFECTS : None
  NOTES        : None
 ************************************************/
globle VOID *FindDefgeneric(name)
  char *name;
  {
   SYMBOL_HN *gsym;
   
   gsym = FindSymbol(name);
   if (gsym == NULL)
     return(NULL);
   return((VOID *) FindDefgenericBySymbol(gsym));
  }
    
/***********************************************************
  NAME         : GetNextDefgeneric
  DESCRIPTION  : Finds first or next generic function
  INPUTS       : The address of the current generic function
  RETURNS      : The address of the next generic function
                   (NULL if none)
  SIDE EFFECTS : None
  NOTES        : If ptr == NULL, the first generic function
                    is returned.
 ***********************************************************/
globle VOID *GetNextDefgeneric(ptr)
  VOID *ptr;
  {
   if (ptr == NULL)
     return((VOID *) GenericList);
   return((VOID *) ((GENERIC_FUNC *) ptr)->nxt);
  }
  
/***********************************************************
  NAME         : GetNextDefmethod
  DESCRIPTION  : Find the next method for a generic function
  INPUTS       : 1) The generic function address
                 2) The index of the current method
  RETURNS      : The index of the next method
                    (0 if none)
  SIDE EFFECTS : None
  NOTES        : If index == 0, the index of the first
                   method is returned
 ***********************************************************/
globle unsigned GetNextDefmethod(ptr,index)
  VOID *ptr;
  unsigned index;
  {
   GENERIC_FUNC *gfunc;
   int mi;
   
   gfunc = (GENERIC_FUNC *) ptr;
   if (index == 0)
     {
      if (gfunc->methods != NULL)
        return(gfunc->methods[0].index);  
      return(0);
     }
   mi = FindMethodByIndex(gfunc,index);
   if ((mi+1) == gfunc->mcnt)
     return(0);
   return(gfunc->methods[mi+1].index);
  }
  
/***************************************************
  NAME         : IsDefgenericDeletable
  DESCRIPTION  : Determines if a generic function
                   can be deleted
  INPUTS       : Address of the generic function
  RETURNS      : TRUE if deletable, FALSE otherwise
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
globle int IsDefgenericDeletable(ptr)
  VOID *ptr;
  {
#if BLOAD_ONLY || RUN_TIME
   return(FALSE);
#else
#if BLOAD || BLOAD_AND_BSAVE
   if (Bloaded())
     return(FALSE);
#endif
   return ((((GENERIC_FUNC *) ptr)->busy == 0) ? TRUE : FALSE);
#endif
  }

/***************************************************
  NAME         : IsDefmethodDeletable
  DESCRIPTION  : Determines if a generic function
                   method can be deleted
  INPUTS       : 1) Address of the generic function
                 2) Index of the method
  RETURNS      : TRUE if deletable, FALSE otherwise
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
globle int IsDefmethodDeletable(ptr,index)
  VOID *ptr;
  unsigned index;
  {
#if MAC_MPW
#pragma unused(index)
#endif
#if BLOAD_ONLY || RUN_TIME
   return(FALSE);
#else
#if BLOAD || BLOAD_AND_BSAVE
   if (Bloaded())
     return(FALSE);
#endif
   return((MethodsExecuting((GENERIC_FUNC *) ptr) == FALSE) ? TRUE : FALSE);
#endif
  }

/*****************************************************
  NAME         : GetDefmethodDescription
  DESCRIPTION  : Prints a synopsis of method parameter
                   restrictions into caller's buffer
  INPUTS       : 1) Caller's buffer
                 2) Buffer size (not including space
                    for terminating '\0')
                 3) Address of generic function
                 4) Index of method
  RETURNS      : Nothing useful
  SIDE EFFECTS : Caller's buffer written
  NOTES        : Terminating '\n' not written
 *****************************************************/
globle VOID GetDefmethodDescription(buf,buflen,ptr,index)
  char *buf;
  int buflen;
  VOID *ptr;
  unsigned index;
  {
   GENERIC_FUNC *gfunc;
   int mi;
   
   gfunc = (GENERIC_FUNC *) ptr;
   mi = FindMethodByIndex(gfunc,index);
   PrintMethod(buf,buflen,&gfunc->methods[mi]);
  }
  
/**********************************************************
  NAME         : CmdUndefgeneric
  DESCRIPTION  : Deletes all methods for a generic function
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : methods deallocated
  NOTES        : CLIPS Syntax: (undefgeneric <name> | *)
 **********************************************************/
globle VOID CmdUndefgeneric()
  {
   DATA_OBJECT temp;
   GENERIC_FUNC *gfunc;

   if (ArgCountCheck("undefgeneric",EXACTLY,1) == -1)
     return;
   if (ArgTypeCheck("undefgeneric",1,SYMBOL,&temp) == FALSE)
     return;
   gfunc = (GENERIC_FUNC *) FindDefgeneric(DOToString(temp));
   if ((gfunc == NULL) ? (strcmp(DOToString(temp),"*") != 0) : FALSE)
     {
      PrintCLIPS(WERROR,"Unable to find generic function ");
      PrintCLIPS(WERROR,DOToString(temp));
      PrintCLIPS(WERROR," in function undefgeneric.\n");
     }
   else
     DeleteDefgeneric((VOID *) gfunc);
  }

/**************************************************************
  NAME         : CmdUndefmethod
  DESCRIPTION  : Deletes one method for a generic function
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : methods deallocated
  NOTES        : CLIPS Syntax: (undefmethod <name> <index> | *)
 **************************************************************/
globle VOID CmdUndefmethod()
  {
   DATA_OBJECT temp;
   GENERIC_FUNC *gfunc;
   unsigned mi;
   
   if (ArgCountCheck("undefmethod",EXACTLY,2) == -1)
     return;
   if (ArgTypeCheck("undefmethod",1,SYMBOL,&temp) == FALSE)
     return;
   gfunc = FindDefgenericBySymbol((SYMBOL_HN *) temp.value);
   if ((gfunc == NULL) ? (strcmp(DOToString(temp),"*") != 0) : FALSE)
     {
      PrintCLIPS(WERROR,"No such generic function ");
      PrintCLIPS(WERROR,DOToString(temp));
      PrintCLIPS(WERROR," in function undefmethod.\n");
      return;
     }
   RtnUnknown(2,&temp);
   if (temp.type == SYMBOL)
     {
      if (strcmp(DOToString(temp),"*") != 0)
        {
         PrintCLIPS(WERROR,"Expected a valid method index in function undefmethod.\n");
         return;
        }
      mi = 0;
     }
   else if (temp.type == INTEGER)
     {
      mi = (unsigned) DOToInteger(temp);
      if (mi == 0)
        {
         PrintCLIPS(WERROR,"Expected a valid method index in function undefmethod.\n");
         return;
        }
     }
   else
     {
      PrintCLIPS(WERROR,"Expected a valid method index in function undefmethod.\n");
      return;
     }
   DeleteDefmethod((VOID *) gfunc,mi);
  }

/***************************************************
  NAME         : RtnGenericUnknown
  DESCRIPTION  : Internal function for getting the
                   value of an argument passed to
                   a gneric function
  INPUTS       : Caller's result value buffer
  RETURNS      : Nothing useful
  SIDE EFFECTS : Caller's buffer set to specified
                   node of GenericStackFrame
  NOTES        : None
 ***************************************************/
globle VOID RtnGenericUnknown(rvalue)
  DATA_OBJECT *rvalue;
  {
   EXPRESSION *myArg;

   myArg = &GenericStackFrame[DOPToInteger(GetFirstArgument()) - 1];

   if (myArg->type == MULTIFIELD)
     CopyMemory(DATA_OBJECT,1,rvalue,myArg->value);
   else
     {
      rvalue->type = myArg->type;
      rvalue->value = myArg->value;
     }
  }
  
/*****************************************************************
  NAME         : GetGenericBind
  DESCRIPTION  : Internal function for looking up the
                    values of parameters or bound variables
                    within a generic function
  INPUTS       : Caller's result value buffer
  RETURNS      : Nothing useful
  SIDE EFFECTS : Caller's buffer set to parameter value in
                   GenericStackFrame or the value in the bind list
  NOTES        : None
 *****************************************************************/
globle VOID GetGenericBind(bind_result)
  DATA_OBJECT *bind_result;
  {
   DATA_OBJECT *bind_ptr;
   EXPRESSION *myArg;
   SYMBOL_HN *var_name;
   EXPRESSION *test_ptr;
   int index;
   
   /*============================================*/
   /* Get the name of the variable being sought. */
   /*============================================*/

   test_ptr = GetFirstArgument();
   var_name = (SYMBOL_HN *) test_ptr->value;

   /*===============================================*/
   /* Search for the variable in the list of binds. */
   /*===============================================*/

   bind_ptr = BindList;
   while (bind_ptr != NULL)
     {
      if (bind_ptr->name == var_name)
        {
         CopyMemory(DATA_OBJECT,1,bind_result,bind_ptr);
         return; 
        }

      bind_ptr = bind_ptr->next;
     }

   /*=====================================================*/
   /* If the variable was not found in the list of binds, */
   /* determine if the variable is only bound on the rhs  */
   /* of the rule.                                        */
   /*=====================================================*/
  
   test_ptr = test_ptr->next_arg;
   index = ValueToInteger(test_ptr->value);
   if (index == 0)
     {    
      SetEvaluationError(TRUE);
      PrintCLIPS(WERROR,"Variable ");
      PrintCLIPS(WERROR,ValueToString(var_name));
      PrintCLIPS(WERROR," unbound in generic function ");
      PrintCLIPS(WERROR,ValueToString(CurrentGeneric->name)); 
      PrintCLIPS(WERROR," method #");
      PrintLongInteger(WERROR,(long) CurrentMethod->index);
      PrintCLIPS(WERROR,".\n");
      bind_result->type = SYMBOL;
      bind_result->value = (VOID *) CLIPSFalseSymbol;
      return;
     }

   /*=============================================================*/
   /* Variable was bound in the argument list of the deffunction. */
   /*=============================================================*/
   if (index > 0)
     {
      myArg = &GenericStackFrame[ValueToInteger(test_ptr->value) - 1];

      if (myArg->type == MULTIFIELD)
        CopyMemory(DATA_OBJECT,1,bind_result,myArg->value);
      else
        {
         bind_result->type = myArg->type;
         bind_result->value = myArg->value;
        }
     }
   else
     GrabGenericWildargs(bind_result,-index);
  }

/****************************************************************
  NAME         : GetGenericWildargs
  DESCRIPTION  : Internal function for accessing the value
                   of the wildcard parameter in generic functions
  INPUTS       : Caller's result value buffer
  RETURNS      : Nothing useful
  SIDE EFFECTS : Caller's buffer set to multi-field holding the
                    wildcard's values from GenericStackFrame
  NOTES        : None
 ****************************************************************/
globle VOID GetGenericWildargs(result)
  DATA_OBJECT *result;
  {
   GrabGenericWildargs(result,DOPToInteger(GetFirstArgument()));
  }
  
/**************************************************************
  NAME         : DeleteDefgeneric
  DESCRIPTION  : Deletes all methods for a generic function
  INPUTS       : The generic-function address (NULL for all)
  RETURNS      : 1 if generic successfully deleted, 0 otherwise
  SIDE EFFECTS : methods deallocated
  NOTES        : None
 **************************************************************/
globle int DeleteDefgeneric(vptr)
  VOID *vptr;
  {
   GENERIC_FUNC *gfunc;
   
#if RUN_TIME || BLOAD_ONLY
   gfunc = (GENERIC_FUNC *) vptr;
   PrintCLIPS(WERROR,"Unable to delete generic function ");
   PrintCLIPS(WERROR,(gfunc != NULL) ? ValueToString(gfunc->name) : "*");
   PrintCLIPS(WERROR,".\n");
   return(0);
#else
   int success = 1;

   gfunc = (GENERIC_FUNC *) vptr;
   if (gfunc == NULL)
     {
      if (ClearDefmethods() == FALSE)
        success = 0;
      if (ClearDefgenerics() == FALSE)
        success = 0;
      return(success);
     }
   else
     return(RemoveGeneric(gfunc));
#endif
  }
  
/**************************************************************
  NAME         : DeleteDefmethod
  DESCRIPTION  : Deletes one method for a generic function
  INPUTS       : 1) Address of generic function (can be NULL)
                 2) Method index (0 for all)
  RETURNS      : 1 if method deleted successfully, 0 otherwise
  SIDE EFFECTS : methods deallocated
  NOTES        : None
 **************************************************************/
globle int DeleteDefmethod(vptr,mi)
  VOID *vptr;
  unsigned mi;
  {
   GENERIC_FUNC *gfunc;
   
#if RUN_TIME || BLOAD_ONLY
   gfunc = (GENERIC_FUNC *) vptr;
   PrintCLIPS(WERROR,"Unable to delete method ");
   PrintCLIPS(WERROR,(gfunc != NULL) ? ValueToString(gfunc->name) : "*");
   PrintCLIPS(WERROR," #");
   PrintLongInteger(WERROR,(long) mi);
   PrintCLIPS(WERROR,".\n");
   return(0);
#else
   unsigned i;
   int nmi;

   gfunc = (GENERIC_FUNC *) vptr;
#if BLOAD || BLOAD_AND_BSAVE
   if (Bloaded() == CLIPS_TRUE)
     {
      PrintCLIPS(WERROR,"Unable to delete method ");
      PrintCLIPS(WERROR,(gfunc != NULL) ? ValueToString(gfunc->name) : "*");
      PrintCLIPS(WERROR," #");
      PrintLongInteger(WERROR,(long) mi);
      PrintCLIPS(WERROR,".\n");
      return(0);
     }
#endif
   if (gfunc == NULL)
     {
      if (mi != 0)
        {
         PrintCLIPS(WERROR,"Incomplete method specification for deletion.\n");
         return(0);
        }
      return(ClearDefmethods());
     }
   if (MethodsExecuting(gfunc))
     {
      MethodAlterError(ValueToString(gfunc->name));
      return(0);
     }
   if (mi == 0)
     {
      for (i = 0 ; i < gfunc->mcnt ; i++)
        DeleteMethodInfo(gfunc,&gfunc->methods[i]);
      if (gfunc->mcnt != 0)
        rm((VOID *) gfunc->methods,(int) (sizeof(METHOD) * gfunc->mcnt));
      gfunc->mcnt = 0;
      gfunc->methods = NULL;
     }
   else
     {
      nmi = CheckMethodExists("undefmethod",gfunc,mi);
      if (nmi == -1)
        return(0);
      RemoveGenericMethod(gfunc,nmi);
     }
   return(1);
#endif
  }
  
#if DEBUGGING_FUNCTIONS

/********************************************************
  NAME         : PPDefgeneric
  DESCRIPTION  : Displays the pretty-print form of
                  a generic function header
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax: (ppdefgeneric <name>)
 ********************************************************/
globle VOID PPDefgeneric()
  {
   DATA_OBJECT temp;
   GENERIC_FUNC *gfunc;

   if (ArgCountCheck("ppdefgeneric",EXACTLY,1) == -1)
     return;
   if (ArgTypeCheck("ppdefgeneric",1,SYMBOL,&temp) == FALSE)
     return;
   gfunc = CheckGenericExists("ppdefgeneric",DOToString(temp));
   if (gfunc == NULL)
     return;
   if (gfunc->pp_form != NULL)
     PrintCLIPS(WDISPLAY,gfunc->pp_form);
  }
  
/***************************************************
  NAME         : CmdListDefgenerics
  DESCRIPTION  : Lists all generic functions
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax: (list-defgenerics)
 ***************************************************/
globle VOID CmdListDefgenerics()
  {
   if (ArgCountCheck("list-defgenerics",EXACTLY,0) == -1)
     return;
   ListDefgenerics();
  }

/**********************************************************
  NAME         : PPDefmethod
  DESCRIPTION  : Displays the pretty-print form of
                  a method
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax: (ppdefmethod <name> <index>)
 **********************************************************/
globle VOID PPDefmethod()
  {
   DATA_OBJECT temp;
   char *gname;
   GENERIC_FUNC *gfunc;
   int gi;

   if (ArgCountCheck("ppdefmethod",EXACTLY,2) == -1)
     return;
   if (ArgTypeCheck("ppdefmethod",1,SYMBOL,&temp) == FALSE)
     return;
   gname = DOToString(temp);
   if (ArgTypeCheck("ppdefmethod",2,INTEGER,&temp) == FALSE)
     return;
   gfunc = CheckGenericExists("ppmethod",gname);
   if (gfunc == NULL)
     return;
   gi = CheckMethodExists("ppmethod",gfunc,DOToInteger(temp));
   if (gi == -1)
     return;
   if (gfunc->methods[gi].pp_form != NULL)
     PrintCLIPS(WDISPLAY,gfunc->methods[gi].pp_form);
  }
  
/******************************************************
  NAME         : CmdListDefmethods
  DESCRIPTION  : Lists a brief description of methods
                   for a particular generic function
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax: (list-defmethods <name>)
 ******************************************************/
globle VOID CmdListDefmethods()
  {
   int argcnt;
   DATA_OBJECT temp;
   GENERIC_FUNC *gfunc;
   
   argcnt = ArgCountCheck("list-defmethods",NO_MORE_THAN,1);
   if (argcnt == -1)
     return;
   if (argcnt == 1)
     {
      if (ArgTypeCheck("list-defmethods",1,SYMBOL,&temp) == FALSE)
        return;
      gfunc = CheckGenericExists("list-defmethods",DOToString(temp));
      if (gfunc != NULL)
        ListDefmethods((VOID *) gfunc);
     }
   else
     ListDefmethods(NULL);
  }

/*******************************************************
  NAME         : GetDefgenericPPForm
  DESCRIPTION  : Gets generic function pretty print form
  INPUTS       : Address of the generic function
  RETURNS      : Generic function ppform
  SIDE EFFECTS : None
  NOTES        : None
 *******************************************************/
globle char *GetDefgenericPPForm(ptr)
  VOID *ptr;
  {
   return(((GENERIC_FUNC *) ptr)->pp_form);
  }

/***************************************************************
  NAME         : GetDefmethodPPForm
  DESCRIPTION  : Getsa generic function method pretty print form
  INPUTS       : 1) Address of the generic function
                 2) Index of the method
  RETURNS      : Method ppform
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************************/
globle char *GetDefmethodPPForm(ptr,index)
  VOID *ptr;
  unsigned index;
  {
   GENERIC_FUNC *gfunc;
   int mi;
   
   gfunc = (GENERIC_FUNC *) ptr;
   mi = FindMethodByIndex(gfunc,index);
   return(gfunc->methods[mi].pp_form);
  }

/***************************************************
  NAME         : ListDefgenerics
  DESCRIPTION  : Lists all generic functions
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
globle VOID ListDefgenerics()
  {
   GENERIC_FUNC *gfunc;
   long cnt = 0;
   
   for (gfunc = GenericList ; gfunc != NULL ; gfunc = gfunc->nxt)
     {
      PrintCLIPS(WDISPLAY,ValueToString(gfunc->name));
      PrintCLIPS(WDISPLAY,"\n");
      cnt++;
     }
   PrintTally(WDISPLAY,cnt,"generic function","generic functions");
  }

/******************************************************
  NAME         : ListDefmethods
  DESCRIPTION  : Lists a brief description of methods
                   for a particular generic function
  INPUTS       : Generic function to list methods for
                   (NULL means list all methods)
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : None
 ******************************************************/
globle VOID ListDefmethods(vptr)
  VOID *vptr;
  {
   GENERIC_FUNC *gfunc;
   int gi;
   long cnt;
   char buf[61];
   
   if (vptr != NULL)
     {
      gfunc = (GENERIC_FUNC *) vptr;
      for (gi = 0 ; gi < gfunc->mcnt ; gi++)
        {
         PrintCLIPS(WDISPLAY,ValueToString(gfunc->name));
         PrintCLIPS(WDISPLAY," #");
         PrintMethod(buf,60,&gfunc->methods[gi]);
         PrintCLIPS(WDISPLAY,buf);
         PrintCLIPS(WDISPLAY,"\n");
        }
      cnt = gfunc->mcnt;
     }
   else
     {
      cnt = 0L;
      for (gfunc = GenericList ; gfunc != NULL ; gfunc = gfunc->nxt)
        {
         cnt += gfunc->mcnt;
         for (gi = 0 ; gi < gfunc->mcnt ; gi++)
           {
            PrintCLIPS(WDISPLAY,ValueToString(gfunc->name));
            PrintCLIPS(WDISPLAY," #");
            PrintMethod(buf,60,&gfunc->methods[gi]);
            PrintCLIPS(WDISPLAY,buf);
            PrintCLIPS(WDISPLAY,"\n");
           }
         if ((gfunc->nxt != NULL) ? (gfunc->nxt->mcnt != 0) : FALSE)
           PrintCLIPS(WDISPLAY,"\n");
        }
     }
   PrintTally(WDISPLAY,cnt,"method","methods");
  }

#endif

/* =========================================
   *****************************************
          INTERNALLY VISIBLE FUNCTIONS
   =========================================
   ***************************************** */

#if (! BLOAD_ONLY) && (! RUN_TIME)

/***************************************************************************
  NAME         : ParseDefgeneric
  DESCRIPTION  : Parses the defgeneric construct
  INPUTS       : The input logical name
  RETURNS      : FALSE if successful parse, TRUE otherwise
  SIDE EFFECTS : Inserts valid generic function defn into generic entry
  NOTES        : CLIPS Syntax :
                 (defgeneric <name> [<comment>])
 ***************************************************************************/
static int ParseDefgeneric(read_source)
  char *read_source;
  {
   SYMBOL_HN *gname;
   GENERIC_FUNC *gfunc;
   int new;
   
   SetPPBufferStatus(ON);
   FlushPPBuffer();
   SavePPBuffer("(defgeneric ");
   SetIndentDepth(3);    

#if BLOAD || BLOAD_AND_BSAVE
   if (Bloaded() == CLIPS_TRUE) 
     {
      PrintCLIPS(WERROR,"\nCannot load defgenerics with binary load in effect.\n");
      return(CLIPS_TRUE);
     }
#endif

   gname = ParseMethodName(read_source);
   if (gname == NULL)
     return(TRUE);

   if (ValidConstruct(ValueToString(gname)) == CLIPS_TRUE)
     {
      PrintCLIPS(WERROR,"Generic functions are not allowed to replace constructs.\n");
      return(TRUE);
     }

#if DEFFUNCTION_CONSTRUCT
   if (FindDeffunctionBySymbol(gname) != NULL)
     {
      PrintCLIPS(WERROR,"Generic functions are not allowed to replace deffunctions.\n");
      return(TRUE);
     }
#endif
     
   gfunc = AddGeneric(gname,&new);
   if (GetPrintWhileLoading())
     {
      if (GetCompilationsWatch())
        {
         PrintCLIPS(WDIALOG,(new == FALSE) ? "Redefining" : "Defining");
         PrintCLIPS(WDIALOG," defgeneric block ");
         PrintCLIPS(WDIALOG,ValueToString(gname));
         PrintCLIPS(WDIALOG,"\n");
        }
      else
        PrintCLIPS(WDIALOG,"^");
     }
   if (GenericInputToken.type != RPAREN)
     {
      PrintCLIPS(WERROR,"Expected ')' to complete defgeneric.\n");
      if (new == TRUE)
        RemoveGeneric(gfunc);
      return(TRUE);
     }
   PPBackup();
   PPBackup();
   SavePPBuffer(")\n");
   if (gfunc->pp_form != NULL)
     {
      rm((VOID *) gfunc->pp_form,(int) (sizeof(char) * (strlen(gfunc->pp_form)+1)));
      gfunc->pp_form = NULL;
     }
#if DEBUGGING_FUNCTIONS
   if (! GetConserveMemory())
     gfunc->pp_form = CopyPPBuffer();
#endif
   return(FALSE);
  }
  
/***************************************************************************
  NAME         : ParseDefmethod
  DESCRIPTION  : Parses the defmethod construct
  INPUTS       : The input logical name
  RETURNS      : FALSE if successful parse, TRUE otherwise
  SIDE EFFECTS : Inserts valid method definition into generic entry
  NOTES        : CLIPS Syntax :
                 (defmethod <name> [<index>] [<comment>]
                    (<restriction>* [<wildcard>])
                    <action>*)
                 <restriction> :== ?<name> |
                                   (?<name> <type>* [<restriction-query>])
                 <wildcard>    :== $?<name>
 ***************************************************************************/
static int ParseDefmethod(read_source)
  char *read_source;
  {
   SYMBOL_HN *gname;
   int rcnt,mposn,mi,new,mnew = FALSE;
   EXPRESSION *params,*actions,*tmp;
   SYMBOL_HN *wildcard;
   METHOD *meth;
   GENERIC_FUNC *gfunc;
   unsigned index;
   
   SetPPBufferStatus(ON);
   FlushPPBuffer();
   SetIndentDepth(3);    
   SavePPBuffer("(defmethod ");

#if BLOAD || BLOAD_AND_BSAVE
   if (Bloaded() == CLIPS_TRUE) 
     {
      PrintCLIPS(WERROR,"\nCannot load defmethods with binary load in effect.\n");
      return(CLIPS_TRUE);
     }
#endif

   gname = ParseMethodNameAndIndex(read_source,&index);
   if (gname == NULL)
     return(TRUE);

   if (ValidConstruct(ValueToString(gname)) == CLIPS_TRUE)
     {
      PrintCLIPS(WERROR,"Generic functions are not allowed to replace constructs.\n");
      return(TRUE);
     }

#if DEFFUNCTION_CONSTRUCT
   if (FindDeffunctionBySymbol(gname) != NULL)
     {
      PrintCLIPS(WERROR,"Generic functions are not allowed to replace deffunctions.\n");
      return(TRUE);
     }
#endif

   if (GetPrintWhileLoading())
     {
      if (GetCompilationsWatch())
        {
         PrintCLIPS(WDIALOG,"Defining defmethod block ");
         PrintCLIPS(WDIALOG,ValueToString(gname));
         PrintCLIPS(WDIALOG,"...\n");
        }
      else
        PrintCLIPS(WDIALOG,"&");
     }
     
   /* ========================================================
      Go ahead and add the header so that the generic function
      can be called recursively
      ======================================================== */
   gfunc = AddGeneric(gname,&new);
   if ((new != TRUE) ? MethodsExecuting(gfunc) : FALSE)
     {
      MethodAlterError(ValueToString(gname));
      return(TRUE);
     }
   IncrementIndentDepth(1);
   rcnt = ParseParameters(read_source,&params,&wildcard);
   DecrementIndentDepth(1);
   if (rcnt == -1)
     {
      if (new == TRUE)
        RemoveGeneric(gfunc);
      return(TRUE);
     }
   PPCRAndIndent();
   for (tmp = params ; tmp != NULL ; tmp = tmp->next_arg)
     {
      if (! ReplaceParameters(((RESTRICTION *) tmp->arg_list)->query,params,wildcard))
        {
         if (new == TRUE)
           RemoveGeneric(gfunc);
         DeleteTempRestricts(params);
         return(TRUE);
        }
     }
   if (new == TRUE)
     {
      meth = NULL;
      mposn = 0;
     }
   else
     meth = FindMethodByRestrictions(gfunc,params,rcnt,wildcard,&mposn);
   if (meth != NULL)
     {
      if ((index != 0) && (index != meth->index))
        {
         PrintCLIPS(WERROR,"New method #");
         PrintLongInteger(WERROR,(long) index);
         PrintCLIPS(WERROR," would be indistinguishable from method #");
         PrintLongInteger(WERROR,(long) meth->index);
         PrintCLIPS(WERROR,".\n");
         DeleteTempRestricts(params);
         return(TRUE);
        }
     }
   else if (index != 0)
     {
      mi = FindMethodByIndex(gfunc,index);
      if (mi == -1)
        mnew = TRUE;
     }
   else
     mnew = TRUE;
   actions = GroupActions(read_source,&GenericInputToken,TRUE,NULL);
   if (actions == NULL)
     {
      DeleteTempRestricts(params);
      if (new == TRUE)
        RemoveGeneric(gfunc);
      return(TRUE);
     }
   actions = CompactActions(actions);
   PPBackup();
   PPBackup();
   SavePPBuffer(GenericInputToken.print_rep);
   SavePPBuffer("\n");
     
   if (! ReplaceParameters(actions,params,wildcard))
     {
      DeleteTempRestricts(params);
      ReturnExpression(actions);
      if (new == TRUE)
        RemoveGeneric(gfunc);
      return(TRUE);
     }
   
#if DEBUGGING_FUNCTIONS
   meth = AddMethod(gfunc,meth,mposn,index,params,rcnt,wildcard,actions,
             GetConserveMemory() ? NULL : CopyPPBuffer());
#else
   meth = AddMethod(gfunc,meth,mposn,index,params,rcnt,wildcard,actions,NULL);
#endif
   DeleteTempRestricts(params);
   ReturnExpression(actions);
   if (GetPrintWhileLoading() && GetCompilationsWatch())
     {
      PrintCLIPS(WDIALOG,"   Method #");
      PrintLongInteger(WDIALOG,(long) meth->index);
      PrintCLIPS(WDIALOG,mnew ? " defined.\n" : " redefined.\n");
     }
   return(FALSE);
  }

/*******************************************************
  NAME         : ParseMethodName
  DESCRIPTION  : Parses the name of the method
  INPUTS       : The logical name of the input source
  RETURNS      : The symbolic name of the method
  SIDE EFFECTS : None
  NOTES        : Assumes "(defmethod " already parsed
 *******************************************************/
static SYMBOL_HN *ParseMethodName(read_source)
  char *read_source;
  {
   SYMBOL_HN *gname;
   
   GetToken(read_source,&GenericInputToken);
   if (GetType(GenericInputToken) != SYMBOL)
     {
      PrintCLIPS(WERROR,"Missing name for defgeneric construct\n");
      return(NULL);
     }
   gname = (SYMBOL_HN *) ClipsGetValue(GenericInputToken);
   SavePPBuffer(" ");

   GetToken(read_source,&GenericInputToken);
   if (GetType(GenericInputToken) == STRING)
     {
      PPCRAndIndent();
      GetToken(read_source,&GenericInputToken);
     }
   else
     {
      PPBackup();
      PPBackup();
      PPCRAndIndent();
      SavePPBuffer(GenericInputToken.print_rep);
     }
   return(gname);
  }
  
/*******************************************************
  NAME         : ParseMethodNameAndIndex
  DESCRIPTION  : Parses the name of the method and
                   optional method index
  INPUTS       : 1) The logical name of the input source
                 2) Caller's buffer for method index
                    (0 if not specified)
  RETURNS      : The symbolic name of the method
  SIDE EFFECTS : None
  NOTES        : Assumes "(defmethod " already parsed
 *******************************************************/
static SYMBOL_HN *ParseMethodNameAndIndex(read_source,index)
  char *read_source;
  unsigned *index;
  {
   SYMBOL_HN *gname;
   
   *index = 0;
   GetToken(read_source,&GenericInputToken);
   if (GetType(GenericInputToken) != SYMBOL)
     {
      PrintCLIPS(WERROR,"Missing name for defmethod construct\n");
      return(NULL);
     }
   gname = (SYMBOL_HN *) ClipsGetValue(GenericInputToken);
   SavePPBuffer(" ");
   GetToken(read_source,&GenericInputToken);
   if (GetType(GenericInputToken) == INTEGER)
     {
      int tmp;
      
      tmp = (int) ValueToLong(ClipsGetValue(GenericInputToken));
      if (tmp < 1)
        {
         PrintCLIPS(WERROR,"Method index out of range.\n");
         return(NULL);
        }
      *index = (unsigned) tmp;
      SavePPBuffer(" ");
      GetToken(read_source,&GenericInputToken);
     }
   if (GetType(GenericInputToken) == STRING)
     {
      PPCRAndIndent();
      GetToken(read_source,&GenericInputToken);
     }
   else
     {
      PPBackup();
      PPBackup();
      PPCRAndIndent();
      SavePPBuffer(GenericInputToken.print_rep);
     }
   return(gname);
  }
  
/************************************************************************
  NAME         : ParseParameters
  DESCRIPTION  : Parses method restrictions
                   (parameter names with class and expression specifiers)
  INPUTS       : 1) The logical name of the input source
                 2) Caller's buffer for the parameter name list
                    (Restriction structures are attached to
                     arg_list pointers of parameter nodes)
                 3) Caller's buffer for wildcard symbol (if any)
  RETURNS      : The number of parameters, or -1 on errors
  SIDE EFFECTS : Memory allocated for parameters and restrictions
                 Parameter names stored in expression list
                 Parameter restrictions stored in contiguous array
  NOTES        : Any memory allocated is freed on errors
                 Assumes first opening parenthesis has been scanned
 ************************************************************************/
static int ParseParameters(read_source,params,wildcard)
  char *read_source;
  EXPRESSION **params;
  SYMBOL_HN **wildcard;
  {
   EXPRESSION *phead = NULL,*pprv;
   SYMBOL_HN *pname;
   RESTRICTION *rtmp;
   int rcnt = 0;
   
   *wildcard = NULL;
   *params = NULL;
   if (GetType(GenericInputToken) != LPAREN)
     {
      PrintCLIPS(WERROR,"Expected a '(' to begin method parameter restrictions.\n");
      return(-1);
     }
   GetToken(read_source,&GenericInputToken);
   while (GenericInputToken.type != RPAREN)
     {
      if (*wildcard != NULL)
        {
         DeleteTempRestricts(phead);
         PrintCLIPS(WERROR,"No parameters allowed after wildcard parameter.\n");
         return(-1);
        }
      if ((GenericInputToken.type == BWORD) || (GenericInputToken.type == BWORDS))
        {
         pname = (SYMBOL_HN *) GenericInputToken.value;
         if (DuplicateParameters(phead,&pprv,pname))
           {
            DeleteTempRestricts(phead);
            return(-1);
           }
         if (GenericInputToken.type == BWORDS)
           *wildcard = pname;
         else
           {
            rtmp = get_struct(restriction);
            rtmp->tcnt = 0;
            rtmp->types = NULL;
            rtmp->query = NULL;
            phead = AddParameter(phead,pprv,pname,rtmp);
            rcnt++;
           }
        }
      else if (GenericInputToken.type == LPAREN)
        {
         GetToken(read_source,&GenericInputToken);
         if (GenericInputToken.type != BWORD)
           {
            DeleteTempRestricts(phead);
            PrintCLIPS(WERROR,"Expected a variable for parameter specification.\n");
            return(-1);
           }
         pname = (SYMBOL_HN *) GenericInputToken.value;
         if (DuplicateParameters(phead,&pprv,pname))
           {
            DeleteTempRestricts(phead);
            return(-1);
           }
         SavePPBuffer(" ");
         rtmp = ParseRestriction(read_source);
         if (rtmp == NULL)
           {
            DeleteTempRestricts(phead);
            return(-1);
           }
         phead = AddParameter(phead,pprv,pname,rtmp);
         rcnt++;
        }
      else
        {
         DeleteTempRestricts(phead);
         PrintCLIPS(WERROR,"Expected a variable or '(' for parameter specification.\n");
         return(-1);
        }
      PPCRAndIndent();
      GetToken(read_source,&GenericInputToken);
     }
   if ((rcnt != 0) || (*wildcard != NULL))
     {
      PPBackup();
      PPBackup();
      SavePPBuffer(")");
     }
   *params = phead;
   return(rcnt);
  }
  
/************************************************************
  NAME         : ParseRestriction
  DESCRIPTION  : Parses the restriction for a parameter of a 
                   method
                 This restriction is comprised of:
                   1) A list of classes (or types) that are
                      allowed for the parameter (None
                      if no type restriction)
                   2) And an optional restriction-query
                      expression
  INPUTS       : The logical name of the input source
  RETURNS      : The address of a RESTRICTION node, NULL on
                   errors
  SIDE EFFECTS : RESTRICTION node allocated
                   Types are in a contiguous array of VOID *
                   Query is an expression
  NOTES        : Assumes "(?<var> " has already been parsed
                 CLIPS Syntax: <type>* [<query>])
 ************************************************************/
static RESTRICTION *ParseRestriction(read_source)
  char *read_source;
  {
   EXPRESSION *types = NULL,*new_types,
              *typesbot,*tmp,*tmp2,
              *query = NULL;
   RESTRICTION *rptr;
   
   GetToken(read_source,&GenericInputToken);
   while (GenericInputToken.type != RPAREN)
     {
      if (query != NULL)
        {
         PrintCLIPS(WERROR,"Query must be last in parameter restriction.\n");
         ReturnExpression(query);
         ReturnExpression(types);
         return(NULL);
        }
      if (GenericInputToken.type == SYMBOL)
        {
         new_types = ValidType((SYMBOL_HN *) GenericInputToken.value);
         if (new_types == NULL)
           {
            ReturnExpression(types);
            ReturnExpression(query);
            return(NULL);
           }
         if (types == NULL)
           types = new_types;
         else
           {
            for (typesbot = tmp = types ; tmp != NULL ; tmp = tmp->next_arg)
              {
               for (tmp2 = new_types ; tmp2 != NULL ; tmp2 = tmp2->next_arg)
                 {
                  if (tmp->value == tmp2->value)
                    {
#if OBJECT_SYSTEM
                     PrintCLIPS(WERROR,"Duplicate classes not allowed in parameter restriction.\n");
#else
                     PrintCLIPS(WERROR,"Duplicate types not allowed in parameter restriction.\n");
#endif
                     ReturnExpression(query);
                     ReturnExpression(types);
                     ReturnExpression(new_types);
                     return(NULL);
                    }
#if OBJECT_SYSTEM
                  if (HasSuperclass((CLASS_TYPE *) tmp->value,
                                     (CLASS_TYPE *) tmp2->value))
                    {
                     PrintCLIPS(WERROR,ValueToString(((CLASS_TYPE *) tmp->value)->name));
                     PrintCLIPS(WERROR," class is redundant.\n");
                     ReturnExpression(query);
                     ReturnExpression(types);
                     ReturnExpression(new_types);
                     return(NULL);
                    }
                  if (HasSuperclass((CLASS_TYPE *) tmp2->value,
                                     (CLASS_TYPE *) tmp->value))
                    {
                     PrintCLIPS(WERROR,ValueToString(((CLASS_TYPE *) tmp2->value)->name));
                     PrintCLIPS(WERROR," class is redundant.\n");
                     ReturnExpression(query);
                     ReturnExpression(types);
                     ReturnExpression(new_types);
                     return(NULL);
                    }
#endif
                 }
               typesbot = tmp;
              }
            typesbot->next_arg = new_types;
           }
        }
      else if (GenericInputToken.type == LPAREN)
        {
         query = Function1Parse(read_source);
         if (query == NULL)
           {
            ReturnExpression(types);
            return(NULL);
           }
         if (GetParsedBindNames() != NULL)
           {
            PrintCLIPS(WERROR,"Binds are not allowed in query expressions.\n");
            ReturnExpression(query);
            ReturnExpression(types);
            return(NULL);
           }
        }
#if DEFGLOBAL_CONSTRUCT
      else if (GenericInputToken.type == GBWORD)
        query = GenConstant(GBWORD,GenericInputToken.value);
#endif
      else
        {
#if OBJECT_SYSTEM
         PrintCLIPS(WERROR,"Expected a valid class name or query.\n");
#else
         PrintCLIPS(WERROR,"Expected a valid type name or query.\n");
#endif
         ReturnExpression(query);
         ReturnExpression(types);
         return(NULL);
        }
      SavePPBuffer(" ");
      GetToken(read_source,&GenericInputToken);
     }
   PPBackup();
   PPBackup();
   SavePPBuffer(")");
   if ((types == NULL) && (query == NULL))
     {
#if OBJECT_SYSTEM
      PrintCLIPS(WERROR,"Expected a valid class name or query.\n");
#else
      PrintCLIPS(WERROR,"Expected a valid type name or query.\n");
#endif
      return(NULL);
     }
   rptr = get_struct(restriction);
   rptr->query = query;
   PackRestrictionTypes(rptr,types);
   ReturnExpression(types);
   return(rptr);
  }
  
/**********************************************************
  NAME         : DuplicateParameters
  DESCRIPTION  : Examines the parameter expression
                   chain for a method looking duplicates.
  INPUTS       : 1) The parameter chain (can be NULL)
                 2) Caller's buffer for address of
                    last node searched (can be used to
                    later attach new parameter)
                 3) The name of the parameter being checked
  RETURNS      : TRUE if duplicates found, FALSE otherwise
  SIDE EFFECTS : Caller's prv address set
  NOTES        : Assumes all parameter list nodes are WORDS
 **********************************************************/
static int DuplicateParameters(head,prv,name)
  EXPRESSION *head,**prv;
  SYMBOL_HN *name;
  {
   *prv = NULL;
   while (head != NULL)
     {
      if (head->value == (VOID *) name)
        {
         PrintCLIPS(WERROR,"Duplicate parameter names not allowed.\n");
         return(TRUE);
        }
      *prv = head;
      head = head->next_arg;
     }
   return(FALSE);
  }
  
/***************************************************
  NAME         : DeleteTempRestricts
  DESCRIPTION  : Deallocates the method
                   temporary parameter list
  INPUTS       : The head of the list
  RETURNS      : Nothing useful
  SIDE EFFECTS : List deallocated
  NOTES        : None
 ***************************************************/
static VOID DeleteTempRestricts(phead)
  EXPRESSION *phead;
  {
   EXPRESSION *ptmp;
   RESTRICTION *rtmp;
   
   while (phead != NULL)
     {
      ptmp = phead;
      phead = phead->next_arg;
      rtmp = (RESTRICTION *) ptmp->arg_list;
      rtn_struct(expr,ptmp);
      ReturnExpression(rtmp->query);
      if (rtmp->tcnt != 0)
        rm((VOID *) rtmp->types,(int) (sizeof(VOID *) * rtmp->tcnt));
      rtn_struct(restriction,rtmp);
     }
  }
  
/*****************************************************************
  NAME         : AddParameter
  DESCRIPTION  : Shoves a new paramter with its restriction
                   onto the list for a method
                 The parameter list is a list of expressions
                   linked by neext_arg pointers, and the
                   arg_list pointers are used for the restrictions
  INPUTS       : 1) The head of the list
                 2) The bottom of the list
                 3) The parameter name
                 4) The parameter restriction
  RETURNS      : The (new) head of the list
  SIDE EFFECTS : New parameter expression node allocated, set,
                   and attached
  NOTES        : None
 *****************************************************************/
static EXPRESSION *AddParameter(phead,pprv,pname,rptr)
  EXPRESSION *phead,*pprv;
  SYMBOL_HN *pname;
  RESTRICTION *rptr;
  {
   EXPRESSION *ptmp;
   
   ptmp = GenConstant(SYMBOL,(VOID *) pname);
   if (phead == NULL)
     phead = ptmp;
   else
     pprv->next_arg = ptmp;
   ptmp->arg_list = (EXPRESSION *) rptr;
   return(phead);
  }

/**********************************************************************
  NAME         : ValidType
  DESCRIPTION  : Examines the name of a restriction type and
                   forms a list of integer-code expressions
                   corresponding to the primitive CLIPS types:
                   1) integer,float,symbol,string and multi-field all
                        go to single-expressions of the apprpriate code
                   2) number goes to an integer followed by a float
                   3) primitive goes to -1 (will be used later)
                 
                 (or a Class address if COOL is installed)
  INPUTS       : The type name
  RETURNS      : The expression chain (NULL on errors)
  SIDE EFFECTS : Expression type chain allocated
                   one or more nodes holding codes for types
                   (or class addresses)
  NOTES        : None
 **********************************************************************/
static EXPRESSION *ValidType(tname)
  SYMBOL_HN *tname;
  {
#if OBJECT_SYSTEM
   CLASS_TYPE *cls;
   
   cls = FindDefclassBySymbol(tname);
   if (cls != NULL)
     return(GenConstant(EXTERNAL_ADDRESS,(VOID *) cls));
   PrintCLIPS(WERROR,"Unknown class in method.\n");
#else
   EXPRESSION *tmp;

   if (strcmp(ValueToString(tname),INTEGER_TYPE_NAME) == 0)
     return(GenConstant(INTEGER,(VOID *) AddLong((long) INTEGER)));
   if (strcmp(ValueToString(tname),FLOAT_TYPE_NAME) == 0)
     return(GenConstant(INTEGER,(VOID *) AddLong((long) FLOAT)));
   if (strcmp(ValueToString(tname),SYMBOL_TYPE_NAME) == 0)
     return(GenConstant(INTEGER,(VOID *) AddLong((long) SYMBOL)));
   if (strcmp(ValueToString(tname),STRING_TYPE_NAME) == 0)
     return(GenConstant(INTEGER,(VOID *) AddLong((long) STRING)));
   if (strcmp(ValueToString(tname),MULTIFIELD_TYPE_NAME) == 0)
     return(GenConstant(INTEGER,(VOID *) AddLong((long) MULTIFIELD)));
   if (strcmp(ValueToString(tname),ADDRESS_TYPE_NAME) == 0)
     return(GenConstant(INTEGER,(VOID *) AddLong((long) EXTERNAL_ADDRESS)));
   if (strcmp(ValueToString(tname),NUMBER_TYPE_NAME) == 0)
     {
      tmp = GenConstant(INTEGER,(VOID *) AddLong((long) INTEGER));
      tmp->next_arg = GenConstant(INTEGER,(VOID *) AddLong((long) FLOAT));
      return(tmp);
     }
   if (strcmp(ValueToString(tname),LEXEME_TYPE_NAME) == 0)
     {
      tmp = GenConstant(INTEGER,(VOID *) AddLong((long) SYMBOL));
      tmp->next_arg = GenConstant(INTEGER,(VOID *) AddLong((long) STRING));
      return(tmp);
     }
   if (strcmp(ValueToString(tname),PRIMITIVE_TYPE_NAME) == 0)
     return(GenConstant(INTEGER,(VOID *) AddLong(-1L)));
   if (strcmp(ValueToString(tname),OBJECT_TYPE_NAME) == 0)
     return(GenConstant(INTEGER,(VOID *) AddLong(-1L)));
   PrintCLIPS(WERROR,"Unknown type in method.\n");
#endif
   return(NULL);
  }
  
/*****************************************************
  NAME         : PackRestrictionTypes
  DESCRIPTION  : Takes the restriction type list
                   and packs it into a contiguous
                   array of VOID *.
  INPUTS       : 1) The restriction structure
                 2) The types expression list
  RETURNS      : Nothing useful
  SIDE EFFECTS : Array allocated
  NOTES        : If the object-system is not being
                   used, then the types are integer
                   codes representing the CLIPS types.
                   -1 means all types are acceptable.
 *****************************************************/
static VOID PackRestrictionTypes(rptr,types)
  RESTRICTION *rptr;
  EXPRESSION *types;
  {
   EXPRESSION *tmp;
   register int i;
   
   rptr->tcnt = 0;
   for (tmp = types ; tmp != NULL ; tmp = tmp->next_arg)
     {
#if ! OBJECT_SYSTEM
      if (ValueToInteger(tmp->value) == -1L)
        {
         rptr->tcnt = 0;
         rptr->types = NULL;
         return;
        }
#endif
      rptr->tcnt++;
     }
   if (rptr->tcnt == 0)
     {
      rptr->types = NULL;
      return;
     }
   rptr->types = (VOID **) gm2((int) (sizeof(VOID *) * rptr->tcnt));
   for (i = 0 , tmp = types ; i < rptr->tcnt ; i++ , tmp = tmp->next_arg)
#if OBJECT_SYSTEM
     rptr->types[i] = (VOID *) tmp->value;
#else
     rptr->types[i] = (VOID *) ValueToInteger(tmp->value);
#endif
  }
  
/****************************************************************
  NAME         : ReplaceParameters
  DESCRIPTION  : Examines an expression for variables
                   and replaces any that correspond to
                   defmethod parameters or globals
                   with function calls that get these
                   variables' values at run-time.
                   For example, generic function arguments
                   are stored an array at run-time, so at
                   parse-time, parameter-references are replaced
                   with function calls referencing this array at
                   the appropriate position.
  INPUTS       : 1) The expression-actions to be examined
                 2) The parameter list
                 3) The wildcard parameter symbol (NULL if none)
  RETURNS      : TRUE if all OK, FALSE otherwise
  SIDE EFFECTS : Variable references replaced with function calls
  NOTES        : None
 ****************************************************************/
static int ReplaceParameters(actions,parameterList,wildcard)
  struct expr *actions;
  struct expr *parameterList;
  SYMBOL_HN *wildcard;
  {
   int position;
   BOOLEAN boundAlso;
   EXPRESSION *arg_lvl;
   SYMBOL_HN *bindName;
   
   while (actions != NULL)
     {
      if ((actions->type == BWORD) || (actions->type == BWORDS))
        {
         /*===============================================*/
         /* See if the variable is in the parameter list. */
         /*===============================================*/
          
         bindName = (SYMBOL_HN *) actions->value;
         position = FindParameter(bindName,parameterList,wildcard);
         
         /*=================================================================*/
         /* Check to see if the variable is bound within the defmethod.     */
         /*=================================================================*/
   
         boundAlso = SearchParsedBindNames(bindName);
         
         /*=============================================*/
         /* If variable is not defined in the parameter */
         /* list or as part of a bind action then...    */
         /*=============================================*/
         
         if ((position == 0) && (boundAlso == FALSE))
           {
            PrintCLIPS(WERROR,"\nUndefined variable ");
            PrintCLIPS(WERROR,ValueToString(bindName));        
            PrintCLIPS(WERROR," referenced in method.\n");
            return(FALSE);
           }
           
         /*===================================================*/
         /* Else if variable is defined in the parameter list */
         /* and not rebound within the defmethod then...     */
         /*===================================================*/
         
         else if ((position > 0) && (boundAlso == FALSE))
           {
            actions->type = FCALL;
            if (bindName != wildcard)
              actions->value = (VOID *) FindFunction("(gnrc-runknown)");
            else
              actions->value = (VOID *) FindFunction("(gnrc-wildargs)");
            actions->arg_list = GenConstant(INTEGER,AddLong((long) position));
           }
           
         /*===========================================================*/
         /* Else the variable is rebound within the deffunction so... */
         /*===========================================================*/
         
         else 
           {
            actions->type = FCALL;
            actions->value = (VOID *) FindFunction("(gnrc-bind)");
            
            actions->arg_list = GenConstant(SYMBOL,bindName);
            arg_lvl = actions->arg_list;

            if (position == 0)
              arg_lvl->next_arg = GenConstant(INTEGER,AddLong(0L));
            else if (bindName != wildcard)
              arg_lvl->next_arg = GenConstant(INTEGER,AddLong((long) position));
            else
              arg_lvl->next_arg = GenConstant(INTEGER,AddLong((long) -position));
           }
        }
#if DEFGLOBAL_CONSTRUCT
      else if (actions->type == GBWORD)
        { if (ReplaceGlobalVariable(actions) == FALSE) return(FALSE); }
#endif
        
      else if (actions->arg_list != NULL)
        {
         if (! ReplaceParameters(actions->arg_list,parameterList,wildcard))
           return(FALSE);
        }
           
      actions = actions->next_arg;
     }
   return(TRUE);
  }
       
/***************************************************
  NAME         : FindParameter
  DESCRIPTION  : Determines the relative position in
                   an n-element list of a certain
                   parameter.  The index is 1..n.
  INPUTS       : 1) Parameter name
                 2) Parameter list
                 3) Wildcard symbol (NULL if none)
  RETURNS      : Index of parameter in list, 0 if
                   not found
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
static int FindParameter(name,parameterList,wildcard)
  SYMBOL_HN *name;
  EXPRESSION *parameterList;
  SYMBOL_HN *wildcard;
  {
   int i = 1;
   
   while (parameterList != NULL)
     {
      if (parameterList->value == (VOID *) name) return(i);
      i++;
      parameterList = parameterList->next_arg;
     }
   /* ===============================================================
      Wildcard is not stored in actual list but know is always at end
      =============================================================== */
   if (name == wildcard)
     return(i);
   return(0);
  }
  
/**********************************************************************
  NAME         : SaveDefgenerics
  DESCRIPTION  : Outputs pretty-print forms of generic function headers
  INPUTS       : The logical name of the output
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : None
 **********************************************************************/
static VOID SaveDefgenerics(log)
  char *log;
  {
   GENERIC_FUNC *gfunc;
   register unsigned i;

   for (gfunc = GenericList ; gfunc != NULL ; gfunc = gfunc->nxt)
     {
      if (gfunc->pp_form != NULL)
        {
         PrintInChunks(log,gfunc->pp_form);
         PrintCLIPS(log,"\n");
        }
      else
        {
         for (i = 0 ; i < gfunc->mcnt ; i++)
           if (gfunc->methods[i].pp_form != NULL)
             break;
         if (i < gfunc->mcnt)
           {
            PrintCLIPS(log,"(defgeneric ");
            PrintCLIPS(log,ValueToString(gfunc->name));
            PrintCLIPS(log,")\n\n");
           }
        }
     }
  }

/**********************************************************************
  NAME         : SaveDefmethods
  DESCRIPTION  : Outputs pretty-print forms of generic function methods
  INPUTS       : The logical name of the output
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : None
 **********************************************************************/
static VOID SaveDefmethods(log)
  char *log;
  {
   GENERIC_FUNC *gfunc;
   register int i;
   
   for (gfunc = GenericList ; gfunc != NULL ; gfunc = gfunc->nxt)
     {
      for (i = 0 ; i < gfunc->mcnt ; i++)
        {
         if (gfunc->methods[i].pp_form != NULL)
           {
            PrintInChunks(log,gfunc->methods[i].pp_form);
            PrintCLIPS(log,"\n");
           }
        }
     }
  }

/**************************************************
  NAME         : RemoveGeneric
  DESCRIPTION  : Removes a generic function node
                   from the generic list along with
                   all its methods
  INPUTS       : The generic function
  RETURNS      : 1 if successful, 0 otherwise
  SIDE EFFECTS : List adjusted
                 Nodes deallocated
  NOTES        : None
 **************************************************/
static int RemoveGeneric(gfunc)
  GENERIC_FUNC *gfunc;
  {
   register int i;
   
#if BLOAD || BLOAD_AND_BSAVE
   if (Bloaded() == CLIPS_TRUE)
     {
      PrintCLIPS(WERROR,"Unable to delete generic function ");
      PrintCLIPS(WERROR,ValueToString(gfunc->name));
      PrintCLIPS(WERROR,".\n");
      return(0);
     }
#endif

   if (gfunc->busy > 0)
     {
      PrintCLIPS(WERROR,"Unable to delete generic function ");
      PrintCLIPS(WERROR,ValueToString(gfunc->name));
      PrintCLIPS(WERROR,".\n");
      return(0);
     }
   
   for (i = 0 ; i < gfunc->mcnt ; i++)
     DeleteMethodInfo(gfunc,&gfunc->methods[i]);
   
   if (GetCompilationsWatch())
     {
      PrintCLIPS(WDIALOG,"Removing defgeneric ");
      PrintCLIPS(WDIALOG,ValueToString(gfunc->name));
      PrintCLIPS(WDIALOG,".\n");
     }
   if (gfunc->mcnt != 0)
     rm((VOID *) gfunc->methods,(int) (sizeof(METHOD) * gfunc->mcnt));
   DecrementSymbolCount(gfunc->name);
   if (gfunc->prv == NULL)
     GenericList = gfunc->nxt;
   else
     gfunc->prv->nxt = gfunc->nxt;
   if (gfunc->nxt == NULL)
     GenericListBottom = gfunc->prv;
   else
     gfunc->nxt->prv = gfunc->prv;
   if (gfunc->pp_form != NULL)
     rm((VOID *) gfunc->pp_form,(int) (sizeof(char) * (strlen(gfunc->pp_form)+1)));
   rtn_struct(generic_func,gfunc);
   return(1);
  }
  
/****************************************************
  NAME         : RemoveGenericMethod
  DESCRIPTION  : Removes a generic function method
                   from the array and removes the
                   generic too if its the last method
  INPUTS       : 1) The generic function
                 2) The array index of the method
  RETURNS      : Nothing useful
  SIDE EFFECTS : List adjusted
                 Nodes deallocated
  NOTES        : Assumes deletion is safe
 ****************************************************/
static VOID RemoveGenericMethod(gfunc,gi)
  GENERIC_FUNC *gfunc;
  int gi;
  {
   METHOD *narr;
   register int b,e;
   
   DeleteMethodInfo(gfunc,&gfunc->methods[gi]);
   if (gfunc->mcnt == 1)
     {
      rm((VOID *) gfunc->methods,(int) sizeof(METHOD));
      gfunc->mcnt = 0;
      gfunc->methods = NULL;
     }
   else
     {
      gfunc->mcnt--;
      narr = (METHOD *) gm2((int) (sizeof(METHOD) * gfunc->mcnt));
      for (b = e = 0 ; b < gfunc->mcnt ; b++ , e++)
        {
         if (b == gi)
           e++;
         CopyMemory(METHOD,1,&narr[b],&gfunc->methods[e]);
        }
      rm((VOID *) gfunc->methods,(int) (sizeof(METHOD) * (gfunc->mcnt+1)));
      gfunc->methods = narr;
     }
  }

#endif

#if ! OBJECT_SYSTEM

/***************************************************
  NAME         : TypeOf
  DESCRIPTION  : Works like "class" in COOL
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax: (type <primitive>)
 ***************************************************/
globle VOID TypeOf(result)
  DATA_OBJECT *result;
  {
   DATA_OBJECT temp;

   result->type = SYMBOL;
   result->value = (VOID *) CLIPSFalseSymbol;
   if (ArgCountCheck("type",EXACTLY,1) != 1)
     {
      SetEvaluationError(TRUE);
      return;
     }
   EvaluateExpression(GetFirstArgument(),&temp);
   switch (temp.type)
     {
      case INTEGER  : result->value = (VOID *) AddSymbol(INTEGER_TYPE_NAME);
                      return;
      case FLOAT    : result->value = (VOID *) AddSymbol(FLOAT_TYPE_NAME);
                      return;
      case SYMBOL     : result->value = (VOID *) AddSymbol(SYMBOL_TYPE_NAME);
                      return;
      case STRING   : result->value = (VOID *) AddSymbol(STRING_TYPE_NAME);
                      return;
      case MULTIFIELD : result->value = (VOID *) AddSymbol(MULTIFIELD_TYPE_NAME);
                      return;
      case EXTERNAL_ADDRESS : result->value = (VOID *) AddSymbol(ADDRESS_TYPE_NAME);
                      return;
      default       : PrintCLIPS(WERROR,"Undefined type in function type.\n");
                      SetEvaluationError(TRUE);
     }
  }

#endif
  
/****************************************************************
  NAME         : GrabGenericWildargs
  DESCRIPTION  : Groups a portion of the GenericStackFrame
                   into a multi-field variable
  INPUTS       : 1) Caller's result value buffer
                 2) Starting index in GenericStackFrame
                      for grouping of arguments into
                      multi-field variable
  RETURNS      : Nothing useful
  SIDE EFFECTS : Multi-field variable allocated and set
                   with corresponding values of GenericStackFrame
  NOTES        : Multi-field IS on list of ephemeral segments
 ****************************************************************/
static VOID GrabGenericWildargs(result,index)
  DATA_OBJECT *result;
  int index;
  {
   register int i,j,k;
   int size;
   struct fact *fptr;
   DATA_OBJECT *val;

   result->type = MULTIFIELD;
   result->begin = 0;
   result->end = -1;
   if (GenericStackSize-index < 0)
     {
      result->value = (VOID *) CreateMultifield(0);
      return;
     }
   for (i = index-1 , size = 0 ; i < GenericStackSize ; i++)
     {
      if (GenericStackFrame[i].type != MULTIFIELD)
        size++;
      else
        size += ((DATA_OBJECT *) GenericStackFrame[i].value)->end - 
                ((DATA_OBJECT *) GenericStackFrame[i].value)->begin + 1;
     }
   result->end = size-1;
   result->value = (VOID *) CreateMultifield(size);
   fptr = (struct fact *) result->value;
   for (i = index-1 , j = 0 ; i < GenericStackSize ; i++)
     {
      if (GenericStackFrame[i].type != MULTIFIELD)
        {
         fptr->atoms[j].type = GenericStackFrame[i].type; 
         fptr->atoms[j].value = GenericStackFrame[i].value; 
         j++;
        }
      else
        {
         val = (DATA_OBJECT *) GenericStackFrame[i].value;
         for (k = val->begin ; k <= val->end ; k++ , j++)
           {
            fptr->atoms[j].type = ((struct fact *) val->value)->atoms[k].type;
            fptr->atoms[j].value = ((struct fact *) val->value)->atoms[k].value;
           }         
        }
     }
  }

/***************************************************
  NAME         : CheckGenericExists
  DESCRIPTION  : Finds the address of named
                  generic function and prints out
                  error message if not found
  INPUTS       : 1) Calling function
                 2) Name of generic function
  RETURNS      : Generic function address (NULL if
                   not found)
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
static GENERIC_FUNC *CheckGenericExists(fname,gname)
  char *fname,*gname;
  {
   GENERIC_FUNC *gfunc;

   gfunc = (GENERIC_FUNC *) FindDefgeneric(gname);
   if (gfunc == NULL)
     {
      PrintCLIPS(WERROR,"Unable to find generic function ");
      PrintCLIPS(WERROR,gname);
      PrintCLIPS(WERROR," in function ");
      PrintCLIPS(WERROR,fname);
      PrintCLIPS(WERROR,".\n");
     }
   return(gfunc); 
  }

/***************************************************
  NAME         : CheckMethodExists
  DESCRIPTION  : Finds the array index of the
                  specified method and prints out
                  error message if not found
  INPUTS       : 1) Calling function
                 2) Generic function address
                 3) Index of method
  RETURNS      : Method array index (-1 if not found)
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
static int CheckMethodExists(fname,gfunc,mi)
  char *fname;
  GENERIC_FUNC *gfunc;
  int mi;
  {
   int fi;

   fi = FindMethodByIndex(gfunc,mi);
   if (fi == -1)
     {
      PrintCLIPS(WERROR,"Unable to find method ");
      PrintCLIPS(WERROR,ValueToString(gfunc->name));
      PrintCLIPS(WERROR," #");
      PrintLongInteger(WERROR,(long) mi);
      PrintCLIPS(WERROR," in function ");
      PrintCLIPS(WERROR,fname);
      PrintCLIPS(WERROR,".\n");
     }
   return(fi); 
  }

#endif

/***************************************************
  NAME         : 
  DESCRIPTION  : 
  INPUTS       : 
  RETURNS      : 
  SIDE EFFECTS : 
  NOTES        : 
 ***************************************************/
