   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*             CLIPS Version 5.10  07/17/91            */
   /*                                                     */
   /*                FACT COMMANDS MODULE                 */
   /*******************************************************/

/*************************************************************/
/* Purpose:                                                  */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Gary D. Riley                                        */
/*                                                           */
/* Contributing Programmer(s):                               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/
 
#define _FACTCOM_SOURCE_

#include <stdio.h>
#define _CLIPS_STDIO_
#include <string.h>

#include "setup.h"
#include "clipsmem.h"
#include "expressn.h"
#include "facts.h"
#include "generate.h"
#include "reteutil.h"
#include "evaluatn.h" 
#include "utility.h"
#include "match.h" 
#include "router.h" 
#include "scanner.h"
#include "constant.h"
#include "parsutil.h"
#include "lgcldpnd.h"
#include "sysprime.h"

#if DEFTEMPLATE_CONSTRUCT
#include "deftempl.h"
#include "deftmfun.h"
#endif

#if OBJECT_SYSTEM
#include "extobj.h"
#endif

#if ANSI_COMPILER  
   VOID                           InitFactCommands(void);
   int                            AssertCommand(void);
   VOID                           RetractCommand(void);
#if DEBUGGING_FUNCTIONS
   VOID                           DependenciesCommand(void);
   VOID                           DependentsCommand(void);
   VOID                           FactsCommand(void);
   static long int                GetFactsArgument(int,int);
#endif
   int                            SetFactDuplicationCommand(void);
   int                            GetFactDuplicationCommand(void);
   VOID                           SaveFactsCommand(void);
   VOID                           LoadFactsCommand(void);
   int                            SaveFacts(char *);
   int                            LoadFacts(char *);
   static struct expr            *StandardLoadFact(char *,struct token *);
   long int                       FactIndexFunction(void);
#else
   VOID                           InitFactCommands();
   int                            AssertCommand();
   VOID                           RetractCommand();
#if DEBUGGING_FUNCTIONS
   VOID                           DependenciesCommand();
   VOID                           DependentsCommand();
   VOID                           FactsCommand();
   static long int                GetFactsArgument();
#endif
   int                            SetFactDuplicationCommand();
   int                            GetFactDuplicationCommand();
   VOID                           SaveFactsCommand();
   VOID                           LoadFactsCommand();
   int                            SaveFacts();
   int                            LoadFacts();
   static struct expr            *StandardLoadFact();
   long int                       FactIndexFunction();
#endif

/************************************************************/
/* InitFactCommands:                                         */
/************************************************************/
globle VOID InitFactCommands()
  {
#if ! RUN_TIME
#if DEBUGGING_FUNCTIONS
   DefineFunction("facts",        'v', PTIF FactsCommand,        "FactsCommand");
   DefineFunction("dependencies", 'v', PTIF DependenciesCommand, "DependenciesCommand");
   DefineFunction("dependents",   'v', PTIF DependentsCommand,   "DependentsCommand");
#endif

   DefineFunction("assert",       'v', PTIF AssertCommand,  "AssertCommand");
   DefineFunction("retract",      'v', PTIF RetractCommand, "RetractCommand");
   
   DefineFunction("get-fact-duplication",'b',
                  GetFactDuplicationCommand,"GetFactDuplicationCommand");
   DefineFunction("set-fact-duplication",'b',
                  SetFactDuplicationCommand,"SetFactDuplicationCommand");
                  
   DefineFunction("save-facts",    'v', PTIF SaveFactsCommand, "SaveFactsCommand");
   DefineFunction("load-facts",    'v', PTIF LoadFactsCommand, "LoadFactsCommand");
   DefineFunction("fact-index",    'l', PTIF FactIndexFunction,"FactIndexFunction");
#endif
  }
 
/*************************************************************/
/* AssertCommand:                                            */
/*************************************************************/
globle int AssertCommand()
  {
   struct element *elem_a;
    
   DATA_OBJECT arg_ptr;
   struct expr *test_ptr;
   struct fact *factPtr;
   struct lelement *elem_ptr;
   struct lelement *head = NULL;
   struct lelement *last_add = NULL, *last_elem, *new_elem;
   int extent, i;
   int count = 0;
   int error = CLIPS_FALSE;
#if DEFTEMPLATE_CONSTRUCT
   int multiSlotPosition;
   int currentPosition = 0;
   SYMBOL_HN *dName = NULL;
#endif

   /*====================================================*/
   /* Determine if a deftemplate fact is being asserted. */
   /*====================================================*/
   
   test_ptr = GetFirstArgument();
   
#if DEFTEMPLATE_CONSTRUCT
   if ((test_ptr != NULL) ? (test_ptr->type == SYMBOL) : CLIPS_FALSE)
     {
      dName =  (SYMBOL_HN *) test_ptr->value;
      multiSlotPosition = GetMultiSlotPosition(dName);
     }
   else
     { multiSlotPosition = -1; }
#endif

   /*==========================================================*/
   /* Evaluate each assert argument constructing the new fact. */
   /*==========================================================*/
   
   while (test_ptr != NULL)
     {
      switch (test_ptr->type)
        {
         case FLOAT:
         case INTEGER:
#if OBJECT_SYSTEM
         case INSTANCE_NAME:
#endif
         case STRING:
         case SYMBOL:
           elem_ptr = get_struct(lelement);
           elem_ptr->type = test_ptr->type;
           elem_ptr->value = test_ptr->value;
           elem_ptr->next = NULL;
           count++;
           break;
           
         default:
#if DEFRULE_CONSTRUCT
           if (test_ptr->type == SCALL_GET_VAR)
             { GetVarSysFunction(&arg_ptr,test_ptr->value); }
           else
#endif
           EvaluateExpression(test_ptr,&arg_ptr);
         
           if (GetType(arg_ptr) == MULTIFIELD)
             {
              extent = (arg_ptr.end - arg_ptr.begin) + 1;
              count += extent;
 
#if DEFTEMPLATE_CONSTRUCT             
              if ((extent != 1) &&
                  ((multiSlotPosition == 0) || (currentPosition < multiSlotPosition)))
                { 
                 error = CLIPS_TRUE;
                 MultiIntoSingleFieldSlotError(currentPosition,dName); 
                }
#endif
              elem_ptr = last_elem = NULL;
              elem_a = ((struct fact *) arg_ptr.value)->atoms;
              i = arg_ptr.begin;
              while (i <= arg_ptr.end)
                {
                 new_elem = get_struct(lelement);
                 new_elem->type = elem_a[i].type;
                 new_elem->value = elem_a[i].value;
                 new_elem->next = NULL;
                 if (last_elem != NULL)
                   { last_elem->next = new_elem; }
                 else
                   { elem_ptr = new_elem; }
                 last_elem = new_elem;
                 i++;
                }
             
             }      
           else
             {
              elem_ptr = get_struct(lelement);
              elem_ptr->type = arg_ptr.type;
              count++;
              elem_ptr->type = arg_ptr.type;
              elem_ptr->value = arg_ptr.value;
              elem_ptr->next = NULL;
             }
             
           break;
        }

      if (last_add == NULL)
        { head = elem_ptr; }
      else
        { last_add->next = elem_ptr; }

      if (elem_ptr != NULL)
        {
         while (elem_ptr->next != NULL)
           { elem_ptr = elem_ptr->next; }
         last_add = elem_ptr;
        }

      test_ptr = GetNextArgument(test_ptr);
#if DEFTEMPLATE_CONSTRUCT
      currentPosition++;
#endif
     }
     
   /*===========================================*/
   /* Copy the list of fact fields to an array. */
   /*===========================================*/
   
   if (count == 0) return(0);
   
   factPtr = (struct fact *) CreateFact(count); 
   elem_a = factPtr->atoms;
   i = 0;
   while (head != NULL)
     {
      elem_a[i].type = head->type;
      elem_a[i].value = head->value;
       
      last_add = head->next;
      rtn_struct(lelement,head);
      head = last_add;
      i++;
     }
     
   if (error)
     {
      ReturnElements(factPtr);
      return(0);
     }
   
   /*================================*/
   /* Add the fact to the fact-list. */
   /*================================*/
   
   AddFact((VOID *) factPtr);

   return(1);
  }

/************************************************************/
/* RetractCommand:                                          */
/************************************************************/
globle VOID RetractCommand()
  {
   long int fact_num;
   int found_fact;
   struct fact *ptr;
   struct expr *test_ptr;
   DATA_OBJECT compute_result;
   int i = 1;

   test_ptr = GetFirstArgument();

   while (test_ptr != NULL)
     {
      EvaluateExpression(test_ptr,&compute_result);
      if (compute_result.type == INTEGER)
        {
         fact_num = ValueToLong(compute_result.value);
         if (fact_num < 0)
           { 
            RemoveAllFacts();
            return;
           }
         found_fact = CLIPS_FALSE;
         ptr = (struct fact *) GetNextFact(NULL);
         while (ptr != NULL)
           {
            if (ptr->factIndex == fact_num)
              { 
               RetractFact((VOID *) ptr);
               ptr = NULL;
               found_fact = CLIPS_TRUE;
              }
            else
              { ptr = ptr->next; }
           }
         if (found_fact == CLIPS_FALSE)
           {
            PrintCLIPS(WERROR,"Unable to find fact f-");
            PrintLongInteger(WERROR,fact_num);
            PrintCLIPS(WERROR,"\n");
           }
        }
      else if (compute_result.type == FACT_ADDRESS)
        { RetractFact(compute_result.value); }
      else
        {
         ExpectedTypeError("retract",i,"fact-address or fact-index");
         SetEvaluationError(TRUE);
        }

      i++;
      test_ptr = GetNextArgument(test_ptr);
     }
  }
  
/************************************************************/
/* SetFactDuplicationCommand:                        */
/************************************************************/
globle int SetFactDuplicationCommand()
  {
   int oldValue;
   DATA_OBJECT arg_ptr;
   
   oldValue = GetFactDuplication();
   
   if (ArgCountCheck("set-fact-duplication",EXACTLY,1) == -1) 
     { return(oldValue); }
     
   RtnUnknown(1,&arg_ptr);
     
   if ((arg_ptr.value == (VOID *) CLIPSFalseSymbol) && (arg_ptr.type == SYMBOL)) 
     { SetFactDuplication(CLIPS_FALSE); }
   else
     { SetFactDuplication(CLIPS_TRUE); }
     
   return(oldValue);
  }
  
/************************************************************/
/* GetFactDuplicationCommand:                        */
/************************************************************/
globle int GetFactDuplicationCommand()
  {
   int oldValue;
   
   oldValue = GetFactDuplication();
   
   if (ArgCountCheck("get-fact-duplication",EXACTLY,0) == -1) 
     { return(oldValue); }
     
   return(oldValue);
  }

/****************************************/
/* FactIndexFunction:                   */
/****************************************/
globle long int FactIndexFunction()
  {
   DATA_OBJECT item;

   if (ArgCountCheck("fact-index",EXACTLY,1) == -1) return(-1L);

   RtnUnknown(1,&item);

   if (GetType(item) == FACT_ADDRESS)
     { return (((struct fact *) ClipsGetValue(item))->factIndex); }
   else
     {
      ExpectedTypeError("fact-index",1,"fact-address");
      return(-1L);
     }
  }
  
#if DEBUGGING_FUNCTIONS

/***************************************************/
/* FactsCommand:  Displays the facts in fact list. */
/*   Syntax: (facts)                               */
/***************************************************/
globle VOID FactsCommand()
  {   
   struct fact *factPtr;
   int num_arg;
   long int start, end, max;
   long count = 0;
   
   if ((num_arg = ArgCountCheck("facts",NO_MORE_THAN,3)) == -1) return;
   
   if ((start = GetFactsArgument(1,num_arg)) == -2) return;
   if ((end = GetFactsArgument(2,num_arg)) == -2) return;
   if ((max = GetFactsArgument(3,num_arg)) == -2) return;
   
   factPtr = (struct fact *) GetNextFact(NULL); 
   while (factPtr != NULL)
     {
      if (GetHaltExecution() == CLIPS_TRUE) return;
      if ((factPtr->factIndex > end) && (end != -1))
        {
         PrintTally(WDISPLAY,count,"fact","facts");
         return;
        }
        
      if (max == 0) 
        {
         PrintTally(WDISPLAY,count,"fact","facts");
         return;
        }
      
      if (factPtr->factIndex >= start)
        {
         PrintFactWithIdentifier(WDISPLAY,factPtr);
         PrintCLIPS(WDISPLAY,"\n");
         count++;
         if (max > 0) max--;
        }
         
      factPtr = (struct fact *) GetNextFact(factPtr);
     }
   
   PrintTally(WDISPLAY,count,"fact","facts");

   return;
  }

/***********************************/
/* GetFactsArgument:               */
/***********************************/
static long int GetFactsArgument(pos,num_arg)
  int pos, num_arg;
  {
   long int fint;
   DATA_OBJECT arg_ptr;
   
   if (pos > num_arg) return(-1L);
     
   if (ArgTypeCheck("facts",pos,INTEGER,&arg_ptr) == CLIPS_FALSE) return(-2L);
   fint = DOToLong(arg_ptr);
   if (fint < 0) 
     {
      ExpectedTypeError("facts",pos,"positive number");
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      return(-2L);
     }
   return(fint);
  }

/****************************************/
/* DependenciesCommand:                 */
/****************************************/
globle VOID DependenciesCommand()
  {
   DATA_OBJECT item;
   struct fact *ptr;

   if (ArgCountCheck("dependencies",EXACTLY,1) == -1) return;

   RtnUnknown(1,&item);

   if (GetType(item) == FACT_ADDRESS)
     {
#if DEFRULE_CONSTRUCT
      ListDependencies((struct fact *) ClipsGetValue(item));
#else
      PrintCLIPS(WDISPLAY,"None\n");
#endif
     }
   else if (GetType(item) == INTEGER)
     {
      if ((ptr = FindIndexedFact(DOToLong(item))) == NULL)
        {
         PrintCLIPS(WERROR,"Unable to find fact f-");
         PrintLongInteger(WERROR,DOToLong(item));
         PrintCLIPS(WERROR,"\n");
        }
      else
        { 
#if DEFRULE_CONSTRUCT
         ListDependencies(ptr);
#else
         PrintCLIPS(WDISPLAY,"None\n");
#endif
        }
     }
   else
     { ExpectedTypeError("dependencies",1,"fact-address or fact-index"); }
  }
  
/****************************************/
/* DependentsCommand:                 */
/****************************************/
globle VOID DependentsCommand()
  {
   DATA_OBJECT item;
   struct fact *ptr;

   if (ArgCountCheck("dependents",EXACTLY,1) == -1) return;

   RtnUnknown(1,&item);

   if (GetType(item) == FACT_ADDRESS)
     { 
#if DEFRULE_CONSTRUCT
      ListDependents((struct fact *) ClipsGetValue(item));
#else
      PrintCLIPS(WDISPLAY,"None\n");
#endif
     }
   else if (GetType(item) == INTEGER)
     {
      if ((ptr = FindIndexedFact(DOToLong(item))) == NULL)
        {
         PrintCLIPS(WERROR,"Fact ");
         PrintLongInteger(WERROR,DOToLong(item));
         PrintCLIPS(WERROR," does not exist\n");
        }
      else
        { 
#if DEFRULE_CONSTRUCT
         ListDependents(ptr);
#else
         PrintCLIPS(WDISPLAY,"None\n");
#endif
        }
     }
   else
     { ExpectedTypeError("dependents",1,"fact-address or fact-index"); }
  }
#endif

/******************************************************/
/* SaveFactsCommand:  Executes the save-facts commands. */
/*   Syntax:  (save-facts <file-name>)                */ 
/******************************************************/ 
globle VOID SaveFactsCommand()
  {
   char *file_found;

   if (ArgCountCheck("save-facts",EXACTLY,1) == -1) return;
   if ((file_found = GetFileName("save-facts",1)) == NULL) return;

   if (SaveFacts(file_found) == CLIPS_FALSE) 
     {
      OpenErrorMessage("save-facts",file_found);
      return;
     }

   return;
  }
  
/******************************************************/
/* LoadFactsCommand:  Executes the load-facts commands. */
/*   Syntax:  (load-facts <file-name>)                */ 
/******************************************************/ 
globle VOID LoadFactsCommand()
  {
   char *file_found;

   if (ArgCountCheck("load-facts",EXACTLY,1) == -1) return;
   if ((file_found = GetFileName("load-facts",1)) == NULL) return;

   if (LoadFacts(file_found) == CLIPS_FALSE) 
     {
      OpenErrorMessage("load-facts",file_found);
      return;
     }
 
   return;
  }

/********************************************************/
/* SaveFacts:  Saves the current set of facts into the */
/*   specified file.                                    */ 
/********************************************************/ 
globle BOOLEAN SaveFacts(file_found)
  char *file_found;
  {
   int tempValue1, tempValue2;
   struct fact *list;
   FILE *file_ptr;
   
   /*======================================================*/
   /* Open the file. Use either "fast save" or I/O Router. */
   /*======================================================*/

   if ((file_ptr = fopen(file_found,"w")) == NULL)
     { return(CLIPS_FALSE); }
   
   SetFastSave(file_ptr);
     
   /*=================*/
   /* Save the facts. */
   /*=================*/
            
   tempValue1 = PreserveEscapedCharacters;
   PreserveEscapedCharacters = CLIPS_TRUE;
   tempValue2 = AddressesToStrings;
   AddressesToStrings = CLIPS_TRUE;
   
   list = (struct fact *) GetNextFact(NULL); 
   while (list != NULL)
     {
      PrintFact((char *) file_ptr,list);
      PrintCLIPS((char *) file_ptr,"\n");
      list = (struct fact *) GetNextFact(list);
     }
     
   PreserveEscapedCharacters = tempValue1;
   AddressesToStrings = tempValue2;
     
   /*=================*/
   /* Close the file. */
   /*=================*/
   
   fclose(file_ptr);
   SetFastSave(NULL);
     
   return(CLIPS_TRUE);
  }

/**********************************************/
/* LoadFacts:  Loads a set of facts from the */
/*   specified file.                          */ 
/**********************************************/ 
globle BOOLEAN LoadFacts(file_found)
  char *file_found;
  {
   FILE *facts_file; 
   struct token input_tkn;
   struct expr *testPtr;
   DATA_OBJECT rv;

   /*======================================================*/
   /* Open the file. Use either "fast save" or I/O Router. */
   /*======================================================*/

   if ((facts_file = fopen(file_found,"r")) == NULL)
     { return(CLIPS_FALSE); }

   SetFastLoad(facts_file);
     
   /*=================*/
   /* Load the facts. */
   /*=================*/

   input_tkn.type = LPAREN;
   while (input_tkn.type != STOP)
     {
      testPtr = StandardLoadFact((char *) facts_file,&input_tkn);
      if (testPtr == NULL) input_tkn.type = STOP;
      else EvaluateExpression(testPtr,&rv);
      ReturnExpression(testPtr);
     }

   /*=================*/
   /* Close the file. */
   /*=================*/
   
   SetFastLoad(NULL);
   fclose(facts_file);
     
   return(CLIPS_TRUE);
  }
 
/**********************************************/
/* StandardLoadFact:                          */
/**********************************************/
static struct expr *StandardLoadFact(logicalName,input_tkn)
  char *logicalName;
  struct token *input_tkn;
  {
   int multi = CLIPS_FALSE, error = CLIPS_FALSE;
   struct expr *temp;
   
   GetToken(logicalName,input_tkn);
   if (input_tkn->type != LPAREN) return(NULL);
   
   temp = GenConstant(FCALL,FindFunction("assert"));
   temp->arg_list = GetRHSPattern(logicalName,input_tkn,&multi,&error,
                                  CLIPS_TRUE,CLIPS_FALSE,RPAREN);
                                  
   if (error == CLIPS_TRUE)
     {
      PrintCLIPS(WERROR,"Function load-facts encountered an error\n");
      SetEvaluationError(CLIPS_TRUE);
      ReturnExpression(temp);
      return(NULL);
     }
   
   if (ExpressionContainsVariables(temp,CLIPS_TRUE))
     {
      ReturnExpression(temp);
      return(NULL);
     }
     
   return(temp);
  }
 
