   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*             CLIPS Version 5.10  07/17/91            */
   /*                                                     */
   /*                  EXPRESSION MODULE                  */
   /*******************************************************/

/*************************************************************/
/* Purpose:                                                  */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Gary D. Riley                                        */
/*                                                           */
/* Contributing Programmer(s):                               */
/*      Brian L. Donnell                                     */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/

#define _EXPRESSN_SOURCE_

#include "setup.h"

#include <stdio.h>
#define _CLIPS_STDIO_
#include <string.h>

#include "constant.h"
#include "scanner.h"
#include "expressn.h"
#include "evaluatn.h"
#include "clipsmem.h"
#include "symbol.h"
#include "router.h"
#include "utility.h"
#include "generate.h"

#if DEFGENERIC_CONSTRUCT
#include "genrccom.h"
#include "genrcfun.h"
#endif

#if DEFFUNCTION_CONSTRUCT
#include "deffnctn.h"
#endif

#if OBJECT_SYSTEM
#include "extobj.h"
#endif

struct FunctionHash 
  {
   struct FunctionDefinition *fdPtr;
   struct FunctionHash *next;
  };

#define SIZE_FUNCTION_HASH 51

/***************************************/
/* LOCAL INTERNAL FUNCTION DEFINITIONS */
/***************************************/

#if ANSI_COMPILER
   static VOID                    InitializeFunctionHashTable(void);
#else
   static VOID                    InitializeFunctionHashTable();
#endif

/***************************************/
/* LOCAL INTERNAL VARIABLE DEFINITIONS */
/***************************************/

   static struct FunctionDefinition     *ListOfFunctions = NULL;
   static struct FunctionHash          **FunctionHashtable;

#if (! RUN_TIME)

/****************************************************************/
/* Function0Parse: Parses a function. Assumes that none of the */
/*   function has been parsed yet.                              */
/****************************************************************/
globle struct expr *Function0Parse(logicalName)
  char *logicalName;
  {
   struct token act_tkn;
   struct expr *top;
   
   /*=================================*/
   /* All functions begin with a '('. */
   /*=================================*/
      
   GetToken(logicalName,&act_tkn);
   if (act_tkn.type != LPAREN) 
     { 
      SyntaxErrorMessage("function calls");
      return(NULL);
     }
       
   /*=================================*/
   /* Parse the rest of the function. */
   /*=================================*/
        
   top = Function1Parse(logicalName);
   return(top);
  }

/*******************************************************/
/* Function1Parse: Parses a function. Assumes that the    */
/*   opening left parenthesis has already been parsed. */                             
/*******************************************************/
globle struct expr *Function1Parse(logicalName)
  char *logicalName;
  {
   struct token act_tkn;
   struct expr *top;
     
   /*========================*/
   /* Get the function name. */
   /*========================*/
        
   GetToken(logicalName,&act_tkn);
   if (act_tkn.type != SYMBOL)  
     {
      PrintCLIPS(WERROR,"\nA function name must be a symbol\n");
      return(NULL);
     }
       
   /*=================================*/
   /* Parse the rest of the function. */
   /*=================================*/
        
   top = Function2Parse(logicalName,ValueToString(act_tkn.value));
   return(top);
  }

/*****************************************************/
/* Function2Parse: Parses a function. Assumes that the  */
/*   opening left parenthesis and function name have */
/*   already been parsed.                            */
/*****************************************************/
globle struct expr *Function2Parse(logicalName,name)
  char *logicalName, *name;
  {
   struct FunctionDefinition *ft_ptr;
   struct expr *top;
#if DEFGENERIC_CONSTRUCT
   VOID *gfunc;
#endif
#if DEFFUNCTION_CONSTRUCT
   VOID *dptr;
#endif

   /*================================*/
   /* Has the function been defined? */
   /*================================*/

   ft_ptr = FindFunction(name);

#if DEFGENERIC_CONSTRUCT
   gfunc = FindDefgeneric(name);
#endif

#if DEFFUNCTION_CONSTRUCT
   if ((ft_ptr == NULL)
#if DEFGENERIC_CONSTRUCT
        && (gfunc == NULL)
#endif
     )
     dptr = FindDeffunction(name);
   else
     dptr = NULL;
#endif

   /*=============================*/
   /* Define top level structure. */
   /*=============================*/
   
#if DEFFUNCTION_CONSTRUCT
   if (dptr != NULL)
     top = GenConstant(PCALL,dptr);
   else
#endif
#if DEFGENERIC_CONSTRUCT
   if (gfunc != NULL)
     top = GenConstant(GCALL,gfunc);
   else
#endif
   if (ft_ptr != NULL)
     top = GenConstant(FCALL,ft_ptr);
   else
     {       
      PrintCLIPS(WERROR,"\nMissing function declaration for ");
      PrintCLIPS(WERROR,name);
      PrintCLIPS(WERROR,"\n");
      return(NULL); 
     }

   /*=======================================================*/
   /* Check to see if function has its own parsing routine. */
   /*=======================================================*/

#if DEFGENERIC_CONSTRUCT || DEFFUNCTION_CONSTRUCT
   if (top->type == FCALL)
#endif
     {
      if (ft_ptr->parser != NULL)
        { 
         top = (*ft_ptr->parser)(top,logicalName);
         return(top);
        }
     }
    
   /*========================================*/
   /* Default parsing routine for functions. */
   /*========================================*/

   top = CollectArguments(top,logicalName);
   if (top == NULL) return(NULL);
   return(top);
  }
    
/********************************************************/
/* CollectArguments:                                   */
/********************************************************/
globle struct expr *CollectArguments(top,logicalName)
  struct expr *top;
  char *logicalName;
  { 
   int error_flag;
   struct expr *last_one, *next_one;
   
   /*========================================*/
   /* Default parsing routine for functions. */
   /*========================================*/

   last_one = NULL;

   while (CLIPS_TRUE)
     {
      SavePPBuffer(" ");     
      
      error_flag = CLIPS_FALSE;
      next_one = ArgumentParse(logicalName,&error_flag);

      if (error_flag == CLIPS_TRUE)
        {
         ReturnExpression(top); 
         return(NULL);
        }
         
      if (next_one == NULL)
        {
         PPBackup();
         PPBackup();
         SavePPBuffer(")"); 
         return(top);
        }

      if (last_one == NULL)
        { top->arg_list = next_one; }
      else
        { last_one->next_arg = next_one; }

      last_one = next_one;
     }
  } 

/***********************************************************/
/* ArgumentParse                                          */
/***********************************************************/
globle struct expr *ArgumentParse(logicalName,error_flag)
  char *logicalName;
  int *error_flag;
  {
   struct expr *top;
   struct token arg_tkn;

   GetToken(logicalName,&arg_tkn);
   
   /*============================*/
   /* ')' counts as no argument. */
   /*============================*/
   
   if (arg_tkn.type == RPAREN)
     { return(NULL); }
        
   /*================================*/
   /* Parse constants and variables. */
   /*================================*/
   
   if ((arg_tkn.type == BWORD) || (arg_tkn.type == BWORDS) ||
       (arg_tkn.type == SYMBOL) || (arg_tkn.type == STRING) || 
#if DEFGLOBAL_CONSTRUCT
       (arg_tkn.type == GBWORD) ||
#endif
#if OBJECT_SYSTEM
       (arg_tkn.type == INSTANCE_NAME) ||
#endif
       (arg_tkn.type == FLOAT) || (arg_tkn.type == INTEGER))
     { return(GenConstant(arg_tkn.type,arg_tkn.value)); }

   /*======================*/
   /* Parse function call. */
   /*======================*/

   if (arg_tkn.type != LPAREN)
     {
      PrintCLIPS(WERROR,"Expected a constant, variable, or expression\n");
      *error_flag = CLIPS_TRUE;
      return(NULL);
     }
     
   top = Function1Parse(logicalName);
   if (top == NULL) *error_flag = CLIPS_TRUE;
   return(top);
  }
  
/********************************************************/
/* ParseAtomOrExpression:                               */
/********************************************************/
globle struct expr *ParseAtomOrExpression(logicalName)
  char *logicalName;
  {   
   struct token act_tkn;
   struct expr *rv;

   GetToken(logicalName,&act_tkn);
   if ((act_tkn.type == SYMBOL) || (act_tkn.type == STRING) ||
       (act_tkn.type == INTEGER) || (act_tkn.type == FLOAT) ||
#if OBJECT_SYSTEM
       (act_tkn.type == INSTANCE_NAME) ||
#endif
#if DEFGLOBAL_CONSTRUCT
       (act_tkn.type == GBWORD) ||
#endif
       (act_tkn.type == BWORD) || (act_tkn.type == BWORDS))
     { rv = GenConstant(act_tkn.type,act_tkn.value); }
   else if (act_tkn.type == LPAREN)
     {  
      rv = Function1Parse(logicalName);
      if (rv == NULL) return(NULL);
     }
   else
     {
      PrintCLIPS(WERROR,"Expected a constant, variable, or expression\n");
      return(NULL); 
     }
     
   return(rv);
  }
  
/*************************/
/* AddFunctionParser:    */
/*************************/
globle int AddFunctionParser(functionName,fpPtr)
  char *functionName;
  struct expr *(*fpPtr)();
  {
   struct FunctionDefinition *fdPtr;
   
   fdPtr = FindFunction(functionName);
   if (fdPtr == NULL)
     {
      PrintCLIPS(WERROR,"Function parsers can only be added for existing functions.\n");
      return(0);
     }

   fdPtr->parser = fpPtr;
   
   return(1);
  }
 
/****************************/
/* RemoveFunctionParser:    */
/****************************/
globle int RemoveFunctionParser(functionName)
  char *functionName;
  {
   struct FunctionDefinition *fdPtr;
   
   fdPtr = FindFunction(functionName);
   if (fdPtr == NULL)
     {
      PrintCLIPS(WERROR,"Function parsers can only be removed from existing functions.\n");
      return(0);
     }

   fdPtr->parser = NULL;
   
   return(1);
  }

/************************************************************/
/* ConstantExpression: Returns CLIPS_TRUE if the expression */
/*   is a constant, otherwise CLIPS_FALSE.                  */
/************************************************************/
globle BOOLEAN ConstantExpression(testPtr)
  struct expr *testPtr;
  {
   while (testPtr != NULL)
     {
      if ((testPtr->type != SYMBOL) && (testPtr->type != STRING) &&
#if OBJECT_SYSTEM
          (testPtr->type != INSTANCE_NAME) && (testPtr->type != INSTANCE) &&
#endif
          (testPtr->type != INTEGER) && (testPtr->type != FLOAT))
        { return(CLIPS_FALSE); }
      testPtr = testPtr->next_arg;
     }
     
   return(CLIPS_TRUE);
  }
 
/********************************************************************/
/* ExpressionInstall:  Increments all occurrences in the hash table */
/*   of symbols found in an expression composed of test structures. */ 
/********************************************************************/
globle VOID ExpressionInstall(expression)
  struct expr *expression;
  {
   if (expression == NULL) return;
   
   while (expression != NULL)
     {
#if DEFGENERIC_CONSTRUCT
      if (expression->type == GCALL)
        { ((GENERIC_FUNC *) expression->value)->busy++; }
      else
#endif
#if DEFFUNCTION_CONSTRUCT
      if (expression->type == PCALL)
        { ((struct dfunc *) expression->value)->busy++; }
      else
#endif
        { AtomInstall(expression->type,expression->value); }
      ExpressionInstall(expression->arg_list);
      expression = expression->next_arg;
     }
  }

/**********************************************************************/
/* ExpressionDeinstall:  Increments all occurrences in the hash table */
/*   of symbols found in an expression composed of test structures.   */ 
/**********************************************************************/
globle VOID ExpressionDeinstall(expression)
  struct expr *expression;
  {
   if (expression == NULL) return;
   
   while (expression != NULL)
     {
#if DEFGENERIC_CONSTRUCT
      if (expression->type == GCALL)
        { ((GENERIC_FUNC *) expression->value)->busy--; }
      else
#endif
#if DEFFUNCTION_CONSTRUCT
      if (expression->type == PCALL)
        { ((struct dfunc *) expression->value)->busy--; }
      else
#endif
        { AtomDeinstall(expression->type,expression->value); }
      ExpressionDeinstall(expression->arg_list);
      expression = expression->next_arg;
     }
  }
    
/****************************************************/
/* PackExpression:                                      */  
/****************************************************/
globle struct expr *PackExpression(original)
  struct expr *original;
  {
   struct expr *packPtr;
   
   if (original == NULL) return (NULL);
   packPtr = (struct expr *) gm3((long) sizeof (struct expr) * (long) ExpressionSize(original));
   ListToPacked(original,packPtr,0);
   return(packPtr);
  }
  
/****************************************************/
/* ListToPacked:                                    */  
/****************************************************/
globle int ListToPacked(original,destination,count)
  struct expr *original, *destination;
  int count;
  {
   long i;
   
   if (original == NULL) { return(count); }
   
   while (original != NULL)
     {
      i = count;
      count++;
   
      destination[i].type = original->type;
      destination[i].value = original->value;
   
      if (original->arg_list == NULL)
        { destination[i].arg_list = NULL; }
      else
        {
         destination[i].arg_list = &destination[(long) count];
         count = ListToPacked(original->arg_list,destination,count);
        }
      
      if (original->next_arg == NULL)
        { destination[i].next_arg = NULL; }
      else
        { destination[i].next_arg = &destination[(long) count]; }
        
      original = original->next_arg;
     }

   return(count);
  }
  
/****************************************************/
/* ReturnPackedExpression:                          */  
/****************************************************/
globle VOID ReturnPackedExpression(packPtr)
  struct expr *packPtr;
  {
   if (packPtr != NULL)
     { rm3(packPtr,(int) sizeof (struct expr) * ExpressionSize(packPtr)); }
  }
  
/*****************************************/
/* IdenticalExpression:                  */
/*****************************************/
globle BOOLEAN IdenticalExpression(check_elem,basic_list)
  struct expr *check_elem;
  struct expr *basic_list;
  {
   if ((check_elem == NULL) && (basic_list == NULL))
     { return(CLIPS_TRUE); }  

   if ((check_elem != NULL) && (basic_list == NULL))
     { return(CLIPS_FALSE); }

   if ((check_elem == NULL) && (basic_list != NULL))
     { return(CLIPS_FALSE); }

   if (check_elem->type != basic_list->type)
     { return(CLIPS_FALSE); }
     
   if (check_elem->value != basic_list->value)
    { return (CLIPS_FALSE); }

   if (IdenticalExpression(check_elem->arg_list,basic_list->arg_list) == CLIPS_FALSE)
     { return(CLIPS_FALSE); }

   if (IdenticalExpression(check_elem->next_arg,basic_list->next_arg) == CLIPS_FALSE)
     { return(CLIPS_FALSE); }

   return(CLIPS_TRUE);
  }

#endif

/****************************************************/
/* CountArguments: Returns the number of structures */
/*   stored in an expression as traversed through   */
/*   the next_arg pointer but not the arg_list      */
/*   pointer.                                       */
/****************************************************/
globle int CountArguments(testPtr)
  struct expr *testPtr;
  {
   int size = 0;
   
   while (testPtr != NULL)
     {
      size++;
      testPtr = testPtr->next_arg;
     }
     
   return(size);
  }
  
/****************************************************/
/* CopyExpresssion:                                 */  
/****************************************************/
globle struct expr *CopyExpression(original)
  struct expr *original;
  {
   struct expr *top_level, *next, *last;

   if (original == NULL) return(NULL);

   top_level = GenConstant(original->type,original->value);
   top_level->arg_list = CopyExpression(original->arg_list);

   last = top_level;
   original = original->next_arg;
   while (original != NULL)
     {
      next = GenConstant(original->type,original->value);
      next->arg_list = CopyExpression(original->arg_list);
      
      last->next_arg = next;
      last = next;
      original = original->next_arg;
     }

   return(top_level);
  }

/**********************************************************/
/* ExpressionContainsVariables:                           */
/**********************************************************/
globle BOOLEAN ExpressionContainsVariables(exp_ptr,globalsAreVariables)
  struct expr *exp_ptr;
  BOOLEAN globalsAreVariables;
  {  
   while (exp_ptr != NULL)
     {
      if ((exp_ptr->type == FCALL)
#if DEFGENERIC_CONSTRUCT
           || (exp_ptr->type == GCALL)
#endif
#if DEFFUNCTION_CONSTRUCT
           || (exp_ptr->type == PCALL)
#endif
         )
        {
         if (ExpressionContainsVariables(exp_ptr->arg_list,globalsAreVariables))
           { return(CLIPS_TRUE); }
        }
      else if ((exp_ptr->type == BWORDS) ||
               ((exp_ptr->type == GBWORD) && (globalsAreVariables == CLIPS_TRUE)) ||
               (exp_ptr->type == BWORD) ||
               (exp_ptr->type == FACT_ADDRESS))
        { return(CLIPS_TRUE); }

      exp_ptr = exp_ptr->next_arg;
     }

   return(CLIPS_FALSE);
  } 
  
/*******************************************************************/
/* ReturnExpression:  Returns a multiply linked list of test structures */
/*   to the list of free tests .                                   */
/*******************************************************************/
globle VOID ReturnExpression(waste)
  struct expr *waste;
  {
   register struct expr *tmp;
   
   while (waste != NULL)
     {
      if (waste->arg_list != NULL)
        ReturnExpression(waste->arg_list);
      tmp = waste;
      waste = waste->next_arg;
      rtn_struct(expr,tmp);
     }
  }

/****************************************************/
/* ExpressionSize: Returns the number of structures */
/*   stored in an expression.                       */
/****************************************************/
globle int ExpressionSize(testPtr)
  struct expr *testPtr;
  {
   int size = 0;
   
   while (testPtr != NULL)
     {
      size++;
      if (testPtr->arg_list != NULL)
        { size += ExpressionSize(testPtr->arg_list); }
      testPtr = testPtr->next_arg;
     }
   return(size);
  }
  
/*******************/
/* GetFunctionList */
/*******************/
globle struct FunctionDefinition *GetFunctionList()
  {
   return(ListOfFunctions);
  }

/*******************/
/* SetFunctionList */
/*******************/
globle VOID SetFunctionList(value)
  struct FunctionDefinition *value;
  {
   ListOfFunctions = value;
  }
  
/***********************/
/* InstallFunctionList */
/***********************/
globle VOID InstallFunctionList(value)
  struct FunctionDefinition *value;
  {
   int i;
   struct FunctionHash *fhPtr, *nextPtr;
   
   if (FunctionHashtable != NULL)
     {
      for (i = 0; i < SIZE_FUNCTION_HASH; i++) 
        {
         fhPtr = FunctionHashtable[i];
         while (fhPtr != NULL)
           {
            nextPtr = fhPtr->next;
            rtn_struct(FunctionHash,fhPtr);
            fhPtr = nextPtr;
           }
         FunctionHashtable[i] = NULL;
        }
     }
     
   ListOfFunctions = value;
   
   while (value != NULL)
     {
      AddHashFunction(value);
      value = value->next;
     }
  }
  
/***********************************************************/
/* FindFunction: Returns a pointer to the corresponding   */
/*   FunctionDefinition structure if a function name is in */
/*   the function list, otherwise returns NULL.            */
/***********************************************************/
globle struct FunctionDefinition *FindFunction(functionName)
  char *functionName;
  {
   struct FunctionHash *fhPtr;
   int hashValue;
   SYMBOL_HN *findValue;
   
   hashValue = HashSymbol(functionName,SIZE_FUNCTION_HASH);
   fhPtr = FunctionHashtable[hashValue];
   
   findValue = (SYMBOL_HN *) FindSymbol(functionName);
   
   while (fhPtr != NULL)
     {
      if (fhPtr->fdPtr->callFunctionName == findValue)
        { return(fhPtr->fdPtr); }
     
      fhPtr = fhPtr->next;
     }

   return(NULL);
  }
  
/************************************************************/
/* InitializeFunctionHashTable: Purpose is to initialize */
/*   the function hash table to NULL.                    */
/************************************************************/
static VOID InitializeFunctionHashTable()
  {
   int i;

   FunctionHashtable = (struct FunctionHash **) 
                       gm2((int) sizeof (struct FunctionHash *) * 
                           SIZE_FUNCTION_HASH);
      
   for (i = 0; i < SIZE_FUNCTION_HASH; i++) FunctionHashtable[i] = NULL;
  }
  
/*****************************************************************/
/* AddHashFunction:  Adds a function to the function hash table. */
/*****************************************************************/
globle VOID AddHashFunction(fdPtr)
  struct FunctionDefinition *fdPtr;
  {
   struct FunctionHash *newhash, *temp;
   int hashValue;

   if (FunctionHashtable == NULL) InitializeFunctionHashTable();
   
   newhash = get_struct(FunctionHash);
   newhash->fdPtr = fdPtr;

   hashValue = HashSymbol(fdPtr->callFunctionName->contents,SIZE_FUNCTION_HASH);
   
   temp = FunctionHashtable[hashValue];
   FunctionHashtable[hashValue] = newhash;
   newhash->next = temp;
  }

/**********************************************************************/
/* ParseConstantArguments: Parses a string into a set of constant     */
/* expressions.                                                       */
/**********************************************************************/
globle EXPRESSION *ParseConstantArguments(argstr,error)
  char *argstr;
  int *error;
  {
   EXPRESSION *top = NULL,*bot = NULL,*tmp;
   char *router = "***CLIPSFNXARGS***";
   struct token tkn;
   
   *error = FALSE;
   if (argstr != NULL)
     {
      if (OpenStringSource(router,argstr,0) == 0)
        {
         PrintCLIPS(WERROR,"Cannot read arguments for external call.\n");
         *error = TRUE;
         return(NULL);
        }
        
      GetToken(router,&tkn);
      while (tkn.type != STOP)
        {
         if ((tkn.type != SYMBOL) && (tkn.type != STRING) &&
             (tkn.type != FLOAT) && (tkn.type != INTEGER) &&
             (tkn.type != INSTANCE_NAME))
           {
            PrintCLIPS(WERROR,"Only constant arguments allowed for external CLIPS function call.\n");
            ReturnExpression(top);
            *error = TRUE;
            return(NULL);
           }
         tmp = GenConstant(tkn.type,tkn.value);
         if (top == NULL)
           top = tmp;
         else
           bot->next_arg = tmp;
         bot = tmp;
         GetToken(router,&tkn);
        }
      CloseStringSource(router);
     }
   return(top);
  }

/****************************************************/
/* PrintExpression: Pretty prints a test construct. */
/****************************************************/
globle VOID PrintExpression(fileid,test_ptr)
  char *fileid;
  struct expr *test_ptr;
  {

   if (test_ptr == NULL) 
     { return; }

   while (test_ptr != NULL)
     {
      switch (test_ptr->type)
        {
         case BWORD:
         case GBWORD:
            PrintCLIPS(fileid,"?");
            PrintCLIPS(fileid,ValueToString(test_ptr->value));
            break;
       
         case BWORDS:
            PrintCLIPS(fileid,"$?");
            PrintCLIPS(fileid,ValueToString(test_ptr->value));
            break;

         case FLOAT:
           PrintFloat(fileid,ValueToDouble(test_ptr->value));
           break;
           
         case INTEGER:
           PrintLongInteger(fileid,ValueToLong(test_ptr->value));
           break;
           
         case SYMBOL:
           PrintCLIPS(fileid,ValueToString(test_ptr->value));
           break;
           
         case STRING: 
           PrintCLIPS(fileid,"\"");
           PrintCLIPS(fileid,ValueToString(test_ptr->value)); 
           PrintCLIPS(fileid,"\"");
           break;
          
#if OBJECT_SYSTEM 
         case INSTANCE: 
           PrintAtom(fileid,INSTANCE,test_ptr->value);
           break;
         case INSTANCE_NAME: 
           PrintCLIPS(fileid,"[");
           PrintCLIPS(fileid,ValueToString(test_ptr->value)); 
           PrintCLIPS(fileid,"]");
           break;
#else
        case INSTANCE:
        case INSTANCE_NAME:
           PrintCLIPS(fileid,"(Bad instance reference)");
           break;
#endif
           
         case FCALL:
           PrintCLIPS(fileid,"(");
           PrintCLIPS(fileid,ValueToString(ExpressionFunctionCallName(test_ptr)));
           if (test_ptr->arg_list != NULL) { PrintCLIPS(fileid," "); }
           PrintExpression(fileid,test_ptr->arg_list);
           PrintCLIPS(fileid,")");
           break;
           
         case GCALL:
#if DEFGENERIC_CONSTRUCT
           PrintCLIPS(fileid,"(");
           PrintCLIPS(fileid,ValueToString(((GENERIC_FUNC *) test_ptr->value)->name));
           if (test_ptr->arg_list != NULL) { PrintCLIPS(fileid," "); }
           PrintExpression(fileid,test_ptr->arg_list);
           PrintCLIPS(fileid,")");
#else
           PrintCLIPS(fileid,"(Bad generic function call)");
#endif
           break;

         case PCALL:
#if DEFFUNCTION_CONSTRUCT
           PrintCLIPS(fileid,"(");
           PrintCLIPS(fileid,ValueToString(((struct dfunc *) test_ptr->value)->name));
           if (test_ptr->arg_list != NULL) { PrintCLIPS(fileid," "); }
           PrintExpression(fileid,test_ptr->arg_list);
           PrintCLIPS(fileid,")");
#else
           PrintCLIPS(fileid,"(Bad deffunction call)");
#endif
           break;

         default: 
           CLIPSSystemError("EXPRESSN",1);
           ExitCLIPS(5);
           break;
        }

      test_ptr = test_ptr->next_arg;
      if (test_ptr != NULL) PrintCLIPS(fileid," ");
     }

   return;
  }


