   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*               CLIPS Version 5.10  07/17/91          */
   /*                                                     */
   /*                  CLASS COMMANDS MODULE              */
   /*******************************************************/

/**************************************************************/
/* Purpose: CLIPS Kernel Interface Commands for Object System */
/*                                                            */
/* Principal Programmer(s):                                   */
/*      Brian L. Donnell                                      */
/*                                                            */
/* Contributing Programmer(s):                                */
/*                                                            */
/* Revision History:                                          */
/*                                                            */
/**************************************************************/
   
/* =========================================
   *****************************************
               EXTERNAL DEFINITIONS
   =========================================
   ***************************************** */
#include "setup.h"

#if OBJECT_SYSTEM

#if BLOAD || BLOAD_ONLY || BLOAD_AND_BSAVE
#include "bload.h"
#include "objbin.h"
#endif
#if CONSTRUCT_COMPILER && (! RUN_TIME)
#include "objcmp.h"
#endif
#include "classfun.h"
#include "clipsmem.h"
#include "constant.h"
#include "constrct.h"
#include "facts.h"
#include "inscom.h"
#include "insfun.h"
#include "insquery.h"
#include "msgcom.h"
#include "msgfun.h"
#include "router.h"
#include "scanner.h"
#include "parsutil.h"
#include "utility.h"

#define _CLASSCOM_SOURCE_
#include "classcom.h"

/* =========================================
   *****************************************
                   CONSTANTS
   =========================================
   ***************************************** */
#define SUPERCLASS_RLN     "is-a"
#define ABSTRACT_RLN       "abstract"
#define CONCRETE_RLN       "concrete"
#define HANDLER_DECL       "message-handler"
#define SLOT_RLN           "slot"
#define SLOT_DEF_RLN       "default"
#define SLOT_DEF_DYN_RLN   "default-dynamic"
#define SLOT_NOINH_RLN     "no-inherit"
#define SLOT_INH_RLN       "inherit"
#define SLOT_RDONLY_RLN    "read-only"
#define SLOT_RDWRT_RLN     "read-write"
#define SLOT_SHARE_RLN     "shared"
#define SLOT_LOCAL_RLN     "local"
#define SLOT_MULT_RLN      "multiple"
#define SLOT_SGL_RLN       "single"
#define SLOT_INIT_RLN      "initialize-only"
#define SLOT_COMPOSITE_RLN "composite"
#define SLOT_EXCLUSIVE_RLN "exclusive"

#define CARDINALITY_BIT 0
#define STORAGE_BIT     1
#define ACCESS_BIT      2
#define INHERIT_BIT     3
#define COMPOSITE_BIT   4

/* =========================================
   *****************************************
      INTERNALLY VISIBLE FUNCTION HEADERS
   =========================================
   ***************************************** */
#if ANSI_COMPILER

#if (! BLOAD_ONLY) && (! RUN_TIME)
static int ParseDefclass(char *);
static VOID PurgeUserClassStuff(void);
static SYMBOL_HN *ParseDefclassName(char *);
static CLASS_LINK *ParseSuperclasses(char *,SYMBOL_HN *);
static SLOT_DESC *ParseSlot(char *,SLOT_DESC *);
static EXPRESSION *ParseSlotValue(char *,int *);
static int EvaluateDefaultSlots(CLASS_TYPE *);
static VOID SaveDefclasses(char *);
#endif

static int CheckTwoClasses(char *,CLASS_TYPE **,CLASS_TYPE **);
static SYMBOL_HN *CheckClassAndSlot(char *,CLASS_TYPE **);
static char *GetClassName(char *);

#if DEBUGGING_FUNCTIONS
static CLASS_TYPE *CheckClass(char *,char *);
static int DisplayHandlersInLinks(CLASS_LINK *lk);
static VOID PrintClassBrowse(CLASS_TYPE *,int);
#endif

static VOID *ClassInfoFnxArgs(char *,DATA_OBJECT *,int *);
static int CountSubclasses(CLASS_TYPE *,int,int);
static int StoreSubclasses(VOID *,int,CLASS_TYPE *,int,int);

#else

#if (! BLOAD_ONLY) && (! RUN_TIME)
static int ParseDefclass();
static VOID PurgeUserClassStuff();
static SYMBOL_HN *ParseDefclassName();
static CLASS_LINK *ParseSuperclasses();
static SLOT_DESC *ParseSlot();
static EXPRESSION *ParseSlotValue();
static int EvaluateDefaultSlots();
static VOID SaveDefclasses();
#endif

static int CheckTwoClasses();
static SYMBOL_HN *CheckClassAndSlot();
static char *GetClassName();

#if DEBUGGING_FUNCTIONS
static CLASS_TYPE *CheckClass();
static int DisplayHandlersInLinks();
static VOID PrintClassBrowse();
#endif

static VOID *ClassInfoFnxArgs();
static int CountSubclasses();
static int StoreSubclasses();

#endif

#if (! BLOAD_ONLY) && (! RUN_TIME)
globle struct token ObjectParseToken;
#endif

/* =========================================
   *****************************************
      INTERNALLY VISIBLE GLOBAL VARIABLES
   =========================================
   ***************************************** */
    
/* =========================================
   *****************************************
          EXTERNALLY VISIBLE FUNCTIONS
   =========================================
   ***************************************** */
/**********************************************************
  NAME         : SetupObjectSystem
  DESCRIPTION  : Initializes all COOL constructs, functions,
                   and data structures
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : COOL initialized
  NOTES        : Order of setup calls is important
 **********************************************************/
globle VOID SetupObjectSystem()
  {
   SetupClasses();
   SetupInstances();
   SetupMessageHandlers();
   SetupDefinstances();
   SetupQuery();
#if BLOAD_AND_BSAVE || BLOAD || BLOAD_ONLY
   SetupObjectsBload();
#endif
#if CONSTRUCT_COMPILER && (! RUN_TIME)
   SetupObjectsCompiler();
#endif
  }

#if RUN_TIME
 
/***************************************************
  NAME         : ObjectsRunTimeInitialize
  DESCRIPTION  : Initializes objects system lists
                   in a run-time module
  INPUTS       : 1) Pointer to new class list
                 2) Pointer to new definstances list
                 3) Pointer to new class hash table
  RETURNS      : Nothing useful
  SIDE EFFECTS : Global pointers set
  NOTES        : None
 ***************************************************/
globle VOID ObjectsRunTimeInitialize(clist,dlist,ctable)
  CLASS_TYPE *clist,*ctable[];
  DEFINSTANCES *dlist;
  {   
   QUERY_DELIMETER_SYMBOL = FindSymbol(QUERY_DELIMETER_STRING);
   INIT_SYMBOL = FindSymbol(INIT_STRING);
   DELETE_SYMBOL = FindSymbol(DELETE_STRING);

   SetClassList(clist);
   SetDefinstancesList(dlist);
   
   ClassTable = (CLASS_TYPE **) ctable;
   PrimitiveClassMap[FLOAT] = (CLASS_TYPE *) FindDefclass("FLOAT");
   PrimitiveClassMap[INTEGER] = (CLASS_TYPE *) FindDefclass("INTEGER");
   PrimitiveClassMap[STRING] = (CLASS_TYPE *) FindDefclass("STRING");
   PrimitiveClassMap[SYMBOL] = (CLASS_TYPE *) FindDefclass("SYMBOL");
   PrimitiveClassMap[MULTIFIELD] = (CLASS_TYPE *) FindDefclass("MULTIFIELD");
   PrimitiveClassMap[EXTERNAL_ADDRESS] = 
                                (CLASS_TYPE *) FindDefclass("EXTERNAL-ADDRESS");
  }

#endif

/*********************************************************
  NAME         : SetupClasses
  DESCRIPTION  : Initializes Class Hash Table,
                   Function Parsers, and Data Structures
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS :
  NOTES        : Uses the CLIPS Kernel functions :
                   AddConstruct(),DefineFunction()
 *********************************************************/  
globle VOID SetupClasses()
  {
   AddClearFunction("instances",(VOID (*)()) InstancesPurge,-725);
#if ! RUN_TIME
   InitializeClasses();

#if ! BLOAD_ONLY
   AddConstruct("defclass",ParseDefclass);
   AddClearFunction("defclass",PurgeUserClassStuff,-750);
   DefineFunction("undefclass",'v',PTIF CmdUndefclass,"CmdUndefclass");
   AddSaveFunction("defclass",PTIF SaveDefclasses,10);
#endif

#if DEBUGGING_FUNCTIONS
   DefineFunction("list-defclasses",'v',PTIF CmdListDefclasses,"CmdListDefclasses");
   DefineFunction("ppdefclass",'v',PTIF PPDefclass,"PPDefclass");
   DefineFunction("describe-class",'v',PTIF DescribeClassCmd,"DescribeClassCmd");
   DefineFunction("browse-classes",'v',PTIF BrowseClassesCmd,"BrowseClassesCmd");
#endif   
   
   DefineFunction("superclassp",'b',PTIF IsSuperclass,"IsSuperclass");
   DefineFunction("subclassp",'b',PTIF IsSubclass,"IsSubclass");
   DefineFunction("class-existp",'b',PTIF DoesClassExist,"DoesClassExist");
   DefineFunction("class-slot-existp",'b',
                  PTIF ClassHasSlot,"ClassHasSlot");
   DefineFunction("class-message-handler-existp",'b',
                  PTIF ClassHasHandler,"ClassHasHandler");
   DefineFunction("class-abstractp",'b',PTIF IsClassAbstractCmd,"IsClassAbstractCmd");
   DefineFunction("class-slots",'m',PTIF ClassSlotsCmd,"ClassSlotsCmd");
   DefineFunction("class-superclasses",'m',
                  PTIF ClassSuperclassesCmd,"ClassSuperclassesCmd");
   DefineFunction("class-subclasses",'m',
                  PTIF ClassSubclassesCmd,"ClassSubclassesCmd");
   DefineFunction("class-message-handlers",'m',
                  PTIF ClassHandlersCmd,"ClassHandlersCmd");
   DefineFunction("slot-facets",'m',PTIF SlotFacetsCmd,"SlotFacetsCmd");
   DefineFunction("slot-sources",'m',PTIF SlotSourcesCmd,"SlotSourcesCmd");
#endif     
  }

/***************************************************
  NAME         : GetDefclassName
  DESCRIPTION  : Gets the name of a defclass
  INPUTS       : Pointer to a defclass
  RETURNS      : Name-string of defclass
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
globle char *GetDefclassName(ptr)
  VOID *ptr;
  {
   return(ValueToString(((CLASS_TYPE *) ptr)->name));
  }
  
/*******************************************************************
  NAME         : FindDefclass
  DESCRIPTION  : Looks up a specified class in the class hash table
  INPUTS       : The name-string of the class
  RETURNS      : The address of the found class, NULL otherwise
  SIDE EFFECTS : None
  NOTES        : None
 ******************************************************************/
globle VOID *FindDefclass(cname)
  char *cname;
  {
   SYMBOL_HN *csym;
   
   csym = FindSymbol(cname);
   if (csym == NULL)
     return(NULL);
   return((VOID *) FindDefclassBySymbol(csym));
  }
  
/***********************************************************
  NAME         : GetNextDefclass
  DESCRIPTION  : Finds first or next defclass
  INPUTS       : The address of the current defclass
  RETURNS      : The address of the next defclass
                   (NULL if none)
  SIDE EFFECTS : None
  NOTES        : If ptr == NULL, the first defclass
                    is returned.
 ***********************************************************/
globle VOID *GetNextDefclass(ptr)
  VOID *ptr;
  {
   if (ptr == NULL)
     return((VOID *) ClassList);
   return((VOID *) ((CLASS_TYPE *) ptr)->nxt_lst);
  }
  
/***************************************************
  NAME         : IsDefclassDeletable
  DESCRIPTION  : Determines if a defclass
                   can be deleted
  INPUTS       : Address of the defclass
  RETURNS      : TRUE if deletable, FALSE otherwise
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
globle int IsDefclassDeletable(ptr)
  VOID *ptr;
  {
#if BLOAD_ONLY || RUN_TIME
   return(FALSE);
#else
   CLASS_TYPE *cls;
   
#if BLOAD || BLOAD_AND_BSAVE
   if (Bloaded())
     return(FALSE);
#endif
   cls = (CLASS_TYPE *) ptr;
   if (cls->system == 1)
     return(FALSE);
   return((IsClassBeingUsed(cls) == FALSE) ? TRUE : FALSE);
#endif
  }

/*************************************************************
  NAME         : CmdUndefclass
  DESCRIPTION  : Deletes a class and its subclasses, as
                 well as their associated instances
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : Syntax : CLIPS> (undefclass <class-name> | *)
 *************************************************************/
globle VOID CmdUndefclass()
  {
   char *cname;
   SYMBOL_HN *csym;
   CLASS_TYPE *cls;

   cname = GetClassName("undefclass");
   if (cname == NULL)
      return;
   csym = FindSymbol(cname);
   cls = (csym != NULL) ? FindDefclassBySymbol(csym) : NULL;
   if ((cls == NULL) ? (strcmp(cname,"*") != 0) : FALSE)
     {
      ClassExistError("undefclass",cname);
      return;
     }
   DeleteDefclass((VOID *) cls);
  }
  
/*********************************************************************
  NAME         : IsSuperclass
  DESCRIPTION  : Determines if a class is a superclass of another
  INPUTS       : None
  RETURNS      : TRUE if class-1 is a superclass of class-2
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax : (superclassp <class-1> <class-2>)
 *********************************************************************/
globle BOOLEAN IsSuperclass()
  {
   CLASS_TYPE *c1,*c2;
   
   if (CheckTwoClasses("superclassp",&c1,&c2) == FALSE)
     return(FALSE);
   return(HasSuperclass(c2,c1));
  }
  
/*********************************************************************
  NAME         : IsSubclass
  DESCRIPTION  : Determines if a class is a subclass of another
  INPUTS       : None
  RETURNS      : TRUE if class-1 is a subclass of class-2
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax : (subclassp <class-1> <class-2>)
 *********************************************************************/
globle BOOLEAN IsSubclass()
  {
   CLASS_TYPE *c1,*c2;
   
   if (CheckTwoClasses("subclassp",&c1,&c2) == FALSE)
     return(FALSE);
   return(HasSuperclass(c1,c2));
  }
  
/*****************************************************************
  NAME         : ClassHasSlot
  DESCRIPTION  : Determines if a slot is present in a class
  INPUTS       : None
  RETURNS      : TRUE if the slot exists, FALSE otherwise
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax : (class-slot-existp <class> <slot>)
 *****************************************************************/
globle int ClassHasSlot()
  {
   CLASS_TYPE *cls;
   SYMBOL_HN *ssym;

   ssym = CheckClassAndSlot("class-slot-existp",&cls);
   if (ssym == NULL)
     return(FALSE);
   if (FindClassSlot(cls,ssym) != NULL)
     return(TRUE);
   return(FALSE);
  }
  
/************************************************************************************
  NAME         : ClassHasHandler
  DESCRIPTION  : Determines if a message-handler is present in a class
  INPUTS       : None
  RETURNS      : TRUE if the message header is present, FALSE otherwise
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax : (class-message-handler-existp <class> <hnd> [<type>])
 ************************************************************************************/
globle int ClassHasHandler()
  {
   int argcnt;
   CLASS_TYPE *cls;
   SYMBOL_HN *mname;
   DATA_OBJECT temp;
   unsigned mtype = MPRIMARY;
   
   argcnt = ArgRangeCheck("class-message-handler-existp",2,3);
   if (argcnt == -1)
     return(FALSE);
   if (ArgTypeCheck("class-message-handler-existp",1,SYMBOL,&temp) == FALSE)
     return(FALSE);
   cls = FindDefclassBySymbol((SYMBOL_HN *) ClipsGetValue(temp));
   if (cls == NULL)
     {
      ClassExistError("class-message-handler-existp",DOToString(temp));
      return(FALSE);
     }
   if (ArgTypeCheck("class-message-handler-existp",2,SYMBOL,&temp) == FALSE)
     return(FALSE);
   mname = (SYMBOL_HN *) ClipsGetValue(temp);
   if (argcnt == 3)
     {
      if (ArgTypeCheck("class-message-handler-existp",3,SYMBOL,&temp) == FALSE)
        return(FALSE);
      mtype = HandlerType("class-message-handler-existp",DOToString(temp));
      if (mtype == MERROR)
        {
         SetEvaluationError(TRUE);
         return(FALSE);
        }
     }
   if (FindHandler(cls,mname,mtype,LOOKUP_HANDLER_ADDRESS) != NULL)
     return(TRUE);
   if (mtype != MPRIMARY)
     return(FALSE);
   if (strncmp(GSM_PREFIX,ValueToString(mname),GSMP_LEN) == 0)
     mname = (SYMBOL_HN *) AddSymbol(ValueToString(mname)+GSMP_LEN);
   else if (strncmp(PSM_PREFIX,ValueToString(mname),PSMP_LEN) == 0)
     mname = (SYMBOL_HN *) AddSymbol(ValueToString(mname)+PSMP_LEN);
   else
     return(FALSE);
   if (FindClassSlot(cls,mname) != NULL)
     return(TRUE);
   return(FALSE);
  }
  
/********************************************************
  NAME         : DoesClassExist
  DESCRIPTION  : Determines if a class exists
  INPUTS       : None
  RETURNS      : TRUE if class exists, FALSE otherwise
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax : (class-existp <arg>)
 ********************************************************/
globle BOOLEAN DoesClassExist()
  {
   DATA_OBJECT temp;

   if (ArgCountCheck("class-existp",EXACTLY,1) != 1)
     return(FALSE);
   if (ArgTypeCheck("class-existp",1,SYMBOL,&temp) == FALSE)
     return(FALSE);
   return((FindDefclassBySymbol((SYMBOL_HN *) temp.value) != NULL) ? TRUE : FALSE);
  }
  
/*********************************************************************
  NAME         : IsClassAbstractCmd
  DESCRIPTION  : Determines if direct instances of a class can be made
  INPUTS       : None
  RETURNS      : TRUE (1) if class is abstract, FALSE (0) if concrete
  SIDE EFFECTS : None
  NOTES        : Syntax: (class-abstractp <class>)
 *********************************************************************/
globle int IsClassAbstractCmd()
  {
   DATA_OBJECT tmp;
   CLASS_TYPE *cls;
   
   if (ArgCountCheck("class-abstractp",EXACTLY,1) == -1)
     return(FALSE);
   if (ArgTypeCheck("class-abstractp",1,SYMBOL,&tmp) == FALSE)
     return(FALSE);
   cls = FindDefclassBySymbol((SYMBOL_HN *) tmp.value);
   if (cls == NULL)
     {
      ClassExistError("class-abstractp",ValueToString(tmp.value));
      return(FALSE);
     }
   return(IsClassAbstract((VOID *) cls));
  }
  
/********************************************************************
  NAME         : ClassSlotsCmd
  DESCRIPTION  : Groups slot info for a class into a multifield value
                   for dynamic perusal
  INPUTS       : Data object buffer to hold the slots of the class
  RETURNS      : Nothing useful
  SIDE EFFECTS : Creates a multifield storing the names of
                    the slots of the class
  NOTES        : Syntax: (class-slots <class> [inherit])
 ********************************************************************/
globle VOID ClassSlotsCmd(result)
  DATA_OBJECT *result;
  {
   int inhp;
   VOID *clsptr;
   
   clsptr = ClassInfoFnxArgs("class-slots",result,&inhp);
   if (clsptr == NULL)
     return;
   GetClassSlots(clsptr,result,inhp);
  }
  
/************************************************************************
  NAME         : ClassSuperclassesCmd
  DESCRIPTION  : Groups superclasses for a class into a multifield value
                   for dynamic perusal
  INPUTS       : Data object buffer to hold the superclasses of the class
  RETURNS      : Nothing useful
  SIDE EFFECTS : Creates a multifield storing the names of
                    the superclasses of the class
  NOTES        : Syntax: (class-superclasses <class> [inherit])
 ************************************************************************/
globle VOID ClassSuperclassesCmd(result)
  DATA_OBJECT *result;
  {
   int inhp;
   VOID *clsptr;
   
   clsptr = ClassInfoFnxArgs("class-superclasses",result,&inhp);
   if (clsptr == NULL)
     return;
   GetClassSuperclasses(clsptr,result,inhp);
  }
  
/************************************************************************
  NAME         : ClassSubclassesCmd
  DESCRIPTION  : Groups subclasses for a class into a multifield value
                   for dynamic perusal
  INPUTS       : Data object buffer to hold the subclasses of the class
  RETURNS      : Nothing useful
  SIDE EFFECTS : Creates a multifield storing the names of
                    the subclasses of the class
  NOTES        : Syntax: (class-subclasses <class> [inherit])
 ************************************************************************/
globle VOID ClassSubclassesCmd(result)
  DATA_OBJECT *result;
  {
   int inhp;
   VOID *clsptr;
   
   clsptr = ClassInfoFnxArgs("class-subclasses",result,&inhp);
   if (clsptr == NULL)
     return;
   GetClassSubclasses(clsptr,result,inhp);
  }
  
/*********************************************************************
  NAME         : ClassHandlersCmd
  DESCRIPTION  : Groups message-handlers for a class into a multifield
                   value for dynamic perusal
  INPUTS       : Data object buffer to hold the handlers of the class
  RETURNS      : Nothing useful
  SIDE EFFECTS : Creates a multifield storing the names of
                    the message-handlers of the class
  NOTES        : Syntax: (class-message-handlers <class> [inherit])
 *********************************************************************/
globle VOID ClassHandlersCmd(result)
  DATA_OBJECT *result;
  {
   int inhp;
   VOID *clsptr;
   
   clsptr = ClassInfoFnxArgs("class-message-handlers",result,&inhp);
   if (clsptr == NULL)
     return;
   GetClassMessageHandlers(clsptr,result,inhp);
  }
  
/*********************************************************************
  NAME         : SlotFacetsCmd
  DESCRIPTION  : Groups facets for a class slot into a multifield
                   value for dynamic perusal
  INPUTS       : Data object buffer to hold the facets of the slot
  RETURNS      : Nothing useful
  SIDE EFFECTS : Creates a multifield storing the facets of
                    the slot of the class
  NOTES        : Syntax: (slot-facets <class> <slot>)
 *********************************************************************/
globle VOID SlotFacetsCmd(result)
  DATA_OBJECT *result;
  {
   SYMBOL_HN *ssym;
   CLASS_TYPE *cls;
   
   result->type = SYMBOL;
   result->value = (VOID *) CLIPSFalseSymbol;
   ssym = CheckClassAndSlot("slot-facets",&cls);
   if (ssym == NULL)
     return;
   GetSlotFacets((VOID *) cls,ValueToString(ssym),result);
   if (result->type != MULTIFIELD)
     SlotExistError(ValueToString(ssym),"slot-facets");
  }
  
/*********************************************************************
  NAME         : SlotSourcesCmd
  DESCRIPTION  : Groups source classes for a slot into a multifield
                   value for dynamic perusal
  INPUTS       : Data object buffer to hold the sources of the slot
  RETURNS      : Nothing useful
  SIDE EFFECTS : Creates a multifield storing the sources of
                    the slot of the class
  NOTES        : Syntax: (slot-sources <class> <slot>)
 *********************************************************************/
globle VOID SlotSourcesCmd(result)
  DATA_OBJECT *result;
  {
   SYMBOL_HN *ssym;
   CLASS_TYPE *cls;
   
   result->type = SYMBOL;
   result->value = (VOID *) CLIPSFalseSymbol;
   ssym = CheckClassAndSlot("slot-sources",&cls);
   if (ssym == NULL)
     return;
   GetSlotSources((VOID *) cls,ValueToString(ssym),result);
   if (result->type != MULTIFIELD)
     SlotExistError(ValueToString(ssym),"slot-sources");
  }
  
#if DEBUGGING_FUNCTIONS

/*********************************************************
  NAME         : PPDefclass
  DESCRIPTION  : Displays the pretty print form of
                 a class to the wdialog router.
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : Syntax : CLIPS> (ppdefclass <class-name>)
 *********************************************************/
globle VOID PPDefclass()
  {
   char *cname;
   CLASS_TYPE *cls;

   cname = GetClassName("ppdefclass");
   if (cname == NULL)
     return;
   cls = CheckClass("ppdefclass",cname);
   if (cls == NULL)
     return;
   if (cls->pp_form != NULL)
     PrintCLIPS(WDISPLAY,cls->pp_form);
  }
  
/****************************************************************
  NAME         : DescribeClassCmd
  DESCRIPTION  : Displays direct superclasses and
                   subclasses and the entire precedence
                   list for a class
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : Syntax : CLIPS> (describe-class <class-name>)
 ****************************************************************/
globle VOID DescribeClassCmd()
  {
   char *cname;
   CLASS_TYPE *cls;

   cname = GetClassName("describe-class");   
   if (cname == NULL)
     return;
   cls = CheckClass("describe-class",cname);
   if (cls == NULL)
     return;
   DescribeClass((VOID *) cls);
  }
  
/****************************************************************
  NAME         : BrowseClasses
  DESCRIPTION  : Displays a "graph" of the class hierarchy
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : Syntax : CLIPS> (browse-classes [<class>])
 ****************************************************************/
globle VOID BrowseClassesCmd()
  {
   register CLASS_TYPE *cls;
   int acnt;
   
   acnt = ArgCountCheck("browse-classes",NO_MORE_THAN,1);
   if (acnt == -1)
     return;
   if (acnt == 0)
     {
      /* ================================================
         Find the OBJECT root class (has no superclasses)
         ================================================ */
      cls = ClassList;
      while ((cls != NULL) ? (cls->superlink != NULL) : FALSE)
        cls = cls->nxt_lst;
     }
   else
     {
      DATA_OBJECT tmp;
      
      if (ArgTypeCheck("browse-classes",1,SYMBOL,&tmp) == FALSE)
        return;
      cls = FindDefclassBySymbol((SYMBOL_HN *) tmp.value);
      if (cls == NULL)
        {
         ClassExistError("browse-classes",DOToString(tmp));
         return;
        }
     }
   BrowseClass((VOID *) cls);
  }
  
/*********************************************************
  NAME         : CmdListDefclasses
  DESCRIPTION  : Lists all classes
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : Syntax : CLIPS> (list-defclasses)
 *********************************************************/
globle VOID CmdListDefclasses()
  {
   if (ArgCountCheck("list-defclasses",EXACTLY,0) != 0)
     return;
   ListDefclasses();
  }

/***************************************************
  NAME         : GetDefclassPPForm
  DESCRIPTION  : Getsa defclass pretty print form
  INPUTS       : Address of the defclass
  RETURNS      : Pretty print form
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
globle char *GetDefclassPPForm(ptr)
  VOID *ptr;
  {
   return(((CLASS_TYPE *) ptr)->pp_form);
  }

/****************************************************************
  NAME         : DescribeClass
  DESCRIPTION  : Displays direct superclasses and
                   subclasses and the entire precedence
                   list for a class
  INPUTS       : Class pointer
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : None
 ****************************************************************/
globle VOID DescribeClass(clsptr)
  VOID *clsptr;
  {
   register CLASS_TYPE *cls;
   CLASS_LINK *stmp,*dtmp,pblk;
   register int i,j;
   char buf[82],fmt[12];
   SLOT_DESC *sp;
   int len,maxlen;
   DATA_OBJECT result;
            
   cls = (CLASS_TYPE *) clsptr;
   if (cls->abstract == 1)
     PrintCLIPS(WDISPLAY,"Abstract: intended for inheritance only.\n\n");
   PrintCLIPS(WDISPLAY,"Direct Superclasses:");
   for (stmp = cls->superlink ; stmp != NULL ; stmp = stmp->nxt)
     {
      PrintCLIPS(WDISPLAY," ");
      PrintCLIPS(WDISPLAY,ValueToString(stmp->cls->name));
     }
   PrintCLIPS(WDISPLAY,"\nInheritance Precedence: ");
   PrintCLIPS(WDISPLAY,ValueToString(cls->name));
   for (dtmp = cls->precedence ; dtmp != NULL ; dtmp = dtmp->nxt)
     {
      PrintCLIPS(WDISPLAY," ");
      PrintCLIPS(WDISPLAY,ValueToString(dtmp->cls->name));
     }
   PrintCLIPS(WDISPLAY,"\nDirect Subclasses:");
   for (stmp = cls->sublink ; stmp != NULL ; stmp = stmp->nxt)
     {
      PrintCLIPS(WDISPLAY," ");
      PrintCLIPS(WDISPLAY,ValueToString(stmp->cls->name));
     }
   PrintCLIPS(WDISPLAY,"\n");
   if (cls->itemplate != NULL)
     {
      maxlen = 5;
      for (i = 0 ; i < cls->islot_cnt ; i++)
        {
         len = strlen(ValueToString(cls->itemplate[i]->name));
         if (len > maxlen)
           maxlen = len;
         if (maxlen > 25)
           {
            maxlen = 25;
            break;
           }
        }
      sprintf(fmt,"%%-%d.%ds : ",maxlen,maxlen);
      sprintf(buf,fmt,"SLOTS");
      strcat(buf,"CRD DEF INH ACC STO TYP SOURCE(S)\n");
      PrintCLIPS(WDISPLAY,"\n");
      PrintCLIPS(WDISPLAY,buf);
      for (i = 0 ; i < cls->islot_cnt ; i++)
        {
         sp = cls->itemplate[i];
         sprintf(buf,fmt,ValueToString(sp->name));
         strcat(buf,(sp->multiple == 1) ? "MLT " : "SGL ");
         if (sp->default_specified == 0)
           strcat(buf,"NIL ");
         else
           strcat(buf,(sp->dynamic == 1) ? "DYN " : "STC ");
         strcat(buf,(sp->noinherit == 1) ? "NO  " : "YES ");
         if (sp->initonly == 1)
           strcat(buf,"INT ");
         else if (sp->nowrite == 1)
           strcat(buf," R  ");
         else
           strcat(buf,"RW  ");
         strcat(buf,(sp->shared == 1) ? "SHR " : "LCL ");
         strcat(buf,(sp->composite == 0) ? "EXC " : "CMP ");
         GetSlotSources(clsptr,ValueToString(sp->name),&result);
         PrintCLIPS(WDISPLAY,buf);
         for (j = GetDOBegin(result) ; j <= GetDOEnd(result) ; j++)
           {
             PrintCLIPS(WDISPLAY,ValueToString(GetMFValue(result.value,j)));
             PrintCLIPS(WDISPLAY," ");
            }
         PrintCLIPS(WDISPLAY,"\n");
        }
      PrintCLIPS(WDISPLAY,"\nNOTE: There are implicit get- and put- primary\n");
      PrintCLIPS(WDISPLAY,"        accessor message-handlers for every slot.\n");
     }
     
   pblk.cls = cls;
   pblk.nxt = cls->precedence;
   DisplayHandlersInLinks(&pblk);
  }

/****************************************************************
  NAME         : BrowseClass
  DESCRIPTION  : Displays a "graph" of the class hierarchy
  INPUTS       : Class pointer
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : None
 ****************************************************************/
globle VOID BrowseClass(clsptr)
  VOID *clsptr;
  {
   PrintClassBrowse((CLASS_TYPE *) clsptr,0);
  }
  
/*********************************************************
  NAME         : ListDefclasses
  DESCRIPTION  : Lists all classes
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : None
 *********************************************************/
globle VOID ListDefclasses()
  {
   register CLASS_TYPE *cls;
   long count = 0;
   
   for (cls = ClassList ; cls != NULL ; cls = cls->nxt_lst)
     {
      count++;
      PrintCLIPS(WDISPLAY,ValueToString(cls->name));
      PrintCLIPS(WDISPLAY,"\n");
     }
   PrintTally(WDISPLAY,count,"class","classes");
  }
  
#endif

/********************************************************************
  NAME         : IsClassAbstract
  DESCRIPTION  : Determines if a class is abstract or not
  INPUTS       : Generic pointe1 if class is abstract 0 otherwise
  SIDE EFFECTS : None
  NOTES        : None
 ********************************************************************/
globle int IsClassAbstract(clsptr)
  VOID *clsptr;
  {
   return(((CLASS_TYPE *) clsptr)->abstract);
  }
  
/********************************************************************
  NAME         : GetClassSlots
  DESCRIPTION  : Groups slot info for a class into a multifield value
                   for dynamic perusal
  INPUTS       : 1) Generic pointer to class
                 2) Data object buffer to hold the slots of the class
                 3) Include (1) or exclude (0) inherited slots
  RETURNS      : Nothing useful
  SIDE EFFECTS : Creates a multifield storing the names of
                    the slots of the class
  NOTES        : None
 ********************************************************************/
globle VOID GetClassSlots(clsptr,result,inhp)
  VOID *clsptr;
  DATA_OBJECT *result;
  int inhp;
  {
   int size;
   register CLASS_TYPE *cls;
   register int i;
   
   cls = (CLASS_TYPE *) clsptr;
   size = inhp ? cls->islot_cnt : cls->slot_cnt;
   result->type = MULTIFIELD;
   result->begin = 0;
   result->end = size - 1;
   result->value = (VOID *) CreateMultifield(size);
   if (size == 0)
     return;
   if (inhp)
     {
      for (i = 0 ; i < cls->islot_cnt ; i++)
        {
         SetMFType(result->value,i+1,SYMBOL);
         SetMFValue(result->value,i+1,cls->itemplate[i]->name);
        }
     }
   else
     {
      for (i = 0 ; i < cls->slot_cnt ; i++)
        {
         SetMFType(result->value,i+1,SYMBOL);
         SetMFValue(result->value,i+1,cls->slots[i].name);
        }
     }
  }
  
/************************************************************************
  NAME         : GetClassMessageHandlers
  DESCRIPTION  : Groups handler info for a class into a multifield value
                   for dynamic perusal
  INPUTS       : 1) Generic pointer to class
                 2) Data object buffer to hold the handlers of the class
                 3) Include (1) or exclude (0) inherited handlers
  RETURNS      : Nothing useful
  SIDE EFFECTS : Creates a multifield storing the names and types of
                    the message-handlers of the class
  NOTES        : None
 ************************************************************************/
globle VOID GetClassMessageHandlers(clsptr,result,inhp)
  VOID *clsptr;
  DATA_OBJECT *result;
  int inhp;
  {
   CLASS_LINK *cl,ctmp;
   register int i,j;
   int len,sublen;
   
   ctmp.cls = (CLASS_TYPE *) clsptr;
   ctmp.nxt = inhp ? ctmp.cls->precedence : NULL;
   for (cl = &ctmp , i = 0 ; cl != NULL ; cl = cl->nxt)
     i += cl->cls->handler_cnt;
   len = i * 3;
   result->type = MULTIFIELD;
   result->begin = 0;
   result->end = len - 1;
   result->value = (VOID *) CreateMultifield(len);
   if (len == 0)
     return;
   for (i = 1 ; i <= len ; i++)
     SetMFType(result->value,i,SYMBOL);
   for (cl = &ctmp , sublen = 0 ; cl != NULL ; cl = cl->nxt)
     {
      i = len - (cl->cls->handler_cnt * 3) - sublen + 1;
      for (j = 0 ; j < cl->cls->handler_cnt ; j++ , i += 3)
        {
         SetMFValue(result->value,i,cl->cls->name);
         SetMFValue(result->value,i+1,cl->cls->handlers[j].name);
         SetMFValue(result->value,i+2,AddSymbol(hndquals[cl->cls->handlers[j].type]));
        }
      sublen += cl->cls->handler_cnt * 3;
     }
  }
  
/***************************************************************************
  NAME         : GetClassSuperclasses
  DESCRIPTION  : Groups the names of superclasses into a multifield
                   value for dynamic perusal
  INPUTS       : 1) Generic pointer to class
                 2) Data object buffer to hold the superclasses of the class
                 3) Include (1) or exclude (0) indirect superclasses
  RETURNS      : Nothing useful
  SIDE EFFECTS : Creates a multifield storing the names of
                    the superclasses of the class
  NOTES        : None
 ***************************************************************************/
globle VOID GetClassSuperclasses(clsptr,result,inhp)
  VOID *clsptr;
  DATA_OBJECT *result;
  int inhp;
  {
   CLASS_LINK *cltop,*cl;
   register int i;
   
   cltop = inhp ? ((CLASS_TYPE *) clsptr)->precedence : 
                  ((CLASS_TYPE *) clsptr)->superlink;
   for (cl = cltop , i = 0 ; cl != NULL ; cl = cl->nxt)
     i++;
   result->type = MULTIFIELD;
   result->begin = 0;
   result->end = i - 1;
   result->value = (VOID *) CreateMultifield(i);
   if (i == 0)
     return;
   for (cl = cltop , i = 1 ; cl != NULL ; cl = cl->nxt , i++)
     {
      SetMFType(result->value,i,SYMBOL);
      SetMFValue(result->value,i,cl->cls->name);
     }
  }
  
/**************************************************************************
  NAME         : GetClassSubclasses
  DESCRIPTION  : Groups the names of subclasses for a class into a
                   multifield value for dynamic perusal
  INPUTS       : 1) Generic pointer to class
                 2) Data object buffer to hold the sublclasses of the class
                 3) Include (1) or exclude (0) indirect subclasses
  RETURNS      : Nothing useful
  SIDE EFFECTS : Creates a multifield storing the names
                    the subclasses of the class
  NOTES        : None
 **************************************************************************/
globle VOID GetClassSubclasses(clsptr,result,inhp)
  VOID *clsptr;
  DATA_OBJECT *result;
  int inhp;
  {
   register int i,id;
   
   if ((id = GetTraversalID()) == -1)
     return;
   i = CountSubclasses((CLASS_TYPE *) clsptr,inhp,id);
   ReleaseTraversalID();
   result->type = MULTIFIELD;
   result->begin = 0;
   result->end = i - 1;
   result->value = (VOID *) CreateMultifield(i);
   if (i == 0)
     return;
   if ((id = GetTraversalID()) == -1)
     return;
   StoreSubclasses(result->value,1,(CLASS_TYPE *) clsptr,inhp,id);
   ReleaseTraversalID();
  }

/**************************************************************************
  NAME         : GetSlotFacets
  DESCRIPTION  : Groups the facets of a slot for a class into a
                   multifield value for dynamic perusal
  INPUTS       : 1) Generic pointer to class
                 2) Name of the slot
                 3) Data object buffer to hold the sublclasses of the class
  RETURNS      : Nothing useful
  SIDE EFFECTS : Creates a multifield storing the facets and their values
                    for the slot of the class
  NOTES        : None
 **************************************************************************/
globle VOID GetSlotFacets(clsptr,sname,result)
  VOID *clsptr;
  char *sname;
  DATA_OBJECT *result;
  {
   SYMBOL_HN *ssym;
   register int i;
   register SLOT_DESC *sp;
   
   result->type = SYMBOL;
   result->value = (VOID *) CLIPSFalseSymbol;
   if ((ssym = FindSymbol(sname)) == NULL)
     {
      SetEvaluationError(TRUE);
      return;
     }
   i = FindInstanceTemplateSlot((CLASS_TYPE *) clsptr,ssym);
   if (i == -1)
     {
      SetEvaluationError(TRUE);
      return;
     }
   sp = ((CLASS_TYPE *) clsptr)->itemplate[i];
   result->type = MULTIFIELD;
   result->begin = 0;
   result->end = 5;
   result->value = (VOID *) CreateMultifield(6);
   for (i = 1 ; i <= 6 ; i++)
     SetMFType(result->value,i,SYMBOL);
   SetMFValue(result->value,1,AddSymbol(sp->multiple ? "MLT" : "SGL"));
   if (sp->default_specified == 1)
     SetMFValue(result->value,2,AddSymbol(sp->dynamic ? "DYN" : "STC"));
   else
     SetMFValue(result->value,2,AddSymbol("NIL"));
   SetMFValue(result->value,3,AddSymbol(sp->noinherit ? "NO" : "YES"));
   if (sp->initonly)
     SetMFValue(result->value,4,AddSymbol("INT"));
   else if (sp->nowrite)
     SetMFValue(result->value,4,AddSymbol("R"));
   else
     SetMFValue(result->value,4,AddSymbol("RW"));
   SetMFValue(result->value,5,AddSymbol(sp->shared ? "SHR" : "LCL"));
   SetMFValue(result->value,6,AddSymbol(sp->composite ? "CMP" : "EXC"));
  }

/**************************************************************************
  NAME         : GetSlotSources
  DESCRIPTION  : Groups the sources of a slot for a class into a
                   multifield value for dynamic perusal
  INPUTS       : 1) Generic pointer to class
                 2) Name of the slot
                 3) Data object buffer to hold the sources of the class
  RETURNS      : Nothing useful
  SIDE EFFECTS : Creates a multifield storing the sources for the slot
                   of the class
  NOTES        : None
 **************************************************************************/
globle VOID GetSlotSources(clsptr,sname,result)
  VOID *clsptr;
  char *sname;
  DATA_OBJECT *result;
  {
   SYMBOL_HN *ssym;
   register int i;
   register SLOT_DESC *sp,*csp;
   CLASS_LINK *cl,*ctop,*ctmp;
   
   result->type = SYMBOL;
   result->value = (VOID *) CLIPSFalseSymbol;
   if ((ssym = FindSymbol(sname)) == NULL)
     {
      SetEvaluationError(TRUE);
      return;
     }
   i = FindInstanceTemplateSlot((CLASS_TYPE *) clsptr,ssym);
   if (i == -1)
     {
      SetEvaluationError(TRUE);
      return;
     }
   sp = ((CLASS_TYPE *) clsptr)->itemplate[i];
   i = 1;
   ctop = get_struct(class_link);
   ctop->cls = (CLASS_TYPE *) sp->cls;
   ctop->nxt = NULL;
   if (sp->composite)
     {
      for (cl = sp->cls->precedence ; cl != NULL ; cl = cl->nxt)
        {
         csp = FindClassSlot(cl->cls,sp->name);
         if ((csp != NULL) ? (csp->noinherit == 0) : FALSE)
           {
            ctmp = get_struct(class_link);
            ctmp->cls = cl->cls;
            ctmp->nxt = ctop;
            ctop = ctmp;
            i++;
            if (csp->composite == 0)
              break;
           }
        }
     }
   result->type = MULTIFIELD;
   result->begin = 0;
   result->end = i - 1;
   result->value = (VOID *) CreateMultifield(i);
   for (ctmp = ctop , i = 1 ; ctmp != NULL ; ctmp = ctmp->nxt , i++)
     {
      SetMFType(result->value,i,SYMBOL);
      SetMFValue(result->value,i,ctmp->cls->name);
     }
   DeleteClassLinks(ctop);
  }

/*************************************************************
  NAME         : DeleteDefclass
  DESCRIPTION  : Deletes a class and its subclasses, as
                 well as their associated instances
  INPUTS       : Address of the class (NULL for all classes)
  RETURNS      : 1 if successful, 0 otherwise
  SIDE EFFECTS : None
  NOTES        : None
 *************************************************************/
globle int DeleteDefclass(vptr)
   VOID *vptr;
  {
   CLASS_TYPE *cls;

#if RUN_TIME || BLOAD_ONLY
   cls = (CLASS_TYPE *) vptr;
   PrintCLIPS(WERROR,"Unable to delete class ");
   PrintCLIPS(WERROR,(cls != NULL) ? ValueToString(cls->name) : "*");
   PrintCLIPS(WERROR,".\n");
   return(0);
#else
   cls = (CLASS_TYPE *) vptr;
#if BLOAD || BLOAD_AND_BSAVE
   if (Bloaded())
     {
      PrintCLIPS(WERROR,"Unable to delete class ");
      PrintCLIPS(WERROR,(cls != NULL) ? ValueToString(cls->name) : "*");
      PrintCLIPS(WERROR,".\n");
      return(0);
     }
#endif
   if (cls == NULL)
     return(ClearDefclasses(FALSE));
   if (cls->system == 1)
     {
      PrintCLIPS(WERROR,"Unable to delete class ");
      PrintCLIPS(WERROR,ValueToString(cls->name));
      PrintCLIPS(WERROR,".\n");
      return(0);
     }
   return(DeleteClassUAG(cls));
#endif
  }
  
/*****************************************************
  NAME         : HasSuperclass
  DESCRIPTION  : Determines if class-2 is a superclass
                   of class-1
  INPUTS       : 1) Class-1
                 2) Class-2
  RETURNS      : TRUE if class-2 is a superclass of
                   class-1, FALSE otherwise
  SIDE EFFECTS : None
  NOTES        : None
 *****************************************************/
globle int HasSuperclass(c1,c2)
  CLASS_TYPE *c1,*c2;
  {
   CLASS_LINK *cprec;
   
   for (cprec = c1->precedence ; cprec != NULL ; cprec = cprec->nxt)
     if (cprec->cls == c2)
       return(TRUE);
   return(FALSE);
  }
  
/* =========================================
   *****************************************
          INTERNALLY VISIBLE FUNCTIONS
   =========================================
   ***************************************** */

#if (! BLOAD_ONLY) && (! RUN_TIME)

/************************************************************************
  NAME         : ParseDefclass
  DESCRIPTION  : (defclass ...) is A CLIPS construct (as
                 opposed to a function), thus no variables
                 may be used.  This means classes may only
                 be STATICALLY defined (like rules).
  INPUTS       : The logical name of the router
                    for the parser input
  RETURNS      : FALSE if successful parse, TRUE otherwise
  SIDE EFFECTS : Inserts valid class definition into
                 Class Table.
  NOTES        : CLIPS Syntax :
  
                 (defclass <name> [<comment>]
                    (is-a <superclass-name>+)
                    <class-descriptor>*)
                
                 <class-descriptor> :== (slot <name> <slot-descriptor>*) |
                                        (abstract) | (concrete) |
                                        
                                        These are used only by CRSV:
                                        
                                        (message-handler <name> [<type>])
                                        
                 <slot-descriptor>  :== (default <expression>*) |
                                        (default-dynamic <expression>*) |
                                        (shared) | (local) |
                                        (read-only) | (read-write) | (initialize-only) |
                                        (multiple) | (single) |
                                        (no-inherit) | (inherit) |
                                        (composite) | (exclusive)
                                       
                                        These are used only by CRSV:
                                        
                                        (type ...) |
                                        (min-number-of-elements ...) |
                                        (max-number-of-elements ...) |
                                        (allowed-words ...) |
                                        (allowed-symbols ...) |
                                        (allowed-strings ...) |
                                        (allowed-numbers ...) |
                                        (allowed-integers ...) |
                                        (allowed-floats ...) |
                                        (allowed-values ...) |
                                        (range ...) |
                                        (allowed-instances ...)

 ***********************************************************************/
static int ParseDefclass(read_source)
  char *read_source;
  {
   SYMBOL_HN *cname;
   CLASS_TYPE *cls;
   CLASS_LINK *sclasses,*preclist;
   SLOT_DESC *slots = NULL;
   int abstract = FALSE,concrete_or_abstract = FALSE,parse_error;
   
   SetPPBufferStatus(ON);
   FlushPPBuffer();          
   SetIndentDepth(3);    
   SavePPBuffer("(defclass ");

#if BLOAD || BLOAD_ONLY || BLOAD_AND_BSAVE
   if (Bloaded())
     {
      PrintCLIPS(WERROR,"Cannot load classes with binary load in effect.\n");
      return(TRUE);
     }
#endif
   cname = ParseDefclassName(read_source);
   if (cname == NULL)
     {
      SyntaxErrorMessage("defclass name");
      return(TRUE);
     }
   cls = FindDefclassBySymbol(cname);
   if (GetPrintWhileLoading() == TRUE)
     {
      if (GetCompilationsWatch() == ON)
        {
         PrintCLIPS(WDIALOG,(cls != NULL) ? "Redefining" : "Defining");
         PrintCLIPS(WDIALOG," defclass block ");
         PrintCLIPS(WDIALOG,ValueToString(cname));
         if ((cls != NULL) ? (cls->handlers != NULL) : FALSE)
           PrintCLIPS(WDIALOG," (Message-handlers are unaffected)\n");
         else
           PrintCLIPS(WDIALOG,"\n");
        }
      else
        PrintCLIPS(WDIALOG,"#");
     }
   if (cls != NULL)
     {
      if (IsClassBeingUsed(cls))
        {
         PrintCLIPS(WERROR,ValueToString(cls->name));
         PrintCLIPS(WERROR," class cannot be redefined while\n");
         PrintCLIPS(WERROR,"    outstanding references to it still exist.\n");
         return(TRUE);
        }
     }
     
   sclasses = ParseSuperclasses(read_source,cname);
   if (sclasses == NULL)
     return(TRUE);
   preclist = FindPrecedenceList(cls,sclasses);
   if (preclist == NULL)
     {
      DeleteClassLinks(sclasses);
      return(TRUE);
     }
   parse_error = FALSE;
   GetToken(read_source,&ObjectParseToken);
   while (GetType(ObjectParseToken) != RPAREN)
     {
      if (GetType(ObjectParseToken) != LPAREN)
        {
         SyntaxErrorMessage("defclass");
         parse_error = TRUE;
         break;
        }
      PPBackup();
      PPCRAndIndent();
      SavePPBuffer("(");
      GetToken(read_source,&ObjectParseToken);
      if (GetType(ObjectParseToken) != SYMBOL)
        {
         SyntaxErrorMessage("defclass");
         parse_error = TRUE;
         break;
        }
      if (strcmp(DOToString(ObjectParseToken),SLOT_RLN) == 0)
        {
         SavePPBuffer(" ");
         slots = ParseSlot(read_source,slots);
         if (slots == NULL)
           {
            parse_error = TRUE;
            break;
           }
        }
      else if (strcmp(DOToString(ObjectParseToken),ABSTRACT_RLN) == 0)
        {
         if (concrete_or_abstract == TRUE)
           {
            PrintCLIPS(WERROR,"Class abstract/concrete role already declared.\n");
            parse_error = TRUE;
            break;
           }
         concrete_or_abstract = TRUE;
         abstract = TRUE;
         GetToken(read_source,&ObjectParseToken);
         if (GetType(ObjectParseToken) != RPAREN)
           {
            SyntaxErrorMessage("defclass");
            parse_error = TRUE;
            break;
           }
        }
      else if (strcmp(DOToString(ObjectParseToken),CONCRETE_RLN) == 0)
        {
         if (concrete_or_abstract == TRUE)
           {
            PrintCLIPS(WERROR,"Class abstract/concrete role already declared.\n");
            parse_error = TRUE;
            break;
           }
         concrete_or_abstract = TRUE;
         abstract = FALSE;
         GetToken(read_source,&ObjectParseToken);
         if (GetType(ObjectParseToken) != RPAREN)
           {
            SyntaxErrorMessage("defclass");
            parse_error = TRUE;
            break;
           }
        }
      else if (strcmp(DOToString(ObjectParseToken),HANDLER_DECL) == 0)
        {
         if (ReadUntilClosingParen(read_source,&ObjectParseToken) == 0)
           {
            parse_error = TRUE;
            break;
           }
        }
      else
        {
         SyntaxErrorMessage("defclass");
         parse_error = TRUE;
         break;
        }
      GetToken(read_source,&ObjectParseToken);
     }
   if ((GetType(ObjectParseToken) != RPAREN) || (parse_error == TRUE))
     {
      DeleteClassLinks(sclasses);
      DeleteClassLinks(preclist);
      DeleteSlots(slots);
      return(TRUE);
     }
   SavePPBuffer("\n");

   cls = NewClass();
   if (abstract == TRUE)
     cls->abstract = 1;
   cls->name = cname;
   cls->superlink = sclasses;
   cls->precedence = preclist;
   
   /* =================================
      Shove slots into contiguous array
      ================================= */
   if (slots != NULL)
     PackSlots(cls,slots);
     
   if (! AddClass(cls))
     return(TRUE);
   
   /* ================================================================
      For slots with static default values :
        Evaluate them now and store the value.  Determine the constant
          expression as well and store it.
      ================================================================ */
   cls->busy++;
   if (EvaluateDefaultSlots(cls) == FALSE)
     {
      cls->busy--;
      DeleteClassUAG(cls);
      return(TRUE);
     }
   cls->busy--;
   
#if DEBUGGING_FUNCTIONS
   if (GetConserveMemory() == FALSE)
     cls->pp_form = CopyPPBuffer();
#endif

   return(FALSE);
  }

/****************************************************
  NAME         : PurgeUserClassStuff
  DESCRIPTION  : Clears all user-defined classes as
                   well as user-defined message
                   handlers attached to system
                   classes
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : Classes, instances, handlers deleted
  NOTES        : None
 ****************************************************/
static VOID PurgeUserClassStuff()
  {
   ClearDefclasses(TRUE);
  }

/****************************************************
  NAME         : ParseDefclassName
  DESCRIPTION  : Finds the name of the defclass
  INPUTS       : The logical name of the input source
  RETURNS      : The address of the name - NULL if
                   thter was an error
  SIDE EFFECTS : The token after the name (and the
                 comment, if any) will already be
                 scanned when this function exits
  NOTES        : Uses CLIPS Scanner Function GetToken
 ****************************************************/
static SYMBOL_HN *ParseDefclassName(read_source)
  char *read_source;
  {
   SYMBOL_HN *cname;
   
   GetToken(read_source,&ObjectParseToken);
   if (GetType(ObjectParseToken) != SYMBOL)
     return(NULL);
   cname = (SYMBOL_HN *) ClipsGetValue(ObjectParseToken);
   if (IsSystemClassName(cname) == TRUE)
     {
      PrintCLIPS(WERROR,"Cannot redefine a predefined system class.\n");
      return(NULL);
     }

   /*================================*/
   /* Get comment too, if it exists. */
   /*================================*/

   GetToken(read_source,&ObjectParseToken);
   if (GetType(ObjectParseToken) == STRING)
     {
      PPBackup();
      SavePPBuffer(" ");
      SavePPBuffer(ObjectParseToken.print_rep);
      PPCRAndIndent();          
      GetToken(read_source,&ObjectParseToken);
     }
   else                                 
     {
      PPBackup();
      PPCRAndIndent();
      SavePPBuffer(ObjectParseToken.print_rep);
     }
   return(cname);
  }

 /**************************************************************
  NAME         : ParseSuperclasses
  DESCRIPTION  : Parses the (is-a <superclass>+) portion of
                 the (defclass ...) construct and returns
                 a list of direct superclasses.  The
                 class "standard-class" is the precedence list 
                 for classes with no direct superclasses.
                 The final precedence list (not calculated here)
                 will have the class in question first followed
                 by the merged precedence lists of its direct
                 superclasses.
  INPUTS       : 1) The logical name of the input source
                 2) The symbolic name of the new class
  RETURNS      : The address of the superclass list
                  or NULL if ther was an error
  SIDE EFFECTS : None
  NOTES        : Assumes "(defclass <name> [<comment>] (" 
                 has already been scanned.
                 
                 All superclasses must be defined before
                 their subclasses.  Duplicates in the (is-a
                 ...) list are are not allowed (a class may only
                 inherits from a superclass once).
                 
                 This routine also checks the class-precedence
                 lists of each of the direct superclasses for
                 an occurrence of the new class - i.e. cycles!
                 This can only happen when a class is redefined
                 (a new class cannot have an unspecified
                 superclass).
                 
                 This routine allocates the space for the list
 ***************************************************************/
static CLASS_LINK *ParseSuperclasses(read_source,new_class_name)
  char *read_source;
  SYMBOL_HN *new_class_name;
  {
   CLASS_LINK *clink = NULL,*cbot = NULL,*ctmp;
   CLASS_TYPE *sclass;
   
   if (GetType(ObjectParseToken) != LPAREN)
     {
      SyntaxErrorMessage("defclass inheritance");
      return(NULL);
     }
   GetToken(read_source,&ObjectParseToken);
   if ((GetType(ObjectParseToken) != SYMBOL) ? TRUE : 
       (strcmp(DOToString(ObjectParseToken),SUPERCLASS_RLN) != 0))
     {
      SyntaxErrorMessage("defclass inheritance");
      return(NULL);
     }
   SavePPBuffer(" ");
   GetToken(read_source,&ObjectParseToken);
   while (GetType(ObjectParseToken) != RPAREN)
     {
      if (GetType(ObjectParseToken) != SYMBOL)
        {
         DeleteClassLinks(clink);
         SyntaxErrorMessage("defclass");
         return(NULL);
        }
      if (ClipsGetValue(ObjectParseToken) == (VOID *) new_class_name)
        {
         DeleteClassLinks(clink);
         PrintCLIPS(WERROR,"A class may not have itself as a superclass.\n");
         return(NULL);
        }
      for (ctmp = clink ; ctmp != NULL ; ctmp = ctmp->nxt)
        {
         if (ClipsGetValue(ObjectParseToken) == (VOID *) ctmp->cls->name)
           {
            DeleteClassLinks(clink);
            PrintCLIPS(WERROR,"A class may inherit from a superclass only once.\n");
            return(NULL);
           }
        }
      sclass = FindDefclassBySymbol((SYMBOL_HN *) ClipsGetValue(ObjectParseToken));
      if (sclass == NULL)
        {
         DeleteClassLinks(clink);
         PrintCLIPS(WERROR,"A class must be defined after all its superclasses.\n");
         return(NULL);
        }
      ctmp = get_struct(class_link);
      ctmp->cls = sclass;
      if (clink == NULL)
        clink = ctmp;
      else
        cbot->nxt = ctmp;
      ctmp->nxt = NULL;
      cbot = ctmp;          
     
      SavePPBuffer(" ");
      GetToken(read_source,&ObjectParseToken);
     }
   if (clink == NULL)
     PrintCLIPS(WERROR,"Must have at least one superclass.\n");
   else
     {
      PPBackup();
      PPBackup();
      SavePPBuffer(")");
     }
   return(clink);
  }

/************************************************************
  NAME         : ParseSlot
  DESCRIPTION  : Parses slot definitions for a
                   defclass statement
  INPUTS       : 1) The logical name of the input source
                 2) The current slot list
  RETURNS      : The address of the list of slots,
                   NULL if there was an error
  SIDE EFFECTS : The slot list is allocated
  NOTES        : Assumes "(slot" has already been parsed.
 ************************************************************/
static SLOT_DESC *ParseSlot(read_source,slist)
  char *read_source;
  SLOT_DESC *slist;
  {
   SLOT_DESC *slot;
   char *flag_relation = NULL;
   int default_exp = FALSE,error;
   register unsigned qualbits = 0,tbit = 0;
   
   /* =================================================================
      Bits in qualflags are when slot qualifiers are specified so that
      duplicate or conflicting qualifiers can be detected.
      
      Single/multiple bit-0
      Shared/local bit-1
      Read-only/Read-write bit-2
      Inherit/No-inherit/Initialize-Only bit-3
      Composite/Exclusive bit-4
      
      See the bit macros at the top of this file
      ================================================================= */

   GetToken(read_source,&ObjectParseToken);
   if (GetType(ObjectParseToken) != SYMBOL)
     {
      DeleteSlots(slist);
      SyntaxErrorMessage("defclass slot");
      return(NULL);
     }
   slot = NewSlot((SYMBOL_HN *) ClipsGetValue(ObjectParseToken));
   slist = InsertSlot(slist,slot);
   if (slist == NULL)
     return(NULL);
   GetToken(read_source,&ObjectParseToken);
   IncrementIndentDepth(3);
   while (GetType(ObjectParseToken) == LPAREN)
     {
      PPBackup();
      PPCRAndIndent();
      SavePPBuffer("(");
      GetToken(read_source,&ObjectParseToken);
      if (GetType(ObjectParseToken) != SYMBOL)
        {
         SyntaxErrorMessage("defclass slot");
         DeleteSlots(slist);
         DecrementIndentDepth(3);
         return(NULL);
        }
      if (strcmp(DOToString(ObjectParseToken),SLOT_DEF_RLN) == 0)
        default_exp = TRUE;
      else if (strcmp(DOToString(ObjectParseToken),SLOT_DEF_DYN_RLN) == 0)
        {
         default_exp = TRUE;
         slot->dynamic = 1;
        }
      if (default_exp == TRUE)
        {
         if (slot->default_specified == 1)
           {
            PrintCLIPS(WERROR,"Only one default-value per slot allowed.\n");
            DeleteSlots(slist);
            DecrementIndentDepth(3);
            return(NULL);
           }
         slot->defexp = ParseSlotValue(read_source,&error);
         if (error == TRUE)
           {
            DeleteSlots(slist);
            DecrementIndentDepth(3);
            return(NULL);
           }
         slot->default_specified = 1;
         default_exp = FALSE;
        }
      else if (strcmp(DOToString(ObjectParseToken),SLOT_MULT_RLN) == 0)
        {
         tbit = CARDINALITY_BIT;
         slot->multiple = 1;
         slot->cardinality_specified = 1;
         flag_relation = SLOT_MULT_RLN;
        }
      else if (strcmp(DOToString(ObjectParseToken),SLOT_SGL_RLN) == 0)
        {
         tbit = CARDINALITY_BIT;
         slot->multiple = 0;
         slot->cardinality_specified = 1;
         flag_relation = SLOT_SGL_RLN;
        }
      else if (strcmp(DOToString(ObjectParseToken),SLOT_RDONLY_RLN) == 0)
        {
         tbit = ACCESS_BIT;
         slot->nowrite = 1;
         slot->access_specified = 1;
         flag_relation = SLOT_RDONLY_RLN;
        }
      else if (strcmp(DOToString(ObjectParseToken),SLOT_RDWRT_RLN) == 0)
        {
         tbit = ACCESS_BIT;
         slot->nowrite = 0;
         slot->access_specified = 1;
         flag_relation = SLOT_RDWRT_RLN;
        }
      else if (strcmp(DOToString(ObjectParseToken),SLOT_INIT_RLN) == 0)
        {
         tbit = ACCESS_BIT;
         slot->initonly = 1;
         slot->nowrite = 1;
         slot->access_specified = 1;
         flag_relation = SLOT_INIT_RLN;
        }
      else if (strcmp(DOToString(ObjectParseToken),SLOT_SHARE_RLN) == 0)
        {
         tbit = STORAGE_BIT;
         slot->shared = 1;
         slot->storage_specified = 1;
         flag_relation = SLOT_SHARE_RLN;
        }
      else if (strcmp(DOToString(ObjectParseToken),SLOT_LOCAL_RLN) == 0)
        {
         tbit = STORAGE_BIT;
         slot->shared = 0;
         slot->storage_specified = 1;
         flag_relation = SLOT_LOCAL_RLN;
        }
      else if (strcmp(DOToString(ObjectParseToken),SLOT_NOINH_RLN) == 0)
        {
         tbit = INHERIT_BIT;
         slot->noinherit = 1;
         flag_relation = SLOT_NOINH_RLN;
        }
      else if (strcmp(DOToString(ObjectParseToken),SLOT_INH_RLN) == 0)
        {
         tbit = INHERIT_BIT;
         slot->noinherit = 0;
         flag_relation = SLOT_INH_RLN;
        }
      else if (strcmp(DOToString(ObjectParseToken),SLOT_COMPOSITE_RLN) == 0)
        {
         tbit = COMPOSITE_BIT;
         slot->composite = 1;
         flag_relation = SLOT_COMPOSITE_RLN;
        }
      else if (strcmp(DOToString(ObjectParseToken),SLOT_EXCLUSIVE_RLN) == 0)
        {
         tbit = COMPOSITE_BIT;
         slot->composite = 0;
         flag_relation = SLOT_EXCLUSIVE_RLN;
        }
      else if ((strcmp(DOToString(ObjectParseToken),"type") == 0) ||
               (strcmp(DOToString(ObjectParseToken),"min-number-of-elements") == 0) ||
               (strcmp(DOToString(ObjectParseToken),"max-number-of-elements") == 0) ||
               (strcmp(DOToString(ObjectParseToken),"allowed-lexemes") == 0) ||
               (strcmp(DOToString(ObjectParseToken),"allowed-words") == 0) ||
               (strcmp(DOToString(ObjectParseToken),"allowed-symbols") == 0) ||
               (strcmp(DOToString(ObjectParseToken),"allowed-strings") == 0) ||
               (strcmp(DOToString(ObjectParseToken),"allowed-numbers") == 0) ||
               (strcmp(DOToString(ObjectParseToken),"allowed-floats") == 0) ||
               (strcmp(DOToString(ObjectParseToken),"allowed-integers") == 0) ||
               (strcmp(DOToString(ObjectParseToken),"allowed-instances") == 0) ||
               (strcmp(DOToString(ObjectParseToken),"allowed-values") == 0) ||
               (strcmp(DOToString(ObjectParseToken),"range") == 0))
        {
         if (ReadUntilClosingParen(read_source,&ObjectParseToken) == 0)
           {
            DeleteSlots(slist);
            DecrementIndentDepth(3);
            return(NULL);
           }
        }
      else
        {
         SyntaxErrorMessage("defclass slot");
         DeleteSlots(slist);
         DecrementIndentDepth(3);
         return(NULL);
        }
      if (flag_relation != NULL)
        {
         if (testbit(qualbits,tbit))
           {
            PrintCLIPS(WERROR,flag_relation);
            PrintCLIPS(WERROR," facet previously overridden or used.\n");
            DeleteSlots(slist);
            DecrementIndentDepth(3);
            return(NULL);
           }
         setbit(qualbits,tbit);
         GetToken(read_source,&ObjectParseToken);
         if (GetType(ObjectParseToken) != RPAREN)
           {
            SyntaxErrorMessage("defclass slot");
            DeleteSlots(slist);
            DecrementIndentDepth(3);
            return(NULL);
           }
         flag_relation = NULL;
        }
      GetToken(read_source,&ObjectParseToken);
     }
   if (GetType(ObjectParseToken) != RPAREN)
     {
      SyntaxErrorMessage("defclass slot");
      DeleteSlots(slist);
      DecrementIndentDepth(3);
      return(NULL);
     }
   if (slist == NULL)
     SyntaxErrorMessage("defclass slot");
   DecrementIndentDepth(3);
   if ((slot->dynamic == 0) && (slot->nowrite == 1) && (slot->initonly == 0))
     slot->shared = 1;
   return(slist);
  }

/***************************************************************************
  NAME         : ParseSlotValue
  DESCRIPTION  : Parses list of expressions for
                   slot value
  INPUTS       : 1) The logical name of the input source
                 2) Caller's error flag buffer
  RETURNS      : The address of the value list
  SIDE EFFECTS : Expression is allocated, caller's error flag
                  set on errors
  NOTES        : Uses the CLIPS Kernel function
                   CollectArguments() with a dummy test
                   structure - this function is normally used
                   to parse arguments for function calls
                   
                 The last thing scanned is the closing parenthesis
 ***************************************************************************/
static EXPRESSION *ParseSlotValue(read_source,error)
  char *read_source;
  int *error;
  {
   EXPRESSION *top,*val;
  
   *error = FALSE;
   top = get_struct(expr);
   top->arg_list = NULL;
   top->next_arg = NULL;
   if (CollectArguments(top,read_source) == NULL)
     {
      *error = TRUE;
      return(NULL);
     }
   if (ExpressionContainsVariables(top->arg_list,FALSE) == TRUE)
     {
      *error = TRUE;
      ReturnExpression(top);
      PrintCLIPS(WERROR,"Variables cannot be accessed by slot default-values.\n");
      return(NULL);
     }
   val = top->arg_list;
   top->arg_list = NULL;
   ReturnExpression(top);
   if (val == NULL)
     return(NULL);
   top = PackExpression(val);
   ReturnExpression(val);
   return(top);
  }
  
/*****************************************************
  NAME         : EvaluateDefaultSlots
  DESCRIPTION  : Evaluates "default" slot expressions
                   and replaces them with constant
                   expressions.  Also links slots
                   with classes.
  INPUTS       : The class
  RETURNS      : TRUE if all OK, FALSE otherwise
  SIDE EFFECTS : Non-constant default expressions are
                  deallocated and replaced
  NOTES        : The default expression is packed
 *****************************************************/
static int EvaluateDefaultSlots(cls)
  CLASS_TYPE *cls;
  {
   SLOT_DESC *slots;
   DATA_OBJECT temp;
   register int status,i;
   
   for (i = 0 ; i < cls->slot_cnt ; i++)
     {
      slots = &cls->slots[i];
      if ((slots->dynamic == 0) && (slots->defexp != NULL))
        {
         int oldce,mferror = FALSE;

         if (ConstantExpression(slots->defexp) == FALSE)
           {
            oldce = ExecutingConstruct();
            SetExecutingConstruct(TRUE);
            status = EvaluateAndStoreInDataObject(slots->multiple,slots->defexp,&temp);
            SetExecutingConstruct(oldce);
            ExpressionDeinstall(slots->defexp);
            ReturnPackedExpression(slots->defexp);
            slots->defexp = NULL;
            if (status == FALSE)
              return(FALSE);
            if (ValidSlotValue(&temp,"defclass") == FALSE)
              return(FALSE);
            if ((temp.type == MULTIFIELD) && (slots->multiple == 0))
              mferror = TRUE;
            else if (status == MULTI_SET)
              {
               slots->defexp = SlotValueExpression(&temp);
               ExpressionInstall(slots->defexp);
              }
           }
         else if ((slots->defexp->next_arg != NULL) && (slots->multiple == 0))
           {
            ExpressionDeinstall(slots->defexp);
            ReturnPackedExpression(slots->defexp);
            slots->defexp = NULL;
            mferror = TRUE;
           }
         if (mferror)
           {
            PrintCLIPS(WERROR,ValueToString(slots->name));
            PrintCLIPS(WERROR," in class ");
            PrintCLIPS(WERROR,ValueToString(cls->name));
            PrintCLIPS(WERROR," is not a multifield slot.\n");
            return(FALSE);
           }
        }
     }
   return(TRUE);
  } 

/***************************************************
  NAME         : SaveDefclasses
  DESCRIPTION  : Prints pretty print form of
                   defclasses to specified output
  INPUTS       : The  logical name of the output
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
static VOID SaveDefclasses(log_name)
  char *log_name;
  {
   CLASS_TYPE *cls;
   HANDLER *hnd;
   register int i;
   
   for (cls = ClassList ; cls != NULL ; cls = cls->nxt_lst)
     {
      if (cls->pp_form != NULL)
        {
         PrintInChunks(log_name,cls->pp_form);
         PrintCLIPS(log_name,"\n");
        }
      for (i = 0 ; i < cls->handler_cnt ; i++)
        {
         hnd = &cls->handlers[i];
         if (hnd->pp_form != NULL)
           {
            PrintInChunks(log_name,hnd->pp_form);
            PrintCLIPS(log_name,"\n");
           }
        }
     }
  }

#endif

/******************************************************
  NAME         : CheckTwoClasses
  DESCRIPTION  : Checks for exactly two class arguments
                    for a CLIPS function
  INPUTS       : 1) The function name
                 2) Caller's buffer for first class
                 3) Caller's buffer for second class
  RETURNS      : TRUE if both found, FALSE otherwise
  SIDE EFFECTS : Caller's buffers set
  NOTES        : None
 ******************************************************/
static int CheckTwoClasses(func,c1,c2)
  char *func;
  CLASS_TYPE **c1,**c2;
  {
   DATA_OBJECT temp;

   if (ArgCountCheck(func,EXACTLY,2) != 2)
     return(FALSE);
   if (ArgTypeCheck(func,1,SYMBOL,&temp) == FALSE)
     return(FALSE);
   *c1 = FindDefclassBySymbol((SYMBOL_HN *) temp.value);
   if (*c1 == NULL)
     {
      ClassExistError(func,ValueToString(temp.value));
      return(FALSE);
     }
   if (ArgTypeCheck(func,2,SYMBOL,&temp) == FALSE)
     return(FALSE);
   *c2 = FindDefclassBySymbol((SYMBOL_HN *) temp.value);
   if (*c2 == NULL)
     {
      ClassExistError(func,ValueToString(temp.value));
      return(FALSE);
     }
   return(TRUE);
  }

/********************************************************************
  NAME         : CheckClassAndSlot
  DESCRIPTION  : Checks class and slot argument for various functions
  INPUTS       : 1) Name of the calling function
                 2) Buffer for class address
  RETURNS      : Slot symbol, NULL on errors
  SIDE EFFECTS : None
  NOTES        : None
 ********************************************************************/
static SYMBOL_HN *CheckClassAndSlot(func,cls)
   char *func;
   CLASS_TYPE **cls;
  {
   DATA_OBJECT temp;
   
   if (ArgCountCheck(func,EXACTLY,2) != 2)
     return(NULL);
   if (ArgTypeCheck(func,1,SYMBOL,&temp) == FALSE)
     return(NULL);
   *cls = FindDefclassBySymbol((SYMBOL_HN *) ClipsGetValue(temp));
   if (*cls == NULL)
     {
      ClassExistError(func,DOToString(temp));
      return(NULL);
     }
   if (ArgTypeCheck(func,2,SYMBOL,&temp) == FALSE)
     return(NULL);
   return((SYMBOL_HN *) ClipsGetValue(temp));
  }
  
/*********************************************************
  NAME         : GetClassName
  DESCRIPTION  : Gets a class name-string and checks for
                   only one argument
  INPUTS       : Calling function name
  RETURNS      : Class name (NULL on errors)
  SIDE EFFECTS : None
  NOTES        : None
 *********************************************************/
static char *GetClassName(fname)
  char *fname;
  {
   DATA_OBJECT temp;

   if (ArgCountCheck(fname,EXACTLY,1) == -1)
     return(NULL);
   if (ArgTypeCheck(fname,1,SYMBOL,&temp) == FALSE)
     return(NULL);
   return(DOToString(temp));
  }

#if DEBUGGING_FUNCTIONS

/*****************************************************
  NAME         : CheckClass
  DESCRIPTION  : Used for to check class name for
                 CLIPS class accessor functions such
                 as ppdefclass and undefclass
  INPUTS       : 1) The name of the CLIPS function
                 2) Name of the class
  RETURNS      : The class address,
                   or NULL if ther was an error
  SIDE EFFECTS : None
  NOTES        : None
 ******************************************************/
static CLASS_TYPE *CheckClass(func,cname)
  char *func,*cname;
  {
   CLASS_TYPE *cls = NULL;
   SYMBOL_HN *csym;   

   csym = FindSymbol(cname);
   if (csym != NULL)
     cls = FindDefclassBySymbol(csym);
   if (cls == NULL)
     ClassExistError(func,cname);
   return(cls);
  }
  
/***************************************************
  NAME         : DisplayHandlersInLinks
  DESCRIPTION  : Recursively displays all handlers
                  for a list of linked classes
  INPUTS       : The top of the linked class list
  RETURNS      : TRUE if handlers printed, FALSE
                    otherwise
  SIDE EFFECTS : None
  NOTES        : Used by DescribeClass()
 ***************************************************/
static int DisplayHandlersInLinks(lk)
  CLASS_LINK *lk;
  {
   register int i,rtn;
   register HANDLER *hnd;
   
   rtn = (lk->nxt != NULL) ? DisplayHandlersInLinks(lk->nxt) : FALSE;
   if ((rtn == FALSE) && (lk->cls->handler_cnt != 0))
     {
      PrintCLIPS(WDISPLAY,"\nRecognized message-handlers:\n");
      rtn = TRUE;
     }
   for (i = 0 ; i < lk->cls->handler_cnt ; i++)
     {
      hnd = &lk->cls->handlers[i];
      PrintHandler(WDISPLAY,ValueToString(lk->cls->name),
                   ValueToString(hnd->name),hndquals[hnd->type]);
     }
   return(rtn);
  }
  
/****************************************************************
  NAME         : PrintClassBrowse
  DESCRIPTION  : Displays a "graph" of class and subclasses
  INPUTS       : 1) The class address
                 2) The depth of the graph
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : None
 ****************************************************************/
static VOID PrintClassBrowse(cls,depth)
  CLASS_TYPE *cls;
  int depth;
  {
   register int i;
   register CLASS_LINK *subs;
   
   for (i = 0 ; i < depth ; i++)
     PrintCLIPS(WDISPLAY,"  ");
   PrintCLIPS(WDISPLAY,ValueToString(cls->name));
   if ((cls->superlink != NULL) ? (cls->superlink->nxt != NULL) : FALSE)
     PrintCLIPS(WDISPLAY," *");
   PrintCLIPS(WDISPLAY,"\n");
   for (subs = cls->sublink ; subs != NULL ; subs = subs->nxt)
     PrintClassBrowse(subs->cls,depth+1);
  }
  
#endif
    
/*********************************************************
  NAME         : ClassInfoFnxArgs
  DESCRIPTION  : Examines arguments for:
                   class-slots, class-message-handlers,
                   class-superclasses and class-subclasses
  INPUTS       : 1) Name of function
                 2) Result buffer
  RETURNS      : Pointer to the class on success,
                   NULL on errors
  SIDE EFFECTS : inhp flag set, result buffer set
                   to symbol FALSE
  NOTES        : None
 *********************************************************/
static VOID *ClassInfoFnxArgs(fnx,result,inhp)
  char *fnx;
  DATA_OBJECT *result;
  int *inhp;
  {
   register int acnt;
   VOID *clsptr;
   DATA_OBJECT tmp;

   *inhp = 0;
   result->type = SYMBOL;
   result->value = CLIPSFalseSymbol;
   if ((acnt = ArgRangeCheck(fnx,1,2)) == -1)
     return(NULL);
   if (ArgTypeCheck(fnx,1,SYMBOL,&tmp) == FALSE)
     return(NULL);
   clsptr = (VOID *) FindDefclassBySymbol((SYMBOL_HN *) tmp.value);
   if (clsptr == NULL)
     {
      ClassExistError(fnx,ValueToString(tmp.value));
        return(NULL);
     }
   if (acnt == 2)
     {
      if (ArgTypeCheck(fnx,2,SYMBOL,&tmp) == FALSE)
        return(NULL);
      if (strcmp(ValueToString(tmp.value),"inherit") == 0)
        *inhp = 1;
      else
        {
         SyntaxErrorMessage(fnx);
           return(NULL);
        }
     }
   return(clsptr);
  }

/*****************************************************************
  NAME         : CountSubclasses
  DESCRIPTION  : Counts the number of direct or indirect
                   subclasses for a class
  INPUTS       : 1) Address of class
                 2) Include (1) or exclude (0) indirect subclasses
                 3) Traversal id
  RETURNS      : The number of subclasses
  SIDE EFFECTS : None
  NOTES        : None
 *****************************************************************/
static int CountSubclasses(cls,inhp,tvid)
  CLASS_TYPE *cls;
  int inhp,tvid;
  {
   register int cnt;
   register CLASS_LINK *cl;
   
   for (cnt = 0 , cl = cls->sublink ; cl != NULL ; cl = cl->nxt)
     {
      if (TestTraversalID(cl->cls->tvids,tvid) == 0)
        {
         cnt++;
         SetTraversalID(cl->cls->tvids,tvid);
         if (inhp && (cl->cls->sublink != NULL))
           cnt += CountSubclasses(cl->cls,inhp,tvid);
        }
     }
   return(cnt);
  }
  
/*********************************************************************
  NAME         : StoreSubclasses
  DESCRIPTION  : Stores the names of direct or indirect
                   subclasses for a class in a mutlifield
  INPUTS       : 1) Caller's multifield buffer
                 2) Starting index
                 3) Address of the class
                 4) Include (1) or exclude (0) indirect subclasses
                 5) Traversal id
  RETURNS      : The number of subclass names stored in the multifield
  SIDE EFFECTS : Multifield set with subclass names
  NOTES        : Assumes multifield is big enough to hold subclasses
 *********************************************************************/
static int StoreSubclasses(mfval,si,cls,inhp,tvid)
  VOID *mfval;
  int si;
  CLASS_TYPE *cls;
  int inhp,tvid;
  {
   register CLASS_LINK *cl;
   register int i;
   
   for (i = si , cl = cls->sublink ; cl != NULL ; cl = cl->nxt)
     {
      if (TestTraversalID(cl->cls->tvids,tvid) == 0)
        {
         SetTraversalID(cl->cls->tvids,tvid);
         SetMFType(mfval,i,SYMBOL);
         SetMFValue(mfval,i++,(VOID *) cl->cls->name);
         if (inhp && (cl->cls->sublink != NULL))
           i += StoreSubclasses(mfval,i,cl->cls,inhp,tvid);
        }
     }
   return(i - si);
  }
  
#endif
   
/***************************************************
  NAME         : 
  DESCRIPTION  : 
  INPUTS       : 
  RETURNS      : 
  SIDE EFFECTS : 
  NOTES        : 
 ***************************************************/
