   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*              CLIPS Version 5.10  07/17/91           */
   /*                                                     */
   /*                INSTANCE FUNCTIONS MODULE            */
   /*******************************************************/

/*************************************************************/
/* Purpose:  Internal instance manipulation routines         */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Brian L. Donnell                                     */
/*                                                           */
/* Contributing Programmer(s):                               */
/*                                                           */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/
   
/* =========================================
   *****************************************
               EXTERNAL DEFINITIONS
   =========================================
   ***************************************** */
#include "setup.h"

#if OBJECT_SYSTEM

#include "classfun.h"
#include "clipsmem.h"
#include "constant.h"
#include "engine.h"
#include "evaluatn.h"
#include "facts.h"
#include "generate.h"
#include "msgcom.h"
#include "msgfun.h"
#include "router.h"
#include "scanner.h"
#include "utility.h"

#define _INSFUN_SOURCE_
#include "insfun.h"

/* =========================================
   *****************************************
                   CONSTANTS
   =========================================
   ***************************************** */
#define BIG_PRIME                       11329
#define InstanceSizeHeuristic(ins)      sizeof(INSTANCE_TYPE)

/* =========================================
   *****************************************
               MACROS AND TYPES
   =========================================
   ***************************************** */
typedef struct igarbage
  {
   INSTANCE_TYPE *ins;
   struct igarbage *nxt;
  } IGARBAGE;
   
/* =========================================
   *****************************************
      INTERNALLY VISIBLE FUNCTION HEADERS
   =========================================
   ***************************************** */
#if ANSI_COMPILER
static int InsertSlotOverrides(INSTANCE_TYPE *,EXPRESSION *);
static INSTANCE_TYPE *InstanceLocationInfo(SYMBOL_HN *,INSTANCE_TYPE **,unsigned *);
static INSTANCE_TYPE *NewInstance(void);
static VOID BuildDefaultSlots(void);
static unsigned HashInstance(SYMBOL_HN *);
static VOID InstallInstance(INSTANCE_TYPE *,int);
static VOID StoreValuesInMultifield(DATA_OBJECT *,DATA_OBJECT *,int);
#else
static int InsertSlotOverrides();
static INSTANCE_TYPE *InstanceLocationInfo();
static INSTANCE_TYPE *NewInstance();
static VOID BuildDefaultSlots();
static unsigned HashInstance();
static VOID InstallInstance();
static VOID StoreValuesInMultifield();
#endif

/* =========================================
   *****************************************
      EXTERNALLY VISIBLE GLOBAL VARIABLES
   =========================================
   ***************************************** */
globle INSTANCE_TYPE *InstanceList = NULL;
#if DEBUGGING_FUNCTIONS
globle int WatchInstances = OFF,
globle     WatchSlots = OFF;
#endif
globle int WithinInit = FALSE;
globle int MaintainGarbageInstances = FALSE;
globle int OverrideSlotProtection = FALSE;
globle int ChangesToInstances = FALSE;

/* =========================================
   *****************************************
      INTERNALLY VISIBLE GLOBAL VARIABLES
   =========================================
   ***************************************** */
static INSTANCE_TYPE **InstanceTable = NULL;
static INSTANCE_TYPE *CurrentInstance = NULL;
static INSTANCE_TYPE *InstanceListBottom = NULL;
static IGARBAGE *InstanceGarbageList = NULL;

/* =========================================
   *****************************************
          EXTERNALLY VISIBLE FUNCTIONS
   =========================================
   ***************************************** */

/***************************************************
  NAME         : IncrementInstanceCount
  DESCRIPTION  : Increments instance busy count -
                   prevents it from being deleted
  INPUTS       : The address of the instance
  RETURNS      : Nothing useful
  SIDE EFFECTS : Count set
  NOTES        : None
 ***************************************************/
globle VOID IncrementInstanceCount(vptr)
  VOID *vptr;
  {
   ((INSTANCE_TYPE *) vptr)->busy++;
   ((INSTANCE_TYPE *) vptr)->cls->busy++;
  }
  
/***************************************************
  NAME         : DecrementInstanceCount
  DESCRIPTION  : Decrements instance busy count -
                   might allow it to be deleted
  INPUTS       : The address of the instance
  RETURNS      : Nothing useful
  SIDE EFFECTS : Count set
  NOTES        : None
 ***************************************************/
globle VOID DecrementInstanceCount(vptr)
  VOID *vptr;
  {
   ((INSTANCE_TYPE *) vptr)->busy--;
   ((INSTANCE_TYPE *) vptr)->cls->busy--;
  }
  
/***************************************************
  NAME         : DecrementInstanceDepth
  DESCRIPTION  : Makes the instance appear to have
                   been created a more shallow
                   evaluation depth than it was -
                   used for propagating return
                   values
  INPUTS       : The address of the instance
  RETURNS      : Nothing useful
  SIDE EFFECTS : Depth set
  NOTES        : None
 ***************************************************/
globle VOID DecrementInstanceDepth(vptr)
  VOID *vptr;
  {
   if (((INSTANCE_TYPE *) vptr)->depth > CurrentEvaluationDepth)
     ((INSTANCE_TYPE *) vptr)->depth = CurrentEvaluationDepth;
  }
  
/***************************************************
  NAME         : InitializeInstanceTable
  DESCRIPTION  : Initializes instance hash table
                  to all NULL addresses
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : Hash table initialized
  NOTES        : None
 ***************************************************/
globle VOID InitializeInstanceTable()
  {
   register int i;
   
   InstanceTable = (INSTANCE_TYPE **) 
                    gm2((int) (sizeof(INSTANCE_TYPE *) * INSTANCE_TABLE_HASH_SIZE));
   for (i = 0 ; i < INSTANCE_TABLE_HASH_SIZE ; i++)
     InstanceTable[i] = NULL;
  }

/*******************************************************
  NAME         : CleanupInstances
  DESCRIPTION  : Iterates through instance garbage
                   list looking for nodes that
                   have become unused - and purges
                   them
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : Non-busy instance garabge nodes deleted
  NOTES        : None
 *******************************************************/
globle VOID CleanupInstances()
  {
   IGARBAGE *gprv,*gtmp,*dump;
   
   if (MaintainGarbageInstances)
     return;
   gprv = NULL;
   gtmp = InstanceGarbageList;
   while (gtmp != NULL)
     {
      if ((gtmp->ins->busy == 0) && (gtmp->ins->depth > CurrentEvaluationDepth))
        {
         EphemeralItemCount--;
         EphemeralItemSize -= InstanceSizeHeuristic(gtmp->ins) + sizeof(IGARBAGE);
         rtn_struct(instance,gtmp->ins);
         if (gprv == NULL)
           InstanceGarbageList = gtmp->nxt;
         else
           gprv->nxt = gtmp->nxt;
         dump = gtmp;
         gtmp = gtmp->nxt;
         rtn_struct(igarbage,dump);
        }
      else
        {
         gprv = gtmp;
         gtmp = gtmp->nxt;
        }
     }
  }
  
/******************************************************************
  NAME         : InitializeInstance
  DESCRIPTION  : Called by EvaluateExpression to initialize
                   an instance of a class
  INPUTS       : The address of the result value
  RETURNS      : Nothing useful
  SIDE EFFECTS : Instance intialized
  NOTES        : CLIPS Syntax :
                  (initialize-instance <instance> <slot-overrides>*)
 *******************************************************************/
globle VOID InitializeInstance(result)
  DATA_OBJECT *result;
  {
   INSTANCE_TYPE *ins;
   DATA_OBJECT temp;
   
   SetpType(result,SYMBOL);
   SetpValue(result,(VOID *) CLIPSFalseSymbol);
   if (ArgCountCheck("initialize-instance",AT_LEAST,1) == -1)
     return;
   EvaluateExpression(GetFirstArgument(),&temp);
   if (temp.type == INSTANCE)
     {
      ins = (INSTANCE_TYPE *) temp.value;
      if (ins->garbage == 1)
        {
         StaleInstanceAddress("initialize-instance");
         return;
        }
     }
   else if ((temp.type == INSTANCE_NAME) || 
            (temp.type == SYMBOL))
     {
      ins = FindInstanceBySymbol((SYMBOL_HN *) temp.value);
      if (ins == NULL)
        {
         NoInstanceError(ValueToString(temp.value),"initialize-instance");
         return;
        }
     }
   else
     {
      PrintCLIPS(WERROR,"Expected a valid instance in function initialize-instance.\n");
      return;
     }
   if (CoreInitializeInstance(ins,GetFirstArgument()->next_arg) == TRUE)
     {
      SetpType(result,INSTANCE_NAME);
      SetpValue(result,(VOID *) ins->name);
     }
  }
  
/*****************************************************************************
  NAME         : MakeInstance
  DESCRIPTION  : Called by EvaluateExpression to initialize
                   an instance of a class
  INPUTS       : The address of the result value
  RETURNS      : Nothing useful
  SIDE EFFECTS : Instance intialized
  NOTES        : CLIPS Syntax :
                  (make-instance <instance-name> of <class> <slot-overrides>*)
 *****************************************************************************/
globle VOID MakeInstance(result)
  DATA_OBJECT *result;
  {
   SYMBOL_HN *iname;
   INSTANCE_TYPE *ins;
   DATA_OBJECT temp;
   
   SetpType(result,SYMBOL);
   SetpValue(result,(VOID *) CLIPSFalseSymbol);
   EvaluateExpression(GetFirstArgument(),&temp);
   if ((GetType(temp) != SYMBOL) && 
       (GetType(temp) != INSTANCE_NAME))
     {
      PrintCLIPS(WERROR,"Expected a valid name for new instance.\n");
      SetEvaluationError(TRUE);
      return;
     }
   iname = (SYMBOL_HN *) ClipsGetValue(temp);
   EvaluateExpression(GetFirstArgument()->next_arg,&temp);
   if (GetType(temp) != SYMBOL)
     {
      PrintCLIPS(WERROR,"Expected a valid class name for new instance.\n");
      SetEvaluationError(TRUE);
      return;
     }
   ins = BuildInstance(iname,(SYMBOL_HN *) ClipsGetValue(temp));
   if (ins == NULL)
     return;
   if (CoreInitializeInstance(ins,GetFirstArgument()->next_arg->next_arg) == TRUE)
     {
      SetpType(result,INSTANCE_NAME);
      SetpValue(result,(VOID *) ins->name);
     }
   else
     QuashInstance(ins);
  }
  
/***************************************************
  NAME         : BuildInstance
  DESCRIPTION  : Creates an uninitialized instance
  INPUTS       : 1) Name of the instance
                 2) Name of the class
  RETURNS      : The address of the new instance,
                   NULL on errors
  SIDE EFFECTS : Old definition (if any) is deleted
  NOTES        : None
 ***************************************************/
globle INSTANCE_TYPE *BuildInstance(iname,cname)
  SYMBOL_HN *iname,*cname;
  {
   INSTANCE_TYPE *ins,*iprv;
   CLASS_TYPE *cls;
   unsigned hashval;
   
   cls = FindDefclassBySymbol(cname);
   if (cls == NULL)
     {
      ClassExistError("make-instance",ValueToString(cname));
      return(NULL);
     }
   if (cls->abstract == 1)
     {
      PrintCLIPS(WERROR,"Cannot create instances of abstract class ");
      PrintCLIPS(WERROR,ValueToString(cname));
      PrintCLIPS(WERROR,".\n");
      SetEvaluationError(TRUE);
      return(NULL);
     }
   ins = InstanceLocationInfo(iname,&iprv,&hashval);
   if (ins != NULL)
     {
      if (ins->installed == 0)
        {
         PrintCLIPS(WERROR,"The instance ");
         PrintCLIPS(WERROR,ValueToString(iname));
         PrintCLIPS(WERROR," has a slot-value which depends on the instance definition.\n");
         SetEvaluationError(TRUE);
         return(NULL);
        }
      ins->busy++;
      IncrementSymbolCount(iname);
      if (ins->garbage == 0)
        DirectMessage(NULL,DELETE_SYMBOL,ins,NULL,NULL);
      ins->busy--;
      DecrementSymbolCount(iname);
      if (ins->garbage == 0)
        {
         PrintCLIPS(WERROR,"Unable to delete old instance ");
         PrintCLIPS(WERROR,ValueToString(iname));
         PrintCLIPS(WERROR,".\n");
         SetEvaluationError(TRUE);
         return(NULL);
        }
     }
     
   /* ============================================================= 
      Create the base instance from the defaults of the inheritance
      precedence list
      ============================================================= */
   CurrentInstance = NewInstance();
   CurrentInstance->name = iname;
   CurrentInstance->cls = cls;
   BuildDefaultSlots();
           
   /* ============================================================
      Put the instance in the instance hash table and put it on its
        class's instance list
      ============================================================ */
   CurrentInstance->hashval = hashval;
   if (iprv == NULL)
     {
      CurrentInstance->nxt_hsh = InstanceTable[hashval];
      if (InstanceTable[hashval] != NULL)
        InstanceTable[hashval]->prv_hsh = CurrentInstance;
      InstanceTable[hashval] = CurrentInstance;
     }
   else
     {
      CurrentInstance->nxt_hsh = iprv->nxt_hsh;
      if (iprv->nxt_hsh != NULL)
        iprv->nxt_hsh->prv_hsh = CurrentInstance;
      iprv->nxt_hsh = CurrentInstance;
      CurrentInstance->prv_hsh = iprv;
     }
   
   /* ======================================
      Put instance in global and class lists
      ====================================== */
   if (CurrentInstance->cls->instances == NULL)
     CurrentInstance->cls->instances = CurrentInstance;
   else
     {
      iprv = CurrentInstance->cls->instances;
      while (iprv->nxt_cls != NULL)
        iprv = iprv->nxt_cls;
      iprv->nxt_cls = CurrentInstance;
      CurrentInstance->prv_cls = iprv;
     }
   
   if (InstanceList == NULL)
     InstanceList = CurrentInstance;
   else
     InstanceListBottom->nxt_lst = CurrentInstance;
   CurrentInstance->prv_lst = InstanceListBottom;
   InstanceListBottom = CurrentInstance;
   ChangesToInstances = TRUE;
   
   /* ==============================================================================
      Install the instance's name and slot-value symbols (prevent them from becoming
      ephemeral) - the class name and slot names are accounted for by the class
      ============================================================================== */
   InstallInstance(CurrentInstance,TRUE);

   ins = CurrentInstance;
   CurrentInstance = NULL;
   return(ins);
  }

/*******************************************************************
  NAME         : CoreInitializeInstance
  DESCRIPTION  : Performs the core work for initializing an instance
  INPUTS       : 1) The instance address
                 2) Slot override expressions
  RETURNS      : TRUE if all OK, FALSE otherwise
  SIDE EFFECTS : EvaluationError set on errors - slots evaluated
  NOTES        : None
 *******************************************************************/
globle int CoreInitializeInstance(ins,ovrexp)
  INSTANCE_TYPE *ins;
  EXPRESSION *ovrexp;
  {
   register int i,svaccess;
   INSTANCE_SLOT *sp;
   DATA_OBJECT temp;
   
   if (ins->installed == 0)
     {
      PrintCLIPS(WERROR,"Instance ");
      PrintCLIPS(WERROR,ValueToString(ins->name));
      PrintCLIPS(WERROR," is already being initialized.\n");
      SetEvaluationError(TRUE);
      return(FALSE);
     }  
   
   /* =======================================
      Find all default slot-value expressions
      ======================================= */
   for (i = 0 ; i < ins->cls->islot_cnt ; i++)
     {
      sp = &(ins->slots[i]);
      if ((sp->desc->shared == 0) ? TRUE :
          ((*(sp->valaddr) == NULL) || (sp->desc->dynamic == 1)))
        sp->valexp = sp->desc->defexp;
      else
        sp->valexp = NULL;
      sp->desc->override = 0;
     }
     
   /* =======================================================
      Replace all default-slot values with any slot-overrides
      ======================================================= */
   ins->busy++;
   ins->installed = 0;
   if (InsertSlotOverrides(ins,ovrexp) == FALSE)
      {
       ins->installed = 1;
       ins->busy--;
       return(FALSE);
      }
      
   /* =================================================================
      Now that all the slot expressions are established - replace them
      with their evaluation
      
      If the slots are initialized properly - the init_eval flag will
      be turned off.
      ================================================================= */
   ins->init_eval = 1;
   svaccess = WithinInit;
   WithinInit = TRUE;
   DirectMessage(NULL,INIT_SYMBOL,ins,&temp,NULL);
   WithinInit = svaccess;
   ins->busy--;
   ins->installed = 1;
   if (EvaluationError)
     {
      PrintCLIPS(WERROR,"An error occurred during the initialization of instance ");
      PrintCLIPS(WERROR,ValueToString(ins->name));
      PrintCLIPS(WERROR,".\n");
      return(FALSE);
     }
   return((ins->init_eval == 1) ? FALSE : TRUE);
  }           

/******************************************************
  NAME         : QuashInstance
  DESCRIPTION  : Deletes an instance if it is not in
                   use, otherwise sticks it on the
                   garbage list
  INPUTS       : The instance
  RETURNS      : 1 if successful, 0 otherwise
  SIDE EFFECTS : Instance deleted or added to garbage
  NOTES        : Even though the instance is removed
                   from the class list, hash table and
                   instance list, its links remain
                   unchanged so that outside loops
                   can still determine where the next
                   node in the list is (assuming the
                   instance was garbage collected).
 ******************************************************/
globle int QuashInstance(ins)
  INSTANCE_TYPE *ins;
  {
   register int i,iflag;
   INSTANCE_SLOT *slot;
   IGARBAGE *gptr;
   
   if (ins->garbage == 1)
     return(0);
   if (ins->installed == 0)
     {
      PrintCLIPS(WERROR,"Cannot delete instance ");
      PrintCLIPS(WERROR,ValueToString(ins->name));
      PrintCLIPS(WERROR," during initialization.\n");
      SetEvaluationError(TRUE);
      return(0);
     }
#if DEBUGGING_FUNCTIONS
   if (WatchInstances == ON)
     {
      PrintCLIPS(WTRACE,"<== instance ");
      PrintCLIPS(WTRACE,ValueToString(ins->name));
      PrintCLIPS(WTRACE,"\n");
     }
#endif
   if (ins->prv_hsh != NULL)
     ins->prv_hsh->nxt_hsh = ins->nxt_hsh;
   else
     InstanceTable[ins->hashval] = ins->nxt_hsh;
   if (ins->nxt_hsh != NULL)
     ins->nxt_hsh->prv_hsh = ins->prv_hsh;

   if (ins->prv_cls != NULL)
     ins->prv_cls->nxt_cls = ins->nxt_cls;
   else
     ins->cls->instances = ins->nxt_cls;
   if (ins->nxt_cls != NULL)
     ins->nxt_cls->prv_cls = ins->prv_cls;

   if (ins->prv_lst != NULL)
     ins->prv_lst->nxt_lst = ins->nxt_lst;
   else
     InstanceList = ins->nxt_lst;
   if (ins->nxt_lst != NULL)
     ins->nxt_lst->prv_lst = ins->prv_lst;
   else
     InstanceListBottom = ins->prv_lst;
   iflag = ins->installed;
   InstallInstance(ins,FALSE);
   if (iflag == 1)
     {
      for (i = 0 ; i < ins->cls->islot_cnt ; i++)
        {
         slot = &(ins->slots[i]);
         if ((slot->desc->shared == 1) ? 
             (--slot->desc->share_cnt == 0) : TRUE)
           PutSlotValue(ins,slot,NULL,FALSE);
        }
     }
   if (ins->cls->islot_cnt != 0)
     rm((VOID *) ins->slots,
        (int) (ins->cls->islot_cnt * sizeof(INSTANCE_SLOT)));

   if ((ins->busy == 0) && (ins->depth > CurrentEvaluationDepth) &&
       (MaintainGarbageInstances == FALSE))
     rtn_struct(instance,ins);
   else
     {
      gptr = get_struct(igarbage);
      ins->garbage = 1;
      gptr->ins = ins;
      gptr->nxt = InstanceGarbageList;
      InstanceGarbageList = gptr;
      EphemeralItemCount++;
      EphemeralItemSize += InstanceSizeHeuristic(ins) + sizeof(IGARBAGE);
     }
   ChangesToInstances = TRUE;
   return(1);
  }
  
/***************************************************************************
  NAME         : FindInstanceBySymbol
  DESCRIPTION  : Looks up a specified instance in the instance hash table
  INPUTS       : 1) The CLIPS symbol for the name of the instance
  RETURNS      : The address of the found instance, NULL otherwise
  SIDE EFFECTS : None
  NOTES        : Conflict resolution is done by chaining
 ***************************************************************************/
globle INSTANCE_TYPE *FindInstanceBySymbol(iname)
  SYMBOL_HN *iname;
  {
   INSTANCE_TYPE *ins;
   
   ins = InstanceTable[HashInstance(iname)];
   while (ins != NULL)
     {
      if (ins->name == iname)
        return(ins);
      ins = ins->nxt_hsh;
     }
   return(ins);
  }
  
/********************************************************************
  NAME         : FindInstanceSlot
  DESCRIPTION  : Performs a binary search on an instance's slot array
                   to find a slot by name
  INPUTS       : 1) The address of the instance
                 2) The CLIPS symbolic name of the slot
  RETURNS      : The address of the slot, NULL if not found
  SIDE EFFECTS : None
  NOTES        : None
 ********************************************************************/
globle INSTANCE_SLOT *FindInstanceSlot(ins,sname)
  INSTANCE_TYPE *ins;
  SYMBOL_HN *sname;
  {
   register int i;
   
   i = FindInstanceTemplateSlot(ins->cls,sname);
   return((i != -1) ? &ins->slots[i] : NULL);
  }
  
/********************************************************************
  NAME         : FindInstanceTemplateSlot
  DESCRIPTION  : Performs a binary search on an class's instance
                   template slot array to find a slot by name
  INPUTS       : 1) The address of the class
                 2) The CLIPS symbolic name of the slot
  RETURNS      : The index of the slot, -1 if not found
  SIDE EFFECTS : None
  NOTES        : The class contains an instance template sorted
                   by the hash values (buckets) of the slot names.
                   Each element in this sorted template contains an
                   offset into the "inherited" order template - the
                   order in which the slots are actually stored in
                   the instance.
 ********************************************************************/
globle int FindInstanceTemplateSlot(cls,sname)
  CLASS_TYPE *cls;
  SYMBOL_HN *sname;
  {
   register int b,e,i,j;
   unsigned *arr;
   SLOT_DESC **tmpl;
   
   if (cls->islot_cnt == 0)
     return(-1);
   arr = cls->sordered;
   tmpl = cls->itemplate;
   b = 0;
   e = cls->islot_cnt-1;
   do
     {
      i = (b+e)/2;
      if (sname->bucket == tmpl[arr[i]]->name->bucket)
        {
         for (j = i ; j >= b ; j--)
           {
            if (sname->bucket != tmpl[arr[j]]->name->bucket)
              break;
            if (sname == tmpl[arr[j]]->name)
              return(arr[j]);
           }
         for (j = i+1 ; j <= e ; j++)
           {
            if (sname->bucket != tmpl[arr[j]]->name->bucket)
              break;
            if (sname == tmpl[arr[j]]->name)
              return(arr[j]);
           }
         return(-1);
        }
      else if (sname->bucket < tmpl[arr[i]]->name->bucket)
        e = i-1;
      else
        b = i+1;
     }
   while (b <= e);
   return(-1);
  }

/*********************************************************
  NAME         : EvaluateAndStoreInDataObject
  DESCRIPTION  : Evaluates slot-value expressions
                   and stores the result in a CLIPS
                   Kernel data object
  INPUTS       : 1) Flag indicating if multifields are OK
                 2) The value-expression
                 3) The data object structure
  RETURNS      : 1) FALSE (0) on errors
                 2) MULTI_CLEAR (1) if the vexp is
                    a multi-field of length 0
                 3) MULTI_SET (2) otherwise
  SIDE EFFECTS : Bogus fact structure allocated
                   for storing multi-field values
  NOTES        : None
 *********************************************************/
globle int EvaluateAndStoreInDataObject(mfp,exp,val)
  int mfp;
  EXPRESSION *exp;
  DATA_OBJECT *val;
  {
   DATA_OBJECT *head = NULL,*bot = NULL,*dval;
   int valcnt = 0;
   
   val->type = MULTIFIELD;
   val->begin = 0;
   val->end = -1;
   if (exp == NULL)
     return(MULTI_CLEAR);
   if ((mfp == 0) && (exp->next_arg == NULL))
     {
      if (EvaluateExpression(exp,val) == TRUE)
        return(FALSE);
      return(MULTI_SET);
     }
   while (exp != NULL)
     {
      dval = get_struct(dataObject);
      if (EvaluateExpression(exp,dval) == TRUE)
        {
         rtn_struct(dataObject,dval);
         while (head != NULL)
           {
            dval = head;
            head = head->next;
            rtn_struct(dataObject,dval);
           }
         return(FALSE);
        }
      dval->next = NULL;
      if (GetpType(dval) == MULTIFIELD)
        {
         if (GetpDOLength(dval) < 1)
           rtn_struct(dataObject,dval);
         else
          {
           if (head == NULL)
             head = dval;
           else
             bot->next = dval;
           bot = dval;
           valcnt += GetpDOLength(dval);
          }
        }
      else
        {
         if (head == NULL)
           head = dval;
         else
           bot->next = dval;
         bot = dval;
         valcnt++;
        }
      exp = GetNextArgument(exp);
     }
   if (valcnt == 0)
     val->value = (VOID *) CreateMultifield(0);
   else
     StoreValuesInMultifield(head,val,valcnt);
   return(MULTI_SET);
  }
  
/****************************************************
  NAME         : SlotValueExpression
  DESCRIPTION  : Forms a linked list of constant
                   expressions equivalent to a slot's
                   multi-field value
  INPUTS       : The slot value
  RETURNS      : The (packed) expression list
  SIDE EFFECTS : New expression nodes are allocated
  NOTES        : None
 ****************************************************/
globle EXPRESSION *SlotValueExpression(val)
  DATA_OBJECT *val;
  {
   EXPRESSION *exp = NULL;
   int i,j;
   
   if (val->type == MULTIFIELD)
     {
      if (GetpDOLength(val) > 0)
        {
         exp = (EXPRESSION *) gm2((int) (sizeof(EXPRESSION) * GetpDOLength(val)));
         for (i = GetpDOBegin(val) , j = 0 ; i <= GetpDOEnd(val) ; i++ , j++)
           {
            SetType(exp[j],GetMFType(GetpValue(val),i));
            ClipsSetValue(exp[j],GetMFValue(GetpValue(val),i));
            exp[j].arg_list = NULL;
            if (i == GetpDOEnd(val))
              exp[j].next_arg = NULL;
            else
              exp[j].next_arg = &(exp[j+1]);
           }
        }
     }
   else
     exp = GenConstant(val->type,val->value);
   return(exp);
  }
  
/*******************************************************
  NAME         : PutSlotValue
  DESCRIPTION  : Evaluates new slot-expression and
                   stores it as a multifield
                   variable for the slot.
  INPUTS       : 1) The address of the instance
                    (NULL if no trace-messages desired)
                 2) The address of the slot
                 3) The address of the value
                 4) OK to print trace messages or not
  RETURNS      : SLOT_ERROR, SLOT_EMPTY or SLOT_FILLED
  SIDE EFFECTS : Old value deleted and new one allocated
                 Old value symbols deinstalled
                 New value symbols installed
  NOTES        : None
 *******************************************************/
globle int PutSlotValue(ins,sp,val,trace)
  INSTANCE_TYPE *ins;
  INSTANCE_SLOT *sp;
  DATA_OBJECT *val;
  int trace;
  {
   register DATA_OBJECT *vptr;
   register int status = SLOT_FILLED;
   
   vptr = *(sp->valaddr);
   if (sp->desc->multiple == 0)
     {
      if (val == NULL)
        {
         if (vptr != NULL)
           {
            ValueDeinstall(vptr);
            rtn_struct(dataObject,vptr);
            *(sp->valaddr) = NULL;
           }
         status = SLOT_EMPTY;
        }
      else if (val->type == MULTIFIELD)
        {
         PrintCLIPS(WERROR,ValueToString(sp->desc->name));
         PrintCLIPS(WERROR," in instance ");
         PrintCLIPS(WERROR,ValueToString(ins->name));
         PrintCLIPS(WERROR," is not a multifield slot.\n");
         SetEvaluationError(TRUE);
         return(SLOT_ERROR);
        }
      else
        {
         if (vptr == NULL)
           vptr = *(sp->valaddr) = get_struct(dataObject);
         else
           ValueDeinstall(vptr);
         vptr->type = val->type;
         vptr->value = val->value;
         ValueInstall(vptr);
        }
     }
   else
     {
      if (val == NULL)
        {
         if (vptr != NULL)
           {
            SegmentDeinstall(GetpValue(vptr));
            ReturnElements(GetpValue(vptr));
            rtn_struct(dataObject,vptr);
            *(sp->valaddr) = NULL;
           }
         status = SLOT_EMPTY;
        }
      else
        {
         if (vptr == NULL)
           {
            vptr = *(sp->valaddr) = get_struct(dataObject);
            vptr->type = MULTIFIELD;
           }
         else
           {
            SegmentDeinstall(GetpValue(vptr));
            ReturnElements(GetpValue(vptr));
           }
         if (val->type == MULTIFIELD)
           DuplicateSegment(vptr,val);
         else
           {
            vptr->begin = 0;
            vptr->end = 0;
            vptr->value = (VOID *) CreateFact(1);
            ((SEGMENT) vptr->value)->atoms[0].type = val->type;
            ((SEGMENT) vptr->value)->atoms[0].value = val->value;
           }
         SegmentInstall(vptr->value);
        }
     }
#if DEBUGGING_FUNCTIONS
   if ((WatchSlots == ON) ? (trace == TRUE) : FALSE)
     {
      if (sp->desc->shared)
        PrintCLIPS(WTRACE,"::= shared slot ");
      else
        PrintCLIPS(WTRACE,"::= local slot ");
      PrintCLIPS(WTRACE,ValueToString(sp->desc->name));
      PrintCLIPS(WTRACE," in instance ");
      PrintCLIPS(WTRACE,ValueToString(ins->name));
      PrintCLIPS(WTRACE," <- ");
      if (*(sp->valaddr) != NULL)
        {
         PrintDataObject(WTRACE,*(sp->valaddr));
         PrintCLIPS(WTRACE,"\n");
        }
      else
        PrintCLIPS(WTRACE,"NIL\n");
     }
#endif
   ChangesToInstances = TRUE;
   return(status);
  }

/*****************************************************************************
  NAME         : EvaluateInstanceSlots
  DESCRIPTION  : Calls CLIPS Kernel Expression Evaluator EvaluateExpression
                   for each expression-value of an instance expression
                   
                 Evaluates default slots only - slots that were specified
                 by overrides (sp->override == 1) are ignored)
  INPUTS       : 1) Instance address
  RETURNS      : Nothing useful
  SIDE EFFECTS : Each DATA_OBJECT slot in the instance's slot array is replaced
                   by the evaluation (by EvaluateExpression) of the expression
                   in the slot list.  The old expression-values
                   are deleted.
  NOTES        : CLIPS Syntax: (init-slots <instance>)
 *****************************************************************************/
globle VOID EvaluateInstanceSlots(result)
  DATA_OBJECT *result;
  {
   INSTANCE_TYPE *ins;
   INSTANCE_SLOT *slot;
   DATA_OBJECT temp;
   register int i,ertn;
   
   SetpType(result,SYMBOL);
   SetpValue(result,(VOID *) CLIPSFalseSymbol);
   EvaluationError = FALSE;
   if (CheckCurrentMessage("init-slots",TRUE) == FALSE)
     return;
   ins = (INSTANCE_TYPE *) CurrentMessageFrame->value;
   if (ins->init_eval == 0)
     return;
   for (i = 0 ; i < ins->cls->islot_cnt ; i++)
     {
      slot = &(ins->slots[i]);
      
      /* ===========================================================
         Slot-overrides are just a short-hand for put-slots, so they
         should be done with messages.  Defaults are from the class
         definition and can be placed directly.
         =========================================================== */
      if (slot->desc->override == 1)
        slot->desc->override = 0;
      else if (slot->valexp != NULL)
        {
         ertn = EvaluateAndStoreInDataObject(slot->desc->multiple,slot->valexp,&temp);
         if ((ertn != FALSE) ? (ValidSlotValue(&temp,"init-slots") != FALSE) : FALSE)
           PutSlotValue(ins,slot,(ertn == MULTI_CLEAR) ? NULL : &temp,TRUE);
        }
      else if (slot->desc->default_specified == 1)
        {
         if ((slot->desc->shared == 0) ? TRUE :
             ((slot->desc->dynamic == 1) || (slot->desc->share_cnt == 1)))
           PutSlotValue(ins,slot,NULL,TRUE);
        }
      if (ins->garbage == 1)
        {
         PrintCLIPS(WERROR,ValueToString(ins->name));
         PrintCLIPS(WERROR," instance deleted by slot-override evaluation.\n");
         SetEvaluationError(TRUE);
         return;
        }
      if (EvaluationError)
        return;
     }
   SetpType(result,INSTANCE);
   SetpValue(result,(VOID *) ins);
   ins->init_eval = 0;
  }
  
/*************************************************************
  NAME         : ValidSlotValue
  DESCRIPTION  : Determines if a value is appropriate
                   for a slot-value
  INPUTS       : 1) The value buffer
                 2) Name of function caller
  RETURNS      : TRUE if value is OK, FALSE otherwise
  SIDE EFFECTS : Sets EvaluationError is slot is not OK
  NOTES        : Examines all fields of a multi-field
 *************************************************************/
globle int ValidSlotValue(val,func)
  DATA_OBJECT *val;
  char *func;
  {
   register int i,type;
   
   type = val->type;
   if (val->type == MULTIFIELD)
     {
      for (i = val->begin ; i <= val->end ; i++)
        {
         type = ((struct fact *) val->value)->atoms[i].type;
         if ((type != SYMBOL) && (type != INTEGER) && (type != STRING) &&
             (type != FLOAT) && (type != INSTANCE_NAME) && (type != INSTANCE)
             && (type != EXTERNAL_ADDRESS))
           break;
        }
      if (i > val->end)
        return(TRUE);
     }
   else if (type == SYMBOL)
     return(TRUE);
   else if (type == INTEGER)
     return(TRUE);
   else if (type == STRING)
     return(TRUE);
   else if (type == FLOAT)
     return(TRUE);
   else if (type == INSTANCE_NAME)
     return(TRUE);
   else if (type == INSTANCE)
     return(TRUE);
   else if (type == EXTERNAL_ADDRESS)
     return(TRUE);
   SetEvaluationError(TRUE);
   PrintCLIPS(WERROR,"Illegal value for slot in function ");
   PrintCLIPS(WERROR,func);
   PrintCLIPS(WERROR,".\n");
   return(FALSE);
  }
  
/*********************************************************************
  NAME         : CheckMultifieldSlotModify
  DESCRIPTION  : For the functions mv-slot-replace, insert, & delete
                    as well as direct-replace, insert, & dlete
                    this function gets the slot, index, and optional
                    field-value for these functions
  INPUTS       : 1) A code indicating the type of operation 
                      INSERT  (0) : Requires one index
                      REPLACE (1) : Requires two indices
                      DELETE  (2) : Requires two indices
                 2) Function name-string
                 3) Instance address
                 4) Argument expression chain
                 5) Caller's buffer for index (or beginning of range)
                 6) Caller's buffer for end of range 
                     (can be NULL for INSERT)
                 7) Caller's new-field value buffer
                     (can be NULL for DELETE)
  RETURNS      : The address of the instance-slot,
                    NULL on errors
  SIDE EFFECTS : Caller's index buffer set
                 Caller's new-field value buffer set (if not NULL)
                   Will allocate an ephemeral segment to store more
                     than 1 new field value
                 EvaluationError set on errors
  NOTES        : Assume the argument chain is at least 2
                   expressions deep - slot, index, and optional values
 *********************************************************************/
globle INSTANCE_SLOT *CheckMultifieldSlotModify(code,func,ins,args,rb,re,newval)
  int code;
  char *func;
  INSTANCE_TYPE *ins;
  EXPRESSION *args;
  int *rb,*re;
  DATA_OBJECT *newval;
  {
   DATA_OBJECT temp;
   INSTANCE_SLOT *sp;
   int start;
   
   start = (args == GetFirstArgument()) ? 1 : 2;
   EvaluationError = FALSE;
   EvaluateExpression(args,&temp);
   if (temp.type != SYMBOL)
     {
      ExpectedTypeError(func,start,"symbol");
      SetEvaluationError(TRUE);
      return(NULL);
     }
   sp = FindInstanceSlot(ins,(SYMBOL_HN *) temp.value);
   if (sp == NULL)
     {
      SlotExistError(ValueToString(temp.value),func);
      return(NULL);
     }
   if (sp->desc->multiple == 0)
     {
      PrintCLIPS(WERROR,"Function ");
      PrintCLIPS(WERROR,func);
      PrintCLIPS(WERROR," cannot be used on single-field slot ");
      PrintCLIPS(WERROR,ValueToString(sp->desc->name));
      PrintCLIPS(WERROR," in instance ");
      PrintCLIPS(WERROR,ValueToString(ins->name));
      PrintCLIPS(WERROR,".\n");
      SetEvaluationError(TRUE);
      return(NULL);
     }
   EvaluateExpression(args->next_arg,&temp);
   if (temp.type != INTEGER)
     {
      ExpectedTypeError(func,start+1,"integer");
      SetEvaluationError(TRUE);
      return(NULL);
     }
   args = args->next_arg->next_arg;
   *rb = ValueToInteger(temp.value);
   if ((code == REPLACE) || (code == DELETE))
     {
      EvaluateExpression(args,&temp);
      if (temp.type != INTEGER)
        {
         ExpectedTypeError(func,start+2,"integer");
         SetEvaluationError(TRUE);
         return(NULL);
        }
      *re = ValueToInteger(temp.value);
      args = args->next_arg;
     }
   if ((code == INSERT) || (code == REPLACE))
     {
      if (EvaluateAndStoreInDataObject(1,args,newval) == FALSE)
        return(NULL);
      if (ValidSlotValue(newval,func) == FALSE)
        return(NULL);
     }
   return(sp);
  }

/***************************************************
  NAME         : NoInstanceError
  DESCRIPTION  : Prints out an appropriate error
                  message when an instance cannot be
                  found for a function
  INPUTS       : 1) The instance name
                 2) The function name
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
globle VOID NoInstanceError(iname,func)
  char *iname,*func;
  {
   PrintCLIPS(WERROR,"No such instance ");
   PrintCLIPS(WERROR,iname);
   PrintCLIPS(WERROR," in function ");
   PrintCLIPS(WERROR,func);
   PrintCLIPS(WERROR,".\n");
   SetEvaluationError(TRUE);
  }
  
/***************************************************
  NAME         : SlotExistError
  DESCRIPTION  : Prints out an appropriate error
                  message when a slot cannot be
                  found for a function
  INPUTS       : 1) The slot name
                 2) The function name
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
globle VOID SlotExistError(sname,func)
  char *sname,*func;
  {
   PrintCLIPS(WERROR,"No such slot ");
   PrintCLIPS(WERROR,sname);
   PrintCLIPS(WERROR," in function ");
   PrintCLIPS(WERROR,func);
   PrintCLIPS(WERROR,".\n");
   SetEvaluationError(TRUE);
  }
  
/***************************************************
  NAME         : StaleInstanceAddress
  DESCRIPTION  : Prints out an appropriate error
                  message when an instance address
                  is no longer valid
  INPUTS       : The function name
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
globle VOID StaleInstanceAddress(func)
  char *func;
  {
   PrintCLIPS(WERROR,"Invalid instance address in function ");
   PrintCLIPS(WERROR,func);
   PrintCLIPS(WERROR,".\n");
  }
  
/**********************************************************************
  NAME         : GetInstancesChanged
  DESCRIPTION  : Returns whether instances have changed 
                   (any were added/deleted or slot values were changed)
                   since last time flag was set to FALSE
  INPUTS       : None
  RETURNS      : The instances-changed flag
  SIDE EFFECTS : None
  NOTES        : Used by interfaces to update instance windows
 **********************************************************************/
globle int GetInstancesChanged()
  {
   return(ChangesToInstances);
  }
  
/***************************************************
  NAME         : SetInstancesChanged
  DESCRIPTION  : Sets instances-changed flag (see above)
  INPUTS       : The value (TRUE or FALSE)
  RETURNS      : Nothing useful
  SIDE EFFECTS : The flag is set
  NOTES        : None
 ***************************************************/
globle VOID SetInstancesChanged(changed)
  int changed;
  {
   ChangesToInstances = changed;
  }
  
/* =========================================
   *****************************************
          INTERNALLY VISIBLE FUNCTIONS
   =========================================
   ***************************************** */
/**********************************************************
  NAME         : InsertSlotOverrides
  DESCRIPTION  : Replaces value-expression for a slot
  INPUTS       : 1) The instance address
                 2) The address of the beginning of the
                    list of slot-expressions
  RETURNS      : TRUE if all okay, FALSE otherwise
  SIDE EFFECTS : Old slot expression deallocated
  NOTES        : Assumes symbols not yet installed
                 EVALUATES the slot-name expression but
                    simply copies the slot value-expression
 **********************************************************/
static int InsertSlotOverrides(ins,slot_exp)
  INSTANCE_TYPE *ins;
  EXPRESSION *slot_exp;
  {
   INSTANCE_SLOT *slot;
   int svaccess;
   DATA_OBJECT temp;
   
   EvaluationError = FALSE;
   svaccess = WithinInit;
   while (slot_exp != NULL)
     {
      if ((EvaluateExpression(slot_exp,&temp) == TRUE) ? TRUE :
          (GetType(temp) != SYMBOL))
        {
         PrintCLIPS(WERROR,"Expected a valid slot name for slot-override.\n");
         SetEvaluationError(TRUE);
         return(FALSE);
        }
      slot = FindInstanceSlot(ins,(SYMBOL_HN *) ClipsGetValue(temp));
      if (slot == NULL)
        {
         PrintCLIPS(WERROR,"Slot ");
         PrintCLIPS(WERROR,DOToString(temp));
         PrintCLIPS(WERROR," does not exist in instance ");
         PrintCLIPS(WERROR,ValueToString(ins->name));
         PrintCLIPS(WERROR,".\n");
         SetEvaluationError(TRUE);
         return(FALSE);
        }
      WithinInit = TRUE;
      DirectMessage(PSM_PREFIX,slot->desc->name,
                     ins,NULL,slot_exp->next_arg->arg_list);
      WithinInit = svaccess;
      if (EvaluationError)
        return(FALSE);
      slot->desc->override = 1;
      slot_exp = slot_exp->next_arg->next_arg;
     }
   return(TRUE);
  }
  
/*****************************************************************
  NAME         : InstanceLocationInfo
  DESCRIPTION  : Determines where a specified instance belongs
                   in the instance hash table
  INPUTS       : 1) The CLIPS symbol for the name of the instance
                 2) Caller's buffer for previous node address
                 3) Caller's buffer for hash value
  RETURNS      : The address of the found instance, NULL otherwise
  SIDE EFFECTS : None
  NOTES        : Conflict resolution is done by chaining
 *****************************************************************/
static INSTANCE_TYPE *InstanceLocationInfo(iname,prv,hashval)
  SYMBOL_HN *iname;
  INSTANCE_TYPE **prv;
  unsigned *hashval;
  {
   INSTANCE_TYPE *ins;
   
   *hashval = HashInstance(iname);
   ins = InstanceTable[*hashval];
   *prv = NULL;
   while ((ins != NULL) ? (ins->name != iname) : FALSE)
     {
      *prv = ins;
      ins = ins->nxt_hsh;
     }
   return(ins);
  }
  
/********************************************************
  NAME         : NewInstance
  DESCRIPTION  : Allocates and initializes a new instance
  INPUTS       : None
  RETURNS      : The address of the new instance
  SIDE EFFECTS : None
  NOTES        : None
 ********************************************************/
static INSTANCE_TYPE *NewInstance()
  {
   INSTANCE_TYPE *instance;
   
   instance = get_struct(instance);
   instance->installed = 0;
   instance->garbage = 0;
   instance->init_eval = 0;
   instance->busy = 0;
   instance->depth = 0;
   instance->name = NULL;
   instance->hashval = 0;
   instance->cls = NULL;
   instance->slots = NULL;
   instance->prv_cls = NULL;
   instance->nxt_cls = NULL;
   instance->prv_hsh = NULL;
   instance->nxt_hsh = NULL;
   instance->prv_lst = NULL;
   instance->nxt_lst = NULL;
   return(instance);
  }

/****************************************************************
  NAME         : BuildDefaultSlots
  DESCRIPTION  : The current instance's address is
                   in the global variable CurrentInstance.
                   This builds the slots and the default values
                   from the direct class of the instance and its
                   inheritances.
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : Allocates the slot array for
                   the current instance
  NOTES        : The current instance's address is
                 stored in a global variable
 ****************************************************************/
static VOID BuildDefaultSlots()
  {
   register int i,scnt;
   INSTANCE_SLOT *dst;
   SLOT_DESC **src;
   
   scnt = CurrentInstance->cls->islot_cnt;
   if (scnt > 0)
     {
      CurrentInstance->slots = dst =
         (INSTANCE_SLOT *) gm2((int) (sizeof(INSTANCE_SLOT) * scnt));
      src = CurrentInstance->cls->itemplate;
      for (i = 0 ; i < scnt ; i++)
        {
          if (src[i]->shared == 1)
            {
             src[i]->share_cnt++;
             dst[i].valaddr = &(src[i]->share_val);
            }
          else
            dst[i].valaddr = &(dst[i].val);
          dst[i].desc = src[i];
          dst[i].val = NULL;
          dst[i].valexp = NULL;
        }
     }
  }
  
/*******************************************************
  NAME         : HashInstance
  DESCRIPTION  : Generates a hash index for a given
                 instance name
  INPUTS       : The address of the instance name SYMBOL_HN
  RETURNS      : The hash index value
  SIDE EFFECTS : None 
  NOTES        : Counts on the fact that the symbol
                 has already been hashed into the CLIPS
                 symbol table - uses that hash value
                 multiplied by a prime for a new hash
 *******************************************************/
static unsigned HashInstance(cname)
  SYMBOL_HN *cname;
  {
   unsigned long tally;
   
   tally = ((unsigned long) cname->bucket) * BIG_PRIME;
   return((unsigned) (tally % INSTANCE_TABLE_HASH_SIZE));
  }

/********************************************************
  NAME         : InstallInstance
  DESCRIPTION  : Prevent name and slot value symbols
                   from being ephemeral (all others
                   taken care of by class defn)
  INPUTS       : 1) The address of the instance
                 2) A flag indicating whether to
                    install or deinstall
  RETURNS      : Nothing useful
  SIDE EFFECTS : Symbol counts incremented or decremented
  NOTES        : Slot symbol installations are handled
                   by PutSlotValue
 ********************************************************/
static VOID InstallInstance(ins,set)
  INSTANCE_TYPE *ins;
  int set;
  {
   if (set == TRUE) 
     {
      if (ins->installed)
        return;
#if DEBUGGING_FUNCTIONS
      if (WatchInstances == ON)
        {
         PrintCLIPS(WTRACE,"==> instance ");
         PrintCLIPS(WTRACE,ValueToString(ins->name));
         PrintCLIPS(WTRACE,"\n");
        }
#endif
      ins->installed = 1;
      ins->depth = CurrentEvaluationDepth;
      IncrementSymbolCount(ins->name);
      ins->cls->busy++;
     }
   else
     {
      if (! ins->installed)
        return;
      ins->installed = 0;
      DecrementSymbolCount(ins->name);
      ins->cls->busy--;
     }
  }
  
/***************************************************************
  NAME         : StoreValuesInMultifield
  DESCRIPTION  : Stores evaluated slot value-expressions in a
                   single multifield variable (so other
                   CLIPS Kernel functions can access the
                   slot value)
  INPUTS       : 1) A linked list of values to be stored
                 2) The destination multifield variable
                 3) The total number of ATOMS in the value list
  RETURNS      : Nothing useful
  SIDE EFFECTS : Original value list is destroyed
                 Old values stored in new large
                   multifield variable
  NOTES        : None
 ****************************************************************/
static VOID StoreValuesInMultifield(orgs,dest,vcnt)
  DATA_OBJECT *orgs,*dest;
  int vcnt;
  {
   SEGMENT seg,mseg;
   register int i,j;
   DATA_OBJECT *val;
   TYPE type;
   int vpend;
   
   SetpType(dest,MULTIFIELD);
   SetpDOBegin(dest,1);
   SetpDOEnd(dest,vcnt);
   seg = (SEGMENT) CreateMultifield(vcnt);
   SetpValue(dest,(VOID *) seg);
   i = 1;
   while (orgs != NULL)
     {
      val = orgs;
      orgs = orgs->next;
      type = GetpType(val);
      if (type == MULTIFIELD)
        {
         vpend = GetpDOEnd(val);
         mseg = (SEGMENT) GetpValue(val);
         for (j = GetpDOBegin(val) ; j <= vpend ; j++)
           {
            SetMFType(seg,i,GetMFType(mseg,j));
            SetMFValue(seg,i,GetMFValue(mseg,j));
            i++;
           }
        }
      else
        {
         SetMFType(seg,i,type);
         SetMFValue(seg,i,GetpValue(val));
         i++;
        }
      rtn_struct(dataObject,val);
     }
  }
  
#endif

/***************************************************
  NAME         : 
  DESCRIPTION  : 
  INPUTS       : 
  RETURNS      : 
  SIDE EFFECTS : 
  NOTES        : 
 ***************************************************/


