/*
 * wxclips.cc
 *
 * wxWindows front end to CLIPS
 *
 */

#include <wx.h>

#include <iostream.h>
#include <stdlib.h>
#include <ctype.h>

#include <wx_help.h>
#include "clipscpp.h"
#include "wx_cmds.h"
#include "wxclips.h"

#include "clips.h"

#ifdef wx_x
#include "aiai.xbm"
#endif

#ifdef wx_msw
char *wxClipsIcon = "door_icn";
#endif

#ifdef CLIPS6
typedef void (*ClipsFunction) ();
#else
typedef int (*ClipsFunction) ();
#endif

#ifdef wx_x
#define COMMAND_PANEL_HEIGHT 130
#define COMMAND_TEXT_WIDTH   350
#endif
#ifdef wx_msw
#define COMMAND_PANEL_HEIGHT 100
#define COMMAND_TEXT_WIDTH   400
#endif

extern wxObject *wxGetTypedObject(long, WXTYPE);

int all_watch = 0;
int compilations_watch = 1;
int facts_watch = 0;
int rules_watch = 0;
int activations_watch = 0;
int statistics_watch = 0;
int globals_watch = 0;
int deffunctions_watch = 0;
int limit = -1;

float wxClipsVersion = 1.20;

wxHelpInstance *HelpInstance = NULL;

wxClipsEnvFrame *wxTheClipsFrame = NULL;

// This statement initialises the whole application
MyApp     myApp;

wxFrame *wxStartApp(void)
{
  wxFrame *main_frame = NULL;
  clipsReturnType = clipsUNKNOWN;

  RouteCommand("(return-result (app-on-init))");

  if (clipsReturnType == clipsLONG)
  {
    main_frame = (wxFrame *)wxGetTypedObject(clipsReturnLong, wxTYPE_FRAME);
    return main_frame;
  }
  else return NULL;
/*
  if (!main_frame)
  {
    ClipsError << "Error in wxStartApp: app-on-init should return a main frame ID!\n";
#ifdef wx_x
    if (!wxTheClipsFrame)
      cerr << "Error in wxStartApp: app-on-init should return a main frame ID!\n";
#endif
    return NULL;
  }
  else return main_frame;
 */
}

void wxInitializeClips(void)
{
  // Initialize CLIPS
  wxInitClips();

  AddRunFunction("wxyield", (ClipsFunction)ClipsYield, 100);

#ifdef CLIPS6
  AddRouter("wxrouter", 10,
            FindOutput,
            PrintOutput,
            DummyGetc,
            DummyUngetc,
            ExitOutput);
#else
  AddRouter("wxrouter", 10,
            (ClipsFunction)FindOutput,
            (ClipsFunction)PrintOutput,
            (ClipsFunction)DummyGetc,
            (ClipsFunction)DummyUngetc,
            (ClipsFunction)ExitOutput);
#endif

#ifdef CLIPS6
  Watch("compilations");
  Watch("facts");
  Watch("rules");
  Watch("activations");
  Watch("statistics");
  Watch("globals");
  Watch("all");

  Watch("compilations");
#else
  AddWatchItem("compilations", &compilations_watch, 100);
  AddWatchItem("facts", &facts_watch, 100);
  AddWatchItem("rules", &rules_watch, 100);
  AddWatchItem("activations", &activations_watch, 100);
  AddWatchItem("statistics", &statistics_watch, 100);
  AddWatchItem("globals", &globals_watch, 100);
  AddWatchItem("all", &all_watch, 100);

  SetWatchItem("compilations", compilations_watch);
#endif
}

void Usage(void)
{
#ifdef wx_x
  cout << "WXCLIPS command line options:\n";
  cout << "wxclips [-clips    <filename>]  ; clips batch file\n";
  cout << "        [-start]                ; Call (app-on-init), no debugging window\n";
  cout << "        [-h]                    ; this help screen\n";
  cout << "        [-help]                 ; list of generic X window arguments\n";
  exit(0);
#endif
}

// The `main program' equivalent, creating the windows and returning the
// main frame
wxFrame *MyApp::OnInit(void)
{
  wxInitializeClips();
  wxIPCInitialize();

  HelpInstance = new wxHelpInstance(TRUE);
  HelpInstance->Initialize("wxclips");

  char *batchFile = NULL;
  Bool start = FALSE;

  int i = 1;
  while (i < argc)
  {
    if (strcmp(argv[i], "-clips") == 0)
    {
      i ++;
      if (i < argc)
        batchFile = copystring(argv[i]);
      i ++;
    }
    else if (strcmp(argv[i], "-start") == 0)
    {
      i ++;
      start = TRUE;
    }
#ifdef wx_x
    else if (strcmp(argv[i], "-h") == 0)
    {
      i ++;
      Usage();
    }
#endif
    else
    {
#ifdef wx_x
      cerr << "Invalid command-line option.\n";
#endif
      i ++;
      Usage();
    }
  }

  wxFrame *clipsMainFrame = NULL;
  if (batchFile)
  {
    wxExecuteClipsFile(batchFile);
    if (start)
    {
      clipsMainFrame = wxStartApp();
    }
  }

  if (start && clipsMainFrame)
  {
    return clipsMainFrame;
  }
  else
  {
    wxShowClipsWindow();
    return wxTheClipsFrame;
  }
}

void wxShowClipsWindow(void)
{
  if (wxTheClipsFrame)
    wxTheClipsFrame->Iconize(FALSE);
  else
    wxTheClipsFrame = new wxClipsEnvFrame(NULL, "wxCLIPS", 10, 10, 480, 450);
}

void wxDeleteClipsWindow(void)
{
  if (wxTheClipsFrame && wxTheClipsFrame->OnClose())
  {
    delete wxTheClipsFrame;
    wxTheClipsFrame = NULL;
  }
}

// Send a command to CLIPS
void EnterCommand(wxButton& but, wxCommandEvent& event)
{
  char *text = wxTheClipsFrame->command_line->GetValue();
  if (text)
  {
    char *comm = copystring(text);
    wxTheClipsFrame->text_window->WriteText("CLIPS> ");
    wxTheClipsFrame->text_window->WriteText(comm);
    wxTheClipsFrame->text_window->WriteText("\n");
    RouteCommand(comm);
    delete comm;
    PeriodicCleanup(CLIPS_TRUE,CLIPS_FALSE);
    if (wxTheClipsFrame)
    {
      wxTheClipsFrame->command_line->SetValue("");
      wxTheClipsFrame->command_line->SetFocus();
    }
  }
}

// Define my frame constructor
wxClipsEnvFrame::wxClipsEnvFrame(wxFrame *frame, char *title, int x, int y, int w, int h):
  wxFrame(frame, title, x, y, w, h)
{
  panel = NULL;
  text_window = NULL;

  wxTheClipsFrame = this;
  // Give it a status line
  CreateStatusLine();
  char buf[200];
  sprintf(buf, "Welcome to wxCLIPS %.2f, a GUI environment for CLIPS", wxClipsVersion);
  SetStatusText(buf);

  // Give it an icon
#ifdef wx_msw
  wxIcon *icon = new wxIcon(wxClipsIcon);
#endif
#ifdef wx_x
  wxIcon *icon = new wxIcon(aiai_bits, 64, 64);
#endif

  SetIcon(icon);

  // Make a menubar

  wxMenu *file_menu = new wxMenu;

  file_menu->Append(WXCLIPS_BATCH, "&Batch", "Execute constructs and commands in batch file");
  file_menu->Append(WXCLIPS_LOAD, "&Load definitions", "Load CLIPS constructs only");
  file_menu->Append(WXCLIPS_SAVE, "&Save definitions", "Save CLIPS definitions");
  file_menu->AppendSeparator();
  file_menu->Append(WXCLIPS_BLOAD, "BL&oad", "Load binary CLIPS definitions");
  file_menu->Append(WXCLIPS_BSAVE, "BS&ave", "Save binary CLIPS definitions");
  file_menu->AppendSeparator();
  file_menu->Append(WXCLIPS_QUIT, "&Exit", "Exit from wxCLIPS");

  wxMenu *run_menu = new wxMenu;

  run_menu->Append(WXCLIPS_RESETRUN, "R&eset and Run", "Reset CLIPS and run");
  run_menu->Append(WXCLIPS_RUN, "&Run", "Run CLIPS");
  run_menu->Append(WXCLIPS_RUN_LIMIT, "Run with &limit", "Run with limit");
  run_menu->Append(WXCLIPS_RESET, "Re&set", "Reset CLIPS");
  run_menu->AppendSeparator();
  run_menu->Append(WXCLIPS_CLEAR, "&Clear", "Clear CLIPS");
  run_menu->AppendSeparator();
  run_menu->Append(WXCLIPS_WATCHES, "&Watch", "Select watches");
  run_menu->Append(WXCLIPS_ADD_BREAKPOINT, "&Add breakpoint", "Add a breakpoint on selected rule");
  run_menu->Append(WXCLIPS_REMOVE_BREAKPOINT, "Re&move breakpoint", "Remove a breakpoint from selected rule");
  run_menu->AppendSeparator();
//  run_menu->Append(WXCLIPS_SET_STRATEGY, "&Set strategy");
  run_menu->Append(WXCLIPS_ASSERT_FACT, "Assert &fact", "Assert a fact");

  wxMenu *app_menu = new wxMenu;
  app_menu->Append(WXCLIPS_CLEAN_WINDOWS, "&Clean up windows", "Clean up user-created windows");
  app_menu->Append(WXCLIPS_RUN_APP, "&Start Application", "Call app-on-init");

  wxMenu *list_menu = new wxMenu;
#if !RUN_TIME
  list_menu->Append(WXCLIPS_LIST_RULES, "&Rules", "List rules");
  list_menu->Append(WXCLIPS_LIST_FACTS, "&Facts", "List facts");
  list_menu->Append(WXCLIPS_LIST_MATCHES, "&Matches", "List matches");
  list_menu->Append(WXCLIPS_LIST_AGENDA, "&Agenda", "List agenda");
  list_menu->Append(WXCLIPS_LIST_BREAKPOINTS, "&Breakpoints", "List breakpoints");
  list_menu->Append(WXCLIPS_LIST_TEMPLATES, "&Templates", "List templates");
  list_menu->Append(WXCLIPS_LIST_GLOBALS, "&Globals", "List globals");
//  list_menu->Append(WXCLIPS_LIST_STRATEGY, "&Current strategy");
  list_menu->AppendSeparator();
#endif
  list_menu->Append(WXCLIPS_CLEAR_SCREEN, "&Clear screen", "Clear screen");

  wxMenu *help_menu = new wxMenu;
  help_menu->Append(WXCLIPS_REFERENCE, "wxCLIPS &Reference", "Invoke wxCLIPS manual");
  help_menu->AppendSeparator();
  help_menu->Append(WXCLIPS_ABOUT, "&About wxCLIPS", "About wxCLIPS");

  wxMenuBar *menu_bar = new wxMenuBar;

  menu_bar->Append(file_menu, "&File");
  menu_bar->Append(run_menu, "&Rules");
  menu_bar->Append(app_menu, "&Application");
  menu_bar->Append(list_menu, "&List");
  menu_bar->Append(help_menu, "&Help");

  // Associate the menu bar with the frame
  SetMenuBar(menu_bar);

  // Make a text window
  panel = new wxPanel(this, 0, 0, 500, COMMAND_PANEL_HEIGHT);
  wxButton *button = new wxButton(panel, (wxFunction)EnterCommand,
     "Do");
  button->SetDefault();
//  command_line = new wxMultiText(panel, NULL, "CLIPS> ", "", -1, -1,
//                                  COMMAND_TEXT_WIDTH, 90);
  command_line = new wxText(panel, NULL, "CLIPS> ", "", -1, -1,
                                  COMMAND_TEXT_WIDTH);

  text_window = new wxTextWindow(this, 0, COMMAND_PANEL_HEIGHT, 500, 400, wxBORDER);

  SetSize(x, y, w, h);
  Show(TRUE);
}

wxClipsEnvFrame::~wxClipsEnvFrame(void)
{
  wxTheClipsFrame = NULL;
}

// Intercept menu commands
char *wxBatchFile = NULL;
void wxClipsEnvFrame::OnMenuCommand(int id)
{
  switch (id)
  {
    case WXCLIPS_BATCH:
    {
      char *path = NULL;
      char *name = NULL;
      if (wxBatchFile)
      {
        path = PathOnly(wxBatchFile);
        name = FileNameFromPath(wxBatchFile);
      }

      char *file = wxFileSelector("CLIPS batch file to execute",
                     path, name,
                     "clp", "*.clp", 0, this);
      if (file)
      {
        wxExecuteClipsFile(file);
        if (wxBatchFile) delete wxBatchFile;
        wxBatchFile = copystring(file);
      }
      break;
    }
    case WXCLIPS_LOAD:
    {
      char *file = wxFileSelector("CLIPS file to load",
                     PathOnly(wxBatchFile), NULL,
                     "clp", "*.clp", 0, this);
      if (file)
      {
        SetPrintWhileLoading(CLIPS_TRUE);
        LoadConstructs(file);
        SetPrintWhileLoading(CLIPS_FALSE);
        if (wxBatchFile) delete wxBatchFile;
        wxBatchFile = copystring(file);
      }
      break;
    }
    case WXCLIPS_SAVE:
    {
      char *file = wxFileSelector("CLIPS file to save to",
                     PathOnly(wxBatchFile), NULL,
                     "clp", "*.clp", 0, this);
      if (file)
      {
	SaveConstructs(file);
        if (wxBatchFile) delete wxBatchFile;
        wxBatchFile = copystring(file);
      }
      break;
    }
    case WXCLIPS_BLOAD:
    {
      char *file = wxFileSelector("CLIPS file to binary load",
                     PathOnly(wxBatchFile), NULL,
                     "clp", "*.clp", 0, this);
      if (file)
      {
        SetPrintWhileLoading(CLIPS_TRUE);
	Bload(file);
        SetPrintWhileLoading(CLIPS_FALSE);
        if (wxBatchFile) delete wxBatchFile;
        wxBatchFile = copystring(file);
      }
      break;
    }
    case WXCLIPS_BSAVE:
    {
#if (!RUN_TIME && BLOAD_AND_BSAVE)
      char *file = wxFileSelector("CLIPS file to binary save to",
                     NULL, NULL,
                     "clp", "*.clp", 0, this);
      if (file)
      {
	Bsave(file);
        if (wxBatchFile) delete wxBatchFile;
        wxBatchFile = copystring(file);
      }
#endif
      break;
    }
    case WXCLIPS_RUN:
    {
      RunCLIPS(-1);
      break;
    }
    case WXCLIPS_RUN_LIMIT:
    {
      char *s = IntToString(limit);
      char *new_s = wxGetTextFromUser("Enter limit value: ",
                            "Respond please",s, this);
      if (new_s)
      {
	StringToInt(new_s, &limit);
	RunCLIPS(limit);
      }
      if (s)
	delete s;
      break;
    }
    case WXCLIPS_RESETRUN:
    {
#ifdef CLIPS6
      Reset();
#else
      ResetCLIPS();
#endif
      RunCLIPS(-1);
      break;
    }
    case WXCLIPS_CLEAR:
    {
      ClearCLIPS();
      break;
    }
    case WXCLIPS_RESET:
    {
#ifdef CLIPS6
      Reset();
#else
      ResetCLIPS();
#endif
      break;
    }
    case WXCLIPS_WATCHES:
    {
      wxDialogBox dialog(this, "Watches", TRUE, 100, 100, 200, 200);
      (void)new wxMessage(&dialog, "Edit watches");
      dialog.NewLine();

      wxCheckBox check0(&dialog, NULL, "All");
      check0.SetValue(all_watch);

      dialog.NewLine();

      wxCheckBox check1(&dialog, NULL, "Facts");
      check1.SetValue(facts_watch);

      dialog.NewLine();

      wxCheckBox check2(&dialog, NULL, "Rules");
      check2.SetValue(rules_watch);

      dialog.NewLine();

      wxCheckBox check3(&dialog, NULL, "Activations");
      check3.SetValue(activations_watch);

      dialog.NewLine();

      wxCheckBox check4(&dialog, NULL, "Compilations");
      check4.SetValue(compilations_watch);

      dialog.NewLine();

      wxCheckBox check5(&dialog, NULL, "Statistics");
      check5.SetValue(statistics_watch);

      dialog.NewLine();

      wxCheckBox check6(&dialog, NULL, "Globals");
      check6.SetValue(globals_watch);

      dialog.NewLine();
      (void) new wxButton(&dialog, (wxFunction)GenericOk, "Ok");

      dialog.Fit();
      dialog.Show(TRUE);

      all_watch = check0.GetValue();
      facts_watch = check1.GetValue();
      rules_watch = check2.GetValue();
      activations_watch = check3.GetValue();
      compilations_watch = check4.GetValue();
      statistics_watch = check5.GetValue();
      globals_watch = check6.GetValue();
#ifdef CLIPS6
      if (all_watch)
        Watch("all");
      else Unwatch("all");

      if (facts_watch)
        Watch("facts");
      else Unwatch("facts");

      if (rules_watch)
        Watch("rules");
      else Unwatch("rules");

      if (activations_watch)
        Watch("activations");
      else Unwatch("activations");

      if (compilations_watch)
        Watch("compilations");
      else Unwatch("compilations");

      if (statistics_watch)
        Watch("statistics");
      else Unwatch("statistics");

      if (globals_watch)
        Watch("globals");
      else Unwatch("globals");
#else
      SetWatchItem("all", all_watch);
      SetWatchItem("facts", facts_watch);
      SetWatchItem("rules", rules_watch);
      SetWatchItem("activations", activations_watch);
      SetWatchItem("compilations", compilations_watch);
      SetWatchItem("statistics", statistics_watch);
      SetWatchItem("globals", globals_watch);
#endif
      break;
    }
    case WXCLIPS_ADD_BREAKPOINT:
    {
      char *name = PromptForRuleName();
      if (name)
      {
	void *ptr = FindDefrule(name);
	if (ptr)
	{
	  AddBreakpoint(ptr);
	  char buf[200];
	  sprintf(buf, "Put breakpoint on %s.\n", name);
	  PrintCLIPS("wdialog", buf);
	}
      }
      break;
    }
    case WXCLIPS_REMOVE_BREAKPOINT:
    {
      char *name = PromptForRuleName();
      if (name)
      {
	void *ptr = FindDefrule(name);
	if (ptr)
	{
          RemoveBreakpoint(ptr);
	  char buf[200];
          sprintf(buf, "Removed breakpoint on %s.\n", name);
	  PrintCLIPS("wdialog", buf);
	}
      }
      break;
    }
    case WXCLIPS_ASSERT_FACT:
    {
      char *s = wxGetTextFromUser(
        "Enter fact (no outer brackets): ", "Respond please","", this);
      if (s)
      {
        char buf[300];
        sprintf(buf, "Asserting fact %s\n", s);
        PrintCLIPS("wdialog", buf);
        AssertString(s);
      }
      break;
    }
    case WXCLIPS_LIST_AGENDA:
    {
#if (!RUN_TIME && DEBUGGING_FUNCTIONS)
      ListAgenda();
#endif
      break;
    }
    case WXCLIPS_LIST_RULES:
    {
#if (!RUN_TIME && DEFRULE_CONSTRUCT && DEBUGGING_FUNCTIONS)
#ifdef CLIPS6
      ListDefrules("stdout", NULL);
#else
      ListDefrules();
#endif
#endif
      break;
    }
    case WXCLIPS_LIST_FACTS:
    {
#if (!RUN_TIME && DEFRULE_CONSTRUCT && DEBUGGING_FUNCTIONS)
#ifdef CLIPS6
      ListDeffacts("stdout", NULL);
#else
      ListDeffacts();
#endif
#endif
      break;
    }
    case WXCLIPS_LIST_TEMPLATES:
    {
#if (!RUN_TIME && DEFTEMPLATE_CONSTRUCT && DEBUGGING_FUNCTIONS)
#ifdef CLIPS6
      ListDeftemplates("stdout", NULL);
#else
      ListDeftemplates();
#endif
#endif
      break;
    }
    case WXCLIPS_LIST_GLOBALS:
    {
#if (!RUN_TIME && DEFGLOBAL_CONSTRUCT && DEBUGGING_FUNCTIONS)
#ifdef CLIPS6
      ListDefglobals("stdout", NULL);
#else
      ListDefglobals();
#endif
#endif
      break;
    }
    case WXCLIPS_LIST_BREAKPOINTS:
    {
#if (!RUN_TIME && DEBUGGING_FUNCTIONS)
      ListBreakpoints();
#endif
      break;
    }
    case WXCLIPS_CLEAR_SCREEN:
    {
      text_window->Clear();
      break;
    }
    case WXCLIPS_CLEAN_WINDOWS:
    {
      wxcCleanWindows();
      break;
    }
    case WXCLIPS_RUN_APP:
    {
      wxcCleanWindows();
      (void)wxStartApp();
      break;
    }
    case WXCLIPS_REFERENCE:
    {
      HelpInstance->LoadFile("wxclips");
      HelpInstance->DisplayContents();
      break;
    }

    case WXCLIPS_ABOUT:
    {
      char aboutBuffer[200];
#ifdef CLIPS6
      char *clipsVersion = "6.0";
#else
      char *clipsVersion = "5.1";
#endif
      sprintf(aboutBuffer, "wxCLIPS\nNASA (c) and Julian Smart, AIAI (c) 1994\nCLIPS Version %s, wxCLIPS Version %.2f",
              clipsVersion, wxClipsVersion);
      wxMessageBox(aboutBuffer, "About wxCLIPS");
      break;
    }
    case WXCLIPS_QUIT:
    {
      OnClose();
      delete this;
      break;
    }
  }
}

// Size the subwindows when the frame is resized
void wxClipsEnvFrame::OnSize(int w, int h)
{
  if (panel && text_window)
  {
/*
    int width, height;
    GetClientSize(&width, &height);
    panel->SetSize(0, 0, width, COMMAND_PANEL_HEIGHT);
    text_window->SetSize(0, COMMAND_PANEL_HEIGHT, width, height-COMMAND_PANEL_HEIGHT);
*/
    int width, height;
    GetClientSize(&width, &height);

    int textWidth, textHeight, textX, textY;
    command_line->GetSize(&textWidth, &textHeight);
    command_line->GetPosition(&textX, &textY);

    int newTextWidth = width - textX - 5;
    int panelHeight = max((textHeight + textY + 5), 20);
    
    panel->SetSize(0, 0, width, panelHeight);
    text_window->SetSize(0, panelHeight, width, height-panelHeight);

    command_line->SetSize(-1, -1, newTextWidth, -1);
  }
}

// Define the behaviour for the frame closing
// - must delete all frames except for the main one.
Bool wxClipsEnvFrame::OnClose(void)
{
  wxcCleanWindows();
  return TRUE;
}

char *wxClipsEnvFrame::PromptForRuleName(void)
{
#ifdef CLIPS6
  return NULL;
#else
  wxStringList strings;
  void *ptr = GetNextDefrule(NULL);
  while (ptr)
  {
    char *name = GetDefruleName(ptr);
    strings.Add(name);
    ptr = GetNextDefrule(ptr);
  }
  char **sarray = strings.ListToArray();
  char *the_name = wxGetSingleChoice("Select a rule", "Selection",
       strings.Number(), sarray, this);

  delete sarray;

  return the_name;
#endif
}


void GenericOk(wxButton& but, wxCommandEvent& event)
{
  wxDialogBox *dialog = (wxDialogBox *)but.GetParent();

  dialog->Show(FALSE);
}


// Compatibility
#ifdef wx_msw
int system(char *command)
{
  wxExecute(command);
  return 1;
}
#endif

// I/O Routers
int FindOutput(char *name)
{
  if ((strcmp(name, "wtrace") == 0) ||
      (strcmp(name, "stdout") == 0) ||
      (strcmp(name, "werror") == 0) ||
      (strcmp(name, "wclips") == 0) ||
      (strcmp(name, "wdisplay") == 0) ||
      (strcmp(name, "wagenda") == 0) ||
      (strcmp(name, "wdialog") == 0))
    return (CLIPS_TRUE);
  else
    return (CLIPS_FALSE);
}

int PrintOutput(char *name, char *str)
{
  if (wxTheClipsFrame)
    wxTheClipsFrame->text_window->WriteText(str);
  return 1;
}

int ExitOutput(int exitcode)
{
  return 1;
}

// I don't know why J.S.'s method of specifying routers blows up in 6...
// but this seems to work. RobW

int DummyGetc(char * c)
{
  return 1;
}

int DummyUngetc(int i, char * c)
{
  return 1;
}

int ClipsYield(void)
{
  wxYield();
  return 1;
}

/*
 * Functions specific to this front end
 *
 */
long clipsQuit()
{
  wxTheClipsFrame->OnClose();
  delete wxTheClipsFrame;
  return 1;
}

/*
 * Required by wx_cmds or CLIPS itself
 *
 */

void ClipsErrorFunction(char *s)
{
  if (wxTheClipsFrame)
    *(wxTheClipsFrame->text_window) << s;
#ifdef wx_x
  else
    cerr << s;
#endif
}

/*
 * TEMPORARY - for LEO2KADS!!
 *
 */
/* 
extern "C" void clipsWritePrompt(void)
{
}
extern "C" void clipsWriteStatusLine(void)
{
}
extern "C" char *clipsGetChoice(void)
{
  return NULL;
}

extern "C" void clipsSetMenuDescriptions(void)
{
}

extern "C" char *clipsGetText(void)
{
  return NULL;
}

extern "C" void clipsWriteInformation(void)
{
}

extern "C" void clipsDisplayHelpText(void)
{
}

extern "C" long clipsGetChar(void)
{
  return 0;
}

extern "C" void clipsDisplayFile(void)
{
}
*/

void UserFunctions()
{
  wxUserFunctions();

  DefineFunction("quit", 'l', PTIF clipsQuit, "clipsQuit");
/*
  // Temporary, for Leo2KADS !!!
  DefineFunction("get-choice", 's', PTIF clipsGetChoice, "clipsGetChoice");
  DefineFunction("get-char", 'l', PTIF clipsGetChar, "clipsGetChar");
  DefineFunction("set-menu-descriptions", 'v', PTIF clipsSetMenuDescriptions, "clipsSetMenuDescriptions");
  DefineFunction("get-text", 's', PTIF clipsGetText, "clipsGetText");
  DefineFunction("write-information", 'v', PTIF clipsWriteInformation, "clipsWriteInformation");
  DefineFunction("write-prompt", 'v', PTIF clipsWritePrompt, "clipsWritePrompt");
  DefineFunction("write-status-line", 'v', PTIF clipsWriteStatusLine, "clipsWriteStatusLine");
  DefineFunction("display-help", 'v', PTIF clipsDisplayHelpText, "clipsDisplayHelpText");
  DefineFunction("display-file", 'v', PTIF clipsDisplayFile, "clipsDisplayFile");
*/
}
