   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*              CLIPS Version 5.10  07/17/91           */
   /*                                                     */
   /*                INSTANCE COMMAND MODULE              */
   /*******************************************************/

/*************************************************************/
/* Purpose:  CLIPS Kernel Interface Commands for Instances   */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Brian L. Donnell                                     */
/*                                                           */
/* Contributing Programmer(s):                               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/
   
/* =========================================
   *****************************************
               EXTERNAL DEFINITIONS
   =========================================
   ***************************************** */
#include "setup.h"

#if OBJECT_SYSTEM

#include "classcom.h"
#include "classfun.h"
#include "clipsmem.h"
#include "constant.h"
#include "evaluatn.h"
#include "facts.h"
#include "generate.h"
#include "insfun.h"
#include "insquery.h"
#include "msgcom.h"
#include "msgfun.h"
#include "multivar.h"
#include "router.h"
#include "scanner.h"
#include "utility.h"

#define _INSCOM_SOURCE_
#include "inscom.h"

#if (! BLOAD_ONLY) && (! RUN_TIME)
extern struct token ObjectParseToken;
#else
static struct token ObjectParseToken;
#endif

/* =========================================
   *****************************************
                   CONSTANTS
   =========================================
   ***************************************** */
#define ALL_QUALIFIER  "inherit"
#define CLASS_RLN      "of"

/* =========================================
   *****************************************
               MACROS AND TYPES
   =========================================
   ***************************************** */

/* =========================================
   *****************************************
      INTERNALLY VISIBLE FUNCTION HEADERS
   =========================================
   ***************************************** */
#if ANSI_COMPILER
static EXPRESSION *ParseSimpleInstance(EXPRESSION *,char *);

#if ! RUN_TIME
static EXPRESSION *ParseSlotOverrides(char *,int *);
#endif

static long TabulateInstances(int,char *,CLASS_TYPE *,int);
static VOID PrintInstance(char *,INSTANCE_TYPE *,char *);
static INSTANCE_TYPE *CheckInstanceAndSlot(char *,INSTANCE_SLOT **,int);
static INSTANCE_TYPE *CheckMultifieldSlotInstance(char *,int,int);
static INSTANCE_SLOT *FindISlotByName(INSTANCE_TYPE *,char *);

#else
static EXPRESSION *ParseSimpleInstance();

#if ! RUN_TIME
static EXPRESSION *ParseSlotOverrides();
#endif

static long TabulateInstances();
static VOID PrintInstance();
static INSTANCE_TYPE *CheckInstanceAndSlot();
static INSTANCE_TYPE *CheckMultifieldSlotInstance();
static INSTANCE_SLOT *FindISlotByName();

#endif

/* =========================================
   *****************************************
          EXTERNALLY VISIBLE FUNCTIONS
   =========================================
   ***************************************** */

/*********************************************************
  NAME         : SetupInstances
  DESCRIPTION  : Initializes instance Hash Table,
                   Function Parsers, and Data Structures
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : Uses the CLIPS Kernel functions :
                   DefineFunction()
 *********************************************************/
globle VOID SetupInstances()
  {
   InitializeInstanceTable();

#if ! RUN_TIME

   DefineFunction("initialize-instance",'u',
                  PTIF InitializeInstance,"InitializeInstance");
   AddFunctionParser("initialize-instance",ParseInitializeInstance);
   DefineFunction("make-instance",'u',PTIF MakeInstance,"MakeInstance");
   AddFunctionParser("make-instance",ParseInitializeInstance);

   DefineFunction("init-slots",'u',PTIF EvaluateInstanceSlots,
                  "EvaluateInstanceSlots");
   DefineFunction("delete-instance",'v',PTIF DeleteInstance,"DeleteInstance");
   DefineFunction("unmake-instance",'v',PTIF UnmakeInstance,"UnmakeInstance");
#if DEBUGGING_FUNCTIONS
   DefineFunction("instances",'v',PTIF CmdListInstances,"CmdListInstances");
   DefineFunction("ppinstance",'v',PTIF PPInstance,"PPInstance");
#endif
   
   DefineFunction("save-instances",'v',PTIF SaveInstancesCommand,"SaveInstancesCommand");
   DefineFunction("load-instances",'v',PTIF LoadInstancesCommand,"LoadInstancesCommand");
   DefineFunction("symbol-to-instance-name",'u',
                  PTIF SymbolToInstanceName,"SymbolToInstanceName");
   DefineFunction("instance-name-to-symbol",'u',
                  PTIF InstanceNameToSymbol,"InstanceNameToSymbol");
   DefineFunction("instance-address",'u',PTIF GetInstanceAddressCmd,"GetInstanceAddressCmd");
   DefineFunction("instance-addressp",'b',PTIF IsInstanceAddress,"IsInstanceAddress");
   DefineFunction("instance-namep",'b',PTIF IsInstanceName,"IsInstanceName");
   DefineFunction("instance-name",'u',PTIF GetInstanceNameCmd,"GetInstanceNameCmd");
   DefineFunction("instancep",'b',PTIF IsInstance,"IsInstance");
   DefineFunction("instance-existp",'b',PTIF DoesInstanceExist,"DoesInstanceExist");
   DefineFunction("class",'u',PTIF GetInstanceClassCmd,"GetInstanceClassCmd");
   DefineFunction("slot-boundp",'b',PTIF IsSlotBound,"IsSlotBound");
   DefineFunction("slot-existp",'b',PTIF DoesSlotExist,"DoesSlotExist");
   DefineFunction("slot-writablep",'b',PTIF IsSlotWritable,"IsSlotWritable");
   DefineFunction("slot-initablep",'b',PTIF IsSlotInitable,"IsSlotInitable");
   DefineFunction("mv-slot-replace",'u',PTIF MultifieldSlotReplace,"MultifieldSlotReplace");
   DefineFunction("mv-slot-insert",'u',PTIF MultifieldSlotInsert,"MultifieldSlotInsert");
   DefineFunction("mv-slot-delete",'u',PTIF MultifieldSlotDelete,"MultifieldSlotDelete");

#endif

#if DEBUGGING_FUNCTIONS
   AddWatchItem("instances",&WatchInstances,75);
   AddWatchItem("slots",&WatchSlots,74);
#endif
  
   AddCleanupFunction("instances",CleanupInstances,0);
  }

/***************************************************
  NAME         : DestroyAllInstances
  DESCRIPTION  : Deallocates all instances,
                  reinitializes hash table and
                  resets class instance pointers
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : All instances deallocated
  NOTES        : None
 ***************************************************/
globle VOID DestroyAllInstances()
  {
   INSTANCE_TYPE *iptr;
   int svmaintain;
   
   svmaintain = MaintainGarbageInstances;
   MaintainGarbageInstances = TRUE;
   iptr = InstanceList;
   while (iptr != NULL)
     {
      DirectMessage(NULL,DELETE_SYMBOL,iptr,NULL,NULL);
      iptr = iptr->nxt_lst;
      while ((iptr != NULL) ? (iptr->garbage == 1) : FALSE)
        iptr = iptr->nxt_lst;
     }
   MaintainGarbageInstances = svmaintain;
  }

/*******************************************************************
  NAME         : CLIPSDeleteInstance
  DESCRIPTION  : DIRECTLY removes a named instance from the
                   hash table and its class's
                   instance list
  INPUTS       : The instance address (NULL to delete all instances)
  RETURNS      : 1 if successful, 0 otherwise
  SIDE EFFECTS : Instance is deallocated
  NOTES        : C interface for deleting instances
 *******************************************************************/
globle int CLIPSDeleteInstance(iptr)
  VOID *iptr;
  {
   INSTANCE_TYPE *ins,*itmp;
   int success = 1;
   
   if (iptr != NULL)
     return(QuashInstance((INSTANCE_TYPE *) iptr));
   ins = InstanceList;
   while (ins != NULL)
     {
      itmp = ins;
      ins = ins->nxt_lst;
      if (QuashInstance((INSTANCE_TYPE *) itmp) == 0)
        success = 0;
     }
   return(success);
  }
  
/*******************************************************************
  NAME         : CLIPSUnmakeInstance
  DESCRIPTION  : Removes a named instance via message-passing
  INPUTS       : The instance address (NULL to delete all instances)
  RETURNS      : 1 if successful, 0 otherwise
  SIDE EFFECTS : Instance is deallocated
  NOTES        : C interface for deleting instances
 *******************************************************************/
globle int CLIPSUnmakeInstance(iptr)
  VOID *iptr;
  {
   INSTANCE_TYPE *ins;
   int success = 1,svmaintain;
   
   svmaintain = MaintainGarbageInstances;
   MaintainGarbageInstances = TRUE;
   ins = (INSTANCE_TYPE *) iptr;
   if (ins != NULL)
     {
      DirectMessage(NULL,DELETE_SYMBOL,ins,NULL,NULL);
      if (ins->garbage == 0)
        success = 0;
     }
   else
     {
      ins = InstanceList;
      while (ins != NULL)
        {
         DirectMessage(NULL,DELETE_SYMBOL,ins,NULL,NULL);
         if (ins->garbage == 0)
           success = 0;
         ins = ins->nxt_lst;
        }
     }
   MaintainGarbageInstances = svmaintain;
   CleanupInstances();
   return(success);
  }
  
#if DEBUGGING_FUNCTIONS

/*******************************************************************
  NAME         : CmdListInstances
  DESCRIPTION  : Lists all instances associated
                   with a particular class
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : CLIPS syntax : (instances [<class-name> [inherit]])
 *******************************************************************/
globle VOID CmdListInstances()
  {
   VOID *cptr;
   int argno;
   DATA_OBJECT temp;

   if ((argno = ArgCountCheck("instances",NO_MORE_THAN,2)) == -1)
     return;
   if (argno == 0)
     ListInstances(NULL,TRUE);
   else
     {
      if (ArgTypeCheck("instances",1,SYMBOL,&temp) == FALSE)
        return;
      cptr = (VOID *) FindDefclassBySymbol((SYMBOL_HN *) temp.value);
      if (cptr == NULL)
        {
         ClassExistError("instances",ValueToString(temp.value));
         return;
        }
      if (argno == 2)
        {
         if (ArgTypeCheck("instances",2,SYMBOL,&temp) == FALSE)
           return;
         if (strcmp(DOToString(temp),ALL_QUALIFIER) != 0)
           {
            SyntaxErrorMessage("function instances");
            return;
           }
         ListInstances(cptr,TRUE);
        }
      else
        ListInstances(cptr,FALSE);
     }
  }
  
/********************************************************
  NAME         : PPInstance
  DESCRIPTION  : Displays the current slot-values
                   of an instance
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : CLIPS syntax : (ppinstance <instance>)
 ********************************************************/
globle VOID PPInstance()
  {
   INSTANCE_TYPE *ins;

   if (CheckCurrentMessage("ppinstance",TRUE) == FALSE)
     return;
   ins = (INSTANCE_TYPE *) CurrentMessageFrame->value;
   if (ins->garbage == 1)
     return;
   PrintInstance(WDISPLAY,ins,"\n");
   PrintCLIPS(WDISPLAY,"\n");
  }

/***************************************************************
  NAME         : ListInstances
  DESCRIPTION  : Lists instances of classes
  INPUTS       : 1) Address of the class (NULL for all classes)
                 2) A flag indicating whether to print instances
                    of subclasses or not
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : None
 **************************************************************/
globle VOID ListInstances(cls,iflag)
  VOID *cls;
  int iflag;
  {
   int id;
   long count;
   
   if (cls == NULL)
     {
      cls = FindDefclass(OBJECT_CLASS_STRING);
      iflag = TRUE;
     }
   if ((id = GetTraversalID()) == -1)
     return;
   count = TabulateInstances(id,WDISPLAY,(CLASS_TYPE *) cls,iflag);
   ReleaseTraversalID();
   PrintTally(WDISPLAY,count,"instance","instances");
  }  

#endif

/*******************************************************
  NAME         : SaveInstances
  DESCRIPTION  : Saves current instances to named file
  INPUTS       : The name of the output file
  RETURNS      : TRUE if file opened OK, FALSE otherwise
  SIDE EFFECTS : Instances saved to file
  NOTES        : None
 *******************************************************/
globle int SaveInstances(file)
  char *file;
  {
   FILE *sfile;
   char *ilog;
   INSTANCE_TYPE *ins;
   INSTANCE_SLOT *sp;
   DATA_OBJECT *vptr;
   ELEMENT_PTR eptr;
   register int i,j;

   if ((sfile = fopen(file,"w")) == NULL)
     return(FALSE);
   ilog = (char *) sfile;
   SetFastSave(sfile);
   for (ins = InstanceList ; (ins != NULL) && (HaltExecution != TRUE) ; 
        ins = ins->nxt_lst)
     {
      PrintCLIPS(ilog,"([");
      PrintCLIPS(ilog,ValueToString(ins->name));
      PrintCLIPS(ilog,"] of ");
      PrintCLIPS(ilog,ValueToString(ins->cls->name));
      for (i = 0 ; i < ins->cls->islot_cnt ; i++)
        {
         sp = &(ins->slots[i]);
         PrintCLIPS(ilog,"\n   (");
         PrintCLIPS(ilog,ValueToString(sp->desc->name));
         vptr = *(sp->valaddr);
         if (vptr != NULL)
           {
            PrintCLIPS(ilog," ");
            if (vptr->type == INSTANCE)
              {
               PrintCLIPS(WERROR,"WARNING: Instance [");
               PrintCLIPS(WERROR,ValueToString(ins->name));
               PrintCLIPS(WERROR,"] has an instance address in its slot value.\n");
               PrintCLIPS(WERROR,"   Writing out name instead.\n");
               PrintCLIPS(ilog,"[");
               PrintCLIPS(ilog,ValueToString(((INSTANCE_TYPE *) vptr->value)->name));
               PrintCLIPS(ilog,"]");
              }
            else if (vptr->type != MULTIFIELD)
              PrintAtom(ilog,vptr->type,vptr->value);
            else
              {
               for (j = vptr->begin ; j <= vptr->end ; j++)
                 {
                  eptr = &((struct fact *) vptr->value)->atoms[j];
                  if (eptr->type == INSTANCE)
                    {
                     PrintCLIPS(WERROR,"WARNING: Instance [");
                     PrintCLIPS(WERROR,ValueToString(ins->name));
                     PrintCLIPS(WERROR,"] has an instance address in its slot value.\n");
                     PrintCLIPS(WERROR,"   Writing out name instead.\n");
                     PrintCLIPS(ilog,"[");
                     PrintCLIPS(ilog,ValueToString(((INSTANCE_TYPE *) eptr->value)->name));
                     PrintCLIPS(ilog,"]");
                    }
                  else
                    PrintAtom(ilog,eptr->type,eptr->value);
                  if (j != vptr->end)
                    PrintCLIPS(ilog," ");
                 }
              }
           }
         PrintCLIPS(ilog,")");
        }
      PrintCLIPS(ilog,")\n\n");
     }
   fclose(sfile);
   SetFastSave(NULL);
   return(TRUE);
  }
  
/**********************************************************
  NAME         : LoadInstances
  DESCRIPTION  : Loads instances from named file
  INPUTS       : The name of the input file
  RETURNS      : TRUE if file processed OK, FALSE otherwise
  SIDE EFFECTS : Instances loaded from file
  NOTES        : None
 **********************************************************/
globle int LoadInstances(file)
  char *file;
  {
   DATA_OBJECT temp;
   FILE *sfile,*svload;
   char *ilog;
   EXPRESSION *top;
   int svprotect;
   
   if ((sfile = fopen(file,"r")) == NULL)
     return(FALSE);
   svload = GetFastLoad();
   ilog = (char *) sfile;
   SetFastLoad(sfile);
   top = GenConstant(FCALL,(VOID *) FindFunction("make-instance"));
   GetToken(ilog,&ObjectParseToken);
   svprotect = OverrideSlotProtection;
   OverrideSlotProtection = TRUE;
   while ((GetType(ObjectParseToken) != STOP) && (HaltExecution != TRUE))
     {
      if (GetType(ObjectParseToken) != LPAREN)
        {
         PrintCLIPS(WERROR,"Expected '(' to begin instance definition.\n");
         rtn_struct(expr,top);
         fclose(sfile);
         SetFastLoad(svload);
         SetEvaluationError(TRUE);
         OverrideSlotProtection = svprotect;
         return(FALSE);
        }
      if (ParseSimpleInstance(top,ilog) == NULL)
        {
         fclose(sfile);
         SetFastLoad(svload);
         OverrideSlotProtection = svprotect;
         return(FALSE);
        }
      EvaluateExpression(top,&temp);
      ReturnExpression(top->arg_list);
      GetToken(ilog,&ObjectParseToken);
     }
   rtn_struct(expr,top);
   fclose(sfile);
   SetFastLoad(svload);
   OverrideSlotProtection = svprotect;
   return((EvaluationError == TRUE) ? FALSE : TRUE);
  }
  
/*********************************************************
  NAME         : CLIPSMakeInstance
  DESCRIPTION  : C Interface for creating and
                   initializing a class instance
  INPUTS       : The make-instance call string,
                    e.g. "([bill] of man (age 34))"
  RETURNS      : The instance address if instance created,
                    NULL otherwise
  SIDE EFFECTS : Creates the instance and returns
                    the result in caller's buffer
  NOTES        : None
 *********************************************************/
globle VOID *CLIPSMakeInstance(mkstr)
  char *mkstr;
  {
   char *router = "***MKINS***";
   struct token tkn;
   EXPRESSION *top;
   DATA_OBJECT result;

   result.type = SYMBOL;
   result.value = (VOID *) CLIPSFalseSymbol;
   if (OpenStringSource(router,mkstr,0) == 0)
     return(NULL);
   GetToken(router,&tkn);
   if (tkn.type == LPAREN)
     {
      top = GenConstant(FCALL,(VOID *) FindFunction("make-instance"));
      if (ParseSimpleInstance(top,router) != NULL)
        {
         GetToken(router,&tkn);
         if (tkn.type == STOP)
           EvaluateExpression(top,&result);
         else
           PrintCLIPS(WERROR,"Expected ')' to end instance definition.\n");
         ReturnExpression(top);      
        }
     }
   else
     PrintCLIPS(WERROR,"Expected '(' to begin instance definition.\n");
   CloseStringSource(router);
   if ((result.type == SYMBOL) && (result.value == (VOID *) CLIPSFalseSymbol))
     return(NULL);
   return((VOID *) FindInstanceBySymbol((SYMBOL_HN *) result.value));
  }

/***************************************************************
  NAME         : CreateRawInstance
  DESCRIPTION  : Creates an empty of instance of the specified
                   class.  No slot-overrides or class defaults
                   are applied.
  INPUTS       : 1) Address of class
                 2) Name of the new instance
  RETURNS      : The instance address if instance created,
                    NULL otherwise
  SIDE EFFECTS : Old instance of same name deleted (if possible)
  NOTES        : None
 ***************************************************************/
globle VOID *CreateRawInstance(cptr,iname)
  VOID *cptr;
  char *iname;
  {
   return((VOID *) BuildInstance(AddSymbol(iname),((CLASS_TYPE *) cptr)->name));
  }
  
/***************************************************************************
  NAME         : FindInstance
  DESCRIPTION  : Looks up a specified instance in the instance hash table
  INPUTS       : Name-string of the instance
  RETURNS      : The address of the found instance, NULL otherwise
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************************************/
globle VOID *FindInstance(iname)
  char *iname;
  {
   SYMBOL_HN *isym;
   
   isym = FindSymbol(iname);
   if (isym == NULL)
     return(NULL);
   return((VOID *) FindInstanceBySymbol(isym));
  }

/***************************************************************************
  NAME         : ValidInstanceAddress
  DESCRIPTION  : Determines if an instance address is still valid
  INPUTS       : Instance address
  RETURNS      : 1 if the address is still valid, 0 otherwise
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************************************/
globle int ValidInstanceAddress(iptr)
  VOID *iptr;
  {
   return((((INSTANCE_TYPE *) iptr)->garbage == 0) ? 1 : 0);
  }
  
/***************************************************
  NAME         : CLIPSTestSlot
  DESCRIPTION  : Tests if slot is bound, writable or
                    initable
  INPUTS       : 1) The instance address
                 2) The slot name-string
                 3) The code for the test:
                    0 - Does slot exist
                    1 - Is slot bound
                    2 - Is slot writable
                    3 - Is slot initable
  RETURNS      : TRUE (1) if test is true, FALSE (0)
                 if test is false, -1 on errors
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
globle int CLIPSTestSlot(ins,sname,code)
  VOID *ins;
  char *sname;
  int code;
  {
   INSTANCE_SLOT *sp;
   
   if (((INSTANCE_TYPE *) ins)->garbage == 1)
     return(-1);
   sp = FindISlotByName((INSTANCE_TYPE *) ins,sname);
   if (sp != NULL)
     {
      switch(code)
        {
         case 0 : return(TRUE);
         case 1 : return((*(sp->valaddr) != NULL) ? TRUE : FALSE);
         case 2 : return((sp->desc->nowrite == 0) ? TRUE : FALSE);
         case 3 : return(((sp->desc->nowrite == 0) || 
                          (sp->desc->initonly == 1)) ? TRUE : FALSE);
        }
     }
   if (code == 0)
     return(FALSE);
   SetEvaluationError(TRUE);
   return(-1);
  }

/***************************************************
  NAME         : CLIPSGetSlot
  DESCRIPTION  : Gets a slot value
  INPUTS       : 1) Instance adress
                 2) Slot name
                 3) Caller's result buffer
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
globle VOID CLIPSGetSlot(ins,sname,result)
  VOID *ins;
  char *sname;
  DATA_OBJECT *result;
  {
   INSTANCE_SLOT *sp;
   
   if (((INSTANCE_TYPE *) ins)->garbage == 1)
     {
      SetEvaluationError(TRUE);
      result->type = SYMBOL;
      result->value = (VOID *) CLIPSFalseSymbol;
      return;
     }
   sp = FindISlotByName((INSTANCE_TYPE *) ins,sname);
   if ((sp == NULL) ? TRUE : (*(sp->valaddr) == NULL))
     {
      SetEvaluationError(TRUE);
      result->type = SYMBOL;
      result->value = (VOID *) CLIPSFalseSymbol;
      return;
     }
   CopyDataObject(result,*(sp->valaddr));
  }

/*********************************************************
  NAME         : CLIPSPutSlot
  DESCRIPTION  : Gets a slot value
  INPUTS       : 1) Instance address
                 2) Slot name
                 3) Caller's new value buffer
  RETURNS      : TRUE if put successful, FALSE otherwise
  SIDE EFFECTS : None
  NOTES        : None
 *********************************************************/
globle int CLIPSPutSlot(ins,sname,val)
  VOID *ins;
  char *sname;
  DATA_OBJECT *val;
  {
   INSTANCE_SLOT *sp;
   DATA_OBJECT result;
   
   if (((INSTANCE_TYPE *) ins)->garbage == 1)
     {
      SetEvaluationError(TRUE);
      return(FALSE);
     }
   result.type = SYMBOL;
   result.value = (VOID *) CLIPSFalseSymbol;
   sp = FindISlotByName((INSTANCE_TYPE *) ins,sname);
   if (sp == NULL)
     SetEvaluationError(TRUE);
   else if (PutSlotValue((INSTANCE_TYPE *) ins,sp,val,TRUE) != SLOT_ERROR)
     result.value = (VOID *) CLIPSTrueSymbol;
   if ((result.type == SYMBOL) && (result.value == (VOID *) CLIPSFalseSymbol))
     return(FALSE);
   return(TRUE);
  }

/***************************************************
  NAME         : GetInstanceName
  DESCRIPTION  : Returns name of instance
  INPUTS       : Pointer to instance
  RETURNS      : Name of instance
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
globle char *GetInstanceName(iptr)
  VOID *iptr;
  {
   if (((INSTANCE_TYPE *) iptr)->garbage == 1)
     return(NULL);
   return(ValueToString(((INSTANCE_TYPE *) iptr)->name));
  }
  
/***************************************************
  NAME         : GetInstanceClass
  DESCRIPTION  : Returns class of instance
  INPUTS       : Pointer to instance
  RETURNS      : Pointer to class of instance
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
globle VOID *GetInstanceClass(iptr)
  VOID *iptr;
  {
   if (((INSTANCE_TYPE *) iptr)->garbage == 1)
     return(NULL);
   return(ValueToString(((INSTANCE_TYPE *) iptr)->cls));
  }
  
/***************************************************
  NAME         : GetNextInstance
  DESCRIPTION  : Returns next instance in list
                 (or first instance in list)
  INPUTS       : Pointer to previous instance
                 (or NULL to get first instance)
  RETURNS      : The next instance or first instance
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
globle VOID *GetNextInstance(iptr)
  VOID *iptr;
  {
   if (iptr == NULL)
     return((VOID *) InstanceList);
   if (((INSTANCE_TYPE *) iptr)->garbage == 1)
     return(NULL);
   return((VOID *) ((INSTANCE_TYPE *) iptr)->nxt_lst);
  }

/***************************************************
  NAME         : GetNextInstanceInClass
  DESCRIPTION  : Finds next instance of class
                 (or first instance of class)
  INPUTS       : 1) Class address
                 2) Instance address
                    (NULL to get first instance)
  RETURNS      : The next or first class instance
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
globle VOID *GetNextInstanceInClass(cptr,iptr)
  VOID *cptr,*iptr;
  {
   if (iptr == NULL)
     return((VOID *) ((CLASS_TYPE *) cptr)->instances);
   if (((INSTANCE_TYPE *) iptr)->garbage == 1)
     return(NULL);
   return((VOID *) ((INSTANCE_TYPE *) iptr)->nxt_cls);
  }

/***************************************************
  NAME         : GetInstancePPForm
  DESCRIPTION  : Writes slot names and values to
                  caller's buffer
  INPUTS       : 1) Caller's buffer
                 2) Size of buffer (not including
                    space for terminating '\0')
                 3) Instance address
  RETURNS      : Nothing useful
  SIDE EFFECTS : Caller's buffer written
  NOTES        : None
 ***************************************************/
globle VOID GetInstancePPForm(buf,buflen,iptr)
  char *buf;
  int buflen;
  VOID *iptr;
  {
   char *pbuf = "***InstancePPForm***";
   
   if (((INSTANCE_TYPE *) iptr)->garbage == 1)
     return;
   if (OpenStringDestination(pbuf,buf,buflen+1) == 0)
     return;
   PrintInstance(pbuf,(INSTANCE_TYPE *) iptr," ");
   CloseStringDestination(pbuf);
  }

/*********************************************************
  NAME         : GetInstanceClassCmd
  DESCRIPTION  : Returns the class of an instance
  INPUTS       : Caller's result buffer
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax : (class <object>)
                 Can also be called by (type <object>)
                   if you have generic functions installed
 *********************************************************/
globle VOID GetInstanceClassCmd(result)
  DATA_OBJECT *result;
  {
   INSTANCE_TYPE *ins;
   char *func;
   DATA_OBJECT temp;
   
   func = ValueToString(((struct FunctionDefinition *)
                       CurrentExpression->value)->callFunctionName);
   result->type = SYMBOL;
   result->value = (VOID *) CLIPSFalseSymbol;
   if (ArgCountCheck(func,EXACTLY,1) != 1)
     {
      SetEvaluationError(TRUE);
      return;
     }
   EvaluateExpression(GetFirstArgument(),&temp);
   if (temp.type == INSTANCE)
     {
      ins = (INSTANCE_TYPE *) temp.value;
      if (ins->garbage == 1)
        {
         StaleInstanceAddress(func);
         SetEvaluationError(TRUE);
         return;
        }
      result->value = (VOID *) ins->cls->name;
     }
   else if (temp.type == INSTANCE_NAME)
     {
      ins = FindInstanceBySymbol((SYMBOL_HN *) temp.value);
      if (ins == NULL)
        {
         NoInstanceError(ValueToString(temp.value),func);
         return;
        }
      result->value = (VOID *) ins->cls->name;
     }
   else
     {
      switch (temp.type)
        {
         case INTEGER  :
         case FLOAT    :
         case SYMBOL     :
         case STRING   :
         case MULTIFIELD : 
         case EXTERNAL_ADDRESS : 
                          result->value = (VOID *) 
                                           PrimitiveClassMap[temp.type]->name;
                         return;
         default       : PrintCLIPS(WERROR,"Undefined type in function ");
                         PrintCLIPS(WERROR,func);
                         PrintCLIPS(WERROR,".\n");
                         SetEvaluationError(TRUE);
        }
     }
  }

#if ! RUN_TIME

/*************************************************************************************
  NAME         : ParseInitializeInstance
  DESCRIPTION  : Parses initialize-instance and make-instance function
                   calls into an EXPRESSION form that
                   can later be evaluated with EvaluateExpression()
  INPUTS       : 1) The address of the top node of the expression
                    containing the initialize-instance function call
                 2) The logical name of the input source
  RETURNS      : The address of the modified expression, or NULL
                    if there is an error
  SIDE EFFECTS : The expression is enhanced to include all
                    aspects of the initialize-instance call
                    (slot-overrides etc.)
                 The "top" expression is deleted on errors.
  NOTES        : This function parses a initialize-instance call into
                 an expression of the following form :
                 
                 (initialize-instance <instance-name> <slot-override>*)
                  where <slot-override> ::= (<slot-name> <expression>+)
                  
                  goes to -->
                  
                  initialize-instance
                      |
                      V
                  <instance or name>-><slot-name>-><dummy-node>...
                                                      |
                                                      V
                                               <value-expression>...
                                             
                  (make-instance <instance> of <class> <slot-override>*)
                  where <slot-override> ::= (<slot-name> <expression>+)
                  
                  goes to -->
                  
                  make-instance
                      |
                      V
                  <instance-name>-><class-name>-><slot-name>-><dummy-node>...
                                                                 |
                                                                 V
                                                          <value-expression>...

 *************************************************************************************/
globle EXPRESSION *ParseInitializeInstance(top,read_source)
  EXPRESSION *top;
  char *read_source;
  {
   int error;
   
   IncrementIndentDepth(3);
   error = FALSE;
   if (top->type == KUNKNOWN)
     top->type = FCALL;
   else
     SavePPBuffer(" ");
   top->arg_list = ArgumentParse(read_source,&error);
   if (error == TRUE)
     {
      ReturnExpression(top);
      DecrementIndentDepth(3);
      return(NULL);
     }
   else if (top->arg_list == NULL)
     {
      SyntaxErrorMessage("instance");
      ReturnExpression(top);
      DecrementIndentDepth(3);
      SetEvaluationError(TRUE);
      return(NULL);
     }
   SavePPBuffer(" ");
   if (top->value == (VOID *) FindFunction("make-instance"))
     {
      GetToken(read_source,&ObjectParseToken);
      if ((GetType(ObjectParseToken) != SYMBOL) ? TRUE :
          (strcmp(CLASS_RLN,DOToString(ObjectParseToken)) != 0))
        {
         SyntaxErrorMessage("make-instance");
         SetEvaluationError(TRUE);
         ReturnExpression(top);
         DecrementIndentDepth(3);
         return(NULL);
        }
      SavePPBuffer(" ");
      top->arg_list->next_arg = ArgumentParse(read_source,&error);
      if (error == TRUE)
        {
         ReturnExpression(top);
         DecrementIndentDepth(3);
         return(NULL);
        }
      else if (top->arg_list->next_arg == NULL)
        {
         SyntaxErrorMessage("instance class");
         ReturnExpression(top);
         DecrementIndentDepth(3);
         SetEvaluationError(TRUE);
         return(NULL);
        }
      PPCRAndIndent();
      GetToken(read_source,&ObjectParseToken);
      top->arg_list->next_arg->next_arg = 
                  ParseSlotOverrides(read_source,&error);
     }
   else
     {
      PPCRAndIndent();
      GetToken(read_source,&ObjectParseToken);
      top->arg_list->next_arg = ParseSlotOverrides(read_source,&error);
     }
   if (error == TRUE)
     {
      ReturnExpression(top);
      DecrementIndentDepth(3);
      return(NULL);
     }
   if (GetType(ObjectParseToken) != RPAREN)
     {
      SyntaxErrorMessage("slot-override");
      ReturnExpression(top);
      DecrementIndentDepth(3);
      SetEvaluationError(TRUE);
      return(NULL);
     }
   DecrementIndentDepth(3);
   return(top);
  }

#endif

/******************************************************
  NAME         : DeleteInstance
  DESCRIPTION  : Removes a named instance from the
                   hash table and its class's
                   instance list
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : Instance is deallocated
  NOTES        : This is an internal function that
                   only be called by a handler
                 CurrentMessageFrame[0] is object being
                   passed a message
 ******************************************************/
globle VOID DeleteInstance()
  {
   if (CheckCurrentMessage("delete-instance",TRUE) == TRUE)
     QuashInstance((INSTANCE_TYPE *) CurrentMessageFrame->value);
  }
  
/*******************************************************************
  NAME         : UnmakeInstance
  DESCRIPTION  : Uses message-passing to delete the
                   specified instance
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : Instance is deallocated
  NOTES        : Syntax: (unmake-instance <instance-expression> | *)
 *******************************************************************/
globle VOID UnmakeInstance()
  {
   DATA_OBJECT val;
   VOID *ins;
   
   if (ArgCountCheck("unmake-instance",EXACTLY,1) == -1)
     return;
   RtnUnknown(1,&val);
   if (val.type == INSTANCE_NAME)
     {
      ins = (VOID *) FindInstanceBySymbol((SYMBOL_HN *) val.value);
      if (ins == NULL)
        {
         NoInstanceError(DOToString(val),"unmake-instance");
         return;
        }
     }
   else if (val.type == INSTANCE)
     ins = val.value;
   else if (val.type == SYMBOL)
     {
      ins = (VOID *) FindInstanceBySymbol((SYMBOL_HN *) val.value);
      if ((ins == NULL) ? (strcmp(DOToString(val),"*") != 0) : FALSE)
        {
         NoInstanceError(DOToString(val),"unmake-instance");
         return;
        }
     }
   else
     {
      ExpectedTypeError("unmake-instance",1,"instance name, instance address or symbol");
      SetEvaluationError(TRUE);
      return;
     }
   if (ins == NULL)
     DestroyAllInstances();
   else if (((INSTANCE_TYPE *) ins)->garbage == 1)
     {
      StaleInstanceAddress("unmake-instance");
      SetEvaluationError(TRUE);
      return;
     }
   else
     DirectMessage(NULL,DELETE_SYMBOL,(INSTANCE_TYPE *) ins,NULL,NULL);
  }

/******************************************************
  NAME         : SaveInstancesCommand
  DESCRIPTION  : CLIPS interface for saving
                   current instances to a file
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : Instances saved to named file
  NOTES        : CLIPS Syntax : (save-instances <file>)
 ******************************************************/
globle VOID SaveInstancesCommand()
  {
   char *file_found;
   DATA_OBJECT temp;

   if (ArgCountCheck("save-instances",EXACTLY,1) == -1) return;

   if (ArgTypeCheck("save-instances",1,SYMBOL_OR_STRING,&temp) == FALSE) return;

   file_found = DOToString(temp);

   if (SaveInstances(file_found) == FALSE) 
     OpenErrorMessage("save-instances",file_found);
  }

/******************************************************
  NAME         : LoadInstancesCommand
  DESCRIPTION  : CLIPS interface for loading
                   instances from a file
  INPUTS       : None
  RETURNS      : Nothing useful
  SIDE EFFECTS : Instances loaded from named file
  NOTES        : CLIPS Syntax : (load-instances <file>)
 ******************************************************/
globle VOID LoadInstancesCommand()
  {
   char *file_found;
   DATA_OBJECT temp;

   if (ArgCountCheck("load-instances",EXACTLY,1) == -1) return;

   if (ArgTypeCheck("load-instances",1,SYMBOL_OR_STRING,&temp) == FALSE) return;

   file_found = DOToString(temp);

   if (LoadInstances(file_found) == FALSE) 
     {
      PrintCLIPS(WERROR,"load-instances could not completely process file ");
      PrintCLIPS(WERROR,file_found);
      PrintCLIPS(WERROR,".\n");
      SetEvaluationError(TRUE);
     }
  }

/*****************************************************************
  NAME         : SymbolToInstanceName
  DESCRIPTION  : Converts a symbol from type SYMBOL
                   to type INSTANCE_NAME
  INPUTS       : The address of the value buffer
  RETURNS      : The new INSTANCE_NAME symbol
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax : (symbol-to-instance-name <symbol>)
 *****************************************************************/
globle VOID SymbolToInstanceName(result)
  DATA_OBJECT *result;
  {
   if (ArgCountCheck("symbol-to-instance-name",EXACTLY,1) != 1)
     {
      SetpType(result,SYMBOL);
      SetpValue(result,(VOID *) CLIPSFalseSymbol);
      return;
     }
   if (ArgTypeCheck("symbol-to-instance-name",1,SYMBOL,result) == FALSE)
     {
      SetpType(result,SYMBOL);
      SetpValue(result,(VOID *) CLIPSFalseSymbol);
      return;
     }
   SetpType(result,INSTANCE_NAME);
  }
  
/*****************************************************************
  NAME         : InstanceNameToSymbol
  DESCRIPTION  : Converts a symbol from type INSTANCE_NAME
                   to type SYMBOL
  INPUTS       : The address of the value buffer
  RETURNS      : The new SYMBOL symbol
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax : (instance-name-to-symbol <iname>)
 *****************************************************************/
globle VOID InstanceNameToSymbol(result)
  DATA_OBJECT *result;
  {
   if (ArgCountCheck("instance-name-to-symbol",EXACTLY,1) != 1)
     {
      SetpType(result,SYMBOL);
      SetpValue(result,(VOID *) CLIPSFalseSymbol);
      return;
     }
   if (ArgTypeCheck("instance-name-to-symbol",1,INSTANCE_NAME,result) == FALSE)
     {
      SetpType(result,SYMBOL);
      SetpValue(result,(VOID *) CLIPSFalseSymbol);
      return;
     }
   SetpType(result,SYMBOL);
  }
  
/*****************************************************************
  NAME         : GetInstanceAddressCmd
  DESCRIPTION  : Returns the address of an instance
  INPUTS       : The address of the value buffer
  RETURNS      : Nothing useful
  SIDE EFFECTS : Stores instance address in caller's buffer
  NOTES        : CLIPS Syntax : (instance-address <instance-name>)
 *****************************************************************/
globle VOID GetInstanceAddressCmd(result)
  DATA_OBJECT *result;
  {
   INSTANCE_TYPE *ins;
   DATA_OBJECT temp;
   
   result->type = SYMBOL;
   result->value = (VOID *) CLIPSFalseSymbol;
   if (ArgCountCheck("instance-address",EXACTLY,1) != 1)
     return;
   if (ArgTypeCheck("instance-address",1,INSTANCE_OR_INSTANCE_NAME,&temp) == FALSE)
     return;
   if (temp.type == INSTANCE)
     {
      ins = (INSTANCE_TYPE *) temp.value;
      if (ins->garbage == 0)
        {
         result->type = INSTANCE;
         result->value = temp.value;
        }
      else
        {
         StaleInstanceAddress("instance-address");
         SetEvaluationError(TRUE);
        }
     }
   else
     {
      ins = FindInstanceBySymbol((SYMBOL_HN *) temp.value);
      if (ins != NULL)
        {
         result->type = INSTANCE;
         result->value = (VOID *) ins;
        }
      else
        NoInstanceError(ValueToString(temp.value),"instance-address");
     }
  }
  
/***************************************************************
  NAME         : GetInstanceNameCmd
  DESCRIPTION  : Gets the name of an INSTANCE
  INPUTS       : The address of the value buffer
  RETURNS      : The INSTANCE_NAME symbol
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax : (instance-name <instance>)
 ***************************************************************/
globle VOID GetInstanceNameCmd(result)
  DATA_OBJECT *result;
  {
   INSTANCE_TYPE *ins;
   DATA_OBJECT temp;
   
   result->type = SYMBOL;
   result->value = (VOID *) CLIPSFalseSymbol;
   if (ArgCountCheck("instance-name",EXACTLY,1) != 1)
     {
      SetEvaluationError(TRUE);
      return;
     }
   if (ArgTypeCheck("instance-name",1,INSTANCE_OR_INSTANCE_NAME,&temp) == FALSE)
     return;
   if (temp.type == INSTANCE)
     {
      ins = (INSTANCE_TYPE *) temp.value;
      if (ins->garbage == 1)
        {
         StaleInstanceAddress("instance-name");
         SetEvaluationError(TRUE);
         return;
        }
     }
   else
     {
      ins = FindInstanceBySymbol((SYMBOL_HN *) temp.value);
      if (ins == NULL)
        {
         NoInstanceError(ValueToString(temp.value),"instance-name");
         return;
        }
     }
   result->type = INSTANCE_NAME;
   result->value = (VOID *) ins->name;
  }
  
/*******************************************************
  NAME         : IsInstanceAddress
  DESCRIPTION  : Determines if a value is of type INSTANCE
  INPUTS       : None
  RETURNS      : TRUE if type INSTANCE, FALSE otherwise
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax : (instance-addressp <arg>)
 *******************************************************/
globle BOOLEAN IsInstanceAddress()
  {
   DATA_OBJECT temp;

   if (ArgCountCheck("instance-addressp",EXACTLY,1) != 1)
     return(FALSE);
   EvaluateExpression(GetFirstArgument(),&temp);
   return((GetType(temp) == INSTANCE) ? TRUE : FALSE);
  }
  
/**************************************************************
  NAME         : IsInstanceName
  DESCRIPTION  : Determines if a value is of type INSTANCE_NAME
  INPUTS       : None
  RETURNS      : TRUE if type INSTANCE_NAME, FALSE otherwise
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax : (instance-namep <arg>)
 **************************************************************/
globle BOOLEAN IsInstanceName()
  {
   DATA_OBJECT temp;

   if (ArgCountCheck("instance-namep",EXACTLY,1) != 1)
     return(FALSE);
   EvaluateExpression(GetFirstArgument(),&temp);
   return((GetType(temp) == INSTANCE_NAME) ? TRUE : FALSE);
  }
  
/*******************************************************
  NAME         : IsInstance
  DESCRIPTION  : Determines if a value is of type INSTANCE
                   or INSTANCE_NAME
  INPUTS       : None
  RETURNS      : TRUE if type INSTANCE_NAME or INSTANCE,
                     FALSE otherwise
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax : (instancep <arg>)
 *******************************************************/
globle BOOLEAN IsInstance()
  {
   DATA_OBJECT temp;

   if (ArgCountCheck("instancep",EXACTLY,1) != 1)
     return(FALSE);
   EvaluateExpression(GetFirstArgument(),&temp);
   if ((GetType(temp) == INSTANCE_NAME) || (GetType(temp) == INSTANCE))
     return(TRUE);
   return(FALSE);
  }
  
/********************************************************
  NAME         : DoesInstanceExist
  DESCRIPTION  : Determines if an instance exists
  INPUTS       : None
  RETURNS      : TRUE if instance exists, FALSE otherwise
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax : (instance-existp <arg>)
 ********************************************************/
globle BOOLEAN DoesInstanceExist()
  {
   DATA_OBJECT temp;

   if (ArgCountCheck("instance-existp",EXACTLY,1) != 1)
     return(FALSE);
   EvaluateExpression(GetFirstArgument(),&temp);
   if (temp.type == INSTANCE)
     return((((INSTANCE_TYPE *) temp.value)->garbage == 0) ? TRUE : FALSE);
   if ((temp.type == INSTANCE_NAME) || (temp.type == SYMBOL))
     return((FindInstanceBySymbol((SYMBOL_HN *) temp.value) != NULL) ? TRUE : FALSE);
   ExpectedTypeError("instance-existp",1,"instance name, instance address or symbol");
   SetEvaluationError(TRUE);
   return(FALSE);
  }
  
/***************************************************************
  NAME         : DoesSlotExist
  DESCRIPTION  : Determines if a slot is present in an instance
  INPUTS       : None
  RETURNS      : TRUE if the slot exists, FALSE otherwise
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax : (slot-existp <instance> <slot>)
 ***************************************************************/
globle BOOLEAN DoesSlotExist()
  {
   INSTANCE_SLOT *sp;
   
   if (CheckInstanceAndSlot("slot-existp",&sp,TRUE) == NULL)
     return(FALSE);
   return((sp != NULL) ? TRUE : FALSE);
  }
  
/**********************************************************************
  NAME         : IsSlotBound
  DESCRIPTION  : Determines if an existing slot has a value bound to it
  INPUTS       : None
  RETURNS      : TRUE if the slot is bound, FALSE otherwise
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax : (slot-boundp <instance> <slot>)
 **********************************************************************/
globle BOOLEAN IsSlotBound()
  {
   INSTANCE_SLOT *sp;
   
   if (CheckInstanceAndSlot("slot-boundp",&sp,FALSE) == NULL)
     return(FALSE);
   return((*(sp->valaddr) != NULL) ? TRUE : FALSE);
  }
  
/**********************************************************************
  NAME         : IsSlotWritable
  DESCRIPTION  : Determines if an existing slot can be written to
  INPUTS       : None
  RETURNS      : TRUE if the slot is writable, FALSE otherwise
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax : (slot-writablep <instance> <slot>)
 **********************************************************************/
globle BOOLEAN IsSlotWritable()
  {
   INSTANCE_SLOT *sp;
   
   if (CheckInstanceAndSlot("slot-writablep",&sp,FALSE) == NULL)
     return(FALSE);
   return((sp->desc->nowrite == 0) ? TRUE : FALSE);
  }
  
/**********************************************************************
  NAME         : IsSlotInitable
  DESCRIPTION  : Determines if an existing slot can be initialized
                   via an init message-handler or slot-override
  INPUTS       : None
  RETURNS      : TRUE if the slot is writable, FALSE otherwise
  SIDE EFFECTS : None
  NOTES        : CLIPS Syntax : (slot-initablep <instance> <slot>)
 **********************************************************************/
globle BOOLEAN IsSlotInitable()
  {
   INSTANCE_SLOT *sp;
   
   if (CheckInstanceAndSlot("slot-initablep",&sp,FALSE) == NULL)
     return(FALSE);
   if ((sp->desc->nowrite == 0) || (sp->desc->initonly == 1))
     return(TRUE);
   return(FALSE);
  }
  
/***********************************************************************************
  NAME         : MultifieldSlotReplace
  DESCRIPTION  : Allows user to replace a specified field of a multi-value slot
                 The slot is directly read (w/o a get- message) and the new
                   slot-value is placed via a put- message.
                 This function is not valid for single-value slots.
  INPUTS       : Caller's result buffer
  RETURNS      : TRUE if multi-value slot successfully modified, FALSE otherwise
  SIDE EFFECTS : Put messsage sent for slot
  NOTES        : CLIPS Syntax : (mv-slot-replace <instance> <slot> 
                                 <range-begin> <range-end> <value>)
 ***********************************************************************************/
globle VOID MultifieldSlotReplace(result)
  DATA_OBJECT *result;
  {
   DATA_OBJECT newval,newseg;
   INSTANCE_TYPE *ins;
   INSTANCE_SLOT *sp;
   int rb,re;
   EXPRESSION arg;
   
   result->type = SYMBOL;
   result->value = (VOID *) CLIPSFalseSymbol;
   ins = CheckMultifieldSlotInstance("mv-slot-replace",AT_LEAST,5);
   if (ins == NULL)
     return;
   sp = CheckMultifieldSlotModify(REPLACE,"mv-slot-replace",ins,
                            GetFirstArgument()->next_arg,&rb,&re,&newval);
   if (sp == NULL)
     return;
   if (ReplaceMultiValueField(&newseg,*(sp->valaddr),
                              rb,re,&newval,"mv-slot-replace") == FALSE)
     return;
   arg.type = MULTIFIELD;
   arg.value = (VOID *) &newseg;
   arg.next_arg = NULL;
   arg.arg_list = NULL;
   DirectMessage("put-",sp->desc->name,ins,result,&arg);
  }
  
/***********************************************************************************
  NAME         : MultifieldSlotInsert
  DESCRIPTION  : Allows user to insert a specified field of a multi-value slot
                 The slot is directly read (w/o a get- message) and the new
                   slot-value is placed via a put- message.
                 This function is not valid for single-value slots.
  INPUTS       : Caller's result buffer
  RETURNS      : TRUE if multi-value slot successfully modified, FALSE otherwise
  SIDE EFFECTS : Put messsage sent for slot
  NOTES        : CLIPS Syntax : (mv-slot-insert <instance> <slot> <index> <value>)
 ***********************************************************************************/
globle VOID MultifieldSlotInsert(result)
  DATA_OBJECT *result;
  {
   DATA_OBJECT newval,newseg;
   INSTANCE_TYPE *ins;
   INSTANCE_SLOT *sp;
   int index;
   EXPRESSION arg;
   
   result->type = SYMBOL;
   result->value = (VOID *) CLIPSFalseSymbol;
   ins = CheckMultifieldSlotInstance("mv-slot-insert",AT_LEAST,4);
   if (ins == NULL)
     return;
   sp = CheckMultifieldSlotModify(INSERT,"mv-slot-insert",ins,
                            GetFirstArgument()->next_arg,&index,NULL,&newval);
   if (sp == NULL)
     return;
   if (InsertMultiValueField(&newseg,*(sp->valaddr),
                             index,&newval,"mv-slot-insert") == FALSE)
     return;
   arg.type = MULTIFIELD;
   arg.value = (VOID *) &newseg;
   arg.next_arg = NULL;
   arg.arg_list = NULL;
   DirectMessage("put-",sp->desc->name,ins,result,&arg);
  }
  
/***********************************************************************************
  NAME         : MultifieldSlotDelete
  DESCRIPTION  : Allows user to delete a specified field of a multi-value slot
                 The slot is directly read (w/o a get- message) and the new
                   slot-value is placed via a put- message.
                 This function is not valid for single-value slots.
  INPUTS       : Caller's result buffer
  RETURNS      : TRUE if multi-value slot successfully modified, FALSE otherwise
  SIDE EFFECTS : Put messsage sent for slot
  NOTES        : CLIPS Syntax : (mv-slot-delete <instance> <slot>
                                 <range-begin> <range-end>)
 ***********************************************************************************/
globle VOID MultifieldSlotDelete(result)
  DATA_OBJECT *result;
  {
   DATA_OBJECT newseg;
   INSTANCE_TYPE *ins;
   INSTANCE_SLOT *sp;
   int rb,re;
   EXPRESSION arg;
   
   result->type = SYMBOL;
   result->value = (VOID *) CLIPSFalseSymbol;
   ins = CheckMultifieldSlotInstance("mv-slot-delete",EXACTLY,4);
   if (ins == NULL)
     return;
   sp = CheckMultifieldSlotModify(DELETE,"mv-slot-delete",ins,
                            GetFirstArgument()->next_arg,&rb,&re,NULL);
   if (sp == NULL)
     return;
   if (DeleteMultiValueField(&newseg,*(sp->valaddr),rb,re,"mv-slot-delete") == FALSE)
     return;
   arg.type = MULTIFIELD;
   arg.value = (VOID *) &newseg;
   arg.next_arg = NULL;
   arg.arg_list = NULL;
   DirectMessage("put-",sp->desc->name,ins,result,&arg);
  }
  
/* =========================================
   *****************************************
          INTERNALLY VISIBLE FUNCTIONS
   =========================================
   ***************************************** */

/****************************************************************************
  NAME         : ParseSimpleInstance
  DESCRIPTION  : Parses instances from file for load-instances
                   into an EXPRESSION forms that
                   can later be evaluated with EvaluateExpression()
  INPUTS       : 1) The address of the top node of the expression
                    containing the make-instance function call
                 2) The logical name of the input source
  RETURNS      : The address of the modified expression, or NULL
                    if there is an error
  SIDE EFFECTS : The expression is enhanced to include all
                    aspects of the make-instance call
                    (slot-overrides etc.)
                 The "top" expression is deleted on errors.
  NOTES        : The name, class, values etc. must be constants.

                 This function parses a make-instance call into
                 an expression of the following form :
                                             
                  (make-instance <instance> of <class> <slot-override>*)
                  where <slot-override> ::= (<slot-name> <expression>+)
                  
                  goes to -->
                  
                  make-instance
                      |
                      V
                  <instance-name>-><class-name>-><slot-name>-><dummy-node>...
                                                                 |
                                                                 V
                                                          <value-expression>...

 ****************************************************************************/
static EXPRESSION *ParseSimpleInstance(top,read_source)
  EXPRESSION *top;
  char *read_source;
  {
   EXPRESSION *exp,*vals,*vbot,*tval;
   int type;

   GetToken(read_source,&ObjectParseToken);
   if ((GetType(ObjectParseToken) != INSTANCE_NAME) &&
       (GetType(ObjectParseToken) != SYMBOL))
     {
      SyntaxErrorMessage("make-instance");
      SetEvaluationError(TRUE);
      ReturnExpression(top);
      return(NULL);
     }
   top->arg_list = GenConstant(INSTANCE_NAME,
                               (VOID *) ClipsGetValue(ObjectParseToken));
   GetToken(read_source,&ObjectParseToken);
   if ((GetType(ObjectParseToken) != SYMBOL) ? TRUE :
       (strcmp(CLASS_RLN,DOToString(ObjectParseToken)) != 0))
     {
      SyntaxErrorMessage("make-instance");
      SetEvaluationError(TRUE);
      ReturnExpression(top);
      return(NULL);
     }
   GetToken(read_source,&ObjectParseToken);
   if (GetType(ObjectParseToken) != SYMBOL)
     {
      SyntaxErrorMessage("make-instance");
      SetEvaluationError(TRUE);
      ReturnExpression(top);
      return(NULL);
     }
   top->arg_list->next_arg = GenConstant(SYMBOL,
                               (VOID *) ClipsGetValue(ObjectParseToken));

   exp = top->arg_list->next_arg;
   GetToken(read_source,&ObjectParseToken);
   while (GetType(ObjectParseToken) == LPAREN)
     {
      GetToken(read_source,&ObjectParseToken);
      if (GetType(ObjectParseToken) != SYMBOL)
        {
         SyntaxErrorMessage("slot-override");
         SetEvaluationError(TRUE);
         ReturnExpression(top);
         return(NULL);
        }
      exp->next_arg = GenConstant(SYMBOL,(VOID *) ClipsGetValue(ObjectParseToken));
      exp->next_arg->next_arg = GenConstant(SYMBOL,(VOID *) CLIPSTrueSymbol);
      exp = exp->next_arg->next_arg;
      GetToken(read_source,&ObjectParseToken);
      vals = vbot = NULL;
      while (GetType(ObjectParseToken) != RPAREN)
        {
         type = GetType(ObjectParseToken);
         if ((type != SYMBOL) && (type != STRING) &&
             (type != FLOAT) && (type != INTEGER) && (type != INSTANCE_NAME))
           {
            SyntaxErrorMessage("slot-override");
            SetEvaluationError(TRUE);
            ReturnExpression(top);
            ReturnExpression(vals);
            return(NULL);
           }
         tval = GenConstant(type,(VOID *) ClipsGetValue(ObjectParseToken));
         if (vals == NULL)
           vals = tval;
         else
           vbot->next_arg = tval;
         vbot = tval;
         GetToken(read_source,&ObjectParseToken);
        }
      exp->arg_list = vals;
      GetToken(read_source,&ObjectParseToken);
     }
   if (GetType(ObjectParseToken) != RPAREN)
     {
      SyntaxErrorMessage("slot-override");
      ReturnExpression(top);
      SetEvaluationError(TRUE);
      return(NULL);
     }
   return(top);   
  }

#if ! RUN_TIME

/********************************************************************************
  NAME         : ParseSlotOverrides
  DESCRIPTION  : Forms expressions for slot-overrides
  INPUTS       : 1) The logical name of the input
                 2) Caller's buffer for error flkag
  RETURNS      : Address override expressions, NULL
                   if none or error.
  SIDE EFFECTS : Slot-expression built
                 Caller's error flag set
  NOTES        : <slot-override> ::= (<slot-name> <value>*)*
  
                 goes to
                 
                 <slot-name> --> <dummy-node> --> <slot-name> --> <dummy-node>...
                                       |
                                       V
                               <value-expression> --> <value-expression> --> ...
                               
                 Assumes first token has already been scanned
 ********************************************************************************/
static EXPRESSION *ParseSlotOverrides(read_source,error)
  char *read_source;
  int *error;
  {
   EXPRESSION *top = NULL,*bot = NULL,*exp;
   
   while (GetType(ObjectParseToken) == LPAREN)
     {
      *error = FALSE;
      exp = ArgumentParse(read_source,error);
      if (*error == TRUE)
        {
         ReturnExpression(top);
         return(NULL);
        }
      else if (exp == NULL)
        {
         SyntaxErrorMessage("slot-override");
         *error = TRUE;
         ReturnExpression(top);
         SetEvaluationError(TRUE);
         return(NULL);
        }
      exp->next_arg = GenConstant(SYMBOL,(VOID *) CLIPSTrueSymbol);
      if (CollectArguments(exp->next_arg,read_source) == NULL)
        {
         *error = TRUE;
         ReturnExpression(top);
         return(NULL);
        }
      if (top == NULL)
        top = exp;
      else
        bot->next_arg = exp;
      bot = exp->next_arg;
      PPCRAndIndent();
      GetToken(read_source,&ObjectParseToken);
     }
   PPBackup();
   PPBackup();
   SavePPBuffer(ObjectParseToken.print_rep);
   return(top);
  }

#endif

/******************************************************
  NAME         : TabulateInstances
  DESCRIPTION  : Displays all instances for a class
  INPUTS       : 1) The traversal id for the classes
                 2) The logical name of the output
                 3) The class address
                 4) A flag indicating whether to
                    print out instances of subclasses
                    or not.
  RETURNS      : The number of instances (including
                    subclasses' instances)
  SIDE EFFECTS : None
  NOTES        : None
 ******************************************************/
static long TabulateInstances(id,log,cls,iflag)
  int id;
  char *log;
  CLASS_TYPE *cls;
  int iflag;
  {
   INSTANCE_TYPE *ins;
   CLASS_LINK *sub;
   long count = 0;
   int pflag;

   if (TestTraversalID(cls->tvids,id))
     return(0L);
   SetTraversalID(cls->tvids,id);
   pflag = iflag;
   for (ins = cls->instances ; ins != NULL ; ins = ins->nxt_cls)
     {
      if (pflag != FALSE)
        {
         pflag = FALSE;
         PrintCLIPS(log,"\nCLASS ");
         PrintCLIPS(log,ValueToString(cls->name));
         PrintCLIPS(log,":\n");
        }
      count++;
      PrintCLIPS(log,ValueToString(ins->name));
      PrintCLIPS(log,"\n");
     }
   if (iflag != FALSE)
     {
      for (sub = cls->sublink ; sub != NULL ; sub = sub->nxt)
        count += TabulateInstances(id,log,sub->cls,iflag);
     }
   return(count);
  }
  
/***************************************************
  NAME         : PrintInstance
  DESCRIPTION  : Displays an instance's slots
  INPUTS       : 1) Logical name for output
                 2) Instance address
                 3) String used to separate
                    slot printouts
  RETURNS      : Nothing useful
  SIDE EFFECTS : None
  NOTES        : Assumes instance is valid
 ***************************************************/
static VOID PrintInstance(log_name,ins,separator)
  char *log_name;
  INSTANCE_TYPE *ins;
  char *separator;
  {
   register int i;
   register INSTANCE_SLOT *sp;

   PrintCLIPS(log_name,ValueToString(ins->name));
   PrintCLIPS(log_name," of ");
   PrintCLIPS(log_name,ValueToString(ins->cls->name));
   for (i = 0 ; i < ins->cls->islot_cnt ; i++)
     {
      PrintCLIPS(log_name,separator);
      sp = &ins->slots[i];
      PrintCLIPS(log_name,"(");
      PrintCLIPS(log_name,ValueToString(sp->desc->name));
      if (*(sp->valaddr) != NULL)
        {
         PrintCLIPS(log_name," ");
         PrintDataObject(log_name,*(sp->valaddr));
        }
      PrintCLIPS(log_name,")");
     }
  }

/****************************************************
  NAME         : CheckInstanceAndSlot
  DESCRIPTION  : Looks up a slot for a particular
                   instance
  INPUTS       : 1) Name of CLIPS function
                 2) Caller's buffer for slot address
                 3) Flag indicating whether it is an
                    error or not if the slot does not
                    exist
  RETURNS      : Address of instance (NULL if error)
  SIDE EFFECTS : Caller's buffer holds address of
                   slot (NULL is slot does not exist)
  NOTES        : None
 ****************************************************/
static INSTANCE_TYPE *CheckInstanceAndSlot(func,sp,no_exist_ok)
  char *func;
  INSTANCE_SLOT **sp;
  int no_exist_ok;
  {
   DATA_OBJECT temp;
   INSTANCE_TYPE *ins;
   
   if (ArgCountCheck(func,EXACTLY,2) == -1)
     return(NULL);
   if (ArgTypeCheck(func,1,INSTANCE_OR_INSTANCE_NAME,&temp) == FALSE)
     return(NULL);
   if (temp.type == INSTANCE)
     {
      ins = (INSTANCE_TYPE *) temp.value;
      if (ins->garbage == 1)
        {
         StaleInstanceAddress(func);
         return(NULL);
        }
     }
   else
     {
      ins = FindInstanceBySymbol((SYMBOL_HN *) temp.value);
      if (ins == NULL)
        {
         NoInstanceError(ValueToString(temp.value),func);
         return(NULL);
        }
     }
   if (ArgTypeCheck(func,2,SYMBOL,&temp) == FALSE)
     return(NULL);
   *sp = FindInstanceSlot(ins,ClipsGetValue(temp));
   if ((*sp == NULL) ? (no_exist_ok == FALSE) : FALSE)
     {
      SlotExistError(DOToString(temp),func);
      return(NULL);
     }
   return(ins);
  }

/**********************************************************************
  NAME         : CheckMultifieldSlotInstance
  DESCRIPTION  : Checks the argument count and gets the instance
                   for the functions mv-slot-replace, insert and delete
  INPUTS       : 1) The function name
                 2) The number of arguments qualifier (EXACTLY, etc.)
                 3) The expected number of arguments
  RETURNS      : The instance address, NULL on errors
  SIDE EFFECTS : None
  NOTES        : None
 **********************************************************************/
static INSTANCE_TYPE *CheckMultifieldSlotInstance(func,expqual,expnum)
  char *func;
  int expqual,expnum;
  {
   INSTANCE_TYPE *ins;
   DATA_OBJECT temp;
   
   if (ArgCountCheck(func,expqual,expnum) == -1)
     {
      SetEvaluationError(TRUE);
      return(NULL);
     }
   if (ArgTypeCheck(func,1,INSTANCE_OR_INSTANCE_NAME,&temp) == FALSE)
     {
      SetEvaluationError(TRUE);
      return(NULL);
     }
   if (temp.type == INSTANCE)
     {
      ins = (INSTANCE_TYPE *) temp.value;
      if (ins->garbage == 1)
        {
         StaleInstanceAddress(func);
         SetEvaluationError(TRUE);
         return(NULL);
        }
     }
   else
     {
      ins = FindInstanceBySymbol((SYMBOL_HN *) temp.value);
      if (ins == NULL)
        NoInstanceError(ValueToString(temp.value),func);
     }
   return(ins);
  }
        
/***************************************************
  NAME         : FindISlotByName
  DESCRIPTION  : Looks up an instance slot by
                   instance name and slot name
  INPUTS       : 1) Instance address
                 2) Instance name-string
  RETURNS      : The instance slot address, NULL if
                   does not exist
  SIDE EFFECTS : None
  NOTES        : None
 ***************************************************/
static INSTANCE_SLOT *FindISlotByName(ins,sname)
  INSTANCE_TYPE *ins;
  char *sname;
  {
   SYMBOL_HN *ssym;
   
   ssym = FindSymbol(sname);
   if (ssym == NULL)
     return(NULL);
   return(FindInstanceSlot(ins,ssym));
  }
  
#endif
 
/***************************************************
  NAME         : 
  DESCRIPTION  : 
  INPUTS       : 
  RETURNS      : 
  SIDE EFFECTS : 
  NOTES        : 
 ***************************************************/


