   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*                  A Product Of The                   */
   /*             Software Technology Branch              */
   /*             NASA - Johnson Space Center             */
   /*                                                     */
   /*             CLIPS Version 5.10  07/17/91            */
   /*                                                     */
   /*                  EVALUATION MODULE                  */
   /*******************************************************/

/*************************************************************/
/* Purpose:                                                  */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Gary D. Riley                                        */
/*                                                           */
/* Contributing Programmer(s):                               */
/*      Brian L. Donnell                                     */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/

#define _EVALUATN_SOURCE_

#include <stdio.h>
#define _CLIPS_STDIO_
#include <string.h>

#include "setup.h"
#include "constant.h"
#include "symbol.h"
#include "clipsmem.h"
#include "expressn.h"
#include "evaluatn.h"
#include "facts.h"
#include "utility.h"
#include "router.h"
#include "sysprime.h"

#if DEFFUNCTION_CONSTRUCT || DEFGENERIC_CONSTRUCT
#include "generate.h"
#include "scanner.h"
#endif

#if DEFGENERIC_CONSTRUCT
#include "genrccom.h"
#include "genrcfun.h"
#endif
#if DEFFUNCTION_CONSTRUCT
#include "deffnctn.h"
#endif
#if DEFGLOBAL_CONSTRUCT
#include "defglobl.h"
#endif
#if OBJECT_SYSTEM
#include "extobj.h"
#endif

/***************************************/
/* LOCAL INTERNAL FUNCTION DEFINITIONS */
/***************************************/

#if ANSI_COMPILER  
   static VOID                    NonexistantError(char *,char *,int);
   static VOID                    WrongTypeError(char *,char *,int,char *);
#else
   static VOID                    NonexistantError();
   static VOID                    WrongTypeError();
#endif

/****************************************/
/* GLOBAL INTERNAL VARIABLE DEFINITIONS */
/****************************************/

   globle struct expr          *CurrentExpression = NULL;
   globle int                   EvaluationError = CLIPS_FALSE;
   globle int                   HaltExecution = CLIPS_FALSE;
   globle int                   CurrentEvaluationDepth = 0;
   globle struct dataObject    *BindList = NULL;

/*###############################################*/
/*###############################################*/
/*######                                   ######*/
/*###### FUNCTION DEFINITION AND EXECUTION ######*/
/*######                                   ######*/
/*###############################################*/
/*###############################################*/

/********************************************************************/
/* EvaluateExpression: Evaluates a CLIPS expression.  Returns FALSE */
/*   if no errors occurred during evaluation, otherwise TRUE.       */
/********************************************************************/
globle int EvaluateExpression(problem,compute_result)
  struct expr *problem;
  DATA_OBJECT_PTR compute_result;
  {
  
   if (problem == NULL)
     {
      compute_result->type = SYMBOL;
      compute_result->value = (VOID *) CLIPSFalseSymbol;
      PropogateReturnValue(compute_result);
      return(EvaluationError);
     }

   switch (problem->type)
     {
      case STRING:
      case SYMBOL:
      case FLOAT:
      case INTEGER:
#if OBJECT_SYSTEM
      case INSTANCE_NAME:
      case INSTANCE:
#endif
      case EXTERNAL_ADDRESS:
        compute_result->type = problem->type;
        compute_result->value = problem->value;
        break;
      case PCALL:
#if DEFFUNCTION_CONSTRUCT
        if (problem->value != NULL)
           CallDeffunction((struct dfunc *) problem->value,
                           problem->arg_list,compute_result);
        else
#endif
           {
            PrintCLIPS(WERROR,"WARNING: attempted to call a deffunction ");
            PrintCLIPS(WERROR,"which does not exist.\n");
            compute_result->type = SYMBOL;
            compute_result->value = (VOID *) CLIPSFalseSymbol;
           }
        break;
      case GCALL:
#if DEFGENERIC_CONSTRUCT
         if (problem->value != NULL)
           GenericDispatch((GENERIC_FUNC *) problem->value,
                                  problem->arg_list,compute_result);
         else
#endif    
           {
            PrintCLIPS(WERROR,"WARNING: attempted to call a generic function ");
            PrintCLIPS(WERROR,"which does not exist.\n");
            compute_result->type = SYMBOL;
            compute_result->value = (VOID *) CLIPSFalseSymbol;
           }
         break;
 
#if DEFRULE_CONSTRUCT        
      case SCALL_GET_VAR:
        GetVarSysFunction(compute_result,problem->value);
        break;
#endif
        
      case FCALL:
        {
         struct expr *old_arg;
         struct FunctionDefinition *fptr;
   
         old_arg = CurrentExpression;
         CurrentExpression = problem;
         fptr = (struct FunctionDefinition *) problem->value;
         switch(fptr->returnValueType)
           {
            case 'v' :  
              (* (VOID (*)(VOID_ARG)) fptr->functionPointer)();
              compute_result->type = RVOID;
              compute_result->value = (VOID *) CLIPSFalseSymbol;
              break;
            case 'b' : 
              compute_result->type = SYMBOL; 
              if ((* (int (*)(VOID_ARG)) fptr->functionPointer)())
                compute_result->value = (VOID *) CLIPSTrueSymbol;
              else
                compute_result->value = (VOID *) CLIPSFalseSymbol;
              break;
            case 'a' : 
              compute_result->type = EXTERNAL_ADDRESS; 
              compute_result->value = 
                             (* (VOID *(*)(VOID_ARG)) fptr->functionPointer)();
              break;
            case 'i' : 
              compute_result->type = INTEGER;
              compute_result->value = (VOID *)
                AddLong((long) (* (int (*)(VOID_ARG)) fptr->functionPointer)());  
              break;
            case 'l' :  
              compute_result->type = INTEGER;
              compute_result->value = (VOID *)
                 AddLong((* (long int (*)(VOID_ARG)) fptr->functionPointer)());
              break;
            case 'f' :          
              compute_result->type = FLOAT;
              compute_result->value = (VOID *)
                 AddDouble((double) (* (float (*)(VOID_ARG)) fptr->functionPointer)()); 
              break;
            case 'd' :           
              compute_result->type = FLOAT;
              compute_result->value = (VOID *)
                 AddDouble((* (double (*)(VOID_ARG)) fptr->functionPointer)()); 
              break;
            case 's' : 
              compute_result->type = STRING;
              compute_result->value = (VOID *)
                (* (SYMBOL_HN *(*)(VOID_ARG)) fptr->functionPointer)();
              break;
            case 'w' :  
              compute_result->type = SYMBOL;
              compute_result->value = (VOID *)
                (* (SYMBOL_HN *(*)(VOID_ARG)) fptr->functionPointer)();
              break;
#if OBJECT_SYSTEM
            case 'x' : 
              compute_result->type = INSTANCE; 
              compute_result->value = 
                             (* (VOID *(*)(VOID_ARG)) fptr->functionPointer)();
              break;
            case 'o' :  
              compute_result->type = INSTANCE_NAME;
              compute_result->value = (VOID *)
                (* (SYMBOL_HN *(*)(VOID_ARG)) fptr->functionPointer)();
              break;
#endif
            case 'c' :  
              {  
               char cbuff[2];
  
               cbuff[0] = (* (char (*)(VOID_ARG)) fptr->functionPointer)();
               cbuff[1] = EOS;
               compute_result->type = SYMBOL;
               compute_result->value = (VOID *) AddSymbol(cbuff);
               break;
              }
            case 'm' : 
            case 'u' : 
#if ANSI_COMPILER
              (* (VOID (*)(DATA_OBJECT_PTR)) 
                 fptr->functionPointer)(compute_result);
#else
              (* (VOID (*)()) fptr->functionPointer)(compute_result);
#endif
              break;
              
            default : 
               CLIPSSystemError("EVALUATN",2);
               ExitCLIPS(5);
               break;
            }
           
        CurrentExpression = old_arg;
        break;
        }
        
     case MULTIFIELD:
        compute_result->type = MULTIFIELD;
        compute_result->value = ((DATA_OBJECT_PTR) (problem->value))->value;
        compute_result->begin = ((DATA_OBJECT_PTR) (problem->value))->begin;
        compute_result->end = ((DATA_OBJECT_PTR) (problem->value))->end;
        break;
        
#if DEFGLOBAL_CONSTRUCT
     case GBWORD:
        if (GetDefglobalValue(ValueToString(problem->value),compute_result)) break;
#endif
     case BWORDS:
     case BWORD:
        if (GetBoundVariable(compute_result,(SYMBOL_HN *) problem->value) == CLIPS_FALSE)
          {
           PrintCLIPS(WERROR,"ERROR: Variable ");
           PrintCLIPS(WERROR,ValueToString(problem->value));
           PrintCLIPS(WERROR," is unbound\n");
           compute_result->type = SYMBOL;
           compute_result->value = (VOID *) CLIPSFalseSymbol;
           EvaluationError = CLIPS_TRUE;
          }
        break;
     default:
        CLIPSSystemError("EVALUATN",3);
        ExitCLIPS(5);
        break;
     }
   PropogateReturnValue(compute_result);
   return(EvaluationError);
  }

/*************************************************************/
/* DefineFunction: Used to define a system or user external */
/*   function so that CLIPS can access it.                   */
/*************************************************************/
globle int DefineFunction(name,return_type,pointer,defn_name)
  char *name, *defn_name;
  int return_type;
  int (*pointer)();
  {
   struct FunctionDefinition *new_function;

   if ( (return_type != 'i') &&
        (return_type != 'f') &&
        (return_type != 'd') &&
        (return_type != 's') &&
        (return_type != 'w') &&
        (return_type != 'c') &&
        (return_type != 'v') &&
        (return_type != 'm') &&
        (return_type != 'l') &&
        (return_type != 'b') &&
        (return_type != 'a') &&
#if OBJECT_SYSTEM
        (return_type != 'o') &&
        (return_type != 'x') &&
#endif
        (return_type != 'u') )
     { return(0); }
   
   new_function = get_struct(FunctionDefinition);
   new_function->callFunctionName = (SYMBOL_HN *) AddSymbol(name);
   new_function->returnValueType = (char) return_type; 
   new_function->functionPointer = pointer;
   new_function->next = GetFunctionList();
   new_function->actualFunctionName = defn_name;
   new_function->parser = NULL;

   IncrementSymbolCount(new_function->callFunctionName);
   SetFunctionList(new_function);
   AddHashFunction(new_function);
 
   return(1);
  }
  
/*************************************************************/
/* RtnLexeme:  Purpose is to return a pointer to a character */
/* array that represents an argument to a function call,     */
/* called by the fctn.                                       */
/*************************************************************/
globle char *RtnLexeme(string_pos)
  int string_pos; 
  {
   int count = 1;
   DATA_OBJECT result;
   struct expr *argPtr;

   /*=====================================================*/
   /* Find the appropriate argument in the argument list. */
   /*=====================================================*/

   argPtr = CurrentExpression->arg_list;
   while ((argPtr != NULL) && (count < string_pos))
     {
      count++; 
      argPtr = argPtr->next_arg; 
     }

   if (argPtr == NULL)
     {
      NonexistantError("RtnLexeme",ValueToString(((struct FunctionDefinition *) CurrentExpression->value)->callFunctionName),string_pos);
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      return(NULL);
     }

   /*=================================================*/
   /* Return the value associated with that argument. */
   /*=================================================*/

   EvaluateExpression(argPtr,&result);
   if ((result.type != SYMBOL) && 
#if OBJECT_SYSTEM
       (result.type != INSTANCE_NAME) &&
#endif
       (result.type != STRING))
     {
      WrongTypeError("RtnLexeme",ValueToString(((struct FunctionDefinition *) CurrentExpression->value)->callFunctionName),
                       string_pos,"symbol, string, or instance name");
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      return(NULL);
     }

   return(ValueToString(result.value));
  }
  
/**************************************************************/
/* RtnDouble:  Returns the nth argument of a function call. The  */
/*   argument should be a floating point number, otherwise an */
/*   error will occur.                                        */
/**************************************************************/
globle double RtnDouble(float_pos)
  int float_pos;
  {
   int count = 1;
   DATA_OBJECT result;
   struct expr *argPtr;

   /*=====================================================*/
   /* Find the appropriate argument in the argument list. */
   /*=====================================================*/
   
   argPtr = CurrentExpression->arg_list;
   while ((argPtr != NULL) && (count < float_pos))
     {
      count++; 
      argPtr = argPtr->next_arg; 
     }

   if (argPtr == NULL)
     {
      NonexistantError("RtnDouble",ValueToString(((struct FunctionDefinition *) CurrentExpression->value)->callFunctionName),float_pos);
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      return(1.0);
     }

   /*=================================================*/
   /* Return the value associated with that argument. */
   /*=================================================*/

   EvaluateExpression(argPtr,&result);
   
   if (result.type == FLOAT)
     { return(ValueToDouble(result.value)); }
   else if (result.type == INTEGER)
     { return((double) ValueToLong(result.value)); }
   else 
     {
      WrongTypeError("rfloat",ValueToString(((struct FunctionDefinition *) CurrentExpression->value)->callFunctionName),
                       float_pos,"number");
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      return(1.0);
     }
  }
  
/*************************************************************/
/* RtnLong:  Returns the nth argument of a function call. */
/*   The argument should be an integer, otherwise an error   */
/*   will occur.                                             */
/*************************************************************/
globle long RtnLong(int_pos)
  int int_pos;
  {
   int count = 1;
   DATA_OBJECT result;
   struct expr *argPtr;

   /*=====================================================*/
   /* Find the appropriate argument in the argument list. */
   /*=====================================================*/
   
   argPtr = CurrentExpression->arg_list;
   while ((argPtr != NULL) && (count < int_pos))
     {
      count++; 
      argPtr = argPtr->next_arg; 
     }

   if (argPtr == NULL)
     {
      NonexistantError("RtnLong",ValueToString(((struct FunctionDefinition *) CurrentExpression->value)->callFunctionName),int_pos);
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      return(1L);
     }

   /*=================================================*/
   /* Return the value associated with that argument. */
   /*=================================================*/

   EvaluateExpression(argPtr,&result);
   
   if (result.type == FLOAT)
     { return((long) ValueToDouble(result.value)); }
   else if (result.type == INTEGER)
     { return(ValueToLong(result.value)); }
   else 
     {
      WrongTypeError("RtnLong",ValueToString(((struct FunctionDefinition *) CurrentExpression->value)->callFunctionName),
                       int_pos,"number");
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      return(1L);
     }
  }
  
/**********************************************************/
/* RtnUnknown: returns an argument thats type is unknown. */
/**********************************************************/
globle DATA_OBJECT_PTR RtnUnknown(arg_pos,val_ptr)
  int arg_pos;
  DATA_OBJECT_PTR val_ptr;
  {
   static int count;
   static struct expr *argPtr;

   /*=====================================================*/
   /* Find the appropriate argument in the argument list. */
   /*=====================================================*/

   count = 1;
   argPtr = CurrentExpression->arg_list;
   while ((argPtr != NULL) && (count < arg_pos))
     {
      count++; 
      argPtr = argPtr->next_arg; 
     }

   if (argPtr == NULL)
     {
      NonexistantError("RtnUnknown",ValueToString(((struct FunctionDefinition *) CurrentExpression->value)->callFunctionName),arg_pos);
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      return(NULL);
     }

   /*=================================================*/
   /* Return the value associated with that argument. */
   /*=================================================*/

   EvaluateExpression(argPtr,val_ptr);
   return(val_ptr);
  }

/********************************************************/
/* rmultype:  Returns the type of the nth element of an */
/*   argument of type MULTIFIELD.                         */
/********************************************************/
globle int rmultype(result,nth)
  DATA_OBJECT_PTR result;
  int nth; 
  {
   int length;
   struct element *elm_ptr;

   if (result->type != MULTIFIELD) return(0);

   length = (result->end - result->begin) + 1;
   if ((nth > length) || (nth < 1)) return(0);
   
   elm_ptr = ((struct fact *) result->value)->atoms;
   
   if (elm_ptr[result->begin + nth - 1].type == INTEGER) return (NUMBER);
   else return(elm_ptr[result->begin + nth - 1].type);
  }

/********************************************************/
/* rmulvalue:                                          */
/********************************************************/
globle VOID *rmulvalue(result,nth)
  DATA_OBJECT_PTR result;
  int nth; 
  {
   int length;
   struct element *elm_ptr;

   if (result->type != MULTIFIELD) return(NULL);

   length = (result->end - result->begin) + 1;
   if ((nth > length) || (nth < 1)) return(NULL);

   elm_ptr = ((struct fact *) result->value)->atoms;
   return(elm_ptr[result->begin + nth - 1].value);
  }

/********************************************************/
/* rmulstring:                                          */
/********************************************************/
globle char *rmulstring(result,nth)
  DATA_OBJECT_PTR result;
  int nth; 
  {
   int length;
   struct element *elm_ptr;

   if (result->type != MULTIFIELD) return(NULL);

   length = (result->end - result->begin) + 1;
   if ((nth > length) || (nth < 1)) return(NULL);

   elm_ptr = ((struct fact *) result->value)->atoms;
   return(ValueToString(elm_ptr[result->begin + nth - 1].value));
  }
  
/********************************************************/
/* rmulhash:                                          */
/********************************************************/
globle SYMBOL_HN *rmulhash(result,nth)
  DATA_OBJECT_PTR result;
  int nth; 
  {
   int length;
   struct element *elm_ptr;

   if (result->type != MULTIFIELD) return(NULL);

   length = (result->end - result->begin) + 1;
   if ((nth > length) || (nth < 1)) return(NULL);

   elm_ptr = ((struct fact *) result->value)->atoms;
   return((SYMBOL_HN *) elm_ptr[result->begin + nth - 1].value);
  }

/********************************************************/
/* rmulfloat:                                          */
/********************************************************/
globle double rmulfloat(result,nth)
  DATA_OBJECT_PTR result;
  int nth; 
  {
   int length;
   struct element *elm_ptr;

   if (result->type != MULTIFIELD) return(0.0);

   length = (result->end - result->begin) + 1;
   if ((nth > length) || (nth < 1)) return(0.0);

   elm_ptr = ((struct fact *) result->value)->atoms;
   if (elm_ptr[result->begin + nth - 1].type == INTEGER)
     { return((float) ((INTEGER_HN *) elm_ptr[result->begin + nth - 1].value)->contents); }
   else 
     { return(((FLOAT_HN *) elm_ptr[result->begin + nth - 1].value)->contents); }
  }
  
/********************************************************/
/* AssignUnknown:                                       */
/********************************************************/
globle VOID AssignUnknown(unk_type,string,number,unk_val)
  SYMBOL_HN *string;
  int unk_type;
  double number;
  DATA_OBJECT_PTR unk_val;
  { 
   unk_val->type = unk_type;
   if (unk_type == FLOAT)
     { unk_val->value = (VOID *) AddDouble(number); }
   else if (unk_type == INTEGER)
     { unk_val->value = (VOID *) AddLong((long) number); }
   else
     { unk_val->value = (VOID *) string; }
  }

/******************************************************************/
/* RtnArgCount: Returns the length of the argument list for a     */
/*   function call.  Useful for system and user defined functions */
/*   which accept a variable number of arguments.                 */
/******************************************************************/
globle int RtnArgCount()
  {
   int count = 0;
   struct expr *argPtr;

   argPtr = CurrentExpression->arg_list;

   while (argPtr != NULL)
     {
      count++; 
      argPtr = argPtr->next_arg;
     }
   
   return(count);
  }
  
/*********************************************/
/* NonexistantError:                         */
/*********************************************/
static VOID NonexistantError(acc_fun_name,fun_name,arg_num)
  char *acc_fun_name, *fun_name;
  int arg_num;
  {
   PrintCLIPS(WERROR,"ERROR: Function ");
   PrintCLIPS(WERROR,acc_fun_name);
   PrintCLIPS(WERROR," received a request from function ");
   PrintCLIPS(WERROR,fun_name);
   PrintCLIPS(WERROR," for argument #");
   PrintLongInteger(WERROR,(long int) arg_num);
   PrintCLIPS(WERROR," which is non-existent\n");
  }
  
/*********************************************/
/* WrongTypeError:                           */
/*********************************************/
static VOID WrongTypeError(acc_fun_name,fun_name,arg_num,type)
  char *acc_fun_name, *fun_name, *type;
  int arg_num;
  {
   PrintCLIPS(WERROR,"ERROR: Function ");
   PrintCLIPS(WERROR,acc_fun_name);
   PrintCLIPS(WERROR," received a request from function ");
   PrintCLIPS(WERROR,fun_name);
   PrintCLIPS(WERROR," for argument #");
   PrintLongInteger(WERROR,(long int) arg_num);
   PrintCLIPS(WERROR," which is not of type ");
   PrintCLIPS(WERROR,type);
   PrintCLIPS(WERROR,"\n");
  }
    
/***************************************************/
/* SetHaltExecution:                               */
/***************************************************/
globle VOID SetEvaluationError(value)
  int value;
  { 
   EvaluationError = value;
   if (value == CLIPS_TRUE) HaltExecution = CLIPS_TRUE;
  }
  
/***************************************************/
/* GetEvaluationError:                                */
/***************************************************/
globle int GetEvaluationError()
  {
   return(EvaluationError);
  }
  
/***************************************************/
/* SetHaltExecution:                               */
/***************************************************/
globle VOID SetHaltExecution(value)
  int value;
  { HaltExecution = value; }
  
/***************************************************/
/* GetHaltExecution:                                */
/***************************************************/
globle int GetHaltExecution()
  {
   return(HaltExecution);
  }
  
/**********************************************************/
/* ArgCountCheck:  Checks that a function has the correct */
/*   number of arguments.                                 */
/**********************************************************/
globle int ArgCountCheck(fun_name,check_val,exp_num)
  char *fun_name;
  int check_val, exp_num;
  {
   int num_a;

   num_a = RtnArgCount();
   if (check_val == EXACTLY)
     { if (num_a == exp_num) return(num_a); }
   else if (check_val == AT_LEAST)
     { if (num_a >= exp_num) return(num_a); }
   else if (check_val == NO_MORE_THAN)
     { if (num_a <= exp_num) return(num_a); }
   else
     {
      PrintCLIPS(WERROR,"Function ArgCountCheck received an invalid argument\n");
      return(0);
     }

   ExpectedCountError(fun_name,check_val,exp_num);

   SetHaltExecution(CLIPS_TRUE);
   SetEvaluationError(CLIPS_TRUE);
   return(-1);
  }
  
/************************************************************/
/* ArgRangeCheck:  Checks that a function has the correct */
/*   range of arguments.                                    */
/************************************************************/
globle int ArgRangeCheck(fun_name,min,max)
  char *fun_name;
  int min,max;
  {
   int num_a;

   num_a = RtnArgCount();
   if ((num_a < min) || (num_a > max))
     {
      PrintCLIPS(WERROR,"ERROR: Function ");
      PrintCLIPS(WERROR,fun_name);
      PrintCLIPS(WERROR," expected at least ");
      PrintLongInteger(WERROR,(long) min);
      PrintCLIPS(WERROR," and no more than ");
      PrintLongInteger(WERROR,(long) max);
      PrintCLIPS(WERROR," arguments.\n");
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      return(-1);
     }
   return(num_a);
  }

/***********************************************************/
/* ArgTypeCheck:  Checks that a function has the correct */
/*   type for a particular argument.                       */
/***********************************************************/
globle int ArgTypeCheck(fun_name,arg_num,exp_type,val_ptr)
  char *fun_name;
  int exp_type;
  int arg_num;
  DATA_OBJECT_PTR val_ptr;
  {

   RtnUnknown(arg_num,val_ptr);
   
   if (EvaluationError) return(CLIPS_FALSE);
   
   if (val_ptr->type != exp_type)
     { 
      if ((exp_type == INTEGER_OR_FLOAT) &&
          ((val_ptr->type == INTEGER) || (val_ptr->type == FLOAT))) 
        { return(CLIPS_TRUE); }
        
      if ((exp_type == SYMBOL_OR_STRING) &&
          ((val_ptr->type == SYMBOL) || (val_ptr->type == STRING))) 
        { return(CLIPS_TRUE); }
        
#if OBJECT_SYSTEM
      if (((exp_type == SYMBOL_OR_STRING) || (exp_type == SYMBOL)) &&
          (val_ptr->type == INSTANCE_NAME)) 
        { return(CLIPS_TRUE); }
        
      if ((exp_type == INSTANCE_NAME) &&
          ((val_ptr->type == INSTANCE_NAME) || (val_ptr->type == SYMBOL)))
        { return(CLIPS_TRUE); }
        
      if ((exp_type == INSTANCE_OR_INSTANCE_NAME) &&
          ((val_ptr->type == INSTANCE) || (val_ptr->type == INSTANCE_NAME) || 
           (val_ptr->type == SYMBOL)))
        { return(CLIPS_TRUE); }
#endif
        
      if ((val_ptr->type == INTEGER) && (exp_type == FLOAT)) 
        {
         val_ptr->type = FLOAT;
         val_ptr->value = (VOID *) AddDouble((double) ValueToLong(val_ptr->value));
         return(CLIPS_TRUE);
        }   
        
      if ((val_ptr->type == FLOAT) && (exp_type == INTEGER)) 
        {
         val_ptr->type = INTEGER;
         val_ptr->value = (VOID *) AddLong((long) ValueToDouble(val_ptr->value));
         return(CLIPS_TRUE);
        }
       
      if (exp_type == FLOAT) ExpectedTypeError(fun_name,arg_num,"float");
      else if (exp_type == INTEGER) ExpectedTypeError(fun_name,arg_num,"integer");
      else if (exp_type == SYMBOL) ExpectedTypeError(fun_name,arg_num,"symbol");
      else if (exp_type == STRING) ExpectedTypeError(fun_name,arg_num,"string");
      else if (exp_type == MULTIFIELD) ExpectedTypeError(fun_name,arg_num,"multifield");
      else if (exp_type == INTEGER_OR_FLOAT) ExpectedTypeError(fun_name,arg_num,"integer or float");
      else if (exp_type == SYMBOL_OR_STRING) ExpectedTypeError(fun_name,arg_num,"symbol or string");
#if OBJECT_SYSTEM      
      else if (exp_type == INSTANCE_NAME) ExpectedTypeError(fun_name,arg_num,"instance name");
      else if (exp_type == INSTANCE) ExpectedTypeError(fun_name,arg_num,"instance");
      else if (exp_type == INSTANCE_OR_INSTANCE_NAME) ExpectedTypeError(fun_name,arg_num,"instance or instance name");
#endif
        
      SetHaltExecution(CLIPS_TRUE);
      SetEvaluationError(CLIPS_TRUE);
      return(CLIPS_FALSE);
     }

   return(CLIPS_TRUE);
  }
  
/*******************************************/
/* ReturnValues:   */       
/*******************************************/
globle VOID ReturnValues(garbagePtr)
  DATA_OBJECT_PTR garbagePtr;
  {
   DATA_OBJECT_PTR nextPtr;
   
   while (garbagePtr != NULL)
     {
      nextPtr = garbagePtr->next;
      ValueDeinstall(garbagePtr);
      rtn_struct(dataObject,garbagePtr);
      garbagePtr = nextPtr;
     }
  }
  
/***********************************************************/
/* PrintDataObject:                                            */
/***********************************************************/
globle VOID PrintDataObject(fileid,argPtr)
  char *fileid;
  DATA_OBJECT_PTR argPtr;
  {
   struct element *elem_ptr;
   int i; 
   
   switch(argPtr->type)
     {
      case RVOID:
        break;
      case SYMBOL:
      case STRING:
      case INTEGER:
      case FLOAT:
      case EXTERNAL_ADDRESS:
      case FACT_ADDRESS:
#if OBJECT_SYSTEM
      case INSTANCE_NAME:
      case INSTANCE:
#endif
        PrintAtom(fileid,argPtr->type,argPtr->value); 
        break;
      case MULTIFIELD:
        PrintCLIPS(fileid,"(");
        elem_ptr = ((struct fact *) argPtr->value)->atoms;
        i = argPtr->begin;
        while (i <= argPtr->end)
          {
           PrintAtom(fileid,elem_ptr[i].type,elem_ptr[i].value);
           i++;
           if (i <= argPtr->end) PrintCLIPS(fileid," ");
          }
        PrintCLIPS(fileid,")");
        break;
                
      default:
        PrintCLIPS(fileid,"<<<???UnknownPrintType"); 
        PrintLongInteger(fileid,(long int) argPtr->type);
        PrintCLIPS(fileid,"???>>>");
        SetHaltExecution(CLIPS_TRUE);
        SetEvaluationError(CLIPS_TRUE);
        break;
     }
  }
   
/**********************************************************/
/* SetMultifieldErrorValue:                                  */
/**********************************************************/
globle VOID SetMultifieldErrorValue(sub_value)
  DATA_OBJECT_PTR sub_value;
  {
   sub_value->type = MULTIFIELD;
   sub_value->value = CreateMultifield(0);
   sub_value->begin = 1;
   sub_value->end = 0;
  }
  
/**********************************************************/
/* ValueInstall:                                  */
/**********************************************************/
globle VOID ValueInstall(vPtr)
  DATA_OBJECT *vPtr;
  {
   if (vPtr->type == MULTIFIELD) SegmentInstall((struct fact *) vPtr->value);
   else AtomInstall(vPtr->type,vPtr->value);
  }
    
/**********************************************************/
/* ValueDeinstall:                                  */
/**********************************************************/
globle VOID ValueDeinstall(vPtr)
  DATA_OBJECT *vPtr;
  {
   if (vPtr->type == MULTIFIELD) SegmentDeinstall((struct fact *) vPtr->value);
   else AtomDeinstall(vPtr->type,vPtr->value);
  }

/**********************************************************/
/* PropogateReturnValue:                                  */
/**********************************************************/
globle VOID PropogateReturnValue(vPtr)
  DATA_OBJECT *vPtr;
  {
   int i;
   struct fact *factPtr;
   struct element *elem_ptr;
   
   switch (vPtr->type)
     {
#if OBJECT_SYSTEM
      case INSTANCE        : DecrementInstanceDepth(vPtr->value);
                             break;
        
      case INSTANCE_NAME   :
#endif
      case INTEGER  :
      case FLOAT   :
      case SYMBOL     :
      case STRING   : 
        if (((SYMBOL_HN *) vPtr->value)->markedEphemeral)
          { 
           if (((SYMBOL_HN *) vPtr->value)->depth > CurrentEvaluationDepth) 
             { ((SYMBOL_HN *) vPtr->value)->depth = CurrentEvaluationDepth; }
          }
        break;
                      
      case MULTIFIELD :
        factPtr = (struct fact *) vPtr->value;
        if (factPtr->depth > CurrentEvaluationDepth)
          factPtr->depth = CurrentEvaluationDepth;
        elem_ptr = factPtr->atoms;
        i = vPtr->begin;
        while (i <= vPtr->end)
          {
           switch (elem_ptr[i].type)
             {
#if OBJECT_SYSTEM
              case INSTANCE        : DecrementInstanceDepth(elem_ptr[i].value);
                                     break;
              case INSTANCE_NAME   :
#endif
              case INTEGER  :
              case FLOAT   :
              case SYMBOL     :
              case STRING   : 
                if (((SYMBOL_HN *) elem_ptr[i].value)->markedEphemeral)
                  { 
                    if (((SYMBOL_HN *) elem_ptr[i].value)->depth > CurrentEvaluationDepth) 
                      { ((SYMBOL_HN *) elem_ptr[i].value)->depth = CurrentEvaluationDepth; }
                  }
                break;
             }
           i++;
          }
        break;
        
      case FACT_ADDRESS :
        factPtr = (struct fact *) vPtr->value;
        if (factPtr->depth > CurrentEvaluationDepth)
          { factPtr->depth = CurrentEvaluationDepth; }
        break;
     }
  
  }

/**************************************************************/
/* GetBoundVariable:                             */
/**************************************************************/
globle BOOLEAN GetBoundVariable(vPtr,varName)
  DATA_OBJECT_PTR vPtr;
  SYMBOL_HN *varName;
  {
   DATA_OBJECT_PTR bindPtr;
   
   bindPtr = BindList;
   while (bindPtr != NULL)
     {
      if (bindPtr->name == varName)
        {
         vPtr->type = bindPtr->type;
         vPtr->value = bindPtr->value;
         vPtr->begin = bindPtr->begin;
         vPtr->end = bindPtr->end; 
         return(CLIPS_TRUE); 
        }

      bindPtr = bindPtr->next;
     }
   return(CLIPS_FALSE);
  }  

#if DEFFUNCTION_CONSTRUCT || DEFGENERIC_CONSTRUCT

/**********************************************************************/
/* CLIPSFunctionCall: Allows CLIPS Deffunctions and Generic Functions */
/* to be called from C.  Allows only constants as arguments.          */
/**********************************************************************/
globle int CLIPSFunctionCall(name,args,result)
  char *name,*args;
  DATA_OBJECT *result;
  {
#if DEFGENERIC_CONSTRUCT
   VOID *gfunc;
#endif
#if DEFFUNCTION_CONSTRUCT
   VOID *dptr;
#endif
   struct FunctionDefinition *fptr;
   EXPRESSION *argexps, *top;
   int error = CLIPS_FALSE;

   if (CurrentEvaluationDepth == 0) SetHaltExecution(CLIPS_FALSE);
   EvaluationError = CLIPS_FALSE;
   
   result->type = SYMBOL;
   result->value = (VOID *) CLIPSFalseSymbol;
   
   argexps = ParseConstantArguments(args,&error);
   if (error == CLIPS_TRUE) return(CLIPS_TRUE);

#if DEFFUNCTION_CONSTRUCT
   if ((dptr = FindDeffunction(name)) != NULL)
     {
      top = GenConstant(PCALL,dptr);
      top->arg_list = argexps;
      error = EvaluateExpression(top,result);
     }
   else
#endif
#if DEFGENERIC_CONSTRUCT
   if ((gfunc = FindDefgeneric(name)) != NULL)
     {
      top = GenConstant(GCALL,gfunc);
      top->arg_list = argexps;
      error = EvaluateExpression(top,result);
     }
   else
#endif
   
   if ((fptr = FindFunction(name)) != NULL)
     {
      top = GenConstant(FCALL,fptr);
      top->arg_list = argexps;
      error = EvaluateExpression(top,result);
     }
   else
     {
      PrintCLIPS(WERROR,"No function, generic function, or deffunction of name ");
      PrintCLIPS(WERROR,name);
      PrintCLIPS(WERROR," exists for external call.\n");
      top = argexps;
      error = CLIPS_TRUE;
     }
     
   ReturnExpression(top);
   return(error);
  }

#endif

/*******************/
/* CopyDataObject  */
/*******************/
globle VOID CopyDataObject(dst,src)
  DATA_OBJECT *dst,*src;
  {
   if (src->type != MULTIFIELD)
     {
      CopyMemory(DATA_OBJECT,1,dst,src);
      dst->next = NULL;
     }
   else
     {
      DuplicateSegment(dst,src);
      AddToSegmentList((SEGMENT) dst->value);
     }
  }
  
