/* ************************************************************************
   FILE : POP.C
      This file contains the support routines that handle pop-up menus,
   pop-up warnings, and pop-up text-entry windows.
   *********************************************************************** */
   
/* =======================================================================
   ***********************************************************************
                             HEADER FILES
   ***********************************************************************
   ======================================================================= */
#include "common.h"   /* Standard header files, constants, and macros     */
#include <stdarg.h>   /* Variable argument list stuff                     */
#define _NO_STDIO
#include "curses.h"   /* Aspen Scientific Curses V4.0 Header Files -      */
#include "xcurses.h"  /*    Screen interface function prototypes and      */
#include "box.h"      /*      constants                                   */
#include "curproto.h"
#include "color.h"    /* Color codes for visual screen images             */
#include "key.h"      /* Key-binding definitions                          */
#define MENU_SOURCE
#include "pop.h"      /* Menu, pop-up warning, and pop-up text entry junk */
#include "clipswin.h" /* Window Interface Stuff                           */
#include "env.h"      /* Environment memory stuff                         */
#if MOUSE
#include "mouse.h"
#endif

/* =======================================================================
   ***********************************************************************
                    INTERNALLY VISIBLE DATA TYPES
   ***********************************************************************
   ======================================================================= */

/* --------------------------------------------------------------------
   Pointer to a function which returns int -- used by new_menu so that
   the macro va_arg can access a function pointer, for va_arg needs
   to access *INT_FNX_ADDRESS (or a pointer to a pointer to a function
   which returns int).  Yuuck! C unary operator precedence strikes again!
   -------------------------------------------------------------------- */
typedef int (*INT_FNX_ADDRESS)();

/* =======================================================================
   ***********************************************************************
                   EXTERNALLY VISIBLE FUNCTION PROTOTYPES
   ***********************************************************************
   ======================================================================= */
MENU *new_menu(int,int,...);      /* Allocates a new menu                  */
int  create_visual_menu(MENU *);  /* Designs a physical display for a menu */
void draw_pop_up_menu(MENU *);    /* Writes border and items for menu      */
MENU *kill_menu(MENU *);          /* Releases a menu's memory              */
int  activate_pop_up_menu(MENU *,int,int); 
                                  /* Displays and activates a pop-up menu  */

int  pop_up_warning(int,int,int,...);
                                  /* Displays a pop-up warning window      */
char *pop_up_text(char *,char *,int);
                                  /* Displays a pop-up text-entry window   */
void pop_up_highlight(WINDOW *,int,ch_attr,int,ch_attr);
                                  /* Highlights a pop-up menu selection    */
int  change_menu_item(MENU *,int,char *);
                                  /* Allows caller to change menu item in
                                       a "static" menu                     */

/* =======================================================================
   ***********************************************************************
                   INTERNALLY VISIBLE FUNCTION PROTOTYPES
   ***********************************************************************
   ======================================================================= */
static void adjust_menu_levels(MENU *menu);
static void write_text(WINDOW *,int,int,int,int,char *);
                                     /* Writes justified text to a window  */

/* =======================================================================
   ***********************************************************************
                         EXTERNALLY VISIBLE FUNCTIONS
   ***********************************************************************
   ======================================================================= */

/******************************************************************************
 NAME        : new_menu
 PURPOSE     : Allocates a new menu
 DESCRIPTION : Using variable argument lists, this routine stores a menu
               and designs a window for it.
 INPUTS      : 1) A code indicating whether this routine should design a
                  visual representation of the menu or not :
                    VISUAL_POP_UP_MENU_NO  (0) - No visual image needed
                    VISUAL_POP_UP_MENU_YES (1) - Create an image
               2) The number of items in the menu (must be at least 1)
               3) The following three pieces of data for every item :
                 1) A string representing a menu item
                 2) A code indicating the item type :
                      ITEM (0)    - Argument 3) should be the address of a
                                    handler function for that item
                      SUBMENU (1) - Argument 3) should be the address of a
                                    child sub-menu which has been previously
                                    allocated with new_menu() with the code
                                    VISUAL_POP_UP_MENU_YES
                 3) A pointer to a handler function for that item
                    This function must return an integer corresponding to
                    one of the completion codes defined in MENU.H :
                      MENU_ITEM_COMPLETION (1) - Routine successfully completed
                      MENU_ITEM_ABORT (0) - Routine prematurely aborted
                    In addition, the routine must accept ONE integer argument
                    which will be the index of the item selected in the
                    menu (this allows one routine to handle several different
                    items in the menu if necessary).

                    OR

                    The address of the child sub-menu which has been
                    previously allocated.

               Example :

                 { The following is the handler function for item "Foo" }
                 int handle_Foo(index)
                   int index;
                   {
                    { --- Perform Stuff ---- }
                    return(MENU_ITEM_COMPLETION);
                   }

                 { Here is the call that would define a menu with 1 item }
                 { and 1 sub-menu previously stored at addres submenu    }

                 MENU *menu;

                 menu = new_menu(VISUAL_PPO_UP_MENU_YES,1,
                                 "Foo",ITEM,handle_Foo,
                                 "Bar",SUBMENU,submenu);

 RETURNS     : The address of the new allocated menu if everything went OK,
               NULL otherwise.
 NOTES       : None
 ******************************************************************************/
MENU *new_menu(int visual_code,int no_of_items,...)
  {
   MENU *menu;
   register int i;
   char *item_name;
   unsigned char item_type;
   int (*item_handler)(int) = NULL;
   MENU *item_submenu = NULL;
   va_list ap;

   if ((menu = balloc(1,MENU)) == NULL)
     return(NULL);
   menu->border = NULL;
   menu->text = NULL;
   menu->level = 0;
   menu->item_cnt = no_of_items;
   menu->items = NULL;
   if ((menu->items = balloc(no_of_items,MENU_ITEM *)) == NULL)
     return(kill_menu(menu));
   for (i = 0 ; i < no_of_items ; i++)
     (menu->items)[i] = NULL;
   va_start(ap,no_of_items);
   for (i = 0 ; i < no_of_items ; i++)
     {
      item_name = va_arg(ap,char *);
      item_type = va_arg(ap,int);
      if (item_type == ITEM)
        item_handler = va_arg(ap,INT_FNX_ADDRESS);
      else
        {
         item_submenu = va_arg(ap,MENU *);
         adjust_menu_levels(item_submenu);
        }
      if (((menu->items)[i] = balloc(1,MENU_ITEM)) == NULL)
        return(kill_menu(menu));
      (menu->items)[i]->type = item_type;
      if (((menu->items)[i]->name = balloc(strlen(item_name)+1,char)) == NULL)
        return(kill_menu(menu));
      strcpy((menu->items)[i]->name,item_name);
      if (item_type == ITEM)
        (menu->items)[i]->desc.handler = item_handler;
      else
        (menu->items)[i]->desc.submenu = item_submenu;
     }   
   va_end(ap);
   if (visual_code == VISUAL_POP_UP_MENU_YES)
     {
      if (! create_visual_menu(menu))
        return(kill_menu(menu));
     }              
   return(menu);
  }

/******************************************************************************
 NAME        : create_visual_menu
 PURPOSE     : Designs a physical representation of a menu
 DESCRIPTION : This routine creates a physical menu which can be later
               displayed and used by a pop-up routine.  It attaches the
               address of the new window to the menu record.
 INPUTS      : 1) The address of the already created internal menu.
 RETURNS     : TRUE (1) if everything went OK, FALSE (0) otherwise.
                (An error will only occur if the menu is too big for the
                 screen or the memory could not be allocated.)
 NOTES       : 1) Uses the color defns in the file COLOR.C for pop-up menus
               2) The window will have the following dimensions :
                  # lines = # of menu items + 2 for borders
                  # columns = width of biggest menu item + 2 for borders
                  The first item is automatically highlighted.
               3) If a visual menu is already present, that one will be
                  deallocated before the new design is attached.
                    If the this routine fails, the old image will be left
                  intact.
                    This feature allows the caller to easily redraw the
                  visual menu with new colors without having to reallocate
                  a whole new internal menu.
               4) The visual menu is initially created starting at screen
                  coordinates (0,0), but the Curses routine mvwin() can
                  later be called to place the menu wherever desired.
                     The window is actually comprised of two parts :
                  the main window which holds the border, and the text
                  selection window which is a sub-window of the first.
                  The text window is initially set to coordinates (1,1).
                  mvwin() must be applied equally to BOTH windows.
                    The text window should be flushed to the screen AFTER
                  the border window.
               5) WARNING!!! - This routine should not be called if the
                  Curses routine initscr() has noy yet been called!
 ******************************************************************************/
static int create_visual_menu(menu)
  MENU *menu;
  {
   int maxwid,wid;
   register int i;

   /* ===========================================================
      Find the width of the biggest menu item and check the sizes
      =========================================================== */
   for (maxwid = 0 , wid = 0 , i = 0 ; i < menu->item_cnt ; i++)
     {
      wid = strlen((menu->items)[i]->name);
      if (wid > maxwid)
        maxwid = wid;
     }
   if ((menu->item_cnt > CGA_LINES-2) || (maxwid > ALL_COLS-2))
     return(FALSE);
 
   /* =====================================
      Deallocate old images (if they exist)
      ===================================== */
   if (menu->text != NULL)
     delwin(menu->text);
   if (menu->border != NULL)
     delwin(menu->border);

    /* ====================
       Allocate the windows
       ==================== */
   if ((menu->border = newwin(menu->item_cnt+2,maxwid+2,0,0)) == NULL)
     return(FALSE);
   if ((menu->text = subwin(menu->border,menu->item_cnt,maxwid,1,1)) == NULL)
     {
      delwin(menu->border);
      return(FALSE);
     }
   scrollok(menu->text,FALSE);
   nodelay(menu->text,TRUE);
   keypad(menu->text,TRUE);
   draw_pop_up_menu(menu);
   return(TRUE);
  }

/******************************************************************************
 NAME        : draw_pop_up_menu
 PURPOSE     : Draws border and items for pop-up menu with appropriate colors
 DESCRIPTION : 
 INPUTS      : 1) The address of the menu structure
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void draw_pop_up_menu(menu)
  MENU *menu;
  {
   ch_attr border_attr,unhgh_attr,hgh_attr;
   register int i;
   int maxwid;

   /* ========================================
      Define the colors and stats for the menu
      ======================================== */
   border_attr = colors[POP_MENU_BORDER] | colors[POP_MENU_BGD];
   unhgh_attr = colors[POP_MENU_TEXT] | colors[POP_MENU_BGD];
   hgh_attr = flip_fgd_bgd(unhgh_attr);
   getmaxc(menu->text,maxwid);

   /* ===============
      Draw the border
      =============== */
   xpaint(menu->border,border_attr);
   xbox(menu->border,DB_V_LIN,DB_H_LIN,border_attr);
   wmove(menu->border,0,0);

   /* =========================
      Write the menu selections
      ========================= */
   xpaint(menu->text,unhgh_attr);
   wattrset(menu->text,unhgh_attr);
   for (i = 0 ; i < menu->item_cnt ; i++)
     {
      wmove(menu->text,i,maxwid/2-strlen((menu->items)[i]->name)/2);
      waddstr(menu->text,(menu->items)[i]->name);
     }
   pop_up_highlight(menu->text,0,unhgh_attr,0,hgh_attr);
   wmove(menu->text,0,0);
  }

/******************************************************************************
 NAME        : kill_menu
 PURPOSE     : Releases memory associated with a menu
 DESCRIPTION : 
 INPUTS      : 1) Address of the menu to be released
 RETURNS     : NULL so that a user may delete a menu and set the address to
               NULL in one blow.

               Example : menu_address = deallocate_menu(menu_address);

 NOTES       : None
 ******************************************************************************/
MENU *kill_menu(menu)
  MENU *menu;
  {
   register int i;

   if (menu->text != NULL)
     delwin(menu->text);
   if (menu->border != NULL)
     delwin(menu->border);
   if (menu->items != NULL)
     {
      for (i = 0 ; i < menu->item_cnt ; i++)
        {
         if ((menu->items)[i] != NULL)
           {
            if ((menu->items)[i]->name != NULL)
              release(strlen((menu->items)[i]->name)+1,
                              char,(menu->items)[i]->name);
            release(1,MENU_ITEM,(menu->items)[i]);
           }
        }
      release(menu->item_cnt,MENU_ITEM *,menu->items);
     }
   release(1,MENU,menu);
   return(NULL);
  }

/******************************************************************************
 NAME        : activate_pop_up_menu
 PURPOSE     : Displays and activates a pop-up menu
 DESCRIPTION : This routine recursively calls sub-menus
 INPUTS      : 1) The address of the menu to be activated.
               2) The screen row to start the menu on
               3) The screen column to start the menu on
 RETURNS     : ERROR (-1) if the menu could not be displayed for some
               reason or MENU_ABORT (0) if the user cancels the menu, 
               otherwise the routine returns the status code of the
               menu item selected :
                    MENU_ITEM_COMPLETION
                    MENU_ITEM_CLIPS_INPUT
               If the item returns MENU_ITEM_ABORT, the pop-up menu stays
               on the screen cycling for input.
 NOTES       : Assumes Curses is initialized as well as the menu structure
               and windows.
 ******************************************************************************/
int activate_pop_up_menu(menu,start_row,start_col)
  MENU *menu;
  int start_row,start_col;
  {
   int current_selection = 1,
       last_selection = 1;
   int rows,cols;
   ch_attr unhgh_attr,hgh_attr;
   int ch,
       item_execution_rtn,
       quit = FALSE;
   MENU_ITEM *item;
#if MOUSE
   int ms_y,ms_x;
#endif

   if ((menu->border == NULL) || (menu->text == NULL))
     return(ERROR);
   getmaxrc(menu->border,rows,cols);
   if ((rows > LINES) || (cols > COLS))
     return(ERROR);
   if (start_row < 0)
     start_row = 0;
   else if (start_row + rows > LINES)
     start_row = LINES-rows;
   if (start_col < 0)
     start_col = 0;
   else if (start_col + cols > COLS)
     start_col = COLS-cols;
   (void) mvwin(menu->border,start_row,start_col);
   (void) mvwin(menu->text,start_row+1,start_col+1);
   unhgh_attr = colors[POP_MENU_TEXT] | colors[POP_MENU_BGD];
   hgh_attr = flip_fgd_bgd(unhgh_attr);
#if MOUSE
   save_mouse();
   mouse_region(start_row+1,start_col+1,start_row+rows-2,start_col+1);
   set_mouse(start_row+1,start_col+1);
#endif
   insert_pop_up(POP_WIN,menu->border);
   insert_pop_up(POP_WIN,menu->text);
   redisplay_wins();
   hide_cursor();
   item = (menu->items)[current_selection-1];
   while (! quit)
     {
      if (current_selection != last_selection)
        {
         pop_up_highlight(menu->text,last_selection-1,unhgh_attr,
                                     current_selection-1,hgh_attr);
         wmove(menu->text,start_row+current_selection,start_col+1);
#if MOUSE
         set_mouse(start_row+current_selection,start_col+1);
#endif
         wrefresh(menu->text);
         hide_cursor();
         last_selection = current_selection;
         item = (menu->items)[current_selection-1];
        }
      if ((ch = wgetch(menu->text)) != -1)
        {
         ch = key_map(ch);
         if ((ch == NEWLINE) || (ch == MAIN_MENU))
           {
            if (item->type == ITEM)
              item_execution_rtn = (*(item->desc.handler))(current_selection);
            else
              item_execution_rtn = activate_pop_up_menu(item->desc.submenu,
                                   start_row+current_selection+1,start_col+2);
            unhgh_attr = colors[POP_MENU_TEXT] | colors[POP_MENU_BGD];
            hgh_attr = flip_fgd_bgd(unhgh_attr);
            if (item_execution_rtn <= MENU_ABORT)
              redisplay_wins();
            else
              quit = TRUE;
           }
         else if (ch == ESC)
           {
            last_selection = current_selection;
            item_execution_rtn = MENU_ABORT;
            quit = TRUE;
           }
         else if (ch == UP_ARROW)
           current_selection = 
               (current_selection + menu->item_cnt - 2) % menu->item_cnt + 1;
         else if (ch == DOWN_ARROW)
           current_selection = current_selection % menu->item_cnt + 1;
         else if (ch == BUFFER_TOP)
           current_selection = 1;
         else if (ch == BUFFER_END)
           current_selection = menu->item_cnt;
         else if (ch == LEFT_ARROW)
           {
            if (menu->level <= 1)
              {
               last_selection = current_selection;
               item_execution_rtn = MENU_LEFT_ABORT;
               quit = TRUE;
              }
            else
              beep();
           }
         else if (ch == RIGHT_ARROW)
           {
            if (menu->level <= 1)
              {
               last_selection = current_selection;
               item_execution_rtn = MENU_RIGHT_ABORT;
               quit = TRUE;
              }
            else
              beep();
           }
         else if (ch == REDRAW_CMD)
           redisplay_wins();
         else
           beep();
        }
#if MOUSE
      else if (mouse_moved_posn(&ms_y,&ms_x))
        current_selection = ms_y - start_row;
      else if (mouse_clicked(&ch,&ms_y,&ms_x))
        {
         if (ch == RIGHT_BUTTON)
           {
            last_selection = current_selection;
            item_execution_rtn = MENU_ABORT;
            quit = TRUE;
           }
         else if (ch == LEFT_BUTTON)
           {
            if (item->type == ITEM)
              item_execution_rtn = (*(item->desc.handler))(current_selection);
            else
              item_execution_rtn = activate_pop_up_menu(item->desc.submenu,
                                   start_row+current_selection+1,start_col+2);
            unhgh_attr = colors[POP_MENU_TEXT] | colors[POP_MENU_BGD];
            hgh_attr = flip_fgd_bgd(unhgh_attr);
            if (item_execution_rtn <= MENU_ITEM_ABORT)
              redisplay_wins();
            else
              quit = TRUE;
           }
         else
           beep();
        }
#endif
     }
   pop_up_highlight(menu->text,last_selection-1,unhgh_attr,0,hgh_attr);
   delete_pop_up();
   delete_pop_up();
#if MOUSE
   restore_mouse();
#endif
   redisplay_wins();
   return(item_execution_rtn);
  }

/******************************************************************************
 NAME        : pop_up_warning
 PURPOSE     : Displays a pop-up warning message
 DESCRIPTION : Pop-up a warning message and prompts the user for a yes/no
               response.
                 The pop-up window is always centered on the screen.
 INPUTS      : 1) The justification with which the text should be displayed
                  LFT_JUST (0) - Left Justification
                  MID_JUST (1) - Middle (Center) Justification
                  RGT_JUST (2) - Right Justification
               2) A code indicating whether to ring the terminal bell
                    SILENCE (0) - Don't ring bell
                    BEEP (1) - Ring the bell
               3) The number of messages being passed
               4+) The messages to be displayed (May have imbedded CRTs)
                   (If there are more than 1 message - each message should
                    have at least one '\n' at the end to prevent overwriting)
 RETURNS     : 1) ERROR (-1) if the warning message could not fit on the
                  screen or the window could not be allocated
               2) YES (1) if the user indicated "yes" to the routine's prompt
               3) NO (0) if the user indicated "no" to the routine's prompt
 NOTES       : The window will require the following screen dimensions :
                 Lines : the number of lines required by the message +
                           4 for the yes/no prompt + 2 for the border
                 Columns : the number of columns required by the message or
                           the number of columns required by the prompt
                           (whichever is greater) + 2 for the border.
                           The prompt requires at most 42 columns.
 ******************************************************************************/
int pop_up_warning(int just,int bell,int msgno,...)
  {
   register int i,j;
   int wid,maxwid,winwid,
       lns,winlns,y,x; 
   WINDOW *border,
          *text;
   char prompt1[43],
        prompt2[43];
   char *msg,*line;
   int promptwid,
       lstart,ch,
       quit = FALSE;
   va_list ap;

   maxwid = 0;
   lns = 1;
   va_start(ap,msgno);
   for (j = 0 ; j < msgno ; j++)
     {
      msg = va_arg(ap,char *);

      /*=================================================================
        Determine how many lines are required for the message and also
        find the width of the longest line.  (The window's minimum width
        must be big enough for the deletion-message)
        =================================================================*/
      for (wid = 0 , i = 0 ; msg[i] != EOS ; i++)
        {
         if (wid  == COLS-2)
           {
            lns++;
            if (wid > maxwid)
              maxwid = wid;
            wid = 0;
           }
         if ((msg[i] != NEWLINE) ? (msg[i] == CRGRTN) : TRUE)
           {
            lns++;
            if (wid > maxwid)
              maxwid = wid;
            wid = 0;
           }
         else
           wid++;
        }
       if (wid > maxwid)
         maxwid = wid;
     }
   va_end(ap);
   
/* ========================================
      Determine the width of the yes/no prompt
      ======================================== */
#if MOUSE
   if (MOUSE_LOADED)
     {
      strcpy(prompt1,"<RTN> or <LFT-CLICK>  <ESC> or <RGT-CLICK>");
      strcpy(prompt2,"      for YES                for NO       ");
      promptwid = 42;
     }
   else
     {
#endif
      strcpy(prompt1," <RTN>     <ESC>");
      strcpy(prompt2,"for YES   for NO");
      promptwid = 16;
#if MOUSE
     }
#endif

   winwid = ((maxwid > promptwid) ? maxwid : promptwid) + 2;
   winlns = lns + 5;
   if (winwid > COLS)
     return(ERROR);
   if (winlns > LINES)
     return(ERROR);
   y = LINES/2 - winlns/2;
   x = COLS/2 - winwid/2;
   if ((y < 0) || ((y+winlns) > LINES))
     return(ERROR);
   if ((x < 0) || ((x+winwid) > COLS))
     return(ERROR);
   if ((border = newwin(winlns,winwid,y,x)) == NULL)
     return(ERROR);
   scrollok(border,FALSE);
   if ((text = subwin(border,winlns-2,winwid-2,y+1,x+1)) == NULL)
     {
      delwin(border);
      return(ERROR);
     }
   scrollok(text,FALSE);
   nodelay(text,TRUE);
   keypad(text,TRUE);

   xpaint(border,colors[POP_WARN_BGD]);
   xbox(border,DB_V_LIN,DB_H_LIN,colors[POP_WARN_BORDER]|colors[POP_WARN_BGD]);
   xpaint(text,colors[POP_WARN_BGD]);
   wattrset(text,colors[POP_WARN_TEXT]|colors[POP_WARN_BGD]);

   lns = 0;
   va_start(ap,msgno);
   for (j = 0 ; j < msgno ; j++)
     {
      msg = va_arg(ap,char *);
      for (i = lstart = 0, line = msg ; msg[i] != EOS ; i++)
        {
         if (((msg[i] != NEWLINE) ? (msg[i] == CRGRTN) : TRUE)
              || (i == COLS-2))
           {
            ch = msg[i];
            msg[i] = EOS;
            write_text(text,lns++,winwid-2,i-lstart,just,line);
            msg[i] = ch;
            lstart = (i == COLS-2) ? i : i + 1;
            line = msg + lstart;
           }
         else if (msg[i+1] == EOS)
           write_text(text,lns,winwid-2,i-lstart+1,just,line);
        }
     }
   va_end(ap);

   mvwaddstr(text,winlns-4,(winwid-2)/2-promptwid/2,prompt1);
   mvwaddstr(text,winlns-3,(winwid-2)/2-promptwid/2,prompt2);

#if MOUSE
   save_mouse();
#endif
   insert_pop_up(POP_WIN,border);
   insert_pop_up(POP_WIN,text);
   if (bell == BEEP)
     beep();
   touchwin(border);
   wnoutrefresh(border);
   touchwin(text);
   wnoutrefresh(text);
   doupdate();
   hide_cursor();
   while (! quit)
     {
      if ((ch = wgetch(text)) != -1)
        {
         ch = key_map(ch);
         if ((ch == NEWLINE) || (ch == MAIN_MENU))
           {
            ch = YES;
            quit = TRUE;
           }
         else if (ch == ESC)
           {
            ch = NO;
            quit = TRUE;
           }
         else if (ch == REDRAW_CMD)
           redisplay_wins();
         else
           beep();
        }
#if MOUSE
      else if (mouse_clicked(&ch,&y,&x))
        {
         if (ch == LEFT_BUTTON)
           {
            ch = YES;
            quit = TRUE;
           }
         else if (ch == RIGHT_BUTTON)
           {
            ch = NO;
            quit = TRUE;
           }
         else
          beep();
        }
#endif
     }
#if MOUSE
   restore_mouse();
#endif
   delete_pop_up();
   delete_pop_up();
   delwin(text);
   delwin(border);
   redisplay_wins();
   return(ch);
  }

/******************************************************************************
 NAME        : pop_up_text
 PURPOSE     : Grabs temporary text from a pop-up window
 DESCRIPTION : Allows the user to enter text for pop-up menu command arguments
               Used when the program does not want I/O with the main CLIPS
               window.
                 The user signals completion of input by pressing <RETURN> or
               <LEFT-CLICK> (if the mouse is loaded).
                 The user can cancel the window at any time by pressing
               <ESC> or <RIGHT-CLICK> (if the mouse is loaded).
 INPUTS      : 1) A prompt string (May have no imbedded carriage-returns)
               2) The address of a caller-allocated buffer to hold the text
                  entered.
               3) The size of the buffer EXCLUDING the space for a null char
                  (the size must be in the range 1..77)
 RETURNS     : 1) The address of the user's buffer if all went well
                  (the user's buffer will now hold the entered text)
               2) NULL if the user cancelled the pop-up box or the window
                  could not be displayed.
 NOTES       : 1) After calling this routine, whatever was in the user's buffer
                  is overwritten.
               2) The pop-up window is always displayed centered on the screen.
               3) The pop-up window will have the following dimensions :
                  Lines = 1(prompt) + 1(text input) + 2(border) = 4
                  Columns = 2(border) + the greatest of the following :
                                 Caller buffer size + 1
                                 Caller prompt
               4) The input text area can be smaller than the actual window
                  and will be indicated by a different color.
                    In this regard, the routine uses the following mechanism
                  to insure the text area has a different background color :
                     1) The first try is the backgd color which corresponds
                        to the same bit pattern as the foreground color.
                     2) If the first try is the same as the other background,
                        the routine flips the foreground and background colors.

                   Example : Say the main background is BLUE and the text
                             input foreground is defined to be RED.
                             Since the background color corresponding to the
                             RED bit pattern is BLUE, the the fgd and bgd will
                             be flipped and the corresponding text input color
                             will be BLUE on RED.
 ******************************************************************************/
char *pop_up_text(prompt,buf,bufsize)
  char *prompt,*buf;
  int bufsize;
  {
   register int i;
   int promptsize,
       wincols;
   WINDOW *border,*text;
   ch_attr text_bgd,text_attr;
   int quit = FALSE,
       ch,y,x,
       chcnt;
   char *rtn;

   /* ===================================
      Check the validity of the arguments
      =================================== */
   if ((buf == NULL) || (prompt == NULL) || (bufsize < 1) || (bufsize > COLS-3))
     return(NULL);
   for (i = 0 ; prompt[i] != EOS ; i++)
     if ((prompt[i] == CRGRTN) || (prompt[i] == NEWLINE))
       return(NULL);
   if ((promptsize = i) > COLS-2)
     return(NULL);

   wincols = (promptsize > bufsize+1) ? promptsize + 2 : bufsize + 3;
   if ((border = newwin(4,wincols,LINES/2-2,COLS/2-wincols/2)) == NULL)
     return(NULL);
   if ((text = subwin(border,1,bufsize+1,LINES/2,COLS/2-(bufsize+1)/2)) == NULL)
     {
      delwin(border);
      return(NULL);
     }
   scrollok(border,FALSE);
   scrollok(text,FALSE);
   nodelay(text,TRUE);
   keypad(text,TRUE);
   xpaint(border,colors[POP_TEXT_BGD]);
   xbox(border,DB_V_LIN,DB_H_LIN,colors[POP_TEXT_BORDER]|colors[POP_TEXT_BGD]);
   wattrset(border,colors[POP_TEXT_PROMPT]|colors[POP_TEXT_BGD]);
   mvwaddstr(border,1,1,prompt);
   text_attr = colors[POP_TEXT_INPUT] | fgd_reverse(colors[POP_TEXT_INPUT]);
   if ((text_attr & FGD_MSK) == colors[POP_TEXT_BGD])
     text_attr = flip_fgd_bgd(text_attr);
   xpaint(text,text_attr);
   wattrset(text,text_attr);
   wmove(text,0,0);
#if MOUSE
   save_mouse();
#endif
   insert_pop_up(POP_WIN,border);
   insert_pop_up(POP_WIN,text);
   redisplay_wins();
   wrefresh(text);

   chcnt = 0;
   buf[chcnt] = EOS;
   while (! quit)
     {
      if ((ch = wgetch(text)) != -1)
        {
         ch = key_map(ch);
         if ((ch >= LOW_PRN_ASCII) && (ch <= HIGH_PRN_ASCII))
           {
            if (chcnt < bufsize)
              {
               buf[chcnt++] = ch;
               buf[chcnt] = EOS;
               waddch(text,ch);
               wrefresh(text);
              }
            else
              beep();
           }
         else if (ch == DELETE)
           {
            if (chcnt != 0)
              {
               buf[--chcnt] = EOS;
               waddstr(text,DEL_STR);
               wrefresh(text);
              }
            else
             beep();
           }
         else if ((ch == NEWLINE) || (ch == MAIN_MENU))
           {
            if (chcnt == 0)
              rtn = NULL;
            else
              rtn = buf;
            quit = TRUE;
           }
         else if (ch == ESC)
           {
            buf[0] = EOS;
            rtn = NULL;
            quit = TRUE;
           }
         else if (ch == REDRAW_CMD)
           {
            redisplay_wins();
            wrefresh(text);
           }
         else
           beep();
        }
#if MOUSE
      else if (mouse_clicked(&ch,&y,&x))
        {
         if (ch == LEFT_BUTTON)
           {
            if (chcnt == 0)
              rtn = NULL;
            else
              rtn = buf;
            quit = TRUE;
           }
         else if (ch == RIGHT_BUTTON)
           {
            buf[0] = EOS;
            rtn = NULL;
            quit = TRUE;
           }
         else
           beep();
        }
#endif
     }
#if MOUSE
   restore_mouse();
#endif
   delete_pop_up();
   delete_pop_up();
   delwin(text);
   delwin(border);
   redisplay_wins();
   return(rtn);
  }

/******************************************************************************
 NAME        : pop_up_highlight
 PURPOSE     : Highlights a particular row in a pop-up menu
 DESCRIPTION : 
 INPUTS      : 1) the pop-up menu window address
               2) the row to be unhighlighted in relative coordinates
               3) the unhighlighting fgd/bgd color attribute
               4) the row to be highlighted in relative coordinates
               5) the highlighting fgd/bgd color attribute
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void pop_up_highlight(win,urow,uattr,hrow,hattr)
  WINDOW *win;
  int urow,hrow;
  ch_attr uattr,hattr;
  {
   register int i;
   int maxcols;

   getmaxc(win,maxcols);
   wattrset(win,uattr);
   wmove(win,urow,0);
   for (i = 0 ; i < maxcols ; i++)
     waddch(win,winch(win) & A_CHARTEXT);
   wattrset(win,hattr);
   wmove(win,hrow,0);
   for (i = 0 ; i < maxcols ; i++)
     waddch(win,winch(win) & A_CHARTEXT);
  }

/******************************************************************************
 NAME        : change_menu_item
 PURPOSE     : Writes a new selection-string to the given row of menu
 DESCRIPTION : 
 INPUTS      : 1) The address of the menu
               2) The item-number (1..n) of the menu-selection to be changed
               3) The string representing the new menu-item
 RETURNS     : TRUE (1) if all went well, FALSE (0) otherwise
 NOTES       : None
 ******************************************************************************/
int change_menu_item(menu,item,str)
  MENU *menu;
  int item;
  char *str;
  {
   char *newstr,*oldstr;
   register int i;
   int wid,cols;

   getmaxc(menu->text,cols);
   if ((wid = strlen(str)) > cols)
     return(FALSE);
   if ((newstr = balloc(wid+1,char)) == NULL)
     return(FALSE);
   strcpy(newstr,str);
   oldstr = ((menu->items)[item-1])->name;
   release(strlen(oldstr)+1,char,oldstr);
   ((menu->items)[item-1])->name = newstr;
   wattrset(menu->text,mvwinat(menu->text,item-1,0));
   wmove(menu->text,item-1,0);
   for (i = 0 ; i < cols ; i++)
     waddch(menu->text,BLANK);
   mvwaddstr(menu->text,item-1,cols/2-wid/2,newstr);
   return(TRUE);
  }

/* =======================================================================
   ***********************************************************************
                         INTERNALLY VISIBLE FUNCTIONS
   ***********************************************************************
   ======================================================================= */

/******************************************************************************
 NAME        : adjust_menu_levels
 PURPOSE     : Recursively increments level count for menu and all sub-menus
 DESCRIPTION : 
 INPUTS      : 1) The address of the parent-menu
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void adjust_menu_levels(MENU *menu)
  {
   int i;

   menu->level++;
   for (i = 0 ; i < menu->item_cnt ; i++)
     {
      if ((menu->items)[i]->type == SUBMENU)
        adjust_menu_levels((menu->items)[i]->desc.submenu);
     }
  }

/******************************************************************************
 NAME        : write_text
 PURPOSE     : Writes justified text to a window
 DESCRIPTION : 
 INPUTS      : 1) The address of the window
               2) The line number to which the text is to be written
               3) The width of the window
               4) The width of the text
               5) The justification with which the text is to be written
                  LFT_JUST (0) - Left Justification
                  MID_JUST (1) - Middle (Center) Justification
                  RGT_JUST (2) - Right Justification
               6) The text which is to be written
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void write_text(win,ln_no,winwid,lnwid,just,line)
  WINDOW *win;
  int ln_no,winwid,lnwid,just;
  char *line;
  {
   if (just == LFT_JUST)
     mvwaddstr(win,ln_no,0,line);
   else if (just == MID_JUST)
     mvwaddstr(win,ln_no,winwid/2-lnwid/2,line);
   else if (just == RGT_JUST)
     mvwaddstr(win,ln_no,winwid-lnwid,line);
  }

/******************************************************************************
 NAME        : 
 PURPOSE     : 
 DESCRIPTION : 
 INPUTS      : 
 RETURNS     : 
 NOTES       : 
 ******************************************************************************/
