 /*************************************************************************
   FILE : POPUP.C
      This file contains the support routines that handle pop-up menus,
   pop-up warnings, and pop-up text-entry windows.
   *********************************************************************** */
   
/* =======================================================================
   ***********************************************************************
                             HEADER FILES
   ***********************************************************************
   ======================================================================= */

#include "curses.h"   /* Aspen Scientific Curses V4.0 Header Files -      */
#include "xcurses.h"  /*    Screen interface function prototypes and      */
#include "box.h"      /*      constants                                   */

#include "crsv.h"     /* General CRSV Definitions */

#define MENU_SOURCE

#include "pcint.h"    /* Standard header files, constants, and macros     */
#include <stdarg.h>   /* Variable argument list stuff                     */

extern MENU  *options_menu;
extern short  option_flags[10];
extern void   get_environment(void);


/* =======================================================================
   ***********************************************************************
                    INTERNALLY VISIBLE DATA TYPES
   ***********************************************************************
   ======================================================================= */

/* --------------------------------------------------------------------
   Pointer to a function which returns int -- used by new_menu so that
   the macro va_arg can access a function pointer, for va_arg needs
   to access *INT_FNX_ADDRESS (or a pointer to a pointer to a function
   which returns int).  Yuuck! C unary operator precedence strikes again!
   -------------------------------------------------------------------- */

typedef int (*INT_FNX_ADDRESS)();


/* =======================================================================
   ***********************************************************************
                   EXTERNALLY VISIBLE FUNCTION PROTOTYPES
   ***********************************************************************
   ======================================================================= */

MENU *new_menu(int,int,...);      /* Allocates a new menu                  */
int  create_visual_menu(MENU *);  /* Designs a physical display for a menu */
void draw_pop_up_menu(MENU *);    /* Writes border and items for menu      */
MENU *kill_menu(MENU *);          /* Releases a menu's memory              */
int  activate_pop_up_menu(MENU *,int,int,int *); 
                                  /* Displays and activates a pop-up menu  */

int  pop_up_warning(char *,int,int,int);
                                  /* Displays a pop-up warning window      */
char *pop_up_text(char *,char *,int);
                                  /* Displays a pop-up text-entry window   */
void write_text(WINDOW *,int,int,int,int,char *);
                                      /* Writes justified text to a window  */
void change_popup_option(MENU*,int,short[]); 


/* =======================================================================
   ***********************************************************************
                   INTERNALLY VISIBLE FUNCTION PROTOTYPES
   ***********************************************************************
   ======================================================================= */

static void pop_up_highlight(WINDOW *,int,ch_attr,int,ch_attr);
                                     /* Highlights a pop-up menu selection */


/* =======================================================================
   ***********************************************************************
                         EXTERNALLY VISIBLE FUNCTIONS
   ***********************************************************************
   ======================================================================= */

/******************************************************************************
 NAME        : new_menu
 PURPOSE     : Allocates a new menu
 DESCRIPTION : Using variable argument lists, this routine stores a menu
               and designs a window for it.
 INPUTS      : 1) A code indicating whether this routine should design a
                  visual representation of the menu or not :
                    VISUAL_POP_UP_MENU_NO  (0) - No visual image needed
                    VISUAL_POP_UP_MENU_YES (1) - Create an image
               2) The number of items in the menu (must be at least 1)
               3) The following two pieces of data for every item :
                 1) A string representing a menu item
                 2) A pointer to a handler function for that item
                    This function must return an integer corresponding to
                    one of the completion codes defined in MENU.H :
                      MENU_ITEM_COMPLETION (1) - Routine successfully completed
                      MENU_ITEM_ABORT (0) - Routine prematurely aborted
                    In addition, the routine must accept ONE integer argument
                    which will be the index of the item selected in the
                    menu (this allows one routine to handle several different
                    items in the menu if necessary).

               Example :

                 { The following is the handler function for item "Foo" }
                 int handle_Foo(index)
                   int index;
                   {
                    { --- Perform Stuff ---- }
                    return(MENU_ITEM_COMPLETION);
                   }

                 { Here is the call that would define a menu with 1 item }

                 MENU *menu;

                 menu = new_menu(1,"Foo",handle_Foo);

 RETURNS     : The address of the new allocated menu if everything went OK,
               NULL otherwise.
 NOTES       : None
 ******************************************************************************/
MENU *new_menu(int visual_code,int no_of_items,...)
  {
   MENU *menu;
   register int i;
   va_list ap;
   char *item_name;
   int (*item_handler)();

   if ((menu = balloc(1,MENU)) == NULL)
     return(NULL);
   menu->border = NULL;
   menu->text = NULL;
   menu->item_cnt = no_of_items;
   menu->items = NULL;
   if ((menu->items = balloc(no_of_items,MENU_ITEM *)) == NULL)
     return(kill_menu(menu));
   for (i = 0 ; i < no_of_items ; i++)
     (menu->items)[i] = NULL;
   va_start(ap,no_of_items);
   for (i = 0 ; i < no_of_items ; i++)
     {
      item_name = va_arg(ap,char *);
      item_handler = va_arg(ap,INT_FNX_ADDRESS);

      if (((menu->items)[i] = balloc(1,MENU_ITEM)) == NULL)
        return(kill_menu(menu));
      if (((menu->items)[i]->name = balloc(strlen(item_name)+1,char)) == NULL)
        return(kill_menu(menu));
      strcpy((menu->items)[i]->name,item_name);
      (menu->items)[i]->handler = item_handler;
     }   
   va_end(ap);
   if (visual_code == VISUAL_POP_UP_MENU_YES)
     {
      if (! create_visual_menu(menu))
        return(kill_menu(menu));
     }              
   return(menu);
  }

/******************************************************************************
 NAME        : create_visual_menu
 PURPOSE     : Designs a physical representation of a menu
 DESCRIPTION : This routine creates a physical menu which can be later
               displayed and used by a pop-up routine.  It attaches the
               address of the new window to the menu record.
 INPUTS      : 1) The address of the already created internal menu.
 RETURNS     : TRUE (1) if everything went OK, FALSE (0) otherwise.
                (An error will only occur if the menu is too big for the
                 screen or the memory could not be allocated.)
 NOTES       : 1) Uses the color defns in the file COLOR.C for pop-up menus
               2) The window will have the following dimensions :
                  # lines = # of menu items + 2 for borders
                  # columns = width of biggest menu item + 2 for borders
                  The first item is automatically highlighted.
               3) If a visual menu is already present, that one will be
                  deallocated before the new design is attached.
                    If the this routine fails, the old image will be left
                  intact.
                    This feature allows the caller to easily redraw the
                  visual menu with new colors without having to reallocate
                  a whole new internal menu.
               4) The visual menu is initially created starting at screen
                  coordinates (0,0), but the Curses routine mvwin() can
                  later be called to place the menu wherever desired.
                     The window is actually comprised of two parts :
                  the main window which holds the border, and the text
                  selection window which is a sub-window of the first.
                  The text window is initially set to coordinates (1,1).
                  mvwin() must be applied equally to BOTH windows.
                    The text window should be flushed to the screen AFTER
                  the border window.
               5) WARNING!!! - This routine should not be called if the
                  Curses routine initscr() has noy yet been called!
 ******************************************************************************/
static int create_visual_menu(menu)
  MENU *menu;
  {
   int maxwid,wid;
   register int i;

   /* ===========================================================
      Find the width of the biggest menu item and check the sizes
      =========================================================== */
   for (maxwid = 0 , wid = 0 , i = 0 ; i < menu->item_cnt ; i++)
     {
      wid = strlen((menu->items)[i]->name);
      if (wid > maxwid)
        maxwid = wid;
     }
   if ((menu->item_cnt > CGA_LINES-2) || (maxwid > ALL_COLS-2))
     return(FALSE);
 
   /* =====================================
      Deallocate old images (if they exist)
      ===================================== */
   if (menu->text != NULL)
     {
      delwin(menu->text);
     }
   if (menu->border != NULL)
     {
      delwin(menu->border);
     }

    /* ====================
       Allocate the windows
       ==================== */
   if ((menu->border = newwin(menu->item_cnt+2,maxwid+2,0,0)) == NULL)
     return(FALSE);
   if ((menu->text = subwin(menu->border,menu->item_cnt,maxwid,1,1)) == NULL)
     {
      delwin(menu->border);
      return(FALSE);
     }
   scrollok(menu->text,FALSE);
   nodelay(menu->text,TRUE);
   keypad(menu->text,TRUE);
   draw_pop_up_menu(menu);
   return(TRUE);
  }

/******************************************************************************
 NAME        : draw_pop_up_menu
 PURPOSE     : Draws border and items for pop-up menu with appropriate colors
 DESCRIPTION : 
 INPUTS      : 1) The address of the menu structure
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void draw_pop_up_menu(menu)
  MENU *menu;
  {
   ch_attr border_attr,unhgh_attr,hgh_attr;
   register int i;
   int maxwid;

   /* ========================================
      Define the colors and stats for the menu
      ======================================== */
   border_attr = colors[POP_MENU_BORDER] | colors[POP_MENU_BGD];
   unhgh_attr = colors[POP_MENU_TEXT] | colors[POP_MENU_BGD];
   hgh_attr = flip_fgd_bgd(unhgh_attr);
   getmaxc(menu->text,maxwid);

   /* ===============
      Draw the border
      =============== */
   xpaint(menu->border,border_attr);
   xbox(menu->border,DB_V_LIN,DB_H_LIN,border_attr);
   wmove(menu->border,0,0);

   /* =========================
      Write the menu selections
      ========================= */
   xpaint(menu->text,unhgh_attr);
   wattrset(menu->text,unhgh_attr);
   for (i = 0 ; i < menu->item_cnt ; i++)
     {
      wmove(menu->text,i,maxwid/2-strlen((menu->items)[i]->name)/2);
      waddstr(menu->text,(menu->items)[i]->name);
     }
   pop_up_highlight(menu->text,0,unhgh_attr,0,hgh_attr);
   wmove(menu->text,0,0);
  }
  
/******************************************************************************
 NAME        : change_popup_option
 PURPOSE     : This routine is called to mark or unmark the option
	       in the popup routine.Make sure that there is a blank in front
	       of the option for the mark.
 DESCRIPTION : 
 INPUTS      : Menu    : Option menu
	       Item    : Index of the option in the menu (first one is 0)
               Flag_set: Array of flags related to the options.
		  For example, if the menu has 10 options then the array should
		  also have 10 items. Each item is ON or Off depend on the
		  option is marked or not.
 RETURNS     : none
 NOTES       : 
 ******************************************************************************/

void change_popup_option(menu,item,flag_set)
MENU  *menu;
int   item;
short flag_set[];
{
   wmove(menu->text,item-1,0);
   wattrset(menu->text,winat(menu->text));
   if(flag_set[item-1] == FALSE)
      xwaddch(menu->text,(char)4);
   else
      waddch(menu->text,' ');
   wmove(menu->text,item-1,0);
}

/******************************************************************************
 NAME        : kill_menu
 PURPOSE     : Releases memory associated with a menu
 DESCRIPTION : 
 INPUTS      : 1) Address of the menu to be released
 RETURNS     : NULL so that a user may delete a menu and set the address to
               NULL in one blow.

               Example : menu_address = deallocate_menu(menu_address);

 NOTES       : None
 ******************************************************************************/
MENU *kill_menu(menu)
  MENU *menu;
  {
   register int i;

   if (menu->text != NULL)
     {
      delwin(menu->text);
     }
   if (menu->border != NULL)
     {
      delwin(menu->border);
     }
   if (menu->items != NULL)
     {
      for (i = 0 ; i < menu->item_cnt ; i++)
        {
         if ((menu->items)[i] != NULL)
           {
            if ((menu->items)[i]->name != NULL)
              release(strlen((menu->items)[i]->name)+1,
                              char,(menu->items)[i]->name);
            release(1,MENU_ITEM,(menu->items)[i]);
           }
        }
      release(menu->item_cnt,MENU_ITEM *,menu->items);
     }
   release(1,MENU,menu);
   return(NULL);
  }

/******************************************************************************
 NAME        : activate_pop_up_menu
 PURPOSE     : Displays and activates a pop-up menu
 DESCRIPTION : 
 INPUTS      : 1) The address of the menu to be activated.
               2) The screen row to start the menu on
               3) The screen column to start the menu on
               4) The address of a variable where the index
                    (1..n for n selections) can be stored.
                  If this address is NULL the index will not be stored.
 RETURNS     : ERROR (-1) if the menu could not be displayed for some
               reason or MENU_ABORT (0) if the user cancels the menu, 
               otherwise the routine returns the status code of the
               menu item selected :
                    MENU_ITEM_COMPLETION
                    MENU_ITEM_CLIPS_INPUT
               If the item returns MENU_ITEM_ABORT, the pop-up menu stays
               on the screen cycling for input.
 NOTES       : Assumes Curses is initialized as well as the menu structure
               and windows.
 ******************************************************************************/
int activate_pop_up_menu(menu,start_row,start_col,sel)
  MENU *menu;
  int start_row,start_col;
  int *sel;
  {
   int current_selection = 1,
       last_selection = 1;
   int rows,cols;
   ch_attr unhgh_attr,hgh_attr;
   int ch,ms_y,ms_x,ms_button,
       item_execution_rtn,
       quit = FALSE;

   if ((menu->border == NULL) || (menu->text == NULL))
     return(ERROR);
   getmaxrc(menu->border,rows,cols);
   if ((rows > LINES) || (cols > COLS))
     return(ERROR);
   if (start_row < 0)
     start_row = 0;
   else if (start_row + rows > LINES)
     start_row = LINES-rows;
   if (start_col < 0)
     start_col = 0;
   else if (start_col + cols > COLS)
     start_col = COLS-cols;
   (void) mvwin(menu->border,start_row,start_col);
   (void) mvwin(menu->text,start_row+1,start_col+1);
   unhgh_attr = colors[POP_MENU_TEXT] | colors[POP_MENU_BGD];
   hgh_attr = flip_fgd_bgd(unhgh_attr);
   insert_pop_up(menu->border);
   insert_pop_up(menu->text);

   redisplay_wins();
   hide_cursor();
   while (! quit)
     {
      if (current_selection != last_selection)
        {
         pop_up_highlight(menu->text,last_selection-1,unhgh_attr,
                                     current_selection-1,hgh_attr);
         wmove(menu->text,start_row+current_selection,start_col+1);
         wrefresh(menu->text);
         hide_cursor();
         last_selection = current_selection;
        }
      if ((ch = wgetch(menu->text)) != -1)
        {
         ch = key_map(ch);
         if (ch == NEWLINE)
           {
            item_execution_rtn = 
              ((menu->items)[current_selection-1]->handler)(current_selection);
            unhgh_attr = colors[POP_MENU_TEXT] | colors[POP_MENU_BGD];
            hgh_attr = flip_fgd_bgd(unhgh_attr);
            if (item_execution_rtn == MENU_ITEM_ABORT)
              {
                 delete_pop_up(menu->text);
                 delete_pop_up(menu->border);
                 insert_pop_up(menu->border);
                 insert_pop_up(menu->text);
                 redisplay_wins();
              }
            else if(item_execution_rtn == MENU_ITEM_COMPLETION)
              quit = TRUE;
           }
         else if (ch == ESC)
           {
            last_selection = current_selection;
            item_execution_rtn = MENU_ABORT;
            quit = TRUE;
           }
         else if (ch == UP_ARROW)
           current_selection = 
               (current_selection + menu->item_cnt - 2) % menu->item_cnt + 1;
         else if (ch == DOWN_ARROW)
           current_selection = current_selection % menu->item_cnt + 1;
         else if (ch == REDRAW_CMD)
           redisplay_wins();
         else
           beep();
        }
     }
   pop_up_highlight(menu->text,last_selection-1,unhgh_attr,0,hgh_attr);
   delete_pop_up(menu->text);
   delete_pop_up(menu->border);
   if (sel != NULL)
     *sel = current_selection;
   redisplay_wins();
   return(item_execution_rtn);
  }

/******************************************************************************
 NAME        : pop_up_warning
 PURPOSE     : Displays a pop-up warning message
 DESCRIPTION : Pop-up a warning message and prompts the user for a yes/no
               response.
                 The pop-up window is always centered on the screen.
 INPUTS      : 1) The message to be displayed (May have imbedded CRTs)
               2) The justification with which the text should be displayed
                  LFT_JUST (0) - Left Justification
                  MID_JUST (1) - Middle (Center) Justification
                  RGT_JUST (2) - Right Justification
               3) A code indicating whether to ring the terminal bell
                    SILENCE (0) - Don't ring bell
                    BEEP (1) - Ring the bell
 RETURNS     : 1) ERROR (-1) if the warning message could not fit on the
                  screen or the window could not be allocated
               2) YES (1) if the user indicated "yes" to the routine's prompt
               3) NO (0) if the user indicated "no" to the routine's prompt
 NOTES       : The window will require the following screen dimensions :
                 Lines : the number of lines required by the message +
                           4 for the yes/no prompt + 2 for the border
                 Columns : the number of columns required by the message or
                           the number of columns required by the prompt
                           (whichever is greater) + 2 for the border.
                           The prompt requires at most 42 columns.
 ******************************************************************************/
int pop_up_warning(warn_msg,just,bell,confirm)
  char *warn_msg;
  int just,bell,confirm;
  {
   register int i;
   int wid,maxwid,winwid,
       lns,winlns,y,x; 
   WINDOW *border,
          *text;
   char prompt1[43],
        prompt2[43];
   char *line;
   int promptwid,
       lstart,ch,
       quit = FALSE;

   /*=================================================================
     Determine how many lines are required for the message and also
     find the width of the longest line.  (The window's minimum width
     must be big enough for the deletion-message)
     =================================================================*/
   for (maxwid = wid = 0 , i = 0 , lns = 1 ; warn_msg[i] != EOS ; i++)
     {
      if ((warn_msg[i] != NEWLINE) ? (warn_msg[i] == CRGRTN) : TRUE)
        {
         lns++;
         if (wid > maxwid)
           maxwid = wid;
         wid = 0;
        }
      else
        wid++;
     }
    if (wid > maxwid)
      maxwid = wid;

   /* ========================================
      Determine the width of the yes/no prompt
      ======================================== */
   if(confirm)
    {
      strcpy(prompt1," <RTN>     <ESC>");
      strcpy(prompt2,"for YES   for NO");
      promptwid = 16;
    }
   else
    {
      strcpy(prompt1,"                                ");
      strcpy(prompt2,"Hit any key to get back to CRSV ");
      promptwid = 31;

    }
   winwid = ((maxwid > promptwid) ? maxwid : promptwid) + 2;
   winlns = lns + 5;
   if (winwid > COLS)
     return(ERROR);
   if (winlns > LINES)
     return(ERROR);
   y = LINES/2 - winlns/2;
   x = COLS/2 - winwid/2;
   if ((y < 0) || ((y+winlns) > LINES))
     return(ERROR);
   if ((x < 0) || ((y+winwid) > COLS))
     return(ERROR);
   if ((border = newwin(winlns,winwid,y,x)) == NULL)
     return(ERROR);
   scrollok(border,FALSE);
   if ((text = subwin(border,winlns-2,winwid-2,y+1,x+1)) == NULL)
     {
      delwin(border);
      return(ERROR);
     }
   scrollok(text,FALSE);
   nodelay(text,TRUE);

   xpaint(border,colors[POP_WARN_BGD]);
   xbox(border,DB_V_LIN,DB_H_LIN,colors[POP_WARN_BORDER]|colors[POP_WARN_BGD]);
   xpaint(text,colors[POP_WARN_BGD]);
   wattrset(text,colors[POP_WARN_TEXT]|colors[POP_WARN_BGD]);

   for (i = lstart = 0, line = warn_msg, lns = 0 ; warn_msg[i] != EOS ; i++)
     {
      if ((warn_msg[i] != NEWLINE) ? (warn_msg[i] == CRGRTN) : TRUE)
        {
         ch = warn_msg[i];
         warn_msg[i] = EOS;
         write_text(text,lns++,winwid-2,i-lstart,just,line);
         warn_msg[i] = ch;
         lstart = i + 1;
         line = warn_msg + lstart;
        }
      else if (warn_msg[i+1] == EOS)
        write_text(text,lns,winwid-2,i-lstart+1,just,line);
     }
   mvwaddstr(text,winlns-4,(winwid-2)/2-promptwid/2,prompt1);
   mvwaddstr(text,winlns-3,(winwid-2)/2-promptwid/2,prompt2);

   insert_pop_up(border);
   insert_pop_up(text);
   if (bell == BEEP)
     beep();
   redisplay_wins();
   hide_cursor();
   while (! quit)
     {
      if ((ch = wgetch(text)) != -1)
        {
         ch = key_map(ch);
         if(!confirm)
            quit = TRUE;
         else if (ch == NEWLINE)
           {
            ch = YES;
            quit = TRUE;
           }
         else if (ch == ESC)
           {
            ch = NO;
            quit = TRUE;
           }
         else if (ch == REDRAW_CMD)
           redisplay_wins();
         else
           beep();
        }
     }
   delete_pop_up(text);
   delete_pop_up(border);
   delwin(text);
   delwin(border);
   redisplay_wins();
   return(ch);
  }

/******************************************************************************
 NAME        : pop_up_text
 PURPOSE     : Grabs temporary text from a pop-up window
 DESCRIPTION : Allows the user to enter text for pop-up menu command arguments
               Used when the program does not want I/O with the main CLIPS
               window.
                 The user signals completion of input by pressing <RETURN> or
               <LEFT-CLICK> (if the mouse is loaded).
                 The user can cancel the window at any time by pressing
               <ESC> or <RIGHT-CLICK> (if the mouse is loaded).
 INPUTS      : 1) A prompt string (May have no imbedded carriage-returns)
               2) The address of a caller-allocated buffer to hold the text
                  entered.
               3) The size of the buffer EXCLUDING the space for a null char
                  (the size must be in the range 1..77)
 RETURNS     : 1) The address of the user's buffer if all went well
                  (the user's buffer will now hold the entered text)
               2) NULL if the user cancelled the pop-up box or the window
                  could not be displayed.
 NOTES       : 1) After calling this routine, whatever was in the user's buffer
                  is overwritten.
               2) The pop-up window is always displayed centered on the screen.
               3) The pop-up window will have the following dimensions :
                  Lines = 1(prompt) + 1(text input) + 2(border) = 4
                  Columns = 2(border) + the greatest of the following :
                                 Caller buffer size + 1
                                 Caller prompt
               4) The input text area can be smaller than the actual window
                  and will be indicated by a different color.
                    In this regard, the routine uses the following mechanism
                  to insure the text area has a different background color :
                     1) The first try is the backgd color which corresponds
                        to the same bit pattern as the foreground color.
                     2) If the first try is the same as the other background,
                        the routine flips the foreground and background colors.

                   Example : Say the main background is BLUE and the text
                             input foreground is defined to be RED.
                             Since the background color corresponding to the
                             RED bit pattern is BLUE, the the fgd and bgd will
                             be flipped and the corresponding text input color
                             will be BLUE on RED.
 ******************************************************************************/
char *pop_up_text(prompt,buf,bufsize)
  char *prompt,*buf;
  int bufsize;
  {
   register int i;
   int promptsize,
       wincols;
   WINDOW *border,*text;
   ch_attr text_bgd,text_attr;
   int quit = FALSE,
       ch,y,x,
       chcnt;
   char *rtn;

   /* ===================================
      Check the validity of the arguments
      =================================== */
   if ((buf == NULL) || (prompt == NULL) || (bufsize < 1) || (bufsize > COLS-3))
     return(NULL);
   for (i = 0 ; prompt[i] != EOS ; i++)
     if ((prompt[i] == CRGRTN) || (prompt[i] == NEWLINE))
       return(NULL);
   if ((promptsize = i) > COLS-2)
     return(NULL);

   wincols = (promptsize > bufsize+1) ? promptsize + 2 : bufsize + 3;
   if ((border = newwin(4,wincols,LINES/2-2,COLS/2-wincols/2)) == NULL)
     return(NULL);
   if ((text = subwin(border,1,bufsize+1,LINES/2,COLS/2-(bufsize+1)/2)) == NULL)
     {
      delwin(border);
      return(NULL);
     }
   scrollok(border,FALSE);
   scrollok(text,FALSE);
   nodelay(text,TRUE);
   xpaint(border,colors[POP_TEXT_BGD]);
   xbox(border,DB_V_LIN,DB_H_LIN,colors[POP_TEXT_BORDER]|colors[POP_TEXT_BGD]);
   wattrset(border,colors[POP_TEXT_PROMPT]|colors[POP_TEXT_BGD]);
   mvwaddstr(border,1,1,prompt);
   text_attr = colors[POP_TEXT_INPUT] | fgd_reverse(colors[POP_TEXT_INPUT]);
   if ((text_attr & FGD_MSK) == colors[POP_TEXT_BGD])
     text_attr = flip_fgd_bgd(text_attr);
   xpaint(text,text_attr);
   wattrset(text,text_attr);
   wmove(text,0,0);
   insert_pop_up(border);
   insert_pop_up(text);
   redisplay_wins();
   wrefresh(text);

   chcnt = 0;
   buf[chcnt] = EOS;
   while (! quit)
     {
      if ((ch = wgetch(text)) != -1)
        {
         ch = key_map(ch);
         if ((ch >= LOW_PRN_ASCII) && (ch <= HIGH_PRN_ASCII))
           {
            if (chcnt < bufsize)
              {
               buf[chcnt++] = ch;
               buf[chcnt] = EOS;
               waddch(text,ch);
               wrefresh(text);
              }
            else
              beep();
           }
         else if (ch == DELETE)
           {
            if (chcnt != 0)
              {
               buf[--chcnt] = EOS;
               waddstr(text,DEL_STR);
               wrefresh(text);
              }
            else
             beep();
           }
         else if (ch == NEWLINE)
           {
            if (chcnt == 0)
              rtn = NULL;
            else
              rtn = buf;
            quit = TRUE;
           }
         else if (ch == ESC)
           {
            buf[0] = EOS;
            rtn = NULL;
            quit = TRUE;
           }
         else if (ch == REDRAW_CMD)
           {
            redisplay_wins();
            wrefresh(text);
           }
         else
           beep();
        }
     }
   delete_pop_up(text);
   delete_pop_up(border);
   delwin(text);
   delwin(border);
   redisplay_wins();
   return(rtn);
  }

/* =======================================================================
   ***********************************************************************
                         INTERNALLY VISIBLE FUNCTIONS
   ***********************************************************************
   ======================================================================= */

/******************************************************************************
 NAME        : pop_up_highlight
 PURPOSE     : Highlights a particular row in a pop-up menu
 DESCRIPTION : 
 INPUTS      : 1) the pop-up menu window address
               2) the row to be unhighlighted in relative coordinates
               3) the unhighlighting fgd/bgd color attribute
               4) the row to be highlighted in relative coordinates
               5) the highlighting fgd/bgd color attribute
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void pop_up_highlight(win,urow,uattr,hrow,hattr)
  WINDOW *win;
  int urow,hrow;
  ch_attr uattr,hattr;
  {
   register int i;
   int maxcols;

   getmaxc(win,maxcols);
   wattrset(win,uattr);
   wmove(win,urow,0);
   for (i = 0 ; i < maxcols ; i++)
    xwaddch(win,winch(win) & A_CHARTEXT);
   wattrset(win,hattr);
   wmove(win,hrow,0);
   for (i = 0 ; i < maxcols ; i++)
     xwaddch(win,winch(win) & A_CHARTEXT);
  }

/******************************************************************************
 NAME        : write_text
 PURPOSE     : Writes justified text to a window
 DESCRIPTION : 
 INPUTS      : 1) The address of the window
               2) The line number to which the text is to be written
               3) The width of the window
               4) The width of the text
               5) The justification with which the text is to be written
                  LFT_JUST (0) - Left Justification
                  MID_JUST (1) - Middle (Center) Justification
                  RGT_JUST (2) - Right Justification
               6) The text which is to be written
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void write_text(win,ln_no,winwid,lnwid,just,line)
  WINDOW *win;
  int ln_no,winwid,lnwid,just;
  char *line;
  {
   if (just == LFT_JUST)
     mvwaddstr(win,ln_no,0,line);
   else if (just == MID_JUST)
     mvwaddstr(win,ln_no,winwid/2-lnwid/2,line);
   else if (just == RGT_JUST)
     mvwaddstr(win,ln_no,winwid-lnwid,line);
  }

/******************************************************************************
 NAME        : 
 PURPOSE     : 
 DESCRIPTION : 
 INPUTS      : 
 RETURNS     : 
 NOTES       : 
 ******************************************************************************/

