/* ************************************************************************
   FILE : KEY.C
   This file contains the support routines that handle raw keystroke
   mapping to CRSV action codes.
   *********************************************************************** */

/******************************************************************************
 ==============================================================================
                                 HEADER FILES
 ==============================================================================
 ******************************************************************************/

#include "curses.h"     /* Curses constants, macros and function prototypes */
#include "box.h"

#include "crsv.h"

#define KEY_SOURCE

#include "pcint.h"     /* Common constants and memory macros */

/******************************************************************************
 ==============================================================================
                                   CONSTANTS
 ==============================================================================
 ******************************************************************************/


/******************************************************************************
 ==============================================================================
                                    MACROS
 ==============================================================================
 ******************************************************************************/

/* 
 * ---------------------------------------------------------------
 * Returns a string representing the control or extended character
 * --------------------------------------------------------------- 
 */

#define keyname(ch) ((ch<LOW_PRN_ASCII) ? keycnames[ch]:keyexnames[ch-KEY_F(1)])


/******************************************************************************
 ==============================================================================
                     EXTERNALLY VISIBLE FUNCTION PROTOTYPES
 ==============================================================================
 ******************************************************************************/

int key_map(int);          /* Maps raw keyboard strokes into CLIPS actions */


/******************************************************************************
 ==============================================================================
                      EXTERNALLY VISIBLE GLOBAL VARIABLES
 ==============================================================================

 All the Key-Binding tables together take up ~2K - most of which is the
   space taken by the string-tables for the keys for the run-time binding
   routine.

 Possible Improvements :

 1) More space efficient way of determining the string-representation of a key
    (The problem is not the control-keys - those are easy - the problem is that
   Curses has ordered the extended keystrokes in a funky way.  For instance,
   The alphabetic Alt-Characters don't follow any kind of ordering based
   on the ascii one, but rather follows the order on a keyboard (ALt-Q,Alt-W,
   Alt-E, etc.)).

 2) Make the keystroke-string tables local variables to the bind routine.
    (Yuuuck!) - This would probably force your compiler to give you
    mucho more stack space at link-time.  I do not recommend this solution.

* ******************************************************************************/

/* --------------------------------------------------------------------------
   The following two tables are the heart and core of the key-bindings.
   Each CRSV action code is mapped to a particular keystroke by placing
   that code in one of the two tables in the position which corresponds
   to the key desired.
     Example : CTRL-C is bound to the CRSV action EXIT_TO_OS 
                (i.e. abort CRSV)
               The ascii/numeric code of CTRL-C is 3 which places it within
                 the bounds of the standard ascii control character table :
                 keyctab[].  The action for CTRL-C thus corresponds to
                 keyctab[3].  Looking up key-binds thus becomes a trivial task.
               As a side-effect of this method, each action can be bound to
                 several different keystrokes (The key-binding routine will
                 only let you set two at run-time - which I thought was
                 plenty.)
   -------------------------------------------------------------------------- */

unsigned char keyctab[MAX_CNTRL_KEYS] =          /* Control Characters */
      { 
       BAD_KEY,
       BAD_KEY,
       LEFT_ARROW,
       EXIT_TO_OS,     /* CTRL-C */
       CTRL_D,
       BAD_KEY,
       RIGHT_ARROW,
       BAD_KEY,
       DELETE,         /* CTRL-H/DELETE */
       BAD_KEY,
       NEWLINE,        /* CTRL-J/NEWLINE */
       BAD_KEY,
       REDRAW_CMD,
       NEWLINE,       /* CTRL-M/CRGRTN */
       DOWN_ARROW,
       BAD_KEY,
       UP_ARROW,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       CLEAR_WIN,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       ESC,           /* CTRL-[/ESCAPE */
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY
      },

     keyextab[MAX_EXTND_KEYS] =    /* Function Keys and Extended Keypad Keys */
      {
       BAD_KEY,               /* Function Keys */
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,              /* Shift + Function Keys */
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,                  /* Ctrl + Function Keys */
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,                  /* Alt + Function Keys  */
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,               /* BackTab */
       BAD_KEY,               /* ^Break */
       BAD_KEY,               /* CTRL-LEFT */
       BAD_KEY,               /* CTRL-RIGHT */
       BAD_KEY,               /* CTRL-END */
       BAD_KEY,               /* CTRL-PGDN */
       BAD_KEY,               /* CTRL-HOME */
       BAD_KEY,               /* CTRL-PGUP */
       DOWN_ARROW,            /* DOWN */
       UP_ARROW,              /* UP */
       LEFT_ARROW,            /* LEFT */
       RIGHT_ARROW,           /* RIGHT */
       BUFFER_TOP,            /* HOME */
       BUFFER_END,            /* END */
       PAGE_DOWN,             /* PGDN */
       PAGE_UP,               /* PGUP */
       BAD_KEY,               /* KEYPAD DELETE */
       BAD_KEY,               /* INSERT */
       BAD_KEY,               /* PRINT SCREEN */
       BAD_KEY,               /* Alt 1 - 0 */
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,                /* Alt - */
       BAD_KEY,                /* Alt = */
       BAD_KEY,                /* Alt Q */
       BAD_KEY,                /* Alt W */
       BAD_KEY,                /* Alt E */
       RUN_CMD,                /* Alt R */
       BAD_KEY,                /* Alt T */
       BAD_KEY,                /* Alt Y */
       BAD_KEY,                /* Alt U */
       BAD_KEY,                /* Alt I */
       OPTION_MENU,            /* Alt O */
       BAD_KEY,                /* Alt P */
       BAD_KEY,                /* Alt A */
       SPEC_MENU,              /* Alt S */
       BAD_KEY,                /* Alt D */
       FILE_MENU,              /* Alt F */
       BAD_KEY,                /* Alt G */
       BAD_KEY,                /* Alt H */
       BAD_KEY,                /* Alt J */
       BAD_KEY,                /* Alt K */
       BAD_KEY,                /* Alt L */
       BAD_KEY,                /* Alt Z */
       BAD_KEY,              /* Alt X */
       BAD_KEY,                /* Alt C */
       BAD_KEY,                /* Alt V */
       BAD_KEY,                /* Alt B */
       BAD_KEY,                /* Alt N */
       BAD_KEY,                /* Alt M */
       BAD_KEY,                /* CTRL-UP */
       BAD_KEY                 /* CTRL-DOWN */
      };



/******************************************************************************
 ==============================================================================
                         EXTERNALLY VISIBLE FUNCTIONS
 ==============================================================================
 ******************************************************************************/

/******************************************************************************
 NAME        : key_map
 PURPOSE     : Maps raw keystrokes to CLIPS action codes
 DESCRIPTION : 
 INPUTS      : 1) The keystroke to be mapped
 RETURNS     : 1) The CLIPS action code if it was mapped
               2) The original keystroke otherwise
 NOTES       : Only ascii control characters and Curses extended keyboard
               characters are mapped.
 ******************************************************************************/
int key_map(chord)
  int chord;
  {
   if ((chord >= LOW_PRN_ASCII) && (chord <= HIGH_PRN_ASCII))
     return(chord);
   else if (chord > MAX_IBMPC_KEY)
     return(chord);
   else if (chord < LOW_PRN_ASCII)
     return((int) keyctab[chord]);
   else
     return((int) keyextab[chord-KEY_F(1)]);
  }


