/**************************************************************************
   FILE : PCINT.C
     This file contains the main driver for the CRSV interface, all
   the interfaces with CRSV internals, and all of the primary driving
   routines for the visual interface.  Support Routines can be found in
   the files : PCMENU.C, PCPOPUP.C, PCKEY.C
     All of these source files have HEADER files which correspond to them.
   *********************************************************************** */
   
/* =======================================================================
   ***********************************************************************
                             HEADER FILES
   ***********************************************************************
   ======================================================================= */

#include <curses.h>      /* Aspen Scientific Curses V4.0 Header Files -   */
#include <xcurses.h>     /*    Screen interface function prototypes and   */
#include <box.h>         /*      constants                                */

#include "crsv.h"

#define CRSVWIN_SOURCE
#define COLOR_SOURCE

#include "pcint.h"      /* Standard header files, constants, and macros  */
#include "pcconst.h"


/* =======================================================================
   ***********************************************************************
            EXTERN PROTOTYPES FOR CRSV FUNCTIONS NOT IN THIS FILE
   ***********************************************************************
   ======================================================================= */

/* CRSV Menu Interface Functions */

extern int do_main_menu(int),  /* Main menu options */
       do_help(int);

extern int do_add(int),       /* File menu options */
       do_remove(int),
       do_move_up(int),
       do_move_down(int), 
       do_dribble(int),
       do_exit(int);  

extern int do_run(int);     /* Execute menu options */

extern int do_options(int),  /* Options menu options */
       do_be_verbose(int),
       do_crss_ref_rel(int),
       do_style_wrng_anal(int),
       do_rule_deff_sum(int),
       do_ver_rule_w_defrel(int),
       do_crss_ref_usr_func(int),
       do_create_defrel_file(int),
       do_drib_sum(int),
       do_anal_drib(int);

extern int do_pause(int),   /* Special menu options */
           do_clear(int);
   
extern int do_nothing(int);     /* Watch/Unwatch functions */


/* -----------------
 *  from CRSVMEM.C 
 * -----------------
 */

extern long int mem_used();


/* =======================================================================
   ***********************************************************************
                   EXTERNALLY VISIBLE FUNCTION PROTOTYPES
   ***********************************************************************
   ======================================================================= */

void get_environment(void);     
void main(int,char *[]);                /* Main driver                   */

void stop_interface(int);               /* Interface cleanup routine     */

/* Window Interface Routines */

void redisplay_wins(void);              /* Redraws entire screen         */

int  insert_pop_up(WINDOW *);           /* Puts window in pop-up queue   */
int  delete_pop_up(WINDOW*);             /* Takes the specific window from queue  */

void pause_interface(void);   /* Sets terminal to normal state                */
void restart_interface(int); /* Resets terminal to valid state for interface */
int  main_pop_up_start(int);     /* Gets center screen box coord  */

/*  I/O Stufffs */

void crsv_error_message(char*,char*);
void crsv_message(char*);
int crsv_addch(char);          /* Curses linefeed and scroll    
                                           glitches hack-fixes           */
void crsv_advance_row(void);   /* Moves cursor to new line       */
void crsv_addstr(char *);      /* Adds a whole string to window  */
void crsv_scroll(int,int);     /* Handles window paging          */
void scroll_to_end(void);       /* Forces view to end of buffer   */
void pause_output(void);        /* --More-- Utility for output    */
void crsv_clear_win(void);     /* To clear the entire CRSV window */


/* =======================================================================
   ***********************************************************************
                       EXTERNALLY VISIBLE GLOBAL VARIABLES
   ***********************************************************************
   ======================================================================= */

/* Screen Interface Colors */

ch_attr colors[MAX_COLOR_DEFS] =
  {
   A_FWHITE,       /*                               */
   A_FCYAN,        /* stdin color                   */
   A_BBLACK,       /* background color              */
   A_FBLUE,        /* border_color                  */
   A_FLT_RED,      /* advertisement color           */
   A_BBLACK,       /* main menu background color    */
   A_FGREEN,       /* main menu text color          */
   A_FLT_GREEN,    /* --More-- message color        */
   A_BWHITE,       /* pop up menu background color  */
   A_FRED,         /* pop up menu border color      */
   A_FRED,         /* pop up menu text color        */
   A_BCYAN,        /* pop up text background color  */
   A_FBLUE,        /* pop up text border color      */
   A_FBLUE,        /* pop up text prompt color      */
   A_FBLUE,        /* pop up text input color       */
   A_BRED,         /* pop up warn background color  */
   A_FYELLOW,      /* pop up warn border color      */
   A_FYELLOW,      /* pop up warn text color        */
   A_BWHITE,       /* pop up files menu background  */
   A_FBLUE,        /* pop up files menu border      */
   A_FCYAN,        /* pop up files menu text        */
   A_FYELLOW       /* pop up files mouse cursor     */
  };

WINDOW
   *file_win_border      = NULL,
   *file_win_text        = NULL,
   *drib_file_win_border = NULL,
   *drib_file_win_text   = NULL,        
   *def_rel_win_border   = NULL,
   *def_rel_win_text     = NULL,
   *about_crsv_win_border= NULL,
   *about_crsv_win_text  = NULL,
   *debug_on_win         = NULL,
   *mem_used_win         = NULL;

/* -------------------------------------------------------------
   List of Menu Commands and the functions associated with them
   ------------------------------------------------------------- */

MENU *main_menu = NULL,
     *file_menu = NULL,
     *options_menu = NULL,
     *special_menu = NULL;

short option_flags[10] = {FALSE,FALSE,TRUE,FALSE,TRUE,
			  FALSE,FALSE,FALSE,FALSE,FALSE};

short PAUSE_OUT_PUT = FALSE;

/* -----------------------------------------------------------
    Pop-Up Window Queue -- see crsvpc.h for WINBLK data type
   ----------------------------------------------------------- */

WINBLK *pop_up_queue_top = NULL,
       *pop_up_queue_bottom = NULL;

int    num_files;                        /* Number of input files */
char   file_list[MAX_FILES][MAX_NAME];   /* List of file names    */
char   def_rel_file[MAX_NAME];           /* Name for defrelations file */
char   drib_file_name[MAX_NAME];         /* Name for the trace_file */
FILE   *drib_fp;

/* =======================================================================
   ***********************************************************************
                   INTERNALLY VISIBLE FUNCTION PROTOTYPES
   ***********************************************************************
   ======================================================================= */

static void next_event(void);
static void draw_border_line(WINDOW*,int,int);
static void restore_environment(void);
static void init_interface(void);       /* Initial setup of screen       */

/* Window Interface Routines */

static int  allocate_windows(void);     /* Inits screen data structures  */
static int  allocate_menus(void);       /* Initializes menu structures   */
static void draw_border(void);          /* Displays the main border      */
static void draw_advertisement(void);   /* Displays CRSV on the screen  */
static void draw_main_menu(void);       /* Displays the main CRSV menu  */
static void draw_main_window(void);     /* Displays CRSV window text    */
static void draw_about_crsv_win(void);  /* Display detail info. about CRSV */
static void reset_colors(void);         /* Redraws screen with new colors*/
static void update_screen(void);        /* Flushes output to the screen  */
static void highlight_main_menu(int);   /* Highlights main menu selects  */
static void highlight_logo(void);        /* Highlights CRSV logo */
static void deallocate_pop_up_queue(void);
                                        /* Frees pop-up-queue memory     */
static int pop_up_about_crsv(void);     /* pop up CRSV information window */
static void deallocate_windows(void);   /* Frees interface memory        */
static void deallocate_menus(void);     /* Releases menu memory space    */
static void restore_environment(void);  /* Puts terminal in normal state */

/* I/O Stuff */

static int  process_special(int,int *);/* Handles special controls       */


/* =======================================================================
   ***********************************************************************
                       INTERNALLY VISIBLE GLOBAL VARIABLES
   ***********************************************************************
   ======================================================================= */


/* Screen Interface Windows */

static WINDOW
   *header_window        = NULL,
   *advertisement_window = NULL,
   *main_menu_window     = NULL,
   *left_border_window   = NULL,
   *right_border_window  = NULL,
   *footer_window        = NULL,
   *crsv_window          = NULL;

static int crsv_row = 0,
           crsv_col = 0;           /* Current cursor posn in the CRSV pad */

static int crsv_offset = 0;        /* # of pad rows off top of screen      */

static int main_menu_cell_width,    /* Width of a main menu selection cell  */
           main_menu_cell_offset,   /* # of blanks between left edge of     
                                       main menu and first cell             */
           main_menu_cell = 1;      /* Currently selected main menu cell    */


/* ------------------------------------------------------------------------
   Keyboard buffer-queue and unget buffer-stack for getc/ungetc simulations
   ------------------------------------------------------------------------ */

static char unget_buf[UNGET_BUF_SIZE];

/* --------------------------------------------------
   The unget buffer is a first-in/last-out stack
     (Thus requiring only one pointer)
   --------------------------------------------------- */

static int unget_buf_ptr = 0;   /* Pointer to char in unget buffer stack     */

/* ----------------------------------------------------------------------
   Number of input characters echoed to screen by win_getc()
   since last system output - this the maximum number of characters for
   which win_getc() will accept a DELETE.
   ---------------------------------------------------------------------- */

static int input_cnt = 0;

/* -----------------------------------------------------------------------
   Number of lines printed since last echoed input.  This variable is used
   to know when to halt output and query the user when to continue with
   the "--More--" message.
   ----------------------------------------------------------------------- */

static int output_lines = 0;

/* -------------------------------------------------------------------------
   The following map is used to keep track of what column position a newline
   occurred on for a particular line.  The map is used for deletions across
   newlines so as to know where to place the cursor. Besides, for such a
   nice feature, I figured 63 bytes wasn't a big deal.
     The nlmap_ptr index is used to keep track of which position in the map
   corresponds to line 0 -- thus when the window scrolls, it is only 
   necessary to shift this index, not the whole array.
   ------------------------------------------------------------------------- */

unsigned char newline_map[CRSV_PAD_LNS];
static int nlmap_ptr = 0;

/* =======================================================================
   ***********************************************************************
                         EXTERNALLY VISIBLE FUNCTIONS
   ***********************************************************************
   ======================================================================= */

/******************************************************************************
 NAME        : main
 PURPOSE     : Main driver for the CRSV interface
 DESCRIPTION : Initializes the screen interface windows, CRSV data
               data structures, performs any command preprocessing
               required, and activates the CRSV top level command loop.
 INPUTS      : 1) argc - the number of arguments CRSV was called with
               2) *argv[] - string array containing the arguments
 RETURNS     : Nothing useful.
 NOTES       : This function is identical to the standard MAIN.C included
               with CRSV with the exception of the calls to 
               init_interface() (this file).
 ******************************************************************************/
void PC_CRSV_interface()
  {
   num_files         = 0;
   drib_file_name[0] = EOS;
   def_rel_file[0]   = EOS;

   CHECK_RULES     = OFF;             /*  Option flag R  */
   CHECK_RELATIONS = OFF;             /*  Option flag X  */
   CHECK_EX_FLAG   = OFF;             /*  Option flag E  */
   CHECK_COMMENTS  = OFF;             /*  Not used currently, may use later */
   CHECK_STYLE     = ON;              /*  Option flag S  */
   VERBOSE         = OFF;             /*  Option flag V  */
   CREATE_DEFRELS  = OFF;             /*  Option flag C  */
   CHECK_DEFRELS   = ON;              /*  Option flag D  */
   CHECK_DEBUG     = OFF;             /*  Option flag Z  */

   /* ============  dribble file stuffs  =================== */

   ANALYZE_TRACE     = OFF;             /*  Option flag T  */

   /* ============ end of dribble file stuffs ============== */

   init_interface();
   redisplay_wins();
   next_event();
  }

/******************************************************************************
 NAME        : next_event
 PURPOSE     : Grabs characters form the keyboard for the command line
               as well as processing special function characters and mouse
               input.
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : 
 ******************************************************************************/
void next_event()
  {
   int ch,
       mn_menu_sel,
       quit = FALSE;

   mn_menu_sel = main_menu_cell;
   while (! quit)
     {
      if((mn_menu_sel == 0)&&(mn_menu_sel != main_menu_cell))
       {
          highlight_logo();
          redisplay_wins();
       }
      else if((mn_menu_sel > 0)&&(mn_menu_sel != main_menu_cell))
        {
         highlight_main_menu(mn_menu_sel - 1);
         redisplay_wins();
        }

      /* =================================================
         Poll the keyboard and handle any characters there
         ================================================= */
      if ((ch = wgetch(crsv_window)) != -1)
        {
             quit = process_special(key_map(ch),&mn_menu_sel);
        } 
      } 
    }


/******************************************************************************
 NAME        : stop_interface
 PURPOSE     : Cleans up the terminal environment and quits CRSV
 DESCRIPTION : Releases screen data structures, resets the terminal to a
               normal state, and quits CRSV
 INPUTS      : Ignored.
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void stop_interface(num)
  int num;
  {
   deallocate_menus();
   deallocate_windows();
   deallocate_pop_up_queue();
  }

/******************************************************************************
 NAME        : redisplay_wins
 PURPOSE     : Flushes interface to screen
 DESCRIPTION : This routine is used to redraw the interface windows on the
               screen after somethings has been done to disturb them --
               such as pop-up menus or the editor.
                 The routine works by un-optimizing all windows and then
               flushing them to the virtual screen.  The final step is to
               flush the virtual screen to the physical screen so it looks
               like the whole screen comes up at once.
 INPUTS      : None
 RETURNS     : Nothing useful.
 NOTES       : Assumes all windows are currently allocated.
               If there are no pop-up menus on the screen, the cursor is
                 left in the main CRSV window, otherwise it is hidden.
 ******************************************************************************/
void redisplay_wins()
  {
   WINBLK *qptr;

   touchwin(header_window);
   wnoutrefresh(header_window);
   touchwin(advertisement_window);
   wnoutrefresh(advertisement_window);
   touchwin(main_menu_window);
   wnoutrefresh(main_menu_window);
   touchwin(left_border_window);
   wnoutrefresh(left_border_window);
   touchwin(right_border_window);
   wnoutrefresh(right_border_window);
   wattrset(footer_window,colors[WFOOTER]|colors[MAIN_BACKGD]);
   touchwin(footer_window);
   wnoutrefresh(footer_window);
   touchwin(crsv_window);
   pnoutrefresh(crsv_window,crsv_offset,0,CRSV_Y,CRSV_X,
                CRSV_Y+LINES-HDR_LNS-FTR_LNS-1,CRSV_X+CRSV_COLS-1);
   if ((qptr = pop_up_queue_top) != NULL)   
     {
      while (qptr != NULL)
        {
         touchwin(qptr->win);
         wnoutrefresh(qptr->win);
         qptr = qptr->nxt;
        }
      doupdate();
      hide_cursor();

     }
   else
     {
      doupdate();

      /* ===================================================
         Hack to overcome pnoutrefresh() placement of cursor
         =================================================== */
      if (((crsv_row-crsv_offset) < (LINES-HDR_LNS-FTR_LNS)) &&
           (crsv_row >= crsv_offset))
        mvcur(0,0,CRSV_Y+crsv_row-crsv_offset,CRSV_X+crsv_col);
      else
        hide_cursor();
     }
  }

/******************************************************************************
 NAME        : insert_pop_up
 PURPOSE     : Puts a new window in the pop-up queue
 DESCRIPTION : The window is put in a queue which is used by redisplay_wins()
               After the main screen has been redrawn, this queue is 
               followed in order from top to bottom, and all windows found
               therein are flushed to the physical screen.
                 This makes it easy to "remove" pop-ups from the screen.
               Insertions/Deletions take place ONLY at the end of the queue.
 INPUTS      : 1) The address of the window to be put in the queue
                  The window MUST have been previously allocated and drawn.
 RETURNS     : TRUE (1) if everything was OK, FALSE (0) otherwise
 NOTES       : This routine does not check to see if a window is already
                 in the queue.
               Be very careful to maintain the equanimity between the inserts
               and deletes or unpredictable results might occur - it
               probably won't crash, but your screen will most likely look
               very strange.
 ******************************************************************************/
int insert_pop_up(win)
  WINDOW *win;
  {
   WINBLK *qptr;

   if ((qptr = balloc(1,WINBLK)) == NULL)
     return(FALSE);
   qptr->win = win;
   qptr->prv = NULL;
   qptr->nxt = NULL;
   if (pop_up_queue_top == NULL)
     {
      pop_up_queue_top = qptr;
      pop_up_queue_bottom = qptr;
     }
   else
     {
      pop_up_queue_bottom->nxt = qptr;
      qptr->prv = pop_up_queue_bottom;
      pop_up_queue_bottom = qptr;
     }
   return(TRUE);
  }

/******************************************************************************
 NAME        : delete_pop_up
 PURPOSE     : Removes a window from the bottom of the pop-up queue
 DESCRIPTION : See above
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
int delete_pop_up(win)
WINDOW *win;
  {
   WINBLK *qptr;

   if (pop_up_queue_bottom != NULL)
     {
      qptr = pop_up_queue_bottom;
      while((qptr != NULL)&&(qptr->win != win))
        qptr = qptr->prv;
      if(qptr == NULL)
        return(FALSE);
      if(qptr->nxt == NULL)
       {
          pop_up_queue_bottom = pop_up_queue_bottom->prv;
          if (pop_up_queue_bottom == NULL)
             pop_up_queue_top = NULL;
          else
             pop_up_queue_bottom->nxt = NULL;
       }
      else if(qptr->prv == NULL)
       {
           pop_up_queue_top = pop_up_queue_top->nxt;
           if(pop_up_queue_top == NULL)
              pop_up_queue_bottom = NULL;
           else
              pop_up_queue_top->prv = NULL;
       }
      else
       {
	    qptr->nxt->prv = qptr->prv;
            qptr->prv->nxt = qptr->nxt;
       }
      release(1,WINBLK,qptr);
      return(TRUE);
     }
    return(FALSE);
  }

/* =======================================================================
   ***********************************************************************
                         INTERNALLY VISIBLE FUNCTIONS
   ***********************************************************************
   ======================================================================= */

/**********************************************/
void pause_interface()
{
  restore_environment();
}
/**********************************************/
void restart_interface(query)
int query;
{
   if(query)
   {
     printf("\nPress any key to return to CRSV");
     while (wgetch(crsv_window) == -1);
   }
   get_environment();

}

/*********************************************/
void restore_environment()
{
   noraw();
   echo();
   nl();
}
/******************************************************************************
 NAME        : init_interface
 PURPOSE     : Sets up the initial window environment for CRSV
 DESCRIPTION : Allocates the windows, draws the screen setup.
 INPUTS      : None
 RETURNS     : Nothing useful.
 NOTES       : This function must be called before any I/O with CRSV occurs.
               Assumes interface is not already initialized.
 ******************************************************************************/
static void init_interface()
  {
   /* ================================================================
      Load CRSV path, names of color/bind dat files, and save default
      configuration to temporary file
      ================================================================ */

   if (! allocate_windows())
     {
      stop_interface(0);
     }
   if (! allocate_menus())
     {
      stop_interface(0);
     }
   draw_border();
   draw_advertisement();
   draw_main_menu();
   get_environment();

   /* ========================================================================
      draw_main_menu() redraws what is currently in the main window imposing a
      new background on the colors that are already there.  allocate_windows()
      xpaint-ed the main-window already as part of the initialization to 
      make sure that all cells had the STDIN foreground attribute.  This
      insures that the cursor will always be the same color (that's the
      good part), but this assumes that user-input will always be requested
      from the logical name STDIN-a fairly reasonable assumption (that's
      the pseudo-bad-part).  Clearing and scrolling the main CRSV window
      also use this attribute so that the cursor will not change color
      (see do_clear_window() and crsv_advance_row()).
         The bottom line is that we do not need to call draw_main_window()
      until such time as the background color changes .

      draw_main_window();
      ======================================================================= */

  }

/***********************************************************/
void get_environment()
{
	raw();
	nonl();
	noecho();
}


/******************************************************************************
 NAME        : allocate_windows
 PURPOSE     : Gets space for border, advertisement and main menu
 DESCRIPTION : To conserve space, separate segmented windows are used to
               cover the portion of the screen which is not covered by
               the main I/O window, i.e the border, advertisement, and
               main menu.
 INPUTS      : None
 RETURNS     : TRUE (1) if everything went well, FALSE (0) otherwise
 NOTES       : Should be called before interfacing with any Curses stuff
               Assumes windows are not already allocated.
 ******************************************************************************/
static int allocate_windows()
  {
   
   if (initscr() == NULL)
     return(FALSE);
   delwin(stdscr);     /* Don't need stdscr, so don't waste the space */
   stdscr = NULL;
   if ((header_window = newwin(HDR_LNS,HDR_COLS,HDR_Y,HDR_X)) == NULL)
     return(FALSE);
   scrollok(header_window,FALSE);
   if ((advertisement_window = subwin(header_window,AD_LNS,AD_COLS,AD_Y,AD_X))
        == NULL)
     return(FALSE);
   scrollok(advertisement_window,FALSE);
   if ((main_menu_window = subwin(header_window,MN_MENU_LNS,MN_MENU_COLS,
                                           MN_MENU_Y,MN_MENU_X)) == NULL)
     return(FALSE);
   scrollok(main_menu_window,FALSE);
   if ((left_border_window = newwin(LINES-HDR_LNS-FTR_LNS,LBORD_COLS,
                               LBORD_Y,LBORD_X)) == NULL)
     return(FALSE);
   scrollok(left_border_window,FALSE);
   if ((right_border_window = newwin(LINES-HDR_LNS-FTR_LNS,RBORD_COLS,
                                RBORD_Y,RBORD_X)) == NULL)
     return(FALSE);
   scrollok(right_border_window,FALSE);
   if ((footer_window = newwin(FTR_LNS,FTR_COLS,LINES-1,FTR_X)) == NULL)
     return(FALSE);
   scrollok(footer_window,FALSE);

   if ((crsv_window = newpad(CRSV_PAD_LNS,CRSV_COLS)) == NULL)
     return(FALSE);
   keypad(crsv_window,TRUE);
   nodelay(crsv_window,TRUE);
   xpaint(crsv_window,colors[STDIN]|colors[MAIN_BACKGD]);
   wattrset(crsv_window,colors[STDIN]|colors[MAIN_BACKGD]);
   if ((file_win_border = newwin(MAX_FILES + 6,MAX_NAME+7,FILE_WIN_LNS,FILE_WIN_COLS)) == NULL)
     return(FALSE);
   if ((file_win_text = subwin(file_win_border,MAX_FILES + 3,MAX_NAME+3,
                               FILE_WIN_LNS+2,FILE_WIN_COLS+3)) == NULL)
     {
     	 delwin(file_win_border);
     	 return(FALSE);
     }
   xpaint(file_win_border,(colors[POP_TEXT_BGD]|colors[POP_TEXT_INPUT]));
   xbox(file_win_border,DB_V_LIN,DB_H_LIN,colors[POP_TEXT_BORDER]|colors[POP_TEXT_BGD]);
   wattrset(file_win_border,colors[POP_TEXT_PROMPT]|colors[POP_TEXT_BGD]);
   mvwaddstr(file_win_border,1,1,"            List of Files");
   xpaint(file_win_text,A_FBLUE|A_BWHITE);
   xbox(file_win_text,SG_V_LIN,SG_H_LIN,colors[POP_TEXT_BORDER]|colors[POP_TEXT_BGD]);
   wattrset(file_win_text,A_FBLUE|A_BWHITE);
   nodelay(file_win_text,TRUE);
   keypad(file_win_text,TRUE);
   keypad(file_win_border,TRUE);
   insert_pop_up(file_win_border);
   insert_pop_up(file_win_text);
   if ((about_crsv_win_border = newwin(ABOUT_CRSV_LNS,ABOUT_CRSV_COLS,
				ABOUT_CRSV_Y,ABOUT_CRSV_X)) == NULL)
     return(FALSE);
   if ((about_crsv_win_text = subwin(about_crsv_win_border,ABOUT_CRSV_LNS - 2,
		ABOUT_CRSV_COLS -2,ABOUT_CRSV_Y+1,ABOUT_CRSV_X+1))==NULL)
     {
     	 delwin(about_crsv_win_border);
     	 return(FALSE);
     }
   if((debug_on_win = newwin(3,10,ABOUT_CRSV_Y+9,ABOUT_CRSV_X +3)) == NULL)
        return(FALSE);
   if((mem_used_win = subwin(about_crsv_win_text,3,21,ABOUT_CRSV_Y+9,
					ABOUT_CRSV_X + 15)) == NULL)
     {
        delwin(about_crsv_win_text);
        delwin(about_crsv_win_border);
        return(FALSE);
     }
   xpaint(about_crsv_win_border,(A_BWHITE|A_FLT_RED));
   xbox(about_crsv_win_border,DB_V_LIN,DB_H_LIN,
		A_FMAGENTA|A_BWHITE);
   xpaint(about_crsv_win_text,A_FLT_MAGENTA|A_BWHITE);
   wattrset(about_crsv_win_text,A_BWHITE|A_FRED);
   xpaint(debug_on_win,(A_BWHITE|A_FLT_RED));
   xbox(debug_on_win,SG_V_LIN,SG_H_LIN,(A_FBLUE|A_BWHITE));
   wattrset(debug_on_win,(A_BBLACK|A_FWHITE));
   xpaint(mem_used_win,(A_BWHITE|A_FLT_RED));
   xbox(mem_used_win,SG_V_LIN,SG_H_LIN,(A_FBLUE|A_BWHITE));
   wattrset(mem_used_win,(A_BBLACK|A_FWHITE));
   keypad(about_crsv_win_border,TRUE);
   nodelay(about_crsv_win_text,TRUE);
   keypad(about_crsv_win_text,TRUE);
   draw_about_crsv_win();
   return(TRUE);
  }

/******************************************************************************
 NAME        : draw_about_crsv_win()
 PURPOSE     : draw the window containing information about the developers
 DESCRIPTION : 
 INPUTS      : nothing
 RETURNS     : nothing
 NOTES       : 
 ******************************************************************************/
static void draw_about_crsv_win()
{
    wattrset(about_crsv_win_text,(A_FBLUE|A_BWHITE));  
    mvwaddstr(about_crsv_win_text,2,2,"CRSV   Version 1.1   10/28/88");
    wattrset(about_crsv_win_text,A_FRED|A_BWHITE);
    mvwaddstr(about_crsv_win_text,4,2,"A Product of the");
    mvwaddstr(about_crsv_win_text,5,2,"Artificial Intelligence Section");
    mvwaddstr(about_crsv_win_text,6,2,"NASA/Johnson Space Center");
    mvwaddstr(debug_on_win,1,1,"Debug On");
    mvwaddstr(mem_used_win,1,1,"Mem Used:");
}
/******************************************************************************
 NAME        : allocate_menus
 PURPOSE     : Initializes main menu structures and routine handlers
 DESCRIPTION : Refer to the MENU data type in the file MENU.H
               and the calling format description for new_menu() in the
               file MENU.C
 INPUTS      : None
 RETURNS     : TRUE (1) if all went well, FALSE (0) otherwise
 NOTES       : The handler routine for each menu item must take one intger
                 command line argument which indicates the index of the menu
                 item for which the handler routine corresponds.  Also, it must
                 return an integer status code indicating whether the routine
                 was completed normally or not :
                    MENU_ITEM_COMPLETION (1) or MENU_ITEM_ABORT (0)
               Assumes internal menus are not already allocated.
 ******************************************************************************/
static int allocate_menus()
  {
   /* =================================================================
      The main menu physical representation is always on the screen and
      hanlde by special routines in this file - only the internals need
      to be initialized.
      ================================================================= */
   main_menu = new_menu(VISUAL_POP_UP_MENU_NO,4,
                        "File",do_main_menu,
                        "Options",do_main_menu,
                        "Run",do_run,
                        "Special",do_main_menu
                        );
   if (main_menu == NULL)
     return(FALSE);
   file_menu = new_menu(VISUAL_POP_UP_MENU_YES,6,
			"Add File", do_add,
			"Remove File",do_remove,
                        "Move To The Top",do_move_up,
                        "Move To The Bottom",do_move_down,
			"Dribble Off",do_dribble,
                        "Exit Crsv",do_exit
                        );
   if (file_menu == NULL)
     return(FALSE);

   options_menu = new_menu(VISUAL_POP_UP_MENU_YES,8,
   			   " Be Verbose",do_be_verbose,
   			   " Cross Reference Relations",do_crss_ref_rel,
   			   " Style Warning and Analysis",do_style_wrng_anal,
   			   " Rule/Deffact Summaries",do_rule_deff_sum,
   			   " Verify Rules with Defrelations",do_ver_rule_w_defrel,
   			   " Cross Reference User Function",do_crss_ref_usr_func,
   			   " Create Defrelation File",do_create_defrel_file,
   			   " Check Clips Trace File",do_drib_sum
                          );
   if (options_menu == NULL)
     return(FALSE);
   special_menu = new_menu(VISUAL_POP_UP_MENU_YES,2,
			   "--More-- Option Off",do_pause,
			   "Clear CRSV Win",do_clear);
   if(special_menu == NULL)
     return(FALSE);
   return(TRUE);
  }

/******************************************************************************
 NAME        : draw_border
 PURPOSE     : Draws the main border on the screen
 DESCRIPTION : The border is stored in 4 different windows :
                 header, left_border, right_border, and footer
               (note : the advertisement and main_menu windows are sub-windows
                       of the header window)
               This was done in this rather roundabout away to save memory.
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : The border is drawn character by character using the extended
               ascii set. Curses xbox routine is not used since it does not
               provide all the stuff needed to be drawn.
                 This routine is non-portable.
               Uses the colors defined in the colors array in COLOR.C
               Assumes header-window, border-windows, and footer-window
                 have already been allocated.
 ******************************************************************************/
static void draw_border()
  {
   register int i;

   wattrset(header_window,colors[MAIN_BORDER]|colors[MAIN_BACKGD]);
   mvwaddch(header_window,0,0,DB_UL);
   for (i = 1 ; i < AD_COLS+AD_X ; i++)
     mvwaddch(header_window,0,i,DB_H_LIN);
   mvwaddch(header_window,0,i++,DBSG_UP_T);
   for ( ; i < HDR_COLS-1 ; i++)
     mvwaddch(header_window,0,i,DB_H_LIN);
   mvwaddch(header_window,0,i,DB_UR);
   mvwaddch(header_window,1,0,DB_V_LIN);
   mvwaddch(header_window,1,AD_COLS+AD_X,SG_V_LIN);
   mvwaddch(header_window,1,HDR_COLS-1,DB_V_LIN);
   mvwaddch(header_window,2,0,DBSG_LT_T);
   for (i = 1 ; i < AD_COLS+AD_X ; i++)
     mvwaddch(header_window,2,i,SG_H_LIN);
   mvwaddch(header_window,2,i++,SGSG_LO_T);
   for ( ; i < HDR_COLS-1 ; i++)
     mvwaddch(header_window,2,i,SG_H_LIN);
   mvwaddch(header_window,2,i,DBSG_RT_T);

   xpaint(left_border_window,colors[MAIN_BORDER]|colors[MAIN_BACKGD]);
   wattrset(left_border_window,colors[MAIN_BORDER]|colors[MAIN_BACKGD]);
   for (i = 0 ; i < LINES-HDR_LNS-FTR_LNS ; i++)
     mvwaddch(left_border_window,i,0,DB_V_LIN);

   xpaint(right_border_window,colors[MAIN_BORDER]|colors[MAIN_BACKGD]);
   wattrset(right_border_window,colors[MAIN_BORDER]|colors[MAIN_BACKGD]);
   for (i = 0 ; i < LINES-HDR_LNS-FTR_LNS ; i++)
     mvwaddch(right_border_window,i,0,DB_V_LIN);

   xpaint(footer_window,colors[MAIN_BORDER]|colors[MAIN_BACKGD]);
   wattrset(footer_window,colors[MAIN_BORDER]|colors[MAIN_BACKGD]);
   mvwaddch(footer_window,0,0,DB_LL);
   for (i = 1 ; i < FTR_COLS-1 ; i++)
     waddch(footer_window,DB_H_LIN);
   waddch(footer_window,DB_LR);
  }

/******************************************************************************
 NAME        : draw_advertisement
 PURPOSE     : Draws the "CRSV" logo on the screen
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Assumes advertisement-window has been allocated
 ******************************************************************************/
static void draw_advertisement()
  {
   xpaint(advertisement_window,colors[ADVERTISE]|colors[MAIN_BACKGD]);
   wattrset(advertisement_window,colors[ADVERTISE]|colors[MAIN_BACKGD]);
   mvwaddstr(advertisement_window,0,1,CRSV_LOGO);
  }

/******************************************************************************
 NAME        : draw_main_menu
 PURPOSE     : To display the main menu and allocate underlying windows
 DESCRIPTION : Draws the following selections in the main menu window :
                "Run","rEset","Step","File",
                "Debug","Action","eXamine", and "Help"
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Uses the colors defined in the colors array in COLOR.C
               This routine assumes the internal main-menu has already been
                 correctly allocated.  Unpredictable results will occur if
                 it has not.
 ******************************************************************************/
static void draw_main_menu()
  {
   register int i,j,k;           /* Item loop counters                  */
   int item_wid;                 /* String length of item to be printed */
   char *item;                   /* Item to be printed                  */

   main_menu_cell_width = (MN_MENU_COLS) / main_menu->item_cnt;
   main_menu_cell_offset = ((MN_MENU_COLS) % main_menu->item_cnt) / 2;

   /* =======================
      Draw all the selections 
      ======================= */
   xpaint(main_menu_window,colors[MAIN_MENU_TEXT]|colors[MAIN_MENU_BGD]);
   wattrset(main_menu_window,colors[MAIN_MENU_TEXT]|colors[MAIN_MENU_BGD]);
   for (i = 0 ; i < main_menu->item_cnt ; i++)
     {
      /* ===============================================================
         Move to posn necessary to center string in cell and don't write
         any more characters than the cell width 
         =============================================================== */
      item = (main_menu->items)[i]->name;
      item_wid = strlen(item);
      if (item_wid >= main_menu_cell_width)
        j = 0;
      else
        j = main_menu_cell_width/2 - item_wid/2;
      wmove(main_menu_window,0,
            main_menu_cell_offset+main_menu_cell_width*i+j);
      for (k = 0 ; (j < main_menu_cell_width) && (item[k] != EOS) ; j++ , k++)
        waddch(main_menu_window,item[k]);
     }

   /* =====================================
      highlight initial main menu selection
      ===================================== */
   highlight_main_menu(main_menu_cell - 1);

  }

/******************************************************************************
 NAME        : draw_main_window
 PURPOSE     : Used to draw/redraw the main window with a new background color
 DESCRIPTION : The foreground colors of individual characters are left intact
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Uses the colors defined in the colors array in COLOR.C
               Assumes CRSV I/O pad is currently allocated.
 ******************************************************************************/
static void draw_main_window()
  {
   register int i,j;
   int ch,rows,cols;

   getmaxrc(crsv_window,rows,cols);
   wattrset(crsv_window,0x0000);
   for (i = 0 ; i < rows ; i++)
     {
      for (j = 0 ; j < cols ; j++)
        { 
         wmove(crsv_window,i,j); 
         ch = winch(crsv_window);
         ch |= (ch & A_ATTRIBUTES & BGD_MSK) | colors[MAIN_BACKGD];
         waddch(crsv_window,ch);
        }
     }
   wattrset(crsv_window,colors[STDIN]|colors[MAIN_BACKGD]);
/*   wmove(crsv_window,crsv_row,crsv_col);*/
  }

/******************************************************************************
 NAME        : reset_colors
 PURPOSE     : Redraws main-screen colors
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void reset_colors()
  {
   draw_border();
   draw_advertisement();
   draw_main_menu();
   highlight_main_menu(main_menu_cell - 1);
   draw_main_window();
   draw_pop_up_menu(file_menu);
   draw_pop_up_menu(options_menu);
   redisplay_wins();
  }

/******************************************************************************
 NAME        : update_screen
 PURPOSE     : Flushes most current window outputs to the physical screen
 DESCRIPTION : Has the same functionality as redisplay_wins() except does
               not un-optimize windows and thus force complete redraws of
               each.
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Assumes all windows are currently allocated
 ******************************************************************************/
static void update_screen()
  {
   WINBLK *qptr;

   wnoutrefresh(header_window);
   wnoutrefresh(advertisement_window);
   wnoutrefresh(main_menu_window);
   wnoutrefresh(left_border_window);
   wnoutrefresh(right_border_window);
   wattrset(footer_window,colors[WFOOTER]|colors[MAIN_BACKGD]);
   wnoutrefresh(footer_window);
   pnoutrefresh(crsv_window,crsv_offset,0,CRSV_Y,CRSV_X,
                CRSV_Y+LINES-HDR_LNS-FTR_LNS-1,CRSV_X+CRSV_COLS-1);
   if ((qptr = pop_up_queue_top) != NULL)
     {
      while (qptr != NULL)
        {
         wnoutrefresh(qptr->win);
         qptr = qptr->nxt;
        }

      doupdate();
     }
   else
     {
      doupdate();

      /* ===================================================
         Hack to overcome pnoutrefresh() placement of cursor
         =================================================== */
      if (((crsv_row-crsv_offset) < (LINES-HDR_LNS-FTR_LNS)) &&
           (crsv_row >= crsv_offset))
        mvcur(0,0,CRSV_Y+crsv_row-crsv_offset,CRSV_X+crsv_col);
      else
        hide_cursor();

     }
  }

/******************************************************************************
 NAME        : highlight_logo
 PURPOSE     : Outlines current main menu selection
 DESCRIPTION : The routine uses the current background and foreground
               colors to unhighlight the old selection and the flip of
               these to highlight the new selection.
                 This routine sets the main-menu index main_menu_cell to
               the value of 0
 INPUTS      : 
 RETURNS     : Nothing useful
 NOTES       : If the foreground color has a high-bit (i.e FYELLOW, FLT_RED,
               etc.) it will be lost when that color is flipped to a 
               background color since background colors can only have 3 bits
               as opposed to foreground colors which can have 4.
                 Example : yellow on black would be flipped to black on brown
 ******************************************************************************/
static void highlight_logo()
  {
   ch_attr unhgh_attr,hgh_attr;
   register int i;

   unhgh_attr = colors[MAIN_MENU_TEXT] | colors[MAIN_MENU_BGD];

   /* =========================
      Unhighlight old selection
      ========================= */
   wattrset(main_menu_window,unhgh_attr);
   wmove(main_menu_window,0,main_menu_cell_offset+
                         (main_menu_cell - 1)*main_menu_cell_width);
   for (i = 0 ; i < main_menu_cell_width ; i++)
      waddch(main_menu_window,winch(main_menu_window) & A_CHARTEXT);


   /* =========================
      Highlight new selection
      ========================= */
   unhgh_attr = colors[ADVERTISE] | colors[MAIN_MENU_BGD];
   hgh_attr = flip_fgd_bgd(unhgh_attr);
   wattrset(advertisement_window,hgh_attr);
   main_menu_cell = 0;
   wmove(advertisement_window,0,0);
   for (i = 0 ; i < AD_COLS ; i++)
     waddch(advertisement_window,winch(advertisement_window) & A_CHARTEXT);
}

/******************************************************************************
 NAME        : highlight_main_menu
 PURPOSE     : Outlines current main menu selection
 DESCRIPTION : The routine uses the current background and foreground
               colors to unhighlight the old selection and the flip of
               these to highlight the new selection.
                 This routine sets the main-menu index main_menu_cell to
               the value of the new selection index.
 INPUTS      : 
 RETURNS     : Nothing useful
 NOTES       : If the foreground color has a high-bit (i.e FYELLOW, FLT_RED,
               etc.) it will be lost when that color is flipped to a 
               background color since background colors can only have 3 bits
               as opposed to foreground colors which can have 4.
                 Example : yellow on black would be flipped to black on brown
 ******************************************************************************/
static void highlight_main_menu(select)
  int select;
  {
   ch_attr unhgh_attr,hgh_attr;
   register int i;

   unhgh_attr = colors[MAIN_MENU_TEXT] | colors[MAIN_MENU_BGD];
   hgh_attr = flip_fgd_bgd(unhgh_attr);

   /* =========================
      Unhighlight old selection
      ========================= */
   if(main_menu_cell == 0)
    {
       wattrset(advertisement_window,(colors[ADVERTISE]|colors[MAIN_BACKGD]));
       wmove(advertisement_window,0,0);
       for(i = 0;i < AD_COLS;i++)
         waddch(advertisement_window,winch(advertisement_window)& A_CHARTEXT);
    }
   else
    { 
      wattrset(main_menu_window,unhgh_attr);
      wmove(main_menu_window,0,main_menu_cell_offset+
                           ( main_menu_cell - 1)*main_menu_cell_width);
      for (i = 0 ; i < main_menu_cell_width ; i++)
      waddch(main_menu_window,winch(main_menu_window) & A_CHARTEXT);
    }

   /* =========================
      Highlight new selection
      ========================= */
   wattrset(main_menu_window,hgh_attr);
   main_menu_cell = select + 1;
   wmove(main_menu_window,0,main_menu_cell_offset+
                            (main_menu_cell - 1)*main_menu_cell_width);
   for (i = 0 ; i < main_menu_cell_width ; i++)
     waddch(main_menu_window,winch(main_menu_window) & A_CHARTEXT);

  }

/******************************************************************************
 NAME        : main_pop_up_start
 PURPOSE     : Finds screen x-coordinate start of main menu selection box
 DESCRIPTION : Used to start pop-up menus underneath main-menu selections
 INPUTS      : 1) The index (0..to max-selections-1) in the main-menu
 RETURNS     : The screen coordinate of the center of the box
 NOTES       : None
 ******************************************************************************/
int main_pop_up_start(index)
  int index;
  {
   return(AD_COLS+AD_X+1+main_menu_cell_offset+index*main_menu_cell_width);
  }

/******************************************************************************
 NAME        : deallocate_pop_up_queue
 PURPOSE     : Deletes pop_up_queue space
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void deallocate_pop_up_queue()
  {
   WINBLK *qptr;

   while (pop_up_queue_top != NULL)
     {
      qptr = pop_up_queue_top;
      pop_up_queue_top = pop_up_queue_top->nxt;
      release(1,WINBLK,qptr);
     }
   pop_up_queue_bottom= NULL;
  }

/******************************************************************************
 NAME        : deallocate_windows
 PURPOSE     : Frees main screen memory
 DESCRIPTION : Uses Curses routine delwin to free main screen window space
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Should be called after all Curses stuff is finished
 ******************************************************************************/
static void deallocate_windows()
  {
   /* =====================================================================
      Since the advertisement and main_menu windows are sub-windows of the
      header window, they must be deallocated first.
      ===================================================================== */
   if (advertisement_window != NULL)
     {
      delwin(advertisement_window);
      advertisement_window = NULL;
     }
   if (main_menu_window != NULL)
     {
      delwin(main_menu_window);
      main_menu_window = NULL;
     }
   if (header_window != NULL)
     {
      delwin(header_window);
      header_window = NULL;
     }
   if (left_border_window != NULL)
     {
      delwin(left_border_window);
      left_border_window = NULL;
     }
   if (right_border_window != NULL)
     {
      delwin(right_border_window);
      right_border_window = NULL;
     }
   if (footer_window != NULL)
     {
      delwin(footer_window);
      footer_window = NULL;
     }
   if(file_win_text != NULL)
     {
      delwin(file_win_text);
      file_win_text = NULL;
     }
   if(file_win_border != NULL)
     {
      delwin(file_win_border);
      file_win_border = NULL;
     }
   if (crsv_window != NULL)
     {
      delwin(crsv_window);
      crsv_window = NULL;
     }
   if(about_crsv_win_text != NULL)
     {
      delwin(about_crsv_win_text);
      about_crsv_win_text = NULL;
     }
   if(about_crsv_win_border != NULL)
     {
      delwin(about_crsv_win_border);
      about_crsv_win_border = NULL;
     }
   if (stdscr == NULL)
     stdscr = newwin(LINES,COLS,0,0);  /* Hack to prevent endwin from bombing */
   endwin();
  }

/******************************************************************************
 NAME        : deallocate_menus
 PURPOSE     : Releases menu memory space
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void deallocate_menus()
  {
   if (main_menu != NULL)
     main_menu = kill_menu(main_menu);
   if (file_menu != NULL)
     file_menu = kill_menu(file_menu);
   if (options_menu != NULL)
     options_menu = kill_menu(options_menu);
  }


/******************************************************************************
 NAME        : crsv_addch
 PURPOSE     : Prints characters to the main CRSV window
 DESCRIPTION : This routine directly handles all window output, especially
               wrapping and scrolling, since the normal Curses automatic
               stuff screws up non-black background colors.
 INPUTS      : 1) The character to be printed
 RETURNS     : TRUE (1) if character successfully added, FALSE (0) otherwise
 NOTES       : None
 ******************************************************************************/
int crsv_addch(ch)
  char ch;
  {
   if (ch == DELETE)
     {
      if (crsv_col != 0)
        {
         mvwaddstr(crsv_window,crsv_row,crsv_col,DEL_STR);
         crsv_col--;
        }
      else if (crsv_row != 0)
        {
         crsv_row--;
         crsv_col = (int) newline_map[(nlmap_ptr + crsv_row) % CRSV_PAD_LNS];
         if (crsv_col == CRSV_COLS)
           {
            mvwaddstr(crsv_window,crsv_row,crsv_col--,DEL_STR);
            wmove(crsv_window,crsv_row,crsv_col);
           }
         else
           wmove(crsv_window,crsv_row,crsv_col);
         if (crsv_row < crsv_offset)
           {
            crsv_offset = crsv_row - PAGE_LEN/2;
            if (crsv_offset < 0)
              crsv_offset = 0;
            }
        }
      else
        return(FALSE);
     }
   else if ((ch != NEWLINE) ? (ch == CRGRTN) : TRUE)
     crsv_advance_row();
   else if (crsv_col == CRSV_COLS-1)
     {
      if ((ch != NEWLINE) ? (ch != CRGRTN) : FALSE)
        mvwaddch(crsv_window,crsv_row,crsv_col++,ch);
      crsv_advance_row();
     }
   else
     mvwaddch(crsv_window,crsv_row,crsv_col++,ch);

   return(TRUE);
  }

/******************************************************************************
 NAME        : crsv_advance_row
 PURPOSE     : Moves cursor to a new line
 DESCRIPTION : Not only takes care of Curses newline glitches with non-black
               backgrounds but also handles scrolling the CRSV window 
               (if necessary) so as to preserve the background color.
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Only this routine is permitted to scroll the CRSV pad
               via the Curses routines deleteln() and xwclrtoeol().
               Thus, other routines (namely crsv_addch()) can place
               characters in the last row/last column position of the window.
 ******************************************************************************/
void crsv_advance_row()
  {
   register ch_attr tmpattr;

   newline_map[(nlmap_ptr + crsv_row) % CRSV_PAD_LNS] = 
                                                   (unsigned char) crsv_col;
   crsv_col = 0;
   if (crsv_row == (CRSV_PAD_LNS-1))
     {
      tmpattr = getattr(crsv_window);
      nlmap_ptr = (nlmap_ptr + 1) % CRSV_PAD_LNS;
      wmove(crsv_window,0,0);
      wdeleteln(crsv_window);
      wmove(crsv_window,crsv_row,0);
      wattrset(crsv_window,colors[STDIN]|colors[MAIN_BACKGD]);
      xwclrtoeol(crsv_window);
      wattrset(crsv_window,tmpattr);
      wmove(crsv_window,crsv_row,crsv_col);
     }
   else
     wmove(crsv_window,++crsv_row,crsv_col);

   if (crsv_row > (crsv_offset+(LINES-HDR_LNS-FTR_LNS)-1))
     crsv_offset = crsv_row-(LINES-HDR_LNS-FTR_LNS)+1;
  }

/******************************************************************************
 NAME        : crsv_addstr
 PURPOSE     : Writes a whole string to the main CRSV window
 DESCRIPTION : Automatically pauses output if the top line of continuous output
               is about to scroll of the screen, e.g the "More" facility.
 INPUTS      : 1) The address of the string to be printed
 RETURNS     : Nothing useful
 NOTES       : Assumes no "Delete" characters are imbedded in the string.
 ******************************************************************************/
void crsv_addstr(str)
  char *str;
  {
   register int i,ch;

   for (i = 0 ; (ch = str[i]) != EOS ; i++)
     {
      if ((ch != NEWLINE) ? (ch == CRGRTN) : TRUE)
        {
         if ((((output_lines+1) % (LINES-HDR_LNS-FTR_LNS)) == 0)
                                  && (PAUSE_OUT_PUT))
           pause_output();
         else
           update_screen();
         crsv_advance_row();
         output_lines++;
        }
      else if (crsv_col == CRSV_COLS-1)
        {
         if ((ch != NEWLINE) ? (ch != CRGRTN) : FALSE)
           mvwaddch(crsv_window,crsv_row,crsv_col++,ch);
         if ((((output_lines+1) % (LINES-HDR_LNS-FTR_LNS)) == 0)
				&& (PAUSE_OUT_PUT))
           pause_output();
         else
           update_screen();
         crsv_advance_row();
         output_lines++;
        }
      else
        mvwaddch(crsv_window,crsv_row,crsv_col++,ch);
     }
  }




/******************************************************************************
 NAME        : process_special
 PURPOSE     : Runs special interface routines based on control-sequences
 DESCRIPTION : 
 INPUTS      : 1) The CRSV action code to be processed
               2) The address of new-main-menu-selection index in
                  the function next_event();
 RETURNS     : TRUE (1) if a command to CLIPS has been issued,
               FALSE (0) otherwise.
 NOTES       : Called by the function next_event().
 ******************************************************************************/
static int process_special(ch,new_cell)
  int ch,*new_cell;
  {
   MENU_ITEM *item;
   int (*handler)(int) = NULL;
   int main_select = main_menu_cell,
       handler_arg = 0,
       handler_rtn;

   switch(ch)
     {
      case LEFT_ARROW  : *new_cell = (main_menu_cell + main_menu->item_cnt)
                                      % (main_menu->item_cnt+1);
                         break;
      case RIGHT_ARROW : *new_cell = (main_menu_cell + 1) % (main_menu->item_cnt +1);
                         break;
      case UP_ARROW    : crsv_scroll(UP,1);
                         break;
      case DOWN_ARROW  : crsv_scroll(DOWN,1);
                         break;
      case PAGE_UP     : crsv_scroll(UP,PAGE_LEN);
                         break;
      case PAGE_DOWN   : crsv_scroll(DOWN,PAGE_LEN);
                         break;
      case BUFFER_TOP  : crsv_scroll(TOP,0);
                         break;
      case BUFFER_END  : crsv_scroll(BOTTOM,0);
                         break;
      case NEWLINE     : if(main_menu_cell > 0)
                           {
			     handler = (main_menu->items)[main_menu_cell-1]->handler;
                             handler_arg = main_menu_cell - 1;
                             main_select--;
                           }
                         else
			   pop_up_about_crsv();
                         break;
      case REDRAW_CMD  : redisplay_wins();
                         break;
      case FILE_MENU   : handler = do_main_menu;
                         main_select = FILE_INDEX;
                         handler_arg = FILE_INDEX;
                         break;
      case EXIT_TO_OS  : handler = do_exit;
                         main_select = FILE_INDEX;
                         break;
      case RUN_CMD     : handler = do_run ;
                         main_select = EXEC_INDEX;
                         break;
      case SPEC_MENU   : handler = do_main_menu;
			 main_select = SPE_INDEX;
			 handler_arg = SPE_INDEX;
			 break;

/*      case HELP_CMD    : handler = do_help;
                         main_select = HELP_INDEX;
                         break;*/
                         
      case OPTION_MENU : handler = do_main_menu;
                         main_select = OPTS_INDEX;
                         handler_arg = OPTS_INDEX;
                         break;
      
/*      case ESC         : handler = do_exit;
			 main_select = FILE_INDEX;
			 break;
*/
      case DO_NOTHING  : break;
      default          : beep();

     }
   if (handler != NULL)
     {
      *new_cell = main_select + 1;
      highlight_main_menu(main_select);
      update_screen();
      handler_rtn = handler(handler_arg);
      redisplay_wins();
      if (handler_rtn == MENU_ITEM_COMPLETION)
        return(TRUE);
     }
   return(FALSE);
  }

/***********************************************************************
 NAME        : 
 PURPOSE     : 
 DESCRIPTION : 
 INPUTS      : 
 RETURNS     : 
 NOTES       : 
 ***********************************************************************/
static int pop_up_about_crsv()
{
  int ch;

  insert_pop_up(about_crsv_win_border);
  insert_pop_up(about_crsv_win_text);
  insert_pop_up(mem_used_win);

  if(CHECK_DEBUG)
      insert_pop_up(debug_on_win);

  mvwprintw(mem_used_win,1,1,"Mem Used: %8ld", mem_used());
  redisplay_wins();
  wmove(about_crsv_win_text,1,1);
  while(1)
   {
     if((ch = wgetch(about_crsv_win_text)) != -1)
        break;
   }

  if(key_map(ch) == CTRL_D)
    {
       if(!CHECK_DEBUG)
         {
               CHECK_DEBUG = ON;
	       insert_pop_up(debug_on_win);
         }
       else
         {
              delete_pop_up(debug_on_win);
              CHECK_DEBUG = OFF;
         }
   }

   if(CHECK_DEBUG)
      delete_pop_up(debug_on_win);
  
   delete_pop_up(mem_used_win);
   delete_pop_up(about_crsv_win_text);
   delete_pop_up(about_crsv_win_border);
   redisplay_wins();
   return(TRUE);
}
/******************************************************************************
 NAME        : crsv_scroll
 PURPOSE     : Moves the physical display of the CRSV main window
 DESCRIPTION : Redefines the starting row for the refresh of the CRSV pad
 INPUTS      : 1) The direction code : UP (0), DOWN (1), TOP (2), or BOTTOM (3)
               2) The number of lines to scroll (in the cases of TOP/BOTTOM
                  this argument is ignored)
 RETURNS     : Nothing useful
 NOTES       : Automatically adjusts scroll-offset if it would take the
               region beyond the pad's boundaries.
 ******************************************************************************/
static void crsv_scroll(drx,lns)
  int drx,lns;
  {
   register int page_size;
   int scrolled = FALSE;

   page_size = LINES-HDR_LNS-FTR_LNS;
   if (drx == UP)
     {
      if (crsv_offset != 0)
        {
         crsv_offset -= lns;
         if (crsv_offset < 0)
           crsv_offset = 0;
         scrolled = TRUE;
        }
     }
   else if (drx == DOWN)
     {
      if (crsv_row > (crsv_offset+page_size-1))
        {
         crsv_offset += lns;
         if (crsv_offset > (crsv_row-page_size+1))
           crsv_offset = crsv_row-page_size+1;
         scrolled = TRUE;
        }
     }
   else if (drx == TOP)
     {
      if (crsv_offset != 0)
        {
         crsv_offset = 0;
         scrolled = TRUE;
        }
     }
   else if (drx == BOTTOM)
     {
      if (crsv_row > (crsv_offset+page_size-1))
        {
         crsv_offset = crsv_row-page_size+1;
         scrolled = TRUE;
        }
     }
   if (scrolled)
     update_screen();
   else
     beep();
  }

/******************************************************************************
 NAME        : scroll_to_end
 PURPOSE     : Forces main CRSV window display to end of the buffer
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void scroll_to_end()
  {
   if (crsv_row > (crsv_offset+(LINES-HDR_LNS-FTR_LNS)-1))
     crsv_offset = crsv_row-(LINES-HDR_LNS-FTR_LNS)+1;
  }

/******************************************************************************
 NAME        : crsv_clear_win()
 PURPOSE     : reinitialize crsv window
 DESCRIPTION : 
 INPUTS      : 
 RETURNS     : 
 NOTES       : 
 ******************************************************************************/
void crsv_clear_win()
  {
        xpaint(crsv_window,colors[STDIN]|colors[MAIN_BACKGD]);
        crsv_row = 0;
        crsv_col = 0;
        crsv_offset = 0;
  }
/******************************************************************************
 NAME        : pause_output
 PURPOSE     : Holds output and waits for a user to press a key indicating
               to continue
 DESCRIPTION : Used when a stream of continuous output is about to have its
               top line scroll off the screen.
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void pause_output()
  {
   int quit = FALSE,
       ch,y,x;

   wattrset(footer_window,colors[MORE_MSG]|colors[MAIN_BACKGD]);
   mvwaddstr(footer_window,0,FTR_COLS/2-6," -- More -- ");
   update_screen();
   while (! quit)
     {
      if (wgetch(crsv_window) != -1)
        quit = TRUE;

     }
   wattrset(footer_window,colors[MAIN_BORDER]|colors[MAIN_BACKGD]);
   for (x = FTR_COLS/2-6 ; x < FTR_COLS/2+6 ; x++)
     mvwaddch(footer_window,0,x,DB_H_LIN);
   update_screen();
  }

/******************************************************************************
 NAME        : crsv_error_message()
 PURPOSE     : print out the ERROR or WARNING message to CRSV window
 DESCRIPTION : 
 INPUTS      : 
 RETURNS     : 
 NOTES       : 
 ******************************************************************************/
void crsv_error_message(error_str,message_str)
char *error_str,*message_str;

{
   register int i;

   if(error_str[1] == 'E')
     {
        wattrset(crsv_window,A_FLT_RED|colors[MAIN_BACKGD]);
        crsv_addstr(error_str);
     }
   else if(error_str[1] == 'W')
     {
        wattrset(crsv_window,A_FYELLOW|colors[MAIN_BACKGD]);
        crsv_addstr(error_str);
     }
   wattrset(crsv_window,colors[STDIN]|colors[MAIN_BACKGD]);
   crsv_addstr(message_str);
   if(drib_fp != NULL)
     {
       fprintf(drib_fp,"%s%s",error_str,message_str);
     }
}

/******************************************************************************
 NAME        : crsv_message()
 PURPOSE     : send the general message to CRSV window
 DESCRIPTION : 
 INPUTS      : 
 RETURNS     : 
 NOTES       : 
 ******************************************************************************/
void crsv_message(message_str)
char *message_str;

{

   crsv_addstr(message_str);
   if(drib_fp != NULL)
       fprintf(drib_fp,"%s",message_str);
}


/******************************************************************************
 NAME        : 
 PURPOSE     : 
 DESCRIPTION : 
 INPUTS      : 
 RETURNS     : 
 NOTES       : 
 ******************************************************************************/

