/* ************************************************************************
   FILE : MOUSE.C
      This file contains the support routines which handle the mouse.
   *********************************************************************** */

/*  CLIPS Version 4.20 4/29/88 */

#include "common.h"
#define MOUSE_SOURCE
#include "mouse.h"

typedef struct button_node
  {
   int button,                /* Button that was pressed          */
       y,x;                   /* Where it was pressed - in pixels */
   struct button_node *nxt;   /* Pointer to next click            */
  } BUTTONNODE;

int curr_min_y = 0,curr_max_y = 24,
    curr_min_x = 0,curr_max_x = 79;

typedef struct region_node
  {
   int min_y,min_x,
       max_y,max_x,
       y,x;
   struct region_node *nxt;
  } MOUSENODE;

static MOUSENODE *mouse_save_stack = NULL;

int MOUSE_LOADED = FALSE;

#include <dos.h>

void MOUSECML(int *,int *,int *,int *);

static void MOUSECML(M1,M2,M3,M4)
  int *M1,*M2,*M3,*M4;
  {
   union REGS ioregs;

   ioregs.x.ax = *M1;
   ioregs.x.bx = *M2;
   ioregs.x.cx = *M3;
   ioregs.x.dx = *M4;
   int86(0x33,&ioregs,&ioregs);
   *M1 = ioregs.x.ax;
   *M2 = ioregs.x.bx;
   *M3 = ioregs.x.cx;
   *M4 = ioregs.x.dx;
  }

int  init_mouse(void),
     mouse_moved_posn(int *,int *),
     mouse_clicked(int *,int *,int *);

void show_mouse(void),
     hide_mouse(void),
     mouse_cursor(int,int,int),
     get_mouse(int *,int *),
     clear_mouse(void),
     mouse_region(int,int,int,int),
     set_mouse(int,int),
     save_mouse(void),restore_mouse(void);

static int examine_mouse(int *,int *,int *);

int init_mouse()
{
   int M1, M2, M3, M4;
   unsigned long msvect;

   msvect = (unsigned long) getvect(0x33);
   if ((msvect == 0) || (msvect == 0xCF))
     return(FALSE);

   M1 = 0;  /* reset and init mouse */
   M2 = 0;
   M3 = 0;
   M4 = 0;
   (void) MOUSECML(&M1, &M2, &M3, &M4);
   if (M1 == -1)
     {
      MOUSE_LOADED = TRUE;
      return(TRUE);
     }
   else
     return(FALSE);
}

static int examine_mouse(button,x,y)
  int *button,*x,*y;
  {
   int M1 = 5;

   (void) MOUSECML(&M1,button,x,y);
   if (*button != 0)
     return(TRUE);
   return(FALSE);
  }

void show_mouse()
{
   int M1, M2, M3, M4;

   if (! MOUSE_LOADED)
     return;
   M1 = 1;  /* show cursor */
   M2 = 0;
   M3 = 0;
   M4 = 0;
   (void) MOUSECML(&M1, &M2, &M3, &M4);
}

void mouse_cursor(type,scr_mask,curs_mask)
  int type,scr_mask,curs_mask;
  {
   int M1,M2,M3,M4;

   if (! MOUSE_LOADED)
     return;
   M1 = 10;
   M2 = type;
   M3 = scr_mask;
   M4 = curs_mask;
   (void) MOUSECML(&M1,&M2,&M3,&M4);
  }

void hide_mouse()
{
   int M1, M2, M3, M4;

   if (! MOUSE_LOADED)
     return;
   M1 = 2;  /* hide cursor */
   M2 = 0;
   M3 = 0;
   M4 = 0;
   (void) MOUSECML(&M1, &M2, &M3, &M4);
}

void get_mouse(horiz, vert)
int *horiz, *vert;
{
   int M1, M2;

   if (! MOUSE_LOADED)
     return;
   M1 = 3;
   M2 = 0;
   (void) MOUSECML(&M1, &M2, horiz, vert);
}

void clear_mouse()
{
   int button;
   int x, y;

   if (! MOUSE_LOADED)
     return;
   button = LEFT_BUTTON;
   (void) examine_mouse(&button, &x, &y);

   button = MIDDLE_BUTTON;
   (void) examine_mouse(&button, &x, &y);

   button = RIGHT_BUTTON;
   (void) examine_mouse(&button, &x, &y);
}

/* =========================================================================
                HIGHER TEXT LEVEL MOUSE INTERFACE FUNCTIONS
   ========================================================================= */

void mouse_region(min_y,min_x,max_y,max_x)
  int min_y,min_x,max_y,max_x;
  {
   int M1 = 7, M2 = 0;
   int range_min,range_max;

   if (! MOUSE_LOADED)
     return;
   curr_min_y = min_y; curr_max_y = max_y;
   curr_min_x = min_x; curr_max_x = max_x;
   range_min = min_x * PXLS_PER_BLOCK;
   range_max = (max_x+1) * PXLS_PER_BLOCK - 1;
   (void) MOUSECML(&M1,&M2,&range_min,&range_max);

   M1 = 8;

   range_min = min_y * PXLS_PER_BLOCK;
   range_max = (max_y+1) * PXLS_PER_BLOCK - 1;
   (void) MOUSECML(&M1,&M2,&range_min,&range_max);
  }



void set_mouse(vert,horiz)
int vert,horiz;
{
   int M1, M2;
   int ny,nx;

   if (! MOUSE_LOADED)
     return;
   ny = vert*PXLS_PER_BLOCK;
   nx = horiz*PXLS_PER_BLOCK;
   M1 = 4;
   M2 = 0;
   (void) MOUSECML(&M1, &M2, &nx, &ny);
}

int mouse_moved_posn(new_y,new_x)
  int *new_y, *new_x;
  {
   int M1 = 11,M2;

   if (! MOUSE_LOADED)
     return(FALSE);
   (void) MOUSECML(&M1,&M2,new_x,new_y);
   if ((*new_x != 0) || (*new_y != 0))
     {
      get_mouse(new_x,new_y);
      *new_y /= PXLS_PER_BLOCK;
      *new_x /= PXLS_PER_BLOCK;
      return(TRUE);
     }
   else
     return(FALSE);
  }

void save_mouse()
  {
   MOUSENODE *mptr;

   if (! MOUSE_LOADED)
     return;
   if ((mptr = balloc(1,MOUSENODE)) == NULL)
     return;
   get_mouse(&(mptr->x),&(mptr->y));
   mptr->y /= PXLS_PER_BLOCK;
   mptr->x /= PXLS_PER_BLOCK;
   mptr->min_y = curr_min_y;
   mptr->min_x = curr_min_x;
   mptr->max_y = curr_max_y;
   mptr->max_x = curr_max_x;   
   mptr->nxt = mouse_save_stack;
   mouse_save_stack = mptr;
   clear_mouse();
  }

void restore_mouse()
  {
   MOUSENODE *mptr;

   if (! MOUSE_LOADED)
     return;
   if (mouse_save_stack != NULL)
     {
      mptr = mouse_save_stack;
      mouse_save_stack = mouse_save_stack->nxt;
      mouse_region(mptr->min_y,mptr->min_x,mptr->max_y,mptr->max_x);
      set_mouse(mptr->y,mptr->x);
      release(1,MOUSENODE,mptr);
     }
   clear_mouse();
  }

/******************************************************************************/
/* mouse_clicked : If the mouse is clicked, this function stores which button */
/* was pressed and where it was pressed in the caller's buffers.              */
/* Returns : TRUE if clicked, FALSE otherwise                                 */
/******************************************************************************/
int mouse_clicked(button,y,x)
  int *button,*y,*x;
  {
   static BUTTONNODE *bqueue = NULL, *bend = NULL;
   BUTTONNODE *bptr;
   int i,checkb,ny,nx,rtn = FALSE;

   if (! MOUSE_LOADED)
     return(FALSE);

   /* ==================================================================
      Check the queue first for old-clicks that have not been processed 
      ================================================================== */
   if (bqueue != NULL)
     {
      *button = bqueue->button;
      *y = bqueue->y;
      *x = bqueue->x;
      bptr = bqueue;
      bqueue = bqueue->nxt;
      release(1,BUTTONNODE,bptr);
      rtn = TRUE;
     }

   /* =============================================================
      Check for new mouse-clicks and put them in the queue --
      If the queue is empty, don't put the first one in the queue,
      just save it for a later return.
      ============================================================= */
   for (i = LEFT_BUTTON ; i <= MIDDLE_BUTTON ; i++)
     {
      checkb = i;
      if (examine_mouse(&checkb,&nx,&ny))
        {
         if (rtn == FALSE)
           {
            *button = (i != MIDDLE_BUTTON) ? i : LEFT_BUTTON;
            *y = ny / PXLS_PER_BLOCK;
            *x = nx / PXLS_PER_BLOCK;
            rtn = TRUE;
           }
         else
           {
            bptr = salloc(button_node);
            bptr->button = (i != MIDDLE_BUTTON) ? i : LEFT_BUTTON;
            bptr->y = ny / PXLS_PER_BLOCK;
            bptr->x = nx / PXLS_PER_BLOCK;
            bptr->nxt = NULL;
            if (bqueue == NULL)
	      {
               bqueue = bptr;
	       bend = bptr;
	      }
            else
              {
               bend->nxt = bptr;
               bend = bend->nxt;
              }
           }
        }
     }
   return(rtn);
  }
