/* ************************************************************************
   FILE : KEY.C
      This file contains the support routines that handle raw keystroke
   mapping to CLIPS action codes as well as defining new key-bindings.
   *********************************************************************** */

/******************************************************************************
 ==============================================================================
                                 HEADER FILES
 ==============================================================================
 ******************************************************************************/
#include "common.h"     /* Common constants and memory macros               */
#define _NO_STDIO
#include "curses.h"     /* Curses constants, macros and function prototypes */
#include "box.h"
#include "curproto.h"
#define KEY_SOURCE
#include "key.h"        /* Key-binding constants                            */
#include "mouse.h"      /* Mouse-Systems PC-Mouse Stuff                     */
#include "env.h"        /* Key-binding binary data file load/save stuff     */

#if ENVIRONMENT_UTILITY
#include "clpsutil.h"   /* Utility interface routines                       */
#endif

/******************************************************************************
 ==============================================================================
                                   CONSTANTS
 ==============================================================================
 ******************************************************************************/

#define MAX_ACTIONS     28                  /* The number of things that can
                                               be bound to keys               */
#define MAX_ACTION_NAME 18
#define MAX_KEY_NAME    10

#if ENVIRONMENT_UTILITY

#define PKEY_START      MAX_ACTION_NAME+19  /* Where the Key-Bind windows #1  */
#define SKEY_START      PKEY_START+20       /* and #2 start (columns)         */

#define KEYBIND1        0                   /* Indices for action key-binds   */
#define KEYBIND2        1

#define names_attr      A_FGREEN|A_BBLACK   /* Actions window color           */
#define keys_attr       A_FCYAN|A_BBLACK    /* Keys windows color             */
#define hi_keys_attr    A_FHI_WHITE|A_BRED  /* Highlighted selected key color */

#define SELECT          0
#define BIND            1

#endif

/******************************************************************************
 ==============================================================================
                                    MACROS
 ==============================================================================
 ******************************************************************************/
#define hide_cursor() mvcur(0,0,LINES,0)

/* ---------------------------------------------------------------
   Returns a string representing the control or extended character
   --------------------------------------------------------------- */
#define keyname(ch) ((ch<LOW_PRN_ASCII) ? keycnames[ch]:keyexnames[ch-KEY_F(1)])

/******************************************************************************
 ==============================================================================
                     EXTERNALLY VISIBLE FUNCTION PROTOTYPES
 ==============================================================================
 ******************************************************************************/

int key_map(int);          /* Maps raw keyboard strokes into CLIPS actions */
char *key_action(int);     /* Gets an action-string for a keystroke        */

#if ENVIRONMENT_UTILITY

void bind_keys(void);      /* (Re)binds raw keystrokes to CLIPS actions    */

#endif

/******************************************************************************
 ==============================================================================
                     INTERNALLY VISIBLE FUNCTION PROTOTYPES
 ==============================================================================
 ******************************************************************************/

/* ----------------------------------------------
   These are all support routines for bind_keys()
   ---------------------------------------------- */

#if ENVIRONMENT_UTILITY

static void initialize_bind_screen(void), /* (Re)draws keys on screen         */
            kill_windows(void),           /* Deallocates bind utility windows */
            init_actiontab(void),         /* Initializes all binds to -1      */
            set_actiontab(void),          /* Loads action table with binds    */
            set_action(unsigned char,int),/* Sets an individual action bind   */
            write_names(WINDOW *),        /* Writes action-names to window    */
            write_keys(WINDOW *,WINDOW *),/* Writes key-names to window       */
            highlight_key(int,int),       /* (Un)highlights key selections    */
            change_key(int);              /* Sets action to a new bind        */

static int process(int,int *,int *,int *);/* Processes utility command chars  */

#endif

/******************************************************************************
 ==============================================================================
                      EXTERNALLY VISIBLE GLOBAL VARIABLES

 All the Key-Binding tables together take up ~2K - most of which is the
   space taken by the string-tables for the keys for the run-time binding
   routine.

 Possible Improvements :

 1) More space efficient way of determining the string-representation of a key
    (The problem is not the control-keys - those are easy - the problem is that
   Curses has ordered the extended keystrokes in a funky way.  For instance,
   The alphabetic Alt-Characters don't follow any kind of ordering based
   on the ascii one, but rather follows the order on a keyboard (ALt-Q,Alt-W,
   Alt-E, etc.)).

 2) Make the keystroke-string tables local variables to the bind routine.
    (Yuuuck!) - This would probably force your compiler to give you
    mucho more stack space at link-time.  I do not recommend this solution.

 ==============================================================================
 ******************************************************************************/

/* --------------------------------------------------------------------------
   The following two tables are the heart and core of the key-bindings.
   Each CLIPS action code is mapped to a particular keystroke by placing
   that code in one of the two tables in the position which corresponds
   to the key desired.

     Example : CTRL-C is bound to the CLIPS action EXIT_TO_OS 
                (i.e. abort CLIPS)
               The ascii/numeric code of CTRL-C is 3 which places it within
                 the bounds of the standard ascii control character table :
                 keyctab[].  The action for CTRL-C thus corresponds to
                 keyctab[3].  Looking up key-binds thus becomes a trivial task.
               As a side-effect of this method, each action can be bound to
                 several different keystrokes (The key-binding routine will
                 only let you set two at run-time - which I thought was
                 plenty.)
   -------------------------------------------------------------------------- */

unsigned char keyctab[MAX_CNTRL_KEYS] =          /* Control Characters */
      { 
       BAD_KEY,
       BAD_KEY,
       LEFT_ARROW,
       EXIT_TO_OS,     /* CTRL-C */
       BAD_KEY,
       BAD_KEY,
       RIGHT_ARROW,
       BAD_KEY,
       DELETE,         /* CTRL-H/DELETE */
       BAD_KEY,
       NEWLINE,        /* CTRL-J/NEWLINE */
       DESCRIBE_KEY,
       REDRAW_CMD,
       NEWLINE,       /* CTRL-M/CRGRTN */
       DOWN_ARROW,
       BAD_KEY,
       UP_ARROW,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       CLEAR_WIN,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       ESC,           /* CTRL-[/ESCAPE */
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY
      },

     keyextab[MAX_EXTND_KEYS] =    /* Function Keys and Extended Keypad Keys */
      {
       HELP_CMD,               /* Function Keys */
       MAIN_MENU,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,              /* Shift + Function Keys */
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,                  /* Ctrl + Function Keys */
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,                  /* Alt + Function Keys  */
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,               /* BackTab */
       BAD_KEY,               /* ^Break */
       BAD_KEY,               /* CTRL-LEFT */
       BAD_KEY,               /* CTRL-RIGHT */
       BAD_KEY,               /* CTRL-END */
       BAD_KEY,               /* CTRL-PGDN */
       BAD_KEY,               /* CTRL-HOME */
       BAD_KEY,               /* CTRL-PGUP */
       DOWN_ARROW,            /* DOWN */
       UP_ARROW,              /* UP */
       LEFT_ARROW,            /* LEFT */
       RIGHT_ARROW,           /* RIGHT */
       BUFFER_TOP,            /* HOME */
       BUFFER_END,            /* END */
       PAGE_DOWN,             /* PGDN */
       PAGE_UP,               /* PGUP */
       ESC,                   /* KEYPAD DELETE */
       MAIN_MENU,             /* INSERT */
       BAD_KEY,               /* PRINT SCREEN */
       BAD_KEY,               /* Alt 1 - 0 */
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,
       BAD_KEY,                /* Alt - */
       BAD_KEY,                /* Alt = */
       BAD_KEY,                /* Alt Q */
       DBG_TOGGLE,             /* Alt W */
       EXEC_MENU,              /* Alt E */
       RUN_CMD,                /* Alt R */
       RESET_CMD,              /* Alt T */
       BAD_KEY,                /* Alt Y */
       BAD_KEY,                /* Alt U */
       BAD_KEY,                /* Alt I */
       OPTION_MENU,            /* Alt O */
       SPAWN_CLI,              /* Alt P */
       ACTION_MENU,            /* Alt A */
       STEP_CMD,               /* Alt S */
       DBG_MENU,               /* Alt D */
       FILE_MENU,              /* Alt F */
       DOS_CMD,                /* Alt G */
       HELP_CMD,               /* Alt H */
       BAD_KEY,                /* Alt J */
       BAD_KEY,                /* Alt K */
       LOAD_CMD,               /* Alt L */
       BAD_KEY,                /* Alt Z */
       EXAM_MENU,              /* Alt X */
       BAD_KEY,                /* Alt C */
       BAD_KEY,                /* Alt V */
       BAD_KEY,                /* Alt B */
       BAD_KEY,                /* Alt N */
       TOGGLE_MORE,            /* Alt M */
       BAD_KEY,                /* CTRL-UP */
       BAD_KEY                 /* CTRL-DOWN */
      };

/******************************************************************************
 ==============================================================================
                      INTERNALLY VISIBLE GLOBAL VARIABLES
 ==============================================================================
 ******************************************************************************/

/* ----------------------------------------------------------
   The following 3 arrays are the memory-hogs mentioned above
   ---------------------------------------------------------- */

/* ---------------------------------------
   String-representations of CLIPS actions
   --------------------------------------- */

static char actionnames[MAX_ACTIONS][MAX_ACTION_NAME] =  
                                      /* These MUST  be in the same order as */
  {                                   /* KEY.H - a mapping depends on it     */
   "Left-Arrow",
   "Right-Arrow",
   "Up-Arrow",
   "Down-Arrow",
   "Page-Up",
   "Page-Down",
   "Buffer-Start",
   "Buffer-End",
   "Menu Execute",
   "Exit-to-OS",
   "Redraw-Screen",
   "Clear-Window",
   "Load",
   "Reset",
   "Run",
   "Step",
   "Help",
   "Watch/Unwatch All",
   "DOS-Command",
   "DOS-Shell",
   "Toggle --More--",
   "Execute Menu",
   "Debug Menu",
   "File Menu",
   "Action Menu",
   "Examine Menu",
   "Options Menu",
   "Describe Key"
  };

#if ENVIRONMENT_UTILITY

/* ----------------------------------------
   String-representations of keystrokes
     Thes tables are set up in much the
   same way as the key-binding tables above
   ---------------------------------------- */

static char keycnames[MAX_CNTRL_KEYS][MAX_KEY_NAME] =
        {
         "CTRL-@","Ctrl-A","Ctrl-B","Ctrl-C","Ctrl-D","Ctrl-E","Ctrl-F",
         "Ctrl-G","BKSPACE","TAB","NEWLINE","Ctrl-K","Ctrl-L","CRGRTN","Ctrl-N",
         "Ctrl-O","Ctrl-P","Ctrl-Q","Ctrl-R","Ctrl-S","Ctrl-T","Ctrl-U",
         "Ctrl-V","Ctrl-W","Ctrl-X","Ctrl-Y","Ctrl-Z","ESC","Ctrl-\\","Ctrl-]",
         "Ctrl-^","Ctrl-_"
        },

     keyexnames[MAX_EXTND_KEYS][MAX_KEY_NAME] =
        {
         "Fn1","Fn2","Fn3","Fn4","Fn5","Fn6","Fn7","Fn8","Fn9","Fn10",
         "Shft-Fn1","Shft-Fn2","Shft-Fn3","Shft-Fn4","Shft-Fn5","Shft-Fn6",
         "Shft-Fn7","Shft-Fn8","Shft-Fn9","Shft-Fn10",
         "Ctrl-Fn1","Ctrl-Fn2","Ctrl-Fn3","Ctrl-Fn4","Ctrl-Fn5","Ctrl-Fn6",
         "Ctrl-Fn7","Ctrl-Fn8","Ctrl-Fn9","Ctrl-Fn10",
         "Alt-Fn1","Alt-Fn2","Alt-Fn3","Alt-Fn4","Alt-Fn5","Alt-Fn6","Alt-Fn7",
         "Alt-Fn8","Alt-Fn9","Alt-Fn10",
         "BACKTAB","Ctrl-Brk","Ctrl-Lft","Ctrl-Rgt","Ctrl-End","Ctrl-PgDn",
         "Ctrl-Hm","Ctrl-PgUp","Dwn","Up","Lft","Rgt","Hm","End","PgDn","PgUp",
         "Delete","Insert","PrtSc","Alt-1","Alt-2","Alt-3","Alt-4","Alt-5",
         "Alt-6","Alt-7","Alt-8","Alt-9","Alt-10","Alt--","Alt-=",
         "Alt-Q","Alt-W","Alt-E","Alt-R","Alt-T","Alt-Y","Alt-U","Alt-I",
         "Alt-O","Alt-P","Alt-A","Alt-S","Alt-D","Alt-F","Alt-G","Alt-H",
         "Alt-J","Alt-K","Alt-L","Alt-Z","Alt-X","Alt-C","Alt-V","Alt-B",
         "Alt-N","Alt-M","Ctrl-Up","Ctrl-Dn"
        };

/* -------------------------------------------------------------------------
   Action table : Each element of the array is a keystroke which corresponds
   to an action.  Example : The code for EXIT_TO_OS is 0x8F.  To find its
   index in this table, it is bitwise anded with the extension mask 0x80,
   yielding 0x0F (or 15).  Thus, acttab[15][0] and acttab[15][1] correspond
   to the two key-bindings for EXIT_TO_OS.
   ------------------------------------------------------------------------- */

static int acttab[MAX_ACTIONS][2];

/* --------------------------------------------------------------------------
   Since there may be more actions than can fit on the screen and because
   the number is static, the screen buffers for the actions and their key-binds
   are implemented as Curses "pads".  A particular section of the actions
   are displayed by flushing different portions of these "pads" to the
   physical screen.
   -------------------------------------------------------------------------- */

static int offset = 0,            /* How many actions are off top of screen   */
           actions_displayed;     /* The total # of actions that can be
                                     on the screen at any one time            */

static int last_action = 0,
           last_bind = KEYBIND1;


static WINDOW *actions = NULL,     /* The "pads" mentioned above              */
              *names = NULL,
              *pkeys = NULL,
              *skeys = NULL;

#endif

/******************************************************************************
 ==============================================================================
                         EXTERNALLY VISIBLE FUNCTIONS
 ==============================================================================
 ******************************************************************************/

/******************************************************************************
 NAME        : key_map
 PURPOSE     : Maps raw keystrokes to CLIPS action codes
 DESCRIPTION : 
 INPUTS      : 1) The keystroke to be mapped
 RETURNS     : 1) The CLIPS action code if it was mapped
               2) The original keystroke otherwise
 NOTES       : Only ascii control characters and Curses extended keyboard
               characters are mapped.
 ******************************************************************************/
int key_map(chord)
  int chord;
  {
   if ((chord >= LOW_PRN_ASCII) && (chord <= HIGH_PRN_ASCII))
     return(chord);
   else if (chord > MAX_IBMPC_KEY)
     return(chord);
   else if (chord < LOW_PRN_ASCII)
     return((int) keyctab[chord]);
   else
     return((int) keyextab[chord-KEY_F(1)]);
  }

/******************************************************************************
 NAME        : key_action
 PURPOSE     : Returns a string corresponding to the action a key is bound to
 DESCRIPTION : 
 INPUTS      : 1) The keystroke for the action
 RETURNS     : 1) The action-string
               2) NULL if the key-code corresponds to no action
 NOTES       : Only ascii control characters and Curses extended keyboard
               characters are mapped.
 ******************************************************************************/
char *key_action(ch)
  int ch;
  {
   int index;

   if (ch < LOW_PRN_ASCII)
     index = keyctab[ch];
   else if (ch > HIGH_PRN_ASCII)
     index = keyextab[ch-KEY_F(1)];
   else
     return(NULL);
   if (index != BAD_KEY)
     {
      if ((index == ESC) || (index == DELETE))
        return(NULL);
      else if (index == NEWLINE)
        return("Pop-Up-Menu Execute");
      else
        return(actionnames[index & ~EXT_MASK]);
     }
   else
     return(NULL);
  }

#if ENVIRONMENT_UTILITY

/******************************************************************************
 NAME        : bind_keys
 PURPOSE     : Run-time utility for changing CLIPS action key-bindings
 DESCRIPTION : See various notes above.
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : If the ENVIRONMENT_UTILITY flags in the file SETUP.H is not set,
               all the support routines and data structures for this routine
               are not included in the code.
 ******************************************************************************/
void bind_keys()
  {
   int ch,y,x,bindch,
       select,
       quit = FALSE,
       getbind,
       new_action, new_bind;
   char select_instruction[ALL_COLS+1],
        bind_instruction[ALL_COLS+1];

   select_instruction[0] = ACS_LARROW;
   select_instruction[1] = ACS_RARROW;
   select_instruction[2] = ACS_UARROW;
   select_instruction[3] = ACS_DARROW;
   select_instruction[4] = EOS;
   strcat(select_instruction,"-select  <CRT>-Bind/Unbind   F1-Load ");
   strcat(select_instruction,"  F2-Save   F3-Defaults   ESC-Quit");
   strcpy(bind_instruction,
          "<key>-Bind New Key to Action  <del>-Unbind Old Key from Action");
   actions_displayed = LINES-5;
   attrset(A_FYELLOW|A_BBLACK);
   mvaddstr(0,COLS/2-21,"CLIPS DOS Interface Key-Binding Utility");
   attrset(A_FWHITE|A_BBLACK);
   mvaddstr(2,0,"CLIPS Action");
   mvaddstr(2,PKEY_START,"Key-Bind #1");
   mvaddstr(2,SKEY_START,"Key-Bind #2");
   instruction_line(select_instruction);

   /* ====================================
      Initialize Actions and Key Codes Pad
      ==================================== */
   if ((actions = newpad(MAX_ACTIONS,COLS)) == NULL)
     {
      kill_windows();
      return;
     }
   keypad(actions,TRUE);
   nodelay(actions,TRUE);
   if ((names = subpad(actions,MAX_ACTIONS,MAX_ACTION_NAME,0,0)) == NULL)
     {
      kill_windows();
      return;
     }
   xpaint(names,names_attr);
   wattrset(names,names_attr);
   if ((pkeys = subpad(actions,MAX_ACTIONS,MAX_KEY_NAME,0,PKEY_START)) == NULL)
     {
      kill_windows();
      return;
     }
   xpaint(pkeys,keys_attr);
   wattrset(pkeys,keys_attr);
   if ((skeys = subpad(actions,MAX_ACTIONS,MAX_KEY_NAME,0,SKEY_START)) == NULL)
     {
      kill_windows();
      return;
     }
   xpaint(skeys,keys_attr);
   wattrset(skeys,keys_attr);
   init_actiontab();
   set_actiontab();
   write_names(names);
   write_keys(pkeys,skeys);
#if MOUSE
   save_mouse();
   if (MAX_ACTIONS <= actions_displayed)
     mouse_region(LINES-actions_displayed-2,PKEY_START,
             LINES-actions_displayed-2+MAX_ACTIONS-1,SKEY_START+MAX_KEY_NAME-1);
   else
     mouse_region(LINES-actions_displayed-3,PKEY_START,
                  LINES-2,SKEY_START+MAX_KEY_NAME-1);
#endif
   attrset(A_FGREEN|A_BBLACK);
   touchwin(stdscr);
   highlight_key(last_action,last_bind);
   new_action = last_action;
   new_bind = last_bind;
   select = FALSE;
   while (! quit)
     {
      if ((new_action != last_action) || (new_bind != last_bind))
        highlight_key(new_action,new_bind);
      if (select)
        {
         if ((last_action | EXT_MASK) == DESCRIBE_KEY)
           {
            writestring("This binding is fixed!",1);
            select = FALSE;
           }
         else
           {
            instruction_line(bind_instruction); 
            refresh();
            hide_cursor();
#if MOUSE
            save_mouse();
#endif
            getbind = FALSE;
            while (! getbind)
              {
               if ((bindch = wgetch(actions)) != -1)
                 getbind = TRUE;
#if MOUSE
               else if (mouse_clicked(&bindch,&y,&x))
                 {
                  if (bindch == RIGHT_BUTTON)
                    {
                     bindch = ESC;
                     getbind = TRUE;
                    }
                  else
                    beep();
                 }
#endif
              }
#if MOUSE
            restore_mouse();
#endif
            if (bindch != ESC)
              {
               if (((bindch < LOW_PRN_ASCII) || (bindch > KEY_F0))
                     && (bindch != NEWLINE) && (bindch != CRGRTN))
                 change_key(bindch);
               else
                 writestring("Invalid key-binder!",1);
              }
            instruction_line(select_instruction);
            refresh();
            hide_cursor();
            select = FALSE;
           }
        }
      if ((ch = wgetch(actions)) != -1)
        {
         clear_status_line();
         quit = process(ch,&new_action,&new_bind,&select);
        }
#if MOUSE
      else if (mouse_moved_posn(&y,&x))
        {
         clear_status_line();
         y -= LINES-actions_displayed-2;
         if (y < 0)
           new_action = offset-1;
         else if (y > actions_displayed-1)
           new_action = offset+actions_displayed;
         else
           new_action = offset+y;
         if (new_action > MAX_ACTIONS-1)
           new_action = MAX_ACTIONS-1;
         else if (new_action < 0)
           new_action = 0;
         if (last_bind == KEYBIND1)
           new_bind = (x > PKEY_START+MAX_KEY_NAME-1) ? KEYBIND2 : KEYBIND1;
         else
           new_bind = (x < SKEY_START) ? KEYBIND1 : KEYBIND2;            
        }
      else if (mouse_clicked(&ch,&y,&x))
        {
         clear_status_line();
         if (ch == RIGHT_BUTTON)
           quit = TRUE;
         else if (ch == LEFT_BUTTON)
           select = TRUE;
         else
           beep();
        }
#endif
     }
   check_for_save(KEY_BINDS);
   last_action = 0;
   last_bind = KEYBIND1;
   kill_windows();

#if MOUSE
   restore_mouse();
#endif
  }

#endif

#if ENVIRONMENT_UTILITY

/******************************************************************************
 ==============================================================================
                         INTERNALLY VISIBLE FUNCTIONS
 ==============================================================================
 ******************************************************************************/

/******************************************************************************
 NAME        : initialize_bind_screen
 PURPOSE     : Loads internal tables with current key-bindings and writes
                 this information to the screen.
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Used for loading new keys and default keys.
 ******************************************************************************/
static void initialize_bind_screen()
  {
   init_actiontab();
   set_actiontab();
   xpaint(pkeys,keys_attr);
   xpaint(skeys,keys_attr);
   wattrset(pkeys,keys_attr);
   wattrset(skeys,keys_attr);
   write_keys(pkeys,skeys);
  }

/******************************************************************************
 NAME        : process
 PURPOSE     : Handles command characters for binding utility
 DESCRIPTION : 
 INPUTS      : 1) The character to be processed
               2) The address of the current action index
               3) The address of the current key-bind-number index
               4) The address of the indicator to change a key-binding or not
 RETURNS     : TRUE (1) is the character was ESC indicating that the user
               wishes to quit, FALSE (0) otherwise.
 NOTES       : None
 ******************************************************************************/
static int process(ch,action,bind,select)
  int ch,*action,*bind,*select;
  {
   int bindch;
   char inbuf[50];

   switch(ch)
     {
      case ESC       : return(TRUE);
      case KEY_UP    : if (last_action != 0)
                         *action = last_action-1;
                       break;
      case KEY_DOWN  : if (last_action !=  MAX_ACTIONS-1)
                         *action = last_action+1;
                       break;
      case KEY_RIGHT : 
      case KEY_LEFT  : *bind = (last_bind+1) % 2;
                       break;
      case KEY_HOME  : if ((last_action != 0) || (last_bind != KEYBIND1))
                         {
                          *action = 0;
                          *bind = KEYBIND1;
                         }
                       break;
      case KEY_ENDL  : if ((last_action != MAX_ACTIONS-1) ||
                           (last_bind != KEYBIND1))
                         {
                          *action = MAX_ACTIONS-1;
                          *bind = KEYBIND1;
                         }
                       break;
      case KEY_F(1)  : check_for_save(KEY_BINDS);
                       getstring("Load file : ",inbuf,50);
                       if (inbuf[0] == EOS)
                         break;
                       if (load_binds(inbuf) == OK_DATA_FILE)
                         {
                          initialize_bind_screen();
                          highlight_key(0,KEYBIND1);
                          *action = 0;
                          *bind = KEYBIND1;
                          writestring("Key-bindings loaded.",0);
                         }
                       else
                         writestring("Invalid key-bindings file!",1);
                       break;
      case KEY_F(2)  : getstring("Save file : ",inbuf,50);
                       if (inbuf[0] == EOS)
                         break;
                       if (save_binds(inbuf) == OK_DATA_FILE)
                         writestring("Key-bindings saved.",0);
                       else
                         writestring("Could not write to file!",1);
                       break;
      case KEY_F(3)  : check_for_save(KEY_BINDS);
                       writestring("Loading default key-bindings...",0);
                       if (default_binds() == OK_DATA_FILE)
                         {
                          initialize_bind_screen();
                          highlight_key(0,KEYBIND1);
                          *action = 0;
                          *bind = KEYBIND1;
                          writestring("Default key-bindings loaded.",0);
                         }
                       else
                         writestring("Default keys-file corrupted!",1);
                       break;
      case CRGRTN    :
      case NEWLINE   : *select = TRUE;
                       break;
      case CTRL_L    : highlight_key(last_action,last_bind);
      default        : writestring("Key has no function.",1);
     }
   return(FALSE);
  }

/******************************************************************************
 NAME        : kill_windows
 PURPOSE     : Deallocates run-time key-binding utility windows
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void kill_windows()
 {
   if (skeys != NULL)
     {
      delwin(skeys);
      skeys = NULL;
     }
   if (pkeys != NULL)
     {
      delwin(pkeys);
      pkeys = NULL;
     }
   if (names != NULL)
     {
      delwin(names);
      names = NULL;
     }
   if (actions != NULL)
     {
      delwin(actions);
      actions = NULL;
     }
  }

/******************************************************************************
 NAME        : init_actiontab
 PURPOSE     : Sets all action key-binds for run-time utility to -1
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void init_actiontab()
  {
   register int i;

   for (i = 0 ; i < MAX_ACTIONS ; i++)
     {
      acttab[i][KEYBIND1] = -1;
      acttab[i][KEYBIND2] = -1;
     }
  }

/******************************************************************************
 NAME        : set_action_tab
 PURPOSE     : Loads action table with current key-bindings in keyctab[] and
               keyextab[] (see descriptions of these arrays above).
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : This routine peruses the above array in a specific order :
                 first the control character array in sequential order,
                 second the extended character array in sequential order.
               Thus, any control-key-bindings will be found first and
               assigned, and then the extended-keys.  Thus, if there are
               more than two key-binds for an action (see notes above),
               the run-time will only know about the first two it finds in
               its search described above.
 ******************************************************************************/
static void set_actiontab()
  {
   register int i;

   for (i = 0 ; i < MAX_CNTRL_KEYS ; i++)
     set_action(keyctab[i],i);
   for (i = 0 ; i < MAX_EXTND_KEYS ; i++)
     set_action(keyextab[i],i+KEY_F(1));
  }

/******************************************************************************
 NAME        : set_action
 PURPOSE     : Called by set_actiontab to set an individual action-bind
 DESCRIPTION : 
 INPUTS      : 1) The action code
               2) the key-bind
 RETURNS     : Nothing useful
 NOTES       : See note for set_actiontab above.
 ******************************************************************************/
static void set_action(action,key)
  unsigned char action;
  int key;
  {
   register int i;

   if ((action >= (0|EXT_MASK)) && (action <= ((MAX_ACTIONS-1)|EXT_MASK)))
     {
      i = action & ~EXT_MASK;
      if (acttab[i][KEYBIND1] == -1)
        acttab[i][KEYBIND1] = key;
      else if (acttab[i][KEYBIND2] == -1)
        acttab[i][KEYBIND2] = key;
     }
  }

/******************************************************************************
 NAME        : write_names
 PURPOSE     : Writes action string-representations to "pad"
 DESCRIPTION : 
 INPUTS      : 1) The address of the name pad
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void write_names(names)
  WINDOW *names;
  {
   register int i;

   for (i = 0 ; i < MAX_ACTIONS ; i++)
     mvwaddstr(names,i,0,actionnames[i]);
  }

/******************************************************************************
 NAME        : write_keys
 PURPOSE     : Writes string-representations of key-binds to "pads"
 DESCRIPTION : 
 INPUTS      : 1) The address of the first key-bindings pad
               2) The address of the second key-bindings pad
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void write_keys(pkeys,skeys)
  WINDOW *skeys,*pkeys;
  {
   register int i;

   for (i = 0 ; i < MAX_ACTIONS ; i++)
     {
      if (acttab[i][KEYBIND1] != -1)
        mvwaddstr(pkeys,i,0,keyname(acttab[i][KEYBIND1]));
      if (acttab[i][KEYBIND2] != -1)
        mvwaddstr(skeys,i,0,keyname(acttab[i][KEYBIND2]));
     }
  }

/******************************************************************************
 NAME        : highlight_key
 PURPOSE     : Highlights newly selected key and unhighlights old one
 DESCRIPTION : Determines if new selection is currently visible on screen.
               If it isn't, the routine automatically shifts the display of
               the pad so that it is.
 INPUTS      : 1) The action table index (NOT the action code)
                  for the new key-bind
               2) The coulmn of the new key-bind (KEYBIND1 or KEYBIND2)
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void highlight_key(action,bind)
  int action,bind;
  {
   WINDOW *win;
   register int i;

   /* ===========================
       Unhighlight old selection 
      =========================== */
   win = (last_bind == KEYBIND1) ? pkeys : skeys;
   wattrset(win,keys_attr);
   for (i = 0 ; i < MAX_KEY_NAME ; i++)
     mvwaddch(win,last_action,i,mvwinch(win,last_action,i) & A_CHARTEXT);

   /* =========================
       Highlight new selection 
      ========================= */
   win = (bind == KEYBIND1) ? pkeys : skeys;
   wattrset(win,hi_keys_attr);
   for (i = 0 ; i < MAX_KEY_NAME ; i++)
     mvwaddch(win,action,i,mvwinch(win,action,i) & A_CHARTEXT);
   if (action < offset)
     offset = action;
   else if (action > (offset+actions_displayed-1))
     offset = action-actions_displayed+1;
   last_bind = bind;
   last_action = action;

   wnoutrefresh(stdscr);
   pnoutrefresh(names,offset,0,LINES-actions_displayed-2,0,
                               LINES-3,MAX_ACTION_NAME-1);
   pnoutrefresh(pkeys,offset,0,LINES-actions_displayed-2,
                      PKEY_START,LINES-3,PKEY_START+MAX_KEY_NAME-1);
   pnoutrefresh(skeys,offset,0,LINES-actions_displayed-2,
                      SKEY_START,LINES-3,SKEY_START+MAX_KEY_NAME-1);
#if MOUSE
   set_mouse(last_action-offset+LINES-actions_displayed-2,
             (last_bind == KEYBIND1) ? 
             PKEY_START + MAX_KEY_NAME/2 : SKEY_START + MAX_KEY_NAME/2);
#endif
   doupdate();
   hide_cursor();
  }

/******************************************************************************
 NAME        : change_key
 PURPOSE     : Changes the key-binding for the currently selected action-key
 DESCRIPTION : Automatically detects redundancy, and writes the new
               key string-representation to the window "pad"
                 Also allows the user to unbind a key from an action if
               the new key-bind passed is the "DELETE" key.
 INPUTS      : 1) The new key-bind
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void change_key(ch)
  int ch;
  {
   WINDOW *win;
   unsigned char *keytabact;
   int oldch;

   if (ch != DELETE)
     {
      if (ch < LOW_PRN_ASCII)
        keytabact = &keyctab[ch];
      else
        keytabact = &keyextab[ch-KEY_F(1)];
      if (*keytabact != BAD_KEY)
        {
         writestring("Key already bound!",1);
         return;
        }
      if ((last_action | EXT_MASK) == *keytabact)
        return;
      *keytabact = last_action | EXT_MASK;
      if ((oldch = acttab[last_action][last_bind]) != -1)
        {
         if (oldch < LOW_PRN_ASCII)
           keyctab[oldch] = BAD_KEY;
         else
           keyextab[oldch-KEY_F(1)] = BAD_KEY;
        }
      acttab[last_action][last_bind] = ch;
      win = (last_bind == KEYBIND1) ? pkeys : skeys;
      wmove(win,last_action,0);
      xwclrtocol(win,MAX_KEY_NAME-1);
      mvwaddstr(win,last_action,0,keyname(ch));
     }
   else
     {
      if ((oldch = acttab[last_action][last_bind]) != -1)
        {
         if (oldch < LOW_PRN_ASCII)
           keyctab[oldch] = BAD_KEY;
         else
           keyextab[oldch-KEY_F(1)] = BAD_KEY;
        }
      else
        return;
      acttab[last_action][last_bind] = -1;
      win = (last_bind == KEYBIND1) ? pkeys : skeys;
      wmove(win,last_action,0);
      xwclrtocol(win,MAX_KEY_NAME-1);
     }
   highlight_key(last_action,last_bind);
   binds_saved = FALSE;
  }

#endif
