/******************************************************************************
 FILE HELP.C :
   This file contains the routines for the CLIPS MS-DOS Interface on-line
 help utility.
 ******************************************************************************/

/******************************************************************************
 ==============================================================================
                                  HEADER FILES
 ==============================================================================
 ******************************************************************************/
#include "common.h"    /* Standard header files and useful constants/macros  */

#if CLP_TEXTPRO

#include "env.h"       /* Help file path                                     */
#include "curses.h"    /* Aspen Scientific's Curses V4.0 Package             */
#include "xcurses.h"
#include "box.h"
#include "curproto.h"
#include "clipswin.h"  /* redisplay_wins() def'n                             */
#include "color.h"     /* Screen color definitions                           */
#include "key.h"       /* Key-mapping definitions                            */
#if MOUSE
#include "mouse.h"     /* Mouse Systems PC-Mouse Interface Functions         */
#endif

/* CLIPS Definitions needed by clips_evaluate() */

#include "constant.h"
#include "access.h"

extern int add_router(char *,int,int (*)(),int (*)(),
                      int (*)(),int (*)(),int (*)());
extern int del_router(char *);
extern int act_router(char *);
extern int deact_router(char *);
int open_str_source(char *,char *,int);
int close_str_source(char *);
struct test *fctn0_parse(char *);
int generic_compute(struct test *,struct values *);
int returntests(struct test *);

/******************************************************************************
 ==============================================================================
                                   CONSTANTS
 ==============================================================================
 ******************************************************************************/

/******************************************************************************
 ==============================================================================
                                    MACROS
 ==============================================================================
 ******************************************************************************/
#define hide_cursor() mvcur(0,0,LINES,0)

/******************************************************************************
 ==============================================================================
                      EXTERNALLY VISIBLE FUNCTION PROTOTYPES
 ==============================================================================
 ******************************************************************************/
void on_line_help(void);       /* Main on-line help driver      */
void kill_on_line_help(void);  /* Frees on-line help memory     */

/******************************************************************************
 ==============================================================================
                      INTERNALLY VISIBLE FUNCTION PROTOTYPES
 ==============================================================================
 ******************************************************************************/
static int clips_evaluate(char *command);         /* Executes a CLIPS command */
static void init_help_window(void);               /* Draws initial window     */
static void print_pages(char *command);
                                                  /* Support fn. for display  */

/* Help Router Functions */

static int clpwin_help_query(char *logname);
static void clpwin_help_print(char *logname,char *str);

static void clips_file_command(char *buf,char *cmd,char *file);
                     /* Writes  command w/ filename argument in CLIPS format */

/******************************************************************************
 ==============================================================================
                      INTERNALLY VISIBLE GLOBAL VARIABLES
 ==============================================================================
 ******************************************************************************/
static int help_lines = 0,   /* Total number of help-lines allowed on screen */
           help_cols = 0,    /* Total number of help-columns allowed         */
           help_y = 0,       /* Starting screen coords for help-window       */
           help_x = 0,
           curr_line = 0,    /* The current line being printed on the screen */
           curr_col = 0;     /* Current column                               */

static int pages = 0;        /* # of pages in the help-facility              */

static WINDOW *text = NULL,  /* Text-window for help-info                    */
              *border = NULL;

static int top_nl = TRUE;    /* Ignore 1st newline output by print-region    */

/******************************************************************************
 ==============================================================================
                          EXTERNALLY VISIBLE FUNCTIONS
 ==============================================================================
 ******************************************************************************/

/*******************************************************************************
 NAME        : on_line_help
 PURPOSE     : Main driver for the CLIPS MS-DOS Interface on-line help utility
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void on_line_help()
  {
   char command[128];

   help_lines = CGA_LINES-2;
   help_cols = ALL_COLS;
   help_y = LINES/2-help_lines/2+1;
   help_x = 0;
   init_help_window();
   if (pages == 0)
     {
      if (! add_router("clpwin_help",40,clpwin_help_query,
                       (int (*)()) clpwin_help_print,NULL,NULL,NULL))
        {
         beep();
         redisplay_wins();
         return;
        }
      clips_file_command(command,"fetch",clpwin_help_file);
      pages = clips_evaluate(command);
     }
   act_router("clpwin_help");
   print_pages(command);
   deact_router("clpwin_help");
   delwin(text);
   text = NULL;
   delwin(border);
   border = NULL;
   redisplay_wins();
  }

/*******************************************************************************
 NAME        : kill_on_line_help
 PURPOSE     : Releases external file-table info for on-line interface help
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void kill_on_line_help()
  {
   char command[128];

   clips_file_command(command,"toss",clpwin_help_file);
   (void) clips_evaluate(command);
   del_router("clpwin_help");
   pages = 0;
  }

/******************************************************************************
 ==============================================================================
                          INTERNALLY VISIBLE FUNCTIONS
 ==============================================================================
 ******************************************************************************/

/*******************************************************************************
 NAME        : clips_evaluate
 PURPOSE     : Performs a CLIPS command
 DESCRIPTION : Used for "fetch", "print-region", and "toss"
 INPUTS      : 1) The command to be executed
 RETURNS     : 1) The return value of the CLIPS function (assumes the function
                  returns a number)
               2) or -1 if there was an error
 NOTES       : Uses the following functions :
                 open_str_source()   -- ROUTER.C
                 close_str_source()  -- ROUTER.C
                 fctn0_parse()       -- EXPRESSN.C
                 generic_compute()   -- EVALUATN.C
                 returntests()       -- EXPRESSN.C
 ******************************************************************************/
static int clips_evaluate(char *command)
  {
   VALUE result;
   struct test *top;

   open_str_source("help-command",command,0);
   top = fctn0_parse("help-command");
   if (top == NULL)
     {
      close_str_source("help-command");
      return(-1);
     }
   generic_compute(top,&result);
   returntests(top);
   close_str_source("help-command");
   return((int) get_valfloat(result));
  }

/*******************************************************************************
 NAME        : init_help_window
 PURPOSE     : Draws initial help-window before help-pages are loaded
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void init_help_window()
  {
   char *instr;

   if ((border = newwin(help_lines,help_cols,help_y,help_x)) == NULL)
     {
      beep();
      return;
     }
   if ((text = subwin(border,help_lines-2,help_cols-2,help_y+1,help_x+1))
        == NULL)
     {
      delwin(border);
      beep();
      return;
     }
   keypad(text,TRUE);
   nodelay(text,TRUE);
   xpaint(border,colors[POP_TEXT_BGD]);
   xbox(border,DB_V_LIN,DB_H_LIN,colors[POP_TEXT_BORDER]|colors[POP_TEXT_BGD]);
   wattrset(border,colors[POP_TEXT_BORDER]|colors[POP_TEXT_BGD]);
   wmove(border,0,help_cols/2-17);
   waddstr(border,"CLIPS MS-DOS Interface Information");
#if MOUSE
   if (MOUSE_LOADED)
     instr = "Next : Dn/PgDn/Enter/Lft-Clk  Prev : Up/PgUp  Quit : Esc/Rgt-Clk";
   else
#endif
     instr = "Next : Dn/PgDn/Enter  Prev : Up/PgUp  Quit : Esc";
   mvwaddstr(border,help_lines-1,help_cols/2-strlen(instr)/2,instr);
   xpaint(text,colors[POP_TEXT_PROMPT]|colors[POP_TEXT_BGD]);
   wattrset(text,colors[POP_TEXT_PROMPT]|colors[POP_TEXT_BGD]);
   touchwin(border);
   wnoutrefresh(border);
   touchwin(text);
   wnoutrefresh(text);
   doupdate();
   hide_cursor();
  }

/*******************************************************************************
 NAME        : print_pages
 PURPOSE     : Displays indivudual pages of the help-file on the screen
 DESCRIPTION : 
 INPUTS      : 1) Buffer to store CLIPS commands
 RETURNS     : Nothing useful
 NOTES       : Uses CLIPS print-region calls
                 Assumes help-file has already been "fetched"
 ******************************************************************************/
static void print_pages(char *command)
  {
   static int curr_page = 0;
   int new_page;
   int done = FALSE,
       error = FALSE;
   int ch;
   char pagebuf[10];
#if MOUSE
   int y,x;
#endif

#if MOUSE
   save_mouse();
#endif

   if (curr_page == 0)
     new_page = 1;
   else
     {
      new_page = curr_page;
      curr_page = 0;
     }
   while (! done)
     {
      if (new_page != curr_page)
        {
         xpaint(text,colors[POP_TEXT_PROMPT]|colors[POP_TEXT_BGD]);
         curr_line = curr_col = 0;
         clips_file_command(command,"print-region clpwin-help",
                                     clpwin_help_file);
         sprintf(pagebuf," PAGE%d)",new_page);
         command[strlen(command)-1] = EOS;
         strcat(command,pagebuf);
         top_nl = TRUE;
         if (clips_evaluate(command))
           {
            mvwprintw(border,0,help_cols-10," Pg %2d/%-2d",new_page,pages);
            touchwin(border);
            wnoutrefresh(border);
           }
         else
           {
            wmove(text,(help_lines-2)/2,(help_cols-2)/2-12);
            waddstr(text,"Could not find help file!");
            kill_on_line_help();
            error = TRUE;
           }
         touchwin(text);
         wnoutrefresh(text);
         doupdate();
         hide_cursor();
         curr_page = new_page;
         if (error)
           {
            beep();
            snooze(1100);
#if MOUSE
            restore_mouse();
#endif
            return;
           }
        }
      if ((ch = wgetch(text)) != -1)
        {
         ch = key_map(ch);
         if (ch == ESC)
           done = TRUE;
         else if ((ch == PAGE_UP) || (ch == UP_ARROW))
           new_page = ((new_page+pages-2) % pages) +1;
         else if ((ch == PAGE_DOWN) || (ch == DOWN_ARROW) ||
                  (ch == NEWLINE) || (ch == MAIN_MENU))
           new_page = (new_page % pages) +1;
         else if (ch == BUFFER_TOP)
           {
            if (curr_page != 1)
              new_page = 1;
            else
              beep();
           }
         else if (ch == BUFFER_END)
           {
            if (curr_page != pages)
              new_page = pages;
            else
              beep();
           }
         else
           beep();
        }
#if MOUSE
      else if (mouse_clicked(&ch,&y,&x))
        {
         if (ch == LEFT_BUTTON)
           new_page = (new_page % pages) + 1;
         else if (ch == RIGHT_BUTTON)
           done = TRUE;
         else
           beep();
        }
#endif
     }
#if MOUSE
   restore_mouse();
#endif
  }

/*******************************************************************************
 NAME        : clpwin_help_query
 PURPOSE     : Router recognizer function for help-system
 DESCRIPTION : 
 INPUTS      : 1) The router logical name
 RETURNS     : 1) TRUE (1) if the router was recognized
               2) FALSE (0) otherwise
 NOTES       : None
 ******************************************************************************/
static int clpwin_help_query(char *logname)
  {
   if (strcmp("clpwin-help",logname) == 0)
     return(TRUE);
   if (strcmp("werror",logname) == 0)
     return(TRUE);
   return(FALSE);
  }

/*******************************************************************************
 NAME        : clpwin_help_print
 PURPOSE     : Router print function for help utility
 DESCRIPTION : Prints string to the current help-window
 INPUTS      : 1) The router logical name
               2) The string to be printed
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void clpwin_help_print(char *logname,char *str)
  {
   register int i;

   if (text == NULL)
     return;
   if (strcmp(logname,"clpwin-help") == 0)
     {
      for (i = 0 ; ((curr_line < (help_lines-2)) ? str[i] != EOS : FALSE) ; i++)
        {
         if (str[i] == NEWLINE)
           {
            if ((curr_line == 0) && (curr_col == 0) && top_nl)
              top_nl = FALSE;
            else
              {
               curr_col = 0;
               curr_line++;
              }
           }
         else if (curr_col > (help_cols-3))
           {
            curr_col = 0;
            curr_line++;
            if (curr_line < (help_lines-2))
              mvwaddch(text,curr_line,curr_col,str[i]);
           }
         else
           mvwaddch(text,curr_line,curr_col++,str[i]);
       }
     }
  }

/*******************************************************************************
 NAME        : clips_file_command
 PURPOSE     : Copies command and filename to user's buffer in form :
                 (<command> "<filename>") where all backslashes in the
               filename have been expanded to a pair of backslashes.
 DESCRIPTION : 
 INPUTS      : 1) The buffer to store the expanded command
               2) The CLIPS command
               3) The filename argument
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void clips_file_command(char *buf,char *cmd,char *file)
  {
   register int i,j;

   sprintf(buf,"(%s \"",cmd);
   for (i = strlen(buf) , j = 0 ; file[j] != EOS ; i++ , j++)
     {
      if (file[j] != '\\')
        buf[i] = file[j];
      else
        {
         buf[i] = '\\';
         buf[++i] = '\\';
        }
      buf[i+1] = EOS;
     }
   strcat(buf,"\")");
  }

#endif

/*******************************************************************************
 NAME        : 
 PURPOSE     : 
 DESCRIPTION : 
 INPUTS      : 
 RETURNS     : 
 NOTES       : 
 ******************************************************************************/
