/* ************************************************************************
   FILE : ENV.C
      This file contains the support routines that load the CLIPS
   configuration file, save and load the color/key-bindings
   binary data files, execute DOS commands, examine the status
   of memory, and deal with an EGA Video Card.
     All Non-ANSI/System-Specific/Compiler-Specific code is in this file
   (with the exception of MOUSE.C which also contains compiler/system
    dependent code).
   *********************************************************************** */
   
/******************************************************************************
 ==============================================================================
                                HEADER FILES
 ==============================================================================
 ******************************************************************************/
#include "common.h"     /* Common constants and macros */
#include <conio.h>

/* --------------------------------------------------
   Copied type declaration needed from CURSES.H.
   Did not include actual header file since its macro
   getch() conflicts with the function definition in
   the system file CONIO.H.
   -------------------------------------------------- */
typedef unsigned short ch_attr;

#include "color.h"      /* Color data structures       */
#include "key.h"        /* Binding data structures     */
#define ENV_SOURCE
#include "env.h"        /* Environment data file codes */

#if ! ENVIRONMENT_UTILITY

/* ----------------------
   CLIPS Memory Functions
   ---------------------- */
extern unsigned long actual_mem_used(void);
extern unsigned long actual_pool_size(void);

#endif

/* ---------------------------------------------------------------------
   System files required to work with temporary files, binary data files,
   searching the DOS path, checking for keyboard strokes, and spawning
   processes.
   --------------------------------------------------------------------- */
#include <dos.h>
#include <io.h>
#include <process.h>
#include <fcntl.h>

#if IBM_TBC

#include <dir.h>
#include <sys\stat.h>

#else

#include <sys\types.h>
#include <sys\stat.h>
#include <direct.h>
#include <time.h>

#define MAXPATH  _MAX_PATH
#define MAXDRIVE _MAX_DRIVE
#define MAXDIR   _MAX_DIR

#endif

/******************************************************************************
 ==============================================================================
                                   CONSTANTS
 ==============================================================================
 ******************************************************************************/
#define DOS_SYSTEM_VECT 0x21    /* DOS System Service Interrupt Vector       */
#define DOS_CRIT_VECT   0x24    /* DOS Interrupt Vector for Disk Errors      */
#define DRIVE_DATA_FUNC 0x1C    /* DOS I21H - Function Code : Get Drive Data */
#define DISK_NOT_READY  0xFF    /* Disk Not Ready Code for I21H - F1CH       */

/******************************************************************************
 ==============================================================================
                     EXTERNALLY VISIBLE FUNCTION PROTOTYPES
 ==============================================================================
 ******************************************************************************/

#if ! ENVIRONMENT_UTILITY

int  setup_config(int mono_flag);
                            /* Search for configuration file and do setup     */
void io_error_clips(void),  /* Sets DOS Critical Error Handler                */
     io_error_dos(void);    /* Resets DOS Critical Error Handler to normal    */
void interrupt handle_crit();
                            /* Disk error interrupt handler                   */
int disk_ready(int drive);  /* Gets readiness of disk in specified drive      */

void reset_cwd(void);       /* Resets current-working-directory to startup    */
int  dos_command(char *);   /* Executes a single DOS command                  */
int  spawn_shell(void);     /* Starts a new COMMAND.COM under CLIPS           */
unsigned long mem_left(void);
                            /* Returns amount of free RAM                     */
int  ega_exist(void);       /* Sees if EGA Card is present                    */
void ega_open(void);        /* Enables EGA Video Card                         */
void ega_close(void);       /* Disables EGA Video Card                        */
void snooze(unsigned);      /* Puts system to sleep                           */

#else

int  save_defaults(void);   /* Puts default colors/bindings in temporary file */
void trash_defaults(void);  /* Unlinks defaults temporary file                */
int  default_colors(void);  /* Loads default colors from temporary file       */
int  default_binds(void);   /* Loads default bindings from temporary file     */
int  save_colors(char *);   /* Saves colors to named file                     */
int  save_binds(char *);    /* Saves bindings to named file                   */

#endif

int  load_colors(char *);   /* Loads colors from named file                   */
int  load_binds(char *);    /* Loads bindings from named file                 */

#if ! ENVIRONMENT_UTILITY

int  valid_color_file(char *);      /* Verifies color data files      */
int  valid_bind_file(char *);       /* Verifies bind data files       */
int  valid_color_bind_file(char *); /* Verifies bind-color data files */

#endif

/******************************************************************************
 ==============================================================================
                      INTERNALLY VISIBLE FUNCTION PROTOTYPES
 ==============================================================================
 ******************************************************************************/
#if ! ENVIRONMENT_UTILITY
static int load_configuration_file(void);
                                        /* Looks for and load CLIPSWIN.CFG    */
#else

static int create_data_file(char *);    /* Allocates full data space for
                                                file if it does not exist     */

#endif

static int valid_file(char *,unsigned long); 
                                        /* Checks for good data file          */

/******************************************************************************
 ==============================================================================
                     EXTERNALLY VISIBLE GLOBAL VARIABLES
 ==============================================================================
 ******************************************************************************/
#if ! ENVIRONMENT_UTILITY

void (interrupt *old_handle_crit)();    /* Pointer to old critical
                                           error handler           */

char clipswinpath[MAXDRIVE+MAXDIR+1];     /* Path for CLIPS to search --
                                                Currently unused              */
char user_env_file[MAXPATH+1];            /* User environment file            */
char clpwin_help_file[MAXPATH+1];         /* Interface help-file              */

#else

int colors_saved = TRUE,         /* Indicates when env info is on disk */
    binds_saved = TRUE;

#endif

/******************************************************************************
 ==============================================================================
                     INTERNALLY VISIBLE GLOBAL VARIABLES
 ==============================================================================
 ******************************************************************************/
#if ! ENVIRONMENT_UTILITY
static int sdsk;                     /* Startup disk-drive */
static char swd[MAXDRIVE+MAXDIR+1];  /* Startup directory  */

#else
static char *temp_default_file = NULL;   /* Used to hold default colors/binds
                                              during execution                */
#endif
static unsigned long validkey = 0xFFFFDFAF, /* Color and Bind data valid      */
                     colorkey = 0xFFFF000F, /* Only color data valid          */
                     bindkey  = 0xF000DFAF, /* Only bind data valid            */
                     anykey   = 0xFDEBFCCC,
                     createkey= 0xF000000F;
                                   /* Used to mark/recognize valid data files */


/******************************************************************************
 ==============================================================================
                         EXTERNALLY VISIBLE FUNCTIONS
 ==============================================================================
 ******************************************************************************/

#if ! ENVIRONMENT_UTILITY

/******************************************************************************
 NAME        : setup_config
 PURPOSE     : Establishes color/key-binding environment for CLIPS
 DESCRIPTION : This routine first saves the default key-bindings/colors to
               a temporary binary data file, then it searches the current
               directory and then the DOS path for the CLIPSWIN.CFG
               configuration file and tries to setup the information it
               specifies.  Finally, the routine saves the current disk-drive
               and working directory.
 INPUTS      : 1) Whether the command-line monochrome flag is set or not
 RETURNS     : TRUE (1) if everything went OK, FALSE (0) otherwise.
 NOTES       : If the routine cannot find CLIPSWIN.CFG, then it just assumes
               the following defaults :
                  CLIPS data path : Current working directory (cwd). (Not Used)
                  CLIPSWIN Color/Bindings File : CLIPSENV.DAT in the cwd.
                  CLIPSWIN Help File : CLIPSWIN.HLP in the cwd.
                  
 ******************************************************************************/
int setup_config(int mono_flag)
  {
   int i;

   if (mono_flag)
     {
      for (i = 0 ; i < MAX_COLOR_DEFS ; i++)
        colors[i] = mono[i];
     }
   if (! load_configuration_file())
     return(FALSE);
   if (! mono_flag)
     (void) load_colors(user_env_file);
   (void) load_binds(user_env_file);
#if IBM_TBC
   sdsk = getdisk();
#else
   _dos_getdrive(&sdsk);
   sdsk--;
#endif
   if (getcwd(swd,MAXDRIVE+MAXDIR) == NULL)
     return(FALSE);
   return(TRUE);
  }

/*****************************************************************************
 NAME        : io_error_clips
 PURPOSE     : Sets DOS Critical Error Hanlder (Interrupt 0x24) to
                a special handler
 DESCRIPTION : Uses TurboC routines getvect() and setvect()
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 *****************************************************************************/
void io_error_clips()
  {
#if IBM_TBC
   old_handle_crit = getvect(DOS_CRIT_VECT);
   setvect(DOS_CRIT_VECT,handle_crit);
#else
   old_handle_crit = _dos_getvect(DOS_CRIT_VECT);
   _dos_setvect(DOS_CRIT_VECT,handle_crit);
#endif
  }

/*****************************************************************************
 NAME        : io_error_dos
 PURPOSE     : Resets DOS Critical Error Hanlder (Interrupt 0x24) to
                normal default handler
 DESCRIPTION : Uses TurboC routine setvect()
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 *****************************************************************************/
void io_error_dos()
  {
#if IBM_TBC
   setvect(DOS_CRIT_VECT,old_handle_crit);
#else
   _dos_setvect(DOS_CRIT_VECT,old_handle_crit);
#endif
  }

/*****************************************************************************
 NAME        : handle_crit
 PURPOSE     : Interrupt handler for critical disk errors
 DESCRIPTION : Tells DOS to ignore errors and sets a flag informing
               change_disk() of the error.
 INPUTS      : Intel registers and flags
 RETURNS     : Nothing useful
 NOTES       : Uses Turbo-C interrupt function type
 *****************************************************************************/
void interrupt handle_crit(bp,di,si,ds,es,dx,cx,bx,ax,ip,cs,flgs)
  {
   ax = 0;  /* Ignore critical errors */
  }

/******************************************************************************
 NAME        : disk_ready
 PURPOSE     : Gets readiness of disk in specified drive
 DESCRIPTION : Determines whether disk is ready and what type of disk
               the drive is meant to read
 INPUTS      : 1) The drive number (A = 0, B= 1, etc.)
 RETURNS     : TRUE (1) if the disk is ready, FALSE (0) otherwise
 NOTES       : Uses DOS System Service Interrupt 0x21 - Function 0x1c
               (Get Drive Data)
               DOS 0x21 Function 0x1c - tries to access the disk and may
               cause a trap to 0x24 (critical error handler) if the drive is
               not ready.  Thus, it is important that the 0x24 handler do
               nothing.
 ******************************************************************************/
int disk_ready(drive)
  int drive;
  {
   union REGS regs;
   struct SREGS sregs;

   regs.h.ah = (unsigned char) DRIVE_DATA_FUNC;
   regs.h.dl = (unsigned char) (drive+1);
   int86x(DOS_SYSTEM_VECT,&regs,&regs,&sregs);
   return(regs.h.al != ((unsigned char) DISK_NOT_READY));
  }

/******************************************************************************
 NAME        : reset_cwd
 PURPOSE     : Resets corrennt-working-disk and directory to the startup values
               set by setup_config()
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Uses the Turbo-C Library Routines setdisk() and chdir()
 ******************************************************************************/
void reset_cwd()
  {
#if IBM_MSC
   unsigned drives;

   _dos_setdrive(sdsk+1,&drives);
#else
   (void) setdisk(sdsk);
#endif
   (void) chdir(swd);
  }

/******************************************************************************
 NAME        : dos_command
 PURPOSE     : Executes a single DOS command
 DESCRIPTION : 
 INPUTS      : 1) The command to be executed
 RETURNS     : TRUE (1) if all went well, FALSE (0) otherwise.
 NOTES       : Uses the C-Library function system().
               Assumes that the terminal environment has already been reset
                 to normal.
 ******************************************************************************/
int dos_command(cmd)
  char *cmd;
  {
   if (system(cmd))
     printf("Command execution error or not enough memory!\n\a");
   printf("\nPress any key to return to CLIPS...");
   getch();     /* This is the standard routine - not the Curses one */
   return(TRUE);
  }

/******************************************************************************
 NAME        : spawn_shell
 PURPOSE     : Starts up a new COMMAND.COM under CLIPS
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : TRUE (1) if all went well, FALSE (0) otherwise.
 NOTES       : Uses the C-Library function spawn().
               Typing EXIT at the new COMMAND.COM's prompt will return to CLIPS
               Assumes that the terminal environment has already been reset
                 to normal.
 ******************************************************************************/
int spawn_shell()
  {
   char *cli;
   int clirtn;

   if ((cli = getenv("COMSPEC")) != NULL)
     {
      printf("Type EXIT to return to CLIPS...\n");
      clirtn = spawnl(P_WAIT,cli,cli,NULL);
      if (clirtn == -1)
        {
         printf("Command interpreter error or not enough memory!\n");
         return(FALSE);
        }
     }
   else
     return(FALSE);
   return(TRUE);
  }

/******************************************************************************
 NAME        : mem_left
 PURPOSE     : Determines the amount of RAM still available for use
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : The amount of RAM left
 NOTES       : Uses the Turbo-C Library Routine coreleft().
 ******************************************************************************/
unsigned long mem_left()
  {
#if IBM_TBC
   return(coreleft() + actual_pool_size());
#else
   unsigned avail_segs;
   unsigned long count;

   (void) _dos_allocmem(64000,&avail_segs);
   count = ((unsigned long) avail_segs) * 16L;
   return(count + actual_pool_size());
#endif
  }

/******************************************************************************
 NAME        : ega_exist
 PURPOSE     : Determines the existence or lack of an EGA Video Card
 DESCRIPTION : Generates an interrupt to have the OS look for the EGA
 INPUTS      : None
 RETURNS     : TRUE (1) if EGA capability exists, FALSE (0) otherwise.
 NOTES       : Non-portable

               This code was taken from the source code for the public
               domain MicroEMACS V3.9 Editor.
 ******************************************************************************/
int ega_exist()
  {
   union REGS rg;

   rg.x.ax = 0x1200;
   rg.x.bx = 0xff10;
   int86(0x10,&rg,&rg);            /* If EGA, bh=0-1 and bl=0-3 */
   return ((! (rg.x.bx & 0xfefc)) ? TRUE : FALSE);  /* Yes, it's EGA */
  }

/******************************************************************************
 NAME        : ega_open
 PURPOSE     : Enables the EGA Video Card
 DESCRIPTION : Uses interrupts to get the OS to do the dirty work
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Non-portable

               This function is only used by the interface to reset EGA
               mode after a spawn to DOS - the Curses function initscr()
               handles the reformatting of screen stuff when switching
               between EGA and CGA modes

               This code was taken from the source code for the public
               domain MicroEMACS V3.9 Editor.
 ******************************************************************************/
void ega_open() /* init the computer to work with the EGA */
  {
   union REGS rg;

   /* put the beast into EGA 43 row mode */
   rg.x.ax = 3;
   int86(16, &rg, &rg);

   rg.h.ah = 17;  /* set char. generator function code */
   rg.h.al = 18;  /*  to 8 by 8 double dot ROM         */
   rg.h.bl = 0;  /* block 0                           */
   int86(16, &rg, &rg);

   rg.h.ah = 18;  /* alternate select function code    */
   rg.h.al = 0;  /* clear AL for no good reason        */
   rg.h.bl = 32;  /* alt. print screen routine         */
   int86(16, &rg, &rg);

   rg.h.ah = 1;  /* set cursor size function code */
   rg.x.cx = 0x0607; /* turn cursor on code */
   int86(0x10, &rg, &rg);

   outp(0x3d4, 10); /* video bios bug patch */
   outp(0x3d5, 6);
  }

/******************************************************************************
 NAME        : ega_close
 PURPOSE     : Disables the EGA Video Card
 DESCRIPTION : Uses interrupts to get the OS to do the dirty work
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Non-portable

               This function is only used by the interface to reset CGA
               mode before a spawn to DOS - the Curses function initscr()
               handles the reformatting of screen stuff when switching
               between EGA and CGA modes

               This code was taken from the source code for the public
               domain MicroEMACS V3.9 Editor.
 ******************************************************************************/
void ega_close()
  {
   union REGS rg;

   /* put the beast into 80 column mode */
   rg.x.ax = 3;
   int86(16, &rg, &rg);
  }

/******************************************************************************
 NAME        : snooze
 PURPOSE     : Puts system to sleep for specified amount of time
 DESCRIPTION : 
 INPUTS      : 1) The number of milliseconds to sleep
 RETURNS     : Nothing useful
 NOTES       : Uses the Turbo C Library Routine delay()
 ******************************************************************************/
void snooze(msecs)
  unsigned msecs;
  {
#if IBM_TBC
   delay(msecs);
#else
   int tm = 0;

   do
     {
      if ((tm = clock()) == (clock_t) -1)
        break;
     }
   while (tm/CLK_TCK/1000 < msecs);
#endif
  }

#else

/******************************************************************************
 NAME        : save_defaults
 PURPOSE     : Writes default colors/key-bindings to temporary file
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : TRUE (1) if everything went OK, FALSE (0) otherwise.
 NOTES       : Permission to write to this temporary file is denied
               so that only these default() routines may manipulate the
               file.  This prevents the user from accidentally deleting
               the file (while executing a DOS command from CLIPS or
               in a spawned shell) while the program still needs it.
 ******************************************************************************/
int save_defaults()
  {
   if ((temp_default_file = mktemp("XXXXXX")) == NULL)
     return(FALSE);
   if (save_colors(temp_default_file) != OK_DATA_FILE)
     return(FALSE);
   if (save_binds(temp_default_file) != OK_DATA_FILE)
     return(FALSE);
   if (chmod(temp_default_file,S_IREAD) == -1)
     return(FALSE);
   return(TRUE);
  }

/******************************************************************************
 NAME        : trash_defaults
 PURPOSE     : Deletes the temporary default data file
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void trash_defaults()
  {
   if (chmod(temp_default_file,S_IREAD|S_IWRITE) == -1)
     return;
   (void) unlink(temp_default_file);
  }

/******************************************************************************
 NAME        : default_colors
 PURPOSE     : Loads default colors from temporary data file
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : OK_DATA_FILE (1) if all went well,
               NO_DATA_FILE (0) if it could not find the temporary file,
               BAD_DATA_FILE (-1) if the temporary data file was corrupted.
 NOTES       : None
 ******************************************************************************/
int default_colors()
  {
   int rtn;

   if (chmod(temp_default_file,S_IREAD|S_IWRITE) == -1)
     return(BAD_DATA_FILE);
   if ((rtn = load_colors(temp_default_file)) != OK_DATA_FILE)
     return(rtn);
   if (chmod(temp_default_file,S_IREAD) == -1)
     return(BAD_DATA_FILE);
   return(rtn);
  }

/******************************************************************************
 NAME        : default_binds
 PURPOSE     : Loads default key-bindings from temporary data file
 DESCRIPTION : 
 INPUTS      : none
 RETURNS     : OK_DATA_FILE (1) if all went well,
               NO_DATA_FILE (0) if it could not find the temporary file,
               BAD_DATA_FILE (-1) if the temporary data file was corrupted.
 NOTES       : None
 ******************************************************************************/
int default_binds()
  {
   int rtn;

   if (chmod(temp_default_file,S_IREAD|S_IWRITE) == -1)
     return(BAD_DATA_FILE);
   if ((rtn = load_binds(temp_default_file)) != OK_DATA_FILE)
     return(rtn);
   if (chmod(temp_default_file,S_IREAD) == -1)
     return(BAD_DATA_FILE);
   return(rtn);
  }


/* --------------------------------------------------------------------------
   The following routines write the color and key-binding arrays directly
   to disk in binary form.  Reading these files is also similarly easy.
   The format in the data file is as follows:
      4 bytes  : MATCHKEY
      60 bytes : Color data (2 bytes each for 30 colors)
      32 bytes : Control key bindings data (1 byte each for 32 keys)
      99 bytes : Extended key bindings data (1 byte each for 99 keys)

      ________

      195 bytes

   When data files are being read, they are verified by two factors :
     The first 4 bytes must match the specified key, and the file length
   must be exactly 195 bytes.  This will almost always prevent bad data from
   being read into the arrays, but like anything else, a hacker could
   circumvent this.  These checks are meant only to prevent accidental
   but not deliberate misuse.
   -------------------------------------------------------------------------- */

/******************************************************************************
 NAME        : save_colors
 PURPOSE     : Writes interface color array out to file in binary form
 DESCRIPTION : 
 INPUTS      : 1) The name of the file to which the data is to be written
 RETURNS     : OK_DATA_FILE (1) if everything went OK,
               BAD_DATA_FILE (-1) otherwise
 NOTES       : If the file in question does not already exist, this function
               creates it so that it is also big enough to hold any 
               key-binding data as well.
 ******************************************************************************/
int save_colors(file)
  char *file;
  {
   int handle;
   int nbytes;
   unsigned long key;

   if ((handle = valid_file(file,anykey)) == NO_DATA_FILE)
     handle = create_data_file(file);
   if (handle == BAD_DATA_FILE)
     return(BAD_DATA_FILE);
   if (read(handle,(char *) &key,sizeof(unsigned long)) 
                  != sizeof(unsigned long))
     {
      close(handle);
      return(BAD_DATA_FILE);
     }
   nbytes = sizeof(ch_attr)*MAX_COLOR_DEFS;
   if (write(handle,(char *) colors,nbytes) != nbytes)
     {
      close(handle);
      return(BAD_DATA_FILE);
     }
   if (lseek(handle,0L,SEEK_SET) != 0)
     {
      close(handle);
      return(BAD_DATA_FILE);
     }
   key |= colorkey;
   if (write(handle,(char *) &key,sizeof(unsigned long))
              != sizeof(unsigned long))
     {
      close(handle);
      return(BAD_DATA_FILE);
     }
   close(handle);
   colors_saved = TRUE;
   return(OK_DATA_FILE);
  }

/******************************************************************************
 NAME        : save_binds
 PURPOSE     : Writes interface key-binding arrays out to file in binary form
 DESCRIPTION : 
 INPUTS      : 1) The name of the file to which the data is to be written
 RETURNS     : OK_DATA_FILE (1) if everything went OK,
               BAD_DATA_FILE (-1) otherwise
 NOTES       : If the file in question does not already exist, this function
               creates it so that it is also big enough to hold any 
               color data as well.
 ******************************************************************************/
int save_binds(file)
  char *file;
  {
   int handle;
   long offset;
   int nbytes;
   unsigned long key;

   if ((handle = valid_file(file,anykey)) == NO_DATA_FILE)
     handle = create_data_file(file);
   if (handle == BAD_DATA_FILE)
     return(BAD_DATA_FILE);
   if (read(handle,(char *) &key,sizeof(unsigned long))
        != sizeof(unsigned long))
     {
      close(handle);
      return(BAD_DATA_FILE);
     }
   offset = sizeof(unsigned long) + (sizeof(ch_attr)*MAX_COLOR_DEFS);
   if (lseek(handle,offset,SEEK_SET) != offset)
     {
      close(handle);
      return(BAD_DATA_FILE);
     }
   nbytes = sizeof(unsigned char)*MAX_CNTRL_KEYS;
   if (write(handle,(char *) keyctab,nbytes) != nbytes)
     {
      close(handle);
      return(BAD_DATA_FILE);
     }
   nbytes = sizeof(unsigned char)*MAX_EXTND_KEYS;
   if (write(handle,(char *) keyextab,nbytes) != nbytes)
     {
      close(handle);
      return(BAD_DATA_FILE);
     }
   if (lseek(handle,0L,SEEK_SET) != 0)
     {
      close(handle);
      return(BAD_DATA_FILE);
     }
   key |= bindkey;
   if (write(handle,(char *) &key,sizeof(unsigned long))
              != sizeof(unsigned long))
     {
      close(handle);
      return(BAD_DATA_FILE);
     }
   close(handle);
   binds_saved = TRUE;
   return(OK_DATA_FILE);
  }

#endif

/******************************************************************************
 NAME        : load_colors
 PURPOSE     : Reads color array from binary data file
 DESCRIPTION : 
 INPUTS      : 1) The name of the file
 RETURNS     : 1) OK_DATA_FILE if colors successfully loaded
               2) NO_DATA_FILE if the file didn't exist
               3) BAD_DATA_FILE if the file was an invalid data file
 NOTES       : None
 ******************************************************************************/
int load_colors(file)
  char *file;
  {
   int handle;
   int nbytes;
   long offset;

   handle = valid_file(file,colorkey);
   if ((handle == BAD_DATA_FILE) || (handle == NO_DATA_FILE))
     return(handle);
   offset = sizeof(unsigned long);
   if (lseek(handle,offset,SEEK_SET) != offset)
     {
      close(handle);
      return(BAD_DATA_FILE);
     }
   nbytes = sizeof(ch_attr)*MAX_COLOR_DEFS;
   if (read(handle,(char *) colors,nbytes) != nbytes)
     {
      close(handle);
      return(BAD_DATA_FILE);
     }
   close(handle);
#if ENVIRONMENT_UTILITY
   colors_saved = TRUE;
#endif
   return(OK_DATA_FILE);
  }

/******************************************************************************
 NAME        : load_binds
 PURPOSE     : Reads key-binding arrays from binary data file
 DESCRIPTION : 
 INPUTS      : 1) The name of the file
 RETURNS     : 1) OK_DATA_FILE if colors successfully loaded
               2) NO_DATA_FILE if the file didn't exist
               3) BAD_DATA_FILE if the file was an invalid data file
 NOTES       : None
 ******************************************************************************/
int load_binds(file)
  char *file;
  {
   int handle;
   int nbytes;
   long offset;

   handle = valid_file(file,bindkey);
   if ((handle == BAD_DATA_FILE) || (handle == NO_DATA_FILE))
     return(handle);
   offset = sizeof(unsigned long) + (MAX_COLOR_DEFS*sizeof(ch_attr));
   if (lseek(handle,offset,SEEK_SET) != offset)
     {
      close(handle);
      return(BAD_DATA_FILE);
     }
   nbytes = sizeof(unsigned char)*MAX_CNTRL_KEYS;
   if (read(handle,(char *) keyctab,nbytes) != nbytes)
     {
      close(handle);
      return(BAD_DATA_FILE);
     }
   nbytes = sizeof(unsigned char)*MAX_EXTND_KEYS;
   if (read(handle,(char *) keyextab,nbytes) != nbytes)
     {
      close(handle);
      return(BAD_DATA_FILE);
     }
   close(handle);
#if ENVIRONMENT_UTILITY
   binds_saved = TRUE;
#endif
   return(OK_DATA_FILE);
  }

#if ! ENVIRONMENT_UTILITY
/******************************************************************************
 NAME        : valid_color_file,valid_bind_file,valid_color_bind_file
 PURPOSE     : Determines whether a file holds valid colors/binds data or not
 DESCRIPTION : 
 INPUTS      : 1) The name of the file
 RETURNS     : TRUE (1) if the file was valid, FALSE (0) otherwise
 NOTES       : Uses valid_file() defined below.
 ******************************************************************************/
int valid_color_file(file)
  char *file;
  {
   int handle;

   if ((handle = valid_file(file,colorkey)) < 0)
     return(FALSE);
   else
     close(handle);
   return(TRUE);
  }

int valid_bind_file(file)
  char *file;
  {
   int handle;

   if ((handle = valid_file(file,bindkey)) < 0)
     return(FALSE);
   else
     close(handle);
   return(TRUE);
  }

int valid_color_bind_file(file)
  char *file;
  {
   int handle;

   if ((handle = valid_file(file,validkey)) < 0)
     return(FALSE);
   else
     close(handle);
   return(TRUE);
  }
#endif

/* **************************************************************************
   ==========================================================================
                        INTERNALLY VISIBLE FUNCTIONS
   ==========================================================================
   ************************************************************************** */

#if ! ENVIRONMENT_UTILITY

/******************************************************************************
 NAME        : load_configuration_file
 PURPOSE     : Reads CLIPSWIN.CFG to setup path environments
 DESCRIPTION : Establishes a path that CLIPS can use to find files
               (currently not used), establishes the path to the
               environment colors/key-bindings file (if one exists),
               and establishes the path to the interface help file.
 INPUTS      : None
 RETURNS     : TRUE (1) if there was no file, or it was correct,
               FALSE (0) if the file CLIPSWIN.CFG had errors
 NOTES       : None
 ******************************************************************************/
static int load_configuration_file()
  {
   FILE *fp;
   char pathvar[20],pathassgn[5];
   int len;
   char cwd[MAXDRIVE+MAXDIR];
#if IBM_TBC
   char *cfgpath;
#else
   char cfgpath[MAXPATH];
#endif

   getcwd(cwd,MAXDRIVE+MAXDIR);
#if IBM_TBC
   cfgpath = searchpath("clipswin.cfg");
#else
   _searchenv("clipswin.cfg","PATH",cfgpath);
#endif
   if (cwd[(len = strlen(cwd))-1] != '\\')
      {
       cwd[len] = '\\';
       cwd[len+1] = EOS;
      }
#if IBM_TBC
   if (cfgpath == NULL)
#else
   if (cfgpath[0] == EOS)
#endif
     {
      strcpy(clipswinpath,cwd);
      sprintf(user_env_file,"%s%s",cwd,"CLIPSENV.DAT");
      sprintf(clpwin_help_file,"%s%s",cwd,"CLIPSWIN.HLP");
      return(TRUE);
     }
   if ((fp = fopen(cfgpath,"r")) == NULL)
     return(FALSE);
   if (fscanf(fp,"%s%s%s",pathvar,pathassgn,clipswinpath) != 3)
     return(FALSE);
   if ((strcmp(pathvar,"CLIPSPATH") == 0) ? (strcmp(pathassgn,"=") != 0) 
                                             : TRUE)
     return(FALSE);
   if (fscanf(fp,"%s%s%s",pathvar,pathassgn,user_env_file) != 3)
     return(FALSE);
   if ((strcmp(pathvar,"ENVPATH") == 0) ? (strcmp(pathassgn,"=") != 0) : TRUE)
     return(FALSE);
   if (fscanf(fp,"%s%s%s",pathvar,pathassgn,clpwin_help_file) != 3)
     return(FALSE);
   if ((strcmp(pathvar,"HELPPATH") == 0) ? (strcmp(pathassgn,"=") != 0) : TRUE)
     return(FALSE);
   if (strcmp(clipswinpath,"DEFAULT") == 0)
     strcpy(clipswinpath,cwd);
   if (strcmp(user_env_file,"DEFAULT") == 0)
     sprintf(user_env_file,"%s%s",cwd,"CLIPSENV.DAT");
   if (strcmp(clpwin_help_file,"DEFAULT") == 0)
     sprintf(clpwin_help_file,"%s%s",cwd,"CLIPSWIN.HLP");
   fclose(fp);
   return(TRUE);
  }

#else

/******************************************************************************
 NAME        : create_data_file
 PURPOSE     : Allocates full data space for non-existent data file
 DESCRIPTION : If the data file already exists, this function returns an error.
               Otherwise, it creates the data file with enough room for
               colors and key-bindings, and it writes the matchkey at the front.
 INPUTS      : 1) The name of the file
 RETURNS     : The file handle if the file was successfully created.
               BAD_DATA_FILE (-1) if the file already existed.
 NOTES       : None
 ******************************************************************************/
static int create_data_file(file)
  char *file;
  {
   int fhandle;
   int nbytes;
   char *tempbuf;

   if ((fhandle = open(file,O_RDWR|O_CREAT|O_EXCL|O_BINARY,S_IREAD|S_IWRITE))
     == -1)
     return(BAD_DATA_FILE);

   /* ==========================================
      Write the recognition createkey to the file
      ========================================== */
   if (write(fhandle,(char *) &createkey,sizeof(unsigned long))
               != sizeof(unsigned long))
     {
      close(fhandle);
      (void) unlink(file);
      return(BAD_DATA_FILE);
     }
   
   /* ==========================================
      Allocate enough space for all the data
      ========================================== */
   nbytes = ((MAX_COLOR_DEFS) * sizeof(ch_attr)) +
            ((MAX_CNTRL_KEYS + MAX_EXTND_KEYS) * sizeof(unsigned char));
   if ((tempbuf = balloc(nbytes,unsigned char)) == NULL)
     {
      close(fhandle);
      (void) unlink(file);
      return(BAD_DATA_FILE);
     }
   if (write(fhandle,tempbuf,nbytes) != nbytes)
     {
      release(nbytes,char,tempbuf);
      close(fhandle);
      (void) unlink(file);
      return(BAD_DATA_FILE);
     }     
   release(nbytes,char,tempbuf);
   if (lseek(fhandle,0L,SEEK_SET) != 0)
     {
      close(fhandle);
      return(BAD_DATA_FILE);
     }
   return(fhandle);
  }

#endif

/******************************************************************************
 NAME        : valid_file
 PURPOSE     : Checks the format of an input file
 DESCRIPTION : This routine checks to see if a file exists, and if it does,
               checks the matchkey and the filelength.  If any of these
               criteria are not satisfied, the routine return ERROR.
 INPUTS      : 1) The name of the file
               2) The key to look for
 RETURNS     : 1) The handle of file opened for read/write if all was OK,
               2) NO_DATA_FILE if the file did not exist.
               3) BAD_DATA_FILE if the file was not in the right format
 NOTES       : None
 ******************************************************************************/
static int valid_file(file,matchkey)
  char *file;
  unsigned long matchkey;
  {
   int fhandle;
   long nbytes;
   unsigned long key;

   if ((fhandle = open(file,O_RDWR|O_BINARY,S_IREAD|S_IWRITE)) == -1)
     return(NO_DATA_FILE);
   nbytes = sizeof(unsigned long) +
            ((MAX_COLOR_DEFS) * sizeof(ch_attr)) +
            ((MAX_CNTRL_KEYS + MAX_EXTND_KEYS) * sizeof(unsigned char));
   if (filelength(fhandle) != nbytes)
     {
      close(fhandle);
      return(BAD_DATA_FILE);
     }
   if (read(fhandle,(char *) &key,sizeof(unsigned long)) 
            != sizeof(unsigned long))
     {
      close(fhandle);
      return(BAD_DATA_FILE);
     }
   if (matchkey == anykey)
     {
      if ((key != validkey) && (key != colorkey) && (key != bindkey)
                            && (key != createkey))
        {
         close(fhandle);
         return(BAD_DATA_FILE);
        }
     }
   else if ((key & matchkey) != matchkey)
     {
      close(fhandle);
      return(BAD_DATA_FILE);
     }
   if (lseek(fhandle,0L,SEEK_SET) != 0)
     {
      close(fhandle);
      return(BAD_DATA_FILE);
     }
   return(fhandle);
  }
