/******************************************************************************
 FILE DIRMENU.C :
   This file contains the routines for menu-interfaces with MS-DOS directories.
 ******************************************************************************/

/******************************************************************************
 ==============================================================================
                                  HEADER FILES
 ==============================================================================
 ******************************************************************************/
#include "common.h"    /* Standard header files and useful constants/macros  */
#include "curses.h"    /* Aspen Scientific's Curses V4.0 Package Definitions */
#include "xcurses.h"
#include "box.h"
#include "curproto.h"
#define DIRMENU_SOURCE
#include "dirmenu.h"
#include "color.h"     /* Color definitions                                  */
#include "key.h"       /* Key-code and mapping definitions                   */
#include "env.h"       /* Disk Error Handler definitions                     */

#if MOUSE
#include "mouse.h"     /* Mouse Systems PC-Mouse Interface Routines Defns    */
#endif

/* --- DOS Header Files --- */

#if IBM_TBC
#include <io.h>
#include <dir.h>
#else
#include <direct.h>

#define MAXPATH  _MAX_PATH
#define MAXDRIVE _MAX_DRIVE
#define MAXDIR   _MAX_DIR
#define MAXFILE  _MAX_FNAME
#define MAXEXT   _MAX_EXT

#define FA_DIREC  _A_SUBDIR
#define FA_LABEL  _A_VOLID
#define FA_RDONLY _A_RDONLY

#define DRIVE     0x01
#define DIRECTORY 0x02
#define FILENAME  0x04
#define EXTENSION 0x08

#endif
#include <dos.h>

/* --- DOS Routines used in this file :

   Turbo-C : getdisk(),setdisk()
             getcwd()
             findfirst(),findnext()
             chdir()
             fnsplit()

   MSC     : _dos_getdrive(),_dos_setdrive()
             getcwd()
             _dos_findfirst(),_dos_findnext()
             chdir()
             _splitpath()

   -------------------------------------------- */

/******************************************************************************
 ==============================================================================
                                   CONSTANTS
 ==============================================================================
 ******************************************************************************/
#define DSK_LABEL_SIZE 11      /* Size of DOS Disk Volume Labels              */

#define DRV_SEPARATOR  ':'     /* Marker between drive and dirs in path       */
#define DIR_SEPARATOR  '\\'    /* Marker between directories in the path      */
#define EXT_SEPARATOR  '.'     /* Marker between filename and extension       */

#define ELIGIBLE_FILES FA_RDONLY|FA_DIREC

/* Display Window Diemnsions */

#define PROMPT_ROWS       1
#define PROMPT_OFFSET_Y   1
#define PROMPT_OFFSET_X   1
#define CWD_ROWS          3
#define CWD_COLS          (MAXFILE+MAXEXT+1)
#define CWD_OFFSET_Y      (PROMPT_OFFSET_Y+1)
#define CWD_OFFSET_X      1
#define FILE_ROWS         7
#define FILE_COLS         (MAXFILE+MAXEXT+1)
#define FILE_OFFSET_Y     (CWD_OFFSET_Y+CWD_ROWS+1)
#define FILE_OFFSET_X     1
#define SCROLL_ROWS       FILE_ROWS
#define SCROLL_COLS       3
#define SCROLL_OFFSET_Y   FILE_OFFSET_Y
#define SCROLL_OFFSET_X   (FILE_OFFSET_X+FILE_COLS-1)
#define NEW_FILE_ROWS     3
#define NEW_FILE_COLS     FILE_COLS
#define NEW_FILE_OFFSET_Y (FILE_OFFSET_Y+FILE_ROWS-1)
#define NEW_FILE_OFFSET_X FILE_OFFSET_X
#define DRIVE_ROWS        3
#define DRIVE_COLS        7
#define DRIVE_OFFSET_Y    CWD_OFFSET_Y
#define DRIVE_OFFSET_X    (FILE_OFFSET_X+FILE_COLS+SCROLL_COLS)
#define LABEL_ROWS        3
#define LABEL_COLS        (DSK_LABEL_SIZE+2)
#define LABEL_OFFSET_Y    DRIVE_OFFSET_Y
#define LABEL_OFFSET_X    (DRIVE_OFFSET_X+DRIVE_COLS+1)
#define HELP_ROWS         7
#define HELP_COLS         17
#define HELP_OFFSET_Y     FILE_OFFSET_Y
#define HELP_OFFSET_X     DRIVE_OFFSET_X
#define DISPLAY_ROWS      (PROMPT_ROWS+CWD_ROWS+FILE_ROWS+NEW_FILE_ROWS+2)
#define DISPLAY_COLS      (FILE_COLS+SCROLL_COLS+DRIVE_COLS+LABEL_COLS+3)

#define PROMPT_COLS       (DISPLAY_COLS-2)

#define CWD_ACTIVE_ROWS   (DISPLAY_ROWS-CWD_OFFSET_Y-1)

#define MOUSE_ROWS        (CWD_ROWS+FILE_ROWS+NEW_FILE_ROWS-2)
#define MOUSE_COLS        (CWD_COLS+SCROLL_COLS+DRIVE_COLS-2)
#define MOUSE_OFFSET_Y    (CWD_OFFSET_Y+1)
#define MOUSE_OFFSET_X    (CWD_OFFSET_X+1)

/* Mouse Cursor Definitions */

#define MSCURS_SCREEN_MASK 0x7000    /* Preserve bgd/fgd colors   */
#define MSCURS_CURSOR_MASK 0x0004    /* Diamond-Cursor            */

/* File-Menu Command Codes */

#define DIRS           0
#define FILES          1

#define NOTHING        0
#define SET_DRIVE      1 
#define SET_DIRECTORY  2
#define SET_FILE       3
#define CHOOSE         4
#define CANCEL         5
#define PG_UP          6
#define PG_DN          7
#define NEW_FILE       8

/******************************************************************************
 ==============================================================================
                                    MACROS
 ==============================================================================
 ******************************************************************************/
#define hide_cursor() mvcur(0,0,LINES,0)
#define ms_map(z,y,x) (*(mouse_map+(y+z*MOUSE_ROWS)*MOUSE_COLS+x))

#define nfdisp(win,off,x) (prefresh(win,0,off,disp_y+NEW_FILE_OFFSET_Y+1, \
                                    disp_x+NEW_FILE_OFFSET_X+1, \
                                    disp_y+NEW_FILE_OFFSET_Y+1, \
                                    disp_x+NEW_FILE_OFFSET_X+NEW_FILE_COLS-2), \
                           mvcur(0,0,disp_y+NEW_FILE_OFFSET_Y+1, \
                                 disp_x+NEW_FILE_OFFSET_X+1+ \
                                 ((x < NEW_FILE_COLS-2) ? x : NEW_FILE_COLS-3)))

/******************************************************************************
 ==============================================================================
                            DATA TYPES AND STRUCTURES
 ==============================================================================
 ******************************************************************************/

/* -- These nodes form a directory stack which represents the cwd --- */

typedef struct pathnode
  {
   char dir[MAXDIR+2];
   struct pathnode *nxt;
  } PATHNODE;

/* -- These nodes form a circularly-linked list of files in a directory --- */

typedef struct filenode
  {
   char file[MAXFILE+MAXEXT+1];
   char attrib;
   struct filenode *prv,*nxt;
  } FILENODE;

/******************************************************************************
 ==============================================================================
                      EXTERNALLY VISIBLE FUNCTION PROTOTYPES
 ==============================================================================
 ******************************************************************************/
char *pop_up_files(char *prompt,char *pattern,int (*valid)(char *),
                   unsigned char access);
                                        /* Displays a menu of files in CWD */

/******************************************************************************
 ==============================================================================
                      INTERNALLY VISIBLE FUNCTION PROTOTYPES
 ==============================================================================
 ******************************************************************************/
static void get_disk_label(void);       /* Finds current disk volume label */

static int  init_path_stack(void),      /* Builds directory stack from cwd */
            push_dir(char *dir);        /* Adds a directory to the stack   */
static void pop_dir(void),              /* Removes the top directory from
                                             the stack                     */
            deallocate_path_stack(void);/* Releases directory stack memory */

static int  init_file_list(char *pattern,int (*valid)(char *)),
                                         /* Builds list of files in CWD     */
            add_file(char *file,char attrib),
                                         /* Adds a file to the list         */
            compare(FILENODE *f1,FILENODE *f2);
                                         /* Determines file sorting order   */
static void deallocate_file_list(void);  /* Releases file list memory       */

static int  init_display(char *prompt,unsigned char access),
                                         /* Initializes base dialog-box     */
            draw_box_with_scroll_bars(WINDOW *win,char *label,
                                      int rows,int cols,int y,int x),
#if MOUSE
            init_mouse_map(unsigned char access),
                                         /* Inits mouse-coord map           */
#endif
                                         /* Draws a box on window w/ scroll */
            init_path_menu(void),        /* Initializes dir-stack menu      */
            init_file_menu(void);        /* Initializes file-list menu      */
static void deallocate_display(void),    /* Releases display memory         */
            deallocate_mouse_map(void),  /* Releases mouse map memory   */
            deallocate_path_menu(void),  /* Releases dir-stack menu memory  */
            deallocate_file_menu(void);  /* Releases file-list menu memory  */

static int  get_file(unsigned char access);
                                         /* File-menu selection loop routine*/
static void highlight(int menu,int flag,int select);
                                         /* (Un)highlights dir/files menus  */
static void show_path_menu(int active),  /* Displays directory-stack menu   */
            show_file_menu(void);        /* Diasplays file-list menu        */
static int  get_command(int active,int *select,int limit,int max_display,
                        unsigned char access);
                                         /* Gets user mouse/keyboard cmds   */
static int  change_disk(void);           /* Resets default disk drive       */
static int  change_directory(int select),/* Resets default directory        */
            select_file(int select,int *command,unsigned char access),
                                         /* Stores menu file selection      */
            new_file(unsigned char access);
                                         /* Lets user enter new file-name   */
static void grab_text(char *text,WINDOW *menu,int row);
                                         /* Grabs text from a menu          */
static int check_overwrite(char *fname); /* Checks for ok-write on file     */
static void adjust_new_file_path(void);  /* Checks path of new file-names   */

/******************************************************************************
 ==============================================================================
                      EXTERNALLY VISIBLE GLOBAL VARIABLES
 ==============================================================================
 ******************************************************************************/

/******************************************************************************
 ==============================================================================
                      INTERNALLY VISIBLE GLOBAL VARIABLES
 ==============================================================================
 ******************************************************************************/
static int cdsk;                    /* Current disk drive        */

static char label[DSK_LABEL_SIZE+1]; /* Current disk label        */
   
static char cwd[MAXDRIVE+MAXDIR+1],  /* Current working directory */
            file[MAXPATH+1];         /* File selected             */

static PATHNODE *path = NULL;        /* Directory Stack           */
static int dir_count = 0;            /* # of directories in stack */

static FILENODE *files = NULL;       /* File List                 */
static int file_count = 0;           /* # of files in list        */

static WINDOW *display = NULL,       /* Dialog-box window         */
              *path_menu = NULL,     /* Directory stack window    */
              *dir_box = NULL ,      /* Box for active dir-menu   */
              *file_menu = NULL;     /* File list window          */

static int disp_y,disp_x;    /* Starting screen coordinates of display */

static int path_offset = 0,   /* Number of dirs off top of menu         */
           file_offset = 0;   /* Number of files off top of menu       */

#if MOUSE
/* Mouse-coord action-code map */

unsigned char *mouse_map = NULL;
#endif

/******************************************************************************
 ==============================================================================
                          EXTERNALLY VISIBLE FUNCTIONS
 ==============================================================================
 ******************************************************************************/

/*****************************************************************************
 NAME        : pop_up_files
 PURPOSE     : Displays a menu of files and lets the user select one
 DESCRIPTION : 
 INPUTS      : 1) A prompt string for the file-menu
                  (The string have no imbedded carriage-returns and will be
                   truncated after 39 characters)
               2) A filename/extension wildcard pattern to select eligible files
                  This pattern must NOT contain any drive/directory information
                  or unpredictable results will occur.  If the pattern is NULL
                  or "*.*", all files will qualify.
               3) The address of a file-validating function.
                  This function must accept a single-string representing a
                  file-name and return  TRUE (1) if the file qualifies for
                  inclusion on the list or FALSE (0) otherwise.
                    If this address is NULL, all files that match the
                  pattern will qualify.
               4) A bit-flag indicating the mode of access desired for the
                  selected file - READ (1), WRITE (2), or READ/WRITE (3)
                  WRITE access will allow the user an option to enter a new
                  file-name for the cwd.  The routine will also perform a
                  double-check prompt when trying to overwrite a file that
                  already exists.
 RETURNS     : 1) The address of a static buffer allocated by this routine which
                  holds the file name
               2) NULL if the menu was cancelled or there was an error
 NOTES       : The address returned is the address of a static buffer,
               and it will be overwritten on each subsequent call.  If the
               caller wishes to preserve this string, it should copy it.
 *****************************************************************************/
char *pop_up_files(char *prompt,char *pattern,int (*valid)(char *),
                   unsigned char access)
  {
   int done = FALSE;

#if IBM_TBC
   cdsk = getdisk();
#else
   _dos_getdrive(&cdsk);
   cdsk--;
#endif
   get_disk_label();
   if (getcwd(cwd,MAXDRIVE+MAXDIR+1) == NULL)
     return(NULL);
   if (! init_display(prompt,access))
     return(NULL);
   file[0] = EOS;
   while (! done)
     {
      if ((dir_count = init_path_stack()) == ERROR)
        {
         deallocate_display();
         return(NULL);
        }
      if (! init_path_menu())
        {
         deallocate_path_stack();
         deallocate_display();
         return(NULL);
        }
      deallocate_path_stack();
      if ((file_count = init_file_list(pattern,valid)) == ERROR)
        {
         deallocate_display();
         deallocate_path_menu();
         return(NULL);
        }
      if (! init_file_menu())
        {
         deallocate_path_menu();
         deallocate_file_list();
         deallocate_display();
         return(NULL);
        }
      deallocate_file_list();
#if MOUSE
     if (! init_mouse_map(access))
       {
        deallocate_file_menu();
        deallocate_path_menu();
        deallocate_display();
        return(NULL);
       }
#endif

      done = get_file(access);

#if MOUSE
      deallocate_mouse_map();
#endif
      deallocate_file_menu();
      deallocate_path_menu();

     }
   deallocate_display();
   if (file[0] == EOS)
     return(NULL);
   return(file);
  }

/******************************************************************************
 ==============================================================================
                          INTERNALLY VISIBLE FUNCTIONS
 ==============================================================================
 ******************************************************************************/

/*****************************************************************************
 NAME        : get_disk_label
 PURPOSE     : Determines the volume label (if any) for the current disk drive
 DESCRIPTION : Method : Does a wildcard search of the root directory of the
               current disk for files with the attribute FA_LABEL
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Uses Turbo_c Library Routines findfirst() and findnext()
 *****************************************************************************/
static void get_disk_label()
  {
   char path[7];
   int done;
#if IBM_TBC
   struct ffblk fblock;

   sprintf(path,"%c:\\*.*",cdsk+'A');
   done = findfirst(path,&fblock,FA_LABEL);
   while (! done)
     {
      if (fblock.ff_attrib & FA_LABEL)
        {
         if (strlen(fblock.ff_name) < MAXFILE)
           strcpy(label,fblock.ff_name);
         else
           {
            (fblock.ff_name)[MAXFILE-1] = EOS;
            sprintf(label,"%s%s",fblock.ff_name,fblock.ff_name+MAXFILE);
            (fblock.ff_name)[MAXFILE] = EXT_SEPARATOR;
           }
         return;
        }
      done = findnext(&fblock);
     }
#else
   struct find_t fblock;

   sprintf(path,"%c:\\*.*",cdsk+'A');
   done = _dos_findfirst(path,_A_VOLID,&fblock);
   while (! done)
     {
      if (fblock.attrib & _A_VOLID)
        {
         if (strlen(fblock.name) < MAXFILE)
           strcpy(label,fblock.name);
         else
           {
            (fblock.name)[MAXFILE-1] = EOS;
            sprintf(label,"%s%s",fblock.name,fblock.name+MAXFILE);
            (fblock.name)[MAXFILE] = EXT_SEPARATOR;
           }
         return;
        }
      done = _dos_findnext(&fblock);
     }
#endif
   strcpy(label,"NONE");
  }

/*****************************************************************************
 NAME        : init_path_stack
 PURPOSE     : Allocates and initializes the directory stack
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : 1) The number of directories in the stack
               2) ERROR (-1) if there was an error.
 NOTES       : None
 *****************************************************************************/
static int init_path_stack()
  {
   register int i,j;
   int dcnt;

   dcnt = 0;
   for (i = 0 ; cwd[i] != DIR_SEPARATOR ; i++)
     { }
   for (j = i ; cwd[i] != EOS ; i++)
     {
      if (cwd[i] == DIR_SEPARATOR)
        {
         cwd[i] = EOS;
         if (! push_dir(cwd+j))
           {
            cwd[i] = DIR_SEPARATOR;
            return(ERROR);
           }
         cwd[i] = DIR_SEPARATOR;
         dcnt++;
         j = i+1;
        }
     }
   if (cwd[j] != EOS)
     {
      if (! push_dir(cwd+j))
        return(ERROR);
      dcnt++;
     }
   return(dcnt);
  }

/*****************************************************************************
 NAME        : push_dir
 PURPOSE     : Allocates space for and adds a directory to the stack
 DESCRIPTION : 
 INPUTS      : 1) The address of a string holding the directory to be added
 RETURNS     : TRUE (1) if everything ok, FALSE (0) otherwise
 NOTES       : This routine deallocates the path-stack if any push_dir() call
                 fails
 *****************************************************************************/
static int push_dir(char *dir)
  {
   PATHNODE *ptmp;

   if ((ptmp = balloc(1,PATHNODE)) == NULL)
     {
      deallocate_path_stack();
      return(FALSE);
     }
   (ptmp->dir)[0] = EOS;
   sprintf(ptmp->dir,"%c%s",DIR_SEPARATOR,dir);
   ptmp->nxt = path;
   path = ptmp;
   return(TRUE);
  }

/*****************************************************************************
 NAME        : pop_dir
 PURPOSE     : Removes and deallocates the directory at the top of the stack
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 *****************************************************************************/
static void pop_dir()
  {
   PATHNODE *ptmp;

   if (path != NULL)
     {
      ptmp = path;
      path = path->nxt;
      release(1,PATHNODE,ptmp);
     }
  }

/*****************************************************************************
 NAME        : deallocate_path_stack
 PURPOSE     : Releases any memory associated with the directory stack
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 *****************************************************************************/
static void deallocate_path_stack()
  {
   PATHNODE *ptmp;

   while (path != NULL)
     {
      ptmp = path;
      path = path->nxt;
      release(1,PATHNODE,ptmp);
     }
  }

/*****************************************************************************
 NAME        : init_file_list
 PURPOSE     : Builds a singly-linked list of files for the cwd
 DESCRIPTION : 
 INPUTS      : 1) The filename/extension pattern to search for (see above)
               2) The address of a file-validating function (see above)
 RETURNS     : 1) The number of files in the list
               2) or ERROR (-1) if there was an error
 NOTES       : Uses the Turbo-C Library Routines findfirst(), findnext()
 *****************************************************************************/
static int init_file_list(char *pattern,int (*valid)(char *))
  {
   register int done;
   unsigned char mode = ELIGIBLE_FILES;
   int fcnt;
#if IBM_TBC
   char path[MAXPATH+1];
   struct ffblk fblock;

   if (cwd[strlen(cwd)-1] != DIR_SEPARATOR)
     sprintf(path,"%s%c%s",cwd,DIR_SEPARATOR,
                               (pattern != NULL) ? pattern : "*.*");
   else
     sprintf(path,"%s%s",cwd,(pattern != NULL) ? pattern : "*.*");
   done = findfirst(path,&fblock,mode);
   fcnt = 0;
   while (! done)
     {
      if ((strcmp(fblock.ff_name,".") != 0) ?
          (strcmp(fblock.ff_name,"..") != 0) : FALSE)
        {
         if ((fblock.ff_attrib & FA_DIREC) ? TRUE :
             ((valid == NULL) ? TRUE : valid(fblock.ff_name)))
           {
            if (! add_file(fblock.ff_name,fblock.ff_attrib))
              return(ERROR);
            fcnt++;
           }
        }
      done = findnext(&fblock);
     }
#else
   char path[MAXPATH+1];
   struct find_t fblock;

   if (cwd[strlen(cwd)-1] != DIR_SEPARATOR)
     sprintf(path,"%s%c%s",cwd,DIR_SEPARATOR,
                               (pattern != NULL) ? pattern : "*.*");
   else
     sprintf(path,"%s%s",cwd,(pattern != NULL) ? pattern : "*.*");
   done = _dos_findfirst(path,mode,&fblock);
   fcnt = 0;
   while (! done)
     {
      if ((strcmp(fblock.name,".") != 0) ?
          (strcmp(fblock.name,"..") != 0) : FALSE)
        {
         if ((fblock.attrib & _A_SUBDIR) ? TRUE :
             ((valid == NULL) ? TRUE : valid(fblock.name)))
           {
            if (! add_file(fblock.name,fblock.attrib))
              return(ERROR);
            fcnt++;
           }
        }
      done = _dos_findnext(&fblock);
     }
#endif
   if (fcnt == 0)
     {
      if (! add_file("",0))
        return(ERROR);
     }
   return(fcnt);
  }

/*****************************************************************************
 NAME        : add_file
 PURPOSE     : Adds a file to the linked list for the cwd
 DESCRIPTION : 
 INPUTS      : 1) The file to be added
               2) The attributes of the file
 RETURNS     : TRUE (1) if all went well, FALSE (0)
 NOTES       : If there is an error, this routine calls deallocate_file_list().
               The all entries are inserted in a sorted order :
                 1) All directories first
                    a) common extensions grouped together in alphabetical order
                    b) extensions alphabetized
                 2) Rest of files grouped by extensions in alphabetical order
                 3) Extensions are alphabetized
 *****************************************************************************/
static int add_file(char *file,char attrib)
  {
   FILENODE *ftmp,*fptr;
   register int val;

   if ((ftmp = balloc(1,FILENODE)) == NULL)
     {
      deallocate_file_list();
      return(FALSE);
     }
   (ftmp->file)[0] = EOS;
#if IBM_TBC
   if (attrib & FA_DIREC)
#else
   if (attrib & _A_SUBDIR)
#endif
     sprintf(ftmp->file,"%c%s",DIR_SEPARATOR,file);
   else
     strcpy(ftmp->file,file);
   ftmp->attrib = attrib;
   ftmp->prv = ftmp;
   ftmp->nxt = ftmp;
   if (files == NULL)
     files = ftmp;
   else
     {
      if (compare(ftmp,files) < 0)
        {
         files->prv->nxt = ftmp;
         ftmp->prv = files->prv;
         ftmp->nxt = files;
         files->prv = ftmp;
         files = ftmp;
        }
      else
        {
         fptr = files->nxt;
         while ((fptr != files) ? (compare(ftmp,fptr) > 0) : FALSE)
           fptr = fptr->nxt;
         fptr->prv->nxt = ftmp;
         ftmp->prv = fptr->prv;
         ftmp->nxt = fptr;
         fptr->prv = ftmp;
        }
     }
   return(TRUE);
  }

/*****************************************************************************
 NAME        : compare
 PURPOSE     : Returns an integer code indicating whether one filenode
               should precede another in a sorted list or not
 DESCRIPTION : The ordering is made by the following rules :
                 1) Directories precede regular files
                 2) Files of the same type (directory/regular) are
                    grouped by their 3 letter extension
                 3) Files are alphabetized by their 3-letter extension first
                 3) Files within an extension group are alphabetized by their
                    filename.
 INPUTS      : 1) The address of the first file-list node
               2) The address of the second file-list node
 RETURNS     : 1) < 0 if f1 should precede f2
               2)   0 if f1 and f2 are equivalent
               3) > 0 if f1 should succeed f2
 NOTES       : None
 *****************************************************************************/
static int compare(FILENODE *f1,FILENODE *f2)
  {
   char *ext1,*ext2,
        sv1,sv2;
   int val;

   if ((f1->attrib & FA_DIREC) && (! (f2->attrib & FA_DIREC)))
     val = -1;
   else if ((! (f1->attrib & FA_DIREC)) && (f2->attrib & FA_DIREC))
     val = 1;
   else
     {
      for (ext1 = f1->file ; ((*ext1 != EOS) ? 
                              (*ext1 != EXT_SEPARATOR) : FALSE) ; ext1++) ;
      for (ext2 = f2->file ; ((*ext2 != EOS) ? 
                              (*ext2 != EXT_SEPARATOR) : FALSE) ; ext2++) ;
      if ((val = strcmp(ext1,ext2)) == 0)
        {
         sv1 = *ext1;
         *ext1 = EOS;
         sv2 = *ext2;
         *ext2 = EOS;
         val = strcmp(f1->file,f2->file);
         *ext1 = sv1;
         *ext2 = sv2;
        }
     }
   return(val);
  }

/*****************************************************************************
 NAME        : deallocate_file_list
 PURPOSE     : Release all memory associated with the file list for the cwd
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 *****************************************************************************/
static void deallocate_file_list()
  {
   FILENODE *ftmp1,*ftmp2;

   if (files != NULL)
     {
      files->prv->nxt = NULL;
      files->prv = NULL;
      ftmp1 = files;
      while (ftmp1 != NULL)
        {
         ftmp2 = ftmp1;
         ftmp1 = ftmp1->nxt;
         release(1,FILENODE,ftmp2);
        }
     }
   files = NULL;
  }

/*****************************************************************************
 NAME        : init_display
 PURPOSE     : Creates and draws base dialog-box for file-access menus
 DESCRIPTION : 
 INPUTS      : 1) A prompt-string for the file-menu
               2) Access mode : READ and/or WRITE
 RETURNS     : TRUE (1) if everything went ok, FALSE (0) otherwise
 NOTES       : None
 *****************************************************************************/
static int init_display(char *prompt,unsigned char access)
  {
   WINDOW *area = NULL;

   disp_y = LINES/2-DISPLAY_ROWS/2;
   disp_x = COLS/2-DISPLAY_COLS/2;
   if ((display = newwin(DISPLAY_ROWS,DISPLAY_COLS,disp_y,disp_x)) == NULL)
     return(FALSE);
   wattrset(display,colors[POP_FILE_TEXT]|colors[POP_FILE_BGD]);
   keypad(display,TRUE);
   nodelay(display,TRUE);
   xpaint(display,colors[POP_FILE_TEXT]|colors[POP_FILE_BGD]);
   xbox(display,DB_V_LIN,DB_H_LIN,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
   if (strlen(prompt) > PROMPT_COLS)
     prompt[PROMPT_COLS] = EOS;
   mvwaddstr(display,PROMPT_OFFSET_Y,PROMPT_OFFSET_X,prompt);
   if ((area = subwin(display,CWD_ROWS,CWD_COLS,
                      disp_y+CWD_OFFSET_Y,disp_x+CWD_OFFSET_X)) == NULL)
     return(FALSE);
   xbox(area,SG_V_LIN,SG_H_LIN,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
   wattrset(area,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
   mvwaddstr(area,0,CWD_COLS/2-4,"DIRECTORY");
   delwin(area);
   if (! draw_box_with_scroll_bars(display,"FILES",
                                   FILE_ROWS,FILE_COLS,
                                   disp_y+FILE_OFFSET_Y,disp_x+FILE_OFFSET_X))
     return(FALSE);
   if (access & WRITE)
     {
      if ((area = subwin(display,NEW_FILE_ROWS,NEW_FILE_COLS,
                   disp_y+NEW_FILE_OFFSET_Y,disp_x+NEW_FILE_OFFSET_X)) == NULL)
        return(FALSE);
      xbox(area,SG_V_LIN,SG_H_LIN,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
      wattrset(area,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
      mvwaddch(area,0,0,SGSG_LT_T);
      mvwaddch(area,0,NEW_FILE_COLS-1,SGSG_X);
      wattrset(area,colors[POP_FILE_TEXT]|colors[POP_FILE_BGD]);
      mvwaddstr(area,1,3,"NEW FILE");
      delwin(area);
     }
   if ((area = subwin(display,DRIVE_ROWS,DRIVE_COLS,
                      disp_y+DRIVE_OFFSET_Y,disp_x+DRIVE_OFFSET_X)) == NULL)
     return(FALSE);
   xbox(area,SG_V_LIN,SG_H_LIN,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
   wattrset(area,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
   mvwaddstr(area,0,1,"DRIVE");
   wattrset(area,colors[POP_FILE_TEXT]|colors[POP_FILE_BGD]);
   mvwprintw(area,1,3,"%c:",cdsk+'A');
   delwin(area);
   if ((area = subwin(display,LABEL_ROWS,LABEL_COLS,
                      disp_y+LABEL_OFFSET_Y,disp_x+LABEL_OFFSET_X)) == NULL)
     return(FALSE);
   xbox(area,SG_V_LIN,SG_H_LIN,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
   wattrset(area,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
   mvwaddstr(area,0,LABEL_COLS/2-3,"VOLUME");
   wattrset(area,colors[POP_FILE_TEXT]|colors[POP_FILE_BGD]);
   mvwaddstr(area,1,LABEL_COLS/2-strlen(label)/2,label);
   delwin(area);
   if ((area = subwin(display,HELP_ROWS,HELP_COLS,
                      disp_y+HELP_OFFSET_Y,disp_x+HELP_OFFSET_X)) == NULL)
     return(FALSE);
   wattrset(area,colors[POP_FILE_TEXT]|colors[POP_FILE_BGD]);
   mvwaddstr(area,0,0,"<RTN>-SELECT     ");
   wmove(area,1,0);
   xwaddch(area,'<'); xwaddch(area,ACS_UARROW);
   xwaddch(area,ACS_DARROW); xwaddch(area,'>');
   waddstr(area," -MOVE       ");
   mvwaddstr(area,2,0,"<ESC>-CANCEL     ");
   mvwaddstr(area,3,0,"<F1> -DRIVE      ");
   mvwaddstr(area,4,0,"<F2> -DIRECTORIES");
   mvwaddstr(area,5,0,"<F3> -OLD FILES  ");
   if (access & WRITE)
     mvwaddstr(area,6,0,"<F4> -NEW FILE   ");
   delwin(area);
   wattrset(display,colors[POP_FILE_TEXT]|colors[POP_FILE_BGD]);
#if MOUSE
   save_mouse();
   mouse_region(disp_y+MOUSE_OFFSET_Y,disp_x+MOUSE_OFFSET_X,
                disp_y+MOUSE_OFFSET_Y+MOUSE_ROWS-1,
                disp_x+MOUSE_OFFSET_X+MOUSE_COLS-1);
   mouse_cursor(SOFTWARE_CURSOR,MSCURS_SCREEN_MASK,
                                MSCURS_CURSOR_MASK | colors[POP_FILE_MOUSE]);
#endif
   touchwin(display);
   wrefresh(display);
   hide_cursor();
   return(TRUE);
  }

/*****************************************************************************
 NAME        : draw_box_with_scroll_bars
 PURPOSE     : Draws a box on a window which has scroll bars to the side
 DESCRIPTION : 
 INPUTS      : 1) The address of the window on which the box is to be drawn
               2) A string-label for the box
               3) Rows in the box (including borders)
               4) Cols in the box (including borders, excluding scroll-boxes)
               5) Absolute screen y-start
               6) Absolute screen x-start
 RETURNS     : TRUE (1) if everything went ok, FALSE (0) otherwise.
 NOTES       : The window being drawn on must have at least the following
               dimensions : <rows> rows by <cols+SCROLL_COLS(3)-1> columns
               If <rows> < 5, only the box/label and not the scroll bars 
               will be drawn.
 *****************************************************************************/
static int draw_box_with_scroll_bars(WINDOW *win,char *label,
                                     int rows,int cols,int y,int x)
  {
   WINDOW *area = NULL;
   register int i;

   if ((area = subwin(win,rows,cols+SCROLL_COLS-1,y,x)) == NULL)
     return(FALSE);
   xpaint(area,colors[POP_FILE_TEXT]|colors[POP_FILE_BGD]);
   xbox(area,SG_V_LIN,SG_H_LIN,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
   wattrset(area,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
   mvwaddstr(area,0,cols/2-strlen(label)/2,label);
   if (rows >= 5)
     {
      mvwaddch(area,0,cols-1,SGSG_UP_T);
      mvwaddch(area,1,cols-1,SG_V_LIN);
      mvwaddch(area,2,cols-1,SGSG_LT_T);
      mvwaddch(area,2,cols,SG_H_LIN);
      mvwaddch(area,2,cols+1,SGSG_RT_T);
      for (i = 3 ; i < rows-3 ; i++)
        mvwaddch(area,i,cols-1,SG_V_LIN);
      mvwaddch(area,rows-3,cols-1,SGSG_LT_T);
      mvwaddch(area,rows-3,cols,SG_H_LIN);
      mvwaddch(area,rows-3,cols+1,SGSG_RT_T);
      mvwaddch(area,rows-2,cols-1,SG_V_LIN);
      mvwaddch(area,rows-1,cols-1,SGSG_LO_T);
      wattrset(area,flip_fgd_bgd(colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]));
     }
   else if (rows == 4)
     {
      mvwaddch(area,0,cols-1,SGSG_UP_T);
      mvwaddch(area,1,cols-1,SG_V_LIN);
      mvwaddch(area,2,cols-1,SG_V_LIN);
      mvwaddch(area,3,cols-1,SGSG_LO_T);
      wattrset(area,flip_fgd_bgd(colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]));
     }
   delwin(area);
   return(TRUE);
  }

#if MOUSE
/*****************************************************************************
 NAME        : init_mouse_map
 PURPOSE     : Initializes moues-input y,x map
 DESCRIPTION : The mouse is constrained to a region MOUSE_ROWS by MOUSE_COLS
               on the display window.  During execution, the display is in 1
               of two states : DIRS (0) - the directory-stack menu is active,
               FILES (1) - the file-list menu is active.
                 Depending on which state is current, mouse-input at particular
               coordinates on the display have different meanings.  The map
               defined below contains action-codes corresponding to mouse
               movement/clicks at a particular position in the window.  The
               codes are as follows :
                 NOTHING         (0)  - No functionality
                 SET_DRIVE       (1)  - Set new disk drive
                 SET_DIRECTORY   (2)  - Activate directory-stack menu
                 SET_FILE        (3)  - Activate file-list menu
                 CHOOSE          (5)  - Menu selection
                 PG_UP           (6)  - Scroll menu up
                 PG_DN           (7)  - Scroll menu down
                 NEW_FILE        (8)  - Pick new file for cwd
               The map contains two entries for each possible mouse-posn
                 in the window (1 for each display-state).
               The action-code for a mouse-posn in a particular display state
                 can be accessed by the ms_map() macro (defined above).
                 
                 Ex : Action-Code for mouse-position (y = 5, x = 6) in
                        files-menu active state :

                      code = ms_map(FILES,5,6) -->
                             (*mouse_map((5+1*MOUSE_ROWS)*MOUSE_COLS+6))
 INPUTS      : 1) Access mode : READ and/or WRITE
 RETURNS     : TRUE (1) if everything went ok, FALSE (0) otherwise
 NOTES       : The map is allocated contiguously with one malloc() call
               to minimize waste due to paragraph alignment
 *****************************************************************************/
static int init_mouse_map(unsigned char access)
  {
   register int i,j;
   int dir_rows;

   if (! MOUSE_LOADED)
     return(TRUE);
   if ((mouse_map = balloc(2*MOUSE_ROWS*MOUSE_COLS,unsigned char)) == NULL)
     return(FALSE);
   for (i = 0 ; i < 2*MOUSE_ROWS*MOUSE_COLS ; i++)
     *(mouse_map+i) = NOTHING;
   for (i = DRIVE_OFFSET_X-1 ; i < DRIVE_OFFSET_X+DRIVE_COLS-3 ; i++)
     {
      ms_map(DIRS,0,i) = SET_DRIVE;
      ms_map(FILES,0,i) = SET_DRIVE;
     }
   dir_rows = ((dir_count+2) < CWD_ACTIVE_ROWS) ? dir_count+2 : CWD_ACTIVE_ROWS;
   for (i = 0 ; i < CWD_COLS-2 ; i++)
     {
      for (j = 0 ; j < dir_rows-2 ; j++)
        ms_map(DIRS,j,i) = CHOOSE;
      for (j = dir_rows-1 ; j < FILE_OFFSET_Y-MOUSE_OFFSET_Y+FILE_ROWS-1 ; j++)
        ms_map(DIRS,j,i) = SET_FILE;
     }
   if (dir_count > (CWD_ACTIVE_ROWS-2))
     {
      ms_map(DIRS,0,CWD_COLS-1) = PG_UP;
      ms_map(DIRS,dir_rows-3,CWD_COLS-1) = PG_DN;
     }

   for (i = 0 ; i < FILE_COLS-2 ; i++)
     {
      ms_map(FILES,0,i) = SET_DIRECTORY;
      for (j = FILE_OFFSET_Y-MOUSE_OFFSET_Y+1 ; 
           j < FILE_OFFSET_Y-MOUSE_OFFSET_Y+FILE_ROWS-1 ; j++)
        {
         if ((j-(FILE_OFFSET_Y-MOUSE_OFFSET_Y+1)) < file_count)
           ms_map(FILES,j,i) = CHOOSE;
        }
     }
   if (access & WRITE)
     {
      for (i = 0 ; i < FILE_COLS-2 ; i++)
        ms_map(FILES,NEW_FILE_OFFSET_Y-MOUSE_OFFSET_Y+1,i) = NEW_FILE;
     }
   if (file_count > (FILE_ROWS-2))
     {
      ms_map(FILES,FILE_OFFSET_Y-MOUSE_OFFSET_Y+1,FILE_COLS-1) = PG_UP;
      ms_map(FILES,FILE_OFFSET_Y-MOUSE_OFFSET_Y+FILE_ROWS-2,FILE_COLS-1) = 
             PG_DN;
     }
   return(TRUE);
  }
#endif

/*****************************************************************************
 NAME        : deallocate_display
 PURPOSE     : Releases dialog-display window memory
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 *****************************************************************************/
static void deallocate_display()
  {
   if (display != NULL)
     {
      delwin(display);
      display = NULL;
     }
#if MOUSE
   restore_mouse();
#endif
  }

#if MOUSE
/*****************************************************************************
 NAME        : deallocate_mouse_map
 PURPOSE     : Releases memory for mouse-input map
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 *****************************************************************************/
static void deallocate_mouse_map()
  {
   if (mouse_map != NULL)
     {
      release(2*MOUSE_ROWS*MOUSE_COLS,unsigned char,mouse_map);
      mouse_map = NULL;
     }
  }
#endif

/*****************************************************************************
 NAME        : init_path_menu
 PURPOSE     : Creates and draws the directory-stack menu
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : TRUE (1) if everything went ok, FALSE (0) otherwise
 NOTES       : The path-menu is a Curses pad
 *****************************************************************************/
static int init_path_menu()
  {
   PATHNODE *ptmp;
   register int i;

   if (dir_count != 0)
     {
      if ((path_menu = newpad(dir_count,CWD_COLS-2)) == NULL)
        return(FALSE);
      xpaint(path_menu,colors[POP_FILE_TEXT]|colors[POP_FILE_BGD]);
      wattrset(path_menu,colors[POP_FILE_TEXT]|colors[POP_FILE_BGD]);
      for (ptmp = path , i = 0 ; ptmp != NULL ; ptmp = ptmp->nxt , i++)
        mvwaddstr(path_menu,i,0,ptmp->dir);
     }
   return(TRUE);
  }

/*****************************************************************************
 NAME        : deallocate_path_menu
 PURPOSE     : Releases path-menu memory
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 *****************************************************************************/
static void deallocate_path_menu()
  {
   if (path_menu != NULL)
     {
      delwin(path_menu);
      path_menu = NULL;
     }
  }

/*****************************************************************************
 NAME        : init_file_menu
 PURPOSE     : Creates and draws the file-list menu
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : TRUE (1) if everything went ok, FALSE (0) otherwise
 NOTES       : The file-list menu is a Curses pad
 *****************************************************************************/
static int init_file_menu()
  {
   FILENODE *ftmp;
   register int i;

   if (file_count != 0)
    {
     if ((file_menu = newpad(file_count,FILE_COLS-2)) == NULL)
       return(FALSE);
     xpaint(file_menu,colors[POP_FILE_TEXT]|colors[POP_FILE_BGD]);
     wattrset(file_menu,colors[POP_FILE_TEXT]|colors[POP_FILE_BGD]);
     mvwaddstr(file_menu,0,0,files->file);
     for (ftmp = files->nxt , i = 1 ; ftmp != files ; ftmp = ftmp->nxt , i++)
       mvwaddstr(file_menu,i,0,ftmp->file);
     }
   else if ((file_menu = newpad(1,1)) != NULL)
     xpaint(file_menu,colors[POP_FILE_TEXT]|colors[POP_FILE_BGD]);
   else
     return(FALSE);
   wattrset(display,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
   wmove(display,FILE_OFFSET_Y+FILE_ROWS-1,FILE_OFFSET_X+1);
   for (i = 0 ; i < FILE_COLS-2 ; i++)
     waddch(display,SG_H_LIN);
   mvwprintw(display,FILE_OFFSET_Y+FILE_ROWS-1,FILE_OFFSET_X+FILE_COLS/2-4,
                     "%d File(s)",file_count);
   touchwin(display);
   wrefresh(display);
   hide_cursor();
   return(TRUE);
  }

/*****************************************************************************
 NAME        : deallocate_file_menu
 PURPOSE     : Releases file-menu memory
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 *****************************************************************************/
static void deallocate_file_menu()
  {
   if (file_menu != NULL)
     {
      delwin(file_menu);
      file_menu = NULL;
     }
  }


/*****************************************************************************
 NAME        : get_file
 PURPOSE     : Displays directory-stack menu and file-menu on screen and
               lets user select a file
 DESCRIPTION : 
 INPUTS      : 1) The type of access for the file : READ and/or WRITE
 RETURNS     : 1) TRUE (1) if all was successful and a file was selected
               2) FALSE (0) if all was successful but the CWD has changed
                             (see below)
               3) ERROR (-1) if there was an error in the routine
 NOTES       : If this routine returns FALSE, it has had the side effects of
               changing the current-working directory and possibly the default
               disk drive as well.  This means the parameters need to be
               recalculated and this routine should be called again.
 *****************************************************************************/
static int get_file(unsigned char access)
  {
   int done = FALSE;
   int active_menu = FILES,
       file_select = 0,
       dir_select = 0;
   int command,drows;

   if (dir_count > 1)
     {
      drows = (dir_count+2 < CWD_ACTIVE_ROWS) ? dir_count+2 : CWD_ACTIVE_ROWS;
      if ((dir_box = newwin(drows,CWD_COLS+SCROLL_COLS-1,
                            disp_y+CWD_OFFSET_Y,disp_x+CWD_OFFSET_X)) == NULL)
        return(ERROR);
      if (! draw_box_with_scroll_bars(dir_box,"DIRECTORY",drows,CWD_COLS,
                                      disp_y+CWD_OFFSET_Y,disp_x+CWD_OFFSET_X))
        {
         delwin(dir_box);
         return(ERROR);
        }
     }
   show_path_menu(FALSE);
   if (file_count != 0)
     highlight(FILES,ON,file_select);
   show_file_menu();
   doupdate();
   hide_cursor();
   while (! done)
     {
      if (active_menu == DIRS)
        command = get_command(active_menu,&dir_select,
                              dir_count,drows-2,access);
      else
        command = get_command(active_menu,&file_select,
                              file_count,FILE_ROWS-2,access);
      if (command == SET_DRIVE)
        done = change_disk();
      else if (command == SET_DIRECTORY)
        {
         if (dir_count > 1)
           {
            active_menu = DIRS;
            highlight(FILES,OFF,file_select);
            show_file_menu();
            touchwin(dir_box);
            wnoutrefresh(dir_box);
            highlight(DIRS,ON,dir_select);
            show_path_menu(TRUE);
            doupdate();
            hide_cursor();
           }
         else
           beep();
        }
      else if (command == SET_FILE)
        {
         active_menu = FILES;
         highlight(DIRS,OFF,dir_select);
         touchwin(display);
         wnoutrefresh(display);
         show_path_menu(FALSE);
         highlight(FILES,ON,file_select);
         show_file_menu();
         doupdate();
         hide_cursor();
        }
      else if (command == CHOOSE)
        {
         if (active_menu == DIRS)
           {
            if (dir_count > 1)
              {
               done = change_directory(dir_select);
               command = SET_DIRECTORY;
              }
           }
         else if (file_count > 0)
           done = select_file(file_select,&command,access);
        }
      else if (command == NEW_FILE)
        done = new_file(access);
      else
        done = TRUE;
     }
   if (dir_box != NULL)
     delwin(dir_box);
   dir_box = NULL;
   touchwin(display);
   wnoutrefresh(display);
   return((command == CHOOSE) || (command == CANCEL) || (command == NEW_FILE));
  }

/*****************************************************************************
 NAME        : highlight
 PURPOSE     : (un)highlights selections in the files/directory menus
 DESCRIPTION : 
 INPUTS      : 1) A code indicating which menu pad is being accessed
                  DIRS (0) - Directory Stack Menu
                  FILES (1) - Files Menu
               2) A flag indicating whether to highlight or unhighlight
                  OFF (0) - Unhighlight
                  ON (1) - Highlight
               3) The index of the selection in the menu to be (un)highlighted
                  Directory Stack - [0..(max_directories-1)]
                  Files - [0..(max_files-1)]
 RETURNS     : Nothing useful
 NOTES       : Assumes the selection index is valid and that the Curses pads
               path_menu and file_menu have already been successfully allocated
 *****************************************************************************/
static void highlight(int menu,int flag,int select)
  {
   ch_attr attr;
   register int i;

   attr = colors[POP_FILE_TEXT]|colors[POP_FILE_BGD];
   if (flag == ON)
     attr = flip_fgd_bgd(attr);
   if (menu == DIRS)
     {
      wattrset(path_menu,attr);
      for (i = 0 ; i < CWD_COLS-2 ; i++)
        waddch(path_menu,mvwinch(path_menu,select,i) & A_CHARTEXT);
      if (select < path_offset)
        path_offset = select;
      else if (select > path_offset + CWD_ACTIVE_ROWS-3)
        path_offset = select-(CWD_ACTIVE_ROWS-3);
     }
   else
     {
      wattrset(file_menu,attr);
      for (i = 0 ; i < FILE_COLS-2 ; i++)
        waddch(file_menu,mvwinch(file_menu,select,i) & A_CHARTEXT);
      if (select < file_offset)
        file_offset = select;
      else if (select > file_offset + FILE_ROWS-3)
        file_offset = select-(FILE_ROWS-3);
     }
  }

/*****************************************************************************
 NAME        : show_path_menu
 PURPOSE     : Displays the directory-stack menu
 DESCRIPTION : 
 INPUTS      : 1) A flag indicating whether the path-menu is active or not
 RETURNS     : Nothing useful
 NOTES       : None
 *****************************************************************************/
static void show_path_menu(int active)
  {
   int offset,rows;

   if (active)
     {
      offset = path_offset;
      if ((dir_count+2) < CWD_ACTIVE_ROWS)
        rows = dir_count;
      else
        {
         rows = CWD_ACTIVE_ROWS-2;
         wattrset(dir_box,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
         wmove(dir_box,1,SCROLL_OFFSET_X-FILE_OFFSET_X+1);
         xwaddch(dir_box,(path_offset != 0) ? ACS_UARROW : BLANK);
         wmove(dir_box,CWD_ACTIVE_ROWS-2,SCROLL_OFFSET_X-FILE_OFFSET_X+1);
         xwaddch(dir_box,((dir_count-path_offset) > (CWD_ACTIVE_ROWS-2)) 
                          ? ACS_DARROW : BLANK);
         wnoutrefresh(dir_box);
        }
     }
   else
     {
      offset = 0;
      rows = 1;
     }
   touchwin(path_menu);
   pnoutrefresh(path_menu,offset,0,
                disp_y+CWD_OFFSET_Y+1,disp_x+CWD_OFFSET_X+1,
                disp_y+CWD_OFFSET_Y+rows,
                disp_x+CWD_OFFSET_X+CWD_COLS-2);
  }

/*****************************************************************************
 NAME        : show_file_menu
 PURPOSE     : Displays the file-list menu
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 *****************************************************************************/
static void show_file_menu()
  {
   wattrset(display,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
   wmove(display,FILE_OFFSET_Y+1,SCROLL_OFFSET_X+1);
   xwaddch(display,(file_offset != 0) ? ACS_UARROW : BLANK);
   wmove(display,FILE_OFFSET_Y+FILE_ROWS-2,SCROLL_OFFSET_X+1);
   xwaddch(display,((file_count-file_offset) > (FILE_ROWS-2)) 
                   ? ACS_DARROW : BLANK);
   wnoutrefresh(display);
   touchwin(file_menu);
   pnoutrefresh(file_menu,file_offset,0,
                disp_y+FILE_OFFSET_Y+1,disp_x+FILE_OFFSET_X+1,
                disp_y+FILE_OFFSET_Y+FILE_ROWS-2,
                disp_x+FILE_OFFSET_X+FILE_COLS-2);
  }

/*****************************************************************************
 NAME        : get_command
 PURPOSE     : Determines user commands for file-menu from mouse-input and
               keystrokes
 DESCRIPTION : This routine handles the highlighting of new selections
 INPUTS      : 1) A code indicating the current active-menu
                  DIRS (0) - Directory Stack Menu
                  FILES (1) - Files Menu
               2) The address where the selection index for the active-menu
                  is currently stored
               3) The maximum number of items in the menu
               4) The maximum number of items that can be displayed at one time
               5) Access mode : READ and/or WRITE
RETURNS     : One of the following command codes :
                 SET_DRIVE     (1) - User wants to change the disk drive
                 SET_DIRECTORY (2) - User wants to activate directory menu
                 SET_FILE      (3) - User wants to activate file menu
                 CHOOSE        (4) - Choose currently selected directory/file
                 CANCEL        (5) - Quit
                 NEW_FILE      (8) - Enter a new file for the cwd
 NOTES       : None
 *****************************************************************************/
static int get_command(int active,int *select,int limit,int max_display,
                       unsigned char access)
  {
   register int done = FALSE;
   int cmd,ch,kch,
       new_select;
#if MOUSE
   int y,x;
#endif

#if MOUSE
   show_mouse();
#endif
   new_select = *select;
   while (! done)
     {
      if (new_select != *select)
        {
         highlight(active,OFF,*select);
         highlight(active,ON,new_select);
         *select = new_select;
         if (active == DIRS)
           show_path_menu(TRUE);
         else
           show_file_menu();
#if MOUSE
         hide_mouse();
#endif
         doupdate();
         hide_cursor();
#if MOUSE
         show_mouse();
#endif
        }
      if ((ch = wgetch(display)) != -1)
        {
         if (ch == KEY_F(1))
           {
            cmd = SET_DRIVE;
            done = TRUE;
           }
         else if (ch == KEY_F(2))
           {
            if (active != DIRS)
              {
               cmd = SET_DIRECTORY;
               done = TRUE;
              }
            else
              beep();
           }
         else if (ch == KEY_F(3))
           {
            if (active != FILES)
              {
               cmd = SET_FILE;
               done = TRUE;
              }
            else
              beep();
           }
         else if (ch == KEY_F(4))
           {
            if ((active == FILES) && (access & WRITE))
              {
               cmd = NEW_FILE;
               done = TRUE;
              }
            else
              beep();
           }
         else if (((kch = key_map(ch)) != NEWLINE) ? (kch == MAIN_MENU) : TRUE)
           {
            cmd = CHOOSE;
            done = TRUE;
           }
         else if (kch == ESC)
           {
            cmd = (active == DIRS) ? SET_FILE : CANCEL;
            done = TRUE;
           }
         else if ((kch == UP_ARROW) || (kch == PAGE_UP) || (kch == BUFFER_TOP))
           {
            if (*select > 0)
              {
               if (kch == UP_ARROW)
                 new_select = (*select)-1;
               else if ((kch == PAGE_UP) && (*select > max_display-1))
                 new_select = (*select)-max_display;
               else
                 new_select = 0;
              }
            else
              beep();
           }
         else if ((kch == DOWN_ARROW) || (kch == PAGE_DOWN) || 
                  (kch == BUFFER_END))
           {
            if (*select < limit-1)
              {
               if (kch == DOWN_ARROW)
                 new_select = (*select)+1;
               else if ((kch == PAGE_DOWN) && 
                        ((limit-(*select)) > max_display))
                 new_select = (*select)+max_display;
               else
                 new_select = limit-1;
              }
            else
              beep();
           }
         else
           beep();
        }
#if MOUSE
      else if (mouse_moved_posn(&y,&x))
        {
         cmd = ms_map(active,y-disp_y-MOUSE_OFFSET_Y,x-disp_x-MOUSE_OFFSET_X);
         if (cmd == CHOOSE)
           {
            if (active == DIRS)
              new_select = y-disp_y-CWD_OFFSET_Y-1+path_offset;
            else
              new_select = y-disp_y-FILE_OFFSET_Y-1+file_offset;
            if (new_select > limit-1)
              new_select = *select;
           }
        }
      else if (mouse_clicked(&ch,&y,&x))
        {
         if (ch == LEFT_BUTTON)
           {
            cmd = ms_map(active,y-disp_y-MOUSE_OFFSET_Y,
                                x-disp_x-MOUSE_OFFSET_X);
            switch(cmd)
              {
               case PG_UP         : if (*select > 0)
                                      {
                                       if (*select > max_display-1)
                                         new_select = (*select)-max_display;
                                       else
                                         new_select = 0;
                                      }
                                    else
                                      beep();
                                    break;
               case PG_DN         : if (*select < limit-1)
                                      {
                                       if ((limit-(*select)) > max_display)
                                         new_select = (*select)+max_display;
                                       else
                                         new_select = limit-1;
                                      }
                                    else
                                      beep();
                                    break;
               case CHOOSE        : 
               case SET_DRIVE     : 
               case SET_DIRECTORY : 
               case SET_FILE      : 
               case NEW_FILE      : done = TRUE;
                                    break;
               default            : beep();
              }
           }
         else if (ch == RIGHT_BUTTON)
           {
            cmd = (active == DIRS) ? SET_FILE : CANCEL;
            done = TRUE; 
           }
         else
           beep();
        }
#endif
     }
#if MOUSE
   hide_mouse();
#endif
   return(cmd);
  }

/*****************************************************************************
 NAME        : change_disk
 PURPOSE     : Resets the default disk drive
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : TRUE (1) if the drive was changed, FALSE (0) otherwise
 NOTES       : None
 *****************************************************************************/
static int change_disk()
  {
   int dsk,
       done = FALSE;
   char newcwd[MAXDRIVE+MAXDIR+1];
   register int i;
   int rtn = TRUE;
#if IBM_MSC
   unsigned drives;
#endif
#if MOUSE
   int y,x;   

   hide_mouse();
#endif
   wattrset(display,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]|A_BLINK);
   mvwaddstr(display,HELP_OFFSET_Y-1,HELP_OFFSET_X,"Press drive-letter...");
   wrefresh(display);
   hide_cursor();
   while (! done)
     {
      if ((dsk = wgetch(display)) != -1)
        {
         if (dsk == ESC)
           done = TRUE;
         else if ((dsk >= 'a') && (dsk <= 'z'))
           {
            dsk -= 'a'-'A';
            done = TRUE;
           }
         else if ((dsk >= 'A') && (dsk <= 'Z'))
           done = TRUE;
         else
           beep();
        }
#if MOUSE
      else if (mouse_clicked(&dsk,&y,&x))
        {
         if (dsk == RIGHT_BUTTON)
           {
            dsk = ESC;
            done = TRUE;
           }
         else
           beep();
        }
#endif
     }
   wattrset(display,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
   mvwaddstr(display,HELP_OFFSET_Y-1,HELP_OFFSET_X,"                     ");
   wrefresh(display);
   hide_cursor();
   if (dsk == ESC)
     {
#if MOUSE
      show_mouse();
#endif
      return(FALSE);
     }
   if (disk_ready(dsk-'A'))
     {
#if IBM_TBC
      (void) setdisk(dsk-'A');
#else
      _dos_setdrive(dsk-'A'+1,&drives);
#endif
      if (getcwd(newcwd,MAXDRIVE+MAXDIR+1) == NULL)
        {
         beep();
         rtn = FALSE;
         if (disk_ready(cdsk-'A'))
#if IBM_TBC
           (void) setdisk(cdsk);
#else
           _dos_setdrive(cdsk+1,&drives);
#endif
        }
      else
        {
         strcpy(cwd,newcwd);
         cdsk = dsk-'A';
         get_disk_label();
         wattrset(display,colors[POP_FILE_TEXT]|colors[POP_FILE_BGD]);
         mvwaddch(display,DRIVE_OFFSET_Y+1,DRIVE_OFFSET_X+3,dsk);
         for (i = LABEL_OFFSET_X+1 ; i < LABEL_OFFSET_X+LABEL_COLS-1 ; i++)
           mvwaddch(display,LABEL_OFFSET_Y+1,i,BLANK);
         mvwaddstr(display,LABEL_OFFSET_Y+1,
                           LABEL_OFFSET_X+LABEL_COLS/2-strlen(label)/2,label);
         wrefresh(display);
         hide_cursor();
        }
     }
   else
     {
      beep();
      rtn = FALSE;
     }
#if MOUSE
   show_mouse();
#endif
   return(rtn);
  }

/*****************************************************************************
 NAME        : change_directory
 PURPOSE     : Builds new working-directory from the fragment of the stack
               below and including the current selection
 DESCRIPTION : 
 INPUTS      : 1) The index of the selection in the directory-menu
 RETURNS     : TRUE
 NOTES       : This routine has the side effects of changing the global
               variable cwd (which hold the current-working directory), so 
               that when the network of routines is called again, they
               will analyze the new directory.  The actual working directory
               is also changed.
 *****************************************************************************/
static int change_directory(int select)
  {
   char newdir[CWD_COLS-1];
   register int i;

   if (select == 0)
     return(TRUE);
   file[0] = cdsk+'A';
   file[1] = DRV_SEPARATOR;
   file[2] = EOS;
   if (select != dir_count-1)
     {
      for (i = dir_count-2 ; i >= select ; i--)
        {
         grab_text(newdir,path_menu,i);
         strcat(file,newdir);
        }
     }
   else
     {
      file[2] = DIR_SEPARATOR;
      file[3] = EOS;
     }
   (void) chdir(file);
   if (disk_ready(cdsk))
     strcpy(cwd,file);
   else
     {
      beep();
      file[0] = EOS;
      return(FALSE);
     }
   file[0] = EOS;
   return(TRUE);
  }

/*****************************************************************************
 NAME        : select_file
 PURPOSE     : Stores name of user-selected file for return to caller
 DESCRIPTION : Reads the file directly from the file_menu window.
               Directory files are recognized by noting that their names
               are preceded by a backslash.
 INPUTS      : 1) The index of the selection in the file-menu
               2) An address where a change toi the command code can be stored
                  (if necessary) - this used to reset the code from CHOOSE to
                  SET_DIRECTORY if the user-selected file is a directory.
               3) Type of file access req'd : READ and/or WRITE
 RETURNS     : TRUE
 NOTES       : This routine has the side effects of changing the global
               variable cwd (which hold the current-working directory), so 
               that when the network of routines is called again, they
               will analyze the new directory.  The actual working directory
               is also changed.
 *****************************************************************************/
static int select_file(int select,int *command,unsigned char access)
  {
   char newfile[FILE_COLS-1];

   grab_text(newfile,file_menu,select);
   if (cwd[strlen(cwd)-1] == DIR_SEPARATOR)
     strcpy(file,cwd);
   else
     sprintf(file,"%s%c",cwd,DIR_SEPARATOR);
   if (newfile[0] == DIR_SEPARATOR)
     {
      *command = SET_DIRECTORY;
      strcat(file,newfile+1);
      (void) chdir(file);
      if (disk_ready(cdsk))
        strcpy(cwd,file);
      else
        {
         beep();
         file[0] = EOS;
         return(FALSE);
        }
      file[0] = EOS;
      return(TRUE);
     }
   else
     strcat(file,newfile);
   if (access == WRITE)
     {
      if (! check_overwrite(newfile))
        {
         file[0] = EOS;
         return(FALSE);
        }
     }
   return(TRUE);
  }

/*****************************************************************************
 NAME        : new_file
 PURPOSE     : Allows the user to enter the name of a new file for the cwd
 DESCRIPTION : This file-name is appended to the cwd-path-string and returned
               to the caller of pop_up_files() in the static buffer file[].
 INPUTS      : 1) File access : READ and/or WRITE
 RETURNS     : TRUE (1) if a new file was entered, FALSE (0) if the option
               was cancelled
 NOTES       : None
 *****************************************************************************/
static int new_file(unsigned char access)
  {
   WINDOW *area = NULL;
   int ch,
       offset = 0,
       done = FALSE;
   register int i = 0,j;
#if MOUSE
   int y,x;
#endif

   if ((area = newpad(1,MAXPATH)) == NULL)
     return(FALSE);
   xpaint(area,colors[POP_FILE_TEXT]|colors[POP_FILE_BGD]);
   wattrset(area,colors[POP_FILE_TEXT]|colors[POP_FILE_BGD]);
   wmove(area,0,0);
   nfdisp(area,0,0);
   while (! done)
     {
      if ((ch = wgetch(display)) != -1)
        {
         ch = key_map(ch);
         if ((ch > BLANK) && (ch <= HIGH_PRN_ASCII))
           {
            if (i != MAXPATH-1)
              {
               if ((ch >= 'a') && (ch <= 'z'))
                 ch -= 'a'-'A';
               file[i++] = ch;
               file[i] = EOS;
               waddch(area,ch);
               if (i >= (NEW_FILE_COLS-2))
                 offset = i-(NEW_FILE_COLS-3);
               nfdisp(area,offset,i);
              }
            else
              beep();
           }
         else if (ch == DELETE)
           {
            if (i != 0)
              {
               file[--i] = EOS;
               waddstr(area,DEL_STR);
               if (i >= (NEW_FILE_COLS-2))
                 offset = i-(NEW_FILE_COLS-3);
               else
                 offset = 0;
               nfdisp(area,offset,i);
              }
            else
              beep();
           }
         else if ((ch == NEWLINE) || (ch == MAIN_MENU))
           done = TRUE;
         else if (ch == ESC)
           {
            file[0] = EOS;
            done = TRUE;
           }
         else
           beep();
        }
#if MOUSE
      else if (mouse_clicked(&ch,&y,&x))
        {
         if (ch == LEFT_BUTTON)
           done = TRUE;
         else if (ch == RIGHT_BUTTON)
           {
            file[0] = EOS;
            done = TRUE;
           }
         else
           beep();
        }
#endif
     }
   xpaint(area,colors[POP_FILE_TEXT]|colors[POP_FILE_BGD]);
   mvwaddstr(area,0,2,"NEW FILE");
   nfdisp(area,0,0);
   hide_cursor();
   delwin(area);
   if (file[0] != EOS)
     {
      adjust_new_file_path();
      if ((access == WRITE) ? (! check_overwrite(file)) : FALSE)
        {
         file[0] = EOS;
         return(FALSE);
        }
      return(TRUE);
     }
   return(FALSE);
  }

/*****************************************************************************
 NAME        : grab_text
 PURPOSE     : Reads text from a menu window and stores it in caller's buffer
 DESCRIPTION : 
 INPUTS      : 1) The caller's buffer address
               2) The address of the menu-window to get the text from
               3) The row coordinate of the text in the menu
 RETURNS     : Nothing useful
 NOTES       : Starts grabbing text at the 1st non-white-space and then
                 stops at the next white-space.
               Assumes the user's buffer is big enough to hold up to
                 <window-columns> characters
 *****************************************************************************/
static void grab_text(char *text,WINDOW *menu,int row)
  {
   register int i,j;
   int cols;
   char ch = BLANK;

   text[0] = EOS;
   getmaxc(menu,cols);
   for (i = 0 ; ((i < cols) ? (ch <= BLANK) : FALSE) ; i++)
     ch = mvwinch(menu,row,i) & A_CHARTEXT;
   for (j = 0 ; ((i < cols) ? (ch > BLANK) : FALSE) ; i++)
     {
      text[j++] = ch;
      text[j] = EOS;
      ch = mvwinch(menu,row,i) & A_CHARTEXT;
     }
  }

/*****************************************************************************
 NAME        : check_overwrite
 PURPOSE     : Check to see if file exists and if it does prompts user for
               overwrite privileges
 DESCRIPTION : 
 INPUTS      : 1) The name of the file (full-path name)
 RETURNS     : TRUE (1) if overwrite permission is granted, FALSE (0) otherwise
 NOTES       : None
 *****************************************************************************/
static int check_overwrite(char *fname)
  {
   register int i;
   int ch,
       done = FALSE,
       query = FALSE;
#if MOUSE
   int y,x;
#endif

   if (access(fname,6) == 0)     /* Check for existence and read/write privs */
     {
      wattrset(display,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]|A_BLINK);
      wmove(display,FILE_OFFSET_Y-1,FILE_OFFSET_X+1);
      wprintw(display,"Overwrite file? [y/n]");
      wrefresh(display);
      hide_cursor();
      while (! done)
        {
         if ((ch = wgetch(display)) != -1)
           {
            if ((ch == 'y') || (ch == 'Y'))
              query = TRUE;
            done = TRUE;
           }
#if MOUSE
         else if (mouse_clicked(&ch,&y,&x))
           {
            if (ch == LEFT_BUTTON)
              query = TRUE;
            done = TRUE;             
           }
#endif
        }
      wattrset(display,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
      wmove(display,FILE_OFFSET_Y-1,FILE_OFFSET_X+1);
      for (i = FILE_OFFSET_X+1 ; i < DISPLAY_COLS-1 ; i++)
        waddch(display,BLANK);
      wrefresh(display);
      hide_cursor();
      return(query);
     }
   return(TRUE);
  }

/*****************************************************************************
 NAME        : adjust new_file_path
 PURPOSE     : Stuffs current disk-drive/working-directory into file name if
               these parameters not already specified.
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Uses the Turbo-C Library Routine fnsplit()
 *****************************************************************************/
static void adjust_new_file_path()
  {
   int pthflgs,i;
   char adjfile[MAXPATH+1],
        drive[MAXDRIVE+1],
        dir[MAXDIR+1],
        name[MAXFILE+1],
        ext[MAXEXT+1];

#if IBM_TBC
   pthflgs = fnsplit(file,NULL,dir,name,ext);
#else
   _splitpath(file,drive,dir,name,ext);
   pthflgs = 0x00;
   if (drive[0] != EOS)
     pthflgs |= DRIVE;
   if (dir[0] != EOS)
     pthflgs |= DIRECTORY;
   if (name[0] != EOS)
     pthflgs |= FILENAME;
   if (ext[0] != EOS)
     pthflgs |= EXTENSION;
#endif
   if (! pthflgs)
     {
      file[0] = EOS;
      beep();
      return;
     }
   adjfile[0] = EOS;
   if (pthflgs & DRIVE)
     {
      adjfile[0] = file[0];
      adjfile[1] = DRV_SEPARATOR; 
      adjfile[2] = EOS; 
      if (pthflgs & DIRECTORY)
        strcat(adjfile,dir);
     }
   else
     {
      adjfile[0] = cdsk+'A';
      adjfile[1] = DRV_SEPARATOR; 
      adjfile[2] = EOS; 
      if (pthflgs & DIRECTORY)
        strcat(adjfile,dir);
      else
        strcat(adjfile,cwd+2);
      i = strlen(adjfile);
      if (adjfile[i-1] != DIR_SEPARATOR)
        {
         adjfile[i++] = DIR_SEPARATOR;
         adjfile[i] = EOS;
        }
     }
   strcat(adjfile,name);
   if (pthflgs & EXTENSION)
     strcat(adjfile,ext);
   strcpy(file,adjfile);
  }
