/* ************************************************************************
   FILE : COLOR.C
      This file contains the support routines that determine the colors
   of objects on the CLIPS screen as well as defining new colors.
      The run-time color setting facility uses some interface routines
   defined in KEY.C for the run-time key-binding facility.
   *********************************************************************** */

/* =======================================================================
  ***********************************************************************
                             HEADER FILES
   ***********************************************************************
   ======================================================================= */
#include "common.h"   /* Standard header files, constants, and macros    */
#define _NO_STDIO
#include <curses.h>   /* Aspen Scientific Curses V4.0 Header Files -     */
#include <xcurses.h>  /*    Screen interface function prototypes and     */
#include <box.h>      /*      constants                                  */
#include "curproto.h"
#define COLOR_SOURCE
#include "color.h"    /* Color constant and access routines defns        */
#if ENVIRONMENT_UTILITY
#include "clpsutil.h" /* Utility interface routines                      */
#include "env.h"      /* Memory status function external definition      */
#include "key.h"      /* Run-time binding utilities generic routines     */
#include "mouse.h"
#endif

/* =======================================================================
   ***********************************************************************
                                 CONSTANTS
   ***********************************************************************
   ======================================================================= */

#if ENVIRONMENT_UTILITY

#if MOUSE
#define MAX_MENU_ITEMS MAX_COLOR_DEFS
#else
#define MAX_MENU_ITEMS MAX_COLOR_DEFS-1
#endif

#define MAX_COLOR_NAME 17   /* Max length of color item description */

#define PALETTE_ROWS 18
#define PALETTE_COLS 6
/* Palette-Y defined at run-time by # of lines on the screen */
#define PALETTE_X    1

/* Menu-Rows defined at run-time by # of lines on the screen    */
#define MENU_COLS    MAX_COLOR_NAME+2
/* Menu-Y defined at run-time by # of lines on the screen    */
#define MENU_X       PALETTE_X+PALETTE_COLS+1

/* Color demonstration window dimensions */

#define DEMO_ROWS 20
#define DEMO_COLS 52
/* Demo-Y defined at run-time by # of lines on the screen    */
#define DEMO_X    ALL_COLS-DEMO_COLS

/* -- color_file_io() flags --- */

#define LOAD     0
#define SAVE     1
#define DEFAULTS 2
#define MONO     3

#endif

/* =======================================================================
   ***********************************************************************
                                  MACROS
   ***********************************************************************
   ======================================================================= */
#define hide_cursor() mvcur(0,0,LINES,0)

/* =======================================================================
   ***********************************************************************
                      EXTERNALLY VISIBLE FUNCTION PROTOTYPES
   ***********************************************************************
   ======================================================================= */
#if ENVIRONMENT_UTILITY
void set_colors(void);     /* Run-time color setup */
#endif

/* =======================================================================
   ***********************************************************************
                      INTERNALLY VISIBLE FUNCTION PROTOTYPES
   ***********************************************************************
   ======================================================================= */
#if ENVIRONMENT_UTILITY
static void create_palette(void), /* Allocates 16-color palette window      */
            create_menu(void),    /* Allocates color menu pad               */
            create_demos(void),   /* Allocates color demonstration windows  */
            deallocate_windows(void),
                                  /* Releases memory for all windows        */
            draw_palette(void),   /* Draws 16-color palette window          */
            draw_menu(void),      /* Draws color menu pad                   */
            draw_demo(void);      /* Draws color demonstration window       */

static void draw_demo_background(void),
            draw_demo_border(void),
            draw_demo_stats(void),
            draw_demo_logo(void),
            draw_demo_main_menu(void),
            draw_demo_clips_io(void),
            draw_demo_pop_menu(void),
            draw_demo_pop_text(void),
            draw_demo_pop_warn(void),
            draw_demo_file_menu(void),
            refresh_demo(void);   /* Flushes demo-windows to virtual screen */

static void select_loop(void),    /* Item selection loop                    */
            set_loop(int),        /* Color setting loop                     */
            menu_choose(int),     /* Highlights current menu selection      */
            palette_choose(int),  /* Labels current palette selection       */
            place_palette_choice(int);
                                  /* Support routine for above              */
static void color_file_io(int);   /* Performs loads, saves, and defaults    */
#endif

/* =======================================================================
   ***********************************************************************
                       EXTERNALLY VISIBLE GLOBAL VARIABLES
   ***********************************************************************
   ======================================================================= */

/* Screen Interface Colors */

ch_attr colors[MAX_COLOR_DEFS] =
  {
   A_FRED,         /* wclips color                  */
   A_FBROWN,       /* wdialog color                 */
   A_FCYAN,        /* wdisplay color                */
   A_FGREEN,       /* werror color                  */
   A_FWHITE,       /* wtrace color                  */
   A_FCYAN,        /* wagenda color                 */
   A_FCYAN,        /* stdin color                   */
   A_FCYAN,        /* stdout color                  */
   A_BBLACK,       /* background color              */
   A_FBLUE,        /* border_color                  */
   A_FLT_RED,      /* advertisement color           */
   A_BBLACK,       /* main menu background color    */
   A_FGREEN,       /* main menu text color          */
   A_FWHITE,       /* Memory status message color   */
   A_FBROWN,       /* Watch status message color    */
   A_FLT_GREEN,    /* --More-- message color        */
   A_BWHITE,       /* pop up menu background color  */
   A_FRED,         /* pop up menu border color      */
   A_FRED,         /* pop up menu text color        */
   A_BCYAN,        /* pop up text background color  */
   A_FBLUE,        /* pop up text border color      */
   A_FBLUE,        /* pop up text prompt color      */
   A_FBLUE,        /* pop up text input color       */
   A_BRED,         /* pop up warn background color  */
   A_FYELLOW,      /* pop up warn border color      */
   A_FYELLOW,      /* pop up warn text color        */
   A_BWHITE,       /* pop up files menu background  */
   A_FBLUE,        /* pop up files menu border      */
   A_FRED,        /* pop up files menu text        */
   A_FYELLOW       /* pop up files mouse cursor     */
  };

ch_attr mono[MAX_COLOR_DEFS] =
  {
   /* WCLIPS          */ A_FWHITE,
   /* WDIALOG         */ A_FWHITE,
   /* WDISPLAY        */ A_FWHITE,
   /* WERROR          */ A_FWHITE,
   /* WTRACE          */ A_FWHITE,
   /* WAGENDA         */ A_FWHITE,
   /* STDIN           */ A_FWHITE,
   /* STDOUT          */ A_FWHITE,
   /* MAIN_BACKGD     */ A_BBLACK,
   /* MAIN_BORDER     */ A_FHI_WHITE,
   /* ADVERTISE       */ A_FWHITE,
   /* MAIN_MENU_BGD   */ A_BBLACK,
   /* MAIN_MENU_TEXT  */ A_FWHITE,
   /* MEMORY_MSG      */ A_FWHITE,
   /* WATCH_MSG       */ A_FWHITE,
   /* MORE_MSG        */ A_FHI_WHITE,
   /* POP_MENU_BGD    */ A_BBLACK,
   /* POP_MENU_BORDER */ A_FHI_WHITE,
   /* POP_MENU_TEXT   */ A_FWHITE,
   /* POP_TEXT_BGD    */ A_BBLACK,
   /* POP_TEXT_BORDER */ A_FHI_WHITE,
   /* POP_TEXT_PROMPT */ A_FWHITE,
   /* POP_TEXT_INPUT  */ A_BBLACK,
   /* POP_WARN_BGD    */ A_BBLACK,
   /* POP_WARN_BORDER */ A_FHI_WHITE,
   /* POP_WARN_TEXT   */ A_FWHITE,
   /* POP_FILE_BGD    */ A_BBLACK,
   /* POP_FILE_BORDER */ A_FHI_WHITE,
   /* POP_FILE_TEXT   */ A_FWHITE,
   /* POP_FILE_MOUSE  */ A_FHI_WHITE
  };

/* =======================================================================
   ***********************************************************************
                       INTERNALLY VISIBLE GLOBAL VARIABLES
   ***********************************************************************
   ======================================================================= */

#if ENVIRONMENT_UTILITY

/* ----------------------------------------
   Names corresponding to the color choices
   ---------------------------------------- */

static char itemnames[MAX_MENU_ITEMS][MAX_COLOR_NAME+1] =
  {
   "  CLIPS Prompt   ",
   "   Dialog Msgs   ",
   "  Display Msgs   ",
   "   Error Msgs    ",
   "   Trace Msgs    ",
   "   Agenda Msgs   ",
   " Standard Input  ",
   " Standard Output ",
   "    Main Bgd     ",
   "   Main Border   ",
   "   CLIPS Logo    ",
   "  Main Menu Bgd  ",
   "  Main Menu Fgd  ",
   "  Memory Status  ",
   "  Watch Status   ",
   " -- More -- Msg  ",
   "  Sub-Menu Bgd   ",
   " Sub-Menu Border ",
   "  Sub-Menu Fgd   ",
   "  Pop Text Bgd   ",
   " Pop Text Border ",
   " Pop Text Prompt ",
   "  Pop Text Fgd   ",
   "   Warning Bgd   ",
   " Warning Border  ",
   "   Warning Fgd   ",
   " Files Menu Bgd  ",
   "Files Menu Border",
   " Files Menu Fgd  "
#if MOUSE
  ,"Files Menu Mouse "
#endif
  };

/* -- Routines for updates of each color item -- */

static void (*update[MAX_MENU_ITEMS])(void) =
  {
   draw_demo_clips_io,
   draw_demo_clips_io,
   draw_demo_clips_io,
   draw_demo_clips_io,
   draw_demo_clips_io,
   draw_demo_clips_io,
   draw_demo_clips_io,
   draw_demo_clips_io,
   draw_demo_background,
   draw_demo_border,
   draw_demo_logo,
   draw_demo_main_menu,
   draw_demo_main_menu,
   draw_demo_stats,
   draw_demo_stats,
   draw_demo_stats,
   draw_demo_pop_menu,
   draw_demo_pop_menu,
   draw_demo_pop_menu,
   draw_demo_pop_text,
   draw_demo_pop_text,
   draw_demo_pop_text,
   draw_demo_pop_text,
   draw_demo_pop_warn,
   draw_demo_pop_warn,
   draw_demo_pop_warn,
   draw_demo_file_menu,
   draw_demo_file_menu,
   draw_demo_file_menu
#if MOUSE
   ,draw_demo_file_menu
#endif
  };

/* -- Color-Setting Utility Windows -- */

static WINDOW *palette = NULL,
              *menu_box = NULL,
              *menu = NULL,
              *demo = NULL,
              *mainmenudemo = NULL,
              *popmenudemo = NULL,
              *poptextdemo = NULL,
              *popwarndemo = NULL,
              *filemenudemo = NULL;

/* --- Palette start row --- */
static int palette_y = 0;

/* -- Menu pad dimensions -- */
static int menu_offset = 0,      /* Number of items above top of display */
           menu_rows = 0,
           menu_y = 0;

/* --- Menu/Palette Selection indices ---- */

static int menu_select = 0,
           palette_select = 0;

/* --- Bitmap designating which colors are background colors --- */

static unsigned long bgd_map = 0x00000000;

/* =======================================================================
   ***********************************************************************
                      EXTERNALLY VISIBLE FUNCTIONS
   ***********************************************************************
   ======================================================================= */

/******************************************************************************
 NAME        : set_colors
 PURPOSE     : Allows the user to redefine screen colors at run-time
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
void set_colors()
  {
   create_palette();
   if (palette == NULL)
     return;
   create_menu();
   if ((menu == NULL) || (menu_box == NULL))
     {
      deallocate_windows();
      return;
     }
   create_demos();
   if ((demo == NULL) || (mainmenudemo == NULL) || (popmenudemo == NULL) ||
       (poptextdemo == NULL) || (popwarndemo == NULL) || (filemenudemo == NULL))
     {
      deallocate_windows();
      return;
     }
   bgd_map = setbit(bgd_map,MAIN_BACKGD);
   bgd_map = setbit(bgd_map,MAIN_MENU_BGD);
   bgd_map = setbit(bgd_map,POP_MENU_BGD);
   bgd_map = setbit(bgd_map,POP_TEXT_BGD);
   bgd_map = setbit(bgd_map,POP_WARN_BGD);
   bgd_map = setbit(bgd_map,POP_FILE_BGD);
   attrset(A_FYELLOW|A_BBLACK);
   mvaddstr(0,COLS/2-22,"CLIPS DOS Interface Color-Setting Utility");
   attrset(A_FWHITE|A_BBLACK);
   mvaddstr(2,PALETTE_X,"Colors");
   mvaddstr(2,MENU_X,"Screen Items");
   mvaddstr(2,DEMO_X,"Demonstration Display");
   draw_palette();
   draw_menu();
   draw_demo();
#if MOUSE
   save_mouse();
#endif
   touchwin(stdscr);
   wnoutrefresh(stdscr);
   touchwin(menu_box);
   wnoutrefresh(menu_box);
   refresh_demo();
   touchwin(menu);
   touchwin(palette);
   wnoutrefresh(palette);

   select_loop();

   deallocate_windows();
#if MOUSE
   restore_mouse();
#endif
  }

/* =======================================================================
   ***********************************************************************
                      INTERNALLY VISIBLE FUNCTIONS
   ***********************************************************************
   ======================================================================= */

/******************************************************************************
 NAME        : create_palette
 PURPOSE     : Allocates space for the 16-color palette window and places
               it on the screen
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void create_palette()
  {
   palette_y = LINES/2-PALETTE_ROWS/2;
   if (palette_y < 3)
     palette_y = 3;
   else if (palette_y + PALETTE_ROWS > LINES-2)
     palette_y = LINES-2-PALETTE_ROWS;
   palette = newwin(PALETTE_ROWS,PALETTE_COLS,palette_y,PALETTE_X);
   nodelay(palette,TRUE);
   keypad(palette,TRUE);
  }

/******************************************************************************
 NAME        : create_menu
 PURPOSE     : Allocates space for the color-settings menu pad
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void create_menu()
  {
   if ((MAX_MENU_ITEMS+2) > (LINES-5))
     menu_rows = LINES-5;
   else
     menu_rows = MAX_MENU_ITEMS+2;
   menu_y = LINES/2-menu_rows/2;
   if (menu_y < 3)
     menu_y = 3;
   else if (menu_y + menu_rows > LINES-2)
     menu_y = LINES-2-menu_rows;
   menu_box = newwin(menu_rows,MENU_COLS,menu_y,MENU_X);
   if (menu_box == NULL)
     return;
   menu = newpad(MAX_MENU_ITEMS,MAX_COLOR_NAME);
   nodelay(menu,TRUE);
   keypad(menu,TRUE);
   xbox(menu_box,SG_V_LIN,SG_H_LIN,A_FWHITE);
  }

/******************************************************************************
 NAME        : create_demos
 PURPOSE     : Allocates space for a color demonstration window and draws
               a white single-line border around it.
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void create_demos()
  {
   int demo_y;

   demo_y = LINES/2-DEMO_ROWS/2;
   if (demo_y < 3)
     demo_y = 3;
   else if (demo_y + DEMO_ROWS > LINES-2)
     demo_y = LINES-2-DEMO_ROWS;
   if ((demo = newwin(DEMO_ROWS,DEMO_COLS,demo_y,DEMO_X)) == NULL)
     return;
   xbox(demo,SG_V_LIN,SG_H_LIN,A_FWHITE|A_BBLACK);
   if ((mainmenudemo = newwin(1,DEMO_COLS-12,demo_y+2,DEMO_X+10)) == NULL)
     return;
   if ((popmenudemo = newwin(4,15,demo_y+3,DEMO_X+20)) == NULL)
     return;
   if ((poptextdemo = newwin(4,15,demo_y+8,DEMO_X+22)) == NULL)
     return;
   if ((popwarndemo = newwin(4,15,demo_y+13,DEMO_X+24)) == NULL)
     return;
   if ((filemenudemo = newwin(4,15,demo_y+13,DEMO_X+3)) == NULL)
     return;
  }

/******************************************************************************
 NAME        : deallocate_windows
 PURPOSE     : Releases memory for all windows
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void deallocate_windows()
  {
   if (palette != NULL)
     {
      delwin(palette);
      palette = NULL;
     }
   if (menu_box != NULL)
     {
      delwin(menu_box);
      menu_box = NULL;
     }
   if (menu != NULL)
     {
      delwin(menu);
      menu = NULL;
     }
   if (demo != NULL)
     {
      delwin(demo);
      demo = NULL;
     }
   if (mainmenudemo != NULL)
     {
      delwin(mainmenudemo);
      mainmenudemo = NULL;
     }
   if (popmenudemo != NULL)
     {
      delwin(popmenudemo);
      popmenudemo = NULL;
     }
   if (poptextdemo != NULL)
     {
      delwin(poptextdemo);
      poptextdemo = NULL;
     }
   if (popwarndemo != NULL)
     {
      delwin(popwarndemo);
      popwarndemo = NULL;
     }
   if (filemenudemo != NULL)
     {
      delwin(filemenudemo);
      filemenudemo = NULL;
     }
  }

/******************************************************************************
 NAME        : draw_palette
 PURPOSE     : Draws 16-color palette
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Assumes palette window has been successfully allocated
 ******************************************************************************/
static void draw_palette()
  {
   register int i,j;

   xbox(palette,SG_V_LIN,SG_H_LIN,A_FWHITE);
   for (i = 1 ; i < PALETTE_ROWS-1 ; i++)
     {
      wattrset(palette,fgd_color(i-1) | A_BBLACK);
      wmove(palette,i,2);
      for (j = 2 ; j <= 4 ; j++)
        xwaddch(palette,ACS_BLOCK);
     }
  }

/******************************************************************************
 NAME        : draw_menu
 PURPOSE     : Fills in item names
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Assumes menu pad has been successfully allocated
               The menu is a pad (i.e. it can be bigger than the screen)
               and is displayed in sections using pnoutrefresh.
 ******************************************************************************/
static void draw_menu()
  {
   register int i;

   wattrset(menu,A_FWHITE);
   for (i = 0 ; i < MAX_MENU_ITEMS ; i++)
     mvwaddstr(menu,i,0,itemnames[i]);
  }

/******************************************************************************
 NAME        : draw_demo
 PURPOSE     : Draws a scaled-down version of the CLIPS interface screens to
               demonstrate current color settings
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Uses the colors defined in the colors array above.
               Assumes demo windows have been successfully allocated
 ******************************************************************************/
static void draw_demo()
  {
   draw_demo_background();
   draw_demo_border();
   draw_demo_logo();
   draw_demo_main_menu();
   draw_demo_clips_io();
   draw_demo_pop_menu();
   draw_demo_pop_text();
   draw_demo_pop_warn();
   draw_demo_file_menu();
  }

static void draw_demo_background()
  {
   register int i,j;

   wattrset(demo,0x0000);
   for (i = 1 ; i < DEMO_ROWS-1 ; i++)
     for (j = 1 ; j < DEMO_COLS-1 ; j++)
       waddch(demo,(mvwinch(demo,i,j) & BGD_MSK) | colors[MAIN_BACKGD]);
  }

static void draw_demo_border()
  {
   register int i,j;

   wattrset(demo,colors[MAIN_BORDER]|colors[MAIN_BACKGD]);
   mvwaddch(demo,1,1,DB_UL);
   for (j = 2 ; j < 9 ; j++)
     mvwaddch(demo,1,j,DB_H_LIN);
   mvwaddch(demo,1,j++,DBSG_UP_T);
   for ( ; j < DEMO_COLS-2 ; j++)
     mvwaddch(demo,1,j,DB_H_LIN);
   mvwaddch(demo,1,j,DB_UR);
   mvwaddch(demo,2,1,DB_V_LIN);
   mvwaddch(demo,2,9,SG_V_LIN);
   mvwaddch(demo,2,DEMO_COLS-2,DB_V_LIN);
   mvwaddch(demo,3,1,DBSG_LT_T);
   for (j = 2 ; j < 9 ; j++)
     mvwaddch(demo,3,j,SG_H_LIN);
   mvwaddch(demo,3,j++,SGSG_LO_T);
   for ( ; j < DEMO_COLS-2 ; j++)
     mvwaddch(demo,3,j,SG_H_LIN);
   mvwaddch(demo,3,j,DBSG_RT_T);
   for (i = 4 ; i < DEMO_ROWS-2 ; i++)
     {
      mvwaddch(demo,i,1,DB_V_LIN);
      mvwaddch(demo,i,DEMO_COLS-2,DB_V_LIN);
     }
   mvwaddch(demo,DEMO_ROWS-2,1,DB_LL);
   for (j = 2 ; j < DEMO_COLS-2 ; j++)
     mvwaddch(demo,DEMO_ROWS-2,j,DB_H_LIN);
   mvwaddch(demo,DEMO_ROWS-2,DEMO_COLS-2,DB_LR);
   
   draw_demo_stats();
  }

static void draw_demo_stats()
  {
   wattrset(demo,colors[WATCH_MSG]|colors[MAIN_BACKGD]);
   mvwaddstr(demo,DEMO_ROWS-2,2,"Watch R/F/A");
   wattrset(demo,colors[MORE_MSG]|colors[MAIN_BACKGD]);
   mvwaddstr(demo,DEMO_ROWS-2,DEMO_COLS/2-6," -- More -- ");
   wattrset(demo,colors[MEMORY_MSG]|colors[MAIN_BACKGD]);
   mvwaddstr(demo,DEMO_ROWS-2,DEMO_COLS-19,"000000 Bytes Left");
  }

static void draw_demo_logo()
  {
   wattrset(demo,colors[ADVERTISE]|colors[MAIN_BACKGD]);
   mvwaddstr(demo,2,3,"CLIPS");
  }

static void draw_demo_main_menu()
  {
   register int j;

   wattrset(mainmenudemo,colors[MAIN_MENU_TEXT]|colors[MAIN_MENU_BGD]);   
   for (j = 0 ; j < DEMO_COLS-12 ; j++)
     mvwaddch(mainmenudemo,0,j,BLANK);
   mvwaddstr(mainmenudemo,0,0," Unactive ");
   wattrset(mainmenudemo,
            flip_fgd_bgd(colors[MAIN_MENU_TEXT]|colors[MAIN_MENU_BGD]));
   waddstr(mainmenudemo,"  Active  ");
  }

static void draw_demo_clips_io()
  {
   wattrset(demo,colors[WCLIPS]|colors[MAIN_BACKGD]);
   mvwaddstr(demo,4,2,"CLIPS> ");
   wattrset(demo,colors[STDIN]|colors[MAIN_BACKGD]);
   mvwaddstr(demo,5,2,"Standard input");
   wattrset(demo,colors[STDOUT]|colors[MAIN_BACKGD]);
   mvwaddstr(demo,6,2,"Standard output");
   wattrset(demo,colors[WDIALOG]|colors[MAIN_BACKGD]);
   mvwaddstr(demo,7,2,"Dialog messages");
   wattrset(demo,colors[WDISPLAY]|colors[MAIN_BACKGD]);
   mvwaddstr(demo,8,2,"Display output");
   wattrset(demo,colors[WERROR]|colors[MAIN_BACKGD]);
   mvwaddstr(demo,9,2,"Error messages");
   wattrset(demo,colors[WTRACE]|colors[MAIN_BACKGD]);
   mvwaddstr(demo,10,2,"Trace messages");
   wattrset(demo,colors[WAGENDA]|colors[MAIN_BACKGD]);
   mvwaddstr(demo,11,2,"Agenda messages");
  }
   
static void draw_demo_pop_menu()
  {
   register ch_attr attr;

   xpaint(popmenudemo,colors[POP_MENU_BGD]);
   xbox(popmenudemo,DB_V_LIN,DB_H_LIN,
                    colors[POP_MENU_BORDER]|colors[POP_MENU_BGD]);
   attr = colors[POP_MENU_TEXT]|colors[POP_MENU_BGD];
   wattrset(popmenudemo,attr);
   mvwaddstr(popmenudemo,1,1,"  Unactive   ");
   wattrset(popmenudemo,flip_fgd_bgd(attr));
   mvwaddstr(popmenudemo,2,1,"   Active    ");
  }

static void draw_demo_pop_text()
  {
   register ch_attr attr;

   xpaint(poptextdemo,colors[POP_TEXT_BGD]);
   xbox(poptextdemo,DB_V_LIN,DB_H_LIN,
                    colors[POP_TEXT_BORDER]|colors[POP_TEXT_BGD]);
   wattrset(poptextdemo,colors[POP_TEXT_PROMPT]|colors[POP_TEXT_BGD]);
   mvwaddstr(poptextdemo,1,1,"Enter text : ");
   attr = colors[POP_TEXT_INPUT]|fgd_reverse(colors[POP_TEXT_INPUT]);
   if ((attr & FGD_MSK) == POP_TEXT_BGD)
     attr = flip_fgd_bgd(attr);
   wattrset(poptextdemo,attr);
   mvwaddstr(poptextdemo,2,5,"input ");
  }

static void draw_demo_pop_warn()
  {
   xpaint(popwarndemo,colors[POP_WARN_BGD]);
   xbox(popwarndemo,DB_V_LIN,DB_H_LIN,
                    colors[POP_WARN_BORDER]|colors[POP_WARN_BGD]);
   wattrset(popwarndemo,colors[POP_WARN_TEXT]|colors[POP_WARN_BGD]);
   mvwaddstr(popwarndemo,1,1,"   WARNING!  ");
  }

static void draw_demo_file_menu()
  {
   register ch_attr attr;

   xpaint(filemenudemo,colors[POP_FILE_BGD]);
   xbox(filemenudemo,DB_V_LIN,DB_H_LIN,
                      colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
   wattrset(filemenudemo,colors[POP_FILE_BORDER]|colors[POP_FILE_BGD]);
   mvwaddch(filemenudemo,1,7,SG_V_LIN);
   mvwaddch(filemenudemo,2,7,SG_V_LIN);
   attr = colors[POP_FILE_TEXT]|colors[POP_FILE_BGD];
   wattrset(filemenudemo,attr);
   mvwaddstr(filemenudemo,1,1,"DIRS");
   wattrset(filemenudemo,flip_fgd_bgd(attr));
   mvwaddstr(filemenudemo,2,1,"FILES");
#if MOUSE
   wattrset(filemenudemo,colors[POP_FILE_MOUSE]|colors[POP_FILE_BGD]);
   wmove(filemenudemo,1,8);
   xwaddch(filemenudemo,ACS_DIAMOND);
#endif
 }

/******************************************************************************
 NAME        : refresh_demo
 PURPOSE     : Flushes demo-images to virtual screen
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : Uses Curses touchwin() and wnoutrefresh() - use doupdate()
               to display images on physical screen.
 ******************************************************************************/
static void refresh_demo()
  {
   touchwin(demo);
   wnoutrefresh(demo);
   touchwin(mainmenudemo);
   wnoutrefresh(mainmenudemo);
   touchwin(popmenudemo);
   wnoutrefresh(popmenudemo);
   touchwin(poptextdemo);
   wnoutrefresh(poptextdemo);
   touchwin(popwarndemo);
   wnoutrefresh(popwarndemo);
   touchwin(filemenudemo);
   wnoutrefresh(filemenudemo);
  }

/******************************************************************************
 NAME        : select_loop
 PURPOSE     : Processes color-setting menu selections
 DESCRIPTION : 
 INPUTS      : None
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void select_loop()
  {
   int quit = FALSE,
       color_set = FALSE,
       ch,y,x,
       choice;
   
   instruction_line("\030\031-Select <CRT>-Change  F1-Load F2-Save F3-Defaults F4-Monochrome <ESC>-Quit");
   wnoutrefresh(stdscr);
#if MOUSE
   mouse_region(menu_y,MENU_X+1,menu_y+menu_rows-1,MENU_X+1);
#endif
   choice = menu_select = 0;
   menu_choose(choice);
   while (! quit)
     {
      if (color_set)
        {
#if MOUSE
         save_mouse();
#endif
         touchwin(palette);
         set_loop(choice);
#if MOUSE
         restore_mouse();
#endif
         instruction_line("\030\031-Select <CRT>-Change  F1-Load F2-Save F3-Defaults F4-Monochrome <ESC>-Quit");
         touchwin(stdscr);
         wnoutrefresh(stdscr);         
         touchwin(palette);
         wnoutrefresh(palette);         
         touchwin(menu_box);
         wnoutrefresh(menu_box);         
         refresh_demo();
         touchwin(menu);
         menu_choose(choice);
         color_set = FALSE;
        }
      if (choice != menu_select)
        menu_choose(choice);
      if ((ch  = wgetch(menu)) != -1)
        {
         clear_status_line();
         switch(ch)
           {
            case NEWLINE    : 
            case CRGRTN     : color_set = TRUE;
                              break;
            case ESC        : quit = TRUE;
                              break;
#if IBM_TBC                              
            case KEY_F(1)   : color_file_io(LOAD);
                              break;
            case KEY_F(2)   : color_file_io(SAVE);
                              break;
            case KEY_F(3)   : color_file_io(DEFAULTS);
                              break;
            case KEY_F(4)   : color_file_io(MONO);
                              break;
#endif
            default : ch = key_map(ch);
                      switch(ch)
                        {
                         case UP_ARROW   : if (choice != 0)
                                             choice--;
                                           break;
                         case DOWN_ARROW : if (choice != MAX_MENU_ITEMS-1)
                                             choice++;
                                           break;
                         case BUFFER_TOP : choice = 0;
                                           break;
                         case BUFFER_END : choice = MAX_MENU_ITEMS-1;
                                           break;
                         default         : beep();
                        }
           }
        }
#if MOUSE
      else if (mouse_moved_posn(&y,&x))
        {
         clear_status_line();
         y -= menu_y+1;
         if (y < 0)
           choice = menu_offset-1;
         else if (y > menu_rows-3)
           choice = menu_offset+menu_rows-2;
         else
           choice = menu_offset+y;
         if (choice > MAX_MENU_ITEMS-1)
           choice = MAX_MENU_ITEMS-1;
         else if (choice < 0)
           choice = 0;
        }
      else if (mouse_clicked(&ch,&y,&x))
        {
         clear_status_line();
         if (ch == LEFT_BUTTON)
           color_set = TRUE;
         else if (ch == RIGHT_BUTTON)
           quit = TRUE;
         else
           beep();
        }
#endif
     }
   check_for_save(COLOR_SETTINGS);
  }

/******************************************************************************
 NAME        : set_loop
 PURPOSE     : Processes new color-settings palette selections
 DESCRIPTION : 
 INPUTS      : 1) The array-index of the item for which color is being set
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void set_loop(item)
  int item;
  {
   int quit = FALSE,
       ch,y,x,
       choice;

   instruction_line("\030\031-Change Color  <ESC,CRT>-Quit");
   wnoutrefresh(stdscr);
#if MOUSE
   mouse_region(palette_y+1,PALETTE_X+1,palette_y+PALETTE_ROWS-2,PALETTE_X+1);
#endif
   choice = (testbit(bgd_map,item)) ? bgd_index(colors[item]) : 
                                      fgd_index(colors[item]);
   palette_select = choice;
   palette_choose(choice);
   while (! quit)
     {
      if (choice != palette_select)
        {
         colors_saved = FALSE;
         colors[item] = (testbit(bgd_map,item)) ? bgd_color(choice) :
                                                  fgd_color(choice);
         (*(update[item]))();
         refresh_demo();
         palette_choose(choice);
        }
      if ((ch = wgetch(palette)) != -1)
        {
         ch = key_map(ch);
         switch(ch)
           {
            case NEWLINE    :
            case ESC        : quit = TRUE;
                              break;
            case UP_ARROW   : if (testbit(bgd_map,item))
                                choice = (choice+MAX_BGDS-1)%MAX_BGDS;
                              else
                                choice = (choice+MAX_FGDS-1)%MAX_FGDS;
                              break;
            case DOWN_ARROW  : if (testbit(bgd_map,item))
                                choice = (choice+1)%MAX_BGDS;
                              else
                                choice = (choice+1)%MAX_FGDS;
                              break;
            default         : beep();
           }
        }
#if MOUSE
      else if (mouse_moved_posn(&y,&x))
        {
         choice = y-palette_y-1;
         if (testbit(bgd_map,item) ? (choice > MAX_BGDS-1) : FALSE)
           choice = MAX_BGDS-1;
        }
      else if (mouse_clicked(&ch,&y,&x))
        {
         if ((ch == LEFT_BUTTON) || (ch == RIGHT_BUTTON))
           quit = TRUE;
         else
           beep();
        }
#endif
     }
  }

/******************************************************************************
 NAME        : menu_choose
 PURPOSE     : (Un)higlights (old)new menu selections
 DESCRIPTION : 
 INPUTS      : 1) The index of the new selection
 RETURNS     : Nothing useful
 NOTES       : The menu is a pad, and this routine takes care of displaying
               the section of the pad that contains the new selection
 ******************************************************************************/
static void menu_choose(choice)
  int choice;
  {
   register int i;

   wattrset(menu,A_FWHITE|A_BBLACK);
   for (i = 0 ; i < MAX_COLOR_NAME ; i++)
     mvwaddch(menu,menu_select,i,mvwinch(menu,menu_select,i) & A_CHARTEXT);

   menu_select = choice;

   wattrset(menu,A_FBLACK|A_BWHITE);
   for (i = 0 ; i < MAX_COLOR_NAME ; i++)
     mvwaddch(menu,menu_select,i,mvwinch(menu,menu_select,i) & A_CHARTEXT);

   if (menu_select < menu_offset)
     menu_offset = menu_select;
   else if (menu_select > (menu_offset+menu_rows-3))
     menu_offset = menu_select-menu_rows+3;
   pnoutrefresh(menu,menu_offset,0,menu_y+1,MENU_X+1,
                                   menu_y+menu_rows-2,MENU_X+MAX_COLOR_NAME);
   if (testbit(bgd_map,choice))
     place_palette_choice(bgd_index(colors[choice]));
   else
     place_palette_choice(fgd_index(colors[choice]));
   doupdate();
   hide_cursor();
#if MOUSE
   set_mouse(menu_select-menu_offset+menu_y+1,MENU_X+1);
#endif
  }

/******************************************************************************
 NAME        : palette_choose
 PURPOSE     : Points arrow at currently defined color for item.
 DESCRIPTION : 
 INPUTS      : 1) The index of the color in the palette
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void palette_choose(choice)
  int choice;
  {
   place_palette_choice(choice);
   doupdate();
   hide_cursor();
#if MOUSE
   set_mouse(palette_select+palette_y+1,PALETTE_X+1);
#endif
  }

/******************************************************************************
 NAME        : place_aplette_choice
 PURPOSE     : Points arrow at color of current selection
 DESCRIPTION : 
 INPUTS      : 1) The palette index (0..15)
 RETURNS     : Nothing useful
 NOTES       : None
 ******************************************************************************/
static void place_palette_choice(choice)
  int choice;
  {
   wattrset(palette,A_FGREEN|A_BBLACK);
   mvwaddch(palette,palette_select+1,1,BLANK);
   palette_select = choice;
   wmove(palette,palette_select+1,1);
   xwaddch(palette,ACS_RARROW);
   wnoutrefresh(palette);
  }
#if IBM_TBC
/******************************************************************************
 NAME        : color_file_io
 PURPOSE     : Performs loads, saves, default restorations, and monochrome
 DESCRIPTION : 
 INPUTS      : 1) The file-io code : 
                  LOAD (0), SAVE(1), DEFAULTS (2), MONO (3)
 RETURNS     : Nothing useful
 NOTES       : Uses interface routines in CLPSUTIL.C
 ******************************************************************************/
static void color_file_io(code)
  int code;
  {
   char file[50];
   register int i;

   if (code == LOAD)
     {
      check_for_save(COLOR_SETTINGS);
      getstring("Load file : ",file,50);
      if (file[0] == EOS)
        return;
      writestring("Loading colors...",0);
      if (load_colors(file) == OK_DATA_FILE)
        {
         draw_demo();
         refresh_demo();
         menu_choose(menu_select);
         writestring("Colors loaded.",0);
        }
      else
        writestring("Invalid colors file!",1);
     }
   else if (code == SAVE)
     {
      getstring("Save file : ",file,50);
      if (file[0] == EOS)
        return;
      if (save_colors(file) == OK_DATA_FILE)
        writestring("Colors saved.",0);
      else
        writestring("Could not write to file!\n",1);
     }
   else if (code == DEFAULTS)
     {
      check_for_save(COLOR_SETTINGS);
      writestring("Loading default colors...",0);
      if (default_colors() == OK_DATA_FILE)
        {
         draw_demo();
         refresh_demo();
         menu_choose(menu_select);
         writestring("Default colors loaded.",0);
        }
      else
        writestring("Default colors-file corrupted!",1);
     }
   else
     {
      check_for_save(COLOR_SETTINGS);
      writestring("Loading monochrome settings...",0);
      for (i = 0 ; i < MAX_COLOR_DEFS ; i++)
        colors[i] = mono[i];
      draw_demo();
      refresh_demo();
      menu_choose(menu_select);
      writestring("Monochrome settings loaded.",0);
     }
  }
#endif
#endif
